/* pms.c -- Package Management System
 * Created: Mon Nov  1 17:52:39 1993 by faith@cs.unc.edu
 * Revised: Wed Jul 20 12:56:56 1994 by faith@cs.unc.edu
 * Copyright 1993, 1994 Rickard E. Faith (faith@cs.unc.edu)
 * Copyright 1994 Kevin E. Martin (martin@cs.unc.edu)
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2, or (at your option) any
 * later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 675 Mass Ave, Cambridge, MA 02139, USA.
 * 
 * $Id: pms.c,v 3.33 1994/07/20 17:00:48 root Exp $
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <fcntl.h>
#include <string.h>
#include <getopt.h>
#include <glob.h>
#include <unistd.h>
#include <time.h>
#include <pwd.h>
#include <grp.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <string.h>
#include <alloca.h>
#include <ftw.h>

				/* The modified ftw will never halt because
                                   it can't read a file. */

extern int pms_ftw(__const char *__dir,
		   __ftw_func_t __func,
		   int __descriptors);
extern int pms_system(register const char *line);


#define VERSION            "$Revision: 3.33 $"

#define DEFAULT_PMSPATH      "var/adm/pms"
#define DEFAULT_SOURCEPATH   "usr/src"
#define ALTERNATE_NOTES_NAME "NOTES"

enum Functions { install, list, query, deinstall, fixperms, tar, build, none };

typedef struct info {
   char *filename;
   char *owner;
   int  mode;
} Info;

int                   Debug           = 0;
static int            NoAutomatic     = 0;
static char           *Root           = "/";
static char           *RootSlash;
static char           *TarPath        = "./";
static int            Compressed      = 0;
static char           *PMSPath        = NULL;
static char           *AltPath        = NULL;
static char           *SourcePath     = NULL;
static int            NewPMSPath      = 0;
static int            NewSourcePath   = 0;
static int            UseTarDotOld    = 0;
static int            BuildCompile    = 0;
static int            BuildInstall    = 0;
static int            BuildAll        = 0;
static char           *MyPath         = NULL;
static char           *PackageName    = NULL;
static char           *PackageVersion = NULL;
static char           *PackageWhere   = NULL;
static int            NeedsLdconfig   = 0;
static enum Functions Function        = none;
static int            PrivateTarFiles = 0;
static int            Verbose         = 0;

static char *version( void )
{
   static char buffer[100];
   char        *pt;
   
   if (strchr( VERSION, ':')) { /* Assume VERSION is an RCS Revision string */
      strcat( buffer, strchr( VERSION, ':' ) + 2 );
      pt = strrchr( buffer, '$') - 1;
      if (!pt)                  /* Stripped RCS Revision string? */
            pt = buffer + strlen( buffer ) - 1;
      if (*pt != ' ')
            ++pt;
      *pt = '\0';
   } else {                     /* Assume VERSION is a number */
      strcat( buffer, VERSION );
   }
   return buffer;
}

static int Mkdir( const char *path, mode_t mode )
{
   if (Debug) {
      printf( "mkdir( %s, %04o )\n", path, mode );
      return 0;
   } else
	 return mkdir( path, mode );
}

static int Unlink( const char *path )
{
   if (Debug) {
      printf( "unlink( %s )\n", path );
      return 0;
   } else
	 return unlink( path );
}

static int Rename( const char *oldpath, const char *newpath, int message )
{
   int retcode;
   
   Unlink( newpath );
   if (Debug) {
      printf( "rename( %s, %s )\n", oldpath, newpath );
      return 0;
   } else {
      retcode = rename( oldpath, newpath );
      if (message && retcode) {
	 fprintf( stderr,
		  "pms: rename of \"%s\" to \"%s\" failed\n",
		  oldpath,
		  newpath );
	 perror( "pms" );
      }
   }

   return retcode;
}

static int Symlink( const char *oldpath, const char *newpath )
{
   if (Debug) {
      printf( "symlink( %s, %s )\n", oldpath, newpath );
      return 0;
   } else
	 return symlink( oldpath, newpath );
}

				/* pms_system is just like system, except
                                   /bin/sh is not invoked unless that
                                   argument is a shell script.  This means
                                   that the semantics may be quite
                                   different from system(3). */

static int System( const char *string )
{
   int retcode;
   
   fflush( stdout );
   retcode = pms_system( string );
   printf( "\n" );
   return retcode;
}

static int Chown( const char *path, uid_t owner, gid_t group )
{
   if (Debug) {
      printf( "chown( %s, %d, %d )\n", path, owner, group );
      return 0;
   } else
	 return chown( path, owner, group );
}

static int Chmod( const char *path, mode_t mode )
{
   if (mode) {			/* we give symlinks mode 0 */
      if (Debug) {
	 printf( "chmod( %s, %04o )\n", path, mode );
	 return 0;
      } else
	    return chmod( path, mode );
   }
   return 0;
}

static char *ResolveFileName( const char *package )
{
   int    fd = open( package, O_RDONLY );
   glob_t pglob = { 0, NULL, 0, 0 }; 

   if (fd < 0) {
      char *pattern = alloca( strlen( package ) + 2 );
      int  i;
      char *retval = NULL;
      
      strcpy( pattern, package );
      strcat( pattern, "*" );
      if (!glob( pattern, 0, NULL, &pglob )) {
	 if (pglob.gl_pathc == 1) {
	    retval = strdup( pglob.gl_pathv[0] );
	 } else {
	    int count = 0;
	    int last  = 0;
	    char *exact_pat = alloca( strlen( package ) + 8 );
	    
	    strcpy( exact_pat, package );
	    strcat( exact_pat, ".tar.gz" );

	    for (i = 0; i < pglob.gl_pathc; i++) {
	       if (!strcmp( pglob.gl_pathv[i], exact_pat )) {
		  count = 1;
		  last = i;
		  break;
	       } else if (strstr( strchr( pglob.gl_pathv[i], '.' ), "tar" )) {
		  ++count;
		  last = i;
	       }
	    }

	    if (count == 1)
		  retval = strdup( pglob.gl_pathv[ last ] );
	    else {
	       fprintf( stderr, "pms: ambiguous package name\n" );
	       exit( 2 );
	    }
	 }
      }
      globfree( &pglob );
      return retval;
   } else {
      close( fd );
      return strdup( package );
   }
}

static char *ResolvePackageName( const char *package )
{
   const char *pt;
   char       *start;
   char       *end;

   if (!(pt = strrchr( package, '/' )))
	 pt = package;
   else
	 ++pt;
   start = strdup( pt );
   
   if (!(end = strrchr( start, '.' )))
	 return start;
   
   if (!strcmp( end, ".z" )
       || !strcmp( end, ".gz" )
       || !strcmp( end, ".Z" )
       || !strcmp( end, ".Notes" )) {
   
      *end = '\0';
      if (!(end = strrchr( start, '.' )))
	    return start;
   }

   if (!strcmp( end, ".tar" )
       || !strcmp( end, ".taz")
       || !strcmp( end, ".tgz")
       || !strcmp( end, ".Notes" )) {
	 
      *end = '\0';
   }

   return start;
}

static char **GetList( const char *filename, int *count )
{
   char **retval;
   
   static char **fetch( char *command )
	 {
	    FILE *str;
	    char buffer[4096];
	    int  i;

	    *count = 0;
	    retval = malloc( sizeof( char * ) );

	    sprintf( buffer, "%s %s", command, filename );

	    if (!(str = popen( buffer, "r" ))) {
	       fprintf( stderr, "pms: cannot open \"%s\"\n", buffer );
	       exit( 2 );
	    }

	    while (fgets( buffer, 4096, str )) {
	       int len = strlen( buffer );
	       
	       if (len && buffer[ len - 1 ] == '\n')
		     buffer[ len - 1 ] = '\0';
	       retval[ *count ] = strdup( buffer );
	       ++*count;
	       retval = realloc( retval, (*count + 1) * sizeof( char * ) );
	    }
	    retval[ *count ] = NULL;

	    if (pclose( str )) {
	       for (i = 0; i < *count; i++)
		     free( retval[i] );
	       free( retval );
	       return NULL;
	    }

	    return retval;
	 }

   if (!fetch( "tar tvvf 2>/dev/null" ) || !*count) {
      Compressed = 1;
      fetch( "tar ztvvf" );
   }
   
   return retval;
}

static Info *MakeInfo( char **list, int count, size_t *size )
{
   int  i, j;
   Info *info = malloc( count * sizeof( Info ) );
   char *pt, *pt2;

   *size = 0;

   for (i = 0; i < count; i++) {
      int value = 1;
      
      info[i].mode = 0;
      if (list[i][0] != 'l') {	/* skip symlinks */
	 for (j = 9; j; j--) {
	    if (list[i][j] != '-')
		  info[i].mode += value;
	    value <<= 1;
	 }
	 if (list[i][9] == 't')
	       info[i].mode += S_ISVTX;
	 if (list[i][3] == 's')
	       info[i].mode += S_ISUID;
	 if (list[i][6] == 's')
	       info[i].mode += S_ISGID;
      }
	 
      pt = strchr( list[i], ' ' );
      ++pt;
      pt2 = strchr( pt, ' ' );
      --pt2;
      info[i].owner = malloc( pt2 - pt + 2 );
      strncpy( info[i].owner, pt, pt2 - pt + 1 );
      info[i].owner[ pt2 - pt + 1 ] = '\0';
      if ((pt = strchr( info[i].owner, '/' )))
	    *pt = ':';

      *size += (+(atoi( pt2 + 1 ) / 4096) * 4) + 4;

      pt = strrchr( list[i], ' ' );
      if (pt[-1] == '>' && pt[-2] == '-' && pt[-3] == ' ') {
	 pt -= 3;
	 *pt = '\0'; 
	 info[i].filename = strdup( strrchr( list[i], ' ' ) + 1 );
	 *pt = ' ';
      } else if (pt[-1] == 'o' && pt[-2] == 't' && pt[-3] == ' ') {
	 pt -= 8;
	 *pt = '\0'; 
	 info[i].filename = strdup( strrchr( list[i], ' ' ) + 1 );
	 *pt = ' ';
      } else {
	 info[i].filename = strdup( pt + 1 );
      }
   }
   
   return info;
}

static void DeleteFiles( Info *info, int count )
{
   int  i;
   char buffer[4096];

   for (i = 0; i < count; i++) {
      if (strstr( info[i].filename, "usr/src" )) {
	 printf( "     *NOT* deleting %s%s\n", RootSlash, info[i].filename );
      } else {
	 printf( "Deleting %s%s\n", RootSlash, info[i].filename );
	 sprintf( buffer, "%s%s", RootSlash, info[i].filename );
	 Unlink( buffer );
      }
   }
}

static void DeleteOldFiles( Info *info, int count )
{
   int  i;
   char buffer[4096];

   for (i = 0; i < count; i++) {
      sprintf( buffer, "%s%s.old", RootSlash, info[i].filename );
      Unlink( buffer );
   }
}

static void MakePath( const char *path )
{
   char *pt;

   for (pt = strchr( path, '/' ); pt != NULL; pt = strchr( pt + 1, '/' )) {
      
      /* Portions of the following code are from make_dirs() in extract.c
      in tar-1.11.2, written 19 Nov 1985 by John Gilmore,
      ihnp4!hoptoad!gnu.  This code is Copyright (C) 1988, 1992, 1993 Free
      Software Foundation, and is distributable under the terms of the GNU
      General Public License. */

				/* avoid mkdir of empty string, if leading or
				   double '/' */

      if (pt == path || pt[-1] == '/') continue;

				/* Avoid mkdir where last part of path is
                                   '.' */

      if (pt[-1] == '.' && (pt == path + 1 || pt[-2] == '/')) continue;
      *pt = '\0';		/* Truncate the path there */
      Mkdir( path, 0755 );
      *pt = '/';
   }
   Mkdir( path, 0755 );
}

static void MakeDirs( Info *info, int count )
{
   int         i;
   char        buffer[4096];
   char        *pt;
   static char cache[4096] = { 0, };

   for (i = 0; i < count; i++) {
      sprintf( buffer, "%s%s", RootSlash, info[i].filename );
      if ((pt = strrchr( buffer, '/'))) {
	 *pt = '\0';
	 if (strcmp( cache, buffer )) MakePath( buffer );
	 strcpy( cache, buffer );
      }
   }
}
   
static void MoveFiles( Info *info, int count )
{
   int  i;
   char buffer1[4096];
   char buffer2[4096];

   for (i = 0; i < count; i++) {
      int  len = strlen( info[i].filename );
      char *basename;

      basename = strrchr( info[i].filename, '/' );
      if (!basename) basename = info[i].filename;
      else           ++basename;

      if (!strncmp( basename, "lib", 3 ) && strstr( basename, ".so." )) {
	 char major[4096];
	 char full_major[4096];
	 char path[4096];
	 char *pt;

	 strcpy( path, info[i].filename );
	 pt = strrchr( path, '/' );
	 if (pt) *pt = '\0';

	 sprintf( buffer1, "%s%s", RootSlash, info[i].filename );
	 sprintf( buffer2, "%s%s/tmp-%s", RootSlash, path, basename );

	 if (!access( buffer1, R_OK )) {
	    
	    printf( "WARNING: This tar file contains shared libraries.\n"
		    "         The current \"%s\" file will be renamed.\n"
		    "         RUN LDCONFIG AFTER PMS COMPLETES!\n",
		    info[i].filename );

	    strcpy( major, basename );
	    pt = strchr( strstr( major, ".so." ) + 4, '.' );
	    if (pt) {
	       *pt = '\0';
	       sprintf( full_major, "%s%s/%s", RootSlash, path, major );
	       Rename( buffer1, buffer2, 0 );
	       Unlink( full_major );
	       Symlink( buffer2, full_major );
	    }

	    ++NeedsLdconfig;
	 }
      } else if (!(len >=  8
		      && !strcmp( info[i].filename + len - 8, "bin/gzip"))
		 && !(len >=  9
		      && !strcmp( info[i].filename + len - 9, "lib/ld.so"))) {
	 struct stat sb;
	 
	 sprintf( buffer1, "%s%s", RootSlash, info[i].filename );
	 sprintf( buffer2, "%s%s.old", RootSlash, info[i].filename );

	 if (stat( buffer1, &sb ) || !S_ISDIR( sb.st_mode)) {
				/* Rename stale links, but don't rename
				   directories */
	    Rename( buffer1, buffer2, 0 );
	 }
	 if (len >= 7 && !strcmp( info[i].filename + len - 7, "bin/tar"))
	       ++UseTarDotOld;
      }

   }
}

static int UntarFiles( const char *filename )
{
   char buffer[4096];
   
   if (UseTarDotOld) {		/* Maybe we are untarring tar? */
      if (Compressed)
	    sprintf( buffer, "tar.old zxvvpCf %s %s", Root, filename );
      else
	    sprintf( buffer, "tar.old xvvpCf %s %s", Root, filename );
      if (!System( buffer ))
	    return 0;
   }

   if (Compressed)
	 sprintf( buffer, "tar zxvvpCf %s %s", Root, filename );
   else
	 sprintf( buffer, "tar xvvpCf %s %s", Root, filename );
   
   return System( buffer );
}

static void MakePMSPath( void )
{
   MakePath( PMSPath );
}

static void MakeSourcePath( void )
{
   MakePath( SourcePath );
}

static void MakeTimestampFile( const char *pkg )
{
   FILE          *str;
   char          buffer[4096];
   time_t        t;
   struct passwd *pw;
   
   if (Debug)
	 return;

   MakePMSPath();

   sprintf( buffer, "%s/%s.Timestamp", PMSPath, pkg );

   if (!(str = fopen( buffer, "w" ))) {
      fprintf( stderr,
	       "pms: cannot create new timestamp file \"%s\"\n",
	       buffer );
   }

   time( &t );
   pw = getpwuid( getuid() );
   
   fprintf( str, "%s installed by %s, %s", pkg, pw->pw_name, ctime( &t ) );

   free( pw );
   fclose( str );
}

static void MakeNotesFile( const char *pkg, Info *info, int count )
{
   char   buffer[4096];
   int    fd;
   FILE   *str;
   int    i;
   time_t t;

   if (Debug)
	 return;
   
   time( &t );
   MakePMSPath();

   sprintf( buffer, "%s/%s.Notes", PMSPath, pkg );

   if ((fd = open( buffer, O_RDONLY )) < 0) {
      if (!(str = fopen( buffer, "w" ))) {
	 fprintf( stderr,
		  "pms: cannot create new notes file \"%s\"\n",
		  buffer );
	 exit( 2 );
      }
      fprintf( str, "%s\n", pkg );
      fprintf( str, "%%%%\n" );
      fprintf( str, "%%%%\n" );
      fprintf( str, "pms %s installed these files on %s",
	       version(),
	       ctime( &t ) );
      for (i = 0; i < count; i++) {
	 fprintf( str,
		  "* %-40s %-15s %04o\n",
		  info[i].filename,
		  info[i].owner,
		  info[i].mode );
      }
      fclose( str );
   }
}

static int RunPostInstallationScript( const char *pkg )
{
   char   buffer[4096];
   FILE   *str;
   FILE   *strOut;
   char   *tmp_file;
   int    flag        = 0;
   int    haveSpecial = 0;
   int    retval      = 0;
   time_t t;

   if (NoAutomatic) return 0;

   time( &t );
   
   sprintf( buffer, "%s/%s.Notes", PMSPath, pkg );
   if (!(str = fopen( buffer, "r" ))) {
      fprintf( stderr,
	       "pms: cannot open new notes file (\"%s\") for read\n",
	       buffer );
      exit( 2 );
   }

   tmp_file = tempnam( "/tmp", "PMS" );
   if (!(strOut = fopen( tmp_file, "w" ))) {
      fprintf( stderr, "pms: cannot open temporary file \"%s\"\n", tmp_file );
      exit( 2 );
   }

   fprintf( strOut, "#!/bin/sh -e\n" );
   fprintf( strOut, "# This is an automatically generated script\n" );
   fprintf( strOut, "# Generated by pms %s, on %s", version(), ctime( &t ) );
   fprintf( strOut, "# Build script for %s\n", pkg );
   fprintf( strOut, "\n" );
   fprintf( strOut, "# Set up root's path\n" );
   fprintf( strOut,
	    "export PATH=.:/sbin:/bin:/usr/sbin:/usr/bin:/usr/bin/X11"
	    ":$PATH\n" );
   fprintf( strOut, "\n" );
   fprintf( strOut, "# Environment variables (automatically generated):\n" );
   fprintf( strOut, "\n" );

   fprintf( strOut, "export ROOTDIR=%s\n",     Root );
   fprintf( strOut, "export SOURCEDIR=%s\n",   SourcePath );
   fprintf( strOut, "export NOTESDIR=%s\n",    PMSPath );
   fprintf( strOut, "export ALTNOTESDIR=%s\n", AltPath );
   fprintf( strOut, "export BINTARDIR=%s\n",   TarPath );
   if (Verbose)
	 fprintf( strOut, "set -x\n" ); /* skip the environment variables */

   while (fgets( buffer, 4095, str )) {
      buffer[ strlen( buffer ) - 1 ] = '\0'; /* kill newline */
      if (flag) {
	 if (!strncmp( buffer, "%%", 2 )) break;
	 if (!strncmp( buffer, "exit", 4 )) break; /* don't copy patches */
	 if (buffer[0] == '%'
	     && buffer[1] == 'i'
	     && (buffer[2] == ' ' || buffer[2] == '\t')) {
	    ++haveSpecial;
	    fprintf( strOut, "%s\n", buffer + 3 );
	 }
      } else {
	 if (!strncmp( buffer, "%%", 2 )) {
	    flag = 1;
	    fprintf( strOut, "\n" );
	    fprintf( strOut,
		     "# Post-installation commands (from Notes file):\n" );
	    fprintf( strOut, "\n" );
	 }
      }
   }

   fclose( str );
   fclose( strOut );
   
   if (haveSpecial) {
      printf( "Running special post-installation script:\n\n" );
      Chmod( tmp_file, 0700 );
      retval = System( tmp_file );
   }
   
   Unlink( tmp_file );

   return retval;
}

static int Install( const char *package )
{
   char   *filename = ResolveFileName( package );
   char   *pkg      = ResolvePackageName( package );
   char   **list;
   int    count;
   Info   *info;
   size_t size;

   if (!filename) {
      fprintf( stderr, "pms: \"%s\" not found\n", package );
      exit( 2 );
   }

   printf( "Installing package \"%s\" from \"%s\"\n", pkg, filename );
   
   if (!(list = GetList( filename, &count ))) {
      fprintf( stderr, "pms: Cannot get directory from tar\n" );
      exit( 2 );
   }

   info = MakeInfo( list, count, &size );
   printf( "Estimate %d kb required to install\n", size );

   MoveFiles( info, count );
   MakeDirs( info, count );
   if (!UntarFiles( filename )) {
      DeleteOldFiles( info, count );
      MakeNotesFile( pkg, info, count );
      MakeTimestampFile( pkg );
   }

   return RunPostInstallationScript( pkg );
}

static void PrintInfo( const char *pkg )
{
   FILE *str;
   char buffer[4096];

   printf( "\n%s:\n", pkg );
   
   sprintf( buffer, "%s/%s.Notes", PMSPath, pkg );
   
   if (!(str = fopen( buffer, "r" )))
	 return;

   while (fgets( buffer, 4095, str )) {
      if (!strncmp( buffer, "%%", 2 ))
	    break;
      if (buffer[0] == '%') {
	 switch (buffer[1]) {
	 case 'c': printf( "  Compiled with:   %s", buffer + 3 );  break;
	 case 'l': printf( "  Linked with:     %s", buffer + 3 );  break;
	 case 'n': printf( "  Package name:    %s", buffer + 3 );  break;
	 case 'v': printf( "  Package version: %s", buffer + 3 );  break;
	 case 'b': printf( "  Build by:        %s", buffer + 3 );  break;
	 case 'd': printf( "  Built on:        %s", buffer + 3 );  break;
	 case 'w':
	    printf( "  Build in:        /usr/src/%s", buffer + 3 );
	    break;
	 case 'f':
	    if (buffer[2] != ' ')
		  printf( "  FTP source %d from: %s",
			  atoi( buffer + 2 ),
			  buffer + (buffer[3] == ' ' ? 4 : 5) );
	    else
		  printf( "  FTP source from: %s", buffer + 3 );
	    break;
	 case 't':
	    if (buffer[2] != ' ')
		  printf( "  Source file %d:     %s",
			  atoi( buffer + 2 ),
			  buffer + (buffer[3] == ' ' ? 4 : 5) );
	    else
		  printf( "  Source file:     %s", buffer + 3 );
	    break;
	 case 'p':
	    if (buffer[2] != ' ')
		  printf( "  Patch file %d:      %s",
			  atoi( buffer + 2 ),
			  buffer + (buffer[3] == ' ' ? 4 : 5) );
	    else
		  printf( "  Patch file:      %s", buffer + 3 );
	    break;
	 }
      } else 
	    printf( "  %s", buffer );
   }
   fclose( str );

   sprintf( buffer, "%s/%s.Timestamp", PMSPath, pkg );
   
   if (!(str = fopen( buffer, "r" )))
	 return;

   while (fgets( buffer, 4095, str )) {
      if (!strncmp( buffer, "%%", 2 ))
	    break;
      printf( "   %s", buffer );
   }
   fclose( str );
}

static int CheckInfo( const char *pkg )
{
   FILE *str;
   char buffer[4096];
   int  HaveName    = 0;
   int  HaveVersion = 0;
   int  HaveBuilder = 0;
   int  HaveDate    = 0;
   int  HaveFtp     = 0;
   int  HaveTar     = 0;
   int  HaveWhere   = 0;
   int  RetVal      = 0;

   if (Function == build || Function == tar) {
      /* When doing a build, look in the AltPath first. */
      sprintf( buffer, "%s/%s.Notes", AltPath, pkg );
      if (!(str = fopen( buffer, "r" ))) {
	 fprintf( stderr,
		  "pms: Cannot open notes file for build (\"%s\")\n",
		  buffer );
	 sprintf( buffer, "%s/%s.Notes", PMSPath, pkg );
	 if (!(str = fopen( buffer, "r" ))) {
	    fprintf( stderr,
		     "pms: Cannot open notes file for build (\"%s\")\n",
		     buffer );
	    exit( 2 );
	 } else {
	    fprintf( stderr,
		     "pms: Using alternate notes file for build (\"%s\")\n",
		     buffer );
	 }
      }
   } else {
      sprintf( buffer, "%s/%s.Notes", PMSPath, pkg );
      if (!(str = fopen( buffer, "r" ))) {
	 fprintf( stderr,
		  "pms: Cannot open notes file for build (\"%s\")\n",
		  buffer );
	 sprintf( buffer, "%s/%s.Notes", AltPath, pkg );
	 if (!(str = fopen( buffer, "r" ))) {
	    fprintf( stderr,
		     "pms: Cannot open notes file (\"%s\")\n",
		     buffer );
	    exit( 2 );
	 } else {
	    fprintf( stderr,
		     "pms: Using alternate notes file (\"%s\")\n",
		     buffer );
	 }
      }
   }

   while (fgets( buffer, 4095, str )) {
      if (!strncmp( buffer, "%%", 2 )) break;
      if (buffer[0] == '%') switch (buffer[1]) {
      case 'n':
	 ++HaveName;
	 PackageName = strdup( buffer + 3 );
	 PackageName[ strlen( PackageName ) - 1 ] = '\0';
	 break;
      case 'v':
	 ++HaveVersion;
	 PackageVersion = strdup( buffer + 3 );
	 PackageVersion[ strlen( PackageVersion ) - 1 ] = '\0';
	 break;
      case 'b': ++HaveBuilder;                                        break;
      case 'd': ++HaveDate;                                           break;
      case 'f': ++HaveFtp;                                            break;
      case 't': ++HaveTar;                                            break;
      case 'w': ++HaveWhere;
	 PackageWhere = strdup( buffer + 3 );
	 PackageWhere[ strlen( PackageWhere ) - 1 ] = '\0';
	 break;
      }
   }
   fclose( str );

   if (HaveName > 1 || HaveVersion > 1 || HaveWhere > 1) {
      printf( "Notes file has more than one %%n, %%v, or %%w field\n" );
      ++RetVal;
   }

   if (!HaveName) {
      printf( "Notes file missing package name (%%n)\n" );
      ++RetVal;
   }
   if (!HaveVersion) {
      printf( "Notes file missing pacakge version (%%v)\n" );
      ++RetVal;
   }
   if (!HaveBuilder) {
      printf( "Notes file missing builder's email address (%%b)\n" );
      ++RetVal;
   }
   if (!HaveDate) {
      printf( "Notes file missing date of build (%%b)\n" );
      ++RetVal;
   }
   if (!HaveFtp) {
      printf( "Notes file missing ftp site (%%f)\n" );
      ++RetVal;
   }
   if (!HaveTar) {
      printf( "Notes file missing tar file name (%%t)\n" );
      ++RetVal;
   }
   if (!HaveWhere) {
      printf( "Notes file missing subdirectory (%%w)\n" );
      ++RetVal;
   }

   return RetVal;
}

static int List( void )
{
   glob_t pglob = { 0, NULL, 0, 0 };
   char   buffer[4096];
   int    i;
   
   sprintf( buffer, "%s/*.Notes", PMSPath ); /* */
   
   if (!glob( buffer, 0, NULL, &pglob )) {
      for (i = 0; i < pglob.gl_pathc; i++) {
	 char *pkg = ResolvePackageName( pglob.gl_pathv[i] );
	 
	 PrintInfo( pkg );

	 free( pkg );
      }
   }
   globfree( &pglob );

   return 0;
}

static Info *GetInfo( const char *pkg, int *count )
{
   char buffer[4096];
   FILE *str;
   int  flag = 0;
   Info *info = malloc( sizeof( Info ) );

   *count = 0;

   sprintf( buffer, "%s/%s.Notes", PMSPath, pkg );

   if (!(str = fopen( buffer, "r" ))) {
      fprintf( stderr, "pms: package \"%s\" was never installed!\n", pkg );
      exit( 2 );
   }

   while (fgets( buffer, 4095, str )) {
      if (!strncmp( buffer, "%%", 2 ))
	    ++flag;
      if (flag == 2 && buffer[0] == '*') {
	 char *pt;
	 
	 info[*count].filename = strdup( strtok( buffer, "* " ) );
	 pt = strtok( NULL, " " );
	 if (pt)
	       info[*count].owner = strdup( pt );
	 else
	       info[*count].owner = strdup( "root:system" );
	 pt = strtok( NULL, " " );
	 if (pt)
	       info[*count].mode = strtol( pt, NULL, 8 );
	 else
	       info[*count].mode = 0;
	 info = realloc( info, (++*count + 1) * sizeof( Info ) );
      }
   }
   fclose( str );
   
   return info;
}

static void CopyNotesFile( const char *pkg )
{
   char   new_notes_file[4096];
   char   tmp_notes_file[4096];
   char   old_notes_file[4096];
   char   buffer[4096];
   char   tmp[4096];
   FILE   *old_notes;
   FILE   *tmp_notes;
   int    flag = 0;
   time_t t;
   char   *builder   = getenv( "PMS_BUILDER" );
   char   *libraries = getenv( "PMS_LIBRARIES" );
   char   *compiler  = getenv( "PMS_COMPILER" );
   int    needs_doc  = 0;

   static int printer( const char *file, struct stat *sb, int flag )
	 {
	    struct passwd *pw;
	    struct group  *gr;
	    mode_t        mode;

	    if (sb) {
	       mode = sb->st_mode & 07777;
	       gr = getgrgid( sb->st_gid );
	       pw = getpwuid( sb->st_uid );
	    } else {
	       mode = 0644;
	       gr = getgrgid( 0 );
	       pw = getpwuid( 0 );
	    }

	    switch (flag) {
	    case FTW_F:
	       if (*file == '/') ++file;
	       if (pw && gr) {
		  fprintf( tmp_notes,
			   "* %-40s %8s:%-8s %04o\n",
			   file,
			   pw->pw_name,
			   gr->gr_name,
			   mode );
	       } else {
		  fprintf( tmp_notes,
			   "* %-40s %8s:%-8s %04o\n",
			   file,
			   "root",
			   "system",
			   mode );
	       }
	       break;
	    case FTW_NS:
	    case FTW_DNR:
	       fprintf( stderr,
			"pms: cannot read \"%s%s\"\n",
			RootSlash,
			file );
	       exit( 2 );
	       break;
	    }
	    
	    return 0;
	 }

   static void check_file( const char *tmp ) {
      struct stat sb;
      struct stat sl;
	       
      if (stat( tmp, &sb )) {
	 fprintf( stderr, "pms: cannot stat \"%s\"\n", tmp );
	 exit( 2 );
      }

      if (lstat( tmp, &sl )) {
	 fprintf( stderr, "pms: cannot lstat \"%s\"\n", tmp );
	 exit( 2 );
      }

      if (S_ISDIR( sb.st_mode ) && !S_ISLNK( sl.st_mode )) {
	 if (pms_ftw( tmp, printer, 10 )) {
	    fprintf( stderr, "pms: pms_ftw failed\n" );
	    perror( "pms" );
	    exit( 2 );
	 }
      } else {
	 printer( tmp, &sb, FTW_F );
      }
   }
   
   time( &t );
   
   sprintf( old_notes_file, "%s/%s.Notes", AltPath, pkg );
   if (!(old_notes = fopen( old_notes_file, "r" ))) {
      sprintf( old_notes_file, "%s/%s.Notes", PMSPath, pkg );
      if (!(old_notes = fopen( old_notes_file, "r" ))) {
	 fprintf( stderr, "pms: package \"%s\" has no notes file!\n", pkg );
	 exit( 2 );
      } else {
	 printf( "Sourcing installed notes file (\"%s\")\n", old_notes_file );
      }
   } else {
      printf( "Sourcing pristine notes file (\"%s\")\n", old_notes_file );
   }

   sprintf( new_notes_file, "%s/%s.Notes", PMSPath, pkg );
   sprintf( tmp_notes_file, "%s/%s.Notes.tmp", PMSPath, pkg );

   if (!(tmp_notes = fopen( tmp_notes_file, "w" ))) {
      fprintf( stderr,
	       "pms: cannot open temporary notes file (\"%s\")\n",
	       tmp_notes_file );
      exit( 2 );
   }

   flag = 0;
   while (fgets( buffer, 4095, old_notes )) {
      if (!strncmp( buffer, "%%", 2 )) ++flag;
      if (flag == 1 && !strncmp( buffer, "%doc", 4 )) ++needs_doc;
      if (!flag) {
	 if (builder && !strncmp( buffer, "%b ", 3 )) {
	    fprintf( tmp_notes, "%%b %s\n", builder );
	 } else if (libraries && !strncmp( buffer, "%l ", 3 )) {
	    fprintf( tmp_notes, "%%l %s\n", libraries );
	 } else if (compiler && !strncmp( buffer, "%c ", 3 )) {
	    fprintf( tmp_notes, "%%c %s\n", compiler );
	 } else if (!strncmp( buffer, "%d", 2 )) {
	    fprintf( tmp_notes, "%%d %s", ctime( &t ) );
	 } else {
	    fputs( buffer, tmp_notes );
	 }
      } else if (flag == 2
		 && buffer[0] == '*'
		 && (buffer[1] == ' ' || buffer[1] == '\t')) {

	 char        tmp[4096];
	 char        *pt = strpbrk( buffer + 2, " \n\t" );

	 if (pt) *pt = '\0';
	 
	 sprintf( tmp, "%s%s", RootSlash, buffer + 2 );

	 if (!strpbrk( tmp, "*?" )) {
	    check_file( tmp );
	 } else {
	    glob_t pglob = { 0, NULL, 0, 0 };
	    int    i;

	    if (glob( tmp, 0, NULL, &pglob )) {
	       fprintf( stderr, "pms: no match for \"%s\"\n", tmp );
	       exit( 2 );
	    }

	    for (i = 0; i < pglob.gl_pathc; i++) {
	       check_file( pglob.gl_pathv[i] );
	    }
	 }
	 
      } else
	    fputs( buffer, tmp_notes );
   }

   if (needs_doc) {
      sprintf( tmp,
	       "%susr/doc/%s/%s-%s",
	       RootSlash,
	       PackageWhere,
	       PackageName,
	       PackageVersion );
      check_file( tmp );
   }
   
   sprintf( tmp, "%s/%s.Notes", PMSPath, pkg );
   printer( tmp, NULL, FTW_F );
   
   fclose( tmp_notes );
   fclose( old_notes );

   Rename( tmp_notes_file, new_notes_file, 1 );
}

static int Query( const char *package )
{
   char *pkg = ResolvePackageName( package );
   Info *info;
   int  count;
   int  i;

   PrintInfo( pkg );

   info = GetInfo( pkg, &count );

   for (i = 0; i < count; i++)
	 printf( "%-50s %s %04o\n",
		 info[i].filename,
		 info[i].owner,
		 info[i].mode );

   free( pkg );

   return 0;
}


static int Deinstall( const char *package )
{
   char   *pkg  = ResolvePackageName( package );
   Info   *info;
   int    count;
   glob_t pglob = { 0, NULL, 0, 0 };
   char   buffer[4096];
   int    i;

   info = GetInfo( pkg, &count );

   printf( "*** DEINSTALLATION of %s ***\n", pkg );
   PrintInfo( pkg );
   DeleteFiles( info, count );

   if (!NewPMSPath) {
				/* FIXME! */
      sprintf( buffer, "%s/%s.*", PMSPath, pkg );

      if (!glob( buffer, 0, NULL, &pglob )) {
	 for (i = 0; i < pglob.gl_pathc; i++) {
	    printf( "Deleting %s\n", pglob.gl_pathv[i] );
	    Unlink( pglob.gl_pathv[i] );
	 }
      }
   }
   
   globfree( &pglob );

   return 0;
}

static int Fixperms( const char *package )
{
   char          *pkg = ResolvePackageName( package );
   Info          *info;
   int           count;
   struct passwd *pw;
   struct group  *gr;
   char          buffer[4096];
   int           i;

   info = GetInfo( pkg, &count );

   printf( "Fixing permissions for:\n" );
   PrintInfo( pkg );
   
   for (i = 0; i < count; i++) {
				/* This should be optimized */
      pw = getpwnam( strtok( info[i].owner, ":" ) );
      gr = getgrnam( strtok( NULL, ":" ) );
      sprintf( buffer, "%s%s", RootSlash, info[i].filename );
      Chown( buffer, pw->pw_uid, gr->gr_gid );
      Chmod( buffer, info[i].mode );
   }

   return 0;
}

static int Tar( const char *package )
{
   char *pkg = ResolvePackageName( package );
   Info *info;
   int  count;
   FILE *str;
   char *tmp_file;
   char buffer[4096];
   int  i;
   int  have_notes = 0;
   int  retval;

   if (CheckInfo( pkg )) {
      fprintf( stderr, "pms: notes file is missing required information\n" );
      exit( 2 );
   }
   CopyNotesFile( pkg );
   
   info = GetInfo( pkg, &count );

   printf( "Making tar distribution for:\n" );
   PrintInfo( pkg );

   tmp_file = tempnam( "/tmp", "PMS" );
   if (!(str = fopen( tmp_file, "w" ))) {
      fprintf( stderr, "pms: cannot open temporary file \"%s\"\n", tmp_file );
      exit( 2 );
   }
   
   for (i = 0; i < count; i++) {
      char *tmp = info[i].filename;
      int  len  = strlen( tmp );
      if (tmp[ strlen( tmp ) - 1 ] != '/')
	    fprintf( str, "%s\n", tmp );
      if (len > 6 && !strcmp( tmp + len - 6, ".Notes" )) ++have_notes;
   }

   if (!have_notes) {
				/* Always include the .Notes file */
      sprintf( buffer, "%s/%s.Notes", PMSPath, pkg );
      if (buffer[0] == '/')
	    fputs( buffer + 1, str );
      else
	    fputs( buffer, str );
   }
      
   fclose( str );

   if (chdir( Root )) {
      fprintf( stderr, "pms: cannot change directory to \"%s\"\n", Root );
      exit( 2 );
   }

   if (TarPath[ strlen( TarPath ) - 1 ] == '/')
	 sprintf( buffer, "tar zScvvpTf %s %s%s.tar.gz",
		  tmp_file, TarPath, pkg );
   else
	 sprintf( buffer, "tar zScvvpTf %s %s/%s.tar.gz",
		  tmp_file, TarPath, pkg );

   retval = System( buffer );

   Unlink( tmp_file );

   return retval;
}

static int Build( const char *package )
{
   char   *pkg = ResolvePackageName( package );
   FILE   *str;
   char   buffer[4095];
   int    flag = 0;
   FILE   *strOut;
   char   *tmp_file;
   time_t t;
   int    retval;
   char   name[1024]    = { 0, };
   char   ver[1024]     = { 0, };
   char   ftpdir[1024]  = { 0, };
   char   tarfile[1024] = { 0, };
   char   where[1024]   = { 0, };
   char   patch[1024]   = { 0, };
   int    need_chown    = 0;
   int    have_doc      = 0;

   time( &t );
   MakeSourcePath();

   printf( "Building \"%s\" from source. . .\n", pkg );
   if (CheckInfo( pkg )) {
      fprintf( stderr, "pms: notes file is missing required information\n" );
      exit( 2 );
   }

   sprintf( buffer, "%s/%s.Notes", AltPath, pkg );
   
   if (Function == build) {
      if (!(str = fopen( buffer, "r" ))) {
	 sprintf( buffer, "%s/%s.Notes", PMSPath, pkg );
	 if (!(str = fopen( buffer, "r" )))
	       return -1;
	 else
	       printf( "Sourcing installed notes file (\"%s\")\n",
		       buffer );
      } else
	    printf( "Sourcing pristine notes file (\"%s\")\n",
		    buffer );
   } else {
      fprintf( stderr,
	       "pms (internal error): in Build(), but Function != build\n" );
      exit( 3 );
   }

   tmp_file = tempnam( "/tmp", "PMS" );
      if (!(strOut = fopen( tmp_file, "w" ))) {
      fprintf( stderr, "pms: cannot open temporary file \"%s\"\n", tmp_file );
      exit( 2 );
   }

   fprintf( strOut, "#!/bin/sh -ex\n" );
   fprintf( strOut, "# This is an automatically generated script\n" );
   fprintf( strOut, "# Generated by pms %s, on %s", version(), ctime( &t ) );
   fprintf( strOut, "# Build script for %s\n", pkg );
   fprintf( strOut, "\n" );
   fprintf( strOut, "# Set up root's path\n" );
   fprintf( strOut,
	    "export PATH=.:/sbin:/bin:/usr/sbin:/usr/bin:/usr/bin/X11"
	    ":$PATH\n" );
   fprintf( strOut, "\n" );
   fprintf( strOut, "# Environment variables (automatically generated):\n" );
   fprintf( strOut, "\n" );

   fprintf( strOut, "export ROOTDIR=%s\n",     Root );
   fprintf( strOut, "export SOURCEDIR=%s\n",   SourcePath );
   fprintf( strOut, "export NOTESDIR=%s\n",    PMSPath );
   fprintf( strOut, "export ALTNOTESDIR=%s\n", AltPath );
   fprintf( strOut, "export BINTARDIR=%s\n",   TarPath );

   while (fgets( buffer, 4095, str )) {
				/* Build script */
      buffer[ strlen( buffer ) - 1 ] = '\0'; /* kill newline */
      if (flag) {
	 if (!strncmp( buffer, "%%", 2 )) break;
	 if (!strncmp( buffer, "exit", 4 )) break; /* don't copy patches */
	 if (!strncmp( buffer, "%setup", 6 )) {
	    fprintf( strOut, "\n" );
	    fprintf( strOut, "# Standard \"setup\" commands:\n" );
	    fprintf( strOut,
		     "if [ ! -d $BUILDDIR ];\n"
		     "   then mkdir $BUILDDIR;\n"
		     "fi\n" );
	    fprintf( strOut, "cd $BUILDDIR\n" );
	    fprintf( strOut, "rm -rf $NAME-$VERSION\n" );
	    fprintf( strOut, "tar zxvf $SRCTARDIR/$TARFILE\n" );
	    fprintf( strOut, "cd $NAME-$VERSION\n" );
	    fprintf( strOut, "\n" );
	 } else if (!strncmp( buffer, "%doc", 4 )) {
	    if (BuildInstall || BuildAll) {
	       fprintf( strOut, "\n" );
	       if (!have_doc) {
		  ++have_doc;
		  fprintf( strOut, "# Standard \"doc\" commands:\n" );
		  fprintf( strOut,
			   "if [ ! -d $ROOTDIR/usr/doc ];\n"
			   "   then mkdir $ROOTDIR/usr/doc;\n"
			   "fi\n" );
		  fprintf( strOut,
			   "if [ ! -d $ROOTDIR/usr/doc/$WHERE ];\n"
			   "   then mkdir $ROOTDIR/usr/doc/$WHERE;\n"
			   "fi\n" );
		  fprintf( strOut,
			   "rm -rf $ROOTDIR/usr/doc/$WHERE/$NAME-$VERSION\n" );
		  fprintf( strOut,
			   "mkdir $ROOTDIR/usr/doc/$WHERE/$NAME-$VERSION\n" );
	       }
	       if (buffer[5]) {
		  fprintf( strOut,
			   "cp -p %s $ROOTDIR/usr/doc/$WHERE/$NAME-$VERSION\n",
			   buffer + 5 );
		  ++need_chown;
	       }
	       fprintf( strOut, "\n" );
	    }
	 } else if (buffer[0] != '%'
		    || buffer[1] != 'i'
		    || (buffer[2] != ' ' && buffer[2] != '\t')) {
	    if (buffer[0] == '*' && (buffer[1] == ' ' || buffer[1] == '\t')) {
	       if (BuildInstall || BuildAll) {
		  fprintf( strOut, "%s\n", buffer + 2 );
	       }
	    } else if (buffer[0] == '*'
		       && buffer[1] == '*'
		       && (buffer[2] == ' ' || buffer[2] == '\t')) {
	       if (BuildAll) {
		  fprintf( strOut, "%s\n", buffer + 3 );
	       }
	    } else {
	       fprintf( strOut, "%s\n", buffer );
	    }
	 }
      } else {
	 if (!strncmp( buffer, "%%", 2 )) {
	    flag = 1;
	    fprintf( strOut, "\n" );
	    fprintf( strOut, "# Installation commands (from Notes file):\n" );
	    fprintf( strOut, "\n" );
	 } else if (buffer[0] == '%') {
	    switch (buffer[1]) {
	    case 'n':
	       strcpy( name, buffer + 3 );
	       fprintf( strOut, "export NAME=%s\n", name );
	       break;
	    case 'v': 
	       strcpy( ver, buffer + 3 );
	       fprintf( strOut, "export VERSION=%s\n", ver );
	       fprintf( strOut,
			"export NOTESFILE=%s/%s-%s.bin.Notes\n",
			PMSPath,
			name,
			ver );
	       if (TarPath[ strlen( TarPath ) - 1 ] == '/')
		     fprintf( strOut,
			      "export BINTARFILE=%s%s-%s.bin.tar.gz\n",
			      TarPath,
			      name,
			      ver );
	       else	
		     fprintf( strOut,
			      "export BINTARFILE=%s/%s-%s.bin.tar.gz\n",
			      TarPath,
			      name,
			      ver );
	       break;
	    case 'f':
	       if (buffer[2] == ' ') {
		  strcpy( ftpdir, buffer + 3 );
		  fprintf( strOut, "export FTPDIR=%s\n", ftpdir );
	       } else {
		  strcpy( ftpdir, buffer + (buffer[3] == ' ' ? 4 : 5) );
		  fprintf( strOut,
			   "export FTPDIR%d=%s\n",
			   atoi( buffer + 2 ),
			   ftpdir );
	       }
	       break;
	    case 't':
	       if (buffer[2] == ' ') {
		  strcpy( tarfile, buffer + 3 );
		  fprintf( strOut, "export TARFILE=%s\n", tarfile );
		  fprintf( strOut,
			   "export FTPTARFILE=%s/%s\n",
			   ftpdir,
			   tarfile );
	       } else {
		  strcpy( tarfile, buffer + (buffer[3] == ' ' ? 4 : 5) );
		  fprintf( strOut,
			   "export TARFILE%d=%s\n",
			   atoi( buffer + 2 ),
			   tarfile );
		  fprintf( strOut,
			   "export FTPTARFILE%d=%s/%s\n",
			   atoi( buffer + 2 ),
			   ftpdir,
			   tarfile );
	       }
	       break;
	    case 'w':
	       strcpy( where, buffer + 3 );
	       fprintf( strOut, "export WHERE=%s\n", where );
	       fprintf( strOut, "export BUILDDIR=%s/%s\n", SourcePath, where );
	       fprintf( strOut,
			"export SRCTARDIR=%s/%sTAR_FILES/%s\n",
			SourcePath,
			PrivateTarFiles ? "PRIVATE_" : "",
			where );
	       break;
	    case 'p':
	       if (buffer[2] == ' ') {
		  strcpy( patch, buffer + 3 );
		  fprintf( strOut, "export PATCHFILE=%s\n", patch );
	       } else {
		  strcpy( patch, buffer + (buffer[3] == ' ' ? 4 : 5) );
		  fprintf( strOut,
			   "export PATCHFILE%d=%s\n",
			   atoi( buffer + 2 ),
			   patch );
	       }
	       break;
	    }
	 }
      }
   }
   if (!flag) {
      fprintf( stderr, "pms: could not find shell script in Notes file\n" );
      exit( 2 );
   }

   if (need_chown) {
      fprintf( strOut,
	       "chown -R `id -un`:`id -gn`"
	       " $ROOTDIR/usr/doc/$WHERE/$NAME-$VERSION\n" );
      fprintf( strOut,
	       "chmod -R g-w,o-w,a+rX"
	       " $ROOTDIR/usr/doc/$WHERE/$NAME-$VERSION\n" );
   }
   
   if (BuildAll) {
      fprintf( strOut, "\n" );
      fprintf( strOut, "# Packaging command (automatically generated):\n" );
      fprintf( strOut, "\n" );
      fprintf( strOut,
	       "%s -t %s -R %s -T %s -P %s -S %s\n",
	       MyPath,
	       package,
	       Root,
	       TarPath,
	       PMSPath,
	       SourcePath );
   }

   fprintf( strOut, "\n" );
   fprintf( strOut, "# End of commands, exit:\n" );
   fprintf( strOut, "exit 0\n" );

   fclose( strOut );
   fclose( str );

   Chmod( tmp_file, 0700 );
   printf( "Executing shell script:\n\n" );
   retval = System( tmp_file );
   Unlink( tmp_file );

   printf( "Build complete.\n" );
   if (!retval) PrintInfo( pkg );
   
   return retval;
}

static void usage( void )
{
   fprintf( stderr,
	    "This is pms %s: Copyright 1993,1994 Rickard E. Faith\n"
	    "usage: pms [-i package]        (install)\n"
	    "           [-l]                (list installed)\n"
	    "           [-q package]        (query)\n"
	    "           [-d package]        (deinstall)\n"
	    "           [-f package]        (fixperms)\n"
	    "           [-t package]        (make tar file)\n"
	    "           [-R root]           (change root)\n"
	    "           [-T path]           (tar path)\n"
	    "           [-P path]           (pms path)\n"
	    "           [-S path]           (source path)\n"
	    "           [-B opt package]    (build)\n"
	    "           [-p]                (use private tar files)\n"
	    "           [-v]                (verbose installation script)\n",
	    version() );
   exit( 1 );
}

int main( int argc, char **argv )
{
   int            c;
   char           *package = "";
   int            retval   = 0;
   time_t         start, stop;
   char           *tmp;

   time( &start );

   if (argv[0][0] == '.') {
      char *cwd = get_current_dir_name();
      MyPath = malloc( strlen( cwd ) + strlen( argv[0] ) );
      if (argv[1])
	    sprintf( MyPath, "%s%s", cwd, argv[0] + 1 );
      else
	    sprintf( MyPath, "%s", cwd );
      free( cwd );
   } else
	 MyPath = argv[0];

   tmp = getenv( "PMS_VERBOSE" );
   if (tmp) ++Verbose;
   tmp = getenv( "ROOTDIR" );
   if (tmp) Root = strdup( tmp );
   tmp = getenv( "BINTARDIR" );
   if (tmp) TarPath = strdup( tmp );
   tmp = getenv( "NOTESDIR" );
   if (tmp) PMSPath = strdup( tmp );
   
   while ((c = getopt( argc, argv, "vi:lq:d:f:t:pDnR:T:P:S:B:" )) != EOF)
	 switch (c) {
	 case 'D': ++Debug;                                break;
	 case 'n': ++NoAutomatic;                          break;
	 case 'i': package = optarg; Function = install;   break;
	 case 'l':                   Function = list;      break;
	 case 'q': package = optarg; Function = query;     break;
	 case 'd': package = optarg; Function = deinstall; break;
	 case 'f': package = optarg; Function = fixperms;  break;
	 case 't': package = optarg; Function = tar;       break;
	 case 'R': Root    = optarg;                       break;
	 case 'T': TarPath = optarg;                       break;
	 case 'P': PMSPath = optarg;                       break;
	 case 'S': SourcePath = optarg;                    break;
	 case 'p': ++PrivateTarFiles;                      break;
	 case 'v': ++Verbose; putenv( "PMS_VERBOSE=1" );   break;
	 case 'B':
	    Function = build;
	    if (optarg[0] == 'c')      BuildCompile = 1;
	    else if (optarg[0] == 'i') BuildInstall = 1;
	    else if (optarg[0] == 'a') BuildAll     = 1;
	    else usage();
	    break;
	 default:  usage();                                break;
	 }

   if (Function == build) {
      if (argc - optind != 1) usage();
      package = argv[optind];
   } else if (argc - optind) usage();

   RootSlash = malloc( strlen( Root ) + 2 );
   strcpy( RootSlash, Root );
   if (RootSlash[ strlen( RootSlash ) - 1 ] != '/')
	 strcat( RootSlash, "/" );
   if (PMSPath) {
      NewPMSPath = 1;
   } else {
      PMSPath = malloc( strlen( RootSlash ) + sizeof( DEFAULT_PMSPATH ) );
      sprintf( PMSPath, "%s%s", RootSlash, DEFAULT_PMSPATH );
   }
   
   if (SourcePath) {
      NewSourcePath = 1;
   } else {
      SourcePath = getenv( "SOURCEDIR" );
      if (SourcePath) {
	 NewSourcePath = 1;
      } else {
	 SourcePath = malloc( strlen( RootSlash )
			      + sizeof( DEFAULT_SOURCEPATH ) );
	 sprintf( SourcePath, "%s%s", RootSlash, DEFAULT_SOURCEPATH );
      }
   }

   AltPath = malloc( strlen( SourcePath )
		     + sizeof( ALTERNATE_NOTES_NAME ) + 2 );
   sprintf( AltPath, "%s/%s", SourcePath, ALTERNATE_NOTES_NAME );
		     
   if (TarPath[0] == '.') {
      char *cwd = get_current_dir_name();
      char *tmp = strdup( TarPath );
      TarPath = malloc( strlen( cwd ) + strlen( TarPath ) );
      if (tmp[1])
	    sprintf( TarPath, "%s%s", cwd, tmp + 1 );
      else
	    sprintf( TarPath, "%s", cwd );
      free( tmp );
      free( cwd );
   } else
	 MyPath = argv[0];
   
   if (Debug) {
      printf( "Function = " );
      switch (Function) {
      case install:   printf( "install" );    break;
      case list:      printf( "list" );       break;
      case query:     printf( "query" );      break;
      case deinstall: printf( "desinstall" ); break;
      case fixperms:  printf( "fixperms" );   break;
      case tar:       printf( "tar" );        break;
      case none:      printf( "none" );       break;
      case build:
	 printf( "build" );
	 if (BuildCompile) printf( " *compile*" );
	 if (BuildInstall) printf( " *install*" );
	 if (BuildAll)     printf( " *all*" );
	 break;
      }
      printf( "\n" );
      printf( "Package     = %s\n", package );
      printf( "Root        = %s\n", Root );
      printf( "PMSTARDIR   = %s\n", TarPath );
      printf( "PMSNOTESDIR = %s\n", PMSPath );
      printf( "ALTNOTESDIR = %s\n", AltPath );
      printf( "SOURCEDIR   = %s\n", SourcePath );
   }
   
   switch (Function)  {
   case install:   retval = Install( package );   break;
   case list:      retval = List();               break;
   case query:     retval = Query( package );     break;
   case deinstall: retval = Deinstall( package ); break;
   case fixperms:  retval = Fixperms( package );  break;
   case tar:       retval = Tar( package );       break;
   case build:     retval = Build( package );     break;
   default:        usage();                       break;
   }

   time( &stop );

   if (Function == build) {
      long int elapsed = stop - start;
      long int hours, minutes, seconds;
      
      hours   =  elapsed / 3600L;
      elapsed -= 3600L * hours;
      minutes =  elapsed / 60L;
      elapsed -= 60L * minutes;
      seconds =  elapsed;

      if (retval) {
	 printf( "\nBuild time: %02ld:%02ld:%02ld (for %s, status = %d) ***\n",
		 hours,
		 minutes,
		 seconds,
		 ResolvePackageName( package ),
		 retval);
      } else {
	 printf( "\nBuild time: %02ld:%02ld:%02ld (for %s)\n",
		 hours,
		 minutes,
		 seconds,
		 ResolvePackageName( package ) );
      }
   }

   if (NeedsLdconfig) {
      printf( "******************************************************\n"
              "WARNING: Important system libraries have been renamed.\n"
	      "         You must run ldconfig now!\n"
	      "******************************************************\n" );
   }

				/* Can't return full range, and retval for
                                   some erros is 32256, which gets returned
                                   as 0 */

   if (retval) return 1;
   return 0;
}
