/***************************************************************************
 * LPRng - An Extended Print Spooler System
 *
 * Copyright 1988-1995 Patrick Powell, San Diego State University
 *     papowell@sdsu.edu
 * See LICENSE for conditions of use.
 *
 ***************************************************************************
 * MODULE: jobcontrol.c
 * PURPOSE: read and write the spool queue control file
 **************************************************************************/

static char *const _id =
"$Id: jobcontrol.c,v 3.0 1996/05/19 04:06:03 papowell Exp $";

#include "lp.h"
#include "lp_config.h"
#include "printcap.h"

/***************************************************************************
Get_job_control( struct control_file )
	Get the job control file from the spool directory and decode the information
	in it. The control file has the lines:
		hold <time>
		priority <time>
 ***************************************************************************/

#define HOLD     1
#define PRIORITY 2
#define REMOVE   3
#define ACTIVE   4
#define MOVE     5
#define REDIRECT 6
#define ERROR    7

#define PAIR(X) { #X, INTEGER_K, (void *)0, X }

static struct keywords status_key[] = {
{ "hold", INTEGER_K, (void *)0, HOLD },
{ "priority", INTEGER_K, (void *)0, PRIORITY },
{ "remove", INTEGER_K, (void *)0, REMOVE },
{ "active", INTEGER_K, (void *)0, ACTIVE },
{ "move", INTEGER_K, (void *)0, MOVE },
{ "redirect", STRING_K, (void *)0, REDIRECT },
{ "error", STRING_K, (void *)0, ERROR },
	{ 0 },
};

static char *Hold_file_pathname( struct control_file *cf );

int Get_job_control( struct control_file *cf )
{
	/* now we parse the file */
	struct stat statb;
	char buffer[SMALLBUFFER];
	long value;
	char *s, *t, *end;
	int i, fd, len;
	
	cf->priority_time = 0;
	cf->hold_time = 0;

	s = Hold_file_pathname( cf );
	fd = Checkread( s, &statb );
	DEBUG5("Get_job_control: file '%s', fd %d, Auto_hold %d", s, fd, Auto_hold);
	if( fd < 0 && Auto_hold ){
		cf->hold_time = time( (void *)0 );
	}
	/* if no change, forget it */
	if( fd < 0 || (cf->hstatb.st_ctime == statb.st_ctime) ){
		close( fd );
		return( 0 );
	}

	/* get the current fd status */
	cf->hstatb = statb;

	buffer[0] = 0;
	/* get the values from the file */
	if( fd >= 0 && (len = statb.st_size) != 0 ){
		if( len >= sizeof(buffer) ){
			len = sizeof(buffer) - 1;
		}
		for( s = buffer;
			len > 0 && (i = read( fd, s, len)) > 0;
			len -= i, s += i );
		*s = 0;
	}
	close(fd);

	DEBUG9("Get_job_control: '%s'", buffer );
	for( s = buffer; s && *s; s = end ){
		end = strchr( s, '\n' );
		if( end ){
			*end++ = 0;
		}
		/* we check for the names now */
		if( (t = strchr( s, ' ')) ) *t++ = 0;
		/* skip white space */
		if( t ) while( isspace( *t ) ) ++t;
		DEBUG9("Get_job_control: '%s' '%s'", s,t );
		for( i = 0;
			status_key[i].keyword
				&& strcasecmp( status_key[i].keyword, s);
			++i );
		if( status_key[i].keyword ){
			value = 0;
			if( t ) value = strtol( t, (void *)0, 0 );
			DEBUG9("Get_job_control: found '%s' '%s' value %d",
				s, t, value );

			switch( status_key[i].maxval ){
			case HOLD: cf->hold_time = value; break;
			case PRIORITY: cf->priority_time = value; break;
			case REMOVE: cf->remove_time = value; break;
			case MOVE: cf->move_time = value; break;
			case ACTIVE:
				cf->active = value;
				if( cf->active && kill( cf->active, 0 ) == -1 ){
					cf->active = 0;
				}
				break;
			case REDIRECT:
				if( t && *t ){
					strncpy( cf->redirect, t, sizeof( cf->redirect) );
				} else {
					cf->redirect[0] = 0;
				}
				break;
			case ERROR:
				if( t && *t && cf->error[0] == 0 ){
					strncpy( cf->error, t, sizeof( cf->error) );
				}
				break;
			default: break;
			}
		}
	}
	DEBUG4("Get_job_control: hold 0x%x, priority 0x%x, remove 0x%x",
		cf->hold_time, cf->priority_time, cf->remove_time );
	return( 1 );
}

int Set_job_control( struct control_file *cf )
{
	struct stat statb;
	char buffer[SMALLBUFFER];
	char *s, *t;
	int lock, create;
	int i, value, fd, len;
	
	value = 0;
	buffer[0] = 0;
	buffer[sizeof(buffer)-1] = 0;
	for( i = 0; status_key[i].keyword ; ++i ){
		len = strlen(buffer);
		s = buffer+len;
		len = sizeof( buffer ) - len - 1 ;
		t = 0;
		switch( status_key[i].maxval ){
		case HOLD:		value = cf->hold_time; break;
		case PRIORITY:	value = cf->priority_time; break;
		case REMOVE:	value = cf->remove_time; break;
		case ACTIVE:	value = cf->active; break;
		case MOVE:		value = cf->move_time; break;
		case REDIRECT:	t = cf->redirect; break;
		case ERROR:		t = cf->error; break;
		default: break;
		}
		if( t == 0 ){
			plp_snprintf( s, len, "%s %d\n",
				status_key[i].keyword, value );
		} else if( *t ) {
			plp_snprintf( s, len, "%s %s\n",
				status_key[i].keyword, t );
		}
	}

	s = Hold_file_pathname( cf );
	DEBUG8("Set_job_control: file '%s', '%s'",s, buffer );
	fd = Lockf( s, &lock, &create, &statb );
	if( fd < 0 ){
		logerr( LOG_ERR,
			"Set_job_control: cannot create hold file '%s'",s);
		return( 1 );
	}
	if( lock == 0 ){
		DEBUG8("Set_job_control: waiting for lock" );
		/* we want to lock the file */
		lock = Do_lock( fd, s, 1 );
	}
	if( lock <= 0 ){
		Errorcode = JABORT;
		logerr_die( LOG_ERR,
			"Set_job_control: cannot lock file '%s'",s);
	}
	if( ftruncate( fd, 0 ) < 0 ){
		Errorcode = JABORT;
		logerr_die( LOG_ERR,
			"Set_job_control: cannot truncate hold file '%s'",s);
	}
	if( Write_fd_str( fd, buffer ) < 0 ){
		Errorcode = JABORT;
		logerr_die( LOG_ERR,
			"Set_job_control: cannot write file '%s'",s);
	}
	close( fd );
	return( 0 );
}

/***************************************************************************
 * int Remove_job_control( struct control_file *cf )
 *  unlink the job control file;
 *  return 1 if success, 0 if failed
 ***************************************************************************/
int Remove_job_control( struct control_file *cf )
{
	struct stat statb;
	char *s;

	s = Hold_file_pathname( cf );
	/* remove it and then check it was removed */
	unlink(s);
	return( stat( s, &statb ) != -1 );
}


/***************************************************************************
 * char *Hold_file_pathname( struct control_file *cf )
 *  get the hold file name for the job and put it in
 *  the control file.
 ***************************************************************************/
char *Hold_file_pathname( struct control_file *cf )
{
	struct dpathname dpath;
	char *s;
	/*
	 * get the hold file pathname
	 */
	if( cf->hold_file == 0 ){
		dpath = *CDpathname;
		s = Add_path( &dpath, cf->name );
		s[dpath.pathlen] = 'h';
		cf->hold_file = add_buffer( &cf->control_file, strlen(s)+1 );
		strcpy( cf->hold_file, s );
	}
	DEBUG7("Hold_file_pathname: '%s' -> '%s'", cf->name, cf->hold_file );
	return( cf->hold_file );
}
