/***************************************************************************
 * LPRng - An Extended Print Spooler System
 *
 * Copyright 1988-1995 Patrick Powell, San Diego State University
 *     papowell@sdsu.edu
 * See LICENSE for conditions of use.
 *
 ***************************************************************************
 * MODULE: sendstatus.c
 * PURPOSE: Send a status request to the remote host
 *
 **************************************************************************/

static char *const _id =
"$Id: sendstatus.c,v 3.0 1996/05/19 04:06:10 papowell Exp $";

#include "lp.h"


/***************************************************************************
Commentary:
The protocol used to send a status request to a remote host consists of the
following:

Client                                   Server
- short format
\3printername option option\n - request
                                         status\n
										....
                                         status\n
										<close connection>

- long format
\4printername option option\n - request


The requestor:
1.  makes a connection (connection timeout)
2.  sends the \3printername options line
3.  reads from the connection until nothing left

void Send_statusrequest( char *printer,	* name of printer *
	char *host,					* name of remote host *
	int format,					* long format == 1, short == 0 *
	char **options,				* options to send *
	int connect_timeout,		* timeout on connection *
	int transfer_timeout,		* timeout on transfer *
	int output )				* output fd

 **************************************************************************/


/***************************************************************************
 * Send_statusrequest
 * 1. Open connection to remote host
 * 2. Send a line of the form:
 *     - short status  \3Printer <options>
 *     - long status   \4Printer <options>
 * 3. Read from the connection until closed and echo on fd 1
 ***************************************************************************/

#define MAX_SHORT_STATUS 8
void Send_statusrequest( char *printer,	/* name of printer */
	char *host,					/* name of remote host */
	int format,					/* long format == 1, long long = 2, short == 0 */
	char **options,				/* options to send */
	int connect_timeout,		/* timeout on connection */
	int transfer_timeout,		/* timeout on transfer */
	int output )				/* output fd */
{
	char *s, *end;
	int i, status, wstatus;
	int sock;		/* socket to use */
	char line[LINEBUFFER];
	char statusline[256];
	char msg[LINEBUFFER];
	char buffers[MAX_SHORT_STATUS][256];	/* status lines */
	char line_header[64];
	int header_len;
	int cnt, next;
	int sendformat;
	int err;
	int max_count;

	DEBUG4("Send_statusrequest: connect_timeout %d, transfer_timeout %d",
			connect_timeout, transfer_timeout );
	sock = Link_open( host, 0, connect_timeout );
	err = errno;
	if( sock < 0 ){
		plp_snprintf( line, sizeof(line)-2,
			"cannot open connection to `%s@%s' - %s",
			printer, host, Errormsg(err) );
		s = line;
		goto error;
	}
	max_count = 0;
	if( format ){
		max_count = (1 << (format-1));
		if( max_count > MAX_SHORT_STATUS ){
			max_count = 0;
		}
	}

	/* now format the option line */
	line[0] = 0;
	safestrncat( line, printer );
	for( ; options && (s = *options); ++options ){
		safestrncat( line, " " );
		safestrncat( line, s );
	}
	if( strlen( line ) >= sizeof( line ) - 1 ){
		s = "too many options or options too long";
		goto error;
	}
	/* long format = REQ_DLONG, short = REQ_DSHORT */
	sendformat =  (format)?REQ_DLONG:REQ_DSHORT;
	DEBUG4("Send_statusrequest: sending '\\%d'%s'", format, line );
	/* open a connection */
	status = Link_send( host, &sock, transfer_timeout, sendformat, line, '\n', 0 );
	DEBUG4("Send_statusrequest: Interactive %d, format %d, max_count %d",
		Interactive, format, max_count );
	/* report the status */
	if( status == 0 ){
		if( Interactive == 0 || format == 0 || max_count == 0){
			/* short status or all status */
			while( status == 0 ){
				i = 64*1024;
				status = Link_file_read( host, &sock, transfer_timeout,
					transfer_timeout, output, &i, (void *)0 );
			}
		} else {
			/* long status - trim lines */
			DEBUG4("Send_statusrequest: long status" );
			cnt = next = header_len = 0;
			wstatus = 0;
			s = statusline;
			s[0] = 0;
			statusline[sizeof(statusline)-1] = 0;
			while( wstatus == 0 && status == 0 ){
				DEBUG4("Send_statusrequest: left '%s'", statusline );
				s = &statusline[strlen(statusline)];
				i = sizeof(statusline) - (s - statusline) - 1;
				if( i <= 0 ){
					s = "status line too long!";
					goto error;
				}
				status = Link_read( host, &sock, transfer_timeout,
					s, &i );
				/* check to see if we have a status line */
				s[i] = 0;
				DEBUG4("Send_statusrequest: got '%s'", s );
				/* now we have to split line up */
				for( s = statusline; s && *s; s = end ){
					end = strchr( s, '\n' );
					if( end ){
						*end++ = 0;
					} else {
						/* copy uncompleted line to start of line */
						strncpy( statusline, s, sizeof(statusline) );
						s = statusline+strlen(statusline);
						break;
					}
					DEBUG4("Send_statusrequest: line found '%s'", s );
					if( header_len == 0
						|| strncmp( s, line_header, header_len ) ){
						/* back up to the start */
						DEBUG4("Send_statusrequest: dumping cnt %d, next %d",
							cnt, next );
						next = next - cnt;
						if( next < 0 ) next += MAX_SHORT_STATUS;
						wstatus = 0;
						for( i = 0; wstatus == 0 && i < cnt; ++i ){
							wstatus = Link_send( host, &output, 
								transfer_timeout, 0, buffers[next], '\n', 0 );
							next = (next+1) % MAX_SHORT_STATUS;
						}
						cnt = next = 0;
					}
					/* copy to the next currently available line */
					DEBUG4("Send_statusrequest: adding cnt %d, next %d",
						cnt, next );
					strncpy( buffers[next], s, sizeof(buffers[0]) );
					next = (next+1) % MAX_SHORT_STATUS;
					if( cnt == 0 ){
						char *t;
						int l;
						/* find the header */
						line_header[0] = 0;
						header_len = 0;
						if( (t = strchr( s, ':' )) &&
							(l = (t - s)) < sizeof(line_header)-1 ){
							strncpy( line_header, s, l );
							line_header[l] = 0;
							header_len = l;
						}
					}
					if( cnt < max_count ) ++cnt;
				}
			}
			if( wstatus == 0 && cnt ){
				/* back up to the start */
				DEBUG4("Send_statusrequest: dumping cnt %d, next %d",
					cnt, next );
				next = next - cnt;
				if( next < 0 ) next += MAX_SHORT_STATUS;
				wstatus = 0;
				for( i = 0; wstatus == 0 && i < cnt; ++i ){
					wstatus = Link_send( host, &output, 
						transfer_timeout, 0, buffers[next], '\n', 0 );
					next = (next+1) % MAX_SHORT_STATUS;
				}
			}
		}
	}
	Link_close( &sock );
	return;

error:
	plp_snprintf( msg,sizeof(msg), "Host '%s' - %s\n",
		host, s );
	if( Write_fd_str( output, msg ) < 0 ){
		cleanup(0);
		exit(-2);
	}
	return;
}
