//		Copyright (c) 1999 by
//		Advanced Visual Systems Inc.
//		All Rights Reserved
//
//	This software comprises unpublished confidential information of
//	Advanced Visual Systems Inc. and may not be used, copied or made
//	available to anyone, except in accordance with the license
//	under which it is furnished.
//
//      This file is under Perforce control
//      $Id: //depot/express/fcs70/dbil/dbil_attr.hxx#1 $
//

// file: dbil_attr.hxx
// author: Patrick Li, Advanced Visual Systems
// purpose: Define class for DBIL Attribute
//
// This class will be used to store 'attributes' or 'columns' of data of type int, double, float,
//  string, or any other primitive data type supported by standard RDBMS's. This class will be used
//  to store a single attribute, or an array of attributes. The class can be used by itself, or it
//  will be frequently used to store attributes as part of a specific shape.
//
#ifndef DBIL_ATTR_DEFINED
#define DBIL_ATTR_DEFINED 1

#include <avs/dbil.h>

// Define DBIL_Attribute class
//
class DBIL_Attribute {
public:
   // Constructors
   //
   DBIL_Attribute();
   DBIL_Attribute( const char *name );
   DBIL_Attribute( const DBIL_Attribute &attr );
   DBIL_Attribute &operator=(const DBIL_Attribute &);

   // Destructors
   //
   virtual ~DBIL_Attribute();

   // Free all memory currently allocated for data, and set size to 0
   //
   void Reset();

   // Set and get attributes name
   //
   DBIL_Status SetName( const char *name );
   const char *GetName() const { return m_name; }

   // Set and get attributes type
   //
   DBIL_Status SetType( const DBIL_DATA_TYPES &type );
   const DBIL_DATA_TYPES &GetType() const { return m_type; }

   // Set and get attributes size 
   //   The size is the number of elements the attribute can hold
   //   (i.e. the array size). The set methods (Allocate and Reallocate)
   //   set the size parameter and attempt to allocate space for the
   //   requested number of elements. Obviously type must be set before
   //   calling this routine...
   //
   int GetSize() const { return m_size; }
   int GetLength(int index) const {return m_len?m_len[index]:m_elen;}
   void *Allocate( int size, int len = 0 );
   void *Reallocate( int size );

   // Set an attributes value to NULL.
   //   Since you can't really set a number to NULL, this will
   //   map the value to the best default (typically 0)
   //
   void SetNull();
   void SetNull( int index );
   int  GetNullFlag() const { return m_null_flag; }
   void SetNullFlag(int flag) { m_null_flag = flag; }

   unsigned char GetNullByte()   const { return m_null_byte; }
   short         GetNullShort()  const { return m_null_short; }
   int           GetNullInt()    const { return m_null_int; }
   int           GetNullUInt()   const { return m_null_uint; }
   float         GetNullFloat()  const { return m_null_float; }
   double        GetNullDouble() const { return m_null_double; }
   const char*   GetNullString() const { return m_null_string; }

   void SetAttrNull(unsigned char v) {m_null_byte = v; }
   void SetAttrNull(short v)         {m_null_short = v; }
   void SetAttrNull(int v)           {m_null_int = v; }
   void SetAttrNull(unsigned int v)  {m_null_uint = v; }
   void SetAttrNull(float v)         {m_null_float = v; }
   void SetAttrNull(double v)        {m_null_double = v; }
   void SetNullValue(char * string);


   unsigned char GetNullValueByte() const;
   short GetNullValueShort() const;
   int GetNullValueInt() const;
	unsigned int GetNullValueUInt() const;
   float GetNullValueFloat() const;
   double GetNullValueDouble() const;
   char* GetNullValueString() const;

   int IsNullValue(unsigned char value) const;
   int IsNullValue(short value) const;
   int IsNullValue(int value) const;
   int IsNullValue(unsigned int value) const;
   int IsNullValue(float value) const;
   int IsNullValue(double value) const;
   int IsNullValue(const char * value) const;
   int ValueIsNull(int index) const;
   int ValueIsNull() const;


   // Various methods for setting this attributes value
   //   The single argument version will automatically set this attributes
   //   type, size = 1, and value. This is convenient when the attribute is
   //   used to store a single value. The second method requires that the
   //   attribute be previously setup, and memory allocated.
   //
   void SetValue( unsigned char value );
   void SetValue( int index, unsigned char value );

   void SetValue( short value );
   void SetValue( int index, short value );

   void SetValue( int value );
   void SetValue( int index, int value );

   void SetValue( unsigned int value );
   void SetValue( int index, unsigned int value );

   void SetValue( float value );
   void SetValue( int index, float value );

   void SetValue( double value );
   void SetValue( int index, double value );

   void SetValue( const char *value );
   void SetValue( int index, const char *value );

   void Set( DBIL_Attribute &attr, int attr_offset );
   void SetValue( int index, DBIL_Attribute &attr, int actual_size=0 );

   void SetValue (int index, const void *value, int size, DBIL_DATA_TYPES type, int len);

   // Get value of this attribute
   //
   unsigned char GetByte() const;
   unsigned char GetByte( int index ) const;

   short GetShort() const;
   short GetShort( int index ) const;

   int GetInt() const;
   int GetInt( int index ) const;

   unsigned int GetUInt() const;
   unsigned int GetUInt( int index ) const;

   float GetFloat() const;
   float GetFloat( int index ) const;

   double GetDouble() const;
   double GetDouble( int index ) const;

   const char *GetString() const;
   const char *GetString( int index ) const;

   // Get pointer to this attributes data, I know this violates pure data
   //  hideing, but the only way around it that I could find would require
   //  multiple memory allocations and copies...
   //
   void *GetDataPtr();
   void *GetPtr() const { return m_ptr; }
   void SetFunc(void * in_func) { m_free_ptr = (funcp *) in_func;}
   void *GetDataPtr( int index );
   void SetPtr(void * in_ptr) { m_ptr = in_ptr; }

private:
   unsigned char m_null_byte; // Individual attributes
   short         m_null_short;// may have individual null
   int           m_null_int;  // sentinal values, hence the
   unsigned int  m_null_uint;
   float         m_null_float;
   double        m_null_double;
   char          *m_null_string;

   // Utility to copy data from one attribute to another
   //
   void DuplicateData( const DBIL_Attribute *src );

   // Attribute parameters
   //
   int m_size;		// number of elements
   int m_elen;		// length in bytes of elements
   char *m_name;
   DBIL_DATA_TYPES m_type;
   short m_chunking;		// whether this string attribute allocates

   // memory in chunks or one per string.

   // Define pointers to where the data will be stored
   //
   void *m_ptr;
   funcp *m_free_ptr;
   int  *m_len;
   void *m_data;
   int   m_null_flag; // For global use only:
};
#endif
