//		Copyright (c) 1999 by
//		Advanced Visual Systems Inc.
//		All Rights Reserved
//
//	This software comprises unpublished confidential information of
//	Advanced Visual Systems Inc. and may not be used, copied or made
//	available to anyone, except in accordance with the license
//	under which it is furnished.
//
//      This file is under Perforce control
//      $Id: //depot/express/fcs70/dbil/dbil_field.hxx#1 $
//

//
// file: dbil_field.hxx
// author: Patrick Li, Advanced Visual Systems
// purpose: Define class for DBIL_Field
//
//   A Field can hold an image (2D array) or a volume (3D array) of nodes. It supports all
//   the basic data types, and vector lengths etc. A field is always accessed in conjunction
//   with a shape which is defined as the extent of the field.
//
#ifndef DBIL_FIELD_DEFINED
#define DBIL_FIELD_DEFINED 1

#include "avs/dbil.h"

// Key to put in packed bytestream to help read it back
//
#define DBIL_BLOB_KEY 0x00000001
#define DBIL_BLOB_REV 0x00000001

// Define DBIL_Field class
//
class DBIL_Field {
public:
   // Constructors
   //
   DBIL_Field();
   DBIL_Field( const DBIL_Field &fld );

   // Destructor
   //
   virtual ~DBIL_Field();

   // Clear fields data buffers etc..
   //
   void Reset();

   // Set and Get a field's name
   //
   void SetName( const char *name_in );
   const char *GetName() const { return name; }

   // Get fields dimensions
   //
   int GetNDims() const { return ndims; }
   const int *GetDims() const { return dims; }

   // Get info about the fields size
   //
   int GetVeclen() const { return veclen; }
   DBIL_DATA_TYPES GetType() const { return type; }

   // Set and Get a fields extents
   //
   void SetExtents( float *extents_in );
   const float *GetExtents() const { return extents; }

   // Return a pointer to a fields data
   //
   int GetDataSize() const { return dsize; }
   const void *GetData() const { return data; }
   const void *GetBuffer() const { return buffer; }
   int GetBufferSize() const { return dsize + sizeof(long); }

   // Set a value in this field based on specified value in src field
   //
   void SetValue( int x, int y, int z, int ix, int iy, int iz, DBIL_Field *src );

   // Bulk load this field from memory
   //
   DBIL_Status Load( char *name, DBIL_DATA_TYPES type, int ndims, int *dims, int veclen, void *data );

   // Pack and Unpack this field to/from a binary stream
   //
   int Pack( char **buffer ) const;
   DBIL_Status Unpack( char *buffer, int buflen );

   // Check byte ordering in field and correct if necesary
   //
   DBIL_Status CorrectData();

private:
   // Local parameters
   //
   char *name;
   int ndims;
   int *dims;
   int veclen;
   DBIL_DATA_TYPES type;
   int dsize;
   void *data;
   void *buffer;
   float *extents;

   // Create private copy constructor and assignment operator 
   // to invalidate copys and assignments
   //
   DBIL_Field &operator=(const DBIL_Field &) { return *this; }
};
#endif
