//		Copyright (c) 1999 by
//		Advanced Visual Systems Inc.
//		All Rights Reserved
//
//	This software comprises unpublished confidential information of
//	Advanced Visual Systems Inc. and may not be used, copied or made
//	available to anyone, except in accordance with the license
//	under which it is furnished.
//
//      This file is under Perforce control
//      $Id: //depot/express/fcs70/dbil/dbil_shape.hxx#1 $
//

//
// file: dbil_shape.hxx
// author: Patrick Li, Advanced Visual System
// purpose: Define DBIL_Shape class
//
// This class contains all the methods and functionality required to store a 'shape'. A shape in DBIL can
// be a 2 dimensional point, polyline, or polygon. It can also have one or more attributes, and a field.
// An example of a shapes attributes would be population, and average_age, if the shape were a state. A
// field would be a dted cell, and the shape would be the footprint of the cell.
//
#ifndef DBIL_SHAPE_DEFINED
#define DBIL_SHAPE_DEFINED

#include "avs/dbil.h"
#include "dbil_attr.hxx"
#include "dbil_field.hxx"

// Define DBIL_Shape class
//
class DBIL_Shape {
public:
   // Constructors and Destructor
   //
   DBIL_Shape();
   DBIL_Shape( const DBIL_Shape &shape );

   virtual ~DBIL_Shape();

   // Utilties
   //
   void SetName( const char *name );
   const char *GetName() const { return m_name; }

   void SetType( const DBIL_SHAPES type );
   const DBIL_SHAPES &GetType() const { return m_type; }
   // Methods to interact with shape's point data
   //
   DBIL_Status SetNumPoints( int n );
   DBIL_Status AddPoints( int n );

   void SetPoint( int index, float x, float y );
   void SetPoint( int index, float x, float y, float z );

   void SetDimensions(int dim){m_dims = dim;}
   int GetDimensions(){return m_dims;}

   int GetNumPoints() const { return m_npoints; }
   const float *GetXArray() const { return m_xdata; }
   const float *GetYArray() const { return m_ydata; }
   const float *GetZArray() const { return m_zdata; }

   // Methods to set and get shape part info for multi part shapes
   //
   DBIL_Status SetNumParts( int num_parts );
   int GetNumParts() const;

   void SetPart( int index, int start, int stop );
   void GetPart( int index, int *start, int *stop ) const;

   // Methods to interact with shapes field data
   //
   DBIL_Status AddField();
   DBIL_Status AddField( const DBIL_Field *fld );
   DBIL_Status AddField( char *buffer, int len );
   DBIL_Status AddField( char *name, DBIL_DATA_TYPES type, int ndims, int *dims, int veclen, void *data );
   DBIL_Field *GetField() const { return m_field; }
   void DeleteField();

   // Methods to interact with shape's attribute data
   //
   DBIL_Status CreateAttrs( int num_attrs );
   DBIL_Status CreateAttrs( int num_attrs, DBIL_Attribute *attrs );
   int GetNumAttrs() const { return m_nattrs; }

   void SetAttribute( int index, const DBIL_Attribute &attr );
   DBIL_Attribute *GetAttribute( int index ) const;

   // Utility to print a shape to stdout
   //
   void Print() const;

protected:

private:
   char *m_name;
   DBIL_SHAPES m_type;

   // Coordinate info
   //
   int m_npoints;
   int m_dims;
   float *m_xdata;
   float *m_ydata;
   float *m_zdata;

   // Part info for multi part shapes
   //
   int m_num_parts;
   int *m_part_start;
   int *m_part_stop;

   // We can currently only hold one field per shape
   //
   DBIL_Field *m_field;

   // A shape can have several attributes
   //
   int m_nattrs;
   DBIL_Attribute *m_attrs;

   // Free local memory
   //
   void freemem();

   // Create private copy constructor and assignment operator 
   // to invalidate copys and assignments
   //
   DBIL_Shape &operator=(const DBIL_Shape &) { return *this; }
};
#endif
