//*-*-c++-*-*
/*
                        Copyright (c) 1996 by
                        Advanced Visual Systems Inc.
                        All Rights Reserved

        This software comprises unpublished confidential information of
        Advanced Visual Systems Inc. and may not be used, copied or made
        available to anyone, except in accordance with the license
        under which it is furnished.

        This file is under Perforce control
        $Id: //depot/express/fcs70/gis/xp_mods/modGrid.cxx#1 $
*/

#define XP_WIDE_API	/* Use Wide APIs */

#include "GISGrid.hxx"

//
//      function prototype
//
//      Note: the function Index() is defined in modTrfm.cxx

xp_long Index(xp_long, xp_long, xp_long);

//
//      mfunction:      generateGrid
//
//      Given start and stop conditions, run through a 
//      grid of latitude and longitude data and put it 
//      into a mesh.
//

/* 64-bit porting. Only Modified Internally */
int
GIS_GISGrid::generateGrid(OMevent_mask event_mask, int seq_num)
{
  // westLongitude (OMXdouble read req)
  // eastLongitude (OMXdouble read req)
  // southLatitude (OMXdouble read req)
  // northLatitude (OMXdouble read req)
  // meridianIncrement (OMXdouble read req)
  // parallelIncrement (OMXdouble read req)
  // outLLA (Mesh write)

  LatLon        llStart;
  LatLon        urEnd;
  xp_long       totalLats;
  xp_long       totalLons;
  xp_long		totalIndices;

  // iterates
  double        iLat;
  double        iLon;
  xp_long       coordIterate = 0;
  xp_long       nodeIterate = 0;
  double        hold;
  int           bracketing_zero = 0;

  // Set the start and stop conditions
  llStart.setLat(southLatitude);
  llStart.setLon(westLongitude);
  urEnd.setLat(northLatitude);
  urEnd.setLon(eastLongitude);

  // Compute the number of points we are talking about here
  // There are a couple of special cases

  // latitude first

  hold = (urEnd.fetchLat() - llStart.fetchLat());
  if (hold == 0.0)
    totalLats = 1;      // can only do one line this direction
  else
    {
      hold = hold/parallelIncrement;
      if (floor(hold) == hold)
        totalLats = (xp_long)(floor(hold)) + 1; // nice and even division
      else
        totalLats = (xp_long)(floor(hold)) + 2; // everybody else
    }

  // now long
  hold = (urEnd.fetchLon() - llStart.fetchLon());
  if (hold < 0)
    {
      hold += 360.0;
      bracketing_zero = 1;
    }
  if (hold == 0.0)
    totalLons = 1;      // can only do one line this direction
  else
    {
      hold = hold / meridianIncrement;
      if (floor(hold) == hold)
        totalLons = (xp_long)(floor(hold)) + 1; // nice and even division
      else
        totalLons = (xp_long)(floor(hold)) + 2; // everybody else
    }

  // Set up the mesh data...
  outLLA.ncell_sets = 1;        // only one cell_set is necessary
  outLLA.nspace = 3;            // 3D is us

  // how many total polylines
  hold = (((totalLats - 1) * totalLons) +
	  ((totalLons - 1) * totalLats));
  outLLA.cell_set[0].ncells = (xp_long)hold;

  // How many vertices on the total number of polylines?
  hold = totalLats * totalLons;
  if(hold <= 0.0) {
    outLLA.nnodes = 0;
    outLLA.ncell_sets = 0;
    return 0;
  }

  totalIndices = ((xp_long) hold);
  outLLA.nnodes = ((xp_long) hold);

  coordIterate = 0;
  nodeIterate = 0;

  // OK, if we are really going to do this, let's go
  if (outLLA.nnodes > (xp_long)0)
    {
      // Get pointers to the mesh data
      float *coordPtr =
        (float *)outLLA.coordinates.values.ret_typed_array_ptr(OM_GET_ARRAY_WR,
                                                               OM_TYPE_FLOAT,
                                                               (xp_long *)NULL);
      if (coordPtr == (void *)NULL)
        {
          ERRerror ("GISGrid::generateGrid", 0, ERR_ORIG,
                    "Could not get out coordinates" );
          outLLA.nnodes = 0;
          outLLA.ncell_sets = 0;
          return 0;
        }

      // get the pointer to the array for connectivity
      // (*not* a polyline; its an ordinary "Line" cell set)
      xp_long *polyPtr = 
        (xp_long *)outLLA.cell_set[0].node_connect_list.ret_typed_array_ptr(OM_GET_ARRAY_WR,
                                                                        OM_TYPE_LONG,
                                                                        (xp_long *)NULL);

      if (polyPtr == (void *)NULL)
        {
          ERRerror ("GISGrid::generateGrid", 0, ERR_ORIG,
                    "Could not get out node connect list" );
          outLLA.nnodes = 0;
          outLLA.ncell_sets = 0;
          return 0;
        }

      coordIterate = 0;
      nodeIterate = 0;
      // Run through and load up the mesh 

      // latitude run first
      iLat = llStart.fetchLat();
      while (1)
        {
          iLon = llStart.fetchLon();
          while(1)
            {
              // location of the start of the line
              coordPtr[Index(coordIterate,1,3)] = (float)iLat;
              coordPtr[Index(coordIterate,0,3)] = (float)iLon;
              coordPtr[Index(coordIterate,2,3)] = 0.0;  // no altitude components!
              coordIterate++;

              // This is the termination condition for longitude
              if (iLon == urEnd.fetchLon() || 
                  iLon == (urEnd.fetchLon() + 360.))
                break;

	      // Record two polylines per run (three on first run)

	      // FIRST POLYLINE
	      // Record the first connectivity of these guys
	      polyPtr[nodeIterate++] = coordIterate-1;

	      // Record the second connectivity for these guys
	      polyPtr[nodeIterate++] = coordIterate;

	      // SECOND POLYLINE
	      if ((coordIterate + totalLons) < totalIndices)
		{
		  polyPtr[nodeIterate++] = coordIterate;
		  polyPtr[nodeIterate++] = coordIterate + totalLons;
		}

	      // If this is the first longitude, do a third line as well
	      if ((iLon == llStart.fetchLon()) && 
		  ((coordIterate - 1 + totalLons) < totalIndices))
		{
		  // THIRD POLYLINE
		  polyPtr[nodeIterate++] = coordIterate - 1;
		  polyPtr[nodeIterate++] = coordIterate - 1 + totalLons;
		}

              // increment the longitude counter
              iLon += meridianIncrement;

              // If we overshot, make sure you put a gridline at the
              // last latitude
              if (bracketing_zero) {
                if(iLon > urEnd.fetchLon() + 360.0) {
                  iLon = urEnd.fetchLon() + 360.0;
                }
              }
              else {
                if(iLon > urEnd.fetchLon()) {
                  iLon = urEnd.fetchLon();
                }
              }
            } /* inner while */

          // This is the termination condition for latitude
          if (iLat == urEnd.fetchLat())
            break;

          // increment the latitude counter
          iLat += parallelIncrement;

          // If we overshot, make sure you put a gridline at the
          // last latitude
          if (iLat > urEnd.fetchLat())
            iLat = urEnd.fetchLat();
        } /* outer while */

      // free em up
      ARRfree(coordPtr);
      ARRfree(polyPtr);
    }

  // All done

  // return 1 for success
  return(1);
}



