//*-*-c++-*-*
/*
                        Copyright (c) 1996 by
                        Advanced Visual Systems Inc.
                        All Rights Reserved

        This software comprises unpublished confidential information of
        Advanced Visual Systems Inc. and may not be used, copied or made
        available to anyone, except in accordance with the license
        under which it is furnished.

        This file is under Perforce control
        $Id: //depot/express/fcs70/gis/xp_mods/moddem.cxx#1 $
*/

#define XP_WIDE_API	/* Use Wide APIs */

#include <assert.h>

#include "GISDEM.hxx"


//
//      function prototypes
//
//      Note: the function Index() is defined in modtrfm.cxx

xp_long Index(xp_long, xp_long, xp_long);

/* 64-bit porting. Only Modified Internally */
int
GIS_READERS_GISDEM::ingest(OMevent_mask event_mask, int seq_num)
{
  // Iterators
  xp_long               coordIterate;
  xp_long               nodeIterate;
  float                 subSampleFactor;
  float                 trueElevation;
  double                ssRowInc;
  double                ssColInc;
  int                   ssMaxRows;
  int                   ssMaxCols;
  int                   moreRecords;

  // Convience pointers
  float                 *coordPtr;      // data array pointer
  xp_long               *dimPtr;        // polyline connectivity array pointer
  float                 *nodePtr;       // f(x,y,z) 

  // File I/O information
  UtString              fName;          // Full pathname to the file

  // if read_data isn't set, get outta here
  if (! read_data)
    return (1);         // All is ok, tho

  // Retrieve the pointer to the DEM_LogicalRecord class. It
  // should have been instanced when this class was created.
  DEM_LogicalRecord *DEM;
  DEM = (DEM_LogicalRecord *)ret_class_ptr("DEM_LogicalRecord");

  // Set up the file name and ingest the header information 
  // stored in DEM logical record A.
  // If the file isn't there, exit
  fName = (char *)demFile;
  DEM->setFN(fName);

  // The status check has bogus numbers, but it will tell folks
  // something is going on. These file readers are all self
  // contained, so I can't really get periodic status checks back
  // with them.
  OMstatus_check(15, "GISDEM", NULL);
  Information = "DEM reads take 1 min";
  if (DEM->ingest() < 0)
    // Didn't work, just return
    {
      status = "Header Read Failed";
      Information = "Header Read Failed";
      OMstatus_check(100, "GISDEM", NULL);
      return (1);
    }

  // Put up some static information from the header
  QuadName = DEM->fetchQuadrangle().c_str();
  Information = DEM->fetchInfo().c_str();

  // We only deal with geographic data, so jettison if anything else
  if (DEM->fetchGroundPlan() != 0)
    {
      status ="GroundPlan not 0";
      Information = "GroundPlan not 0";
      OMstatus_check(100, "GISDEM", NULL);
      return (1);
    }

  status = "File is ok";

  // SubSample stuff.
  //
  // The user inputs subsampling in terms of a percentage of 
  // samples that they want removed from the data stream.
  // Clamp the values of this down, and store it...
  //
  // 1.0 = 100 percent of the samples
  // 0.0 = 0 percent of the samples
  //
  if (subsample < 0.0)
    subSampleFactor = 0.0;
  else
    if (subsample > 1.0)
      subSampleFactor = 1.0;
    else
      subSampleFactor = subsample;

  OMstatus_check(0, "GISDEM", NULL);


  // NOTE TO FUTURE DEVELOPERS: 
  //
  // This is disgusting, but to handle all  DEM data types necessitates a 
  // two-pass process: one pass to read the file into an array, and
  // the second pass to put the data into a Mesh. This is 
  // because DEM file are set up to have a variable number of rows
  // of data - the maximum number of available columns cannot be
  // determined from the header (logical record A)...this number can 
  // only be determined by reading each logical record B and looking at
  // the value of the first (m,n) element. 
  //
  // However, most DEMs (the 1d x 1d and 30' x 30' file) are guaranteed to be 
  // square, I believe. Although this is never explicitly stated, it is 
  // implied by the DEM Data User's Guide under "Data Characteristics". Quoting:
  // "The unit of coverage is a 1- x 1- degree block. Elevation data on the integer 
  //  degree lines (all four sides) overlap with the
  //  corresponding profiles on the surrounding eight blocks....
  //  The 1-degree DEM mosaic data set is characteristically the same as the source 
  //  1- by 1-degree DEM unit of coverage..."
  // 
  // I'm not happy with this arrangement, but I am going to stick to it until 
  // I can figure out the most efficient way to read in the information and determine
  // the size from the data sets without unnecessary disk reads.
  //
  // - rjd, feb 96

  int maxRows = DEM->fetchColumns();
  int maxCols = DEM->fetchColumns();

  // Change the max values based on the subsampling
  ssMaxRows = (int) ((maxRows * subSampleFactor) + 0.49999);  // round  
  ssMaxCols = (int) ((maxCols * subSampleFactor) + 0.49999);

  if ( ((ssMaxRows * ssMaxCols) < 1) )
    {
      status = "Nothing left after subsample";
      return 1;
    }

  // Must make sure that the first and last rows and columns
  // are included for the resultant subsampled grid. If they
  // are not, there will be a noticeable gap in the output
  // when two or more DEM files are placed side by side.

  ssRowInc = ((maxRows-1)/(double)(ssMaxRows-1)); // how many to skip?
  ssColInc = ((maxCols-1)/(double)(ssMaxCols-1));

  //printf( "\nssRowInc: %f, ssColInc: %f\n", ssRowInc, ssColInc );

  //
  // Set up the mesh data...
  //
  dimPtr =
    (xp_long *)demGrid.dims.ret_typed_array_ptr(OM_GET_ARRAY_WR,
                                            OM_TYPE_LONG,
                                            (xp_long *)NULL);

  if (dimPtr == (void *)NULL)
    {
      status = "dimPtr is NULL";
      return 1;
    }

  // Set up the dimensions of the mesh field.
  dimPtr[0] = ssMaxCols;
  dimPtr[1] = ssMaxRows;

  //printf( "Dst Cols: %d,  Dst Rows: %d\n", ssMaxCols, ssMaxRows );

  // Set up the total number of values to receive
  demGrid.coordinates.nvals = (xp_long)ssMaxCols * ssMaxRows;

  coordIterate = 0;
  nodeIterate = 0;

  // Set up the starting values for the x and y run
  float xStart, yStart;
  float xInc, yInc, zInc;

  xInc = (float) ((DEM->fetchRez())[0]);
  yInc = (float) ((DEM->fetchRez())[1]);
  zInc = (float) ((DEM->fetchRez())[2]);

  // Should have everything all ready to go. Run through 
  // in a row/column manner and load up the arrays.
  // get the pointers to the mesh data
  coordPtr = 
    (float *)demGrid.coordinates.values.ret_typed_array_ptr(OM_GET_ARRAY_RW,
                                                            OM_TYPE_FLOAT,
                                                            (xp_long *)NULL);

  if (coordPtr == (void *)NULL)
    {
      status = "coordPtr is NULL";
      return 1;
    }

  // Turn on the nodedata field
  demGrid.nnode_data = 1;
  // Tell the node data that we are just a nvals x 1
  demGrid.node_data[0].veclen = 1;
  // vnals should be filled in and automagically set to demGrid.coordinate.nvals

  // get the pointer to the node data
  nodePtr = 
    (float *)demGrid.node_data[0].values.ret_typed_array_ptr(OM_GET_ARRAY_WR,
                                                             OM_TYPE_FLOAT,
                                                            (xp_long *)NULL);

  if (nodePtr == (void *)NULL)
    {
      status = "nodePtr is NULL";
      return 1;
    }

  coordIterate = 0;
  nodeIterate= 0;

  // Set up the units and labels
  demGrid.node_data[0].labels = "elevation";
  demGrid.node_data[0].units = "meters";

  demGrid.coordinates.labels = "location";
  demGrid.coordinates.units = "degrees, degrees, meters";

  // Run through and load up the mesh's polylines running 
  // fastest in the row direction

  int prevColIndex = -1;

  for ( int i = 0; i < ssMaxCols; ++i )
    {
      // RowIndex means which col, *not* position in col.
      int demColIndex = (int) ( (i * ssColInc) + 0.499999 );  // round

      assert( demColIndex < maxCols );

      int colsToRead = demColIndex - prevColIndex;
      prevColIndex = demColIndex;

      //printf( "Reading records: %d\n", colsToRead );
      for( int ii = 0; ii < colsToRead; ++ii ) {
         moreRecords = DEM->readRecord();
         if ( moreRecords < 0 )
            break;
      }

      OMstatus_check( (i*100)/ssMaxCols, "GISDEM", NULL);

      //printf( "Columns- src:%d dst:%d\n", demColIndex, i );

      xStart = DEM->fetchBaseX();	// Does this vary?
      yStart = DEM->fetchBaseY();

      for ( int j = 0; j < ssMaxRows; ++j )
        {
          // Line locations...The DEM folks properly store x/y: x=longitude, y = latitude
          // The coordinates are in seconds, so convert to decimal 
          // degrees before continuing...
          coordPtr[Index(coordIterate,0,3)] = (float) (xStart/3600.0);
          coordPtr[Index(coordIterate,1,3)] = (float) (yStart/3600.0);
          yStart += (yInc * ssRowInc);

          // RowIndex means which row, *not* position in row.
          int demRowIndex = (int) ( (j * ssRowInc) + 0.499999 );  // round

          assert( demRowIndex < maxRows );
          //if( i == 0 ) printf( "Rows - src: %d, dst: %d\n", demRowIndex, j );

          // Elevations are always represented in DEM files as  meters
          // only. Find the local datum, then add
          // the global radius to get the true elevation for display.
          trueElevation = DEM->fetchElevation(demRowIndex,0) + 
              DEM->fetchLocalDatum();

          coordPtr[Index(coordIterate,2,3)] = trueElevation;

          // Load up the node data
          nodePtr[nodeIterate] = trueElevation;

          // Done with this location
          coordIterate++;
          nodeIterate++;
        }
    }

  // That oughta do it...
  OMstatus_check(100, "GISDEM", NULL);

  // free em up
  ARRfree(coordPtr);
  ARRfree(dimPtr);
  ARRfree(nodePtr);

  // All done
  // return 1 for success
  return(1);
}
