/*
			Copyright (c) 1994 by
			Advanced Visual Systems Inc.
			All Rights Reserved

	This software comprises unpublished confidential information of
	Advanced Visual Systems Inc. and may not be used, copied or made
	available to anyone, except in accordance with the license
	under which it is furnished.

	This file is under Perforce control
	$Id: //depot/express/fcs70/gmod/file.c#1 $
*/

#define XP_WIDE_API	/* Use Wide APIs */

#include <stdio.h>
#include <stdlib.h>

#ifndef MSDOS
#include <sys/types.h>
#include <rpc/types.h>
#include <rpc/xdr.h>
#else
#ifdef WIN32
#define NOGDI /* Squash conflict over 'ERROR' */
#define WIN32_LEAN_AND_MEAN
#endif
#include <fcntl.h>
#include <io.h>
#include <avs\pc_xdr\types.h>
#include <avs\pc_xdr\xdr.h>
#endif

#include <avs/gmod.h>
#include "uci_file.h"
#include <avs/om.h>
#include <avs/om_att.h>
#include <avs/om_type.h>
#include <avs/err.h>
#include <avs/f_utils.h>

#define ASCII_MODE 0
#define BIN_MODE   1
#define XDR_MODE   2

#ifdef MSDOS
/* Our own, somewhat outdated, version of regular expressions. */
#include <avs/regexp.h>
#else
/* Otherwise use POSIX regular expressions */
#define REGEXP_POSIX
#include <regex.h>
#endif

#define BUF_SIZE 256
#define BLOCK_SIZE 1024

typedef  unsigned char byte;

static int XDR_TypeSize[7] = { 1, 1, 2, 4, 4, 8, sizeof(xp_long)};

int file_long_type (OMobj_id, int *);
int file_get_data_type (OMobj_id, int *);
int file_get_array_dims (OMobj_id, int *, xp_long *);

int file_get_sub_farray (OMobj_id, int, xp_long *, xp_long *, xp_long *, float *);
int file_get_sub_rarray (OMobj_id, int, xp_long *, xp_long *, xp_long *, double *);
int file_get_sub_larray (OMobj_id, int, xp_long *, xp_long *, xp_long *, xp_long *);
int file_get_sub_iarray (OMobj_id, int, xp_long *, xp_long *, xp_long *, int *);
int file_get_sub_barray (OMobj_id, int, xp_long *, xp_long *, xp_long *, byte *);
int file_get_sub_sarray (OMobj_id, int, xp_long *, xp_long *, xp_long *, short *);
int file_get_array (OMobj_id, int *, char **, int *, xp_long *, int);

int file_get_real_val (OMobj_id, double *);

int file_expr_get_long_val (OMobj_id, xp_long *);
int file_skip_line_get_long_val (OMobj_id, xp_long *);
int file_skip_word_get_long_val (OMobj_id, xp_long *);
int file_valid_obj (OMobj_id obj_id, OMobj_id templ_id, int mode);

int file_ascii_get_array(OMobj_id elem_id, int *in_type, char **array,
			 int *ndim, xp_long *dims, int mode);
int file_array_ascii_bin(OMobj_id elem_id);
int file_bin_get_real_val(OMobj_id elem_id, double *val);
int file_get_array_dims_bin(OMobj_id elem_id, int *ndim, xp_long *dims);
int file_bin_sub_array(OMobj_id elem_id, int ndim, xp_long *dims, xp_long *min_rng,
		       xp_long *max_rng, void *array, int type,
		       OMsub_array_func func);

/* 64-bit porting. Only Modified Internally */
void FILEinit(void)
{
	OMtype_id type;

	type = OMcreate_type("Tfile_obj",OM_type_val_func, 0);
	OMset_method(OM_METH_GET_DATA_TYPE,type,file_get_data_type);
	OMset_method(OM_METH_GET_SUB_RARRAY,type,file_get_sub_rarray);
	OMset_method(OM_METH_GET_SUB_BARRAY,type,file_get_sub_barray);
	OMset_method(OM_METH_GET_SUB_SARRAY,type,file_get_sub_sarray);
	OMset_method(OM_METH_GET_SUB_FARRAY,type,file_get_sub_farray);
	OMset_method(OM_METH_GET_SUB_LARRAY,type,file_get_sub_larray);
	OMset_method(OM_METH_GET_SUB_IARRAY,type,file_get_sub_iarray);
	OMset_method(OM_METH_GET_REAL_VAL,type,file_get_real_val);
	OMset_method(OM_METH_GET_LONG_VAL,type,OMreal_get_long_val);
	OMset_method(OM_METH_GET_INT_VAL,type,OMreal_get_int_val);
	OMset_method(OM_METH_GET_ARRAY,type,file_get_array);
	OMset_method(OM_METH_GET_ARRAY_DIMS,type,file_get_array_dims);
	OMset_method(OM_METH_VALID_OBJ,type,file_valid_obj);

	type = OMcreate_type("Tfile_find_expr",OM_type_val_func, 0);
	OMset_method(OM_METH_GET_INT_VAL,type,OMlong_get_int_val);
	OMset_method(OM_METH_GET_LONG_VAL,type,file_expr_get_long_val);
	OMset_method(OM_METH_GET_DATA_TYPE,type,file_long_type);

	type = OMcreate_type("Tfile_skip_lines",OM_type_val_func, 0);
	OMset_method(OM_METH_GET_INT_VAL,type,OMlong_get_int_val);
	OMset_method(OM_METH_GET_LONG_VAL,type,file_skip_line_get_long_val);
	OMset_method(OM_METH_GET_DATA_TYPE,type,file_long_type);

	type = OMcreate_type("Tfile_skip_words",OM_type_val_func, 0);
	OMset_method(OM_METH_GET_INT_VAL,type,OMlong_get_int_val);
	OMset_method(OM_METH_GET_LONG_VAL,type,file_skip_word_get_long_val);
	OMset_method(OM_METH_GET_DATA_TYPE,type,file_long_type);
}

int FileUpdate(FileStruct *file, FileBitmask *Bmk)
{
	FileBitmask OutBmk;

	memset((char *)&OutBmk, 0, sizeof(FileBitmask));
	OMcstruct_resolve_ptr(file, NULL);

	if (file->status && file->fptr) {
		/**  close the open file **/

		fclose(file->fptr);
		file->fptr = NULL;
		file->cur_offset = 0;
		file->status = 0;
		OutBmk.cur_offset = 1;
		OutBmk.status = 1;
	}
	if ((file->fptr = FILEfopen(file->name, SIO_R_BIN)) == NULL) {
		ERRerror("FileUpdate",1,ERR_ORIG,
			 "cannot open file %s", file->name);
		file->status = 0;
		OutBmk.status = 1;
		OutBmk.fptr = 1;
		OMcstruct_update_om(file, &OutBmk);
	}
	file->status = 1;
	file->cur_offset = 0;
	OutBmk.cur_offset = 1;
	OutBmk.status = 1;
	OutBmk.fptr = 1;

	OMcstruct_update_om(file, &OutBmk);
	return(1);
}

int FileDel (FileStruct *file, FileBitmask *Bmk)
{
	if (file->status && file->fptr)
		fclose(file->fptr);
	return(1);
}

/* 64-bit porting. Directly Modified */
int file_long_type (OMobj_id elem_id, int *type)
{
	*type = DTYPE_LONG;
	return(1);
}

/* 64-bit porting. Directly Modified */
int
file_expr_get_long_val (OMobj_id elem_id, xp_long *val)
{
	OMobj_id file_id, fptr_id, pat_id;
	FILE *fp;
	int retStat, offset; 
	xp_long count, start;
	char buf1[128];
	char *pattern;
	char string[BUF_SIZE];
#ifdef REGEXP_POSIX
	regex_t rdata;
#else
	struct regexp *rdata;
#endif

	file_id = OMfind_subobj(elem_id,OMstr_to_name("file"),OM_OBJ_RD);
	if (OMis_null_obj(file_id)) {
		return(0);
	}
	if ((retStat = OMget_obj_val(file_id,&file_id)) != 1) {
		if (retStat == -1) {
			ERRerror("file_expr_get_long_val",1,ERR_CHAIN,
				 "can't get file object: %s",
				 OMret_obj_path(elem_id,buf1,sizeof(buf1)));
		}
		return(retStat);
	}

	retStat = OMget_name_int_val (elem_id, OMstr_to_name("offset"), &offset);
	if (retStat != 1) {
		if (retStat == -1) {
			ERRerror("file_expr_get_long_val",1,ERR_CHAIN,
				 "can't get file offset: %s",
				 OMret_obj_path(elem_id,buf1,sizeof(buf1)));
		}
		return(retStat);
	}

	fptr_id = OMfind_subobj(file_id, OMstr_to_name("fptr"), OM_OBJ_RD);
	if (OMis_null_obj(fptr_id)) {
		ERRerror("file_expr_get_long_val",1,ERR_CHAIN,
			 "can't get file pointer object: %s",
			 OMret_obj_path(elem_id,buf1,sizeof(buf1)));
		return(-1);
	}
	retStat = OMget_ptr_val(fptr_id, (void **)&fp, 0);
	if (retStat != 1 || fp == NULL) {
		if (retStat == 1 && fp == NULL) retStat = 0;
		if (retStat == -1) {
			ERRerror("file_expr_get_long_val",1,ERR_CHAIN,
				 "file pointer is not set: %s",
				 OMret_obj_path(elem_id,buf1,sizeof(buf1)));
		}
		return(retStat);
	}

	pat_id = OMfind_subobj(elem_id, OMstr_to_name("pattern"), OM_OBJ_RD);
	if (OMis_null_obj(pat_id)) {
		ERRerror("file_expr_get_long_val",1,ERR_CHAIN,
			 "can't get pattern object: %s",
			 OMret_obj_path(elem_id,buf1,sizeof(buf1)));
		return(-1);
	}
	if ((retStat = OMget_str_val(pat_id, &pattern, 0)) != 1 || pattern == NULL) {
		if (retStat == -1) {
			ERRerror("file_expr_get_long_val",1,ERR_CHAIN,
				 "can't get pattern value: %s",
				 OMret_obj_path(elem_id,buf1,sizeof(buf1)));
		}
		if (pattern == NULL)
			retStat = 0;
		return(retStat);
	}


	start = offset;
	retStat = fseek (fp, start, SEEK_SET);
	if (retStat != 0) {
		ERRerror("file_expr_get_long_val",2,ERR_CHAIN,
			 "fseek failed to position at %ld, %s", start,
			 OMret_obj_path(elem_id,buf1,sizeof(buf1)));
		free(pattern);
		return(-1);
	}

#ifdef REGEXP_POSIX
        retStat = regcomp(&rdata, pattern, 0);
	if (retStat != 0) {
#else
	rdata = regcomp(pattern);
	if (!rdata) {
#endif
		ERRerror("file_expr_get_long_val",2,ERR_CHAIN,
			 "pattern %s couldn't compile in file: %s", pattern,
			 OMret_obj_path(elem_id,buf1,sizeof(buf1)));
		free(pattern);
		return(-1);
	}

	retStat = 1;
	count = offset;
	while (retStat) {
		if (fgets(string, BUF_SIZE, fp) != NULL) {
#ifdef REGEXP_POSIX
			regmatch_t rmatch;
			if (regexec(&rdata, string, 1, &rmatch, 0) == 0) {
				count += rmatch.rm_eo;
#else
			if (regexec(rdata, string)) {
				count += rdata->endp[0] - string;
#endif
				retStat = 0;
			}
			else {
				count += strlen(string);
			}
		}
		else {
			ERRerror("file_expr_get_long_val",2,ERR_CHAIN,
				 "pattern %s not found in file: %s", pattern,
				 OMret_obj_path(elem_id,buf1,sizeof(buf1)));
#ifdef REGEXP_POSIX
			regfree(&rdata);
#else
			free(rdata);
#endif
			free(pattern);
			return(-1);
		}
	}
	*val = count;

#ifdef REGEXP_POSIX
	regfree(&rdata);
#else
	free(rdata);
#endif
	free(pattern);

	return(1);
}

/* 64-bit porting. Directly Modified */
int
file_skip_line_get_long_val (OMobj_id elem_id, xp_long *val)
{
	OMobj_id file_id, fptr_id;
	FILE *fp;
	int i, retStat, offset, nlines;
	xp_long count, start;
	char buf1[128];
	char string[BUF_SIZE];


	file_id = OMfind_subobj(elem_id,OMstr_to_name("file"),OM_OBJ_RD);
	if (OMis_null_obj(file_id)) {
		return(0);
	}
	if ((retStat = OMget_obj_val(file_id,&file_id)) != 1) {
		if (retStat == -1) {
			ERRerror("file_skip_lines_get_long_val",1,ERR_CHAIN,
				 "can't get file object: %s",
				 OMret_obj_path(elem_id,buf1,sizeof(buf1)));
		}
		return(retStat);
	}

	retStat = OMget_name_int_val (elem_id, OMstr_to_name("offset"), &offset);
	if (retStat != 1) {
		if (retStat == -1) {
			ERRerror("file_skip_lines_get_long_val",1,ERR_CHAIN,
				 "can't get file offset value: %s",
				 OMret_obj_path(elem_id,buf1,sizeof(buf1)));
		}
		return(retStat);
	}

	retStat = OMget_name_int_val (elem_id, OMstr_to_name("nlines"), &nlines);
	if (retStat != 1) {
		if (retStat == -1) {
			ERRerror("file_skip_lines_get_long_val",1,ERR_CHAIN,
				 "can't get nlines value: %s",
				 OMret_obj_path(elem_id,buf1,sizeof(buf1)));
		}
		return(retStat);
	}

	fptr_id = OMfind_subobj(file_id, OMstr_to_name("fptr"), OM_OBJ_RD);
	if (OMis_null_obj(fptr_id)) {
		ERRerror("file_skip_lines_get_long_val",1,ERR_CHAIN,
			 "can't get file pointer object: %s",
			 OMret_obj_path(elem_id,buf1,sizeof(buf1)));
		return(-1);
	}
	retStat = OMget_ptr_val(fptr_id, (void **)&fp, 0);
	if (retStat != 1 || fp == NULL) {
		if (retStat == 1 && fp == NULL) retStat = 0;
		if (retStat == -1) {
			ERRerror("file_skip_lines_get_long_val",1,ERR_CHAIN,
				 "file pointer is not set: %s",
				 OMret_obj_path(elem_id,buf1,sizeof(buf1)));
		}
		return(retStat);
	}

	start = offset;
	retStat = fseek (fp, start, SEEK_SET);
	if (retStat != 0) {
		ERRerror("file_skip_lines_get_long_val",2,ERR_CHAIN,
			 "fseek failed to position at %ld, %s", start,
			 OMret_obj_path(elem_id,buf1,sizeof(buf1)));
		return(-1);
	}

	count = offset;
	for (i=0; i< nlines; i++) {
		if (fgets(string, BUF_SIZE, fp) != NULL) {
			count += strlen(string);
		}
		else {
			ERRerror("file_skip_lines_get_long_val",2,ERR_CHAIN,
				 "cannot skip more then %d lines: %s", i,
				 OMret_obj_path(elem_id,buf1,sizeof(buf1)));
			return(-1);
		}
	}
	*val = count;
	return(1);
}

/* 64-bit porting. Directly Modified */
int
file_skip_word_get_long_val (OMobj_id elem_id, xp_long *val)
{
	OMobj_id file_id, fptr_id;
	FILE *fp;
	int i, retStat, offset, nwords;
	xp_long count, start;
	char buf1[128];
	char string[BUF_SIZE], *str;

	file_id = OMfind_subobj(elem_id,OMstr_to_name("file"),OM_OBJ_RD);
	if (OMis_null_obj(file_id)) {
		return(0);
	}
	if ((retStat = OMget_obj_val(file_id,&file_id)) != 1) {
	    if (retStat == -1) {
		ERRerror("file_skip_words_get_long_val",1,ERR_CHAIN,
			 "can't get file object: %s",
			 OMret_obj_path(elem_id,buf1,sizeof(buf1)));
	    }
	    return(retStat);
	}

	retStat = OMget_name_int_val (elem_id, OMstr_to_name("offset"), &offset);
	if (retStat != 1) {
		if (retStat == -1) {
			ERRerror("file_skip_words_get_long_val",1,ERR_CHAIN,
				 "can't get file offset value: %s",
				 OMret_obj_path(elem_id,buf1,sizeof(buf1)));
		}
		return(retStat);
	}

	retStat = OMget_name_int_val (elem_id, OMstr_to_name("nwords"), &nwords);
	if (retStat != 1) {
		if (retStat == -1) {
			ERRerror("file_skip_words_get_long_val",1,ERR_CHAIN,
				 "can't get nwords value: %s",
				 OMret_obj_path(elem_id,buf1,sizeof(buf1)));
		}
		return(retStat);
	}

	fptr_id = OMfind_subobj(file_id, OMstr_to_name("fptr"), OM_OBJ_RD);
	if (OMis_null_obj(fptr_id)) {
		ERRerror("file_skip_words_get_long_val",1,ERR_CHAIN,
			 "can't get file pointer object: %s",
			 OMret_obj_path(elem_id,buf1,sizeof(buf1)));
		return(-1);
	}
	retStat = OMget_ptr_val(fptr_id, (void **)&fp, 0);
	if (retStat != 1 || fp == NULL) {
		if (retStat == 1 && fp == NULL) retStat = 0;
		if (retStat == -1) {
			ERRerror("file_skip_words_get_long_val",1,ERR_CHAIN,
				 "file pointer is not set: %s",
				 OMret_obj_path(elem_id,buf1,sizeof(buf1)));
		}
		return(retStat);
	}

	start = offset;
	retStat = fseek (fp, start, SEEK_SET);
	if (retStat != 0) {
		ERRerror("file_skip_words_get_long_val",2,ERR_CHAIN,
			 "fseek failed to position at %ld: %s", start,
			 OMret_obj_path(elem_id,buf1,sizeof(buf1)));
		return(-1);
	}

	count = offset;
	if (fgets(string, BUF_SIZE, fp) == NULL) {
		ERRerror("file_skip_words_get_long_val",1,ERR_CHAIN,
			 "cannot read line: %s",
			 OMret_obj_path(elem_id,buf1,sizeof(buf1)));
		return(-1);
	}

	str = string;
	/** skip leading spaces **/
	while(*(str) == ' ')
		str++;

	for (i=0; i< nwords; i++) {
		/** skip word **/
		while(*(str) && *(str) != ' ')
			str++;
		if (!(*str)) {
			ERRerror("file_skip_words_get_long_val",2,ERR_CHAIN,
				 "cannot skip more than %d words: %s",i,
				 OMret_obj_path(elem_id,buf1,sizeof(buf1)));
			return(-1);
		}
		/** skip trailing spaces **/
		while(*(str) == ' ')
			str++;

	}
	count += str-string;
	*val = count;
	return(1);
}

/* 64-bit porting. Directly Modified */
int file_get_array(OMobj_id elem_id, int *in_type, char **array,
                   int *ndim, xp_long *dims, int mode)
{
	int ascii_bin;
	char buf1[128];

	ascii_bin = file_array_ascii_bin(elem_id);

	if (ascii_bin == ASCII_MODE)
		return(file_ascii_get_array(elem_id, in_type, array, ndim, dims, mode));
	else if (ascii_bin == BIN_MODE ||  ascii_bin == XDR_MODE)
		return(OMsub_array_get_array(elem_id, in_type, array, ndim, dims, mode));
	else {
		ERRerror("file_get_array",1,ERR_CHAIN,
			 "invalid ascii/binary parameter for get array operation: %s",
			 OMret_obj_path(elem_id,buf1,sizeof(buf1)));
		return(-1);
	}
}

/* 64-bit porting. Directly Modified */
int file_get_sub_barray(OMobj_id elem_id, int ndim, xp_long *dims,
                        xp_long *min_rng, xp_long *max_rng, unsigned char *array)
{
	int ascii_bin;
	char buf1[128];

	ascii_bin = file_array_ascii_bin(elem_id);

	if (ascii_bin == ASCII_MODE)
		return(OMarray_get_sub_barray(elem_id,ndim,dims,min_rng,max_rng,array));
	else if (ascii_bin == BIN_MODE ||  ascii_bin == XDR_MODE)
		return(file_bin_sub_array(elem_id,ndim,dims,min_rng,max_rng,
					  array,OM_TYPE_BYTE,OMget_sub_barray));
	else {
		ERRerror("file_get_array",1,ERR_CHAIN,
			 "invalid ascii/binary parameter for get array operation: %s",
			 OMret_obj_path(elem_id,buf1,sizeof(buf1)));
		return(-1);
	}
}

/* 64-bit porting. Directly Modified */
int file_get_sub_sarray(OMobj_id elem_id, int ndim, xp_long *dims,
                        xp_long *min_rng, xp_long *max_rng, short *array)
{
	int ascii_bin;
	char buf1[128];

	ascii_bin = file_array_ascii_bin(elem_id);

	if (ascii_bin == ASCII_MODE)
		return(OMarray_get_sub_sarray(elem_id,ndim,dims,min_rng,max_rng,array));
	else if (ascii_bin == BIN_MODE ||  ascii_bin == XDR_MODE)
		return(file_bin_sub_array(elem_id,ndim,dims,min_rng,max_rng,
					  array,OM_TYPE_SHORT,OMget_sub_sarray));
	else {
		ERRerror("file_get_array",1,ERR_CHAIN,
			 "invalid ascii/binary parameter for get array operation: %s",
			 OMret_obj_path(elem_id,buf1,sizeof(buf1)));
		return(-1);
	}
}

/* 64-bit porting. Directly Modified */
int file_get_sub_iarray(OMobj_id elem_id, int ndim, xp_long *dims,
                        xp_long *min_rng, xp_long *max_rng, int *array)
{
	int ascii_bin;
	char buf1[128];

	ascii_bin = file_array_ascii_bin(elem_id);

	if (ascii_bin == ASCII_MODE)
		return(OMarray_get_sub_iarray(elem_id,ndim,dims,min_rng,max_rng,array));
	else if (ascii_bin == BIN_MODE ||  ascii_bin == XDR_MODE)
		return(file_bin_sub_array(elem_id,ndim,dims,min_rng,max_rng,
					  array,OM_TYPE_INT,OMget_sub_iarray));
	else {
		ERRerror("file_get_array",1,ERR_CHAIN,
			 "invalid ascii/binary parameter for get array operation: %s",
			 OMret_obj_path(elem_id,buf1,sizeof(buf1)));
		return(-1);
	}
}

/* 64-bit porting. Newly Introduced */
int file_get_sub_larray(OMobj_id elem_id, int ndim, xp_long *dims,
                        xp_long *min_rng, xp_long *max_rng, xp_long *array)
{
	int ascii_bin;
	char buf1[128];

	ascii_bin = file_array_ascii_bin(elem_id);

	if (ascii_bin == ASCII_MODE)
		return(OMarray_get_sub_larray(elem_id,ndim,dims,min_rng,max_rng,array));
	else if (ascii_bin == BIN_MODE ||  ascii_bin == XDR_MODE)
		return(file_bin_sub_array(elem_id,ndim,dims,min_rng,max_rng,
					  array,OM_TYPE_LONG,OMget_sub_larray));
	else {
		ERRerror("file_get_array",1,ERR_CHAIN,
			 "invalid ascii/binary parameter for get array operation: %s",
			 OMret_obj_path(elem_id,buf1,sizeof(buf1)));
		return(-1);
	}
}

/* 64-bit porting. Directly Modified */
int file_get_sub_farray(OMobj_id elem_id, int ndim, xp_long *dims,
                        xp_long *min_rng, xp_long *max_rng, float *array)
{
	int ascii_bin;
	char buf1[128];

	ascii_bin = file_array_ascii_bin(elem_id);

	if (ascii_bin == ASCII_MODE)
		return(OMarray_get_sub_farray(elem_id,ndim,dims,min_rng,max_rng,array));
	else if (ascii_bin == BIN_MODE ||  ascii_bin == XDR_MODE)
		return(file_bin_sub_array(elem_id,ndim,dims,min_rng,max_rng,
					  array,OM_TYPE_FLOAT,OMget_sub_farray));
	else {
		ERRerror("file_get_array",1,ERR_CHAIN,
			 "invalid ascii/binary parameter for get array operation: %s",
			 OMret_obj_path(elem_id,buf1,sizeof(buf1)));
		return(-1);
	}
}

/* 64-bit porting. Directly Modified */
int file_get_sub_rarray(OMobj_id elem_id, int ndim, xp_long *dims,
                        xp_long *min_rng, xp_long *max_rng, double *array)
{
	int ascii_bin;
	char buf1[128];

	ascii_bin = file_array_ascii_bin(elem_id);

	if (ascii_bin == ASCII_MODE)
		return(OMarray_get_sub_rarray(elem_id,ndim,dims,min_rng,max_rng,array));
	else if (ascii_bin == BIN_MODE ||  ascii_bin == XDR_MODE)
		return(file_bin_sub_array(elem_id,ndim,dims,min_rng,max_rng,
					  array,OM_TYPE_DOUBLE,OMget_sub_rarray));
	else {
		ERRerror("file_get_array",1,ERR_CHAIN,
			 "invalid ascii/binary parameter for get array operation: %s",
			 OMret_obj_path(elem_id,buf1,sizeof(buf1)));
		return(-1);
	}
}


typedef struct _FileValueType {
   union {
      double  f;
      xp_long l;
      int     i;
      short   s;
      char    c;
   } value;
   int type;
} FileValueType;

/* 64-bit porting. Only Modified Internally */
double file_get_double_data(FileValueType val)
{
   switch(val.type) {
     case DTYPE_DOUBLE:
     case DTYPE_FLOAT:
      return val.value.f;
     case DTYPE_LONG:
      return (double)val.value.l;
     case DTYPE_INT:
      return (double)val.value.i;
     case DTYPE_SHORT:
      return (double)val.value.s;
     case DTYPE_BYTE:
     case DTYPE_CHAR:
      return (double)val.value.c;
   }
   return 0.0;
}

/* 64-bit porting. Newly Introduced */
xp_long file_get_long_data(FileValueType val)
{
   switch(val.type) {
     case DTYPE_DOUBLE:
     case DTYPE_FLOAT:
      return (xp_long)val.value.f;
     case DTYPE_LONG:
      return val.value.l;
     case DTYPE_INT:
      return (xp_long)val.value.i;
     case DTYPE_SHORT:
      return (xp_long)val.value.s;
     case DTYPE_BYTE:
     case DTYPE_CHAR:
      return (xp_long)val.value.c;
   }
   return 0;
}

/* 64-bit porting. Only Modified Internally */
int file_get_int_data(FileValueType val)
{
   switch(val.type) {
     case DTYPE_DOUBLE:
     case DTYPE_FLOAT:
      return (int)val.value.f;
     case DTYPE_LONG:
      return (int)val.value.l;
     case DTYPE_INT:
      return val.value.i;
     case DTYPE_SHORT:
      return (int)val.value.s;
     case DTYPE_BYTE:
     case DTYPE_CHAR:
      return (int)val.value.c;
   }
   return 0;
}

/* 64-bit porting. Only Modified Internally */
short file_get_short_data(FileValueType val)
{
   switch(val.type) {
     case DTYPE_DOUBLE:
     case DTYPE_FLOAT:
      return (short)val.value.f;
     case DTYPE_LONG:
      return (short)val.value.l;
     case DTYPE_INT:
      return (short)val.value.i;
     case DTYPE_SHORT:
      return val.value.s;
     case DTYPE_BYTE:
     case DTYPE_CHAR:
      return (short)val.value.c;
   }
   return 0;
}

/* 64-bit porting. Only Modified Internally */
char file_get_char_data(FileValueType val)
{
   switch(val.type) {
     case DTYPE_DOUBLE:
     case DTYPE_FLOAT:
      return (char)val.value.f;
     case DTYPE_LONG:
      return (char)val.value.l;
     case DTYPE_INT:
      return (char)val.value.i;
     case DTYPE_SHORT:
      return (char)val.value.s;
     case DTYPE_BYTE:
     case DTYPE_CHAR:
      return val.value.c;
   }
   return 0;
}

/* 64-bit porting. Only Modified Internally */
void file_store_data(int dst_type, void* dst_value, FileValueType src)
{
   switch(dst_type) {
     case DTYPE_DOUBLE:
      *((double*)dst_value) = file_get_double_data(src);
      break;
     case DTYPE_FLOAT:
      *((float*)dst_value)  = file_get_double_data(src);
      break;
     case DTYPE_LONG:
      *((xp_long*)dst_value) = file_get_long_data(src);
      break;
     case DTYPE_INT:
      *((int*)dst_value)    = file_get_int_data(src);
      break;
     case DTYPE_SHORT:
      *((short*)dst_value)  = file_get_short_data(src);
      break;
     case DTYPE_BYTE:
      *((unsigned char*)dst_value) = file_get_char_data(src);
      break;
     case DTYPE_CHAR:
      *((char*)dst_value)   = file_get_char_data(src);
      break;
   }
}

/* 64-bit porting. Directly Modified */
int
file_ascii_get_array(OMobj_id elem_id, int *in_type, char **array,
                     int *ndim, xp_long *dims, int mode)
{
	OMobj_id file_id, fptr_id, id;
	FILE *fp;
	int retStat, type, *col, dtype, done, nrow;
	int i, offset;
	xp_long count, calc_dims;
	xp_long i_w, ncol, size;
	xp_long start;
	char buf1[128];
	char form[8];
	double *darr;
	float  *farr;
	xp_long  *larr;
	int    *iarr;
	short  *sarr;
	char   *carr;
	unsigned char *barr;
	char string[BUF_SIZE];

	FileValueType val;

        memset(&val, 0, sizeof(val));

	if (mode != OM_GET_ARRAY_RD) {
		ERRerror("file_ascii_get_array",1,ERR_CHAIN,
			 "invalid mode for get array operation: %s",
			 OMret_obj_path(elem_id,buf1,sizeof(buf1)));
		return(-1);
	}

	file_id = OMfind_subobj(elem_id,OMstr_to_name("file"),OM_OBJ_RD);
	if (OMis_null_obj(file_id))
		return(0);
	if ((retStat = OMget_obj_val(file_id,&file_id)) != 1) {
		if (retStat == -1) {
			ERRerror("file_ascii_get_array",1,ERR_CHAIN,
				 "can't get file object: %s",
				 OMret_obj_path(elem_id,buf1,sizeof(buf1)));
		}
		return(retStat);
	}
	if ((retStat = OMget_name_int_val (elem_id, OMstr_to_name("type"), &type)) != 1) {
		if (retStat == -1) {
			ERRerror("file_ascii_get_array",1,ERR_CHAIN,
				 "can't get variable type: %s",
				 OMret_obj_path(elem_id,buf1,sizeof(buf1)));
		}
		return(retStat);
	}

	if (*in_type == DTYPE_UNSET)
		*in_type = type;

	retStat = OMget_name_int_val (elem_id, OMstr_to_name("offset"), &offset);
	if (retStat != 1) {
		if (retStat == -1) {
			ERRerror("file_ascii_get_array",1,ERR_CHAIN,
				 "can't get file offset: %s",
				 OMret_obj_path(elem_id,buf1,sizeof(buf1)));
		}
		return(retStat);
	}

	fptr_id = OMfind_subobj(file_id, OMstr_to_name("fptr"), OM_OBJ_RD);
	if (OMis_null_obj(fptr_id)) {
		ERRerror("file_ascii_get_array",1,ERR_CHAIN,
			 "can't get file pointer object: %s",
			 OMret_obj_path(elem_id,buf1,sizeof(buf1)));
		return(-1);
	}
	retStat = OMget_ptr_val(fptr_id, (void **)&fp, 0);
	if (retStat != 1 || fp == NULL) {
		if (retStat == 1 && fp == NULL) retStat = 0;
		if (retStat == -1) {
			ERRerror("file_ascii_get_array",1,ERR_CHAIN,
				 "file pointer is not set: %s",
				 OMret_obj_path(elem_id,buf1,sizeof(buf1)));
		}
		return(retStat);
	}
	start = offset;
	retStat = fseek (fp, start, SEEK_SET);
	if (retStat != 0) {
		ERRerror("file_ascii_get_array",2,ERR_CHAIN,
			 "fseek failed to position at %ld: %s", start,
			 OMret_obj_path(elem_id,buf1,sizeof(buf1)));
		return(-1);
	}

	id = OMfind_subobj(elem_id,OMstr_to_name("columns"),OM_OBJ_RD);
	if (OMis_null_obj(file_id)) {
		ERRerror("file_ascii_get_array",1,ERR_CHAIN,
			 "can't get columns object: %s",
			 OMret_obj_path(elem_id,buf1,sizeof(buf1)));
		return(-1);
	}
	dtype = OM_TYPE_INT;
	ncol = 0;
	col = (int *)NULL;
	retStat = OMget_array_sz(id, &dtype, (char **)(&col), &ncol, OM_GET_ARRAY_RD);
	if (retStat != 1) {
		if (retStat == -1) {
			ERRerror("file_ascii_get_array",1,ERR_CHAIN,
				 "cannot get columns array: %s",
				 OMret_obj_path(elem_id,buf1,sizeof(buf1)));
		}
		return(retStat);
	}

	if (*ndim != 0) {
		calc_dims = 0;
		for (size=1,i=0; i<*ndim; i++)
			size *= dims[i];
	}
	else {
		calc_dims = 1;
		*ndim = 1;
		dims[0] = 0;
		nrow = 0;
		for (i_w=0; i_w< ncol; i_w++)
			if (col[i_w])
				nrow += 1;
		size = BLOCK_SIZE;
	}

	*array = (char *)ARRalloc(NULL, *in_type, size, NULL);
	if (*array == NULL) {
		ERRerror("file_ascii_get_array",2,ERR_CHAIN,
			 "unable to allocate %ld elements for array: %s",
			 size, OMret_obj_path(elem_id,buf1,sizeof(buf1)));
		ARRfree(col);
		return(-1);
	}
	count = 0;
	done = 0;

	val.type = type;
	switch (type) {
	      case DTYPE_DOUBLE:
	      case DTYPE_FLOAT:
		strcpy(form, "%lf");
		break;
	      case DTYPE_LONG:
		strcpy(form, "%ld");
		break;
	      case DTYPE_INT:
		strcpy(form, "%d");
		break;
	      case DTYPE_SHORT:
		strcpy(form, "%hd");
		break;
	      case DTYPE_BYTE:
	      case DTYPE_CHAR:
		strcpy(form, "%c");
		break;
	      default:
		ERRerror("file_ascii_get_array",2,ERR_CHAIN,
			 "unknown data type %d: %s", (int)type,
			 OMret_obj_path(elem_id,buf1,sizeof(buf1)));
	}

	switch (*in_type) {
	      case DTYPE_DOUBLE:
		darr = (double *)*array;
		while (!done) {
			if (calc_dims && dims[0]+nrow >= size) {
				size += BLOCK_SIZE;
				*array = (char *)ARRrealloc(*array, type, size, NULL);
				if (*array == NULL) {
					ERRerror("file_ascii_get_array",2,ERR_CHAIN,
						 "unable to allocate %ld elements for array: %s",
						 size, OMret_obj_path(elem_id,buf1,sizeof(buf1)));
					ARRfree(col);
					return(-1);
				}
				darr = (double *)*array;
			}
			for (i_w=0; i_w< ncol; i_w++) {
				if (col[i_w]) {
					retStat = fscanf(fp, form, &val);
					if (retStat != 1) {
						if (calc_dims && dims[0] != 0) {
							done = 1;
							break;
						}
						ERRerror("file_ascii_get_array",2,ERR_CHAIN,
							 "fscanf failed to read %ld element: %s", count,
							 OMret_obj_path(elem_id,buf1,sizeof(buf1)));
						ARRfree(col);
						return(-1);
					}
					file_store_data(DTYPE_DOUBLE, &(darr[count++]), val);
				}
				else {
					retStat = fscanf(fp, "%s", string);
					if (retStat != 1) {
						if (calc_dims && dims[0] != 0) {
							done = 1;
							break;
						}
						ERRerror("file_ascii_get_array",2,ERR_CHAIN,
							 "fscanf failed to read %ld element: %s", count,
							 OMret_obj_path(elem_id,buf1,sizeof(buf1)));
						ARRfree(col);
						return(-1);
					}
				}
			}
			if (calc_dims && done==0)
				dims[0] += nrow;
			else if (count >= size)
				done = 1;
		}
		break;

	      case DTYPE_FLOAT:
		farr = (float *)*array;
		while (!done) {
			if (calc_dims && dims[0]+nrow >= size) {
				size += BLOCK_SIZE;
				*array = (char *)ARRrealloc(*array, type, size, NULL);
				if (*array == NULL) {
					ERRerror("file_ascii_get_array",2,ERR_CHAIN,
						 "unable to allocate %ld elements for array: %s",
						 size, OMret_obj_path(elem_id,buf1,sizeof(buf1)));
					ARRfree(col);
					return(-1);
				}
				farr = (float *)*array;
			}
			for (i_w=0; i_w< ncol; i_w++) {
				if (col[i_w]) {
					retStat = fscanf(fp, form, &val);
					if (retStat != 1) {
						if (calc_dims && dims[0] != 0) {
							done = 1;
							break;
						}
						ERRerror("file_ascii_get_array",2,ERR_CHAIN,
							 "fscanf failed to read %ld element: %s", count,
							 OMret_obj_path(elem_id,buf1,sizeof(buf1)));
						ARRfree(col);
						return(-1);
					}
					file_store_data(DTYPE_FLOAT, &(farr[count++]), val);
				}
				else {
					retStat = fscanf(fp, "%s", string);
					if (retStat != 1) {
						if (calc_dims && dims[0] != 0) {
							done = 1;
							break;
						}
						ERRerror("file_ascii_get_array",2,ERR_CHAIN,
							 "fscanf failed to read %ld element: %s", count,
							 OMret_obj_path(elem_id,buf1,sizeof(buf1)));
						ARRfree(col);
						return(-1);
					}
				}
			}
			if (calc_dims && done==0)
				dims[0] += nrow;
			else if (count >= size)
				done = 1;
		}
		break;

	      case DTYPE_LONG:
		larr = (xp_long *)*array;
		while (!done) {
			if (calc_dims && dims[0]+nrow >= size) {
				size += BLOCK_SIZE;
				*array = (char *)ARRrealloc(*array, type, size, NULL);
				if (*array == NULL) {
					ERRerror("file_ascii_get_array",2,ERR_CHAIN,
						 "unable to allocate %ld elements for array: %s",
						 size, OMret_obj_path(elem_id,buf1,sizeof(buf1)));
					ARRfree(col);
					return(-1);
				}
				larr = (xp_long *)*array;
			}
			for (i_w=0; i_w< ncol; i_w++) {
				if (col[i_w]) {
					retStat = fscanf(fp, form, &val);
					if (retStat != 1) {
						if (calc_dims && dims[0] != 0) {
							done = 1;
							break;
						}
						ERRerror("file_ascii_get_array",2,ERR_CHAIN,
							 "fscanf failed to read %ld element: %s", count,
							 OMret_obj_path(elem_id,buf1,sizeof(buf1)));
						ARRfree(col);
						return(-1);
					}
					file_store_data(DTYPE_LONG, &(larr[count++]), val);
				}
				else {
					retStat = fscanf(fp, "%s", string);
					if (retStat != 1) {
						if (calc_dims && dims[0] != 0) {
							done = 1;
							break;
						}
						ERRerror("file_ascii_get_array",2,ERR_CHAIN,
							 "fscanf failed to read %ld element: %s", count,
							 OMret_obj_path(elem_id,buf1,sizeof(buf1)));
						ARRfree(col);
						return(-1);
					}
				}
			}
			if (calc_dims && done==0)
				dims[0] += nrow;
			else if (count >= size)
				done = 1;
		}
		break;

	      case DTYPE_INT:
		iarr = (int *)*array;
		while (!done) {
			if (calc_dims && dims[0]+nrow >= size) {
				size += BLOCK_SIZE;
				*array = (char *)ARRrealloc(*array, type, size, NULL);
				if (*array == NULL) {
					ERRerror("file_ascii_get_array",2,ERR_CHAIN,
						 "unable to allocate %ld elements for array: %s",
						  size, OMret_obj_path(elem_id,buf1,sizeof(buf1)));
					ARRfree(col);
					return(-1);
				}
				iarr = (int *)*array;
			}
			for (i_w=0; i_w< ncol; i_w++) {
				if (col[i_w]) {
					retStat = fscanf(fp, form, &val);
					if (retStat != 1) {
						if (calc_dims && dims[0] != 0) {
							done = 1;
							break;
						}
						ERRerror("file_ascii_get_array",2,ERR_CHAIN,
							 "fscanf failed to read %ld element: %s", count,
							 OMret_obj_path(elem_id,buf1,sizeof(buf1)));
						ARRfree(col);
						return(-1);
					}
					file_store_data(DTYPE_INT, &(iarr[count++]), val);
				}
				else {
					retStat = fscanf(fp, "%s", string);
					if (retStat != 1) {
						if (calc_dims && dims[0] != 0) {
							done = 1;
							break;
						}
						ERRerror("file_ascii_get_array",2,ERR_CHAIN,
							 "fscanf failed to read %ld element: %s", count,
							 OMret_obj_path(elem_id,buf1,sizeof(buf1)));
						ARRfree(col);
						return(-1);
					}
				}
			}
			if (calc_dims && done==0)
				dims[0] += nrow;
			else if (count >= size)
				done = 1;
		}
		break;

	      case DTYPE_SHORT:
		sarr = (short *)*array;
		while (!done) {
			if (calc_dims && dims[0]+nrow >= size) {
				size += BLOCK_SIZE;
				*array = (char *)ARRrealloc(*array, type, size, NULL);
				if (*array == NULL) {
					ERRerror("file_ascii_get_array",2,ERR_CHAIN,
						 "unable to allocate %ld elements for array: %s",
						  size, OMret_obj_path(elem_id,buf1,sizeof(buf1)));
					ARRfree(col);
					return(-1);
				}
				sarr = (short *)*array;
			}
			for (i_w=0; i_w< ncol; i_w++) {
				if (col[i_w]) {
					retStat = fscanf(fp, form, &val);
					if (retStat != 1) {
						if (calc_dims && dims[0] != 0) {
							done = 1;
							break;
						}
						ERRerror("file_ascii_get_array",2,ERR_CHAIN,
							 "fscanf failed to read %ld element: %s", count,
							 OMret_obj_path(elem_id,buf1,sizeof(buf1)));
						ARRfree(col);
						return(-1);
					}
					file_store_data(DTYPE_SHORT, &(sarr[count++]), val);
				}
				else {
					retStat = fscanf(fp, "%s", string);
					if (retStat != 1) {
						if (calc_dims && dims[0] != 0) {
							done = 1;
							break;
						}
						ERRerror("file_ascii_get_array",2,ERR_CHAIN,
							 "fscanf failed to read %ld element: %s", count,
							 OMret_obj_path(elem_id,buf1,sizeof(buf1)));
						ARRfree(col);
						return(-1);
					}
				}
			}
			if (calc_dims && done==0)
				dims[0] += nrow;
			else if (count >= size)
				done = 1;
		}
		break;

	      case DTYPE_BYTE:
		barr = (byte *)*array;
		while (!done) {
			if (calc_dims && dims[0]+nrow >= size) {
				size += BLOCK_SIZE;
				*array = (char *)ARRrealloc(*array, type, size, NULL);
				if (*array == NULL) {
					ERRerror("file_ascii_get_array",2,ERR_CHAIN,
						 "unable to allocate %ld elements for array: %s",
						  size, OMret_obj_path(elem_id,buf1,sizeof(buf1)));
					ARRfree(col);
					return(-1);
				}
				barr = (byte *)*array;
			}
			for (i_w=0; i_w< ncol; i_w++) {
				if (col[i_w]) {
					retStat = fscanf(fp, form, &val);
					if (retStat != 1) {
						if (calc_dims && dims[0] != 0) {
							done = 1;
							break;
						}
						ERRerror("file_ascii_get_array",2,ERR_CHAIN,
							 "fscanf failed to read %ld element: %s", count,
							 OMret_obj_path(elem_id,buf1,sizeof(buf1)));
						ARRfree(col);
						return(-1);
					}
					file_store_data(DTYPE_BYTE, &barr[count++], val);
				}
				else {
					retStat = fscanf(fp, "%s", string);
					if (retStat != 1) {
						if (calc_dims && dims[0] != 0) {
							done = 1;
							break;
						}
						ERRerror("file_ascii_get_array",2,ERR_CHAIN,
							 "fscanf failed to read %ld element: %s", count,
							 OMret_obj_path(elem_id,buf1,sizeof(buf1)));
						ARRfree(col);
						return(-1);
					}
				}
			}
			if (calc_dims && done==0)
				dims[0] += nrow;
			else if (count >= size)
				done = 1;
		}
		break;

	      case DTYPE_CHAR:
		carr = (char *)*array;
		while (!done) {
			if (calc_dims && dims[0]+nrow >= size) {
				size += BLOCK_SIZE;
				*array = (char *)ARRrealloc(*array, type, size, NULL);
				if (*array == NULL) {
					ERRerror("file_ascii_get_array",2,ERR_CHAIN,
						 "unable to allocate %ld elements for array: %s",
						  size, OMret_obj_path(elem_id,buf1,sizeof(buf1)));
					ARRfree(col);
					return(-1);
				}
				carr = (char *)*array;
			}
			for (i_w=0; i_w< ncol; i_w++) {
				if (col[i_w]) {
					retStat = fscanf(fp, form, &val);
					if (retStat != 1) {
						if (calc_dims && dims[0] != 0) {
							done = 1;
							break;
						}
						ERRerror("file_ascii_get_array",2,ERR_CHAIN,
							 "fscanf failed to read %ld element: %s", count,
							 OMret_obj_path(elem_id,buf1,sizeof(buf1)));
						ARRfree(col);
						return(-1);
					}
					file_store_data(DTYPE_CHAR, &(carr[count++]), val);
				}
				else {
					retStat = fscanf(fp, "%s", string);
					if (retStat != 1) {
						if (calc_dims && dims[0] != 0) {
							done = 1;
							break;
						}
						ERRerror("file_ascii_get_array",2,ERR_CHAIN,
							 "fscanf failed to read %ld element: %s", count,
							 OMret_obj_path(elem_id,buf1,sizeof(buf1)));
						ARRfree(col);
						return(-1);
					}
				}
			}
			if (calc_dims && done==0)
				dims[0] += nrow;
			else if (count >= size)
				done = 1;
		}
		break;
	      default:
		ERRerror("file_ascii_get_array",2,ERR_CHAIN,
			 "unknown data type %d: %s", (int)*in_type, /* 64-bit changes. 'type' changed to '*in_type' */
			 OMret_obj_path(elem_id,buf1,sizeof(buf1)));
		ARRfree(col);
		return(-1);
	}

	ARRfree(col);
	return(1);
}

/* 64-bit porting. Only Modified Internally */
int file_ascii_get_real_val(OMobj_id elem_id, double *val)
{
	OMobj_id file_id, fptr_id;
	FILE *fp;
	int retStat, offset, type;
	xp_long start;
	char buf1[128];
	float  fdata;
	xp_long ldata;
	int    idata;
	short  sdata;
	char   cdata;
	unsigned char bdata;

	file_id = OMfind_subobj(elem_id,OMstr_to_name("file"),OM_OBJ_RD);
	if (OMis_null_obj(file_id))
		return(0);
	if ((retStat = OMget_obj_val(file_id,&file_id)) != 1) {
		if (retStat == -1) {
			ERRerror("file_ascii_get_real_val",1,ERR_CHAIN,
				 "can't get file object: %s",
				 OMret_obj_path(elem_id,buf1,sizeof(buf1)));
		}
		return(retStat);
	}
	if ((retStat = OMget_name_int_val (elem_id, OMstr_to_name("type"), &type)) != 1) {
		if (retStat == -1) {
			ERRerror("file_ascii_get_real_val",1,ERR_CHAIN,
				 "can't get variable type: %s",
				 OMret_obj_path(elem_id,buf1,sizeof(buf1)));
		}
		return(retStat);
	}

	retStat = OMget_name_int_val (elem_id, OMstr_to_name("offset"), &offset);
	if (retStat != 1) {
		if (retStat == -1) {
			ERRerror("file_ascii_get_real_val",1,ERR_CHAIN,
				 "can't get file offset: %s",
				 OMret_obj_path(elem_id,buf1,sizeof(buf1)));
		}
		return(retStat);
	}

	fptr_id = OMfind_subobj(file_id, OMstr_to_name("fptr"), OM_OBJ_RD);
	if (OMis_null_obj(fptr_id)) {
		ERRerror("file_ascii_get_real_val",1,ERR_CHAIN,
			 "can't get file pointer object: %s",
			 OMret_obj_path(elem_id,buf1,sizeof(buf1)));
		return(-1);
	}
	retStat = OMget_ptr_val(fptr_id, (void **)&fp, 0);
	if (retStat != 1 || fp == NULL) {
		if (retStat == 1 && fp == NULL) retStat = 0;
		if (retStat == -1) {
			ERRerror("file_ascii_get_real_val",1,ERR_CHAIN,
				 "file pointer is not set: %s",
				 OMret_obj_path(elem_id,buf1,sizeof(buf1)));
		}
		return(retStat);
	}
	start = offset;
	retStat = fseek (fp, start, SEEK_SET);
	if (retStat != 0) {
		ERRerror("file_ascii_get_real_val",2,ERR_CHAIN,
			 "fseek failed to position at %ld: %s", start,
			 OMret_obj_path(elem_id,buf1,sizeof(buf1)));
		return(-1);
	}
	switch (type) {
	      case DTYPE_DOUBLE:
		retStat = fscanf(fp, "%f", &fdata);
		if (retStat != 1) {
			ERRerror("file_ascii_get_real_val",1,ERR_CHAIN,
				 "fscanf failed to read object: %s",
				 OMret_obj_path(elem_id,buf1,sizeof(buf1)));
			return(-1);
		}
		*val = (double)fdata;
		break;
	      case DTYPE_FLOAT:
		retStat = fscanf(fp, "%f", &fdata);

		if (retStat != 1) {
			ERRerror("file_ascii_get_real_val",1,ERR_CHAIN,
				 "fscanf failed to read object: %s",
				 OMret_obj_path(elem_id,buf1,sizeof(buf1)));
			return(-1);
		}
		*val = (double)fdata;
		break;
	      case DTYPE_LONG:
		retStat = fscanf(fp, "%ld", &ldata);

		if (retStat != 1) {
			ERRerror("file_ascii_get_real_val",1,ERR_CHAIN,
				 "fscanf failed to read object: %s",
				 OMret_obj_path(elem_id,buf1,sizeof(buf1)));
			return(-1);
		}
		*val = (double)ldata;
		break;
	      case DTYPE_INT:
		retStat = fscanf(fp, "%d", &idata);

		if (retStat != 1) {
			ERRerror("file_ascii_get_real_val",1,ERR_CHAIN,
				 "fscanf failed to read object: %s",
				 OMret_obj_path(elem_id,buf1,sizeof(buf1)));
			return(-1);
		}
		*val = (double)idata;
		break;
	      case DTYPE_SHORT:
		retStat = fscanf(fp, "%hd", &sdata);

		if (retStat != 1) {
			ERRerror("file_ascii_get_real_val",1,ERR_CHAIN,
				 "fscanf failed to read object: %s",
				 OMret_obj_path(elem_id,buf1,sizeof(buf1)));
			return(-1);
		}
		*val = (double)sdata;
		break;
	      case DTYPE_BYTE:
		retStat = fscanf(fp, "%c", &bdata);

		if (retStat != 1) {
			ERRerror("file_ascii_get_real_val",1,ERR_CHAIN,
				 "fscanf failed to read object: %s",
				 OMret_obj_path(elem_id,buf1,sizeof(buf1)));
			return(-1);
		}
		*val = (double)bdata;
		break;
	      case DTYPE_CHAR:
		retStat = fscanf(fp, "%c", &cdata);

		if (retStat != 1) {
			ERRerror("file_ascii_get_real_val",1,ERR_CHAIN,
				 "fscanf failed to read object: %s",
				 OMret_obj_path(elem_id,buf1,sizeof(buf1)));
			return(-1);
		}
		*val = (double)cdata;
		break;
	      default:
		ERRerror("file_ascii_get_real_val",2,ERR_CHAIN,
			 "unknown data type %d: %s", (int)type,
			 OMret_obj_path(elem_id,buf1,sizeof(buf1)));
		return(-1);
	}

	return(1);
}


int file_get_data_type (OMobj_id elem_id, int *type)
{
	char buf1[128];
	int retStat;

	retStat = OMget_name_int_val (elem_id, OMstr_to_name("type"), type);
	if (retStat != 1) {
		if (retStat == -1) {
		   ERRerror("read_bin_array",1,ERR_CHAIN,
			    "can't get file type: %s",
			    OMret_obj_path(elem_id,buf1,sizeof(buf1)));
		   return(-1);
		}
	}
	return(retStat);
}

int file_get_real_val(OMobj_id elem_id, double *val)
{
	int ascii_bin;
	char buf1[128];

	ascii_bin = file_array_ascii_bin(elem_id);

	if (ascii_bin == ASCII_MODE)
		return(file_ascii_get_real_val(elem_id, val));
	else if (ascii_bin == BIN_MODE ||  ascii_bin == XDR_MODE)
		return(file_bin_get_real_val(elem_id, val));
	else {
		ERRerror("file_get_real_val",1,ERR_CHAIN,
			 "invalid ascii/binary parameter for get real val: %s",
			 OMret_obj_path(elem_id,buf1,sizeof(buf1)));
		return(-1);
	}
}

/* 64-bit porting. Only Modified Internally */
int file_bin_get_real_val(OMobj_id elem_id, double *val)
{
	OMobj_id file_id, fptr_id;
	FILE *fp;
	int retStat, offset, type;
	xp_long start;
	char buf1[128];
	double ddata;
	float  fdata;
	xp_long ldata;
	int    idata;
	short  sdata;
	char   cdata;
	unsigned char bdata;
	int mode;
	XDR xdrs;

	file_id = OMfind_subobj(elem_id,OMstr_to_name("file"),OM_OBJ_RD);
	if (OMis_null_obj(file_id)) {
		return(0);
	}
	if ((retStat = OMget_obj_val(file_id,&file_id)) != 1) {
	    if (retStat == -1) {
		ERRerror("file_bin_get_real_val",1,ERR_CHAIN,
			 "can't get file object: %s",
			 OMret_obj_path(elem_id,buf1,sizeof(buf1)));
	    }
	    return(retStat);
	}
	if ((retStat = OMget_name_int_val (elem_id, OMstr_to_name("type"), &type)) != 1) {
		if (retStat == -1) {
			ERRerror("file_bin_get_real_val",1,ERR_CHAIN,
				 "can't get variable type: %s",
				 OMret_obj_path(elem_id,buf1,sizeof(buf1)));
		}
		return(retStat);
	}

	retStat = OMget_name_int_val (elem_id, OMstr_to_name("offset"), &offset);
	if (retStat != 1) {
		if (retStat == -1) {
			ERRerror("file_bin_get_real_val",1,ERR_CHAIN,
				 "can't get file offset: %s",
				 OMret_obj_path(elem_id,buf1,sizeof(buf1)));
		}
		return(retStat);
	}

	fptr_id = OMfind_subobj(file_id, OMstr_to_name("fptr"), OM_OBJ_RD);
	if (OMis_null_obj(fptr_id)) {
		ERRerror("file_bin_get_real_val",1,ERR_CHAIN,
			 "can't get file pointer object: %s",
			 OMret_obj_path(elem_id,buf1,sizeof(buf1)));
		return(-1);
	}
	retStat = OMget_ptr_val(fptr_id, (void **)&fp, 0);
	if (retStat != 1 || fp == NULL) {
		if (retStat == 1 && fp == NULL) retStat = 0;
		if (retStat == -1) {
			ERRerror("file_bin_get_real_val",1,ERR_CHAIN,
				 "file pointer is not set: %s",
				 OMret_obj_path(elem_id,buf1,sizeof(buf1)));
		}
		return(retStat);
	}
	start = offset;

	mode = file_array_ascii_bin(elem_id);

	if (mode == XDR_MODE) {
		xdrstdio_create (&xdrs, fp, XDR_DECODE);
	}

	retStat = fseek (fp, start, SEEK_SET);
	if (retStat != 0) {
		ERRerror("file_bin_get_real_val",2,ERR_CHAIN,
			 "fseek failed to position at %ld: %s", start,
			 OMret_obj_path(elem_id,buf1,sizeof(buf1)));
		return(-1);
	}
	switch (type) {
	      case DTYPE_DOUBLE:
		if (mode == BIN_MODE)
			retStat = (int)fread(&ddata, (size_t)DTYPEtype_size[type], (size_t)1, fp);
		else
			retStat = xdr_double (&xdrs, &ddata);

		if (retStat != 1) {
			ERRerror("file_bin_get_real_val",1,ERR_CHAIN,
				 "fread failed to read object: %s",
				 OMret_obj_path(elem_id,buf1,sizeof(buf1)));
			return(-1);
		}
		*val = ddata;
		break;
	      case DTYPE_FLOAT:
		if (mode == BIN_MODE)
			retStat = (int)fread(&fdata, (size_t)DTYPEtype_size[type], (size_t)1, fp);
		else
			retStat = xdr_float (&xdrs, &fdata);

		if (retStat != 1) {
			ERRerror("file_bin_get_real_val",1,ERR_CHAIN,
				 "fread failed to read object: %s",
				 OMret_obj_path(elem_id,buf1,sizeof(buf1)));
			return(-1);
		}
		*val = (double)fdata;
		break;
	      case DTYPE_LONG:
		if (mode == BIN_MODE)
			retStat = (int)fread(&ldata, (size_t)DTYPEtype_size[type], (size_t)1, fp);
		else
			retStat = xdr_long (&xdrs, &ldata);

		if (retStat != 1) {
			ERRerror("file_bin_get_real_val",1,ERR_CHAIN,
				 "fread failed to read object: %s",
				 OMret_obj_path(elem_id,buf1,sizeof(buf1)));
			return(-1);
		}
		*val = (double)ldata;
		break;
	      case DTYPE_INT:
		if (mode == BIN_MODE)
			retStat = (int)fread(&idata, (size_t)DTYPEtype_size[type], (size_t)1, fp);
		else
			retStat = xdr_int (&xdrs, &idata);

		if (retStat != 1) {
			ERRerror("file_bin_get_real_val",1,ERR_CHAIN,
				 "fread failed to read object: %s",
				 OMret_obj_path(elem_id,buf1,sizeof(buf1)));
			return(-1);
		}
		*val = (double)idata;
		break;
	      case DTYPE_SHORT:
		if (mode == BIN_MODE)
			retStat = (int)fread(&sdata, (size_t)DTYPEtype_size[type], (size_t)1, fp);
		else
			retStat = xdr_short (&xdrs, &sdata);

		if (retStat != 1) {
			ERRerror("file_bin_get_real_val",1,ERR_CHAIN,
				 "fread failed to read object: %s",
				 OMret_obj_path(elem_id,buf1,sizeof(buf1)));
			return(-1);
		}
		*val = (double)sdata;
		break;
	      case DTYPE_BYTE:
		retStat = (int)fread(&bdata, (size_t)DTYPEtype_size[type], (size_t)1, fp);

		if (retStat != 1) {
			ERRerror("file_bin_get_real_val",1,ERR_CHAIN,
				 "fread failed to read object: %s",
				 OMret_obj_path(elem_id,buf1,sizeof(buf1)));
			return(-1);
		}
		*val = (double)bdata;
		break;
	      case DTYPE_CHAR:
		retStat = (int)fread(&cdata, (size_t)DTYPEtype_size[type], (size_t)1, fp);

		if (retStat != 1) {
			ERRerror("file_bin_get_real_val",1,ERR_CHAIN,
				 "fread failed to read object: %s",
				 OMret_obj_path(elem_id,buf1,sizeof(buf1)));
			return(-1);
		}
		*val = (double)cdata;
		break;
	      default:
		ERRerror("file_bin_get_real_val",2,ERR_CHAIN,
			 "unknown data type %d: %s", (int)type,
			 OMret_obj_path(elem_id,buf1,sizeof(buf1)));
		return(-1);
	}
	if (mode == XDR_MODE) {
		xdr_destroy(&xdrs);
	}
	return(1);
}

/* 64-bit porting. Directly Modified */
int file_get_array_dims (OMobj_id elem_id, int *ndim, xp_long *dims)
{
	char *array;
	int retStat, type, ascii_bin;
	char buf1[128];

	if (dims == NULL) {
		/* don't compute dims, just return ndim... */
		*ndim = 1;
		return(1);
	}
	ascii_bin = file_array_ascii_bin(elem_id);

	if (ascii_bin == ASCII_MODE) {
		*ndim = 0;
		type = DTYPE_UNSET;
		array = NULL;
		retStat = file_ascii_get_array(elem_id, &type, &array, ndim, dims, OM_GET_ARRAY_RD);
		if (array)
			ARRfree(array);
		return(retStat);
	}
	else if (ascii_bin == BIN_MODE ||  ascii_bin == XDR_MODE )
		return(file_get_array_dims_bin(elem_id, ndim, dims));
	else {
		ERRerror("file_get_array_dims",1,ERR_CHAIN,
			 "invalid ascii/binary parameter : %s",
			 OMret_obj_path(elem_id,buf1,sizeof(buf1)));
		return(-1);
	}

}

/* 64-bit porting. Directly Modified */
int file_get_array_dims_bin(OMobj_id elem_id, int *ndim, xp_long *dims)
{
	OMobj_id file_id, fptr_id;
	FILE *fp;
	int retStat, offset, type, done;
	xp_long size, count, start;
	char buf1[128];
	unsigned char *array;
	int stride;
	int mode;

	*ndim = 1;
	if (dims == NULL) {
		/* don't compute dims, just return ndim... */
		return(1);
	}
	file_id = OMfind_subobj(elem_id,OMstr_to_name("file"),OM_OBJ_RD);
	if (OMis_null_obj(file_id))
		return(0);
	if ((retStat = OMget_obj_val(file_id,&file_id)) != 1) {
		if (retStat == -1) {
			ERRerror("get_dims_bin_array",1,ERR_CHAIN,
				 "can't get file object: %s",
				 OMret_obj_path(elem_id,buf1,sizeof(buf1)));
		}
		return(retStat);
	}
	retStat = OMget_name_int_val (elem_id, OMstr_to_name("type"), &type);
	if (retStat != 1) {
		if (retStat == -1) {
		   ERRerror("get_dims_bin_array",1,ERR_CHAIN,
			    "can't get file type: %s",
			    OMret_obj_path(elem_id,buf1,sizeof(buf1)));
		}
		return(retStat);
	}

	retStat = OMget_name_int_val (elem_id, OMstr_to_name("offset"), &offset);
	if (retStat != 1) {
		if (retStat == -1) {
		   ERRerror("get_dims_bin_array",1,ERR_CHAIN,
			    "can't get file offset: %s",
			    OMret_obj_path(elem_id,buf1,sizeof(buf1)));
		}
		return(retStat);
	}
	retStat = OMget_name_int_val (elem_id, OMstr_to_name("stride"), &stride);
	if (retStat != 1)
		stride = 0;

	fptr_id = OMfind_subobj(file_id, OMstr_to_name("fptr"), OM_OBJ_RD);
	if (OMis_null_obj(fptr_id)) {
		ERRerror("get_dims_bin_array",1,ERR_CHAIN,
			 "can't get file pointer object: %s",
			 OMret_obj_path(elem_id,buf1,sizeof(buf1)));
		return(-1);
	}
	retStat = OMget_ptr_val(fptr_id, (void **)&fp, 0);
	if (retStat != 1 || fp == NULL) {
		if (retStat == 1 && fp == NULL) retStat = 0;
		if (retStat == -1) {
			ERRerror("get_dims_bin_array",1,ERR_CHAIN,
				 "file pointer is not set: %s",
				 OMret_obj_path(elem_id,buf1,sizeof(buf1)));
		}
		return(retStat);
	}
 	dims[0] = 0;
	start = offset;
	size = BLOCK_SIZE;
	array = (unsigned char *)ARRalloc(NULL, type, size, NULL);
	if (array == NULL) {
		ERRerror("get_dims_bin_array",2,ERR_CHAIN,
			"unable to allocate %ld elements for array: %s",size,
			OMret_obj_path(elem_id,buf1,sizeof(buf1)));
		return(-1);
	}

	mode = file_array_ascii_bin(elem_id);

	retStat = fseek (fp, start, SEEK_SET);
	if (retStat != 0) {
		ERRerror("get_dims_bin_array",2,ERR_CHAIN,
			 "fseek failed to position at %ld: %s", start,
			 OMret_obj_path(elem_id,buf1,sizeof(buf1)));
		return(-1);
	}
	done = 0;
	while (!done) {
		if (mode == BIN_MODE)
			count = fread(array, (size_t)DTYPEtype_size[type], (size_t)size, fp);
		else
			count = fread(array, (size_t)XDR_TypeSize[type], (size_t)size, fp);
		dims[0] += count;
		if (count != size)
			done =1;
	}
	if (stride > 1)
		dims[0] = (dims[0]+stride-1)/stride;
	ARRfree(array);
	return(1);
}


/* 64-bit porting. Directly Modified */
int read_bin_array(OMobj_id elem_id, int ndim, xp_long *dims, xp_long *min_rng,
                   xp_long *max_rng, unsigned char **array, int *type)
{
	OMobj_id file_id, fptr_id;
	FILE *fp;
	int i, offset;
	xp_long  retStat, start, end, size;
	xp_long prods[OM_ARRAY_MAXDIM];
	char buf1[128];
	int mode;
	int stride;
	xp_long n;
	XDR xdrs;
	char *addr;

	file_id = OMfind_subobj(elem_id,OMstr_to_name("file"),OM_OBJ_RD);
	if (OMis_null_obj(file_id))
		return(0);
	if ((retStat = OMget_obj_val(file_id,&file_id)) != 1) {
		if (retStat == -1) {
			ERRerror("read_bin_array",1,ERR_CHAIN,
				 "can't get file object: %s",
				 OMret_obj_path(elem_id,buf1,sizeof(buf1)));
		}
		return((int)retStat);
	}
	retStat = OMget_name_int_val (elem_id, OMstr_to_name("type"), type);
	if (retStat != 1) {
		if (retStat == -1) {
		   ERRerror("read_bin_array",1,ERR_CHAIN,
			    "can't get file type: %s",
			    OMret_obj_path(elem_id,buf1,sizeof(buf1)));
		}
		return((int)retStat);
	}

	retStat = OMget_name_int_val (elem_id, OMstr_to_name("offset"), &offset);
	if (retStat != 1) {
		if (retStat == -1) {
		   ERRerror("read_bin_array",1,ERR_CHAIN,
			    "can't get file offset: %s",
			    OMret_obj_path(elem_id,buf1,sizeof(buf1)));
		}
		return((int)retStat);
	}

	retStat = OMget_name_int_val (elem_id, OMstr_to_name("stride"), &stride);
	if (retStat != 1)
		stride = 0;

	fptr_id = OMfind_subobj(file_id, OMstr_to_name("fptr"), OM_OBJ_RD);
	if (OMis_null_obj(fptr_id)) {
		ERRerror("read_bin_array",1,ERR_CHAIN,
			 "can't get file pointer object: %s",
			 OMret_obj_path(elem_id,buf1,sizeof(buf1)));
		return(-1);
	}
	retStat = OMget_ptr_val(fptr_id, (void **)&fp, 0);
	if (retStat != 1 || fp == NULL) {
		if (retStat == 1 && fp == NULL) retStat = 0;
		if (retStat == -1) {
			ERRerror("read_bin_array",1,ERR_CHAIN,
				 "file pointer is not set: %s",
				 OMret_obj_path(elem_id,buf1,sizeof(buf1)));
		}
		return((int)retStat);
	}
	prods[0] = 1;
	for (i=1; i<ndim; i++)
		prods[i] = prods[i-1]*dims[i-1];

	for (start=0,end=0, i=0; i<ndim; i++) {
		start += min_rng[i]*prods[i];
		end   += (max_rng[i]-1)*prods[i];
	}
	size = (end - start+1);
	start = offset + start * DTYPEtype_size[*type];

	mode = file_array_ascii_bin(elem_id);

	if (mode == XDR_MODE) {
		xdrstdio_create (&xdrs, fp, XDR_DECODE);
	}

	retStat = fseek (fp, start, SEEK_SET);
	if (retStat != 0) {
		ERRerror("read_bin_array",2,ERR_CHAIN,
			 "fseek failed to position at %ld: %s", start,
			 OMret_obj_path(elem_id,buf1,sizeof(buf1)));
		return(-1);
	}

	*array = (unsigned char *)ARRalloc(NULL, *type, size, NULL);
	if (*array == NULL) {
		ERRerror("read_bin_array",2,ERR_CHAIN,
			"unable to allocate %ld elements for array: %s",size,
			OMret_obj_path(elem_id,buf1,sizeof(buf1)));
		return(-1);
	}
	if (mode == BIN_MODE || *type == DTYPE_BYTE || *type == DTYPE_CHAR) {
		if (stride <= 1)
			retStat = fread(*array, (size_t)DTYPEtype_size[*type], (size_t)size, fp);
		else {
			addr = (char *)*array;
			for (n=0; n<size; n++) {
				retStat = fread(addr, (size_t)DTYPEtype_size[*type], (size_t)1, fp);
				if (retStat != 1)
					break;

				fseek(fp, (stride-1)*DTYPEtype_size[*type], SEEK_CUR);
				addr += DTYPEtype_size[*type];
			}
			if (retStat != 1)
				retStat = 0;
			else
				retStat = size;
		}
	}
	else {
		addr = (char *)*array;
		switch (*type) {
		      case DTYPE_INT:
			for (n=0; n<size; n++) {
				if ( (retStat = xdr_int (&xdrs, (int *)addr)) != 1)
					break;
				addr += DTYPEtype_size[*type];
				if (stride > 1)
					fseek(fp, (stride-1)*XDR_TypeSize[*type], SEEK_CUR);
			}
			break;
		      case DTYPE_LONG:
			for (n=0; n<size; n++) {
				if ( (retStat = xdr_long (&xdrs, (xp_long *)addr)) != 1)
					break;
				addr += DTYPEtype_size[*type];
				if (stride > 1)
					fseek(fp, (stride-1)*XDR_TypeSize[*type], SEEK_CUR);
			}
			break;
		      case DTYPE_FLOAT:
			for (n=0; n<size; n++) {
				if ( (retStat = xdr_float (&xdrs, (float *)addr)) != 1)
					break;
				addr += DTYPEtype_size[*type];
				if (stride > 1)
					fseek(fp, (stride-1)*XDR_TypeSize[*type], SEEK_CUR);
			}
			break;
		      case DTYPE_DOUBLE:
			for (n=0; n<size; n++) {
				if ( (retStat = xdr_double (&xdrs, (double *)addr)) != 1)
					break;
				addr += DTYPEtype_size[*type];
				if (stride > 1)
					fseek(fp, (stride-1)*XDR_TypeSize[*type], SEEK_CUR);
			}
			break;
		      case DTYPE_SHORT:
			for (n=0; n<size; n++) {
				if ( (retStat = xdr_short (&xdrs, (short *)addr)) != 1)
					break;
				addr += DTYPEtype_size[*type];
				if (stride > 1)
					fseek(fp, (stride-1)*XDR_TypeSize[*type], SEEK_CUR);
			}
			break;
		      default:
			ERRerror("file_bin_get_real_val",2,ERR_CHAIN,
				 "unknown data type %d: %s", (int)*type,
				 OMret_obj_path(elem_id,buf1,sizeof(buf1)));
			return(-1);
		}
		if (retStat != 1)
			retStat = 0;
		else
			retStat = size;
	}
	if (mode == XDR_MODE) {
		xdr_destroy(&xdrs);
	}
	if (retStat != size) {
		ERRerror("read_bin_array",2,ERR_CHAIN,
			 "fread failed to read %ld elements: %s", size,
			 OMret_obj_path(elem_id,buf1,sizeof(buf1)));
		ARRfree(*array);
		return(-1);
	}
	return(1);
}


/* 64-bit porting. Directly Modified */
int file_bin_sub_array(OMobj_id elem_id, int ndim, xp_long *dims,
                       xp_long *min_rng, xp_long *max_rng, void *array, int type,
                       OMsub_array_func func)
{

	unsigned char *arr_ptr;
	xp_long *n_dims, *n_min_rng, *n_max_rng;
	xp_long m_dims[OM_ARRAY_MAXDIM];
	xp_long m_min_rng[OM_ARRAY_MAXDIM], m_max_rng[OM_ARRAY_MAXDIM];
	int r_ndim;
	xp_long r_dims[OM_ARRAY_MAXDIM], r_min_rng[OM_ARRAY_MAXDIM], r_max_rng[OM_ARRAY_MAXDIM];
	int arr_type, retStat, i;
	xp_long arr_size;
	char buf1[128];

	if (ndim == 0) return(1);

	n_dims = m_dims;
	n_min_rng = m_min_rng;
	n_max_rng = m_max_rng;
	memcpy(m_dims,    dims,    ndim*sizeof(xp_long));
	memcpy(m_min_rng, min_rng, ndim*sizeof(xp_long));
	memcpy(m_max_rng, max_rng, ndim*sizeof(xp_long));

	for (i=0; i<ndim; i++) {
		if (i==0)
			retStat = 1;
		if (retStat == 1 && n_dims[i] == n_max_rng[i]-n_min_rng[i] && i<ndim-1) {
			n_dims[i+1] = n_dims[i]*n_dims[i+1];
			n_min_rng[i+1] *= n_dims[i];
			n_max_rng[i+1] *= n_dims[i];
			n_dims[i] = 1;
		}
		else {
			retStat = 0;
		}
	}
	if (ndim != 1) {
		for (r_ndim=0,i=0; i<ndim; i++) {
			if (n_dims[i] != 1) {
				r_dims[r_ndim] = n_dims[i];
				r_min_rng[r_ndim] = n_min_rng[i];
				r_max_rng[r_ndim] = n_max_rng[i];
				r_ndim++;
			}
		}
		if (r_ndim == 1) {
			retStat = read_bin_array(elem_id, ndim, dims, min_rng, max_rng, &arr_ptr, &arr_type);
			if (retStat != 1) {
				if (retStat == -1) {
					ERRerror("file_get_sub_array",1,ERR_CHAIN,
						 "can't read file array: %s",
						 OMret_obj_path(elem_id,buf1,sizeof(buf1)));
				}
				return(retStat);
			}

			DTYPEcvt_copy_array((unsigned char *)array, type, arr_ptr,
					    arr_type,r_max_rng[0] - r_min_rng[0]);
			ARRfree(arr_ptr);
			return(1);
		}
		if (r_ndim == 2) {
			retStat = read_bin_array(elem_id, ndim, dims, min_rng, max_rng, &arr_ptr, &arr_type);
			if (retStat != 1) {
				if (retStat == -1) {
					ERRerror("file_get_sub_array",1,ERR_CHAIN,
						 "can't read file array: %s",
						 OMret_obj_path(elem_id,buf1,sizeof(buf1)));
				}
				return(retStat);
			}

			DTYPEcvt_copy_array_2d((unsigned char *)array, type, arr_ptr,
					       arr_type,r_dims, r_min_rng, r_max_rng, 0);
			ARRfree(arr_ptr);
			return(1);
		}
		else if (r_ndim == 3) {
			retStat = read_bin_array(elem_id, ndim, dims, min_rng, max_rng, &arr_ptr, &arr_type);
			if (retStat != 1) {
				if (retStat == -1) {
					ERRerror("file_get_sub_array",1,ERR_CHAIN,
						 "can't read file array: %s",
						 OMret_obj_path(elem_id,buf1,sizeof(buf1)));
				}
				return(retStat);
			}

			DTYPEcvt_copy_array_3d((unsigned char *)array, type, arr_ptr,
					       arr_type,r_dims, r_min_rng, r_max_rng, 0);
			ARRfree(arr_ptr);
			return(1);
		}
		else {
			arr_size = 1;
			for (i = 0; i < ndim; i++) arr_size *= dims[i];
			retStat=OMcvt_sub_array(elem_id,ndim,dims,min_rng,max_rng,1,&arr_size,
					     (char *)array,DTYPEtype_size[type],
					     (OMsub_array_func)func);
			if (retStat == -2) {
				return(1);
			}
			if (retStat != 1) {
				return(retStat);
			}
		}
	}
	retStat = read_bin_array(elem_id, ndim, dims, min_rng, max_rng, &arr_ptr, &arr_type);
	if (retStat != 1) {
		if (retStat == -1) {
			ERRerror("file_get_sub_array",1,ERR_CHAIN,
				 "can't read file array: %s",
				 OMret_obj_path(elem_id,buf1,sizeof(buf1)));
		}
		return(retStat);
	}

	DTYPEcvt_copy_array((unsigned char *)array, type, arr_ptr,
			    arr_type, n_max_rng[0] - n_min_rng[0]);
	ARRfree(arr_ptr);

	return(1);
}

int file_array_ascii_bin(OMobj_id elem_id)
{
	int retStat, ascii_binary;
	char buf1[128];

	retStat = OMget_name_int_val (elem_id, OMstr_to_name("ascii_binary"), &ascii_binary);
	if (retStat != 1) {
		if (retStat == -1) {
			ERRerror("file_ascii_bin",1,ERR_CHAIN,
				 "can't get file ascii_binary: %s",
				 OMret_obj_path(elem_id,buf1,sizeof(buf1)));
		}
		return(retStat);
	}
	return(ascii_binary);
}

int file_valid_obj (OMobj_id obj_id, OMobj_id templ_id, int mode)
{

	OMobj_id sub_id;

	if (OMget_array_subobj(obj_id, 0, &sub_id,OM_OBJ_RD) == 1 &&
	    (!(mode & OM_VALID_REQ) ||
	     OMget_obj_atts(sub_id, OM_atts_req) == 1) &&
	    OMvalid_obj(sub_id, OMnull_obj,
			OM_VALID_OBJ_DOWN | (mode & OM_VALID_REQ)) != 1)
		return(0);

	if (OMget_array_subobj(obj_id, 1, &sub_id,OM_OBJ_RD) == 1 &&
	    (!(mode & OM_VALID_REQ) ||
	     OMget_obj_atts(sub_id, OM_atts_req) == 1) &&
	    OMvalid_obj(sub_id, OMnull_obj,
			OM_VALID_OBJ_DOWN | (mode & OM_VALID_REQ)) != 1)
		return(0);

	if (OMget_array_subobj(obj_id, 2, &sub_id,OM_OBJ_RD) == 1 &&
	    (!(mode & OM_VALID_REQ) ||
	     OMget_obj_atts(sub_id, OM_atts_req) == 1) &&
	    OMvalid_obj(sub_id, OMnull_obj,
			OM_VALID_OBJ_DOWN | (mode & OM_VALID_REQ)) != 1)
		return(0);

	if (OMget_array_subobj(obj_id, 4, &sub_id,OM_OBJ_RD) == 1 &&
	    (!(mode & OM_VALID_REQ) ||
	     OMget_obj_atts(sub_id, OM_atts_req) == 1) &&
	    OMvalid_obj(sub_id, OMnull_obj,
			OM_VALID_OBJ_DOWN | (mode & OM_VALID_REQ)) != 1)
		return(0);

	return(1);
}
