//			Copyright (c) 1993 by
//			Advanced Visual Systems Inc.
//			All Rights Reserved
//
//	This software comprises unpublished confidential information of
//	Advanced Visual Systems Inc. and may not be used, copied or made
//	available to anyone, except in accordance with the license
//	under which it is furnished.
//
//      This file is under CVS/RCS control in:
//      $Id: //depot/express/fcs70/include/avs/cxxutl/array.cxx#1 $
//

// array.cxx - Source file for UtArrayRep and UtArray

#ifndef CXXUTIL_ARRAY_CXX
#define CXXUTIL_ARRAY_CXX

//----------------------------------------------------------------------
#ifndef CXXUTIL_ARRAY_HXX
#include "array.hxx"
#endif

#ifndef CXXUTIL_MISC_HXX
#include "misc.hxx"
#endif

#include <string.h>


/* Get rid of definitions for these commonly used macros */
#ifdef min
#undef min
#endif

#ifdef max
#undef max
#endif

//----------------------------------------------------------------------
// UtArrayRep
//----------------------------------------------------------------------

//----------------------------------------------------------------------
template <class TYPE>
UtArrayRep<TYPE>::UtArrayRep()
    : _size(0L), _numAlloc(0L), _readOnly(0), _array(0L),
      _refCount(0), _extremesValid(0), _freeData(0L)
//----------------------------------------------------------------------
{}

//----------------------------------------------------------------------
/* 64-bit porting. Directly Modified */
template <class TYPE>
UtArrayRep<TYPE>::UtArrayRep(
    xp_long size)
    : _size(0L), _numAlloc(0L), _readOnly(0), _array(0L),
      _refCount(0), _extremesValid(0), _freeData(0L)
//----------------------------------------------------------------------
{
    _resize(size);
}

//----------------------------------------------------------------------
/* 64-bit porting. Directly Modified */
template <class TYPE>
UtArrayRep<TYPE>::UtArrayRep(
    const TYPE *array,
    xp_long size,
    int readOnly)
    : _size(0L), _numAlloc(0L), _readOnly(readOnly),  _refCount(0),
      _extremesValid(0), _freeData(0L)
//----------------------------------------------------------------------
{
    // Check for null pointer and valid size
    if (array && size>0) {
        // Check for read only data
        if (readOnly) {
             _numAlloc = _size = size;
             _array = (TYPE*)array;
        }
        else {
            _resize(size);
            for (xp_long i=0; i<_size; i++) {
                _array[i] = array[i];
            }
        }
    }
}

//----------------------------------------------------------------------
/* 64-bit porting. Directly Modified */
template <class TYPE>
UtArrayRep<TYPE>::UtArrayRep(
    const TYPE *array,
    xp_long size,
    UtArrayFree *freeData)
    : _size(0L), _numAlloc(0L), _readOnly(0), _refCount(0),
      _extremesValid(0), _freeData(freeData)
//----------------------------------------------------------------------
{
    // Check for null pointer
    if (array) {
        // Check for external free function
        if (_freeData) {
             _numAlloc = _size = size;
             _array = (TYPE*)array;
        }
        _resize(size);
        for (xp_long i=0; i<_size; i++) {
            _array[i] = array[i];
        }
    }
}

//----------------------------------------------------------------------
template <class TYPE>
UtArrayRep<TYPE>::~UtArrayRep()
//----------------------------------------------------------------------
{
    // Dont free if readOnly memory
    if (_numAlloc && !_readOnly) {
        // Use free function if provided
        if (_freeData && _array) {
            _freeData((void*)_array);
            _freeData = 0;
        } else {
            delete [] _array;
        }
    }
}

//----------------------------------------------------------------------
// Set entire array to new array
//----------------------------------------------------------------------
/* 64-bit porting. Directly Modified */
template <class TYPE>
void
UtArrayRep<TYPE>::_set(
    const TYPE *array,
    xp_long newSize)
//----------------------------------------------------------------------
{
    // Check for null pointer
    if (array) {
        if (_numAlloc < newSize) {
            _size = 0;
            _resize(newSize);
            _size = newSize;
        } else {
            _size = newSize;
        }
        for (xp_long i=0; i<_size; i++) {
            _array[i] = array[i];
        }
        _extremesValid = 0;
    }
}

//----------------------------------------------------------------------
// return extremes of array, return = 0 if invalid
//----------------------------------------------------------------------
/* 64-bit porting. Only Modified Internally */
template <class TYPE>
void
UtArrayRep<TYPE>::_updateExtremes()
//----------------------------------------------------------------------
{
    // Check for calculation of extremes
    if (!_extremesValid && _size > 0) {
        _min = _max = _array[0];
        if (_size >= 1) {
            for (xp_long i=1; i<_size; i++) {
                TYPE value = _array[i];
                if (value < _min) _min = value;
                if (value > _max) _max = value;
            }
        }
        _extremesValid = 1;
    }
}

//----------------------------------------------------------------------
// Resize please note that setting it to 0 is legal
//----------------------------------------------------------------------
/* 64-bit porting. Directly Modified */
template <class TYPE>
void
UtArrayRep<TYPE>::_resize(
    xp_long newSize)
//----------------------------------------------------------------------
{
    // If same same as now then just return
    if (newSize == _size) return;
    // Is new size non zero
    if (newSize > 0) {
        // Realloc if outside range
        if (newSize > _numAlloc || newSize < _numAlloc-_addSize()) {
            // get new mem and copy data
            TYPE *tempArray = new TYPE[newSize];
            // copy old stuff if anything there
            if (_size) {
                _size = UtMin(_size, newSize);
                (void) memcpy(tempArray, _array,
                              _size * sizeof(TYPE));
            } else {
                _size = newSize;
            }
            // delete old array
            if (_numAlloc) {
                if (_freeData && _array) {
                    _freeData((void*)_array);
                    _freeData = 0;
                } else {
                    delete [] _array;
                }
            }
            _array = tempArray;
            _numAlloc = newSize;
        }
        else {
            _size = newSize;
        }
    }
    // If zero size then just clear
    else {
        if (_numAlloc) {
            if (_freeData && _array) {
                _freeData((void*)_array);
                _freeData = 0;
            } else {
                delete [] _array;
            }
            _size=_numAlloc=0;
        }
    }
    _extremesValid = 0;
}

//----------------------------------------------------------------------
/* 64-bit porting. Directly Modified */
template <class TYPE>
void
UtArrayRep<TYPE>::_extendSize(
    xp_long index)
//----------------------------------------------------------------------
{
    if (index >= _size) {
        // beyond boundaries then extend size
        if (index > (_numAlloc-1)) {
            xp_long numAlloc = _numAlloc + _addSize();
            _resize(numAlloc);
            _size = index+1;
        }
        else {
            // be sure _size is correct
            _size = index+1;
        }
    }
    _extremesValid = 0;
}


//----------------------------------------------------------------------
// UtArray
//----------------------------------------------------------------------
template <class TYPE>
UtArray<TYPE>::UtArray()
//----------------------------------------------------------------------
{
    _arrayRep = new UtArrayRep<TYPE>;
}

//----------------------------------------------------------------------
/* 64-bit porting. Directly Modified */
template <class TYPE>
UtArray<TYPE>::UtArray(
    xp_long size)
//----------------------------------------------------------------------
{
    _arrayRep = new UtArrayRep<TYPE>(size);
}

//----------------------------------------------------------------------
/* 64-bit porting. Directly Modified */
template <class TYPE>
UtArray<TYPE>::UtArray(
    const TYPE *array,
    xp_long size,
    int readOnly)
//----------------------------------------------------------------------
{
    _arrayRep = new UtArrayRep<TYPE>(array, size, readOnly);
}

//----------------------------------------------------------------------
/* 64-bit porting. Directly Modified */
template <class TYPE>
UtArray<TYPE>::UtArray(
    const TYPE *array,
    xp_long size,
    UtArrayFree *freeData)
//----------------------------------------------------------------------
{
    _arrayRep = new UtArrayRep<TYPE>(array, size, freeData);
}

//----------------------------------------------------------------------
// Copy contructor
//----------------------------------------------------------------------
template <class TYPE>
UtArray<TYPE>::UtArray(
    const UtArray<TYPE> &from)
//----------------------------------------------------------------------
{
    _arrayRep = from._arrayRep;
    _arrayRep->_refCount++;
}

//----------------------------------------------------------------------
template <class TYPE>
UtArray<TYPE>::~UtArray()
//----------------------------------------------------------------------
{
    _delete();
}

//----------------------------------------------------------------------
template <class TYPE>
void
UtArray<TYPE>::_delete()
//----------------------------------------------------------------------
{
    // Check for more references
    if (_arrayRep->_refCount > 0) {
        _arrayRep->_refCount--;
    }
    // If no more references then delete
    else {
        delete _arrayRep;
    }
}

//----------------------------------------------------------------------
// Assignment operator
//----------------------------------------------------------------------
template <class TYPE>
UtArray<TYPE>&
UtArray<TYPE>::operator=(
    const UtArray<TYPE> &from)
//----------------------------------------------------------------------
{
    _delete();
    _arrayRep = from._arrayRep;
    _arrayRep->_refCount++;
    return *this;
}

//----------------------------------------------------------------------
// Set array
//----------------------------------------------------------------------
/* 64-bit porting. Directly Modified */
template <class TYPE>
void
UtArray<TYPE>::set(
    const TYPE array[],
    xp_long size,
    int readOnly)
//----------------------------------------------------------------------
{
    // If readonly we dont copy the data
    if (readOnly) {
        _delete();
        _arrayRep = new UtArrayRep<TYPE>(array, size, readOnly);
    } else {
        // If more are referencing the array or it is shared readonly then
        // make a new one and copy the data to it
        if (_arrayRep->_refCount != 0) {
            _delete();
            _arrayRep = new UtArrayRep<TYPE>(array, size);
        }
        // Ohterwise just modify the one we got
        else {
            _arrayRep->_set(array, size);
        }
    }
}

//----------------------------------------------------------------------
// Set array with user defined free function
//----------------------------------------------------------------------
/* 64-bit porting. Directly Modified */
template <class TYPE>
void
UtArray<TYPE>::set(
    const TYPE array[],
    xp_long size,
    UtArrayFree *freeData)
//----------------------------------------------------------------------
{
    _delete();
    _arrayRep = new UtArrayRep<TYPE>(array, size, freeData);
}

//----------------------------------------------------------------------
// Index function for safe assign to operations
// It will automacally extend size and realloc memory when needed
//----------------------------------------------------------------------
/* 64-bit porting. Directly Modified */
template <class TYPE>
TYPE &
UtArray<TYPE>::operator[](
    xp_long index)
//----------------------------------------------------------------------
{
    // If only one reference and not readonly then just resize
    if (_arrayRep->_refCount == 0 && !_arrayRep->_readOnly) {
        if (index >= _arrayRep->_size) {
            _arrayRep->_extendSize(index);
        }
    }
    // If this is done for shared readonly memory or when more than one
    // reference the same memory then copy to new memory
    else {
        resize(UtMax(_arrayRep->_size, index+1));
    }
    return _arrayRep->_array[index];
}

//----------------------------------------------------------------------
// return min data value
//----------------------------------------------------------------------
template <class TYPE>
const TYPE
UtArray<TYPE>::min()
//----------------------------------------------------------------------
{
    _arrayRep->_updateExtremes();
    return _arrayRep->_min;
}

//----------------------------------------------------------------------
// return max data value
//----------------------------------------------------------------------
template <class TYPE>
const TYPE
UtArray<TYPE>::max()
//----------------------------------------------------------------------
{
    _arrayRep->_updateExtremes();
    return _arrayRep->_max;
}

//----------------------------------------------------------------------
// Remove all from array
//----------------------------------------------------------------------
template <class TYPE>
void
UtArray<TYPE>::clear()
//----------------------------------------------------------------------
{
    _delete();
    _arrayRep = new UtArrayRep<TYPE>;
}

//----------------------------------------------------------------------
// Resize please note that setting it to 0 is legal
// The resize function will preserve any exsisting data
// in index [0] - [newSize-1]
//----------------------------------------------------------------------
/* 64-bit porting. Directly Modified */
template <class TYPE>
void
UtArray<TYPE>::resize(
    xp_long newSize)
//----------------------------------------------------------------------
{
    // If only one reference and not readonly then just resize
    if (_arrayRep->_refCount == 0 && !_arrayRep->_readOnly) {
        _arrayRep->_resize(newSize);
    }
    // Othersize create a new one of the correct size
    else {
        UtArrayRep<TYPE> *newArrayRep
            = new UtArrayRep<TYPE>(_arrayRep->_array,
                                   UtMin(newSize, _arrayRep->_size));
        _delete();
        _arrayRep = newArrayRep;
    }
}

#endif // DON'T ADD ANYTHING AFTER THIS #endif
