/*
			Copyright (c) 1994 by
			Advanced Visual Systems Inc.
			All Rights Reserved

	This software comprises unpublished confidential information of
	Advanced Visual Systems Inc. and may not be used, copied or made
	available to anyone, except in accordance with the license
	under which it is furnished.

	This file is under Perforce control
	$Id: //depot/express/fcs70/include/avs/dmap.h#1 $
*/
#ifndef dmap_h
#define dmap_h

#ifndef PORT_HEADER_IS_INCLUDED
#   include "avs/port.h"
#endif

#ifdef __cplusplus
#   ifndef array_h
#       include "avs/array.h"
#   endif
#   ifndef dbxx_h
#       include "avs/dbxx.h"
#   endif
#endif

#ifdef __cplusplus
extern "C" {
#endif


/* A color as defined by the Data mapping package
 */
typedef struct {
    float v1;
    float v2;
    float v3;
    float v4;
    unsigned int argb;
    unsigned int rgba;	/* OpenGL format */
} DmapColor;

/* User supplied interpolation function.
 */
typedef int (*DmapInterpolateFunc) (
#ifdef USE_PROTOS
    int,          /* The component (0 to 3) of a DmapColor to interpolate */
    double,       /* Minimum value in range for interpolation */
    double,       /* Maximum value in range for interpolation */
    double *,     /* The array of values to interpolate between */
    int,          /* Number of interpolation points (minimum of 2) */
    DmapColor *,  /* The resulting table (already allocated) */
    int           /* The size of the table */
#endif
    );

/* User supplied color conversion function
 */
typedef int (*DmapColorFunc) (
#ifdef USE_PROTOS
    DmapColor *,  /* The source array of DmapColors (already allocated) */
    DmapColor *,  /* The destination array of DmapColors (already allocated) */
    int           /* The number of colors to convert */
#endif
    );

#ifdef __cplusplus
}
#endif

#ifdef __cplusplus

/* DmapColor_c
 *    - This class is used to access a DmapColor as stored in the
 *      database.
 */
class DmapColor_c : public DBelem_c {
    DBreal _v1;
    DBreal _v2;
    DBreal _v3;
    DBreal _v4;

  public:
             DmapColor_c (DBelem_id id);
    virtual ~DmapColor_c ();

    double getv1 () { return _v1; }
    double getv2 () { return _v2; }
    double getv3 () { return _v3; }
    double getv4 () { return _v4; }
    DBreal & setv1 (double value) { _v1 = value; return _v1; }
    DBreal & setv2 (double value) { _v2 = value; return _v2; }
    DBreal & setv3 (double value) { _v3 = value; return _v3; }
    DBreal & setv4 (double value) { _v4 = value; return _v4; }
};

/* DmapValue
 *    - Each control point that exists in a range of data consists of a
 *      DmapColor plus the value that represents it.
 */
class DmapValue : public DmapColor_c {
    DBreal _value;

  public:
             DmapValue (DBelem_id);
    virtual ~DmapValue ();

    double getValue () { return _value; }
    DBreal & set_value (double value) { _value = value; return _value; }
};

/* DmapColorModel
 *    - Colors found in a data map are stored according to a user
 *      color. The DmapColorModel is used to access the current
 *      color model to renderer color model. For example, the renderer
 *      requires RGB colors while the current color model specified may
 *      be HSV.
 *
 *    - The various strings are used to aid with editors that need to
 *      display the color model information.
 */
class DmapColorModel : public DBelem_c {
    DBstring _model;           // The model name
    DBstring _v1Label;         // Labels for the color components.
    DBstring _v2Label;
    DBstring _v3Label;
    DBstring _v4Label;
    DBfunc   _convertToRGB;    // Current convertion functions.
    DBfunc   _convertFromRGB;

  public:
             DmapColorModel (DBelem_id id);
    virtual ~DmapColorModel ();

    char * getModel() { return _model; }
    char * getv1Label () { return _v1Label; }
    char * getv2Label () { return _v2Label; }
    char * getv3Label () { return _v3Label; }
    char * getv4Label () { return _v4Label; }

    int convertToRGB (DmapColor *from, DmapColor *to, int size);
    int convertFromRGB (DmapColor *from, DmapColor *to, int size);
};

/* DmapDataRangeModel
 *    - Just as the color model can be reconfigured, so can a model
 *      for creating the color lookup table for a range of data.
 */
class DmapDataRangeModel : public DBelem_c {
    DBstring _model;             // The model name
    DBfunc   _interpolate;       // The function that computes the table

  public:
             DmapDataRangeModel (DBelem_id id);
    virtual ~DmapDataRangeModel ();

    char * getModel () { return _model; }

    int interpolate (
        int component,                // Component in the table to update
        double minimum,               // Minimum value in range
        double maximum,               // Maximum value in range
        DBelemVector & controlPoints, // Array of Datamap Values
        DmapColor * table,            // Table of Datamap Colors
        int size                      // Size of the table
    );
};

/* DmapDataRange
 *    - A Datamap is made up of one or more ranges of data that specify
 *      a mapping of values to colors.
 *    - Colors are separated into 4 components where each component has
 *      a dynamically specified interpolation function used to compute
 *      the color lookup.
 *    - Control points help with interpolation methods that require more
 *      than a minumum and a maximum.
 */
class DmapDataRange : public DBelem_c {
    float _minimum;               // minumum value of the range
    float _maximum;               // maximum value of the range
    int   _size;                  // size of the lookup table
    int   _changed;		  // contents of range have changed
    DBelemVector   _controlPoints;
    DmapDataRangeModel *_v1Model;  // Interpolation models for color components.
    DmapDataRangeModel *_v2Model;
    DmapDataRangeModel *_v3Model;
    DmapDataRangeModel *_v4Model;

    DmapColor *_table;		// The lookup table (no overhead lookups)
    float _delta;		// The delta between values in the table
    int _table_index;

    // Sequence numbers used to tell when things are changed.
    int _v1seq, _v2seq, _v3seq, _v4seq, _cpseq;

  public:
             DmapDataRange (DBelem_id id);
    virtual ~DmapDataRange ();

    // Recalculate the color table etc.
    virtual int  update ();

    float getMinimum () {return _minimum;}
    float getMaximum () {return _maximum;}
    int   getSize () { return _size; }
    int   getChanged () { return _changed; }
    int   getNumControlPoints () { return _controlPoints.size(); };

    // Return the color for an in-range value.
    DmapColor & getColor (float value) {
	_table_index = (int)(_delta*(value-_minimum));
	if (_table_index < _size)
            return _table[_table_index];
	else return _table[_size-1];
    }
    DmapColor & getMinColor () {return _table[0];}
    DmapColor & getMaxColor () {return _table[_size-1];}

    // Converts the table colors from the model
    void   convertColors (DmapColorModel &model, int changed);

    void  getCPInfo(int j, double *vals);

  private:
    // Update a model whose subelements are out of date.
    void   updateModel (DmapDataRangeModel **model, const char *name);

    // Build the packed ARGB representation from the float values.
    void buildARGBs (DmapColor * table, int size);
};


/* Dmap
 *    - A data map provides the means to map a value to a color. The
 *      map consists of a set of 1 or more ranges where each range
 *      typically defines a new way of interpreting a mapping.
 */
class Dmap : public DBelem_c {
    int             _numRanges;       // The size of ranges array.
    DBelemVector    _ranges;          // The database array of ranges.
    DmapColorModel *_colorModel;      // Currently selected color model.
    DBfunc _getColorsFunc;            // DB func that converts many colors.
    DBfunc _getARGBsFunc;             // DB func that converts many colors.
    DBfunc _getMinMaxFunc;            // DB func that converts many colors.
    DBelem_c _localPtr;               // Needed so that get_color.. can be
                                      // called from info in the database.
    DBArray _rangeArray;              // In memory versions of _ranges.

    // Sequence numbers used to determine when things change.
    int _rangesSeq, _rangesIdSeq, _colorModelSeq;
    int     _refCount;                // Used by geometry flipbook.
  public:
             Dmap (DBelem_id id);
    virtual ~Dmap ();

    // Functions that manipulate ranges.
    xp_long      getNumRanges () { return _rangeArray.size (); }

    // Update the ranges etc.
    virtual void      update ();

    static void       get_colors (
                          Dmap *dmap,
                          char *values,
                          float *colors,
			  int type,
                          xp_ulong size
                       );

    static void       get_argbs(
                          Dmap *dmap,
                          char *values,
                          unsigned int *colors,
			  int type,
                          unsigned int width,
                          unsigned int height,
                          unsigned int src_width,
                          unsigned int dst_width,
			  short *scans,
			  int flag		/* 0 = argb, 1 = rgba */
                       );

    static void       get_minmax(
                          Dmap *dmap,
			  double *minimum,
			  double *maximum
                       );

    xp_long		  getSize();
    int		      getChanged();
    void	      getRangeInfo(int i, int *rangeSize, int *numControlPoints);
    void	      getCPInfo(int i, int j, double *vals);

    inline void              inc_ref() { _refCount++;};
    inline int               dec_ref() { if (--_refCount < 1) {
                                           delete this;
                                           return 1;
                                         }
                                         return 0;
                                       }
  private:
    void	      buildRangeArray ();
};

#endif /* __cplusplus */

#endif /* dmap_h */
