//*-*-c++-*-*
//
//                      Copyright (c) 1996 by
//                      Advanced Visual Systems Inc.
//                      All Rights Reserved
//
//      This software comprises unpublished confidential information of
//      Advanced Visual Systems Inc. and may not be used, copied or made
//      available to anyone, except in accordance with the license
//      under which it is furnished.
//
//      This file is under Perforce control
//      $Id: //depot/express/fcs70/include/avs/gis/cia.h#1 $
//

#ifndef WORLDMAP_HEADER
#define WORLDMAP_HEADER

#ifdef XP_STD_IOSTREAM
#include <iostream>
using std::ostream;
#else
#include <iostream.h>
#endif

// Express file manipulation
#ifndef f_utils_h
#include <avs/f_utils.h>
#endif

#ifndef LATLON_HEADER
#include <avs/gis/latlon.h>
#endif

#ifndef CXXUTIL_STRING_HXX
#include <avs/cxxutl/string.hxx>
#endif

#ifndef CXXUTIL_VECTOR_HXX
#include <avs/cxxutl/vector.hxx>
#endif

//
//      The following enumerations refer to the CIA/USGS overlay subsections
//
#define TOTAL_OVERLAYS  5

enum overlaySections {
  geoBoundary = 0,
  polBoundary = 1,
  coastal = 2,
  river = 3 ,
  usgsPlates = 4,
  lastOS = 5
};


// Create an iterator operator for the overlaySections enum.
//
// Compilers vary as to whether or not they will accept operator++ for
// an enum, but they all accept an ordinary function overloaded on an enum.
//
void increment( overlaySections& );


//
//      The following enumerations refer to the CIA/USGS map subsections
//
#define TOTAL_MAPSECTIONS 5

enum mapSections {
  africa = 0,
  asia = 1,
  europe = 2,
  namerica = 3,
  samerica = 4,
  lastMS = 5
};


// Create an iterator operator for the overlaySections enum.
void increment( mapSections& );


//
//      The following enumerations refer to the various information levels
//
#define TOTAL_INFOLEVELS 18

enum infoLevels {
  everything = -2,
  none = -1,
  both = 0,
  all = 1,
  definite = 1,
  wm_major = 1,
  indefinite = 2,
  additionalMajor = 2,
  other = 3,
  intermediate = 3,
  wm_minor = 4,
  doubleLines = 5,
  intermittentMajor = 6,
  intermittentMinor = 7,
  reefs = 8,
  saltPans1 = 9,
  saltPans2 = 10,
  majorCanals = 10,
  lesserCanals = 11,
  irrigations = 12,
  shelves1 = 13,
  shelves2 = 14,
  glaciers = 15,
  lastIL = 16
};

// Create an iterator operator for the overlaySections enum type.
void increment(infoLevels&);


//
//      WorldMap class
//
//      This class holds pertinenant information regarding the 
//      CIA worldmap database, and the information the 
//      user wishes to gather from it. 
//
class WorldMap {
  // public types
public:
  typedef UtVector<LatLon *> LL_PolyLine;
#if defined(__hpux) && ( __cplusplus < 199707L )
  // Kludge for HP's pre-ANSI compiler.
  typedef UtVector<void *> LL_PolyLineCollection;
#else
  typedef UtVector<LL_PolyLine *> LL_PolyLineCollection;
#endif

  // private instances
private:
  // Information necessary for gathering the CIA data
  UtString      ciaDirectory;           // location of the CIA Map database

  UtString      Continent[TOTAL_MAPSECTIONS];   // Subdirectory for the continent
  UtString      Overlay[TOTAL_OVERLAYS];        // actual feature overlay files

  long          precision;              // In seconds
  int           Verbose;                // Verbosity flag

  long          currentFilePos;         // Current file location

  LatLon        *llBound;               // lowest lat/lon
  LatLon        *urBound;               // highest lat/lon

  // Store the offset for the longitude. This will give the map
  // a place to break it's polylines. Essential for flat world maps.
  LatLon        *longitudeOffset;	// Defaults to 0


  int           pointsOfInterest[TOTAL_OVERLAYS][TOTAL_INFOLEVELS];     // Parameters for what we want to see

  LL_PolyLineCollection polylineCollection[TOTAL_OVERLAYS];      // Ordered collection of Ordered Collections
                                                        // of polylines

  void          decimate(overlaySections);      // Degrade the data 

  int           readInMap(overlaySections sect,
                          mapSections maps,
                          UtString fname);     // Read in the data and load up points

  int           readMap(FILE *ptrFile, 
                        unsigned long *x);      // Pull a word out of the file
  int           readMap(FILE *ptrFile,
                        unsigned short *x);     // Pull a short out of the file
  int           readMap(FILE *ptrFile,
                        long *x);               // Pull a long out of the file

  void          nextPoint(FILE *ptrFile,
                          long *ixp, 
                          long *yxp);           // Retrieve the next data point

  int           withinExtents(long xlo, long ylo,
                              long xhi, long yhi,
                              long xpos, long ypos); // Does the point fall in the extents?


  // protected instances
protected:

  // public instances
public:

  // Constructors and destructors
  // new directory, all else defaults
  WorldMap();
  WorldMap(UtString dir);
  ~WorldMap();                          // destroys any collections

  // set some of the internal CIA information
  void          setCIADirectory(const UtString& dir)    { ciaDirectory = dir; }
  void          setContinentDir(mapSections sect, 
                                const UtString& cont)   { Continent[sect] = cont; }
  void          setOverlaysFile(overlaySections sect,
                                const UtString& over)   { Overlay[sect] = over; }
  void          setLowLeftBoundary(LatLon *ll)          { delete llBound; llBound = ll; }
  void          setUpRightBoundary(LatLon *ur)          { delete urBound; urBound = ur; }
  void          setPrecision(long sec)                  { precision = sec; }

  // retrieve some of the CIA information
  UtString      fetchCIADirectory()                     { return ciaDirectory; }
  UtString      fetchContinentDir(mapSections sect)     { return Continent[sect]; }
  UtString      fetchOverlaysFile(overlaySections sect) { return Overlay[sect]; }
  LatLon        *fetchLowLeftBoundary()                 { return llBound; }
  LatLon        *fetchUpRightBoundary()                 { return urBound; }

  int           insideBoundary(LatLon);     // Returns TRUE or FALSE
  long          fetchPrecision()                        { return precision; }

  // set various information levels of the map files
  void          setPOI(overlaySections overlay,
                       infoLevels      information,
                       int             flag) { pointsOfInterest[overlay][information] = flag; }
  // retrieve various information levels of the map files
  int           fetchPOI(overlaySections overlay,
                         infoLevels      info) { return pointsOfInterest[overlay][info]; }

  // is anything in a particular POI on?
  int           isPOIOn(overlaySections overlay)
  {
    int result = 0;
    for (int i=0; i<=TOTAL_INFOLEVELS; i++)
        result |= fetchPOI(overlay, (infoLevels) i);
    return result;
  }


  // Force the class to read in the file based on current information
  void          readData();

  // Reset all of the collections
  void          clearMaps();

  // Return the ObjectSequence collection
  LL_PolyLineCollection& fetchPolygonCollection(overlaySections overlay)
  { return polylineCollection[overlay]; }

  // Given a collection of Polygons, return a specific polyline
  LL_PolyLine * fetchPolyline(overlaySections overlay, int polygon)
  { return (LL_PolyLine *)(polylineCollection[overlay][polygon]);}

  // Given a collection of Polylines, return a specific LatLonAlt
  LatLon * fetchLatLon(overlaySections overlay, int polygon, int location)
  { return (*fetchPolyline(overlay, polygon))[location]; }

#if 0
  // Deal with the longitude offset 
  void		setOffsetLon(double Long)
  {
    longitudeOffset->setLon(Long);
  }

  double        fetchOffsetLon()
  {
    return longitudeOffset->fetchLon();
  }
#endif

  // Misc
  void          setVerbosity(int flag)          { Verbose = flag; }
  int           fetchVerbosity()                { return Verbose; }
  friend        ostream &operator<<(ostream& out, WorldMap& WP);
  friend        ostream &operator<<(ostream& out, WorldMap *WP);

};

#endif
