//*-*-c++-*-*
//
//                      Copyright (c) 1996 by
//                      Advanced Visual Systems Inc.
//                      All Rights Reserved
//
//      This software comprises unpublished confidential information of
//      Advanced Visual Systems Inc. and may not be used, copied or made
//      available to anyone, except in accordance with the license
//      under which it is furnished.
//
//      This file is under Perforce control
//      $Id: //depot/express/fcs70/include/avs/gis/transfrm.h#1 $
//


#ifndef PROJECTION_HEADER
#define PROJECTION_HEADER


#ifndef CXXUTIL_STRING_HXX
#include <avs/cxxutl/string.hxx>
#endif

#ifndef LATLON_HEADER
#include <avs/gis/latlon.h>
#endif

#ifndef POINT_HEADER
#include <avs/gis/points.h>
#endif

#ifdef _AIX
// This is a hack to get around some IBM goofiness
#undef _ABS
#endif

#include <math.h>
#include <float.h>


//----------------
//      Useful globals
//----------------

#ifndef M_PI
#define M_PI            3.14159265358979323846
#endif

#ifndef TRUE
#define TRUE 1
#endif
#ifndef FALSE
#define FALSE 0
#endif

#ifndef NO_PROJECTION_AVAILABLE
#define NO_PROJECTION_AVAILABLE -10
#endif


//
//      The following enumeration defines the projection types
//
enum projType {
    cartesian,
    mercator,
    lambert_cyl,
    gnomonic,
    stereo,
    ortho,
    postel,
    lambert_azi,
    lambert_conical,
    albers,
    cassini_soldner,
    bonne,
    werner,
    sanson_flamsteed,
    globe
};

//
//      The following class defines a Projection object
//      that can be inherited by other transform classes.
//
//      The projection class works in a non-defined double
//      precision world space. All attempts at defining this
//      world space should be created outside of this class.
//
class Projection {
    // private instances
private: 
    // Information for the physical coordinates
    int         nDimensions;            // number of coordinates
    LatLon      *llCoordinates;         // The coordinate values in latlon
    double      Altitude;               // the altitude
    double      *worldCoordinates;      // The coordinate values in world coords

    // Global information.
    // What projection is the current inCoord?
    projType    currentProjection;      

    // Current globe radius for globe projection
    double      globeRadius;     // equitorial radius in km

    // These two points indicate the center of the map
    LatLon      *baseLatLon;

    // These two points are the 1st and 2nd standard parallel
    // Only use the longitude value. 
    LatLon      *firstParallel;
    LatLon      *secondParallel;

    // Store the offset for the longitude origin.
    LatLon	*longitudeOffset;   // defaults to 0 degrees longitude


    // public instances
public:
    // Constructors - Just the most common for now
    Projection();                        // Default. Assumes default values.

    Projection(projType Type);           // Set the type, retain other defaults

    Projection(projType Type,
               double lat,
               double lon);              // Set the type & the lat lon

    Projection(projType Type, 
               double radius, 
               double lat, 
               double lon);              // Set the type & the lat lon

    // Destructor
    ~Projection();

    // Fetch and set member functions - set functions
    // contain error checking, so they are not inlined.
    projType            fetchProjection()               { return currentProjection; }
    UtString            fetchProjectionName();
    int                 setProjection(projType); 
    int                 setProjection(UtString type);
    double              fetchGlobeRadius()              { return globeRadius; }
    int                 setGlobeRadius(double); 

    // lats and lons can be set and fetched in different ways 
    double              fetchBaseLat()              { return baseLatLon->fetchLat(); }
    void                fetchBaseLat(double *D,
                                     double *M,
                                     double *S)     { baseLatLon->fetchLat(D, M, S); }
    void                fetchBaseLat(double *lat)   { baseLatLon->fetchLat(lat); }
    double              fetchBaseLon()              { return baseLatLon->fetchLon(); }
    void                fetchBaseLon(double *D,
                                     double *M,
                                     double *S)     { baseLatLon->fetchLon(D, M, S); }
    void                fetchBaseLon(double *lon)   { baseLatLon->fetchLon(lon); }

    void                setBaseLat(double lat)      { baseLatLon->setLat(lat); }
    void                setBaseLat(double D,
                                   double M,
                                   double S)        { baseLatLon->setLat(D,M,S); }
    void                setBaseLat(double *lat)     { baseLatLon->setLat(lat); }

    void                setBaseLon(double lon)      { baseLatLon->setLon(lon); }
    void                setBaseLon(double D,
                                   double M,
                                   double S)        { baseLatLon->setLon(D,M,S); }
    void                setBaseLon(double *lon)     { baseLatLon->setLon(lon); }

    void                setBaseLatLon(double lat, 
                                      double lon )  { baseLatLon->setLat(lat); 
                                                      baseLatLon->setLon(lon); }
    void                setBaseLatLon(double latD,  
                                      double latM,  
                                      double latS,
                                      double lonD,  
                                      double lonM,  
                                      double lonS)  { baseLatLon->setLat(latD, latM, latS);
                                                      baseLatLon->setLon(lonD, lonM, lonS); }
    void                setBaseLatLon(double *lat, 
                                      double *lon)  { baseLatLon->setLat(lat); 
                                                      baseLatLon->setLon(lon); }
    LatLon*             fetchBaseLatLon()           { return baseLatLon; }

    double              fetchFirstPar()             { return firstParallel->fetchLon(); }
    void                fetchFirstPar(double *D,
                                      double *M,
                                      double *S)    { firstParallel->fetchLon(D, M, S); }
    void                fetchFirstPar(double *par)  { firstParallel->fetchLon(par); }
    void                setFirstPar(double par)     { firstParallel->setLon(par); }
    void                setFirstPar(double D,
                                    double M,
                                    double S)       { firstParallel->setLon(D, M, S); }
    void                setFirstPar(double *par)    { firstParallel->setLon(par); }

    double              fetchSecondPar()            { return secondParallel->fetchLon(); }
    void                fetchSecondPar(double *D,
                                       double *M,
                                       double *S)   { secondParallel->fetchLon(D,M,S); }
    void                fetchSecondPar(double *par) { secondParallel->fetchLon(par); }
    void                setSecondPar(double par)    { secondParallel->setLon(par); }
    void                setSecondPar(double D,
                                     double M,
                                     double S)      { secondParallel->setLon(D,M,S); }
    void                setSecondPar(double *par)   { secondParallel->setLon(par); }

    // Coordinate mainupulation. Adding and removing points
    void                removeCoords();

    int                 fetchNDims()                    { return nDimensions; }
    void                setLLCoordinates(double, 
                                            double,
                                            double);  // as lat/lot in decimals + altitude
    void                setLLCoordinates(double,
                                            double,
                                            double,
                                            double,
                                            double,
                                            double,
                                            double); // as lat/lon in DMS + altitude
    void                fetchLLCoordinates(double *lat,
                                              double *lon,
                                              double *altitude) { 
      *lat = llCoordinates->fetchLat();
      *lon = llCoordinates->fetchLon();
      *altitude = Altitude; 
    }
    LatLon              *fetchLLCoordinates(double *alt) {
      *alt = Altitude;
      return llCoordinates;
    }
    LatLon              *fetchLLCoordinates() {
      return llCoordinates;
    }
    int                 fetchWorldCoordinates(double **out);
    int                 fetchWorldCoordinates(WorldPoint *out);
    int                 fetchWorldCoordinates(double *, 
                                              double *,
                                              double *);
    void                setWorldCoordinates(WorldPoint in);
    void                setWorldCoordinates(double *in);
    void                setWorldCoordinates(double x, 
                                            double y,
                                            double z);

    // Map projection member functions - here we go... 
    int                 Cartesian();
    int                 Globe();                
    int                 Mercator();
    int                 LambertCylindric();
    int                 Gnomonic();
    int                 Stereo();
    int                 Orthographic();
    int                 Postel();
    int                 LambertAzimuthal();
    int                 LambertConical();
    int                 Albers();
    int                 CassiniSoldner();
    int                 Bonne();
    int                 Werner();
    int                 SansonFlamsteed();

    // Inverse Map projection member functions - here we go again... 
    int                 inverseCartesian();
    int                 inverseGlobe();         
    int                 inverseMercator();
    int                 inverseLambertCylindric();
    int                 inverseGnomonic();
    int                 inverseStereo();
    int                 inverseOrthographic();
    int                 inversePostel();
    int                 inverseLambertAzimuthal();
    int                 inverseLambertConical();
    int                 inverseAlbers();
    int                 inverseCassiniSoldner();
    int                 inverseBonne();
    int                 inverseWerner();
    int                 inverseSansonFlamsteed();

    // All-In-One functions for whiners

    // From LATLON to WORLD COORDINATES
    int                 FromTo(LatLon inLatLon,
                               double Altitude,
                               projType proj,
                               double   **result);
    int                 FromTo(LatLon inLatLon,
                               double Altitude,
                               double   **result);
    int                 FromTo(LatLon inLatLon,
                               double   **result);

    // From WORLD COORDINATES to LATLON
    int                 FromTo(double   *input,
                               projType proj,
                               LatLon   *output,
                               double   *altitude);
    int                 FromTo(double   *input,
                               LatLon   *output,
                               double   *altitude);
    int                 FromTo(double   *input,
                               LatLon   *output);
    LatLon*             FromTo(double   *input);
    LatLon*             FromTo(WorldPoint input);
    LatLon*             FromTo(double, double, double);

    // Deal with the longitude offset 
    void                setOffsetLon(double Long)
      {
        delete longitudeOffset;
        longitudeOffset = new LatLon(0.0, Long);
      }

    double              fetchOffsetLon()
      {
        return longitudeOffset->fetchLon();
      }


    // Utilities
    double              LonToRad(double value)   // rads to degrees.
      {
        LatLon Intermediate;
        Intermediate.setLon(value + longitudeOffset->fetchLon());
        return ((Intermediate.fetchLon()/180.0) * M_PI);
      }

    double              LonToDeg(double value)   // degrees to rads
      {
        LatLon Intermediate;
        Intermediate.setLon(((value/M_PI) * 180.0) - longitudeOffset->fetchLon() );
        return Intermediate.fetchLon();
      }
    double              LatToRad(double value)   // rads to degrees.
      {
        LatLon Intermediate;
        Intermediate.setLat(value);
        return ((Intermediate.fetchLat()/180.0) * M_PI);
      }

    double              LatToDeg(double value)   // degrees to rads
      {
        LatLon Intermediate;
        Intermediate.setLat((value/M_PI) * 180.0);
        return Intermediate.fetchLat();
      }
};


#endif
