//			Copyright (c) 1993 by
//			Advanced Visual Systems Inc.
//			All Rights Reserved
//
//	This software comprises unpublished confidential information of
//	Advanced Visual Systems Inc. and may not be used, copied or made
//	available to anyone, except in accordance with the license
//	under which it is furnished.
//
//	This file is under Perforce control
//	$Id: //depot/express/fcs70/include/avs/omx.hxx#1 $
//
#ifndef OMX_INCLUDED
#define OMX_INCLUDED

#include <avs/om.h>
#include <avs/arr.h>

#ifdef __cplusplus

/*
 * Create a new a new OMobj_id with OMcreate_obj_from_path, using the
 * obj_id argument as a parent for this new object.  This OMobj_id
 * will be destroyed when the C++ class is deleted
 */
const int OMX_CREATE_NEW_OBJ = 1;
/*
 * This OMXobj refers to a subobject of the supplied parent OMXobj.  In
 * this case, the obj_id argument should be the OMobj_id of the parent
 * object (assumed to be obtained with OM_OBJ_RD mode).  In this case
 * we often end up looking up the OMobj_id for this object with a subroutine
 * call each time that this obj_id is accessed.
 */
const int OMX_CREATE_SUB_OBJ = 2;
/*
 * Attach this OMXobj to an existing OMobj_id.  This OMobj_id is assumed
 * to be obtained with RW mode and cannot change during the lifetime of
 * the C++ class (or sub OMXobj's will not update themselves)
 */
const int OMX_EXISTING_OBJ = 3;
/*
 * Attach this OMXobj to an existing OMobj_id that might change later on
 * with a call to set_obj_id.  All sub OMXobj's will attach themselves in
 * reference mode so that they look up their OMobj_id relative to the
 * parent's id with each call.
 */
const int OMX_TRANSIENT_OBJ = 4;

/////////////////////////////////////
class OMXobj {
/////////////////////////////////////
  public:
                        OMXobj (
                            OMobj_id obj_id /* = OMnull_obj */,
			    const char *name = NULL,
			    int mode = OMX_CREATE_NEW_OBJ,
			    OMXobj *parent = NULL,
			    const char *class_path = NULL);
    virtual             ~OMXobj ();

    //
    // This is an internal routine used to access the OMXobj's this pointer
    // from a constructor without errors.
    //
    OMXobj *		ret_OMXobj_this();

    const char *        name () const;
    OMobj_id		obj_id(int mode = OM_OBJ_RW) const;

    void        	set_obj_id (OMobj_id obj_id);

    OMobj_id		obj_val() const;

    operator 		OMobj_id () const;

    int             	operator== (OMobj_id rhs) const;
    int             	operator!= (OMobj_id rhs) const;

    //
    // methods controlling the execution.. begin and end operation
    //
    unsigned int       	push_ctx (int state = 0,
				  int event_mode = 0,
				  int child_event_mode = 0);
    int 	       	pop_ctx ();

    //
    // methods dealing with connections: add/del/set
    //
    int			add_obj_ref(OMobj_id conn_id,int mode = 0);
    int			del_obj_ref(OMobj_id conn_id,int mode = 0);
    int			set_obj_ref(OMobj_id conn_id,int mode = 0);
    int			insert_obj_ref(xp_long ind, OMobj_id conn_id, int mode = 0);

    int			set_obj_val(OMobj_id conn_id,int mode = 0);

    int			valid_obj(OMobj_id templ = OMnull_obj, int mode = 0);
    int			get_obj_seq(OMobj_id templ = OMnull_obj,
				    int mode = OM_SEQ_VAL) const;
    int			changed(int seq_num, OMobj_id templ = OMnull_obj,
				int mode = OM_SEQ_VAL);

	int 		get_array(int *type, void **array, int *ndim,
				  xp_long *dims, int mode);
#ifdef WORDLENGTH_64
	int 		get_array_n(int *type, void **array, int *ndim,
				  int *dims, int mode);
#endif

    void		free_array(void *array);
    void 		*ret_array_ptr(int mode = OM_GET_ARRAY_RW,
				       xp_long *size = NULL, int *type = NULL);
#ifdef WORDLENGTH_64
    void 		*ret_array_ptr_n(int mode = OM_GET_ARRAY_RW,
				       int *size = NULL, int *type = NULL);
#endif

    void 		*ret_typed_array_ptr(int mode = OM_GET_ARRAY_RW,
				             int type = OM_TYPE_INT,
					     xp_long *size = NULL);
#ifdef WORDLENGTH_64
    void 		*ret_typed_array_ptr_n(int mode = OM_GET_ARRAY_RW,
				             int type = OM_TYPE_LONG,
					     int *size = NULL);
#endif

    char                *ret_str_array_val(xp_long index,
					    char *buffer = NULL,
					    xp_long buffer_size = 0 );
    int                 set_str_array_val(xp_long index, const char *value);
    int 		set_array(int type, void *array, xp_long len,
				  int mode = OM_SET_ARRAY_COPY);

    int 		get_array_size(xp_long *size) const;
#ifdef WORDLENGTH_64
    int 		get_array_size_n(int *size) const;
#endif

    int 		set_array_size(xp_long size);
    xp_long		ret_array_size() const;
    xp_long             ret_num_subobjs() const;
    OMobj_id            ret_subobj_id( xp_long i, int mode = OM_OBJ_RD );
    int			is_type_of(OMtype_id type) const;

    virtual const char	*class_name();

  protected:
    unsigned int	created:1;      /* do we destroy OMobj_id in delete? */
    unsigned int	is_ref:1;       /* is a reference to parent OMXobj?  */
    unsigned int	need_sub_refs:1; /* should sub OMXobj's be refs?     */
    unsigned int	is_value:1;     /* is not a group reference/ptr?     */
    OMXobj 		*parent;	/* OMXobj of parent (or NULL)        */
    union {
       OMobj_id         obj_id;		/* if ref = 0, obj_id to use         */
       const char	*sub_name;	/* else path name relative to parent */
    } v;
  private:
    OMXobj(OMXobj &) {}			/* don't allow copy constructor     */
};


inline
OMXobj *OMXobj::ret_OMXobj_this()
{
   return(this);
}

/*
 * Returns the "validity" of the object (i.e. whether or not it currently
 * has a valid value)
 */
inline
int OMXobj::valid_obj(OMobj_id templ, int mode)
{
   return(OMvalid_obj(obj_id(OM_OBJ_RD),templ,mode));
}

/*
 * Returns the sequence number of the object
 */
inline
int  OMXobj::get_obj_seq(OMobj_id templ, int mode) const
{
   return(OMget_obj_seq(obj_id(OM_OBJ_RD),templ,mode));
}

/*
 * Returns the "changed" state of the object with respect to the
 * supplied sequence number.
 */
inline
int  OMXobj::changed(int seq_num, OMobj_id templ, int mode)
{
   return(get_obj_seq(templ,mode) > seq_num);
}

inline
int  OMXobj::operator == (OMobj_id rhs) const
{
    OMobj_id my_id = obj_id(OM_OBJ_RD);
    return (my_id.proc_id == rhs.proc_id) && (my_id.elem_id == rhs.elem_id);
}

inline
int  OMXobj::operator != (OMobj_id rhs) const
{
    OMobj_id my_id = obj_id(OM_OBJ_RD);
    return !((my_id.proc_id == rhs.proc_id) &&
	     (my_id.elem_id == rhs.elem_id));
}

inline
unsigned int OMXobj::push_ctx (int state, int event_mode, int child_event_mode)
{
    return(OMpush_ctx (obj_id(),state,event_mode,child_event_mode));
}

inline
int OMXobj::pop_ctx ()
{
    return(OMpop_ctx (obj_id()));
}

inline
OMobj_id OMXobj::obj_val() const
{
   OMobj_id val_id;
   if (OMget_obj_val(obj_id(),&val_id) != 1) val_id = OMnull_obj;
   return(val_id);
}

inline
void OMXobj::free_array(void *array)
{
   ARRfree(array);
}

inline
int OMXobj::is_type_of(OMtype_id type) const
{
   return OMobj_is_type_of( obj_id(), type );
}

/////////////////////////////////////
class OMXgroup : public OMXobj {
/////////////////////////////////////
    public:
			OMXgroup (
			    OMobj_id obj_id = OMinst_obj,
			    const char *name = NULL,
			    int mode = OMX_CREATE_NEW_OBJ,
			    OMXobj *parent_ptr = NULL,
			    const char *class_path = "group");

   virtual void    	*ret_class_ptr(const char *, void *chain = NULL) const;
   virtual void    	*ret_omx_ptr(const char *);
   virtual void		init_class_ptr();
   virtual void		free_class_ptr();
   OMobj_id		find_subobj(OMobj_name name, int mode = OM_OBJ_RD);
   OMobj_id		find_subobj(const char *name, int mode = OM_OBJ_RD);
   void 	*class_ptr;

   //
   // Flag error messages when an attempt is made to assign an OMXgroup
   // to something else.
   //
   private:
    OMXgroup &operator=(OMXobj &) {return(*this);}
};

inline
OMobj_id OMXgroup::find_subobj(OMobj_name name, int mode)
{
   return OMfind_subobj(obj_id(), name, mode);
}

inline
OMobj_id OMXgroup::find_subobj(const char *name, int mode)
{
   return find_subobj(OMstr_to_name(name), mode);
}

/////////////////////////////////////
class OMXappl : public OMXgroup {
/////////////////////////////////////
   public:
			OMXappl (
			    OMobj_id obj_id = OMinst_obj,
			    const char *name = NULL,
			    int mode = OMX_CREATE_NEW_OBJ,
			    OMXobj *parent_ptr = NULL,
			    const char *class_path = "application");
};

/////////////////////////////////////
class OMXbprim : public OMXobj {
/////////////////////////////////////
   public:
			OMXbprim (
			    OMobj_id obj_id = OMinst_obj,
			    const char *name = NULL,
			    int mode = OMX_CREATE_NEW_OBJ,
			    OMXobj *parent_ptr = NULL,
			    const char *class_path = NULL);
      int		get_int_val (int &value, int dflt = 0) const;
      int		set_int_val (int v);
      int		get_long_val (xp_long &value, xp_long dflt = 0) const;
      int		set_long_val (xp_long v);
      int		get_real_val (double &value, double dflt = 0) const;
      int		set_real_val (double v);
      int		get_data_type();
};

inline
int OMXbprim::set_int_val (int val)
{
   return(OMset_int_val(obj_id(), val));
}

inline
int OMXbprim::set_long_val (xp_long val)
{
   return(OMset_long_val(obj_id(), val));
}

inline
int OMXbprim::set_real_val (double val)
{
   return(OMset_real_val(obj_id(), val));
}

/*
 * This class has all of the data type methods but will require explicit
 * casts since we don't know what to do by default.
 */
/////////////////////////////////////
class OMXprim : public OMXbprim {
/////////////////////////////////////
   public:
			OMXprim (
			    OMobj_id obj_id = OMinst_obj,
			    const char *name = NULL,
			    int mode = OMX_CREATE_NEW_OBJ,
			    OMXobj *parent_ptr = NULL,
			    const char *class_path = "prim");
    			operator double () const;
    			operator float () const;
    			operator xp_long () const;
#ifdef WORDLENGTH_64
    			operator int () const;
#endif

      OMXprim & 	operator= (double rhs);
      double  		operator += (double rhs);
      double  		operator -= (double rhs);
      int  		operator == (double rhs) const;
      int  		operator != (double rhs) const;
      int  		operator < (double rhs) const;
      int  		operator <= (double rhs) const;
      int  		operator > (double rhs) const;
      int  		operator >= (double rhs) const;

      // hidden methods from OMXobj:
      // without these, the operator==(double) above hides rather than
      // overloading the OMXobj:: version.  So we just provide a
      // drop-through version here.  See the C++ ARM section 13.1 for why
      // you need this.
      int  		operator == (OMobj_id rhs) const
                           { return OMXobj::operator==(rhs);}
      int  		operator != (OMobj_id rhs) const
                           { return OMXobj::operator!=(rhs);}

      OMXprim & 	operator= (float rhs);
      float  		operator += (float rhs);
      float  		operator -= (float rhs);
      int  		operator == (float rhs) const;
      int  		operator != (float rhs) const;
      int  		operator < (float rhs) const;
      int  		operator <= (float rhs) const;
      int  		operator > (float rhs) const;
      int  		operator >= (float rhs) const;

	  OMXprim & 	operator= (xp_long rhs);
      xp_long  		operator += (xp_long rhs);
      xp_long  		operator -= (xp_long rhs);
      int  		operator == (xp_long rhs) const;
      int  		operator != (xp_long rhs) const;
      int  		operator < (xp_long rhs) const;
      int  		operator <= (xp_long rhs) const;
      int  		operator > (xp_long rhs) const;
      int  		operator >= (xp_long rhs) const;

#ifdef WORDLENGTH_64
      OMXprim & 	operator= (int rhs);
      int  			operator += (int rhs);
      int  			operator -= (int rhs);
      int  		operator == (int rhs) const;
      int  		operator != (int rhs) const;
      int  		operator < (int rhs) const;
      int  		operator <= (int rhs) const;
      int  		operator > (int rhs) const;
      int  		operator >= (int rhs) const;
#endif

      virtual const char *class_name();

      inline OMXprim & 	operator= (const OMXbprim &rhs);

      // Block automatically generated assignment operator that
      // interferes with OMXprim::operator= (const OMXbprim &rhs)
      // See GNAT 7968 / CFS PR 10631.
      OMXprim & operator= (const OMXprim &rhs) {
         return operator=( (const OMXbprim &)rhs );
      }
};

inline
OMXprim & OMXprim::operator= (const OMXbprim &rhs)
{
   set_obj_val (rhs.obj_id());
   return *this;
}

inline
OMXprim::operator double () const
{ double value; get_real_val (value,0.0); return value; }

inline
OMXprim::operator float () const
{ double value; get_real_val (value,0.0); return (float) value; }

inline
OMXprim::operator xp_long () const
{ xp_long value; get_long_val (value,0); return value; }

#ifdef WORDLENGTH_64
inline
OMXprim::operator int () const
{ int value; get_int_val (value,0); return value; }
#endif

inline
OMXprim & OMXprim::operator= (double rhs) { set_real_val (rhs); return *this; }

inline
int OMXprim::operator == (double rhs) const
{ return operator double () == rhs; }

inline
int OMXprim::operator != (double rhs) const
{ return operator double () != rhs; }

inline
int OMXprim::operator < (double rhs) const
{ return operator double () < rhs; }

inline
int OMXprim::operator <= (double rhs) const
{ return operator double () <= rhs; }

inline
int OMXprim::operator > (double rhs) const
{ return operator double () > rhs; }

inline
int OMXprim::operator >= (double rhs) const
{ return operator double () >= rhs; }

inline
double OMXprim::operator += (double rhs)
{
   double value;
   get_real_val(value,0);
   set_real_val(value + rhs);
   return value + rhs;
}

inline
double OMXprim::operator -= (double rhs)
{
   double value;
   get_real_val(value,0);
   set_real_val(value - rhs);
   return value - rhs;
}

inline
OMXprim & OMXprim::operator= (float rhs) { set_real_val (rhs); return *this; }

inline
int OMXprim::operator == (float rhs) const
{ return operator float () == rhs; }

inline
int OMXprim::operator != (float rhs) const
{ return operator float () != rhs; }

inline
int OMXprim::operator < (float rhs) const
{ return operator float () < rhs; }

inline
int OMXprim::operator <= (float rhs) const
{ return operator float () <= rhs; }

inline
int OMXprim::operator > (float rhs) const
{ return operator float () > rhs; }

inline
int OMXprim::operator >= (float rhs) const
{ return operator float () >= rhs; }

inline
float OMXprim::operator += (float rhs)
{
   double value;
   get_real_val(value,0);
   set_real_val(value + rhs);
   return (float) (value + rhs);
}

inline
float OMXprim::operator -= (float rhs)
{
   double value;
   get_real_val(value,0);
   set_real_val(value - rhs);
   return (float) (value - rhs);
}

inline
OMXprim & OMXprim::operator= (xp_long rhs) { set_long_val (rhs); return *this; }

inline
int OMXprim::operator == (xp_long rhs) const
{ return operator xp_long () == rhs; }

inline
int OMXprim::operator != (xp_long rhs) const
{ return operator xp_long () != rhs; }

inline
int OMXprim::operator < (xp_long rhs) const
{ return operator xp_long () < rhs; }

inline
int OMXprim::operator <= (xp_long rhs) const
{ return operator xp_long () <= rhs; }

inline
int OMXprim::operator > (xp_long rhs) const
{ return operator xp_long () > rhs; }

inline
int OMXprim::operator >= (xp_long rhs) const
{ return operator xp_long () >= rhs; }

inline
xp_long OMXprim::operator += (xp_long rhs)
{
   xp_long value;
   get_long_val(value,0);
   set_long_val(value + rhs);
   return value + rhs;
}

inline
xp_long OMXprim::operator -= (xp_long rhs)
{
   xp_long value;
   get_long_val(value,0);
   set_long_val(value - rhs);
   return value - rhs;
}

#ifdef WORDLENGTH_64
inline
OMXprim & OMXprim::operator= (int rhs) { set_int_val (rhs); return *this; }

inline
int OMXprim::operator == (int rhs) const
{ return operator int () == rhs; }

inline
int OMXprim::operator != (int rhs) const
{ return operator int () != rhs; }

inline
int OMXprim::operator < (int rhs) const
{ return operator int () < rhs; }

inline
int OMXprim::operator <= (int rhs) const
{ return operator int () <= rhs; }

inline
int OMXprim::operator > (int rhs) const
{ return operator int () > rhs; }

inline
int OMXprim::operator >= (int rhs) const
{ return operator int () >= rhs; }

inline
int OMXprim::operator += (int rhs)
{
   int value;
   get_int_val(value,0);
   set_int_val(value + rhs);
   return value + rhs;
}

inline
int OMXprim::operator -= (int rhs)
{
   int value;
   get_int_val(value,0);
   set_int_val(value - rhs);
   return value - rhs;
}
#endif

/////////////////////////////////////
class OMXint : public OMXbprim {
/////////////////////////////////////
   public:
			OMXint (
			    OMobj_id obj_id = OMinst_obj,
			    const char *name = NULL,
			    int mode = OMX_CREATE_NEW_OBJ,
			    OMXobj *parent_ptr = NULL,
			    const char *class_path = "int");

      /* Assignment operators */
      OMXint & 		operator= (const int rhs);
      OMXint & 		operator= (const OMXbprim &rhs);
      OMXint & 		operator= (const OMXint &rhs);

      /* Cast operator */
      			operator int () const;

      int		operator ! () const;
      int		operator ++();
      int		operator --();
      int		operator += (int rhs) ;
      int		operator -= (int rhs) ;
      int		operator *= (int rhs) ;
      int		operator /= (int rhs) ;
      int		operator %= (int rhs) ;

      /* Comparison operators */
      int  		operator == (int rhs) const;
      int  		operator != (int rhs) const;
      int  		operator < (int rhs) const;
      int  		operator <= (int rhs) const;
      int  		operator > (int rhs) const;
      int  		operator >= (int rhs) const;

      // expose hidden OMXobj functions
      int  		operator == (OMobj_id rhs) const
                           { return OMXobj::operator==(rhs);}
      int  		operator != (OMobj_id rhs) const
                           { return OMXobj::operator!=(rhs);}

      int		get_val (int &value, int dflt = 0) const;
      int		set_val (int v);

      // resolve ambiguities between OMXint -> int and OMXint -> OMXobj
      int  		operator == (const OMXint &rhs) const;
      int  		operator != (const OMXint &rhs) const;
      virtual const char *class_name();
};

/* Assignment operators */
inline
OMXint & OMXint::operator= (const int rhs)
{
    set_int_val (rhs);
    return *this;
}

inline
OMXint & OMXint::operator= (const OMXbprim &rhs)
{
   int tmp;
   rhs.get_int_val(tmp);
   set_int_val (tmp);
   return *this;
}

inline
OMXint & OMXint::operator= (const OMXint &rhs)
{
   int tmp;
   rhs.get_int_val(tmp);
   set_int_val (tmp);
   return *this;
}

inline
OMXint::operator int () const
{ int value; get_int_val (value,0); return value; }

inline
int OMXint::operator ! () const
{ int value; get_int_val (value,0); return !value; }

inline
int OMXint::operator ++ ()
{
   int tmp = operator int() + 1;
   set_int_val(tmp);
   return(tmp);
}

inline
int OMXint::operator -- ()
{
   int tmp = operator int() - 1;
   set_int_val(tmp);
   return(tmp);
}

inline
int OMXint::operator ==(const OMXint &rhs) const
{ return operator int() == rhs.operator int();}

inline
int OMXint::operator !=(const OMXint &rhs) const
{ return operator int() != rhs.operator int();}

inline
int OMXint::operator += (int rhs)
{
   int tmp = operator int() + rhs;
   set_int_val(tmp);
   return(tmp);
}

inline
int OMXint::operator -= (int rhs)
{
   int tmp = operator int() - rhs;
   set_int_val(tmp);
   return(tmp);
}

inline
int OMXint::operator *= (int rhs)
{
   int tmp = operator int() * rhs;
   set_int_val(tmp);
   return(tmp);
}

inline
int OMXint::operator /= (int rhs)
{
   int tmp = operator int() / rhs;
   set_int_val(tmp);
   return(tmp);
}

inline
int OMXint::operator %= (int rhs)
{
   int tmp = operator int() % rhs;
   set_int_val(tmp);
   return(tmp);
}

/*
   Comparison operators
   (this may only be needed because we override != and == in OMXobj)
   */
inline
int OMXint::operator == (int rhs) const
{ return operator int () == rhs; }

inline
int OMXint::operator != (int rhs) const
{ return operator int () != rhs; }

inline
int OMXint::operator < (int rhs) const
{ return operator int () < rhs; }

inline
int OMXint::operator <= (int rhs) const
{ return operator int () <= rhs; }

inline
int OMXint::operator > (int rhs) const
{ return operator int () > rhs; }

inline
int OMXint::operator >= (int rhs) const
{ return operator int () >= rhs; }

/* Compatibility routines */
inline
int OMXint::get_val (int &value, int dflt) const {
   return(get_int_val(value, dflt));
}
inline
int OMXint::set_val (int value) { return(set_int_val(value)); }

/* 64-bit porting. Newly Introduced */
/////////////////////////////////////
class OMXlong : public OMXbprim {
/////////////////////////////////////
   public:
			OMXlong (
			    OMobj_id obj_id = OMinst_obj,
			    const char *name = NULL,
			    int mode = OMX_CREATE_NEW_OBJ,
			    OMXobj *parent_ptr = NULL,
			    const char *class_path = "long");

      /* Assignment operators */
      OMXlong & 		operator= (const xp_long rhs);
      OMXlong & 		operator= (const OMXbprim &rhs);
      OMXlong & 		operator= (const OMXlong &rhs);

      /* Cast operator */
      			operator xp_long () const;

      xp_long		operator ! () const;
      xp_long		operator ++();
      xp_long		operator --();
      xp_long		operator += (xp_long rhs) ;
      xp_long		operator -= (xp_long rhs) ;
      xp_long		operator *= (xp_long rhs) ;
      xp_long		operator /= (xp_long rhs) ;
      xp_long		operator %= (xp_long rhs) ;

      /* Comparison operators */
      int  		operator == (xp_long rhs) const;
      int  		operator != (xp_long rhs) const;
      int  		operator < (xp_long rhs) const;
      int  		operator <= (xp_long rhs) const;
      int  		operator > (xp_long rhs) const;
      int  		operator >= (xp_long rhs) const;

      // expose hidden OMXobj functions
      int  		operator == (OMobj_id rhs) const
                           { return OMXobj::operator==(rhs);}
      int  		operator != (OMobj_id rhs) const
                           { return OMXobj::operator!=(rhs);}

      int		get_val (xp_long &value, xp_long dflt = 0) const;
      int		set_val (xp_long v);

      // resolve ambiguities between OMXint -> int and OMXint -> OMXobj
      int  		operator == (const OMXlong &rhs) const;
      int  		operator != (const OMXlong &rhs) const;
      virtual const char *class_name();
};

/* Assignment operators */
inline
OMXlong & OMXlong::operator= (const xp_long rhs)
{
    set_long_val (rhs);
    return *this;
}

inline
OMXlong & OMXlong::operator= (const OMXbprim &rhs)
{
   xp_long tmp;
   rhs.get_long_val(tmp);
   set_long_val (tmp);
   return *this;
}

inline
OMXlong & OMXlong::operator= (const OMXlong &rhs)
{
   xp_long tmp;
   rhs.get_long_val(tmp);
   set_long_val (tmp);
   return *this;
}

inline
OMXlong::operator xp_long () const
{ xp_long value; get_long_val (value,0); return value; }

inline
xp_long OMXlong::operator ! () const
{ xp_long value; get_long_val (value,0); return !value; }

inline
xp_long OMXlong::operator ++ ()
{
   xp_long tmp = operator xp_long() + 1;
   set_long_val(tmp);
   return(tmp);
}

inline
xp_long OMXlong::operator -- ()
{
   xp_long tmp = operator xp_long() - 1;
   set_long_val(tmp);
   return(tmp);
}

inline
int OMXlong::operator ==(const OMXlong &rhs) const
{ return operator xp_long() == rhs.operator xp_long();}

inline
int OMXlong::operator !=(const OMXlong &rhs) const
{ return operator xp_long() != rhs.operator xp_long();}

inline
xp_long OMXlong::operator += (xp_long rhs)
{
   xp_long tmp = operator xp_long() + rhs;
   set_long_val(tmp);
   return(tmp);
}

inline
xp_long OMXlong::operator -= (xp_long rhs)
{
   xp_long tmp = operator xp_long() - rhs;
   set_long_val(tmp);
   return(tmp);
}

inline
xp_long OMXlong::operator *= (xp_long rhs)
{
   xp_long tmp = operator xp_long() * rhs;
   set_long_val(tmp);
   return(tmp);
}

inline
xp_long OMXlong::operator /= (xp_long rhs)
{
   xp_long tmp = operator xp_long() / rhs;
   set_long_val(tmp);
   return(tmp);
}

inline
xp_long OMXlong::operator %= (xp_long rhs)
{
   xp_long tmp = operator xp_long() % rhs;
   set_long_val(tmp);
   return(tmp);
}

/*
   Comparison operators
   (this may only be needed because we override != and == in OMXobj)
   */
inline
int OMXlong::operator == (xp_long rhs) const
{ return operator xp_long () == rhs; }

inline
int OMXlong::operator != (xp_long rhs) const
{ return operator xp_long () != rhs; }

inline
int OMXlong::operator < (xp_long rhs) const
{ return operator xp_long () < rhs; }

inline
int OMXlong::operator <= (xp_long rhs) const
{ return operator xp_long () <= rhs; }

inline
int OMXlong::operator > (xp_long rhs) const
{ return operator xp_long () > rhs; }

inline
int OMXlong::operator >= (xp_long rhs) const
{ return operator xp_long () >= rhs; }

/* Compatibility routines */
inline
int OMXlong::get_val (xp_long &value, xp_long dflt) const {
   return(get_long_val(value, dflt));
}
inline
int OMXlong::set_val (xp_long value) { return(set_long_val(value)); }


/////////////////////////////////////
class OMXreal : public OMXbprim {
/////////////////////////////////////
   public:
			OMXreal (
			    OMobj_id obj_id = OMinst_obj,
			    const char *name = NULL,
			    int mode = OMX_CREATE_NEW_OBJ,
			    OMXobj *parent_ptr = NULL,
			    const char *class_path = "double");

      /* Assignment operators */
      OMXreal & 	operator= (double rhs);
      OMXreal & 	operator= (const OMXbprim &rhs);
      OMXreal & 	operator= (const OMXreal &rhs);

      /* Cast operators */
			operator double () const;

      double		operator ++();
      double		operator --();
      double		operator += (double rhs) ;
      double		operator -= (double rhs) ;
      double		operator *= (double rhs) ;
      double		operator /= (double rhs) ;

      /* Comparison operators */
      int  		operator == (double rhs) const;
      int  		operator != (double rhs) const;
      int  		operator < (double rhs) const;
      int  		operator <= (double rhs) const;
      int  		operator > (double rhs) const;
      int  		operator >= (double rhs) const;

      // expose hidden OMXobj functions
      int  		operator == (OMobj_id rhs) const
                           { return OMXobj::operator==(rhs);}
      int  		operator != (OMobj_id rhs) const
                           { return OMXobj::operator!=(rhs);}

      // resolve ambiguities between OMXint -> int and OMXint -> OMXobj
      int  		operator == (OMXreal &rhs) const;
      int  		operator != (OMXreal &rhs) const;
      virtual const char *class_name();
};

/* Assignment operators */
inline
OMXreal & OMXreal::operator= (double rhs) { set_real_val (rhs); return *this; }

inline
OMXreal & OMXreal::operator= (const OMXbprim &rhs)
{
   double tmp;
   rhs.get_real_val(tmp);
   set_real_val (tmp);
   return *this;
}

inline
OMXreal & OMXreal::operator= (const OMXreal &rhs)
{
   double tmp;
   rhs.get_real_val(tmp);
   set_real_val (tmp);
   return *this;
}

/* Cast operators */
inline
OMXreal::operator double () const
{ double value; get_real_val (value,0.0); return value; }


inline
double OMXreal::operator ++ ()
{
   double tmp = operator double() + 1.0;
   set_real_val(tmp);
   return(tmp);
}

inline
double OMXreal::operator -- ()
{
   double tmp = operator double() - 1.0;
   set_real_val(tmp);
   return(tmp);
}

inline
int OMXreal::operator ==(OMXreal &rhs) const
{ return operator double() == rhs.operator double();}

inline
int OMXreal::operator !=(OMXreal &rhs) const
{ return operator double() != rhs.operator double();}

inline
double OMXreal::operator += (double rhs)
{
   double tmp = operator double() + rhs;
   set_real_val(tmp);
   return(tmp);
}

inline
double OMXreal::operator -= (double rhs)
{
   double tmp = operator double() - rhs;
   set_real_val(tmp);
   return(tmp);
}

inline
double OMXreal::operator *= (double rhs)
{
   double tmp = operator double() * rhs;
   set_real_val(tmp);
   return(tmp);
}

inline
double OMXreal::operator /= (double rhs)
{
   double tmp = operator double() / rhs;
   set_real_val(tmp);
   return(tmp);
}

/* Comparison operators */
inline
int OMXreal::operator == (double rhs) const
{ return operator double () == rhs; }

inline
int OMXreal::operator != (double rhs) const
{ return operator double () != rhs; }

inline
int OMXreal::operator < (double rhs) const
{ return operator double () < rhs; }

inline
int OMXreal::operator <= (double rhs) const
{ return operator double () <= rhs; }

inline
int OMXreal::operator > (double rhs) const
{ return operator double () > rhs; }

inline
int OMXreal::operator >= (double rhs) const
{ return operator double () >= rhs; }

/////////////////////////////////////
class OMXstr: public OMXbprim {
/////////////////////////////////////
  public:
			OMXstr (
			    OMobj_id obj_id = OMinst_obj,
			    const char *name = NULL,
			    int mode = OMX_CREATE_NEW_OBJ,
			    OMXobj *parent_ptr = NULL,
			    const char *path = "string");
    virtual             ~OMXstr ();

    // getting and setting
    int                 set_str_val (const char *str);
    int                 get_str_val (char **value, char *dflt = 0);
			operator char * ();

    int                 operator== (const char *rhs);
    int 		operator== (OMXstr &rhs);
    int 		operator== (xp_long);

    int                 operator!= (const char *rhs);
    int 		operator!= (OMXstr &rhs);
    int 		operator!= (xp_long);

    OMXstr &		operator= (const char *rhs);
    OMXstr & 		operator= (OMXstr &rhs);
    OMXstr & 		operator= (const xp_long rhs);
    OMXstr & 		operator= (const xp_ulong rhs);

    OMXstr &		operator+= (const char *rhs);
    OMXstr & 		operator+= (OMXstr &rhs);
    OMXstr & 		operator+= (const xp_long rhs);
    OMXstr & 		operator+= (const xp_ulong rhs);

    // expose hidden OMXobj functions
    int  		operator == (OMobj_id rhs) const
                           { return OMXobj::operator==(rhs);}
    int  		operator != (OMobj_id rhs) const
                           { return OMXobj::operator!=(rhs);}

    virtual const char	*class_name();
  private:
    char *_save_str;
};

inline
OMXstr::operator char * ()
{ char *_v;  get_str_val (&_v); return _v; }

inline
OMXstr & OMXstr::operator= (const char *rhs)
{
    set_str_val (rhs);
    return *this;
}

inline
OMXstr & OMXstr::operator= (OMXstr &rhs)
{
   set_str_val ((const char *) rhs);
   return *this;
}

inline
OMXstr & OMXstr::operator= (const xp_long rhs)
{
    char  _v[20];
    sprintf( _v, "%ld", rhs );
    set_str_val ( _v );
    return *this;
}

inline
OMXstr & OMXstr::operator= (const xp_ulong rhs)
{
    char  _v[20];
    sprintf( _v, "0x%lx", rhs );
    set_str_val ( _v );
    return *this;
}
/////////////////////////////////////
class OMXenum: public OMXstr {
/////////////////////////////////////
  public:
			OMXenum (
			    OMobj_id obj_id = OMinst_obj,
			    const char *name = NULL,
			    int mode = OMX_CREATE_NEW_OBJ,
			    OMXobj *parent_ptr = NULL,
			    const char *path = "string");

			operator char * () ;
      			operator int ();

    OMXenum &		operator= (const char *rhs);
    OMXenum & 		operator= (OMXenum &rhs);
    OMXenum & 		operator= (const int rhs);

    virtual const char	*class_name();
  private:
};

inline
OMXenum::operator char * () {
    char *_v;  get_str_val (&_v);
    return _v;
}

inline
OMXenum::operator int () {
    int value; get_int_val (value,0);
    return value;
}

inline
OMXenum & OMXenum::operator= (const char *rhs)
{
    set_str_val (rhs);
    return *this;
}

inline
OMXenum & OMXenum::operator= (OMXenum &rhs)
{
   set_str_val ((const char *) rhs);
   return *this;
}

inline
OMXenum & OMXenum::operator= (const int rhs)
{
    set_int_val ( rhs );
    return *this;
}

/////////////////////////////////////
class OMXgroup_array : public OMXobj {
/////////////////////////////////////
    public:
			OMXgroup_array (
			    OMobj_id obj_id,
			    const char *name = NULL,
			    int mode = OMX_CREATE_NEW_OBJ,
			    OMXobj *parent_ptr = NULL);

      virtual ~OMXgroup_array();

      OMXgroup &operator[](const xp_long index);
      virtual OMXgroup *ret_array_member (OMobj_id val_id);

    private:
       void reshape();
       xp_long _arr_size;
       OMXgroup **_arr_values;
};

typedef OMXobj OMXprim_array;
typedef OMXobj OMXobj_array; // links that are not scalar
typedef OMXreal OMXdouble;
typedef OMXreal OMXfloat;
typedef OMXint OMXchar;
typedef OMXint OMXbyte;
typedef OMXint OMXshort;
typedef OMXobj OMXptr;
typedef OMXprim_array OMXchar_array;
typedef OMXprim_array OMXbyte_array;
typedef OMXprim_array OMXshort_array;
typedef OMXprim_array OMXint_array;
typedef OMXprim_array OMXlong_array;
typedef OMXprim_array OMXfloat_array;
typedef OMXprim_array OMXdouble_array;
typedef OMXprim_array OMXstr_array;
typedef OMXprim_array OMXptr_array;
typedef OMXprim_array OMXenum_array;

#if defined(WORDLENGTH_64) && !defined(XP_WIDE_API)
	#define		get_array		get_array_n
	#define		ret_array_ptr	ret_array_ptr_n
	#define		ret_typed_array_ptr	 ret_typed_array_ptr_n
	#define 	get_array_size	get_array_size_n
#endif

#endif
#endif /* OMX_INCLUDED */
