/*
			Copyright (c) 1998 by
			Advanced Visual Systems Inc.
			All Rights Reserved

	This software comprises unpublished confidential information of
	Advanced Visual Systems Inc. and may not be used, copied or made
	available to anyone, except in accordance with the license
	under which it is furnished.

	This file is under Perforce control
	$Id: //depot/express/fcs70/modules/arr_coll.c#1 $
*/

#define XP_WIDE_API	/* Use Wide APIs */

#include <stdio.h>

#include <avs/util.h>
#include <avs/err.h>
#include <avs/om.h>
#include <avs/om_att.h>
#include <avs/dtype.h>
#include <avs/om_type.h>

#define ERR_RETURN(A)  {ERRerror("array_collect",1,ERR_ORIG, A);return(0);}
#define MAX_CLASS 64
#define MAX_NAME_LEN  1024

static char buf[MAX_NAME_LEN];

static int  GetInClass (OMobj_id obj, char *class_name);
static int  SetOutDim (OMobj_id obj_id, int dim);

static int  AreTheSame(OMobj_id ctx_id, OMobj_id obj_id, char *path_from_parent);
static int  CopyData(OMobj_id ctx_id, OMobj_id out_ctx_id,
		     OMobj_id obj_id, OMobj_id out_obj_id,
		     OMobj_id par_id);

static int  CreatePrim (OMobj_id par_id, char *name, int type, OMobj_id *obj_id);
static int  CreateGroup(OMobj_id par_id, char *name, OMobj_id *out_obj_id);

static int  CopyPrim(OMobj_id out_id, OMobj_id obj_id);
static int  CopyGroup(OMobj_id out_id, OMobj_id obj_id);

int DVarray_collect_delete(OMobj_id     obj_id)
{
	OMobj_id nout;

	nout = OMfind_subobj(obj_id, OMstr_to_name("arr_out"), OM_OBJ_RW);
	if (!OMis_null_obj(nout)) {
		OMclr_obj_atts(nout, OM_atts_locked);
		OMuser_destroy_obj(nout);
	}
	return(1);
}

int DVarray_collect_update(OMobj_id     obj_id, 
			   OMevent_mask event_mask, 
			   int		seq_num	)
{
	OMobj_id in, out, nout, val_id;
	int  stat, index, dim, run;
	char class_name[MAX_NAME_LEN];

	in = OMfind_subobj(obj_id, OMstr_to_name("in"), OM_OBJ_RD);
	if (OMget_obj_val(in, &in) != 1)
		return(0);

	out = OMfind_subobj(obj_id, OMstr_to_name("out"), OM_OBJ_RW);

	if (OMget_name_int_val(obj_id, OMstr_to_name("index"), &index) != 1 || index < 0) {
		ERR_RETURN("bad index value");
	}
	if (OMget_name_int_val(obj_id, OMstr_to_name("dim"), &dim) != 1 || dim <=0)
		return(0);
	if (OMget_name_int_val(obj_id, OMstr_to_name("run"), &run) != 1) {
		ERR_RETURN("error getting run");
	}
	if (index > dim -1) {
		ERR_RETURN("bad index value");
	}

	if (run == 0) {
		/* loop has not run yet */
		/* start it over         */
		index = 0;
		run = 1;
		if (OMset_name_int_val(obj_id, OMstr_to_name("index"), index) != 1) {
			ERR_RETURN("error setting index");
		}
		if (OMset_name_int_val(obj_id, OMstr_to_name("run"), run) != 1) {
			ERR_RETURN("error setting run");
		}
		return(1);
	}
	if (index == 0) {
		/* delete arr_out object */
		nout = OMfind_subobj(obj_id, OMstr_to_name("arr_out"), OM_OBJ_RW);
		if (!OMis_null_obj(nout)) {
			OMclr_obj_atts(nout, OM_atts_locked);
			OMuser_destroy_obj(nout);
		}

		/* create new arr_out object */
		stat = GetInClass(in, class_name);
		if (stat != 1) {
			ERR_RETURN("error getting input object class");
		}
		if (OMparse_buffer(obj_id, class_name,0) != 1) {
			ERR_RETURN("error creating out object");
		}
		nout = OMfind_subobj(obj_id, OMstr_to_name("arr_out"), OM_OBJ_RW);
		if (OMis_null_obj(nout)) {
			ERR_RETURN("cannot find arr_out object");
		}
		stat = SetOutDim (nout, dim);
		if (stat != 1) {
			ERR_RETURN("error setting out dim");
		}
	}
	else {
		nout = OMfind_subobj(obj_id, OMstr_to_name("arr_out"), OM_OBJ_RW);
		if (OMis_null_obj(nout)) {
			ERR_RETURN("cannot find arr_out object");
		}
	}

	/* copy data from in into arr_out[index] */

	if (OMget_array_val(nout, index, &val_id, OM_OBJ_RW) != 1) {
		ERR_RETURN("cannot get out array member");
	}
	strcpy(class_name, "");

	stat = CopyData(in, val_id, in, val_id, in);
	if (stat != 1) {
		ERR_RETURN("error setting out object");
	}
	if (index == dim-1) {
		/* end of loop */
		nout = OMfind_subobj(obj_id, OMstr_to_name("arr_out"), OM_OBJ_RW);
		if (OMis_null_obj(nout)) {
			ERR_RETURN("cannot find arr_out object");
		}

		stat = OMset_obj_ref(out, nout, 0);
		if (stat < 1) {
			ERR_RETURN("error setting reference");
		}
		run = 0;
	}
	else {
		/* continue run loop */
		index++;
		run = 1;
		if (OMset_name_int_val(obj_id, OMstr_to_name("index"), index) != 1) {
			ERR_RETURN("error setting index");
		}
	}
	if (OMset_name_int_val(obj_id, OMstr_to_name("run"), run) != 1) {
		ERR_RETURN("error setting run");
	}

	return(1);
}

static int  GetInClass(OMobj_id obj_id, char *class_name)
{
	OMobj_id classes[MAX_CLASS];
	OMobj_name obj_name;
	int i, stat, num_classes;

	stat = OMget_user_classes(obj_id, classes, &num_classes, MAX_CLASS, 0, OM_CLASS_FOLLOW_REFS);
	if (stat != 1 || num_classes == 0) {
		strcpy(class_name, "group");
	}
	else {
		stat = OMget_obj_name(classes[0], &obj_name);
		if (stat != 1) {
			ERR_RETURN(" cannot get class name");
		}
		strcpy(class_name, OMname_to_str(obj_name));
		for (i=1; i<num_classes; i++) {
			stat = OMget_obj_name(classes[i], &obj_name);
			if (stat != 1) {
				ERR_RETURN(" cannot get class name");
			}
			strcat(class_name, "+");
			strcat(class_name, OMname_to_str(obj_name));
		}
	}
	strcat(class_name, "+write arr_out;");
	return(1);
}

static int SetOutDim (OMobj_id obj_id, int dim)
{
	OMobj_id delem, dims_elem;
	int stat;

	delem = OMcreate_obj(OM_NULL_NAME, OM_type_raw_int, OMnull_obj,
			     OMnull_obj, OMlocal_proc_id);
	if (OMis_null_obj(delem))
		return(0);
	OMset_int_val(delem, dim);
	dims_elem = OMcreate_obj(OM_name_dims, OM_type_val_list, obj_id,
				 OMnull_obj, OMlocal_proc_id);
	if (OMis_null_obj(dims_elem))
		return(0);
	stat = OMadd_subobj(dims_elem, delem);

	return(stat);
}

/* 64-bit porting. Only Modified Internally */
static int CopyData(OMobj_id ctx_id, OMobj_id out_ctx_id,
		    OMobj_id obj_id, OMobj_id out_obj_id,
		    OMobj_id par_id)
{
	OMobj_id val_id, out_val_id;
	OMobj_name obj_name;
	int i, type, ndim;
	int stat=1;
	xp_long ndim_w, dims[OM_ARRAY_MAXDIM];

	if (OMget_data_type(obj_id, &type) == 1) {
		/* prim object */

		if(OMget_obj_atts(obj_id, OM_atts_virtual) == 1 ||
		   OMget_obj_atts(obj_id, OM_atts_wasvirtual) == 1 ||
		   OMget_obj_atts(obj_id, OM_atts_nosave) == 1)
			return(1);
		stat = OMget_obj_name(obj_id, &obj_name);
		if (stat != 1) {
			ERR_RETURN(" cannot get obj name");
		}
		if (AreTheSame(par_id, obj_id, OMname_to_str(obj_name)))
			return(1);

		/* else write it out */
		stat = CopyPrim(out_obj_id, obj_id);
	}
	else {	/* we are a group */
		if (OMget_array_dims(obj_id,&ndim,dims) == 1 && ndim != 0) {
			/* We are an array of groups*/
			OMget_array_size(obj_id, &ndim_w);
			ndim = (int)ndim_w;
			for (i = 0; i < ndim; i++) {
				if (OMget_array_val(obj_id,i,&val_id,OM_OBJ_RD) == 1 &&
				    OMget_obj_val(val_id,&val_id) == 1 && 
				    OMget_array_val(out_obj_id,i,&out_val_id,OM_OBJ_RW) == 1 &&
				    OMget_obj_val(out_val_id,&out_val_id) == 1) {
					CopyData(ctx_id, out_ctx_id, val_id, out_val_id,
						 obj_id);
				}
			}
		}
		else {  /* just group */
			stat = CopyGroup (out_obj_id, obj_id);
			if (stat != 1) {
				ERR_RETURN("cannot set object");
			}
			OMget_num_subobjs_mode(obj_id,&ndim_w,OM_SUB_SYMBOLIC);
			ndim = (int)ndim_w;
			for (i = 0; i < ndim; i++) {
				/* process primitives first */
				if (OMget_array_subobj(obj_id,i,&val_id,OM_OBJ_RD | OM_SUB_SYMBOLIC) == 1) {
					stat = OMget_obj_name(val_id, &obj_name);
					if (stat != 1)
						return(stat);
					val_id = OMfind_subobj(obj_id, obj_name,OM_OBJ_RD);
					if(OMis_null_obj(val_id)) {
						ERR_RETURN("cannot find object  ");
					}
					if (OMget_data_type(val_id, &type) == 1) {
						out_val_id = OMfind_str_subobj(out_obj_id, OMname_to_str(obj_name),OM_OBJ_RW);
						if(OMis_null_obj(out_val_id)) {
							/* prim object does not exist in the template */
							stat = CreatePrim (out_obj_id, OMname_to_str(obj_name), 
									   type, &out_val_id);
							if (stat != 1) {
								ERR_RETURN("cannot create object ");
							}
							stat = CopyPrim(out_val_id, val_id);
							if (stat != 1) {
								ERR_RETURN("cannot set object ");
							}
						}
						else {
							/* prim object does exist in the template, set it*/
							CopyData(ctx_id, out_ctx_id, val_id, out_val_id,
								 obj_id);
						}
					}
				}
			}
			for (i = 0; i < ndim; i++) {
				/* process groups second */
				if (OMget_array_subobj(obj_id,i,&val_id,OM_OBJ_RD | OM_SUB_SYMBOLIC) == 1) {
					stat = OMget_obj_name(val_id, &obj_name);
					if (stat != 1)
						return(stat);
					val_id = OMfind_subobj(obj_id, obj_name,OM_OBJ_RD);
					if(OMis_null_obj(val_id)) {
						ERR_RETURN("cannot find object");
					}
					if (OMget_data_type(val_id, &type) != 1) {
						out_val_id = OMfind_str_subobj(out_obj_id, OMname_to_str(obj_name),OM_OBJ_RW);
						if(OMis_null_obj(out_val_id)) {
							/* group object does not exist in the template */
							stat = CreateGroup (out_obj_id, OMname_to_str(obj_name), 
									   &out_val_id);
							if (stat != 1) {
								ERR_RETURN("cannot create object");
							}
						}
						stat = CopyGroup (out_val_id, val_id);
						if (stat != 1) {
							ERR_RETURN("cannot set object ");
						}
						CopyData(ctx_id, out_ctx_id, val_id, out_val_id,
							 obj_id);
					}
				}
			}
		}
	}
	return(stat);

}

static int AreTheSame(OMobj_id ctx_id, OMobj_id obj_id, char *path_from_parent)
{
	OMobj_id classes[MAX_CLASS], class_sub_id;
	int i, num_classes, type, mode;

	if (OMget_data_type(obj_id, &type) == 1)
		mode =  OM_MATCH_STRUCT |OM_MATCH_EXACT | OM_MATCH_NO_ERROR;
	else
		mode =  OM_MATCH_EXACT | OM_MATCH_NO_ERROR;

	if (OMget_user_classes(ctx_id, classes, &num_classes,MAX_CLASS,0,OM_CLASS_FOLLOW_REFS) == 1) {
		for (i = num_classes-1; i >=0; i--) {
			class_sub_id = OMfind_str_subobj(classes[i],path_from_parent,
							 OM_OBJ_RD);
			if (!OMis_null_obj(class_sub_id)) {
				return(OMmatch_obj(class_sub_id, obj_id, mode));
			}
		}
	}
	return(0);
}

/* 64-bit porting. Only Modified Internally */
static int CopyPrim(OMobj_id out_id, OMobj_id obj_id)
{
	int    type, stat, ndim;
	xp_long i, size, dims[OM_ARRAY_MAXDIM];
	char  *str, *val;
	int    ival;
	xp_long lval;
	double dval;
	OMobj_id dims_id;

	if (OMget_data_type(obj_id, &type) != 1) {
		sprintf(buf, "cannot get type for object");
		ERR_RETURN(buf);
	}
	if (OMset_data_type(out_id, type) != 1) {
		sprintf(buf, "cannot set type for object");
		ERR_RETURN(buf);
	}
	if (OMget_array_dims(obj_id, &ndim, dims) != 1 || ndim == 0) {
		/* scalar */
		switch (type) {
		      case DTYPE_CHAR:
		      case DTYPE_BYTE:
		      case DTYPE_SHORT:
		      case DTYPE_INT:
			stat = OMget_int_val(obj_id, &ival);
			if (stat != 1) {
				return(1);
			}
			stat = OMset_int_val(out_id, ival);
			if (stat != 1) {
				sprintf(buf, "cannot set value for object");
				ERR_RETURN(buf);
			}
			break;

		      case DTYPE_LONG:
			stat = OMget_long_val(obj_id, &lval);
			if (stat != 1) {
				return(1);
			}
			stat = OMset_long_val(out_id, lval);
			if (stat != 1) {
				sprintf(buf, "cannot set value for object");
				ERR_RETURN(buf);
			}
			break;

		      case DTYPE_FLOAT:
		      case DTYPE_DOUBLE:
			stat = OMget_real_val(obj_id, &dval);
			if (stat != 1) {
				return(1);
			}
			stat = OMset_real_val(out_id, dval);
			if (stat != 1) {
				sprintf(buf, "cannot set value for object");
				ERR_RETURN(buf);
			}
			break;

		      case DTYPE_STRING:
			str = NULL;
			stat = OMget_str_val(obj_id, &str, 0);
			if (stat != 1) {
				return(1);
			}
			stat = OMset_str_val(out_id, str);
			free(str);
			if (stat != 1) {
				sprintf(buf, "cannot set value for object");
				ERR_RETURN(buf);
			}
			break;
		      case DTYPE_UNSET:
			return(1);

		      default:
			sprintf(buf, "unsupported type %d for object", type);
			ERR_RETURN(buf);
		}
	}
	else if (type == DTYPE_STRING) {
		if (OMget_array_size(obj_id, &size) != 1 || size == 0) {
			return(1);
		}
		for (i=0; i<size; i++) {
			stat = OMget_str_array_val(obj_id, i, &str, 0);
			if (stat != 1) {
				return(1);
			}
			stat = OMset_str_array_val(out_id, i, str);
			free(str);
			if (stat != 1) {
				sprintf(buf, "cannot set value for object");
				ERR_RETURN(buf);
			}
		}
	}
	else {
		/* array */
		if (OMget_array_dims(out_id, &ndim, dims) != 1 || ndim == 0) {
			/* add dimensions */
			dims_id = OMparse_dims_to_obj(out_id, "[]");
			if (OMis_null_obj(dims_id)) {
				ERR_RETURN("cannot parse dims object");
			}
			OMadd_subobj(out_id, dims_id);
		}
		val = NULL;
		size = 0;
		type = DTYPE_UNSET;
		stat = OMget_array_sz(obj_id, &type, (char **)&val, &size, OM_GET_ARRAY_RD);
		if (stat != 1) {
			return(0);
		}
		if (OMset_array(out_id, type, (void *)val, size, OM_SET_ARRAY_COPY) != 1) {
			sprintf(buf, "cannot set value for object");
			ERR_RETURN(buf);
		}
		ARRfree(val);
	}
	return(1);
}

static int CreateGroup(OMobj_id par_id, char *name, OMobj_id *out_obj_id)
{
	*out_obj_id = OMcreate_obj(OMstr_to_name(name), OMstr_to_type("group"), par_id, 
			      OMnull_obj, OMlocal_proc_id);
	if (OMis_null_obj(*out_obj_id))
		return(0);
	return(1);
}

/* 64-bit porting. Only Modified Internally */
static int CreatePrim (OMobj_id par_id, char *name, int type, OMobj_id *obj_id)
{
	char ctype[64];
	switch (type) {
	      case DTYPE_CHAR:
		strcpy(ctype, "byte");
		break;

	      case DTYPE_BYTE:
		strcpy(ctype, "char");
		break;

	      case DTYPE_SHORT:
		strcpy(ctype, "short");
		break;

	      case DTYPE_INT:
		strcpy(ctype, "int");
		break;

	      case DTYPE_LONG:
		strcpy(ctype, "long");
		break;

	      case DTYPE_FLOAT:
		strcpy(ctype, "float");
		break;

	      case DTYPE_DOUBLE:
		strcpy(ctype, "double");
		break;

	      case DTYPE_STRING:
		strcpy(ctype, "string");
		break;

	      case DTYPE_UNSET:
		return(1);

	      default:
		sprintf(buf, "unsupported type %d for object %s", type, name);
		ERR_RETURN(buf);
	}
	*obj_id = OMcreate_obj(OMstr_to_name(name), OMstr_to_type(ctype), par_id, 
			       OMnull_obj, OMlocal_proc_id);
	if (OMis_null_obj(*obj_id))
		return(0);
	return(1);
}

/* 64-bit porting. Only Modified Internally */
static int CopyGroup(OMobj_id out_obj_id, OMobj_id obj_id)
{
	OMobj_id classes[MAX_CLASS];
	int   i, num_classes, stat,  ndim;
	xp_long ndim_w, ondim_w, dims[OM_ARRAY_MAXDIM];
	OMobj_id dims_id;

	if (OMget_array_dims(obj_id, &ndim, dims) == 1  && ndim != 0) {
		if (OMget_array_dims(out_obj_id, &ndim, dims) != 1 || ndim == 0) {
			/* add dimensions */
			dims_id = OMparse_dims_to_obj(out_obj_id, "[]");
			if (OMis_null_obj(dims_id)) {
				ERR_RETURN("cannot parse dims object");
			}
			OMadd_subobj(out_obj_id, dims_id);
		}
		if (OMget_array_size(obj_id, &ndim_w) == 1) {
			if (OMget_array_size(out_obj_id, &ondim_w) != 1 || ndim_w != ondim_w)
				OMset_array_size(out_obj_id, ndim_w);
		}
	}

	stat = OMget_user_classes(obj_id, classes, &num_classes, MAX_CLASS, 0, OM_CLASS_FOLLOW_REFS);
	if (stat != 1)
		num_classes = 0;
	for (i=0; i<num_classes; i++) {
		stat = OMmerge_obj(classes[i], out_obj_id, 0);
		if (stat != 1)
			return(0);
	}
	return(1);
}

