/*
			Copyright (c) 1994 by
			Advanced Visual Systems Inc.
			All Rights Reserved

	This software comprises unpublished confidential information of
	Advanced Visual Systems Inc. and may not be used, copied or made
	available to anyone, except in accordance with the license
	under which it is furnished.

	This file is under Perforce control
	$Id: //depot/express/fcs70/modules/bounds.c#1 $
*/

#define XP_WIDE_API	/* Use Wide APIs */

#include <avs/util.h>
#include <avs/err.h>
#include <avs/dtype.h>
#include <avs/math.h>
#include <avs/om.h>
#include <avs/fld.h>

#define METHOD_SUCCESS 1
#define METHOD_FAILURE 0

#define ERR_RETURN(A) {ERRerror("bounds", 0, ERR_ORIG, A); \
                       return METHOD_FAILURE;}

#define MAX_NAME_SIZE 1024

#define ALLOC_SIZE 1000
#define ALLOC_GROW 1.5

#define CLEAN_AND_RETURN \
    if (h.xyz) \
        ARRfree(h.xyz); \
    if (h.data) \
        ARRfree(h.data); \
    if (h.line_conn) \
        ARRfree(h.line_conn); \
    if (h.quad_conn) \
        ARRfree(h.quad_conn); \
    ARRfree(dims); \
    return METHOD_FAILURE;

typedef struct _OUTmesh {
    xp_long nnodes;
    float *xyz;
    char *data;
    xp_long   size;
    xp_long   size_incr;    
    xp_long   nline;
    xp_long   *line_conn;
    xp_long   line_size;
    xp_long   line_incr;
    xp_long   nquad;
    xp_long   *quad_conn;
    xp_long   quad_size;
    xp_long   quad_incr;    
} OUTmesh;


int FUNCbounds (OMobj_id in, int hull, int edges, int faces,
                int imin, int imax, int jmin, int jmax, int kmin, int kmax,
                int data, int comp, OMobj_id out);

static int get_line (OMobj_id in, OUTmesh *h, int ndim, xp_long *dims, int dir,
    xp_long *min_ind, xp_long *max_ind, int nspace, int data, int comp, int veclen,
    int dtype);

static int get_line_y (OUTmesh *h, xp_long *dims, xp_long ind,
    int nspace, int data, int veclen, int dtype);

static int get_face (OMobj_id in, OUTmesh *h, int ndim, xp_long *dims, int dir1,
    int dir2, xp_long ind, int norm, int nspace, int data, int comp, int veclen,
    int dtype);


int DVbounds_update(OMobj_id elem)
{
    OMobj_id src_id, dst_id, e_id;
    int hull, data, comp, imin, imax, jmin, jmax, kmin, kmax, edges, faces;

    src_id = OMfind_subobj(elem,OMstr_to_name("in"),OM_OBJ_RD);
    dst_id = OMfind_subobj(elem,OMstr_to_name("out"),OM_OBJ_RW);

    e_id = OMfind_subobj(elem,OMstr_to_name("hull"),OM_OBJ_RD);
    OMget_int_val(e_id, &hull);
    e_id = OMfind_subobj(elem,OMstr_to_name("edges"),OM_OBJ_RD);
    OMget_int_val(e_id, &edges);
    e_id = OMfind_subobj(elem,OMstr_to_name("faces"),OM_OBJ_RD);
    OMget_int_val(e_id, &faces);

    e_id = OMfind_subobj(elem,OMstr_to_name("imin"),OM_OBJ_RD);
    OMget_int_val(e_id, &imin);
    e_id = OMfind_subobj(elem,OMstr_to_name("imax"),OM_OBJ_RD);
    OMget_int_val(e_id, &imax);
    e_id = OMfind_subobj(elem,OMstr_to_name("jmin"),OM_OBJ_RD);
    OMget_int_val(e_id, &jmin);
    e_id = OMfind_subobj(elem,OMstr_to_name("jmax"),OM_OBJ_RD);
    OMget_int_val(e_id, &jmax);
    e_id = OMfind_subobj(elem,OMstr_to_name("kmin"),OM_OBJ_RD);
    OMget_int_val(e_id, &kmin);
    e_id = OMfind_subobj(elem,OMstr_to_name("kmax"),OM_OBJ_RD);
    OMget_int_val(e_id, &kmax);

    e_id = OMfind_subobj(elem,OMstr_to_name("data"),OM_OBJ_RD);
    OMget_int_val(e_id, &data);

    if (data) {
        e_id = OMfind_subobj(elem,OMstr_to_name("component"),OM_OBJ_RD);
        OMget_int_val(e_id, &comp);
    }
    if (FUNCbounds(src_id, hull, edges, faces,
                   imin, imax, jmin, jmax, kmin, kmax,
                   data, comp, dst_id) != METHOD_SUCCESS)
        return METHOD_FAILURE;

    return METHOD_SUCCESS;
}

/* 64-bit porting. Only Modified Internally */
int DV_ARRbounds_update(OMobj_id elem, OMevent_mask event_mask, int seq_num)
{
    OMobj_id src_arr_id, dst_arr_id, e_id;
    xp_long i, num_fields;
    int hull, data, comp, imin, imax, jmin, jmax, kmin, kmax, edges, faces;
    int stat;

    OMobj_id src_id, dst_id;
    OMobj_id src_xform_id, dst_xform_id;
    OMobj_name xform_name = OMstr_to_name("xform");

    src_arr_id = OMfind_subobj(elem,OMstr_to_name("in"),OM_OBJ_RD);
    dst_arr_id = OMfind_subobj(elem,OMstr_to_name("out"),OM_OBJ_RW);

    stat = OMget_array_size( src_arr_id, &num_fields );
    if( stat != OM_STAT_SUCCESS ) {
        if( !OMis_null_obj( dst_arr_id ) ) OMset_array_size( dst_arr_id, 0 );
        return METHOD_FAILURE;
    }
    else if( num_fields == 0 ) {
        if( !OMis_null_obj( dst_arr_id ) ) OMset_array_size( dst_arr_id, 0 );
        return METHOD_SUCCESS;
    }

    e_id = OMfind_subobj(elem,OMstr_to_name("hull"),OM_OBJ_RD);
    OMget_int_val(e_id, &hull);
    e_id = OMfind_subobj(elem,OMstr_to_name("edges"),OM_OBJ_RD);
    OMget_int_val(e_id, &edges);
    e_id = OMfind_subobj(elem,OMstr_to_name("faces"),OM_OBJ_RD);
    OMget_int_val(e_id, &faces);

    e_id = OMfind_subobj(elem,OMstr_to_name("imin"),OM_OBJ_RD);
    OMget_int_val(e_id, &imin);
    e_id = OMfind_subobj(elem,OMstr_to_name("imax"),OM_OBJ_RD);
    OMget_int_val(e_id, &imax);
    e_id = OMfind_subobj(elem,OMstr_to_name("jmin"),OM_OBJ_RD);
    OMget_int_val(e_id, &jmin);
    e_id = OMfind_subobj(elem,OMstr_to_name("jmax"),OM_OBJ_RD);
    OMget_int_val(e_id, &jmax);
    e_id = OMfind_subobj(elem,OMstr_to_name("kmin"),OM_OBJ_RD);
    OMget_int_val(e_id, &kmin);
    e_id = OMfind_subobj(elem,OMstr_to_name("kmax"),OM_OBJ_RD);
    OMget_int_val(e_id, &kmax);

    e_id = OMfind_subobj(elem,OMstr_to_name("data"),OM_OBJ_RD);
    OMget_int_val(e_id, &data);

    if (data) {
        e_id = OMfind_subobj(elem,OMstr_to_name("component"),OM_OBJ_RD);
        OMget_int_val(e_id, &comp);
    }

    stat = OMset_array_size( dst_arr_id, num_fields );
    if( stat != OM_STAT_SUCCESS ) return METHOD_FAILURE;

    for( i = 0; i < num_fields; ++i ) {
        /* Should be OK for some of the blocks to be empty */
        stat = OMget_array_val( src_arr_id, i, &src_id, OM_OBJ_RD );
        if( stat != OM_STAT_SUCCESS ) continue;

        stat = OMget_array_val( dst_arr_id, i, &dst_id, OM_OBJ_RW );
        if( stat != OM_STAT_SUCCESS ) continue;

        stat = FUNCbounds(src_id, hull, edges, faces,
                          imin, imax, jmin, jmax, kmin, kmax,
                          data, comp, dst_id);
#if 0
        if( stat != METHOD_SUCCESS ) {
            ERRerror( "bounds", 1, ERR_ORIG,
                      "Error while processing field: %ld", i );
        }
#endif

        /* This can be done easily in V, but its faster in C. */
        src_xform_id = OMfind_subobj(src_id, xform_name, OM_OBJ_RD);
        dst_xform_id = OMfind_subobj(dst_id, xform_name ,OM_OBJ_RW);
        OMset_obj_ref( dst_xform_id, src_xform_id, 0 );
    }

    return METHOD_SUCCESS;
}

/* 64-bit porting. Only Modified Internally */
int FUNCbounds (OMobj_id in, int hull, int edges, int faces,
                int imin, int imax, int jmin, int jmax, int kmin, int kmax,
                int data, int comp, OMobj_id out)
{
	OUTmesh  h;
	OMobj_id cell_set;
	xp_long *dims, i_w;
	int i, ndim, stat, ncomp, nspace, data_id, null_flag, dims_size;
	xp_long min_rng[4], max_rng[4];
	int out_nsets, set;
	int veclen, dtype;
	char  units[MAX_NAME_SIZE], label[MAX_NAME_SIZE];
	double null_value;

	if (FLDget_dims (in, &dims, &dims_size) != 1) {
		ERR_RETURN("Error getting dims");
	}
	if (FLDget_ndim (in, &ndim) != 1) {
		ERR_RETURN("Error getting ndim");
	}
	if (FLDget_nspace (in, &nspace) != 1) {
		ERR_RETURN("Error getting nspace");
	}
	dtype = 0;
	veclen = 1;
	if (data) {
		if (FLDget_node_data_ncomp(in, &ncomp) != 1) {
			data = 0;
		}
		else {
			if (comp >= ncomp) {
				ERRerror("bounds", 1, ERR_ORIG, "data component %d is out of range, assume 0-th", comp);
				comp = 0;
			}
			if (FLDget_node_data_veclen(in, comp, &veclen) != 1) {
				ERR_RETURN("Error getting veclen");
			}
			if (FLDget_node_data_type(in, comp, &dtype) != 1) {
				ERR_RETURN("Error getting data type");
			}
		}
	}
	/********************************/
	/*   Free pre-allocated arrays  */
	/********************************/
	h.xyz = NULL;
	h.data = NULL;
	h.line_conn = NULL;
	h.quad_conn = NULL;

	if (FLDset_ncell_sets (out, 0) != 1) {
		ERR_RETURN("Error setting ncell_sets");
	}
	if (FLDset_nnodes (out, 0) != 1) {
		ERR_RETURN("Error setting nnodes");
	}
	if (FLDset_coord(out, h.xyz, 0, OM_SET_ARRAY_FREE) != 1) {
		ERR_RETURN("Error setting coord");
	}
	if (FLDset_node_data_ncomp (out, 0) != 1) {
		ERR_RETURN("Error setting nnode_data");
	}

	h.nnodes = 0;
	h.size = ALLOC_SIZE;
	h.size_incr = ALLOC_SIZE;
	h.xyz = (float *)ARRalloc(NULL, DTYPE_FLOAT, nspace*h.size, NULL);
	if (data)
		h.data = (char *)ARRalloc(NULL, dtype, h.size*veclen, NULL);
	else
		h.data = NULL;
	h.nline = 0;
	h.nquad = 0;
	h.line_size = ALLOC_SIZE;
	h.line_incr = ALLOC_SIZE;
	h.line_conn = (xp_long *)ARRalloc(NULL, DTYPE_LONG, 3*h.line_size, NULL);
	h.quad_size = ALLOC_SIZE;
	h.quad_incr = ALLOC_SIZE;
	h.quad_conn = (xp_long *)ARRalloc(NULL, DTYPE_LONG, 4*h.quad_size, NULL);

	if (hull) {
		min_rng[0] = 0;
		max_rng[0] = dims[0];
		for (i=1; i<ndim; i++) {
			min_rng[i] = 0;
			max_rng[i] = 1;
		}
		stat = get_line (in, &h, ndim, dims, 0, min_rng, max_rng, nspace, data, comp, veclen, dtype);
		if (!stat) {
			CLEAN_AND_RETURN;
		}
		if (ndim >1) {
			min_rng[0] = 0;
			max_rng[0] = dims[0];
			min_rng[1] = dims[1]-1;
			max_rng[1] = dims[1];
			for (i=2; i<ndim; i++) {
				min_rng[i] = 0;
				max_rng[i] = 1;
			}
			stat = get_line (in, &h, ndim, dims, 0, min_rng, max_rng, nspace, data, comp, veclen, dtype);
			if (!stat) {
				CLEAN_AND_RETURN;
			}
			min_rng[0] = 0;
			max_rng[0] = 1;
			min_rng[1] = 0;
			max_rng[1] = dims[1];
			for (i=2; i<ndim; i++) {
				min_rng[i] = 0;
				max_rng[i] = 1;
			}
			stat = get_line (in, &h, ndim, dims, 1, min_rng, max_rng, nspace, data, comp, veclen, dtype);
			if (!stat) {
				CLEAN_AND_RETURN;
			}
			min_rng[0] = dims[0]-1;
			max_rng[0] = dims[0];
			min_rng[1] = 0;
			max_rng[1] = dims[1];
			for (i=2; i<ndim; i++) {
				min_rng[i] = 0;
				max_rng[i] = 1;
			}
			stat = get_line (in, &h, ndim, dims, 1, min_rng, max_rng, nspace, data, comp, veclen, dtype);
			if (!stat) {
				CLEAN_AND_RETURN;
			}
		}
		if (ndim > 2) {
			min_rng[2] = dims[2]-1;
			max_rng[2] = dims[2];

			min_rng[0] = 0;
			max_rng[0] = dims[0];
			min_rng[1] = 0;
			max_rng[1] = 1;
			stat = get_line (in, &h, ndim, dims, 0, min_rng, max_rng, nspace, data, comp, veclen, dtype);
			if (!stat) {
				CLEAN_AND_RETURN;
			}

			min_rng[0] = 0;
			max_rng[0] = dims[0];
			min_rng[1] = dims[1]-1;
			max_rng[1] = dims[1];
			stat = get_line (in, &h, ndim, dims, 0, min_rng, max_rng, nspace, data, comp, veclen, dtype);
			if (!stat) {
				CLEAN_AND_RETURN;
			}

			min_rng[0] = 0;
			max_rng[0] = 1;
			min_rng[1] = 0;
			max_rng[1] = dims[1];
			stat = get_line (in, &h, ndim, dims, 1, min_rng, max_rng, nspace, data, comp, veclen, dtype);
			if (!stat) {
				CLEAN_AND_RETURN;
			}

			min_rng[0] = dims[0]-1;
			max_rng[0] = dims[0];
			min_rng[1] = 0;
			max_rng[1] = dims[1];
			stat = get_line (in, &h, ndim, dims, 1, min_rng, max_rng, nspace, data, comp, veclen, dtype);
			if (!stat) {
				CLEAN_AND_RETURN;
			}

			min_rng[2] = 0;
			max_rng[2] = dims[2];

			min_rng[0] = 0;
			max_rng[0] = 1;
			min_rng[1] = 0;
			max_rng[1] = 1;
			stat = get_line (in, &h, ndim, dims, 2, min_rng, max_rng, nspace, data, comp, veclen, dtype);
			if (!stat) {
				CLEAN_AND_RETURN;
			}

			min_rng[0] = dims[0]-1;
			max_rng[0] = dims[0];
			min_rng[1] = 0;
			max_rng[1] = 1;
			stat = get_line (in, &h, ndim, dims, 2, min_rng, max_rng, nspace, data, comp, veclen, dtype);
			if (!stat) {
				CLEAN_AND_RETURN;
			}

			min_rng[0] = 0;
			max_rng[0] = 1;
			min_rng[1] = dims[1]-1;
			max_rng[1] = dims[1];
			stat = get_line (in, &h, ndim, dims, 2, min_rng, max_rng, nspace, data, comp, veclen, dtype);
			if (!stat) {
				CLEAN_AND_RETURN;
			}

			min_rng[0] = dims[0]-1;
			max_rng[0] = dims[0];
			min_rng[1] = dims[1]-1;
			max_rng[1] = dims[1];
			stat = get_line (in, &h, ndim, dims, 2, min_rng, max_rng, nspace, data, comp, veclen, dtype);
			if (!stat) {
				CLEAN_AND_RETURN;
			}
		}
	}
	if (edges) {
		if (imin && ndim == 3) {
			min_rng[0] = 0;
			max_rng[0] = 1;

			min_rng[1] = 0;
			max_rng[1] = dims[1];
			for (i_w=0; i_w<dims[2]; i_w++) {
				min_rng[2] = i_w;
				max_rng[2] = i_w+1;
				stat = get_line (in, &h, ndim, dims, 1, min_rng, max_rng, nspace, data, comp, veclen, dtype);
				if (!stat) {
					CLEAN_AND_RETURN;
				}
			}
			min_rng[2] = 0;
			max_rng[2] = dims[2];
			for (i_w=0; i_w<dims[1]; i_w++) {
				min_rng[1] = i_w;
				max_rng[1] = i_w+1;
				stat = get_line (in, &h, ndim, dims, 2, min_rng, max_rng, nspace, data, comp, veclen, dtype);
				if (!stat) {
					CLEAN_AND_RETURN;
				}
			}
		}
		if (imax && ndim == 3) {
			min_rng[0] = dims[0]-1;
			max_rng[0] = dims[0];

			min_rng[1] = 0;
			max_rng[1] = dims[1];
			for (i_w=0; i_w<dims[2]; i_w++) {
				min_rng[2] = i_w;
				max_rng[2] = i_w+1;
				stat = get_line (in, &h, ndim, dims, 1, min_rng, max_rng, nspace, data, comp, veclen, dtype);
				if (!stat) {
					CLEAN_AND_RETURN;
				}
			}
			min_rng[2] = 0;
			max_rng[2] = dims[2];
			for (i_w=0; i_w<dims[1]; i_w++) {
				min_rng[1] = i_w;
				max_rng[1] = i_w+1;
				stat = get_line (in, &h, ndim, dims, 2, min_rng, max_rng, nspace, data, comp, veclen, dtype);
				if (!stat) {
					CLEAN_AND_RETURN;
				}
			}
		}
		if (jmin && ndim == 3) {
			min_rng[1] = 0;
			max_rng[1] = 1;

			min_rng[0] = 0;
			max_rng[0] = dims[0];
			for (i_w=0; i_w<dims[2]; i_w++) {
				min_rng[2] = i_w;
				max_rng[2] = i_w+1;
				stat = get_line (in, &h, ndim, dims, 0, min_rng, max_rng, nspace, data, comp, veclen, dtype);
				if (!stat) {
					CLEAN_AND_RETURN;
				}
			}
			min_rng[2] = 0;
			max_rng[2] = dims[2];
			for (i_w=0; i_w<dims[0]; i_w++) {
				min_rng[0] = i_w;
				max_rng[0] = i_w+1;
				stat = get_line (in, &h, ndim, dims, 2, min_rng, max_rng, nspace, data, comp, veclen, dtype);
				if (!stat) {
					CLEAN_AND_RETURN;
				}
			}
		}
		if (jmax && ndim == 3) {
			min_rng[1] = dims[1]-1;
			max_rng[1] = dims[1];

			min_rng[0] = 0;
			max_rng[0] = dims[0];
			for (i_w=0; i_w<dims[2]; i_w++) {
				min_rng[2] = i_w;
				max_rng[2] = i_w+1;
				stat = get_line (in, &h, ndim, dims, 0, min_rng, max_rng, nspace, data, comp, veclen, dtype);
				if (!stat) {
					CLEAN_AND_RETURN;
				}
			}
			min_rng[2] = 0;
			max_rng[2] = dims[2];
			for (i_w=0; i_w<dims[0]; i_w++) {
				min_rng[0] = i_w;
				max_rng[0] = i_w+1;
				stat = get_line (in, &h, ndim, dims, 2, min_rng, max_rng, nspace, data, comp, veclen, dtype);
				if (!stat) {
					CLEAN_AND_RETURN;
				}
			}
		}
		if (kmin && ndim >1) {
			min_rng[2] = 0;
			max_rng[2] = 1;

			min_rng[0] = 0;
			max_rng[0] = dims[0];
			for (i_w=0; i_w<dims[1]; i_w++) {
				min_rng[1] = i_w;
				max_rng[1] = i_w+1;
				stat = get_line (in, &h, ndim, dims, 0, min_rng, max_rng, nspace, data, comp, veclen, dtype);
				if (!stat) {
					CLEAN_AND_RETURN;
				}
			}
			if (ndim == 2) {   /* try to avoid getting sub-array in y direction */
				min_rng[1] = 0;
				max_rng[1] = dims[1];
				for (i_w=0; i_w<dims[0]; i_w++) {
					stat = get_line_y (&h, dims, i_w, nspace, data, veclen, dtype);
					if (!stat) {
						CLEAN_AND_RETURN;
					}
				}
			}
			else {
				min_rng[1] = 0;
				max_rng[1] = dims[1];
				for (i_w=0; i_w<dims[0]; i_w++) {
					min_rng[0] = i_w;
					max_rng[0] = i_w+1;
					stat = get_line (in, &h, ndim, dims, 1, min_rng, max_rng, nspace, data, comp, veclen, dtype);
					if (!stat) {
						CLEAN_AND_RETURN;
					}
				}
			}
		}
		if (kmax && ndim == 3) {
			min_rng[2] = dims[2]-1;
			max_rng[2] = dims[2];

			min_rng[1] = 0;
			max_rng[1] = dims[1];
			for (i_w=0; i_w<dims[0]; i_w++) {
				min_rng[0] = i_w;
				max_rng[0] = i_w+1;
				stat = get_line (in, &h, ndim, dims, 1, min_rng, max_rng, nspace, data, comp, veclen, dtype);
				if (!stat) {
					CLEAN_AND_RETURN;
				}
			}
			min_rng[0] = 0;
			max_rng[0] = dims[0];
			for (i_w=0; i_w<dims[1]; i_w++) {
				min_rng[1] = i_w;
				max_rng[1] = i_w+1;
				stat = get_line (in, &h, ndim, dims, 0, min_rng, max_rng, nspace, data, comp, veclen, dtype);
				if (!stat) {
					CLEAN_AND_RETURN;
				}
			}
		}
	}
	if (faces) {
		if (imin && ndim>2) {
			stat = get_face (in, &h, ndim, dims, 1, 2, 0, -1, nspace, data, comp, veclen, dtype);
			if (!stat) {
				CLEAN_AND_RETURN;
			}
		}
		if (imax && ndim>2) {
			stat = get_face (in, &h, ndim, dims, 1, 2, dims[0]-1, 1, nspace, data, comp, veclen, dtype);
			if (!stat) {
				CLEAN_AND_RETURN;
			}
		}
		if (jmin && ndim>2) {
			stat = get_face (in, &h, ndim, dims, 0, 2, 0, 1, nspace, data, comp, veclen, dtype);
			if (!stat) {
				CLEAN_AND_RETURN;
			}
		}
		if (jmax && ndim>2) {
			stat = get_face (in, &h, ndim, dims, 0, 2, dims[1]-1, -1, nspace, data, comp, veclen, dtype);
			if (!stat) {
				CLEAN_AND_RETURN;
			}
		}
		if (kmin && ndim>1) {
			stat = get_face (in, &h, ndim, dims, 0, 1, 0, -1, nspace, data, comp, veclen, dtype);
			if (!stat) {
				CLEAN_AND_RETURN;
			}
		}
		if (kmax && ndim>2) {
			stat = get_face (in, &h, ndim, dims, 0, 1, dims[2]-1, 1, nspace, data, comp, veclen, dtype);
			if (!stat) {
				CLEAN_AND_RETURN;
			}
		}
	}
	if (FLDget_coord_units(in, units, MAX_NAME_SIZE) == 1) {
		if (FLDset_coord_units (out, units) != 1) {
			ERR_RETURN("Error setting units");
		}
	}
	if (FLDset_nspace (out, nspace) != 1) {
		ERR_RETURN("Error setting nspace");
	}
	if (FLDset_nnodes (out, h.nnodes) != 1) {
		ERR_RETURN("Error setting nnodes");
	}
	if (h.nnodes)
		if (FLDset_coord(out, h.xyz, nspace*h.nnodes, OM_SET_ARRAY_FREE) != 1) {
			ERR_RETURN("Error setting coord");
		}
	out_nsets = 0;
	if (FLDset_ncell_sets (out, out_nsets) != 1) {
		ERR_RETURN("Error setting ncell_sets");
	}
	set = 0;
	if (h.nline) {
		if (FLDadd_cell_set(out, "Polyline") != 1) {
			ERR_RETURN("Error setting cell type");
		}
		if (FLDget_cell_set(out, set, &cell_set) != 1) {
			ERR_RETURN("Error getting cell set");
		}
		if (FLDset_npolys(cell_set, h.nline) != 1) {
			ERR_RETURN("Error setting ncells");
		}
		if (FLDset_poly_connect(cell_set, h.line_conn, 2*h.nline,
					OM_SET_ARRAY_FREE) != 1) {
			ERR_RETURN("Error setting cell connect list");
		}
		set++;
	}
	else {
		ARRfree(h.line_conn);
	}
	if (h.nquad) {
		if (FLDadd_cell_set(out, "Quad") != 1) {
			ERR_RETURN("Error setting cell type");
		}
		if (FLDget_cell_set(out, set, &cell_set) != 1) {
			ERR_RETURN("Error getting cell set");
		}
		if (FLDset_ncells(cell_set, h.nquad) != 1) {
			ERR_RETURN("Error setting ncells");
		}
		if (FLDset_node_connect(cell_set, h.quad_conn, 4*h.nquad,
					OM_SET_ARRAY_FREE) != 1) {
			ERR_RETURN("Error setting cell connect list");
		}
	}
	else {
		ARRfree(h.quad_conn);
	}
	if (h.nnodes && data) {
		if (FLDset_node_data_ncomp (out, 1) != 1) {
			ERR_RETURN("Error setting nnode_data");
		}
		if (FLDget_node_data_units(in, comp, units, MAX_NAME_SIZE) != 1) {
			strcpy(units, "");
		}
		if (FLDget_node_data_label(in, comp, label, MAX_NAME_SIZE) != 1) {
			strcpy(label, "");
		}
		if (FLDget_node_null_data(in, comp, &null_flag, (char *)&null_value) != 1) {
			ERR_RETURN("cannot get null data");
		}
		if (FLDset_node_data_comp (out, 0, veclen, label, units) != 1) {
			ERR_RETURN("Error setting node component");
		}
		if (FLDget_node_data_id(in, comp, &data_id) == 1)
			FLDset_node_data_id(out, 0, data_id);
		if (FLDset_node_data(out, 0, (char *)h.data, dtype,
				     veclen*h.nnodes, OM_SET_ARRAY_FREE) != 1) {
			ERR_RETURN("Error setting node data");
		}
		if (FLDcopy_node_minmax(in, out, comp, 0) != 1) {
			ERR_RETURN("Error copying node minmax");
		}
		if (FLDcopy_node_minmax_vec(in, out, comp, 0) != 1) {
			ERR_RETURN("Error copying node minmax");
		}
		if (null_flag) {
			if (FLDset_node_null_data(out, 0, (char *)&null_value, dtype) != 1) {
				ERR_RETURN("Error setting null value");
			}
		}
		else {
			if (FLDset_node_null_flag(out, 0, 0) != 1) {
				ERR_RETURN("Error setting null flag");
			}
		}
	}
	else {
		if (FLDset_node_data_ncomp (out, 0) != 1) {
			ERR_RETURN("Error setting nnode_data");
		}
	}
	ARRfree(dims);
	return METHOD_SUCCESS;
}

/* 64-bit porting. Directly Modified */
static int get_line (in, h, ndim, dims, dir, min_ind, max_ind, nspace, data, comp, veclen, dtype)
    OMobj_id in;
    OUTmesh *h;
    xp_long *dims;
    xp_long *min_ind, *max_ind;
    int ndim, nspace, dir, data, comp, veclen, dtype;
{
	int i;
	xp_long  i_w;
	xp_long nnew;
	xp_long rdims[4], rmin[4], rmax[4];

	nnew = max_ind[dir]-min_ind[dir];
	if (h->nnodes+nnew >= h->size) {
		i_w = nnew/h->size_incr+1;
		h->size += i_w*h->size_incr;
		h->size_incr *= 1.5;
		h->xyz = (float *)ARRrealloc(h->xyz, DTYPE_FLOAT, nspace*h->size, NULL);
		if (data)
			h->data = (char *)ARRrealloc(h->data, dtype, 
						     h->size*veclen, NULL);
	}
	rdims[0] = nspace;
	rmin[0]=0;
	rmax[0]=nspace;
	for (i=0; i<ndim; i++) {
		rdims[i+1] = dims[i];
		rmin[i+1] = min_ind[i];
		rmax[i+1] = max_ind[i];
	}
	if (FLDget_sub_coord(in, ndim+1, rdims, rmin, rmax, h->xyz+h->nnodes*nspace) != 1) {
		ERR_RETURN("cannot get coordinates");
	}
	if (data) {
		rdims[0] = veclen;
		rmax[0]= veclen;
		if (FLDget_sub_node_data(in, comp, &dtype, ndim+1, rdims, rmin, rmax,
					 h->data+DTYPEtype_size[dtype]*h->nnodes*veclen) != 1) {
			ERR_RETURN("cannot get node data");
		}
	}
	if (h->nline+2 >= (h->line_size)) {
		h->line_size += h->line_incr;
		h->line_incr += 1.5;
		h->line_conn = (xp_long *)ARRrealloc(h->line_conn, DTYPE_LONG, 2*h->line_size, NULL);
	}
	h->line_conn[2*h->nline] = h->nnodes;
	h->nnodes += nnew;
	h->line_conn[2*h->nline+1] = h->nnodes-1;
	(h->nline)++;
	return 1;
}

/* 64-bit porting. Directly Modified */
static int get_line_y (h, dims, ind, nspace, data, veclen, dtype)
    OUTmesh *h;
    xp_long *dims, ind;
    int nspace, data, veclen, dtype;
{
	xp_long i, nnew;

	nnew = dims[1];
	if (h->nnodes+nnew >= h->size) {
		i = nnew/h->size_incr+1;
		h->size += i*h->size_incr;
		h->size_incr *= 1.5;
		h->xyz = (float *)ARRrealloc(h->xyz, DTYPE_FLOAT, nspace*h->size, NULL);
		if (data)
			h->data = (char *)ARRrealloc(h->data, dtype, 
						     h->size*veclen, NULL);
	}
	for (i=0; i<nnew; i++) {
		memcpy(h->xyz+(h->nnodes+i)*nspace,
		       h->xyz+(h->nnodes-ind*dims[1]-dims[0]*(dims[1]-i)+ind)*nspace,
		       nspace*sizeof(float));
	}
	if (data) {
		for (i=0; i<nnew; i++) {
			memcpy(h->data+(h->nnodes+i)*DTYPEtype_size[dtype]*veclen,
			       h->data+(h->nnodes-ind*dims[1]-dims[0]*(dims[1]-i)+ind)*DTYPEtype_size[dtype]*veclen,
			       DTYPEtype_size[dtype]*veclen);
		}
	}
	if (h->nline+2 >= (h->line_size)) {
		h->line_size += h->line_incr;
		h->line_incr *= 1.5;
		h->line_conn = (xp_long *)ARRrealloc(h->line_conn, DTYPE_LONG, 2*h->line_size, NULL);
	}
	h->line_conn[2*h->nline] = h->nnodes;
	h->nnodes += nnew;
	h->line_conn[2*h->nline+1] = h->nnodes-1;
	(h->nline)++;
	return 1;
}
/* 64-bit porting. Directly Modified */
static int get_face (in, h, ndim, dims, dir1, dir2, ind, norm, nspace, data, comp, veclen, dtype)
    OMobj_id in;
    OUTmesh *h;
    xp_long *dims, ind;
    int ndim, nspace, dir1, dir2, data, norm, comp, veclen, dtype;
{
	int i;
	xp_long  i_w, j, n, n1, n2, nnew, rdims[4], rmin[4], rmax[4];

	n1 = dims[dir1];
	n2 = dims[dir2];
	nnew = n1*n2;
	if (h->nnodes+nnew >= h->size) {
		n = nnew/h->size_incr+1;
		h->size += n*h->size_incr;
		h->size_incr *= 1.5;
		h->xyz = (float *)ARRrealloc(h->xyz, DTYPE_FLOAT, nspace*h->size, NULL);
		if (data)
			h->data = (char *)ARRrealloc(h->data, dtype, 
						     h->size*veclen, NULL);
	}
	rdims[0] = nspace;
	rmin[0]=0;
	rmax[0]=nspace;
	rmin[dir1+1] = 0;
	rmin[dir2+1] = 0;
	rmax[dir1+1] = dims[dir1];
	rmax[dir2+1] = dims[dir2];
	for (i=0; i<ndim; i++) {
		rdims[i+1] = dims[i];
		if (i != dir1 && i != dir2) {
			rmin[i+1] = ind;
			rmax[i+1] = ind+1;
		}
	}
	if (FLDget_sub_coord(in, ndim+1, rdims, rmin, rmax, h->xyz+h->nnodes*nspace) != 1) {
		ERR_RETURN("cannot get coordinates");
	}
	if (data) {
		rdims[0] = veclen;
		rmax[0]= veclen;
		if (FLDget_sub_node_data(in, comp, &dtype, ndim+1, rdims, rmin, rmax,
					 h->data+DTYPEtype_size[dtype]*h->nnodes*veclen) != 1) {
			ERR_RETURN("cannot get node data");
		}
	}
	if (h->nquad+(n1-1)*(n2-1) >= (h->quad_size)) {
		n = (n1-1)*(n2-1)/h->quad_incr+1;
		h->quad_size += n*h->quad_incr;
		h->quad_incr *= 1.5;
		h->quad_conn = (xp_long *)ARRrealloc(h->quad_conn, DTYPE_LONG, 4*h->quad_size, NULL);
	}
	if (norm >0) {
		for (i_w=0; i_w<n2-1; i_w++) {
			n = i_w*n1+h->nnodes;
			for (j=0; j<n1-1; j++) {
				h->quad_conn[4*h->nquad] = n+j;
				h->quad_conn[4*h->nquad+1] = n+j+1;
				h->quad_conn[4*h->nquad+2] = n+j+1+n1;
				h->quad_conn[4*h->nquad+3] = n+j+n1;
				(h->nquad)++;
			}
		}
	}
	else {
		for (i_w=0; i_w<n2-1; i_w++) {
			n = i_w*n1+h->nnodes;
			for (j=0; j<n1-1; j++) {
				h->quad_conn[4*h->nquad] = n+j;
				h->quad_conn[4*h->nquad+3] = n+j+1;
				h->quad_conn[4*h->nquad+2] = n+j+1+n1;
				h->quad_conn[4*h->nquad+1] = n+j+n1;
				(h->nquad)++;
			}
		}
	}
	h->nnodes += nnew;
	return 1;
}
