/*
			Copyright (c) 1995 by
			Advanced Visual Systems Inc.
			All Rights Reserved

	This software comprises unpublished confidential information of
	Advanced Visual Systems Inc. and may not be used, copied or made
	available to anyone, except in accordance with the license
	under which it is furnished.

	This file is under Perforce control
	$Id: //depot/express/fcs70/modules/cellcntr.c#1 $
*/

#define XP_WIDE_API	/* Use Wide APIs */

#include <avs/util.h>
#include <avs/err.h>
#include <avs/om.h>
#include <avs/fld.h>
#include <avs/arr.h>
#include <avs/data_utils.h>

#define ERR_RETURN(A) ERRerror("DVcell_centres_update", 0, ERR_ORIG, A); return(0);
#define MAX_NAME_LEN 256
#define ALLOC_BLOCK 1024

/* 64-bit porting. Only Modified Internally */
int DVcell_centres_update(OMobj_id mod_id)
{
	OMobj_id in, out, in_set;

	int nsets, set, cell_nnodes, nspace, ndata, null_flag;
	xp_long ncells, nnodes, *node_conn;
	int i, j, n, veclen, out_veclen, dtype, data_id;
	xp_long i_w, size, out_nnodes, count, ncount, npoly, *poly_conn;
	int poly_flag, *poly_nnodes;
	float *in_coord, *out_coord, center[3];
	double null_value;
	char *in_data, *out_data, name[MAX_NAME_LEN], units[MAX_NAME_LEN];


	in = OMfind_subobj(mod_id, OMstr_to_name("in"), OM_OBJ_RD);
	out = OMfind_subobj(mod_id, OMstr_to_name("out"), OM_OBJ_RW);

	if (FLDget_nnodes(in, &nnodes) != 1) {
		ERR_RETURN("Error getting nnodes");
	}
	if (FLDget_nspace (in, &nspace) != 1) {
		ERR_RETURN("Error getting nspace");
	}
	if (FLDset_nspace (out, nspace) != 1) {
		ERR_RETURN("Error setting nspace");
	}
	if (FLDget_ncell_sets(in, &nsets) != 1) {
		ERR_RETURN("cannot get nsets");
	}

	if (nsets < 1)
	    return 0;

	for (ndata=0, out_nnodes=0, set=0; set<nsets; set++) {
		if (FLDget_cell_set(in, set, &in_set) != 1) {
			ERR_RETURN("Error getting cell set");
		}
		if (FLDget_poly_flag(in_set, &poly_flag) != 1) {
			poly_flag = 0;
		}
		if (poly_flag == 0) {
			if (FLDget_ncells(in_set, &ncells) != 1) {
				ERR_RETURN("cannot get ncells");
			}
		}
		else {
			if (FLDget_npolys(in_set, &ncells) != 1) {
				ERR_RETURN("cannot get ncells");
			}
		}
		out_nnodes += ncells;
		if (set == 0) {
			if (FLDget_cell_data_ncomp(in_set, &ndata) != 1)
				ndata = 0;
			if (FLDset_node_data_ncomp(out, ndata) != 1) {
				ERR_RETURN("cannot set ncell_data");
			}
			for (i=0; i<ndata; i++) {
				if (FLDget_cell_data_veclen(in_set, i, &veclen) != 1) {
					ERR_RETURN("Error getting veclen");
				}
				if (FLDget_cell_data_units(in_set, i, units, MAX_NAME_LEN) != 1) {
					strcpy(units, "");
				}
				if (FLDget_cell_data_label(in_set, i, name, MAX_NAME_LEN) != 1) {
					strcpy(name, "");
				}
				if (FLDset_node_data_comp (out, i, veclen, name, units) != 1) {
					ERR_RETURN("Error setting node component");
				}
				if (FLDget_cell_data_type(in_set, i, &dtype) != 1) {
					ERR_RETURN("cannot get cell data type");
				}
				if (FLDset_node_data_type(out, i, dtype) != 1) {
					ERR_RETURN("cannot get node data type");
				}
				if (FLDget_cell_data_id(in_set, i, &data_id) == 1)
					FLDset_node_data_id(out, i, data_id);

				if (FLDget_cell_null_data(in_set, i, &null_flag, (char *)&null_value) != 1) {
					ERR_RETURN("cannot get null data");
				}
				if (null_flag) {
					if (FLDset_node_null_data(out, i, (char *)&null_value, dtype) != 1) {
						ERR_RETURN("Error setting node null value");
					}
				}
				else {
					if (FLDset_node_null_flag(out, i, 0) != 1) {
						ERR_RETURN("Error setting node null flag");
					}
				}
			}
		}
	}

	if (out_nnodes < 1)
	    return 0;

	if (FLDset_nnodes(out, out_nnodes) != 1) {
		ERR_RETURN("Error setting nnodes");
	}

	if (FLDget_coord(in, &in_coord, &size, OM_GET_ARRAY_RD) != 1) {
		ERR_RETURN("Error setting coordinate array");
	}

	if (FLDget_coord(out, &out_coord, &size, OM_GET_ARRAY_WR) != 1) {
		ERR_RETURN("Error setting coordinate array");
	}
	for (i=0; i<ndata; i++) {
		if (FLDget_node_data(out, i, &dtype, &out_data,
				     &size, OM_GET_ARRAY_WR) != 1) {
			ERR_RETURN("cannot get cell data");
		}
		UTILinit_array(out_data, out_nnodes*veclen, (double)(0), dtype);
		ARRfree(out_data);
	}
	for (ncount=0, set=0; set<nsets; set++) {
		if (FLDget_cell_set(in, set, &in_set) != 1) {
			ERR_RETURN("Error getting cell set");
		}
		if (FLDget_poly_flag(in_set, &poly_flag) != 1) {
			poly_flag = 0;
		}

		if (poly_flag == 0) {
			if (FLDget_ncells(in_set, &ncells) != 1) {
				ERR_RETURN("cannot get ncells");
			}
			if (FLDget_cell_set_nnodes(in_set,  &cell_nnodes) != 1) {
				ERR_RETURN("cannot get cell nnodes");
			}
			if (FLDget_node_connect(in_set, &node_conn, &size, OM_GET_ARRAY_RD) != 1) {
				ERR_RETURN("cannot get cell connectivity");
			}
			for (i_w=0; i_w<ncells; i_w++) {
				for (j=0; j<nspace; j++)
					center[j] = 0.0;
				for (n=0; n<cell_nnodes; n++)
					for (j=0; j<nspace; j++)
						center[j] += in_coord[node_conn[i_w*cell_nnodes+n]*nspace+j];
				for (j=0; j<nspace; j++)
					out_coord[(ncount+i_w)*nspace+j] = center[j]/cell_nnodes;
			}
			ARRfree(node_conn);
		}
		else {
			if (FLDget_npolys(in_set, &npoly) != 1) {
				ERR_RETURN("cannot get ncells");
			}
			ncells = npoly;
			if (FLDget_poly_nnodes(in_set, &poly_nnodes, &size, OM_GET_ARRAY_RD) != 1) {
				ERR_RETURN("cannot get poly_nnodes");
			}
			if (FLDget_poly_connect(in_set, &poly_conn, &size, OM_GET_ARRAY_RD) != 1) {
				ERR_RETURN("cannot get poly connectivity");
			}
			for (count=0,i_w=0; i_w<npoly; i_w++) {
				for (j=0; j<nspace; j++)
					center[j] = 0.0;
				if (poly_nnodes[i_w] == 2) {  /* polyline or polytri */
					cell_nnodes = (int)(poly_conn[2*i_w+1]-poly_conn[2*i_w]+1);
					for (n=0; n<cell_nnodes; n++)
						for (j=0; j<nspace; j++)
							center[j] += in_coord[(poly_conn[i_w*2]+n)*nspace+j];
				}
				else {
					cell_nnodes = poly_nnodes[i_w];
					for (n=0; n<cell_nnodes; n++)
						for (j=0; j<nspace; j++)
							center[j] += in_coord[poly_conn[count+n]*nspace+j];
					count += cell_nnodes;
				}
				for (j=0; j<nspace; j++)
					out_coord[(ncount+i_w)*nspace+j] = center[j]/cell_nnodes;
			}
			ARRfree(poly_conn);
			ARRfree(poly_nnodes);
		}

		for (i=0; i<ndata; i++) {
			if (FLDget_cell_data_veclen(in_set, i, &veclen) != 1) {
				continue;
			}
			if (FLDget_node_data_veclen(out, i, &out_veclen) != 1) {
				ERR_RETURN("Error getting node data veclen");
			}
			if (veclen != out_veclen)
				continue;

			if (FLDget_cell_data(in_set, i, &dtype, &in_data, &size, OM_GET_ARRAY_RD) != 1) {
				ERR_RETURN("cannot get in cell data");
			}
			if (FLDget_node_data(out, i, &dtype, &out_data, &size, OM_GET_ARRAY_WR) != 1) {
				ERR_RETURN("cannot get out node data");
			}
			memcpy(out_data+ncount*DTYPEtype_size[dtype]*veclen, in_data,
			       ncells*veclen*DTYPEtype_size[dtype]);
			ARRfree(in_data);
			ARRfree(out_data);
		}
		ncount += ncells;
	}
	ARRfree(in_coord);
	ARRfree(out_coord);
	return(1);
}
