/*
			Copyright (c) 2003 by
			Advanced Visual Systems Inc.
			All Rights Reserved

	This software comprises unpublished confidential information of
	Advanced Visual Systems Inc. and may not be used, copied or made
	available to anyone, except in accordance with the license
	under which it is furnished.

	This file is under Perforce control
	$Id: //depot/express/fcs70/modules/clampcel.c#1 $
*/

#define XP_WIDE_API	/* Use Wide APIs */

#include <avs/util.h>
#include <avs/err.h>
#include <avs/om.h>
#include <avs/dtype.h>
#include <avs/fld.h>
#include <avs/arr.h>
#include <avs/dv_util.h>
#include <avs/data_utils.h>

#define ERR_RETURN(A) ERRerror("clamp_cell", 0, ERR_ORIG, A); return(0);
#define MAX_NAME_SIZE 1024

int FUNCclamp_cell (OMobj_id in, int clamp_comp,
                    int below, double min_val, int above, double max_val,
                    int reset, OMobj_id out);

/* 64-bit porting. Only Modified Internally */
int DVclamp_cell_update(OMobj_id elem_id)
{
	OMobj_id in, e_id;
	OMobj_id out_sets, out, out_set, out_data, sets, e;
	int  comp;
	int  stat, below, above, reset;
	xp_long i, size;
	double min_val, max_val;

	/* Inputs */

	in = OMfind_subobj(elem_id, OMstr_to_name("in"), OM_OBJ_RD);
	e_id = OMfind_subobj(elem_id, OMstr_to_name("component"),
				 OM_OBJ_RD);
	OMget_int_val(e_id, &comp);
	e_id = OMfind_subobj(elem_id, OMstr_to_name("below"),
				 OM_OBJ_RD);
	OMget_int_val(e_id, &below);
	e_id = OMfind_subobj(elem_id, OMstr_to_name("above"),
				 OM_OBJ_RD);
	OMget_int_val(e_id, &above);
	e_id = OMfind_subobj(elem_id, OMstr_to_name("min_value"),
				 OM_OBJ_RD);
	OMget_real_val(e_id, &min_val);
	e_id = OMfind_subobj(elem_id, OMstr_to_name("max_value"),
				 OM_OBJ_RD);
	OMget_real_val(e_id, &max_val);
	e_id = OMfind_subobj(elem_id, OMstr_to_name("reset_minmax"),
				 OM_OBJ_RD);
	OMget_int_val(e_id, &reset);

	/* Outputs */

	out_data = OMfind_subobj(elem_id, OMstr_to_name("out"), OM_OBJ_RW);

	out_sets = OMfind_subobj(elem_id, OMstr_to_name("out_sets"), OM_OBJ_RD);
	out = OMfind_subobj(elem_id, OMstr_to_name("out_cd"), OM_OBJ_RW);
	sets = OMfind_subobj(out, OMstr_to_name("cell_set"), OM_OBJ_RW);

	stat = OMset_obj_ref(sets, OMnull_obj, 0);
	if (stat != 1) {
		ERR_RETURN("cannot set out cell set to NULL");
	}
	stat = OMget_array_size(out_sets, &size);
	if (stat != 1)
		return(0);
	for (i=0; i<size; i++) {

		if ((stat = OMget_array_val(out_sets, i, &e, OM_OBJ_RW)) != 1) {
			ERR_RETURN("cannot get array sub-element of out_sets");
		}
		out_set = OMfind_subobj(e, OMstr_to_name("out_set"), OM_OBJ_RD);
		if (OMis_null_obj(out_set)) {
			ERR_RETURN("cannot find out_set");
		}
		if ((stat = OMadd_obj_ref(sets, out_set, 0)) != 1) {
			ERR_RETURN("cannot add ref");
		}
	}


	if (FUNCclamp_cell(in, comp, below, min_val, above, max_val,
		           reset, out_data)) {
		return(1);
	}
	else return(0);
}

/* 64-bit porting. Only Modified Internally */
int FUNCclamp_cell (OMobj_id in, int clamp_comp,
                    int below, double min_val, int above, double max_val,
                    int reset, OMobj_id out)
{
	int   nsets, cs, veclen, data_id;
	xp_long ncells, size;
	int   poly_flag, data_type, null_flag, ncomp;
	char   *in_cell_data, *out_cell_data;
	double null_value;
	char  label[MAX_NAME_SIZE], units[MAX_NAME_SIZE];
	OMobj_id  cell_set, out_cell_set;

	/* Only look at the first node data component */
	const int clamp_vec = 0;

	if (FLDget_ncell_sets(in, &nsets) != 1) {
		return(0);
	}

	for (cs=0; cs<nsets; cs++) {
		if (FLDget_cell_set(in, cs, &cell_set) != 1) {
			ERR_RETURN("cannot get cell set");
		}
		if (FLDget_cell_set(out, cs, &out_cell_set) != 1) {
			ERR_RETURN("cannot get cell set");
		}
		if (FLDget_poly_flag(cell_set, &poly_flag) != 1) {
			poly_flag = 0;
		}
		if (poly_flag == 0) {
			if (FLDget_ncells(cell_set, &ncells) != 1) {
				ERR_RETURN("cannot get cell set");
			}
		}
		else {
			if (FLDget_npolys(cell_set, &ncells) != 1) {
				ERR_RETURN("cannot get ncells");
			}
		}

		if (FLDget_cell_data_ncomp(cell_set, &ncomp) != 1) {
			ERR_RETURN("Error getting ncomp");
		}
		if (ncomp == 0) {
			if (FLDset_cell_data_ncomp (out_cell_set, 0) != 1) {
				ERR_RETURN("Error setting ncell_data");
			}
			continue;
		}
		if (FLDset_cell_data_ncomp (out_cell_set, 0) != 1) {
			ERR_RETURN("Error setting ncell_data");
		}
		if (FLDset_cell_data_ncomp (out_cell_set, 1) != 1) {
			ERR_RETURN("Error setting ncell_data");
		}
		if (FLDget_cell_data_veclen(cell_set, clamp_vec, &veclen) != 1) {
			ERR_RETURN("Error getting veclen");
		}

		if (veclen <= clamp_comp) {
			if (FLDset_cell_data_ncomp (out_cell_set, 0) != 1) {
				ERR_RETURN("Error setting ncell_data");
			}
			ERRerror("clamp_cell", 1, ERR_ORIG,
				 "bad vector component number, clamp is not performed");
			continue;
		}
		if (FLDget_cell_data_units(cell_set, clamp_vec,
					   units, MAX_NAME_SIZE) != 1) {
			strcpy(units, "");
		}
		if (FLDget_cell_data_label(cell_set, clamp_vec,
					   label, MAX_NAME_SIZE) != 1) {
			strcpy(label, "clamped_cell_data");
		}
		if (FLDset_cell_data_comp (out_cell_set, 0, veclen, label, units) != 1) {
			ERR_RETURN("Error setting cell component");
		}
		if (FLDget_cell_data(cell_set, clamp_vec, &data_type, &in_cell_data,
				     &size, OM_GET_ARRAY_RD) != 1) {
			ERR_RETURN("cannot get cell data");
		}
		if (FLDget_cell_null_data(cell_set, clamp_vec, &null_flag, (char *)&null_value) != 1) {
			ERR_RETURN("cannot get null data");
		}
		if (FLDget_cell_data(out_cell_set, 0, &data_type, &out_cell_data,
				     &size, OM_GET_ARRAY_WR) != 1) {
			ERR_RETURN("Error setting cell data");
		}
		if (FLDget_cell_data_id(cell_set, clamp_vec, &data_id) == 1)
			FLDset_cell_data_id(out_cell_set, 0, data_id);

		/* UTILclamp does _not_ copy over unclamped values */
		memcpy(out_cell_data, in_cell_data,
		       (size_t)ncells*veclen*DTYPEtype_size[data_type]);

		UTILclamp(ncells, in_cell_data, veclen, clamp_comp, data_type,
			  null_flag, (char *)&null_value,
			  below, min_val, above, max_val, out_cell_data);

		if (null_flag)
			if (FLDset_cell_null_data(out_cell_set, 0, (char *)&null_value, data_type) != 1) {
				ERR_RETURN("Error setting null value");
			}
		else
			if (FLDset_cell_null_flag(out_cell_set, 0, 0) != 1) {
				ERR_RETURN("Error setting null flag");
			}

		if (reset) {
			if (FLDreset_cell_minmax(out_cell_set, 0) != 1) {
				ERR_RETURN("Error resetting cell minmax");
			}
			if (FLDreset_cell_minmax_vec(out_cell_set, 0) != 1) {
				ERR_RETURN("Error resetting cell minmax vec");
			}
		}
		else {
			if (FLDcopy_cell_minmax(cell_set, out_cell_set, clamp_vec, 0) != 1) {
				ERR_RETURN("Error copying cell minmax");
			}
			if (FLDcopy_cell_minmax_vec(cell_set, out_cell_set, clamp_vec, 0) != 1) {
				ERR_RETURN("Error copying cell minmax vec");
			}
		}

		ARRfree(in_cell_data);
		ARRfree(out_cell_data);
	}
	return(1);
}
