/*
			Copyright (c) 2000 by
			Advanced Visual Systems Inc.
			All Rights Reserved

	This software comprises unpublished confidential information of
	Advanced Visual Systems Inc. and may not be used, copied or made
	available to anyone, except in accordance with the license
	under which it is furnished.

	This file is under Perforce control
	$Id: //depot/express/fcs70/modules/combine_sets.c#1 $
*/

#define XP_WIDE_API	/* Use Wide APIs */

#include <avs/err.h>
#include <avs/util.h>
#include <avs/om.h>
#include <avs/fld.h>
#include <avs/data_utils.h>

#define METHOD_SUCCESS 1
#define METHOD_FAILURE 0

#define ERR_RETURN(A) {ERRerror("combine_sets", 0, ERR_ORIG, A); \
                       return METHOD_FAILURE;}

typedef enum {
    FLD_Error = -1,
    FLD_Point = 0,
    FLD_Line,
    FLD_Line2,
    FLD_Polyline,
    FLD_Tri,
    FLD_Tri2,
    FLD_Polytri,
    FLD_Quad,
    FLD_Quad2,
    FLD_Tet,
    FLD_Tet2,
    FLD_Hex,
    FLD_Hex2,
    FLD_Pyr,
    FLD_Pyr2,
    FLD_Prism,
    FLD_Prism2,
    FLD_Polyhedron,
    FLD_cell_kind_count
} FLD_cell_kind;


struct _FLD_cell_kind_info {
    int cell_ndim;      /* cell dimensionality */
    int cell_nnodes;    /* number of nodes per cell */
    int cell_corner_nnodes;   /* number of corner nodes per cell */
    int cell_order;     /* cells' order */
    int poly_flag;      /* flag used for polylines, polytriangles */
    int cell_type;
    const char *name;
} FLD_cell_kind_info[FLD_cell_kind_count] = {
    { 0,  1, 1, 1, 0,  1, "Point" },
    { 1,  2, 2, 1, 0,  2, "Line" },
    { 1,  3, 2, 2, 0, 12, "Line2" },
    { 1,  2, 2, 1, 1,  3, "Polyline" },
    { 2,  3, 3, 1, 0,  4, "Tri" },
    { 2,  6, 3, 2, 0, 14, "Tri2" },
    { 2,  3, 3, 1, 1, 10, "Polytri" },
    { 2,  4, 4, 1, 0,  5, "Quad" },
    { 2,  8, 4, 2, 0, 15, "Quad2" },
    { 3,  4, 4, 1, 0,  6, "Tet" },
    { 3, 10, 4, 2, 0, 16, "Tet2" },
    { 3,  8, 8, 1, 0,  7, "Hex" },
    { 3, 20, 8, 2, 0, 17, "Hex2" },
    { 3,  5, 5, 1, 0,  9, "Pyr" },
    { 3, 13, 5, 2, 0, 19, "Pyr2" },
    { 3,  6, 6, 1, 0,  8, "Prism" },
    { 3, 15, 6, 2, 0, 18, "Prism2" }
#if 0
    /* other changes are needed before we support Polyhedron */
    { 3, 3, 3, 1, 1, 19, "Polyhedron" }
#endif
};


#define DEBUG 0

FLD_cell_kind FLD_identify_cell_set( OMobj_id cs_id )
{
    int ndim, nnodes;
    int order, poly;

    FLDget_cell_ndim( cs_id, &ndim );
    /* FLDget_cell_nnodes( cs_id, &nnodes ); */
    OMget_name_int_val( cs_id, OMstr_to_name("cell_nnodes"), &nnodes );
    FLDget_cell_order( cs_id, &order );
    FLDget_poly_flag( cs_id, &poly );

    switch( ndim ) {
      case 0:
        return FLD_Point;

      case 1:  /* Line, Line2, Polyline */
        if( nnodes == 3 )     return FLD_Line2;
        else if( poly == 1 )  return FLD_Polyline;
        else                  return FLD_Line;

      case 2:  /* Tri, Tri2, Polytri, Quad, Quad2, Polyhedron */
        if( nnodes == 3 ) {
           if( poly == 1 ) {
             /* Check poly type to distinguish Polytri and Polyhedron */
             int poly_type = 2;
             OMget_name_int_val( cs_id, OMstr_to_name("poly_type"), &poly_type );
             if( poly_type == 2 )      return FLD_Polytri;
             else if( poly_type == 1 ) return FLD_Polyhedron;
             else return FLD_Error;
           }
           else            return FLD_Tri;
        }
        else if( nnodes == 4 ) return FLD_Quad;
        else if( nnodes == 6 ) return FLD_Tri2;
        else if( nnodes == 8 ) return FLD_Quad2;
        return FLD_Error;

      case 3:  /* Tet, Tet2, Hex, Hex2, Pyr, Pyr2, Prism, Prism2 */
        if( nnodes == 4 )       return FLD_Tet;
        else if( nnodes == 5 )  return FLD_Pyr;
        else if( nnodes == 6 )  return FLD_Prism;
        else if( nnodes == 8 )  return FLD_Hex;
        else if( nnodes == 10 ) return FLD_Tet2;
        else if( nnodes == 13 ) return FLD_Pyr2;
        else if( nnodes == 15 ) return FLD_Prism2;
        else if( nnodes == 20 ) return FLD_Hex2;
        break;
    }

    return FLD_Error;
}


/* FLD.Data_Array */
struct _data_array {
    int  type;       /* DTYPE_INT, DTYPE_FLOAT, etc. */
    int  veclen;
    char label[128];
    char units[128];
    int  null_flag;
    double null_value; /* not necessarly a double: big enough to hold any prim */

    /* need to dynamically allocate */
    char *data;
    char *min;
    char *max;
    xp_long data_size;
    int minmax_size;
};


/* FLD.Cell_Set + FLD.Cell_Data */
struct _cell_set_table {
    xp_long ncells;            /* number of cells      */
    xp_long npolys;
    xp_long node_list_length;
    xp_long poly_list_length;

    int ncell_data;        /* number of cell data components */
    struct _data_array cell_data;

    /* need to dynamically allocate */
    xp_long * node_connect_list;  /* node connectivity array */
    xp_long * poly_connect_list;
};

struct _summary_table {

    struct _cell_set_table cs_table[FLD_cell_kind_count];

    int nfields;

    xp_long nnodes;
    int nspace;

    /* int nnode_data_comps; */
    struct _data_array node_data;

    /* need to dynamically allocate */
    float *coordinates;

};


int FUNCcombine_sets( OMobj_id *in_ids, int num_fields, OMobj_id out_id );

/* 64-bit porting. Only Modified Internally */
int
DV_ARRcombine_sets_update(OMobj_id elem, OMevent_mask event_mask, int seq_num)
{
    OMobj_id in_arr_id, in_id, out_id;
    OMobj_id *in_ids = NULL;
    xp_long i, num_fields;
    int stat;

#if DEBUG
    {
      char buffer[1024];
      fprintf( stderr, "%s\n",
               OMret_obj_path( elem, buffer, sizeof(buffer) ) );
    }
#endif

    in_arr_id = OMfind_subobj(elem, OMstr_to_name("in"), OM_OBJ_RD);

    stat = OMget_array_size( in_arr_id, &num_fields );
    if( stat != OM_STAT_SUCCESS ) return METHOD_FAILURE;

    if( num_fields != 0 ) {
        in_ids = malloc( num_fields * sizeof(OMobj_id) );
        if( !in_ids ) return METHOD_FAILURE;
    }

    for( i = 0; i < num_fields; ++i ) {
        stat = OMget_array_val( in_arr_id, i, &in_id, OM_OBJ_RD );
        if( stat != OM_STAT_SUCCESS )
            in_ids[i] = OMnull_obj;
        else
            in_ids[i] = in_id;
    }

    out_id = OMfind_subobj(elem,OMstr_to_name("out"),OM_OBJ_RW);

    stat = FUNCcombine_sets( in_ids, (int)num_fields, out_id );

    if( in_ids ) free( in_ids );

    return stat;
}


/* 64-bit porting. Only Modified Internally */
int
FUNCcombine_sets( OMobj_id *in_ids, int num_fields, OMobj_id out_id )
{
    OMobj_id in_id;
    int stat;
    int i, j;
    xp_long k;

    /* The master table */
    struct _summary_table table;

    /* Initialize summary table */
    memset( &table, 0, sizeof(struct _summary_table) );
    table.nspace = -1;

    /* Initialize output field */

    FLDset_nnodes( out_id, 0 );
    FLDset_nspace( out_id, 0 );
    FLDset_node_data_ncomp( out_id, 0 );
    FLDset_coord( out_id, NULL, 0, OM_SET_ARRAY_FREE );
    FLDset_ncell_sets( out_id, 0 );

    if( num_fields <= 0 ) return METHOD_FAILURE;

    /* Cruise through the cell sets and determine how much */
    /* memory will be needed in the summary tables.        */

    /* Also, extract from misc information that does not   */
    /* go into dynamically allocated data stuctures.       */

#if DEBUG
    fprintf( stderr, "Sizing Data Structures. Number fields:%d\n", num_fields );
#endif

    for( i = 0; i < num_fields; ++i ) {
        xp_long nnodes; 
        int nsets, nspace_tmp;
        int ncomp = 0;

        in_id = in_ids[i];
        if( OMis_null_obj(in_id) ) continue;

        /* nspaces for all the fields must match */
        stat = FLDget_nspace( in_id, &nspace_tmp );
        if( stat != OM_STAT_SUCCESS ) continue;
        if( table.nspace == -1 ) table.nspace = nspace_tmp;
        else
            if( table.nspace != nspace_tmp ) continue;

        stat = FLDget_nnodes( in_id, &nnodes );
        if( stat != OM_STAT_SUCCESS || nnodes == 0 ) continue;
        table.nnodes += nnodes;

        stat = FLDget_node_data_ncomp( in_id, &ncomp );
        if( (stat == OM_STAT_SUCCESS) && (ncomp > 0) ) {
            char *node_data_tmp;
            xp_long node_data_size;
            int node_data_type;

            FLDget_node_data( in_id,
                              0,   /* Only the first component for now */
                              &node_data_type,
                              &node_data_tmp,
                              &node_data_size,
                              OM_GET_ARRAY_RD );
            ARRfree( node_data_tmp ); node_data_tmp = NULL;

            /* Add up the total number of nodes.  We will   */
            /* be allocating storage to hold all the nodes. */
            table.node_data.data_size += node_data_size;

            /* Better be the same type for all the fields */
            /* Todo: add checking code                    */
            table.node_data.type = node_data_type;

            /* We really only need to get this stuff once. */

            /* Better be the same veclen for all the fields */
            FLDget_node_data_veclen( in_id, 0,
                                     &(table.node_data.veclen) );

            FLDget_node_data_label ( in_id, 0,
                                     table.node_data.label,
                                     sizeof(table.node_data.label) );

            FLDget_node_data_units ( in_id, 0,
                                     table.node_data.units,
                                     sizeof(table.node_data.units) );

            FLDget_node_null_data( in_id, 0,
                                   &table.node_data.null_flag,
                                   (char *)&table.node_data.null_value );

            table.node_data.minmax_size++;

#if DEBUG
            fprintf( stdout, "Combine fld:%d, nnodes:%ld, ncomp:%d, nd_size:%ld, nd_type:%d, nd_veclen:%d\n",
                     i, nnodes, ncomp, node_data_size, node_data_type, table.node_data.veclen );
#endif

        }
#if DEBUG
        else
            fprintf( stdout, "Combine fld:%d, nnodes:%ld, ncomp:%d\n",
                     i, nnodes, ncomp );
#endif


        stat = FLDget_ncell_sets( in_id, &nsets );
        if( stat != OM_STAT_SUCCESS ) continue;

        for( j = 0; j < nsets; ++j ) {   /* loop over cell sets in field */
            OMobj_id cell_set_id;
            int poly_flag, ncell_comp;
            FLD_cell_kind cell_kind;

            FLDget_cell_set( in_id, j, &cell_set_id );
            cell_kind = FLD_identify_cell_set( cell_set_id );

            if( cell_kind == FLD_Polyhedron ) continue;

            /* Cell Data */
            stat = FLDget_cell_data_ncomp( cell_set_id, &ncell_comp );
            if( stat == OM_STAT_SUCCESS && ncell_comp != 0 ) {
                struct _data_array *cell_data = &table.cs_table[cell_kind].cell_data;

                char *cell_data_tmp;
                xp_long cell_data_size;
                int cell_data_type;

                FLDget_cell_data( cell_set_id,
                                  0,   /* Only the first component for now */
                                  &cell_data_type,
                                  &cell_data_tmp,
                                  &cell_data_size,
                                  OM_GET_ARRAY_RD );
                ARRfree( cell_data_tmp ); cell_data_tmp = NULL;

#if DEBUG
                fprintf( stderr,  "Cell_Data fld:%d, cell_set:%d, kind:%d, ncomp:%d, size:%ld\n",
                         i, j, cell_kind, ncell_comp, cell_data_size );
#endif

                /* Add up the total number of cells.  We will   */
                /* be allocating storage to hold all the cells. */
                cell_data->data_size += cell_data_size;

                cell_data->type = cell_data_type;

                /* This should be consistent across all fields. */

                FLDget_cell_data_veclen( cell_set_id, 0,
                                         &(cell_data->veclen) );

                FLDget_cell_data_label ( cell_set_id, 0,
                                         cell_data->label,
                                         sizeof(cell_data->label) );

                FLDget_cell_data_units ( cell_set_id, 0,
                                         cell_data->units,
                                         sizeof(cell_data->units) );

                FLDget_cell_null_data( cell_set_id, 0,
                                       &cell_data->null_flag,
                                       (char *)&cell_data->null_value );

                cell_data->minmax_size++;
            }

            poly_flag =  FLD_cell_kind_info[cell_kind].poly_flag;

            if( poly_flag ) {
                xp_long npolys, *poly_conn_array, poly_conn_size;
                FLDget_npolys( in_id, &npolys );
                FLDget_poly_connect( cell_set_id,
                                     &poly_conn_array,
                                     &poly_conn_size,  /* usually == 2 * npolys */
                                     OM_GET_ARRAY_RD );
                ARRfree( poly_conn_array );
                table.cs_table[cell_kind].poly_list_length += poly_conn_size;
            }
            else {
                xp_long ncells;
                FLDget_ncells( cell_set_id, &ncells );
                table.cs_table[cell_kind].node_list_length += (ncells * FLD_cell_kind_info[cell_kind].cell_nnodes);
            }

        }  /* loop over cell sets */
        table.nfields++;
    }  /* loop over fields */


    /*
     * Allocate the memory in our summary tables.
     */

#if DEBUG
    fprintf( stderr, "Allocating Memory nnodes:%ld\n", table.nnodes );
#endif

    /* coordinates are always of type float */
    if( (table.nnodes * table.nspace) != 0  )
        table.coordinates = malloc( table.nnodes * table.nspace * sizeof(float) );
    else
        table.coordinates = NULL;

    /* reset for later use as index counter */
    table.nnodes = 0;

    if( table.node_data.data_size != 0 ) {
        int bytes_per_node = DTYPEtype_size[table.node_data.type];
        table.node_data.data = malloc( table.node_data.data_size * bytes_per_node );
        table.node_data.min  = malloc( num_fields * bytes_per_node );
        table.node_data.max  = malloc( num_fields * bytes_per_node );

        /* reset for later use as index counter */
        table.node_data.minmax_size = 0;
    }

    for( j = 0; j < FLD_cell_kind_count; ++j ) {

        struct _cell_set_table *cs_table = &table.cs_table[j];

        if( cs_table->cell_data.data_size != 0 ) {
            int bytes_per_node = DTYPEtype_size[cs_table->cell_data.type];

#if DEBUG
            fprintf( stderr,  "Cell_Data kind:%d, allocating data size:%d\n",
                     j, cs_table->cell_data.data_size );
            fprintf( stderr,  "Cell_Data kind:%d, allocating minmax size:%d\n",
                     j, cs_table->cell_data.minmax_size );
#endif

            cs_table->cell_data.data =
                malloc( cs_table->cell_data.data_size * bytes_per_node );

            cs_table->cell_data.min  =
                malloc( cs_table->cell_data.minmax_size * bytes_per_node );
            cs_table->cell_data.max  =
                malloc( cs_table->cell_data.minmax_size * bytes_per_node );

            /* reset for later use as index counter */
            cs_table->cell_data.minmax_size = 0;
        }

        if( cs_table->node_list_length != 0 ) {
            cs_table->node_connect_list =
                malloc( cs_table->node_list_length * sizeof(xp_long) );

            /* reset for later use as index counter */
            cs_table->node_list_length = 0;
        }

        if( cs_table->poly_list_length != 0 ) {
            cs_table->poly_connect_list =
                malloc( cs_table->poly_list_length * sizeof(xp_long) );

            /* reset for later use as index counter */
            cs_table->poly_list_length = 0;
        }
    }

    /*
     * Extract the information out of the fields
     */

#if DEBUG
    fprintf( stderr, "Extracting Field Information\n" );
#endif

    for( i = 0; i < num_fields; ++i ) {
        int nspace_tmp, nsets;
        xp_long nnodes, coord_size;
        float * coords = NULL;

        char * node_data_tmp;
        double min, max;  /* large enough to hold any prim type */
        xp_long node_data_size;
        int node_data_type;
        int ncomp = 0;

        in_id = in_ids[i];
        if( OMis_null_obj(in_id) ) continue;

        /* nspaces for all the fields must match */
        stat = FLDget_nspace( in_id, &nspace_tmp );
        if( stat != OM_STAT_SUCCESS || table.nspace != nspace_tmp ) continue;

        stat = FLDget_ncell_sets( in_id, &nsets );
        if( stat != OM_STAT_SUCCESS || nsets == 0 ) continue;

        stat = FLDget_nnodes( in_id, &nnodes );
        if( stat != OM_STAT_SUCCESS || nnodes == 0 ) continue;

        /* Grid/coordinates */

        FLDget_coord ( in_id,
                       &coords,
                       &coord_size, /* should equal nnodes * nspace */
                       OM_GET_ARRAY_RD );


        if( coords != NULL && coord_size > 0 ) {
            memcpy( &(table.coordinates[table.nspace*table.nnodes]),
                    coords,
                    nnodes*table.nspace*sizeof(float) );

            ARRfree( coords ); coords = NULL;
        }

        /* Node_Data */

        stat = FLDget_node_data_ncomp( in_id, &ncomp );
        if( (stat == OM_STAT_SUCCESS)  && (ncomp > 0) ) {
            int bytes_per_node = DTYPEtype_size[table.node_data.type];

            FLDget_node_data( in_id,
                              0,   /* Only the first component for now */
                              &node_data_type,
                              &node_data_tmp,
                              &node_data_size,
                              OM_GET_ARRAY_RD );

#if DEBUG
            fprintf( stderr, "Node_Data fld:%d, nnodes:%ld, size: %ld, bpn:%d, veclen:%d\n",
                     i, nnodes, node_data_size, bytes_per_node, table.node_data.veclen );
#endif

            memcpy( &(table.node_data.data[table.nnodes*table.node_data.veclen*bytes_per_node]),
                    node_data_tmp,
                    node_data_size*bytes_per_node );

            ARRfree( node_data_tmp ); node_data_tmp = NULL;

            stat = FLDget_node_data_minmax( in_id, 0, (char *)&min, (char *)&max );
            if( stat == OM_STAT_SUCCESS ) {
#if DEBUG
                fprintf( stderr, "Node_Data fld:%d, node_data: min: %f, max: %f, bpn:%d\n",
                         i, *(float *)&min, *(float *)&max, bytes_per_node );
#endif

                memcpy( &(table.node_data.min[table.node_data.minmax_size*bytes_per_node]),
                        (char *)&min,
                        bytes_per_node );

                memcpy( &(table.node_data.max[table.node_data.minmax_size*bytes_per_node]),
                        (char *)&max,
                        bytes_per_node );

                table.node_data.minmax_size++;
            }

            /* Don't update table.nnodes until the end, the old value */
            /* is needed when renumbering cells. */
        }

        for( j = 0; j < nsets; ++j ) {
            OMobj_id cell_set_id;
            struct _cell_set_table * cs_table;

            xp_long ncells, npolys;
            int ncell_comp;
            FLD_cell_kind cell_kind;
            int poly_flag;
            xp_long *node_conn_array = NULL;
            xp_long node_conn_size = 0;
            xp_long poly_conn_size = 0;

            FLDget_cell_set( in_id, j, &cell_set_id );
            FLDget_ncells( cell_set_id, &ncells );
            cell_kind   = FLD_identify_cell_set( cell_set_id );

            if( cell_kind == FLD_Polyhedron ) continue;

            poly_flag   = FLD_cell_kind_info[cell_kind].poly_flag;
            if( poly_flag ) {
                FLDget_npolys( cell_set_id, &npolys );
            }

            /* Just for convenience */
            cs_table = &(table.cs_table[cell_kind]);

            /* Cell Data */

            stat = FLDget_cell_data_ncomp( cell_set_id, &ncell_comp );
            if( stat == OM_STAT_SUCCESS && ncell_comp != 0 ) {
                struct _data_array *cell_data = &cs_table->cell_data;
                char * cell_data_tmp;
                xp_long cell_data_size;
                int cell_data_type;

                xp_long ncells = poly_flag ? cs_table->npolys : cs_table->ncells;

                int bytes_per_node = DTYPEtype_size[cell_data->type];

                FLDget_cell_data( cell_set_id,
                                  0,   /* Only the first component for now */
                                  &cell_data_type,
                                  &cell_data_tmp,
                                  &cell_data_size,
                                  OM_GET_ARRAY_RD );

#if DEBUG
                fprintf( stderr, "Cell_Data fld:%d, cell_set:%d, ncells:%ld, size:%ld, bpn:%d, veclen:%d\n",
                         i, j, ncells, cell_data_size, bytes_per_node, cell_data->veclen );
#endif

                memcpy( &(cell_data->data[ncells*bytes_per_node*cell_data->veclen]),
                        cell_data_tmp,
                        cell_data_size*bytes_per_node );

                ARRfree( cell_data_tmp ); cell_data_tmp = NULL;

                stat = FLDget_cell_data_minmax( cell_set_id, 0, (char *)&min, (char *)&max );

#if DEBUG
                fprintf( stderr, "Cell_Data fld:%d, min: %f, max: %f, sz:%d\n",
                         i, *(float *)&min, *(float *)&max, bytes_per_node );
#endif

                if( stat == OM_STAT_SUCCESS ) {
                    memcpy( &(cell_data->min[cell_data->minmax_size*bytes_per_node]),
                            (char *)&min,
                            bytes_per_node );

                    memcpy( &(cell_data->max[cell_data->minmax_size*bytes_per_node]),
                            (char *)&max,
                            bytes_per_node );

                    cell_data->minmax_size++;
                }
            }

            /* Cells */

            if( poly_flag )
                FLDget_poly_connect( cell_set_id,
                                     &node_conn_array,
                                     &poly_conn_size,  /* == cell_nnodes * ncells */
                                     OM_GET_ARRAY_RD );
            else
                FLDget_node_connect( cell_set_id,
                                     &node_conn_array,
                                     &node_conn_size,  /* == 2 * npolys */
                                     OM_GET_ARRAY_RD );


#if DEBUG
            {
              int cell_type;
              char cell_name[128];
              FLDget_cell_type( cell_set_id, &cell_type );
              FLDget_cell_set_name( cell_set_id, cell_name, sizeof(cell_name) );
              fprintf( stderr, "Cell_Set fld:%d, cs:%d, name:%s, ncells:%d, p:%d, conn_sz:%d\n",
                       i, j, cell_name, ncells, poly_flag,
                       poly_flag?poly_conn_size:node_conn_size );
            }
#endif

            /* Save it away */
            /* Now for fields other than the first one, we need to renumber the nodes */
            /* For the first field, table.nnodes == 0 */

            if( poly_flag ) {
                memcpy( &(cs_table->poly_connect_list[cs_table->poly_list_length]),
                        node_conn_array,
                        poly_conn_size*sizeof(xp_long) );

                if( table.nnodes != 0 )
                    for( k = 0; k < poly_conn_size; ++k ) {
                        cs_table->poly_connect_list[cs_table->poly_list_length+k] += table.nnodes;
                    }

                cs_table->npolys += npolys;
                cs_table->poly_list_length += poly_conn_size;
            }
            else {
                memcpy( &(cs_table->node_connect_list[cs_table->node_list_length]),
                        node_conn_array,
                        node_conn_size*sizeof(xp_long) );

                if( table.nnodes != 0 )
                    for( k = 0; k < node_conn_size; ++k ) {
                        cs_table->node_connect_list[cs_table->node_list_length+k] += table.nnodes;
                    }

                cs_table->ncells += ncells;
                cs_table->node_list_length += node_conn_size;
            }

            ARRfree( node_conn_array ); node_conn_array = NULL;

        } /* loop over cell sets */

        table.nnodes += nnodes;

    } /* loop over fields */


    /*
     * Now write out the collected information into a single field.
     */

#if DEBUG
    fprintf( stderr, "Writing Output Field nnodes:%d\n", table.nnodes );
#endif

    /* Grid */

    FLDset_nnodes( out_id, table.nnodes );
    FLDset_nspace( out_id, table.nspace );

    /* Coordinate data */

    FLDset_coord ( out_id,
                   table.coordinates,
                   table.nnodes * table.nspace,
                   OM_SET_ARRAY_FREE );

    table.coordinates = NULL;  /* its been given to the OM, do not free */

    /* Node Data */

    if( table.node_data.data ) {
        double min, max; /* big enough to hold any prim */

        int null_flag = 0;
        double null_value = 0.0;

        /* Only the first component for now */
        FLDset_node_data_ncomp( out_id, 1 );
        FLDset_node_data_comp( out_id,
                               0,
                               table.node_data.veclen,
                               table.node_data.label,
                               table.node_data.units);

        UTILcalc_min( table.node_data.minmax_size,
                      1, /* veclen == 1 even if the orginal data's veclen != 1 */
                      table.node_data.min,
                      table.node_data.type,
                      null_flag,
                      null_value,
                      &min );

        UTILcalc_max( table.node_data.minmax_size,
                      1,
                      table.node_data.max,
                      table.node_data.type,
                      null_flag,
                      null_value,
                      &max );

#if DEBUG
        fprintf( stderr, "Out: min: %f, max: %f\n",
                 *(float *)&min, *(float *)&max );
#endif

        FLDset_node_data_minmax( out_id, 0,
                                 (char *)&min, (char *)&max,
                                 table.node_data.type );

        free( table.node_data.min ); table.node_data.min = NULL;
        free( table.node_data.max ); table.node_data.max = NULL;

        if( table.node_data.null_flag ) {
            FLDset_node_null_flag( out_id, 0,
                                   table.node_data.null_flag );
            FLDset_node_null_data( out_id, 0,
                                  (char *)&table.node_data.null_value,
                                  table.node_data.type );
        }

        FLDset_node_data( out_id,
                          0,
                          table.node_data.data,
                          table.node_data.type,
                          table.nnodes*table.node_data.veclen,
                          OM_SET_ARRAY_FREE );

        table.node_data.data = NULL;  /* its been given to the OM, do not free */
    }

    /* Cells */

    for( j = 0, k = 0; j < FLD_cell_kind_count; ++j ) {

        if( table.cs_table[j].ncells != 0 || table.cs_table[j].npolys != 0 ) {
            OMobj_id cell_set_id;
            struct _cell_set_table * cs_table = &(table.cs_table[j]);

            FLDadd_cell_set( out_id, FLD_cell_kind_info[j].name );
            FLDget_cell_set( out_id, (int)k, &cell_set_id );
            k++;

            /* Cell_Data */

            if( cs_table->cell_data.data ) {

                struct _data_array *cell_data = &cs_table->cell_data;

                double min, max; /* any prim type */
                int null_flag = 0;
                double null_value = 0.0;

                int poly_flag = FLD_cell_kind_info[j].poly_flag;
                xp_long ncells    = poly_flag ? cs_table->npolys : cs_table->ncells;

                FLDset_cell_data_ncomp( cell_set_id, 1 );
                FLDset_cell_data_comp( cell_set_id,
                                       0,
                                       cell_data->veclen,
                                       cell_data->label,
                                       cell_data->units );

                UTILcalc_min( cell_data->minmax_size,
                              1, /* veclen == 1 even if the orginal data's veclen != 1 */
                              cell_data->min,
                              cell_data->type,
                              null_flag,
                              null_value,
                              &min );

                UTILcalc_max( cell_data->minmax_size,
                              1,
                              cell_data->max,
                              cell_data->type,
                              null_flag,
                              null_value,
                              &max );

                if( cell_data->min ) free( cell_data->min );
                cell_data->min = NULL;
                if( cell_data->max ) free( cell_data->max );
                cell_data->max = NULL;

                FLDset_cell_data_minmax( cell_set_id, 0,
                                         (char *)&min, (char *)&max,
                                         cell_data->type );

#if DEBUG
                fprintf( stderr, "Cell_Data kind:%d, minmax_size:%d, min:%f, max:%f\n",
                         j, cell_data->minmax_size, *(float *)&min, *(float *)&max );
#endif

                if( cell_data->null_flag ) {
                    FLDset_cell_null_flag( out_id, 0,
                                           cell_data->null_flag );
                    FLDset_cell_null_data( out_id, 0,
                                           (char *)&cell_data->null_value,
                                           cell_data->type );
                }

                stat = FLDset_cell_data( cell_set_id,
                                         0,
                                         cell_data->data,
                                         cell_data->type,
                                         ncells*cell_data->veclen,
                                         OM_SET_ARRAY_FREE );

                cell_data->data = NULL; /* its been given to the OM, do not free */

#if DEBUG
                fprintf( stderr, "Cell_Data kind:%d, stat:%d, ncells:%d, size:%d\n",
                         j, stat, ncells, cell_data->data_size );
#endif
            }

            /* Cell_Set */

            if( cs_table->poly_connect_list ) {
                FLDset_npolys( cell_set_id, cs_table->npolys );
                FLDset_poly_connect( cell_set_id,
                                     cs_table->poly_connect_list,
                                     cs_table->poly_list_length,
                                     OM_SET_ARRAY_FREE );
            }

            if( cs_table->node_connect_list ) {
                FLDset_ncells( cell_set_id, cs_table->ncells );
                FLDset_node_connect( cell_set_id,
                                     cs_table->node_connect_list,
                                     cs_table->node_list_length,
                                     OM_SET_ARRAY_FREE );
            }

            cs_table->node_connect_list = NULL; /* its been given to the OM, do not free */
            cs_table->poly_connect_list = NULL; /* its been given to the OM, do not free */
        }
    }

    return METHOD_SUCCESS;
}

