/*
			Copyright (c) 2000 by
			Advanced Visual Systems Inc.
			All Rights Reserved

	This software comprises unpublished confidential information of
	Advanced Visual Systems Inc. and may not be used, copied or made
	available to anyone, except in accordance with the license
	under which it is furnished.

	This file is under Perforce control
	$Id: //depot/express/fcs70/modules/cpy_field.c#1 $
*/

#define XP_WIDE_API	/* Use Wide APIs */

#include <avs/util.h>
#include <avs/err.h>
#include <avs/om.h>
#include <avs/om_att.h>
#include <avs/fld.h>
#include <avs/arr.h>

/*
 This module copies the entire field, including the mesh, the
 cell data, the node data, and the transform.  It should be easy to
 customize by removing unneeded code.  If the parameter mesh_only
 is present and set to non-zero, do not copy the cell data and node data.
*/

static float IXFORM[16] = {1.0, 0.0, 0.0, 0.0,
			   0.0, 1.0, 0.0, 0.0,
			   0.0, 0.0, 1.0, 0.0,
			   0.0, 0.0, 0.0, 1.0};

#define ERR_RETURN(A) ERRerror("copy_field", 0, ERR_ORIG, A); return(0);

typedef enum { FLD_UNIF, FLD_RECT, FLD_STRUCT, FLD_UNSTRUCT } FLD_Type;

static void set_null_field(OMobj_id elem_id);

int FUNCcopy_field( OMobj_id in, OMobj_id parent, int mesh_only, 
                    OMobj_id *out );

/* This module will copy the input field.
*/
int DVcopy_field_update(OMobj_id module_id, OMevent_mask mask, int seq_num)
{
   OMobj_id tmp_id, mesh_only_id, inFld_id, outFld_id;
   int mesh_only;
   int stat;

   /* Get the inputs and outputs to the module. */
   tmp_id = OMfind_subobj(module_id, OMstr_to_name("inFld"), OM_OBJ_RW);
   if (OMget_obj_val(tmp_id, &inFld_id) != OM_STAT_SUCCESS)
      return 0;

   tmp_id = OMfind_subobj(module_id, OMstr_to_name("outFld"), OM_OBJ_RD);
#if 0
   if (OMget_obj_val(tmp_id, &tmp_id) != OM_STAT_SUCCESS)
      return 0;

   /* Clear out the old output field. */
   set_null_field(tmp_id);
#endif

#ifdef CF_DEBUG
   {
      char buffer[128], *str;
      str = OMget_user_class_name(inFld_id, buffer, sizeof(buffer));
      fprintf( stderr, "Class: %s\n", str );
   }
#endif

   /* OK if missing */
   mesh_only_id = OMfind_subobj(module_id, OMstr_to_name("mesh_only"), OM_OBJ_RD);
   if( OMis_null_obj(mesh_only_id) || 
       OMget_int_val(mesh_only_id, &mesh_only) != OM_STAT_SUCCESS )
      mesh_only = 0;

   outFld_id = OMnull_obj;
   stat = FUNCcopy_field( inFld_id, module_id, mesh_only, &outFld_id );
   if( stat != 1 ) {
      ERR_RETURN("Error while copying field" );
   }

   tmp_id = OMfind_subobj(module_id, OMstr_to_name("outFld"), OM_OBJ_RD);
   stat =  OMset_obj_ref_mode(tmp_id, OM_OBJ_REF);
   if (stat != OM_STAT_SUCCESS )
      return(stat);
   stat = OMset_obj_ref(tmp_id, outFld_id, 0);
   if (stat != OM_STAT_SUCCESS ) {
      ERR_RETURN("Could not set outFld reference to Output_Field" );
   }

   return 1;
}


/*
 * If caller provides a outFld (outFld_id_ptr points to a field obj),
 * then parent_id is ignored and can be OMnull_obj.
 *
 * If caller does not provide a outFld (outFld_id_ptr points to OMnull_obj),
 * then outFld will be created as a child of parent_id.  outFld_id_ptr
 * will then point at the newly created field.
 */
/* 64-bit porting. Only Modified Internally */ 
int FUNCcopy_field( OMobj_id inFld_id, OMobj_id parent_id, int mesh_only,
                    OMobj_id * outFld_id_ptr )
{
   OMobj_id outFld_id;
   xp_long nnodes;
   int nspace, ncomps, ndim;
   int ncell_sets, i, j;
   float xform[16];
   FLD_Type fld_type;

   if( outFld_id_ptr == NULL ) return 0;

   /* Create the new output field */
   if( OMis_null_obj( *outFld_id_ptr ) ) {

      /* Caller has not supplied a output field.  Create a output
       * field with a type that matches the input field.
       */

      static OMobj_id msh_unif_tmpl, msh_rect_tmpl, msh_struct_tmpl;
      static OMobj_id data_tmpl, cell_data_tmpl, node_data_tmpl;
      static int init = 0;
      const char *out_fld_name = "Output_Field";

      if( !init ) {
         msh_unif_tmpl =   OMfind_subobj( OMtempl_obj, OMstr_to_name("Mesh_Unif"), OM_OBJ_RD );
         msh_rect_tmpl =   OMfind_subobj( OMtempl_obj, OMstr_to_name("Mesh_Rect"), OM_OBJ_RD );
         msh_struct_tmpl = OMfind_subobj( OMtempl_obj, OMstr_to_name("Mesh_Struct"), OM_OBJ_RD );
         data_tmpl =       OMfind_subobj( OMtempl_obj, OMstr_to_name("Data"), OM_OBJ_RD );
         node_data_tmpl =  OMfind_subobj( OMtempl_obj, OMstr_to_name("Node_Data"), OM_OBJ_RD );
         cell_data_tmpl =  OMfind_subobj( OMtempl_obj, OMstr_to_name("Cell_Data"), OM_OBJ_RD );
         init = 1;
      }

#define match_flags (OM_MATCH_EXACT | OM_MATCH_NO_ERROR)

      if( OMmatch_obj(msh_unif_tmpl, inFld_id, match_flags) == OM_STAT_SUCCESS ) {
         fld_type = FLD_UNIF;
         outFld_id = FLDcreate(parent_id, "Mesh_Unif", out_fld_name );
      }
      else if( OMmatch_obj(msh_rect_tmpl, inFld_id, match_flags) == OM_STAT_SUCCESS ) {
         fld_type = FLD_RECT;
         outFld_id = FLDcreate(parent_id, "Mesh_Rect", out_fld_name );
      }
      else if( OMmatch_obj(msh_struct_tmpl, inFld_id, match_flags) == OM_STAT_SUCCESS )
      {
         fld_type = FLD_STRUCT;
         outFld_id = FLDcreate(parent_id, "Mesh_Struct", out_fld_name );
      }
      else {
         fld_type = FLD_UNSTRUCT;
         outFld_id = FLDcreate(parent_id, "Mesh", out_fld_name );
      }

      if( OMmatch_obj(data_tmpl, inFld_id, match_flags) == OM_STAT_SUCCESS ) {
         OMmerge_obj(data_tmpl, outFld_id, 0);
      }
      else {
        if( OMmatch_obj(node_data_tmpl, inFld_id, match_flags) == OM_STAT_SUCCESS ) {
           OMmerge_obj(node_data_tmpl, outFld_id, 0);
        }
        if( OMmatch_obj(cell_data_tmpl, inFld_id, match_flags) == OM_STAT_SUCCESS ) {
           OMmerge_obj(cell_data_tmpl, outFld_id, 0);
        }
      }

#undef match_flags

      OMset_obj_atts( outFld_id, OM_atts_nosave );

      *outFld_id_ptr = outFld_id;
   }
   else {
      /* Caller has supplied a output field.  Lets hope its the right type! */
      outFld_id = *outFld_id_ptr;
   }

   /* Start getting the information from the input field. */
   if (FLDget_nnodes(inFld_id, &nnodes) != 1)
      return 0;
   if (FLDget_nspace(inFld_id, &nspace) != 1)
      return 0;
   if (FLDget_ncell_sets(inFld_id, &ncell_sets) != 1)
      return 0;
   /* If we have an empty input field, leave a null output. */
   if (nnodes == 0 || ncell_sets == 0) {
      return 0;
   }

   /* Write the information to the output field. */
   if (FLDset_nnodes(outFld_id, nnodes) != 1) {
      set_null_field(outFld_id);
      return 0;
   }
   if (FLDset_nspace(outFld_id, nspace) != 1) {
      set_null_field(outFld_id);
      return 0;
   }

   /* Dimensions */

   if( fld_type != FLD_UNSTRUCT ) {
      xp_long *dims;
      int dims_size;

      if (FLDget_ndim(inFld_id, &ndim) != 1) {
         set_null_field(outFld_id);
         return 0;
      }
      FLDset_ndim(outFld_id, ndim);

      if (FLDget_dims(inFld_id, &dims, &dims_size) != 1) {
         set_null_field(outFld_id);
         return 0;
      }
      if (FLDset_dims(outFld_id, dims) != 1) {
         set_null_field(outFld_id);
         return 0;
      }
      ARRfree(dims);
   }

   /* Coordinates or points */

   if( fld_type == FLD_UNIF || fld_type == FLD_RECT ) {
      /* "point" (axis) information */
      float *in_points, *out_points;
      xp_long in_pts_size, out_pts_size;
      if (FLDget_points(inFld_id, &in_points, &in_pts_size,
                        OM_GET_ARRAY_RD) != 1) {
        return 0;
      }

      FLDget_points(outFld_id, &out_points, &out_pts_size,
                    OM_GET_ARRAY_WR);
      memcpy(out_points, in_points, in_pts_size*sizeof(float));
      ARRfree(in_points);
      ARRfree(out_points);
   }
   else {
      /* Coordinates information */
      xp_long in_coord_size, out_coord_size;
      float *in_coord_array, *out_coord_array;

      if (FLDget_coord(inFld_id, &in_coord_array, &in_coord_size,
                       OM_GET_ARRAY_RD) != 1) {
         return 0;
      }

      if (FLDget_coord(outFld_id, &out_coord_array, &out_coord_size,
                       OM_GET_ARRAY_WR) != 1) {
         ARRfree(in_coord_array);
         set_null_field(outFld_id);
         return 0;
      }

      memcpy(out_coord_array, in_coord_array, in_coord_size* sizeof(float));
      ARRfree(in_coord_array);
      ARRfree(out_coord_array);
   }

   /* Deal with each cell set individually. */
   for (i=0; i<ncell_sets; i++) {
      OMobj_id inCellSet_id, outCellSet_id;
      int cell_type;
      xp_long ncells;
      int cell_ncomps;
      char cellSetName[128], user_name[128];
      xp_long in_conn_size, out_conn_size;
      xp_long *in_conn, *out_conn;

      /* Get the input cell set name. */
      if (FLDget_cell_set(inFld_id, i, &inCellSet_id) != 1) {
         set_null_field(outFld_id);
         return 0;
      }
      if (FLDget_cell_type(inCellSet_id, &cell_type) != 1) {
         set_null_field(outFld_id);
         return 0;
      }
      if (FLDget_cell_set_name(inCellSet_id,
                               cellSetName, sizeof(cellSetName)) != 1) {
         set_null_field(outFld_id);
         return 0;
      }

      /* Add the cell set to the output mesh and get its id. */
      if( fld_type == FLD_UNSTRUCT ) {
         /* The problem here is how to handle a struct/rect/unif field */
         /* with extra cell sets over and above the default ones. */
         if (FLDadd_cell_set(outFld_id, cellSetName) != 1) {
            set_null_field(outFld_id);
            return 0;
         }
      }
      if (FLDget_cell_set(outFld_id, i, &outCellSet_id) != 1) {
         set_null_field(outFld_id);
         return 0;
      }
      /* If we have an input user name, write it to the output. */
      if (FLDget_cell_set_user_name(inCellSet_id,
                                    user_name, sizeof(user_name)) == 1) {
	  FLDset_cell_set_user_name(outCellSet_id, user_name);
      }

      /* We need to deal with certain cell sets differently -
         polyline, polytri and polyhedron.
      */
      if (cell_type == 3 || cell_type == 10 || cell_type == 19) {
         /* Get the number of cells in the cell set and
            make that number in the output mesh.
         */
         if (FLDget_npolys(inCellSet_id, &ncells) != 1) {
            set_null_field(outFld_id);
            return 0;
         }
         if (FLDset_npolys(outCellSet_id, ncells) != 1) {
            set_null_field(outFld_id);
            return 0;
         }

         /* Get the input cell set connectivity and copy it to
            the output cell set.
         */
         if (FLDget_poly_connect(inCellSet_id, &in_conn, &in_conn_size, 
		   OM_GET_ARRAY_RD) != 1) {
            set_null_field(outFld_id);
            return 0;
         }
         if (FLDget_poly_connect(outCellSet_id, &out_conn, &out_conn_size, 
		   OM_GET_ARRAY_WR) != 1) {
            ARRfree(in_conn);
            set_null_field(outFld_id);
            return 0;
         }
         memcpy(out_conn, in_conn, in_conn_size * sizeof(xp_long));
         ARRfree(in_conn);
         in_conn = NULL;
         ARRfree(out_conn);
         out_conn = NULL;

         /* If we have a polyhedron cell set, we also need to deal
            with the nnodes array.
         */
         if (cell_type == 19) {
            int *in_nnodes = NULL, *out_nnodes = NULL;
            xp_long in_nnode_size, out_nnode_size;

            if (FLDget_poly_nnodes(inCellSet_id, &in_nnodes, &in_nnode_size, 
		   OM_GET_ARRAY_RD) != 1) {
               set_null_field(outFld_id);
               return 0;
            }
            if (FLDget_poly_nnodes(outCellSet_id, &out_nnodes, &out_nnode_size,
		   OM_GET_ARRAY_WR) != 1) {
               ARRfree(in_nnodes);
               set_null_field(outFld_id);
               return 0;
            }
            memcpy(out_nnodes, in_nnodes, in_nnode_size * sizeof(int));
            ARRfree(in_nnodes);
            ARRfree(out_nnodes);
         }
      }
      else {
         /* Get the number of cells in the cell set and
            make that number in the output mesh.
         */
         if (FLDget_ncells(inCellSet_id, &ncells) != 1) {
            set_null_field(outFld_id);
            return 0;
         }
         if (FLDset_ncells(outCellSet_id, ncells) != 1) {
            set_null_field(outFld_id);
            return 0;
         }

         /* In a structured cell set, the connectivity info is implicit */
         /* We might catch a few odd situations by directly checking    */
         /* the cell set type for a match with Cell_Set_Struct          */
         if( fld_type == FLD_UNSTRUCT ) {
            /* Get the input cell set connectivity and copy it to
               the output cell set.
            */
            if (FLDget_node_connect(inCellSet_id, &in_conn, &in_conn_size, 
                                    OM_GET_ARRAY_RD) != 1) {
               set_null_field(outFld_id);
               return 0;
            }
            if (FLDget_node_connect(outCellSet_id, &out_conn, &out_conn_size, 
                                    OM_GET_ARRAY_WR) != 1) {
               ARRfree(in_conn);
               set_null_field(outFld_id);
               return 0;
            }
            memcpy(out_conn, in_conn, in_conn_size * sizeof(xp_long));
            ARRfree(in_conn);
            in_conn = NULL;
            ARRfree(out_conn);
            out_conn = NULL;
         }
      }

      /* Cell Properties */
      {
         int nprops, in_props_size, out_props_size;
         float *in_props, *out_props;

         if (FLDget_cell_nprops(inCellSet_id, &nprops) == 1)
            FLDset_cell_nprops(outCellSet_id, nprops);

         /* OK if missing ... do not abort */
         if (FLDget_cell_props(inCellSet_id, &in_props, &in_props_size, 
                               OM_GET_ARRAY_RD) == 1) {
            if (FLDget_cell_props(outCellSet_id, &out_props, &out_props_size, 
                                  OM_GET_ARRAY_WR) == 1) {
              memcpy(out_props, in_props, in_props_size * sizeof(float));
              ARRfree(out_props);
            }
            ARRfree(in_props);
         }
      }

      /*
        Deal with the cell data if there is any.
      */
      if (!mesh_only &&
          FLDget_cell_data_ncomp(inCellSet_id, &cell_ncomps) == 1) {
         if (FLDset_cell_data_ncomp(outCellSet_id, cell_ncomps) == 1) {
            xp_long in_cell_data_size, out_cell_data_size;
            char *in_cell_data = NULL, *out_cell_data = NULL;
            int veclen, cell_id, cell_data_type, cell_null_flag;
            double cell_min, cell_max, cell_null_value; /* large enough to hold any prim */
            char buffer[128];

            for (j=0; j<cell_ncomps; j++) {
	       /* cell data veclen */
	       if (FLDget_cell_data_veclen(inCellSet_id, j, &veclen) == 1)
	          FLDset_cell_data_veclen(outCellSet_id, j, veclen);

	       /* cell data id */
	       if (FLDget_cell_data_id(inCellSet_id, j, &cell_id) == 1)
	          FLDset_cell_data_id(outCellSet_id, j, cell_id);

	       /* actual cell data */
               if (FLDget_cell_data(inCellSet_id, j, &cell_data_type,
                        (char **)&in_cell_data, 
			&in_cell_data_size, OM_GET_ARRAY_RD) == 1) {
                  if (FLDget_cell_data(outCellSet_id, j, &cell_data_type,
                        (char **)&out_cell_data, 
			&out_cell_data_size, OM_GET_ARRAY_WR) == 1) {

                     size_t size = in_cell_data_size;
                     size *= DTYPEtype_size[cell_data_type];
                     memcpy(out_cell_data, in_cell_data, size);
                     ARRfree(in_cell_data);
                     in_cell_data = NULL;
                     ARRfree(out_cell_data);
                     out_cell_data = NULL;
                  }
                  else {
                     ARRfree(in_cell_data);
                     in_cell_data = NULL;
                  }
               }

	       /* cell data null value */
	       if (FLDget_cell_null_data(inCellSet_id, j, &cell_null_flag, (char *)&cell_null_value) == 1) {
	          FLDset_cell_null_flag(outCellSet_id, j, cell_null_flag);
                  if (cell_null_value) {
                     FLDset_cell_null_data(outCellSet_id, j, (char *)&cell_null_value, cell_data_type);
                  }
	       }

	       /* cell data min/max */
	       if (FLDget_cell_data_minmax(inCellSet_id, j, (char *)&cell_min, (char *)&cell_max ) == 1)
	          FLDset_cell_data_minmax(outCellSet_id, j, (char *)&cell_min, (char *)&cell_max, cell_data_type );

               /* cell data label */
               if (FLDget_cell_data_label(inCellSet_id, j, buffer, sizeof(buffer)) == 1)
                  FLDset_cell_data_label(outCellSet_id, j, buffer);

               /* cell data units */
               if (FLDget_cell_data_units(inCellSet_id, j, buffer, sizeof(buffer)) == 1)
                  FLDget_cell_data_units(outCellSet_id, j, buffer, sizeof(buffer));

            } /* loop over components */
         }
      }
   } /* loop over cell sets */

   /* 
      Deal with the node data if there is any.
   */
   if (!mesh_only &&
       FLDget_node_data_ncomp(inFld_id, &ncomps) == 1) {
      if (FLDset_node_data_ncomp(outFld_id, ncomps) == 1) {
         xp_long in_node_data_size, out_node_data_size;
         char *in_node_data = NULL, *out_node_data = NULL;
         int veclen, node_id, node_data_type, node_null_flag;
         double node_min, node_max, node_null_value; /* large enough to hold any prim */
         char buffer[128];

         for (j=0; j<ncomps; j++) {
            /* node data veclen */
            if (FLDget_node_data_veclen(inFld_id, j, &veclen) == 1)
               FLDset_node_data_veclen(outFld_id, j, veclen);

            /* node data id */
            if (FLDget_node_data_id(inFld_id, j, &node_id) == 1)
               FLDset_node_data_id(outFld_id, j, node_id);

            /* actual node data */
            if (FLDget_node_data(inFld_id, j, &node_data_type,
                                 (char **)&in_node_data, 
                                 &in_node_data_size, OM_GET_ARRAY_RD) == 1) {
               if (FLDget_node_data(outFld_id, j, &node_data_type,
                                    (char **)&out_node_data, 
                                    &out_node_data_size, OM_GET_ARRAY_WR) == 1) {

                  size_t size = in_node_data_size;
                  size *= DTYPEtype_size[node_data_type];
                  memcpy(out_node_data, in_node_data, size);

                  ARRfree(in_node_data);
                  in_node_data = NULL;
                  ARRfree(out_node_data);
                  out_node_data = NULL;
               }
               else {
                  ARRfree(in_node_data);
                  in_node_data = NULL;
               }
            }

            /* node data null value */
            if (FLDget_node_null_data(inFld_id, j, &node_null_flag,
                                      (char *)&node_null_value) == 1) {
               FLDset_node_null_flag(outFld_id, j, node_null_flag);
               if (node_null_flag) {
                  FLDset_node_null_data(outFld_id, j, (char *)&node_null_value,
                                        node_data_type);
               }
            }

            /* node data min/max */
            if (FLDget_node_data_minmax(inFld_id, j, (char *)&node_min, (char *)&node_max ) == 1)
               FLDset_node_data_minmax(outFld_id, j, (char *)&node_min, (char *)&node_max, node_data_type);

            /* node data labels */
            if (FLDget_node_data_label(inFld_id, j, buffer, sizeof(buffer)) == 1)
               FLDset_node_data_label(outFld_id, j, buffer);

            /* node data units */
            if (FLDget_node_data_units(inFld_id, j, buffer, sizeof(buffer)) == 1)
               FLDset_node_data_units(outFld_id, j, buffer);

         } /* loop over components */
      }
   } /* if node data */

   /* 
      transform
   */
   if (FLDget_xform(inFld_id, xform) == 1)
      FLDset_xform(outFld_id, xform);

   return 1;
}

/* 64-bit porting. Only Modified Internally */
static void set_null_field(OMobj_id elem_id)
{
   xp_long dims[3];

   dims[0] = 0; dims[1] = 0; dims[2] = 0;
   FLDset_nnodes(elem_id, 0);                 /* no coords */
   FLDset_nspace(elem_id, 0);
   FLDset_dims(elem_id, dims);
   FLDset_coord(elem_id, NULL, 0, OM_SET_ARRAY_FREE);
   FLDset_node_data_ncomp(elem_id, 0);        /* no node data */
   FLDset_ncell_sets(elem_id, 0);             /* no cell sets */
   FLDset_xform(elem_id, IXFORM);             /* default transform */
}
