/*
			Copyright (c) 1993 by
			Advanced Visual Systems Inc.
			All Rights Reserved

	This software comprises unpublished confidential information of
	Advanced Visual Systems Inc. and may not be used, copied or made
	available to anyone, except in accordance with the license
	under which it is furnished.

	This file is under Perforce control
	$Id: //depot/express/fcs70/modules/cpy_mesh.c#1 $
*/

#define XP_WIDE_API	/* Use Wide APIs */

#include <avs/util.h>
#include <avs/err.h>
#include <avs/om.h>
#include <avs/fld.h>
#include <avs/arr.h>

/* This used to be called DVcopy_mesh_update */

#define ERR_RETURN(A) ERRerror("copy_cells", 0, ERR_ORIG, A); return(0);

static void set_null_mesh(OMobj_id elem_id);

/* This module will copy the input cell sets. If there is any
   cell data, the cell data will be copied too.
*/
/* 64-bit porting. Only Modified Internally */ 
int DVcopy_cells_update(OMobj_id elem_id)
{
   OMobj_id tmp_id, inMesh_id, outMesh_id;
   OMobj_id inCellSet_id, outCellSet_id;
   xp_long nnodes;
   int nspace;
   xp_long in_coord_size, out_coord_size;
   float *in_coord_array, *out_coord_array;
   xp_long in_cell_data_size, out_cell_data_size;
   char *in_cell_data, *out_cell_data;
   int ncell_sets, cell_type, i, j;
   xp_long ncells;
   xp_long in_conn_size, out_conn_size;
   xp_long *in_conn, *out_conn;
   xp_long in_nnode_size, out_nnode_size;
   int *in_nnodes, *out_nnodes;
   char cellSetName[128], user_name[128];
   int cell_ncomps, cell_veclen, cell_id, cell_data_type;

   /* Get the inputs and outputs to the module. */
   tmp_id = OMfind_subobj(elem_id, OMstr_to_name("inMesh"), OM_OBJ_RW);
   if (OMget_obj_val(tmp_id, &inMesh_id) != OM_STAT_SUCCESS)
      return 0;

   tmp_id = OMfind_subobj(elem_id, OMstr_to_name("outMesh"), OM_OBJ_RD);
   if (OMget_obj_val(tmp_id, &outMesh_id) != OM_STAT_SUCCESS)
      return 0;

   /* Get the information from the input mesh. */
   if (FLDget_nnodes(inMesh_id, &nnodes) != 1) {
      set_null_mesh(outMesh_id);
      return 0;
   }
   if (FLDget_nspace(inMesh_id, &nspace) != 1) {
      set_null_mesh(outMesh_id);
      return 0;
   }
   if (FLDget_ncell_sets(inMesh_id, &ncell_sets) != 1) {
      set_null_mesh(outMesh_id);
      return 0;
   }
   /* If we have an empty input field, null the output. */
   if (nnodes == 0 || ncell_sets == 0) {
      set_null_mesh(outMesh_id);
      return 0;
   }

   if (FLDget_coord(inMesh_id, (float **)&in_coord_array, &in_coord_size,
	OM_GET_ARRAY_RD) != 1) {
      set_null_mesh(outMesh_id);
      return 0;
   }

   /* Write the information from to the output mesh. */
   if (FLDset_nnodes(outMesh_id, nnodes) != 1) {
      ARRfree(in_coord_array);
      set_null_mesh(outMesh_id);
      return 0;
   }
   if (FLDset_nspace(outMesh_id, nspace) != 1) {
      ARRfree(in_coord_array);
      set_null_mesh(outMesh_id);
      return 0;
   }
   if (FLDset_ncell_sets(outMesh_id, ncell_sets) != 1) {
      ARRfree(in_coord_array);
      set_null_mesh(outMesh_id);
      return 0;
   }
   if (FLDget_coord(outMesh_id, (float **)&out_coord_array, &out_coord_size,
	OM_GET_ARRAY_WR) != 1) {
      ARRfree(in_coord_array);
      set_null_mesh(outMesh_id);
      return 0;
   }
   memcpy(out_coord_array, in_coord_array, in_coord_size* sizeof(float));
   ARRfree(in_coord_array);
   ARRfree(out_coord_array);

   /* Clear all output cell sets so when we do the add cell
      sets they aren't appended to the field.
   */
   if (FLDset_ncell_sets(outMesh_id, 0) != 1) {
      set_null_mesh(outMesh_id);
      return 0;
   }

   /* Deal with each sell set individually. */
   for (i=0; i<ncell_sets; i++) {
      /* Get the input cell set name. */
      if (FLDget_cell_set(inMesh_id, i, &inCellSet_id) != 1) {
         set_null_mesh(outMesh_id);
         return 0;
      }
      if (FLDget_cell_type(inCellSet_id, &cell_type) != 1) {
         set_null_mesh(outMesh_id);
         return 0;
      }
      if (FLDget_cell_set_name(inCellSet_id, cellSetName, 128) != 1) {
         set_null_mesh(outMesh_id);
         return 0;
      }

      /* Add the cell set to the output mesh and get its id. */
      if (FLDadd_cell_set(outMesh_id, cellSetName) != 1) {
         set_null_mesh(outMesh_id);
         return 0;
      }
      if (FLDget_cell_set(outMesh_id, i, &outCellSet_id) != 1) {
         set_null_mesh(outMesh_id);
         return 0;
      }
      /* If we have an input user name, write it to the output. */
      if (FLDget_cell_set_user_name(inCellSet_id, user_name, 128) == 1) {
	  FLDset_cell_set_user_name(outCellSet_id, user_name);
      }

      /* We need to deal with certain cell sets differently -
         polyline, polytri and polyhedron.
      */
      if (cell_type == 3 || cell_type == 10 || cell_type == 19) {
         /* Get the number of cells in the cell set and
            make that number in the output mesh.
         */
         if (FLDget_npolys(inCellSet_id, &ncells) != 1) {
            set_null_mesh(outMesh_id);
            return 0;
         }
         if (FLDset_npolys(outCellSet_id, ncells) != 1) {
            set_null_mesh(outMesh_id);
            return 0;
         }

         /* Get the input cell set connectivity and copy it to
            the output cell set.
         */
         if (FLDget_poly_connect(inCellSet_id, &in_conn, &in_conn_size, 
		   OM_GET_ARRAY_RD) != 1) {
            set_null_mesh(outMesh_id);
            return 0;
         }
         if (FLDget_poly_connect(outCellSet_id, &out_conn, &out_conn_size, 
		   OM_GET_ARRAY_WR) != 1) {
            ARRfree(in_conn);
            set_null_mesh(outMesh_id);
            return 0;
         }
         memcpy(out_conn, in_conn, in_conn_size * sizeof(xp_long));
         ARRfree(in_conn);
         ARRfree(out_conn);

         /* If we have a polyhedron cell set, we also need to deal
            with the nnodes array.
         */
         if (cell_type == 19) {
            if (FLDget_poly_nnodes(inCellSet_id, &in_nnodes, &in_nnode_size, 
		   OM_GET_ARRAY_RD) != 1) {
               set_null_mesh(outMesh_id);
               return 0;
            }
            if (FLDget_poly_nnodes(outCellSet_id, &out_nnodes, &out_nnode_size,
		   OM_GET_ARRAY_WR) != 1) {
               ARRfree(in_nnodes);
               set_null_mesh(outMesh_id);
               return 0;
            }
            memcpy(out_nnodes, in_nnodes, in_nnode_size * sizeof(int));
            ARRfree(in_nnodes);
            ARRfree(out_nnodes);
         }
      }
      else {
         /* Get the number of cells in the cell set and
            make that number in the output mesh.
         */
         if (FLDget_ncells(inCellSet_id, &ncells) != 1) {
            set_null_mesh(outMesh_id);
            return 0;
         }
         if (FLDset_ncells(outCellSet_id, ncells) != 1) {
            set_null_mesh(outMesh_id);
            return 0;
         }

         /* Get the input cell set connectivity and copy it to
            the output cell set.
         */
         if (FLDget_node_connect(inCellSet_id, &in_conn, &in_conn_size, 
		   OM_GET_ARRAY_RD) != 1) {
            set_null_mesh(outMesh_id);
            return 0;
         }
         if (FLDget_node_connect(outCellSet_id, &out_conn, &out_conn_size, 
		   OM_GET_ARRAY_WR) != 1) {
            ARRfree(in_conn);
            set_null_mesh(outMesh_id);
            return 0;
         }
         memcpy(out_conn, in_conn, in_conn_size * sizeof(xp_long));
         ARRfree(in_conn);
         ARRfree(out_conn);
      }

      /* Cell Properties */
      {
         int nprops, in_props_size, out_props_size;
         float *in_props, *out_props;

         if (FLDget_cell_nprops(inCellSet_id, &nprops) == 1)
            FLDset_cell_nprops(outCellSet_id, nprops);

         /* OK if missing ... do not abort */
         if (FLDget_cell_props(inCellSet_id, &in_props, &in_props_size, 
                               OM_GET_ARRAY_RD) == 1) {
            if (FLDget_cell_props(outCellSet_id, &out_props, &out_props_size, 
                                  OM_GET_ARRAY_WR) == 1) {
              memcpy(out_props, in_props, in_props_size * sizeof(float));
              ARRfree(out_props);
            }
            ARRfree(in_props);
         }
      }

      /* Deal with the cell data if there is any. */
      if (FLDget_cell_data_ncomp(inCellSet_id, &cell_ncomps) == 1) {
         if (FLDset_cell_data_ncomp(outCellSet_id, cell_ncomps) == 1) {
            int cell_null_flag;
            double cell_min, cell_max, cell_null_value; /* large enough to hold any prim */
            char buffer[256];
            for (j=0; j<cell_ncomps; j++) {
	       /* cell data veclen */
	       if (FLDget_cell_data_veclen(inCellSet_id, j, &cell_veclen) == 1)
	          FLDset_cell_data_veclen(outCellSet_id, j, cell_veclen);
	       /* cell data id */
	       if (FLDget_cell_data_id(inCellSet_id, j, &cell_id) == 1)
	          FLDset_cell_data_id(outCellSet_id, j, cell_id);
	       /* actual cell data */
               if (FLDget_cell_data(inCellSet_id, j, &cell_data_type, (char **)&in_cell_data, 
			&in_cell_data_size, OM_GET_ARRAY_RD) == 1) {
                  if (FLDget_cell_data(outCellSet_id, j, &cell_data_type, (char **)&out_cell_data, 
			&out_cell_data_size, OM_GET_ARRAY_WR) == 1) {
                     size_t size = in_cell_data_size;
                     size *= DTYPEtype_size[cell_data_type];
                     memcpy(out_cell_data, in_cell_data, size);
                     ARRfree(in_cell_data);
                     ARRfree(out_cell_data);
                  }
                  else ARRfree(in_cell_data);
               }
#if 0
	       /* cell data min/max */
	       if (FLDget_cell_data_minmax(inCellSet_id, j, (char *)&cell_min, (char *)&cell_max ) == 1)
	          FLDset_cell_data_minmax(outCellSet_id, j, (char *)&cell_min, (char *)&cell_max, cell_data_type );
#endif
               /* cell data null value */
	       if (FLDget_cell_null_data(inCellSet_id, j, &cell_null_flag, (char *)&cell_null_value) == 1) {
	          FLDset_cell_null_flag(outCellSet_id, j, cell_null_flag);
                  if (cell_null_value) {
                     FLDset_cell_null_data(outCellSet_id, j, (char *)&cell_null_value, cell_data_type);
                  }
	       }
               /* cell data label */
               if (FLDget_cell_data_label(inCellSet_id, j, buffer, sizeof(buffer)) == 1)
                  FLDset_cell_data_label(outCellSet_id, j, buffer);

               /* cell data units */
               if (FLDget_cell_data_units(inCellSet_id, j, buffer, sizeof(buffer)) == 1)
                  FLDget_cell_data_units(outCellSet_id, j, buffer, sizeof(buffer));
            }
         }
      }
   }

   return(1);
}

/* 64-bit porting. Only Modified Internally */
static void set_null_mesh(OMobj_id elem_id)
{
   xp_long dims[3];

   dims[0] = 0; dims[1] = 0; dims[2] = 0;
   FLDset_nnodes(elem_id, 0);                 /* no coords */
   FLDset_dims(elem_id, dims);
   FLDset_node_data_ncomp(elem_id, 0);        /* no node data */
   FLDset_ncell_sets(elem_id, 0);             /* no cell sets */
}
