/*
			Copyright (c) 1994 by
			Advanced Visual Systems Inc.
			All Rights Reserved

	This software comprises unpublished confidential information of
	Advanced Visual Systems Inc. and may not be used, copied or made
	available to anyone, except in accordance with the license
	under which it is furnished.

	This file is under Perforce control
	$Id: //depot/express/fcs70/modules/crop.c#1 $
*/

#define XP_WIDE_API	/* Use Wide APIs */

#include <avs/util.h>
#include <avs/err.h>
#include <avs/om.h>
#include <avs/fld.h>
#include <avs/dv_util.h>

#define METHOD_SUCCESS 1
#define METHOD_FAILURE 0

#define ERR_RETURN(A) {ERRerror("crop", 0, ERR_ORIG, A); \
                       return METHOD_FAILURE;}

#define MAX_NAME_SIZE 1024

int FUNCcrop (OMobj_id in, int *min, int *max, OMobj_id out);
int FUNCcrop_unif (OMobj_id in, int *min, int *max, OMobj_id out);
int FUNCcrop_rect (OMobj_id in, int *min, int *max, OMobj_id out);

/* 64-bit porting. Only Modified Internally */
int DVcrop_update(OMobj_id elem)
{
    OMobj_id src_id, dst_id, min_id, max_id;
    int *min, *max, type, stat;
    xp_long size;

    src_id = OMfind_subobj(elem,OMstr_to_name("in"),OM_OBJ_RD);
    min_id = OMfind_subobj(elem,OMstr_to_name("min"),OM_OBJ_RD);
    max_id = OMfind_subobj(elem,OMstr_to_name("max"),OM_OBJ_RD);
    dst_id = OMfind_subobj(elem,OMstr_to_name("out"),OM_OBJ_RW);

    type = OM_TYPE_INT;
    size = 0;
    min = (int *)NULL;
    stat = OMget_array_sz(min_id, &type, (char **)(&min),
                          &size, OM_GET_ARRAY_RD);
    if (stat != OM_STAT_SUCCESS)
        return METHOD_FAILURE;
    size = 0;
    max = (int *)NULL;
    stat = OMget_array_sz(max_id, &type, (char **)(&max),
                          &size, OM_GET_ARRAY_RD);
    if (stat != OM_STAT_SUCCESS)
        return METHOD_FAILURE;

    stat = FUNCcrop(src_id, min, max, dst_id);

    ARRfree(min);
    ARRfree(max);
    return stat;
}


/* 64-bit porting. Only Modified Internally */
int DVcrop_unif_update(OMobj_id elem)
{
    OMobj_id src_id, dst_id, min_id, max_id;
    int *min, *max, type, stat;
    xp_long  size;

    src_id = OMfind_subobj(elem,OMstr_to_name("in"),OM_OBJ_RD);
    min_id = OMfind_subobj(elem,OMstr_to_name("min"),OM_OBJ_RD);
    max_id = OMfind_subobj(elem,OMstr_to_name("max"),OM_OBJ_RD);
    dst_id = OMfind_subobj(elem,OMstr_to_name("out"),OM_OBJ_RW);

    type = OM_TYPE_INT;
    size = 0;
    min = (int *)NULL;
    stat = OMget_array_sz(min_id, &type, (char **)(&min),
                          &size, OM_GET_ARRAY_RD);
    if (stat != OM_STAT_SUCCESS)
        return METHOD_FAILURE;
    size = 0;
    max = (int *)NULL;
    stat = OMget_array_sz(max_id, &type, (char **)(&max),
                          &size, OM_GET_ARRAY_RD);
    if (stat != OM_STAT_SUCCESS)
        return METHOD_FAILURE;

    stat = FUNCcrop_unif(src_id, min, max, dst_id);

    ARRfree(min);
    ARRfree(max);
    return stat;
}


/* 64-bit porting. Only Modified Internally */
int DVcrop_rect_update(OMobj_id elem)
{
    OMobj_id src_id, dst_id, min_id, max_id;
    int *min, *max, type, stat;
    xp_long size;

    src_id = OMfind_subobj(elem,OMstr_to_name("in"),OM_OBJ_RD);
    min_id = OMfind_subobj(elem,OMstr_to_name("min"),OM_OBJ_RD);
    max_id = OMfind_subobj(elem,OMstr_to_name("max"),OM_OBJ_RD);
    dst_id = OMfind_subobj(elem,OMstr_to_name("out"),OM_OBJ_RW);

    type = OM_TYPE_INT;
    size = 0;
    min = (int *)NULL;
    stat = OMget_array_sz(min_id, &type, (char **)(&min),
                          &size, OM_GET_ARRAY_RD);
    if (stat != OM_STAT_SUCCESS)
        return METHOD_FAILURE;
    size = 0;
    max = (int *)NULL;
    stat = OMget_array_sz(max_id, &type, (char **)(&max),
                          &size, OM_GET_ARRAY_RD);
    if (stat != OM_STAT_SUCCESS)
        return METHOD_FAILURE;

    stat = FUNCcrop_rect(src_id, min, max, dst_id);

    ARRfree(min);
    ARRfree(max);
    return stat;
}


/* 64-bit porting. Only Modified Internally */
int
DV_ARRcrop_update(OMobj_id elem, OMevent_mask event_mask, int seq_num)
{
    OMobj_id src_arr_id, dst_arr_id;
    xp_long i, num_fields, size;

    OMobj_id src_id, dst_id, min_id, max_id;
    OMobj_id src_xform_id, dst_xform_id;
    OMobj_name xform_name = OMstr_to_name("xform");
    int *min, *max, type, stat;
    int *min_save, *max_save;

    int ndims;
    int multi_params_min = 0;
    int multi_params_max = 0;

    src_arr_id = OMfind_subobj(elem,OMstr_to_name("in"),OM_OBJ_RD);
    min_id = OMfind_subobj(elem,OMstr_to_name("min"),OM_OBJ_RD);
    max_id = OMfind_subobj(elem,OMstr_to_name("max"),OM_OBJ_RD);
    dst_arr_id = OMfind_subobj(elem,OMstr_to_name("out"),OM_OBJ_RW);

    stat = OMget_array_size( src_arr_id, &num_fields );
    if( stat != OM_STAT_SUCCESS ) return METHOD_FAILURE;

    stat = OMset_array_size( dst_arr_id, num_fields );
    if( stat != OM_STAT_SUCCESS ) return METHOD_FAILURE;

    OMget_array_val( src_arr_id, 0, &src_id, OM_OBJ_RD );
    FLDget_ndim( src_id, &ndims );

    type = OM_TYPE_INT;
    size = 0;
    min = (int *)NULL;
    stat = OMget_array_sz(min_id, &type, (char **)(&min),
                          &size, OM_GET_ARRAY_RD);
    if (stat != OM_STAT_SUCCESS)
        return METHOD_FAILURE;

    /* The min params is a 2D array that independently */
    /* specifies the cropping on each block.           */
    if( (ndims * num_fields) == size )
        multi_params_min = 1;

    size = 0;
    max = (int *)NULL;
    stat = OMget_array_sz(max_id, &type, (char **)(&max),
                          &size, OM_GET_ARRAY_RD);
    if (stat != OM_STAT_SUCCESS)
        return METHOD_FAILURE;

    /* The max params is a 2D array that independently */
    /* specifies the cropping on each block.           */
    if( (ndims * num_fields) == size )
        multi_params_max = 1;

    min_save = min;
    max_save = max;

    for( i = 0; i < num_fields; ++i ) {
        stat = OMget_array_val( src_arr_id, i, &src_id, OM_OBJ_RD );
        if( stat != OM_STAT_SUCCESS ) continue;
        stat = OMget_array_val( dst_arr_id, i, &dst_id, OM_OBJ_RW );
        if( stat != OM_STAT_SUCCESS ) continue;

        stat = FUNCcrop(src_id, min, max, dst_id);

        if( multi_params_min ) {
            min += ndims;
        }
        if( multi_params_max ) {
            max += ndims;
        }

#if 0
        if( stat != METHOD_SUCCESS ) {
            ERRerror( "crop", 1, ERR_ORIG,
                      "Error while processing field: %d", i );
        }
#endif

        /* This can be done easily in V, but its faster in C. */
        src_xform_id = OMfind_subobj(src_id, xform_name, OM_OBJ_RD);
        dst_xform_id = OMfind_subobj(dst_id, xform_name ,OM_OBJ_RW);
        OMset_obj_ref( dst_xform_id, src_xform_id, 0 );
    }

    ARRfree(min_save);
    ARRfree(max_save);
    return stat;
}


/* 64-bit porting. Only Modified Internally */
int
DV_ARRcrop_unif_update(OMobj_id elem, OMevent_mask event_mask, int seq_num)
{
    OMobj_id src_arr_id, dst_arr_id;
    xp_long i, num_fields, size;

    OMobj_id src_id, dst_id, min_id, max_id;
    OMobj_id src_xform_id, dst_xform_id;
    OMobj_name xform_name = OMstr_to_name("xform");
    int *min, *max, type, stat;

    src_arr_id = OMfind_subobj(elem,OMstr_to_name("in"),OM_OBJ_RD);
    min_id = OMfind_subobj(elem,OMstr_to_name("min"),OM_OBJ_RD);
    max_id = OMfind_subobj(elem,OMstr_to_name("max"),OM_OBJ_RD);
    dst_arr_id = OMfind_subobj(elem,OMstr_to_name("out"),OM_OBJ_RW);

    stat = OMget_array_size( src_arr_id, &num_fields );
    if( stat != OM_STAT_SUCCESS ) return OM_STAT_FAILURE;

    stat = OMset_array_size( dst_arr_id, num_fields );
    if( stat != OM_STAT_SUCCESS ) return OM_STAT_FAILURE;

    type = OM_TYPE_INT;
    size = 0;
    min = (int *)NULL;
    stat = OMget_array_sz(min_id, &type, (char **)(&min),
                          &size, OM_GET_ARRAY_RD);
    if (stat != OM_STAT_SUCCESS)
        return METHOD_FAILURE;
    size = 0;
    max = (int *)NULL;
    stat = OMget_array_sz(max_id, &type, (char **)(&max),
                          &size, OM_GET_ARRAY_RD);
    if (stat != OM_STAT_SUCCESS)
        return METHOD_FAILURE;

    for( i = 0; i < num_fields; ++i ) {
        stat = OMget_array_val( src_arr_id, i, &src_id, OM_OBJ_RD );
        if( stat != OM_STAT_SUCCESS ) continue;
        stat = OMget_array_val( dst_arr_id, i, &dst_id, OM_OBJ_RW );
        if( stat != OM_STAT_SUCCESS ) continue;

        stat = FUNCcrop_unif(src_id, min, max, dst_id);
#if 0
        if( stat != METHOD_SUCCESS ) {
            ERRerror( "crop_unif", 1, ERR_ORIG,
                      "Error while processing field: %d", i );
        }
#endif

        /* This can be done easily in V, but its faster in C. */
        src_xform_id = OMfind_subobj(src_id, xform_name, OM_OBJ_RD);
        dst_xform_id = OMfind_subobj(dst_id, xform_name ,OM_OBJ_RW);
        OMset_obj_ref( dst_xform_id, src_xform_id, 0 );
    }

    ARRfree(min);
    ARRfree(max);
    return stat;
}


/* 64-bit porting. Only Modified Internally */
int
DV_ARRcrop_rect_update(OMobj_id elem, OMevent_mask event_mask, int seq_num)
{
    OMobj_id src_arr_id, dst_arr_id;
    xp_long i, num_fields, size;

    OMobj_id src_id, dst_id, min_id, max_id;
    OMobj_id src_xform_id, dst_xform_id;
    OMobj_name xform_name = OMstr_to_name("xform");
    int *min, *max, type, stat;

    src_arr_id = OMfind_subobj(elem,OMstr_to_name("in"),OM_OBJ_RD);
    min_id = OMfind_subobj(elem,OMstr_to_name("min"),OM_OBJ_RD);
    max_id = OMfind_subobj(elem,OMstr_to_name("max"),OM_OBJ_RD);
    dst_arr_id = OMfind_subobj(elem,OMstr_to_name("out"),OM_OBJ_RW);

    stat = OMget_array_size( src_arr_id, &num_fields );
    if( stat != OM_STAT_SUCCESS ) return OM_STAT_FAILURE;

    stat = OMset_array_size( dst_arr_id, num_fields );
    if( stat != OM_STAT_SUCCESS ) return OM_STAT_FAILURE;

    type = OM_TYPE_INT;
    size = 0;
    min = (int *)NULL;
    stat = OMget_array_sz(min_id, &type, (char **)(&min),
                          &size, OM_GET_ARRAY_RD);
    if (stat != OM_STAT_SUCCESS)
        return METHOD_FAILURE;
    size = 0;
    max = (int *)NULL;
    stat = OMget_array_sz(max_id, &type, (char **)(&max),
                          &size, OM_GET_ARRAY_RD);
    if (stat != OM_STAT_SUCCESS)
        return METHOD_FAILURE;

    for( i = 0; i < num_fields; ++i ) {
        stat = OMget_array_val( src_arr_id, i, &src_id, OM_OBJ_RD );
        if( stat != OM_STAT_SUCCESS ) continue;
        stat = OMget_array_val( dst_arr_id, i, &dst_id, OM_OBJ_RW );
        if( stat != OM_STAT_SUCCESS ) continue;

        stat = FUNCcrop_rect(src_id, min, max, dst_id);
#if 0
        if( stat != METHOD_SUCCESS ) {
            ERRerror( "crop_rect", 1, ERR_ORIG,
                      "Error while processing field: %d", i );
        }
#endif

        /* This can be done easily in V, but its faster in C. */
        src_xform_id = OMfind_subobj(src_id, xform_name, OM_OBJ_RD);
        dst_xform_id = OMfind_subobj(dst_id, xform_name ,OM_OBJ_RW);
        OMset_obj_ref( dst_xform_id, src_xform_id, 0 );
    }

    ARRfree(min);
    ARRfree(max);
    return stat;
}


/* 64-bit porting. Only Modified Internally */
int FUNCcrop (OMobj_id in, int *min, int *max, OMobj_id out)
{
	xp_long out_dims[3], min_rng[4], max_rng[4], rdims[4];
	xp_long *dims, size;
	int ndim, i, comp, ncomp, null_flag, data_id;
	int type, in_type, veclen, nspace, out_nspace;
	float *xyz, *out_xyz;
	int  out_ndim, dims_size;
	char  units[MAX_NAME_SIZE], label[MAX_NAME_SIZE];
	char   *node_data;
	double null_value;

	if (FLDget_dims (in, &dims, &dims_size) != 1) {
		ERR_RETURN("Error getting dims");
	}
	if (FLDget_ndim (in, &ndim) != 1) {
		ERR_RETURN("Error getting ndim");
	}
	if (FLDget_nspace (in, &nspace) != 1) {
		ERR_RETURN("Error getting nspace");
	}
	for (i=0; i<ndim; i++) {
		if (min[i] < 0) {
			sprintf(label, "invalid min value %d, 0 assumed", min[i]);
			ERRerror("crop", 0, ERR_ORIG, label);
			min[i] = 0;
		}
		if (max[i] > (int)(dims[i]-1)) {
			sprintf(label, "invalid max value %d, %d assumed", max[i], (int)(dims[i]-1));
			ERRerror("crop", 0, ERR_ORIG, label);
			max[i] = (int)(dims[i]-1);
		}

		if (min[i] >= max[i]) {
			ERR_RETURN("invalid parameters: min >= max, crop is not performed");
		}
	}
	if (FLDget_coord(in, &xyz, &size, OM_GET_ARRAY_RD) != 1) {
		ERR_RETURN("cannot get coordinates ");
	}
	/*-----------------------------*/
	/*   OUT FIELD                 */
	/*-----------------------------*/

	out_ndim = ndim;
	memcpy(rdims+1, dims, ndim*sizeof(xp_long));
	for (i=0; i<out_ndim; i++) {
		out_dims[i]  = max[i]-min[i]+1;
		min_rng[i+1] = min[i];
		max_rng[i+1] = max[i]+1;
	}
	if (FLDset_ndim (out, out_ndim) != 1) {
		ERR_RETURN("Error setting ndim");
	}
	if (FLDset_dims (out, out_dims) != 1) {
		ERR_RETURN("Error setting dims");
	}
	if (FLDget_coord_units(in, units, MAX_NAME_SIZE) == 1) {
		if (FLDset_coord_units (out, units) != 1) {
			ERR_RETURN("Error setting units");
		}
	}
	out_nspace = nspace;
	if (FLDset_nspace (out, out_nspace) != 1) {
		ERR_RETURN("Error setting nspace");
	}

	if (FLDget_coord(out, &out_xyz, &size, OM_GET_ARRAY_WR) != 1) {
		ERR_RETURN("Error setting coordinate array");
	}
	rdims[0] = nspace;
	min_rng[0] = 0;
	max_rng[0] = nspace;
	if (FLDget_sub_coord(in, ndim+1, rdims,
			     min_rng, max_rng, out_xyz) != 1) {
		ERR_RETURN("Error getting sub coordinate array");
	}
	if (FLDget_node_data_ncomp(in, &ncomp) != 1) {
		ERR_RETURN("Error getting ncomp");
	}
	if (FLDset_node_data_ncomp (out, ncomp) != 1) {
		ERR_RETURN("Error setting nnode_data");
	}
	for (comp=0; comp<ncomp; comp++) {
		if (FLDget_node_data_units(in, comp, units, MAX_NAME_SIZE) != 1) {
			strcpy(units, "");
		}
		if (FLDget_node_data_label(in, comp, label, MAX_NAME_SIZE) != 1) {
			strcpy(label, "");
		}
		if (FLDget_node_data_veclen(in, comp, &veclen) != 1) {
			ERR_RETURN("Error getting veclen");
		}
		if (FLDset_node_data_comp (out, comp, veclen, label, units) != 1) {
			ERR_RETURN("Error setting node component");
		}
		if (FLDget_node_data_id(in, comp, &data_id) == 1)
			FLDset_node_data_id(out, comp, data_id);

		if (FLDcopy_node_minmax(in, out, comp, comp) != 1) {
			ERR_RETURN("Error copying node minmax");
		}
		if (FLDcopy_node_minmax_vec(in, out, comp, comp) != 1) {
			ERR_RETURN("Error copying node minmax");
		}

		if (FLDget_node_data_type(in, comp, &in_type) != 1) {
			ERR_RETURN("Error copying node minmax");
		}
		type = in_type;
		if (FLDget_node_data(out, comp, &type, &node_data,
				      &size, OM_GET_ARRAY_WR) != 1) {
			ERR_RETURN("Error setting node data");
		}
		rdims[0] = veclen;
		min_rng[0] = 0;
		max_rng[0] = veclen;
		if (FLDget_sub_node_data(in, comp, &type, ndim+1, rdims,
				     min_rng, max_rng, node_data) != 1) {
			ERR_RETURN("cannot get node data");
		}
		if (FLDget_node_null_data(in, comp, &null_flag, (char *)&null_value) != 1) {
			ERR_RETURN("cannot get null data");
		}
		if (null_flag) {
			if (FLDset_node_null_data(out, comp, (char *)&null_value, type) != 1) {
				ERR_RETURN("Error setting null value");
			}
		}
		else {
			if (FLDset_node_null_flag(out, comp, 0) != 1) {
				ERR_RETURN("Error setting null flag");
			}
		}
		ARRfree(node_data);
	}
	ARRfree(dims);
	ARRfree(xyz);
	ARRfree(out_xyz);
	return METHOD_SUCCESS;
}


/* 64-bit porting. Only Modified Internally */
int FUNCcrop_unif (OMobj_id in, int *min, int *max, OMobj_id out)
{
	xp_long out_dims[3], min_rng[4], max_rng[4], rdims[4];
	xp_long *dims, size;
	int ndim, i, comp, ncomp, null_flag, data_id;
	int type, in_type, veclen, nspace, out_nspace;
	float *xyz, *out_xyz;
	int  out_ndim, dims_size;
	char  units[MAX_NAME_SIZE], label[MAX_NAME_SIZE];
	char   *node_data;
	double null_value;

	if (FLDget_dims (in, &dims, &dims_size) != 1) {
		ERR_RETURN("Error getting dims");
	}
	if (FLDget_ndim (in, &ndim) != 1) {
		ERR_RETURN("Error getting ndim");
	}
	if (FLDget_nspace (in, &nspace) != 1) {
		ERR_RETURN("Error getting nspace");
	}
	for (i=0; i<ndim; i++) {
		if (min[i] < 0) {
			sprintf(label, "invalid min value %d, 0 assumed", min[i]);
			ERRerror("crop", 0, ERR_ORIG, label);
			min[i] = 0;
		}
		if (max[i] > (int)(dims[i]-1)) {
			sprintf(label, "invalid max value %d, %d assumed", max[i], (int)(dims[i]-1));
			ERRerror("crop", 0, ERR_ORIG, label);
			max[i] = (int)(dims[i]-1);
		}

		if (min[i] >= max[i]) {
			ERR_RETURN("invalid parameters: min >= max, crop is not performed");
		}
	}
	if (FLDget_points (in, &xyz, &size, OM_GET_ARRAY_RD) != 1) {
		ERR_RETURN("Error getting points");
	}
	/*-----------------------------*/
	/*   OUT FIELD                 */
	/*-----------------------------*/

	out_ndim = ndim;
	memcpy(rdims+1, dims, ndim*sizeof(xp_long));
	for (i=0; i<out_ndim; i++) {
		out_dims[i]  = max[i]-min[i]+1;
		min_rng[i+1] = min[i];
		max_rng[i+1] = max[i]+1;
	}
	if (FLDset_ndim (out, out_ndim) != 1) {
		ERR_RETURN("Error setting ndim");
	}
	if (FLDset_dims (out, out_dims) != 1) {
		ERR_RETURN("Error setting dims");
	}
	if (FLDget_coord_units(in, units, MAX_NAME_SIZE) == 1) {
		if (FLDset_coord_units (out, units) != 1) {
			ERR_RETURN("Error setting units");
		}
	}
	out_nspace = nspace;
	if (FLDset_nspace (out, out_nspace) != 1) {
		ERR_RETURN("Error setting nspace");
	}

	if (FLDset_npoints (out, 2) != 1) {
		ERR_RETURN("Error setting npoints");
	}
	if (FLDget_points (out, &out_xyz, &size, OM_GET_ARRAY_WR) != 1) {
		ERR_RETURN("Error getting points");
	}
	for (i=0; i<out_ndim; i++) {
		out_xyz[i] = xyz[i] + (xyz[i+nspace] - xyz[i])*min[i]/
			(dims[i] - 1.);
		out_xyz[i+out_nspace] = xyz[i] + (xyz[i+nspace] - xyz[i])*max[i]/
			(dims[i] - 1.);
	}
	for (;i<out_nspace; i++) {
		out_xyz[i] = xyz[i];
		out_xyz[i+out_nspace] = xyz[i+out_nspace];
	}

	if (FLDget_node_data_ncomp(in, &ncomp) != 1) {
		ERR_RETURN("Error getting ncomp");
	}
	if (FLDset_node_data_ncomp (out, ncomp) != 1) {
		ERR_RETURN("Error setting nnode_data");
	}
	for (comp=0; comp<ncomp; comp++) {
		if (FLDget_node_data_units(in, comp, units, MAX_NAME_SIZE) != 1) {
			strcpy(units, "");
		}
		if (FLDget_node_data_label(in, comp, label, MAX_NAME_SIZE) != 1) {
			strcpy(label, "");
		}
		if (FLDget_node_data_veclen(in, comp, &veclen) != 1) {
			ERR_RETURN("Error getting veclen");
		}
		if (FLDset_node_data_comp (out, comp, veclen, label, units) != 1) {
			ERR_RETURN("Error setting node component");
		}
		if (FLDget_node_data_id(in, comp, &data_id) == 1)
			FLDset_node_data_id(out, comp, data_id);

		if (FLDcopy_node_minmax(in, out, comp, comp) != 1) {
			ERR_RETURN("Error copying node minmax");
		}
		if (FLDcopy_node_minmax_vec(in, out, comp, comp) != 1) {
			ERR_RETURN("Error copying node minmax");
		}

		if (FLDget_node_data_type(in, comp, &in_type) != 1) {
			ERR_RETURN("Error copying node minmax");
		}
		type = in_type;
		if (FLDget_node_data(out, comp, &type, &node_data,
				      &size, OM_GET_ARRAY_WR) != 1) {
			ERR_RETURN("Error setting node data");
		}
		rdims[0] = veclen;
		min_rng[0] = 0;
		max_rng[0] = veclen;
		if (FLDget_sub_node_data(in, comp, &type, ndim+1, rdims,
				     min_rng, max_rng, node_data) != 1) {
			ERR_RETURN("cannot get node data");
		}
		if (FLDget_node_null_data(in, comp, &null_flag, (char *)&null_value) != 1) {
			ERR_RETURN("cannot get null data");
		}
		if (null_flag) {
			if (FLDset_node_null_data(out, comp, (char *)&null_value, type) != 1) {
				ERR_RETURN("Error setting null value");
			}
		}
		else {
			if (FLDset_node_null_flag(out, comp, 0) != 1) {
				ERR_RETURN("Error setting null flag");
			}
		}
		ARRfree(node_data);
	}
	ARRfree(dims);
	ARRfree(xyz);
	ARRfree(out_xyz);
	return METHOD_SUCCESS;
}

/* 64-bit porting. Only Modified Internally */
int FUNCcrop_rect (OMobj_id in, int *min, int *max, OMobj_id out)
{
	xp_long out_ind, out_dims[3], min_rng[4], max_rng[4], rdims[4];
	xp_long n, *dims, size, offset[3], out_offset[3];
	int ndim, i, j, comp, ncomp, null_flag, data_id;
	int type, in_type, veclen, nspace, out_nspace;
	float *xyz, *out_xyz;
	int out_ndim, dims_size;
	char  units[MAX_NAME_SIZE], label[MAX_NAME_SIZE];
	char   *node_data;
	double null_value;

	if (FLDget_dims (in, &dims, &dims_size) != 1) {
		ERR_RETURN("Error getting dims");
	}
	if (FLDget_ndim (in, &ndim) != 1) {
		ERR_RETURN("Error getting ndim");
	}
	if (FLDget_nspace (in, &nspace) != 1) {
		ERR_RETURN("Error getting nspace");
	}
	for (i=0; i<ndim; i++) {
		if (min[i] < 0) {
			sprintf(label, "invalid min value %d, 0 assumed", min[i]);
			ERRerror("crop",0, ERR_ORIG, label);
			min[i] = 0;
		}
		if (max[i] > (int)(dims[i]-1)) {
			sprintf(label, "invalid max value %d, %d assumed", max[i], (int)(dims[i]-1));
			ERRerror("crop", 0, ERR_ORIG, label);
			max[i] = (int)(dims[i]-1);
		}

		if (min[i] >= max[i]) {
			ERR_RETURN("invalid parameters: min >= max, crop is not performed");
		}
	}
	if (FLDget_points (in, &xyz, &size, OM_GET_ARRAY_RD) != 1) {
		ERR_RETURN("Error getting points");
	}
	/*-----------------------------*/
	/*   OUT FIELD                 */
	/*-----------------------------*/

	out_ndim = ndim;
	memcpy(rdims+1, dims, ndim*sizeof(xp_long));
	for (i=0; i<out_ndim; i++) {
		out_dims[i]  = max[i]-min[i]+1;
		min_rng[i+1] = min[i];
		max_rng[i+1] = max[i]+1;
	}
	if (FLDset_ndim (out, out_ndim) != 1) {
		ERR_RETURN("Error setting ndim");
	}
	if (FLDset_dims (out, out_dims) != 1) {
		ERR_RETURN("Error setting dims");
	}
	if (FLDget_coord_units(in, units, MAX_NAME_SIZE) == 1) {
		if (FLDset_coord_units (out, units) != 1) {
			ERR_RETURN("Error setting units");
		}
	}
	out_nspace = nspace;
	if (FLDset_nspace (out, out_nspace) != 1) {
		ERR_RETURN("Error setting nspace");
	}

	for (n=0,i=0; i<out_ndim; i++)
		n += out_dims[i];
	if (FLDset_npoints (out, n) != 1) {
		ERR_RETURN("Error setting npoints");
	}
	if (FLDget_points (out, &out_xyz, &size, OM_GET_ARRAY_WR) != 1) {
		ERR_RETURN("Error getting points");
	}
	offset[0] = 0;
	out_offset[0] = 0;
	for (i=1; i<ndim; i++)
		offset[i] = offset[i-1] + dims[i-1]*nspace;
	for (i=1; i<out_ndim; i++)
		out_offset[i] = out_offset[i-1] + out_dims[i-1]*out_nspace;

	for (i=0; i<out_ndim; i++) {
		for (n=0; n<out_dims[i]; n++) {
			for (j=0; j<out_ndim; j++) {
				out_ind = out_offset[i]+n*out_nspace+j;
				out_xyz[out_ind] =
					xyz[offset[i]+(n+min[i])*nspace+j];
			}
			for (;j<out_nspace; j++) {
				out_ind = out_offset[i]+n*out_nspace+j;
				out_xyz[out_ind] =
					xyz[offset[i]+(n+min[i])*nspace+j];
			}
		}
	}
	if (FLDget_node_data_ncomp(in, &ncomp) != 1) {
		ERR_RETURN("Error getting ncomp");
	}
	if (FLDset_node_data_ncomp (out, ncomp) != 1) {
		ERR_RETURN("Error setting nnode_data");
	}
	for (comp=0; comp<ncomp; comp++) {
		if (FLDget_node_data_units(in, comp, units, MAX_NAME_SIZE) != 1) {
			strcpy(units, "");
		}
		if (FLDget_node_data_label(in, comp, label, MAX_NAME_SIZE) != 1) {
			strcpy(label, "");
		}
		if (FLDget_node_data_veclen(in, comp, &veclen) != 1) {
			ERR_RETURN("Error getting veclen");
		}
		if (FLDset_node_data_comp (out, comp, veclen, label, units) != 1) {
			ERR_RETURN("Error setting node component");
		}
		if (FLDget_node_data_id(in, comp, &data_id) == 1)
			FLDset_node_data_id(out, comp, data_id);

		if (FLDcopy_node_minmax(in, out, comp, comp) != 1) {
			ERR_RETURN("Error copying node minmax");
		}
		if (FLDcopy_node_minmax_vec(in, out, comp, comp) != 1) {
			ERR_RETURN("Error copying node minmax");
		}

		if (FLDget_node_data_type(in, comp, &in_type) != 1) {
			ERR_RETURN("Error copying node minmax");
		}
		type = in_type;
		if (FLDget_node_data(out, comp, &type, &node_data,
				      &size, OM_GET_ARRAY_WR) != 1) {
			ERR_RETURN("Error setting node data");
		}
		rdims[0] = veclen;
		min_rng[0] = 0;
		max_rng[0] = veclen;
		if (FLDget_sub_node_data(in, comp, &type, ndim+1, rdims,
				     min_rng, max_rng, node_data) != 1) {
			ERR_RETURN("cannot get node data");
		}
		if (FLDget_node_null_data(in, comp, &null_flag, (char *)&null_value) != 1) {
			ERR_RETURN("cannot get null data");
		}
		if (null_flag) {
			if (FLDset_node_null_data(out, comp, (char *)&null_value, type) != 1) {
				ERR_RETURN("Error setting null value");
			}
		}
		else {
			if (FLDset_node_null_flag(out, comp, 0) != 1) {
				ERR_RETURN("Error setting null flag");
			}
		}
		ARRfree(node_data);
	}
	ARRfree(dims);
	ARRfree(xyz);
	ARRfree(out_xyz);
	return METHOD_SUCCESS;
}
