/*
			Copyright (c) 1994 by
			Advanced Visual Systems Inc.
			All Rights Reserved

	This software comprises unpublished confidential information of
	Advanced Visual Systems Inc. and may not be used, copied or made
	available to anyone, except in accordance with the license
	under which it is furnished.

	This file is under Perforce control
	$Id: //depot/express/fcs70/modules/down.c#1 $
*/

#define XP_WIDE_API	/* Use Wide APIs */

#include <avs/util.h>
#include <avs/err.h>
#include <avs/dtype.h>
#include <avs/om.h>
#include <avs/fld.h>
#include <avs/dv_util.h>

#define METHOD_SUCCESS 1
#define METHOD_FAILURE 0

#define ERR_RETURN(A) {ERRerror("downsize", 0, ERR_ORIG, A); \
                       return METHOD_FAILURE;}

#define MAX_NAME_SIZE 1024

int FUNCdownsize (OMobj_id in, float *stride, OMobj_id out);
int FUNCdownsize_unif (OMobj_id in, float *stride, OMobj_id out);
int FUNCdownsize_rect (OMobj_id in, float *stride, OMobj_id out);

/* 64-bit porting. Only Modified Internally */
int DVdownsize_update(OMobj_id elem)
{
    OMobj_id src_id, dst_id, stride_id;
    xp_long size;
    int type, stat;
    float *stride;

    src_id = OMfind_subobj(elem,OMstr_to_name("in"),OM_OBJ_RD);
    stride_id = OMfind_subobj(elem,OMstr_to_name("factor"),OM_OBJ_RD);
    dst_id = OMfind_subobj(elem,OMstr_to_name("out"),OM_OBJ_RW); 

    type = OM_TYPE_FLOAT;
    size = 0;
    stride = (float *)NULL;
    stat = OMget_array_sz(stride_id, &type, (char **)(&stride), 
                          &size, OM_GET_ARRAY_RD);
    if (stat != OM_STAT_SUCCESS)
        return METHOD_FAILURE;

    stat = FUNCdownsize(src_id, stride, dst_id);

    ARRfree(stride);
    return stat;
}


/* 64-bit porting. Only Modified Internally */
int DVdownsize_unif_update(OMobj_id elem)
{
    OMobj_id src_id, dst_id, stride_id;
    xp_long size;
    int type, stat;
    float *stride;

    src_id = OMfind_subobj(elem,OMstr_to_name("in"),OM_OBJ_RD);
    stride_id = OMfind_subobj(elem,OMstr_to_name("factor"),OM_OBJ_RD);
    dst_id = OMfind_subobj(elem,OMstr_to_name("out"),OM_OBJ_RW); 

    type = OM_TYPE_FLOAT;
    size = 0;
    stride = (float *)NULL;
    stat = OMget_array_sz(stride_id, &type, (char **)(&stride), 
                          &size, OM_GET_ARRAY_RD);
    if (stat != OM_STAT_SUCCESS)
        return METHOD_FAILURE;

    stat = FUNCdownsize_unif(src_id, stride, dst_id);

    ARRfree(stride);
    return stat;
}

/* 64-bit porting. Only Modified Internally */
int DVdownsize_rect_update(OMobj_id elem)
{
    OMobj_id src_id, dst_id, stride_id;
    xp_long size;
    int type, stat;
    float *stride;

    src_id = OMfind_subobj(elem,OMstr_to_name("in"),OM_OBJ_RD);
    stride_id = OMfind_subobj(elem,OMstr_to_name("factor"),OM_OBJ_RD);
    dst_id = OMfind_subobj(elem,OMstr_to_name("out"),OM_OBJ_RW); 

    type = OM_TYPE_FLOAT;
    size = 0;
    stride = (float *)NULL;
    stat = OMget_array_sz(stride_id, &type, (char **)(&stride), 
                          &size, OM_GET_ARRAY_RD);
    if (stat != OM_STAT_SUCCESS)
        return METHOD_FAILURE;

    stat = FUNCdownsize_rect(src_id, stride, dst_id);

    ARRfree(stride);
    return stat;
}


/* 64-bit porting. Only Modified Internally */
int
DV_ARRdownsize_update(OMobj_id elem_id, OMevent_mask event_mask, int seq_num)
{
    OMobj_id src_arr_id, dst_arr_id;
    xp_long i, num_fields;

    OMobj_id src_id, dst_id, stride_id;
    OMobj_id src_xform_id, dst_xform_id;
    OMobj_name xform_name = OMstr_to_name("xform");
    xp_long size;
    int type, stat;
    float *stride;

    src_arr_id = OMfind_subobj(elem_id,OMstr_to_name("in"),OM_OBJ_RD);
    stride_id = OMfind_subobj(elem_id,OMstr_to_name("factor"),OM_OBJ_RD);
    dst_arr_id = OMfind_subobj(elem_id,OMstr_to_name("out"),OM_OBJ_RW);

    stat = OMget_array_size( src_arr_id, &num_fields );
    if( stat != OM_STAT_SUCCESS ) return METHOD_FAILURE;

    stat = OMset_array_size( dst_arr_id, num_fields );
    if( stat != OM_STAT_SUCCESS ) return METHOD_FAILURE;

    type = OM_TYPE_FLOAT;
    size = 0;
    stride = (float *)NULL;
    stat = OMget_array_sz(stride_id, &type, (char **)(&stride),
                          &size, OM_GET_ARRAY_RD);
    if (stat != OM_STAT_SUCCESS)
        return METHOD_FAILURE;

    for( i = 0; i < num_fields; ++i ) {
        stat = OMget_array_val( src_arr_id, i, &src_id, OM_OBJ_RD );
        if( stat != OM_STAT_SUCCESS ) continue;
        stat = OMget_array_val( dst_arr_id, i, &dst_id, OM_OBJ_RW );
        if( stat != OM_STAT_SUCCESS ) continue;

        stat = FUNCdownsize(src_id, stride, dst_id);
#if 0
        if( stat != METHOD_SUCCESS ) {
            ERRerror( "downsize", 1, ERR_ORIG,
                      "Error while processing field: %d", i );
        }
#endif

        /* This can be done easily in V, but its faster in C. */
        src_xform_id = OMfind_subobj(src_id, xform_name, OM_OBJ_RD);
        dst_xform_id = OMfind_subobj(dst_id, xform_name ,OM_OBJ_RW);
        OMset_obj_ref( dst_xform_id, src_xform_id, 0 );
    }

    ARRfree(stride);

    return METHOD_SUCCESS;
}

/* 64-bit porting. Only Modified Internally */
int
DV_ARRdownsize_unif_update(OMobj_id elem_id, OMevent_mask event_mask, int seq_num)
{
    OMobj_id src_arr_id, dst_arr_id;
    xp_long i, num_fields;

    OMobj_id src_id, dst_id, stride_id;
    OMobj_id src_xform_id, dst_xform_id;
    OMobj_name xform_name = OMstr_to_name("xform");
    xp_long size;
    int type, stat;
    float *stride;

    src_arr_id = OMfind_subobj(elem_id,OMstr_to_name("in"),OM_OBJ_RD);
    stride_id = OMfind_subobj(elem_id,OMstr_to_name("factor"),OM_OBJ_RD);
    dst_arr_id = OMfind_subobj(elem_id,OMstr_to_name("out"),OM_OBJ_RW);

    stat = OMget_array_size( src_arr_id, &num_fields );
    if( stat != OM_STAT_SUCCESS ) return METHOD_FAILURE;

    stat = OMset_array_size( dst_arr_id, num_fields );
    if( stat != OM_STAT_SUCCESS ) return METHOD_FAILURE;

    type = OM_TYPE_FLOAT;
    size = 0;
    stride = (float *)NULL;
    stat = OMget_array_sz(stride_id, &type, (char **)(&stride),
                          &size, OM_GET_ARRAY_RD);
    if (stat != OM_STAT_SUCCESS)
        return METHOD_FAILURE;

    for( i = 0; i < num_fields; ++i ) {
        stat = OMget_array_val( src_arr_id, i, &src_id, OM_OBJ_RD );
        if( stat != OM_STAT_SUCCESS ) continue;
        stat = OMget_array_val( dst_arr_id, i, &dst_id, OM_OBJ_RW );
        if( stat != OM_STAT_SUCCESS ) continue;

        stat = FUNCdownsize_unif(src_id, stride, dst_id);
#if 0
        if( stat != METHOD_SUCCESS ) {
            ERRerror( "downsize_unif", 1, ERR_ORIG,
                      "Error while processing field: %d", i );
        }
#endif

        /* This can be done easily in V, but its faster in C. */
        src_xform_id = OMfind_subobj(src_id, xform_name, OM_OBJ_RD);
        dst_xform_id = OMfind_subobj(dst_id, xform_name ,OM_OBJ_RW);
        OMset_obj_ref( dst_xform_id, src_xform_id, 0 );
    }

    ARRfree(stride);

    return METHOD_SUCCESS;
}

/* 64-bit porting. Only Modified Internally */
int
DV_ARRdownsize_rect_update(OMobj_id elem_id, OMevent_mask event_mask, int seq_num)
{
    OMobj_id src_arr_id, dst_arr_id;
    xp_long i, num_fields;

    OMobj_id src_id, dst_id, stride_id;
    OMobj_id src_xform_id, dst_xform_id;
    OMobj_name xform_name = OMstr_to_name("xform");
    xp_long size;
    int type, stat;
    float *stride;

    src_arr_id = OMfind_subobj(elem_id,OMstr_to_name("in"),OM_OBJ_RD);
    stride_id = OMfind_subobj(elem_id,OMstr_to_name("factor"),OM_OBJ_RD);
    dst_arr_id = OMfind_subobj(elem_id,OMstr_to_name("out"),OM_OBJ_RW);

    stat = OMget_array_size( src_arr_id, &num_fields );
    if( stat != OM_STAT_SUCCESS ) return METHOD_FAILURE;

    stat = OMset_array_size( dst_arr_id, num_fields );
    if( stat != OM_STAT_SUCCESS ) return METHOD_FAILURE;

    type = OM_TYPE_FLOAT;
    size = 0;
    stride = (float *)NULL;
    stat = OMget_array_sz(stride_id, &type, (char **)(&stride),
                          &size, OM_GET_ARRAY_RD);
    if (stat != OM_STAT_SUCCESS)
        return METHOD_FAILURE;

    for( i = 0; i < num_fields; ++i ) {
        stat = OMget_array_val( src_arr_id, i, &src_id, OM_OBJ_RD );
        if( stat != OM_STAT_SUCCESS ) continue;
        stat = OMget_array_val( dst_arr_id, i, &dst_id, OM_OBJ_RW );
        if( stat != OM_STAT_SUCCESS ) continue;

        stat = FUNCdownsize_rect(src_id, stride, dst_id);
#if 0
        if( stat != METHOD_SUCCESS ) {
            ERRerror( "downsize_rect", 1, ERR_ORIG,
                      "Error while processing field: %d", i );
        }
#endif

        /* This can be done easily in V, but its faster in C. */
        src_xform_id = OMfind_subobj(src_id, xform_name, OM_OBJ_RD);
        dst_xform_id = OMfind_subobj(dst_id, xform_name ,OM_OBJ_RW);
        OMset_obj_ref( dst_xform_id, src_xform_id, 0 );
    }

    ARRfree(stride);

    return METHOD_SUCCESS;
}


/* 64-bit porting. Only Modified Internally */
int FUNCdownsize (OMobj_id in, float *stride, OMobj_id out)
{
	int ndim, i, stat, comp, ncomp, null_flag, data_id;
	xp_long *dims, size;
	int type, veclen, nspace, out_nspace;
	float *xyz, *out_xyz;
	int  out_ndim, dims_size;
	xp_long out_dims[3];
	char  units[MAX_NAME_SIZE], label[MAX_NAME_SIZE];
	char   *node_data, *out_node_data;
	double null_value;

	if (FLDget_dims (in, &dims, &dims_size) != 1) {
		ERR_RETURN("Error getting dims");
	}
	if (FLDget_ndim (in, &ndim) != 1) {
		ERR_RETURN("Error getting ndim");
	}
	if (FLDget_nspace (in, &nspace) != 1) {
		ERR_RETURN("Error getting nspace");
	}
	for (i=0; i<ndim; i++) 
		if (stride[i] <= 0 || stride[i] >= dims[i] ) {
			ARRfree(dims);
			sprintf(label, "invalid factor value %f, downsize is not performed", stride[i]);
			ERR_RETURN(label);
		}
	if (FLDget_coord(in, &xyz, &size, OM_GET_ARRAY_RD) != 1) {
		ERR_RETURN("cannot get coordinates ");
	}
	/*-----------------------------*/
	/*   OUT FIELD                 */
	/*-----------------------------*/

	out_ndim = ndim;
	for (i=0; i<out_ndim; i++) {
		out_dims[i]  = dims[i]/stride[i];
		if (out_dims[i] < 2) out_dims[i] = 2;
	}
	if (FLDset_ndim (out, out_ndim) != 1) {
		ERR_RETURN("Error setting ndim");
	}
	if (FLDset_dims (out, out_dims) != 1) {
		ERR_RETURN("Error setting dims");
	}
	if (FLDget_coord_units(in, units, MAX_NAME_SIZE) == 1) {
		if (FLDset_coord_units (out, units) != 1) {
			ERR_RETURN("Error setting units");
		}
	}
	out_nspace = nspace;
	if (FLDset_nspace (out, out_nspace) != 1) {
		ERR_RETURN("Error setting nspace");
	}

	if (FLDget_coord(out, &out_xyz, &size, OM_GET_ARRAY_WR) != 1) {
		ERR_RETURN("Error setting coordinate array");
	}
	stat = UTILresample(ndim, dims, out_dims, nspace, DTYPE_FLOAT, (char *)xyz, (char *)out_xyz);
	if(!stat) {
		ERR_RETURN("Error resampling coordinates");
	}
	if (FLDget_node_data_ncomp(in, &ncomp) != 1) {
		ERR_RETURN("Error getting ncomp");
	}
	if (FLDset_node_data_ncomp (out, ncomp) != 1) {
		ERR_RETURN("Error setting nnode_data");
	}
	for (comp=0; comp<ncomp; comp++) {
		if (FLDget_node_data_units(in, comp, units, MAX_NAME_SIZE) != 1) {
			strcpy(units, "");
		}
		if (FLDget_node_data_label(in, comp, label, MAX_NAME_SIZE) != 1) {
			strcpy(label, "");
		}
		if (FLDget_node_data_veclen(in, comp, &veclen) != 1) {
			ERR_RETURN("Error getting veclen");
		}
		if (FLDset_node_data_comp (out, comp, veclen, label, units) != 1) {
			ERR_RETURN("Error setting node component");
		}
		if (FLDget_node_data_id(in, comp, &data_id) == 1)
			FLDset_node_data_id(out, comp, data_id);

		if (FLDcopy_node_minmax(in, out, comp, comp) != 1) {
			ERR_RETURN("Error copying node minmax");
		}
		if (FLDcopy_node_minmax_vec(in, out, comp, comp) != 1) {
			ERR_RETURN("Error copying node minmax");
		}

		if (FLDget_node_data_type(in, comp, &type) != 1) {
			ERR_RETURN("Error copying node minmax");
		}

		if (FLDget_node_data(in, comp, &type, &node_data, 
				      &size, OM_GET_ARRAY_RD) != 1) {
			ERR_RETURN("Error getting node data");
		}
		if (FLDget_node_data(out, comp, &type, &out_node_data, 
				      &size, OM_GET_ARRAY_WR) != 1) {
			ERR_RETURN("Error setting node data");
		}
		stat = UTILresample(ndim, dims, out_dims, veclen, 
				    type, node_data, out_node_data);
		if (!stat) {
			ERR_RETURN("Error resampling data");
		}
		if (FLDget_node_null_data(in, comp, &null_flag, (char *)&null_value) != 1) {
			ERR_RETURN("cannot get null data");
		}
		if (null_flag) {
			if (FLDset_node_null_data(out, comp, (char *)&null_value, type) != 1) {
				ERR_RETURN("Error setting null value");
			}
		}
		else {
			if (FLDset_node_null_flag(out, comp, 0) != 1) {
				ERR_RETURN("Error setting null flag");
			}
		}
		ARRfree(node_data);
		ARRfree(out_node_data);
	}
	ARRfree(dims);
	ARRfree(xyz);
	ARRfree(out_xyz);
	return METHOD_SUCCESS;
}


/* 64-bit porting. Only Modified Internally */
int FUNCdownsize_unif (OMobj_id in, float *stride, OMobj_id out)
{
	int ndim, i, stat, comp, ncomp, null_flag, data_id;
	xp_long *dims, size;
	int type, veclen, nspace, out_nspace;
	float *xyz, *out_xyz;
	int  out_ndim, dims_size;
	xp_long out_dims[3];
	char  units[MAX_NAME_SIZE], label[MAX_NAME_SIZE];
	char   *node_data, *out_node_data;
	double null_value;

	if (FLDget_dims (in, &dims, &dims_size) != 1) {
		ERR_RETURN("Error getting dims");
	}
	if (FLDget_ndim (in, &ndim) != 1) {
		ERR_RETURN("Error getting ndim");
	}
	if (FLDget_nspace (in, &nspace) != 1) {
		ERR_RETURN("Error getting nspace");
	}
	for (i=0; i<ndim; i++) 
		if (stride[i] <= 0 || stride[i] >= dims[i] ) {
			ARRfree(dims);
			sprintf(label, "invalid factor value %f, downsize is not performed", stride[i]);
			ERR_RETURN(label);
		}
	if (FLDget_points (in, &xyz, &size, OM_GET_ARRAY_RD) != 1) {
		ERR_RETURN("Error getting points");
	}
	/*-----------------------------*/
	/*   OUT FIELD                 */
	/*-----------------------------*/

	out_ndim = ndim;
	for (i=0; i<out_ndim; i++) {
		out_dims[i]  = dims[i]/stride[i];
		if (out_dims[i] < 2) out_dims[i] = 2;
	}
	if (FLDset_ndim (out, out_ndim) != 1) {
		ERR_RETURN("Error setting ndim");
	}
	if (FLDset_dims (out, out_dims) != 1) {
		ERR_RETURN("Error setting dims");
	}
	if (FLDget_coord_units(in, units, MAX_NAME_SIZE) == 1) {
		if (FLDset_coord_units (out, units) != 1) {
			ERR_RETURN("Error setting units");
		}
	}
	out_nspace = nspace;
	if (FLDset_nspace (out, out_nspace) != 1) {
		ERR_RETURN("Error setting nspace");
	}

	if (FLDset_npoints (out, 2) != 1) {
		ERR_RETURN("Error setting npoints");
	}
	if (FLDget_points (out, &out_xyz, &size, OM_GET_ARRAY_WR) != 1) {
		ERR_RETURN("Error getting points");
	}
	for (i=0; i<out_nspace; i++) {
		out_xyz[i] = xyz[i];
		out_xyz[i+out_nspace] = xyz[i+out_nspace];
	}
	if (FLDget_node_data_ncomp(in, &ncomp) != 1) {
		ERR_RETURN("Error getting ncomp");
	}
	if (FLDset_node_data_ncomp (out, ncomp) != 1) {
		ERR_RETURN("Error setting nnode_data");
	}
	for (comp=0; comp<ncomp; comp++) {
		if (FLDget_node_data_units(in, comp, units, MAX_NAME_SIZE) != 1) {
			strcpy(units, "");
		}
		if (FLDget_node_data_label(in, comp, label, MAX_NAME_SIZE) != 1) {
			strcpy(label, "");
		}
		if (FLDget_node_data_veclen(in, comp, &veclen) != 1) {
			ERR_RETURN("Error getting veclen");
		}
		if (FLDset_node_data_comp (out, comp, veclen, label, units) != 1) {
			ERR_RETURN("Error setting node component");
		}
		if (FLDget_node_data_id(in, comp, &data_id) == 1)
			FLDset_node_data_id(out, comp, data_id);

		if (FLDcopy_node_minmax(in, out, comp, comp) != 1) {
			ERR_RETURN("Error copying node minmax");
		}
		if (FLDcopy_node_minmax_vec(in, out, comp, comp) != 1) {
			ERR_RETURN("Error copying node minmax");
		}

		if (FLDget_node_data_type(in, comp, &type) != 1) {
			ERR_RETURN("Error copying node minmax");
		}

		if (FLDget_node_data(in, comp, &type, &node_data, 
				      &size, OM_GET_ARRAY_RD) != 1) {
			ERR_RETURN("Error setting node data");
		}
		if (FLDget_node_data(out, comp, &type, &out_node_data, 
				      &size, OM_GET_ARRAY_WR) != 1) {
			ERR_RETURN("Error setting node data");
		}
		stat = UTILresample(ndim, dims, out_dims, veclen, 
				    type, node_data, out_node_data);
		if (!stat) {
			ERR_RETURN("Error resampling data");
		}

		if (FLDget_node_null_data(in, comp, &null_flag, (char *)&null_value) != 1) {
			ERR_RETURN("cannot get null data");
		}
		if (null_flag) {
			if (FLDset_node_null_data(out, comp, (char *)&null_value, type) != 1) {
				ERR_RETURN("Error setting null value");
			}
		}
		else {
			if (FLDset_node_null_flag(out, comp, 0) != 1) {
				ERR_RETURN("Error setting null flag");
			}
		}
		ARRfree(node_data);
		ARRfree(out_node_data);
	}
	ARRfree(dims);
	ARRfree(xyz);
	ARRfree(out_xyz);
	return METHOD_SUCCESS;
}

/* 64-bit porting. Only Modified Internally */
int FUNCdownsize_rect (OMobj_id in, float *stride, OMobj_id out)
{
	int ndim, i, stat, comp, ncomp, null_flag, data_id;
	xp_long *dims, n, size, offset, out_offset;
	int type, veclen, nspace, out_nspace;
	float *xyz, *out_xyz;
	int  out_ndim, dims_size;
	xp_long out_dims[3];
	char  units[MAX_NAME_SIZE], label[MAX_NAME_SIZE];
	char   *node_data, *out_node_data;
	double null_value;

	if (FLDget_dims (in, &dims, &dims_size) != 1) {
		ERR_RETURN("Error getting dims");
	}
	if (FLDget_ndim (in, &ndim) != 1) {
		ERR_RETURN("Error getting ndim");
	}
	if (FLDget_nspace (in, &nspace) != 1) {
		ERR_RETURN("Error getting nspace");
	}
	for (i=0; i<ndim; i++) 
		if (stride[i] <= 0 || stride[i] >= dims[i] ) {
			ARRfree(dims);
			sprintf(label, "invalid factor value %f, downsize is not performed", stride[i]);
			ERR_RETURN(label);
		}
	if (FLDget_points (in, &xyz, &size, OM_GET_ARRAY_RD) != 1) {
		ERR_RETURN("Error getting points");
	}
	/*-----------------------------*/
	/*   OUT FIELD                 */
	/*-----------------------------*/

	out_ndim = ndim;
	for (i=0; i<out_ndim; i++) {
		out_dims[i]  = dims[i]/stride[i];
		if (out_dims[i] < 2) out_dims[i] = 2;
	}
	if (FLDset_ndim (out, out_ndim) != 1) {
		ERR_RETURN("Error setting ndim");
	}
	if (FLDset_dims (out, out_dims) != 1) {
		ERR_RETURN("Error setting dims");
	}
	if (FLDget_coord_units(in, units, MAX_NAME_SIZE) == 1) {
		if (FLDset_coord_units (out, units) != 1) {
			ERR_RETURN("Error setting units");
		}
	}
	out_nspace = nspace;
	if (FLDset_nspace (out, out_nspace) != 1) {
		ERR_RETURN("Error setting nspace");
	}

	for (n=0,i=0; i<out_ndim; i++)
		n += out_dims[i];
	if (FLDset_npoints (out, n) != 1) {
		ERR_RETURN("Error setting npoints");
	}
	if (FLDget_points (out, &out_xyz, &size, OM_GET_ARRAY_WR) != 1) {
		ERR_RETURN("Error getting points");
	}
	offset = 0;
	out_offset = 0;
	for (i=0; i<ndim; i++) {
		stat = UTILresample(1, &dims[i], &out_dims[i], nspace, DTYPE_FLOAT, 
				    (char *)(xyz+offset), (char *)(out_xyz+out_offset));
		if(!stat) {
			ERR_RETURN("Error resampling coordinates");
		}
		offset += dims[i]*nspace;
		out_offset += out_dims[i]*nspace;
	}
	if (FLDget_node_data_ncomp(in, &ncomp) != 1) {
		ERR_RETURN("Error getting ncomp");
	}
	if (FLDset_node_data_ncomp (out, ncomp) != 1) {
		ERR_RETURN("Error setting nnode_data");
	}
	for (comp=0; comp<ncomp; comp++) {
		if (FLDget_node_data_units(in, comp, units, MAX_NAME_SIZE) != 1) {
			strcpy(units, "");
		}
		if (FLDget_node_data_label(in, comp, label, MAX_NAME_SIZE) != 1) {
			strcpy(label, "");
		}
		if (FLDget_node_data_veclen(in, comp, &veclen) != 1) {
			ERR_RETURN("Error getting veclen");
		}
		if (FLDset_node_data_comp (out, comp, veclen, label, units) != 1) {
			ERR_RETURN("Error setting node component");
		}
		if (FLDget_node_data_id(in, comp, &data_id) == 1)
			FLDset_node_data_id(out, comp, data_id);

		if (FLDcopy_node_minmax(in, out, comp, comp) != 1) {
			ERR_RETURN("Error copying node minmax");
		}
		if (FLDcopy_node_minmax_vec(in, out, comp, comp) != 1) {
			ERR_RETURN("Error copying node minmax");
		}

		if (FLDget_node_data_type(in, comp, &type) != 1) {
			ERR_RETURN("Error copying node minmax");
		}

		if (FLDget_node_data(in, comp, &type, &node_data, 
				      &size, OM_GET_ARRAY_RD) != 1) {
			ERR_RETURN("Error setting node data");
		}
		if (FLDget_node_data(out, comp, &type, &out_node_data, 
				      &size, OM_GET_ARRAY_WR) != 1) {
			ERR_RETURN("Error setting node data");
		}
		stat = UTILresample(ndim, dims, out_dims, veclen, 
				    type, node_data, out_node_data);
		if (!stat) {
			ERR_RETURN("Error resampling data");
		}

		if (FLDget_node_null_data(in, comp, &null_flag, (char *)&null_value) != 1) {
			ERR_RETURN("cannot get null data");
		}
		if (null_flag) {
			if (FLDset_node_null_data(out, comp, (char *)&null_value, type) != 1) {
				ERR_RETURN("Error setting null value");
			}
		}
		else {
			if (FLDset_node_null_flag(out, comp, 0) != 1) {
				ERR_RETURN("Error setting null flag");
			}
		}
		ARRfree(node_data);
		ARRfree(out_node_data);
	}
	ARRfree(dims);
	ARRfree(xyz);
	ARRfree(out_xyz);
	return METHOD_SUCCESS;
}
