/*
			Copyright (c) 1994 by
			Advanced Visual Systems Inc.
			All Rights Reserved

	This software comprises unpublished confidential information of
	Advanced Visual Systems Inc. and may not be used, copied or made
	available to anyone, except in accordance with the license
	under which it is furnished.

	This file is under Perforce control
	$Id: //depot/express/fcs70/modules/exc_brk3.c#1 $
*/

#define XP_WIDE_API	/* Use Wide APIs */

#include <avs/util.h>
#include <avs/err.h>
#include <avs/om.h>
#include <avs/fld.h>
#include <avs/arr.h>

#define ERR_RETURN(A) ERRerror("excavate brick", 0, ERR_ORIG, A); return(0);

typedef struct _OBJ {
	xp_long connect[1024];
	float coord[256][3];
	float uvws[256][3];
	float norm[256][3];
	xp_long nnodes;
	xp_long ncells;
} OBJ;

int FUNCexcavate_brick3d(OMobj_id in, OMobj_id out,
	int X, int Y, int Z, int Flip_X, int Flip_Y, int Flip_Z,
	int Draw_Sides);

static int create_square(OBJ *obj, 
	float x1, float x2, float y1, float y2, float z1, float z2,
	float u1, float u2, float v1, float v2, float w1, float w2, int N);

static int create_x_L(OBJ *obj, 
	float x, float y1, float y, float y2, float z1, float z, 
	float z2, float u, float v1, float v, float v2, float w1, 
	float w, float w2, int N);

static int create_y_L(OBJ *obj, 
	float x1, float x, float x2, float y, float z1, float z, 
	float z2, float u1, float u, float u2, float v, float w1, 
	float w, float w2, int N);

static int create_z_L(OBJ *obj, 
	float x1, float x, float x2, float y1, float y, float y2, 
	float z,  float u1, float u, float u2, 
	float v1, float v, float v2, float w, int N);


int DVexcavate_brick3d_update(OMobj_id elem_id)
{
	OMobj_id in, out;
	int stat, X, Y, Z, Flip_X, Flip_Y, Flip_Z, Draw_Sides;

	in = OMfind_subobj(elem_id, OMstr_to_name("in"), OM_OBJ_RD);
	out = OMfind_subobj(elem_id, OMstr_to_name("out"), OM_OBJ_RW);

	stat = OMget_name_int_val (elem_id, OMstr_to_name("x"), &X);
	if (stat != 1) {
		ERR_RETURN("cannot get X");
	}
	stat = OMget_name_int_val (elem_id, OMstr_to_name("y"), &Y);
	if (stat != 1) {
		ERR_RETURN("cannot get Y");
	}
	stat = OMget_name_int_val (elem_id, OMstr_to_name("z"), &Z);
	if (stat != 1) {
		ERR_RETURN("cannot get Z");
	}
	stat = OMget_name_int_val (elem_id, OMstr_to_name("flip_x"), &Flip_X);
	if (stat != 1) {
		ERR_RETURN("cannot get Flip_X");
	}
	stat = OMget_name_int_val (elem_id, OMstr_to_name("flip_y"), &Flip_Y);
	if (stat != 1) {
		ERR_RETURN("cannot get Flip_Y");
	}
	stat = OMget_name_int_val (elem_id, OMstr_to_name("flip_z"), &Flip_Z);
	if (stat != 1) {
		ERR_RETURN("cannot get Flip_Z");
	}
	stat = OMget_name_int_val (elem_id, OMstr_to_name("draw_sides"), &Draw_Sides);
	if (stat != 1) {
		ERR_RETURN("cannot get Draw_Sides");
	}
	if (FUNCexcavate_brick3d( in, out, X, Y, Z, Flip_X, Flip_Y, Flip_Z, Draw_Sides))
		return(1);
	else
		return(0);
}


#define FX 1
#define FY 2
#define FZ 4

/* 64-bit porting. Only Modified Internally */
int FUNCexcavate_brick3d( in, out, X, Y, Z, Flip_X, Flip_Y, Flip_Z, Draw_Sides )
    OMobj_id in;
    OMobj_id out;
    int X, Y, Z, Flip_X, Flip_Y, Flip_Z, Draw_Sides;
{
	float *xyz, x, y, z;
	float perx, pery, perz;
	float x1, y1, z1;
	float x2, y2, z2;
	xp_long *dims, size;
	int mode, dims_size;
	OBJ object, *obj;
	OMobj_id cell_set;

	obj = &object;
	obj->nnodes = 0;
	obj->ncells = 0;
	if (FLDget_dims (in, &dims, &dims_size) != 1) {
		ERR_RETURN("Error getting dims");
	}
	if (FLDget_points (in, &xyz, &size, OM_GET_ARRAY_RD) != 1) {
		ERR_RETURN("Error getting points");
	}

	if (FLDset_ncell_sets (out, 0) != 1) {
		ERR_RETURN("Error setting ncell_sets");
	}

	x = xyz[0] + (xyz[3] - xyz[0])*X/(dims[0] - 1.);
	y = xyz[1] + (xyz[4] - xyz[1])*Y/(dims[1] - 1.);
	z = xyz[2] + (xyz[5] - xyz[2])*Z/(dims[2] - 1.);

	x1 = xyz[0];
	x2 = xyz[3];

	y1 = xyz[1];
	y2 = xyz[4];

	z1 = xyz[2];
	z2 = xyz[5];

	ARRfree(xyz);

	perx = (x2-x1 == 0.0) ? 1.0 : (x-x1) / (x2-x1);
	pery = (y2-y1 == 0.0) ? 1.0 : (y-y1) / (y2-y1);
	perz = (z2-z1 == 0.0) ? 1.0 : (z-z1) / (z2-z1);

	mode = Flip_X + (Flip_Y * 2) + (Flip_Z * 4);
	switch (mode) {
	      case 0:
		create_square(obj, x, x, y, y2, z, z2,  perx, perx, pery, 1.0, perz, 1.0, 1);
		create_square(obj, x, x2, y, y, z, z2,  perx, 1.0, pery, pery, perz, 1.0, 1);
		create_square(obj, x, x2, y, y2, z, z,  perx, 1.0, pery, 1.0, perz, perz, 1);

		if (Draw_Sides) {
			create_square(obj, x1, x1, y1, y2, z1, z2,  0.0, 0.0, 0.0, 1.0, 0.0, 1.0, -1);
			create_square(obj, x1, x2, y1, y1, z1, z2,  0.0, 1.0, 0.0, 0.0, 0.0, 1.0, -1);
			create_square(obj, x1, x2, y1, y2, z1, z1,  0.0, 1.0, 0.0, 1.0, 0.0, 0.0, -1);
			create_x_L(obj, x2, y1, y, y2, z1, z, z2,   1.0, 0.0, pery, 1.0, 0.0, perz, 1.0, 1);
			create_y_L(obj, x1, x, x2, y2, z1, z, z2,   0.0, perx, 1.0, 1.0, 0.0, perz, 1.0, 1);
			create_z_L(obj, x1, x, x2, y1, y, y2, z2,   0.0, perx, 1.0, 0.0, pery, 1.0, 1.0, 1);
		}
		break;

	      case FX:
		create_square(obj, x, x, y, y2, z, z2,  perx, perx, pery, 1.0, perz, 1.0, -1);
		create_square(obj, x1, x, y, y, z, z2,  0.0, perx, pery, pery, perz, 1.0, 1);
		create_square(obj, x1, x, y, y2, z, z,  0.0, perx, pery, 1.0, perz, perz, 1);

		if (Draw_Sides) {
			create_square(obj, x2, x2, y1, y2, z1, z2,  1.0, 1.0, 0.0, 1.0, 0.0, 1.0, 1);
			create_square(obj, x1, x2, y1, y1, z1, z2,  0.0, 1.0, 0.0, 0.0, 0.0, 1.0, -1);
			create_square(obj, x1, x2, y1, y2, z1, z1,  0.0, 1.0, 0.0, 1.0, 0.0, 0.0, -1);
			create_x_L(obj, x1, y1, y, y2, z1, z, z2,   0.0, 0.0, pery, 1.0, 0.0, perz, 1.0, -1);
			create_y_L(obj, x2, x, x1, y2, z1, z, z2,   1.0, perx, 0.0, 1.0, 0.0, perz, 1.0, 1);
			create_z_L(obj, x2, x, x1, y1, y, y2, z2,   1.0, perx, 0.0, 0.0, pery, 1.0, 1.0, 1);
		}
		break;

	      case FY:
		create_square(obj, x, x, y1, y, z, z2,  perx, perx, 0.0, pery, perz, 1.0, 1);
		create_square(obj, x, x2, y, y, z, z2,  perx, 1.0, pery, pery, perz, 1.0, -1);
		create_square(obj, x, x2, y1, y, z, z,  perx, 1.0, 0.0, pery, perz, perz, 1);

		if (Draw_Sides) {
			create_square(obj, x1, x1, y1, y2, z1, z2,  0.0, 0.0, 0.0, 1.0, 0.0, 1.0, -1);
			create_square(obj, x1, x2, y2, y2, z1, z2,  0.0, 1.0, 1.0, 1.0, 0.0, 1.0, 1);
			create_square(obj, x1, x2, y1, y2, z1, z1,  0.0, 1.0, 0.0, 1.0, 0.0, 0.0, -1);
			create_x_L(obj, x2, y2, y, y1, z1, z, z2,   1.0, 1.0, pery, 0.0, 0.0, perz, 1.0, 1);
			create_y_L(obj, x1, x, x2, y1, z1, z, z2,   0.0, perx, 1.0, 0.0, 0.0, perz, 1.0, -1);
			create_z_L(obj, x1, x, x2, y2, y, y1, z2,   0.0, perx, 1.0, 1.0, pery, 0.0, 1.0, 1);
		}
		break;

	      case FZ:
		create_square(obj, x, x, y, y2, z1, z,  perx, perx, pery, 1.0, 0.0, perz, 1);
		create_square(obj, x, x2, y, y, z1, z,  perx, 1.0, pery, pery, 0.0, perz, 1);
		create_square(obj, x, x2, y, y2, z, z,  perx, 1.0, pery, 1.0, perz, perz, -1);

		if (Draw_Sides) {
			create_square(obj, x1, x1, y1, y2, z1, z2,  0.0, 0.0, 0.0, 1.0, 0.0, 1.0, -1);
			create_square(obj, x1, x2, y1, y1, z1, z2,  0.0, 1.0, 0.0, 0.0, 0.0, 1.0, -1);
			create_square(obj, x1, x2, y1, y2, z2, z2,  0.0, 1.0, 0.0, 1.0, 1.0, 1.0, 1);
			create_x_L(obj, x2, y1, y, y2, z2, z, z1,   1.0, 0.0, pery, 1.0, 1.0, perz, 0.0, 1);
			create_y_L(obj, x1, x, x2, y2, z2, z, z1,   0.0, perx, 1.0, 1.0, 1.0, perz, 0.0, 1);
			create_z_L(obj, x1, x, x2, y1, y, y2, z1,   0.0, perx, 1.0, 0.0, pery, 1.0, 0.0, -1);
		}
		break;

	      case FX | FY:
		create_square(obj, x, x, y1, y, z, z2,  perx, perx, 0.0, pery, perz, 1.0, -1);
		create_square(obj, x1, x, y, y, z, z2,  0.0, perx, pery, pery, perz, 1.0, -1);
		create_square(obj, x1, x, y1, y, z, z,  0.0, perx, 0.0, pery, perz, perz, 1);

		if (Draw_Sides) {
			create_square(obj, x2, x2, y1, y2, z1, z2,  1.0, 1.0, 0.0, 1.0, 0.0, 1.0, 1);
			create_square(obj, x1, x2, y2, y2, z1, z2,  0.0, 1.0, 1.0, 1.0, 0.0, 1.0, 1);
			create_square(obj, x1, x2, y1, y2, z1, z1,  0.0, 1.0, 0.0, 1.0, 0.0, 0.0, -1);
			create_x_L(obj, x1, y2, y, y1, z1, z, z2,   0.0, 1.0, pery, 0.0, 0.0, perz, 1.0, -1);
			create_y_L(obj, x2, x, x1, y1, z1, z, z2,   1.0, perx, 0.0, 0.0, 0.0, perz, 1.0, -1);
			create_z_L(obj, x2, x, x1, y2, y, y1, z2,   1.0, perx, 0.0, 1.0, pery, 0.0, 1.0, 1);
		}
		break;

	      case FX | FZ:
		create_square(obj, x, x, y, y2, z1, z,  perx, perx, pery, 1.0, 0.0, perz, -1);
		create_square(obj, x1, x, y, y, z1, z,  0.0, perx, pery, pery, 0.0, perz, 1);
		create_square(obj, x1, x, y, y2, z, z,  0.0, perx, pery, 1.0, perz, perz, -1);

		if (Draw_Sides) {
			create_square(obj, x2, x2, y1, y2, z1, z2,  1.0, 1.0, 0.0, 1.0, 0.0, 1.0, 1);
			create_square(obj, x1, x2, y1, y1, z1, z2,  0.0, 1.0, 0.0, 0.0, 0.0, 1.0, -1);
			create_square(obj, x1, x2, y1, y2, z2, z2,  0.0, 1.0, 0.0, 1.0, 1.0, 1.0, 1);
			create_x_L(obj, x1, y1, y, y2, z2, z, z1,   0.0, 0.0, pery, 1.0, 1.0, perz, 0.0, -1);
			create_y_L(obj, x2, x, x1, y2, z2, z, z1,   1.0, perx, 0.0, 1.0, 1.0, perz, 0.0, 1);
			create_z_L(obj, x2, x, x1, y1, y, y2, z1,   1.0, perx, 0.0, 0.0, pery, 1.0, 0.0, -1);
		}
		break;

	      case FY | FZ:
		create_square(obj, x, x, y1, y, z1, z,  perx, perx, 0.0, pery, 0.0, perz, 1);
		create_square(obj, x, x2, y, y, z1, z,  perx, 1.0, pery, pery, 0.0, perz, -1);
		create_square(obj, x, x2, y1, y, z, z,  perx, 1.0, 0.0, pery, perz, perz, -1);

		if (Draw_Sides) {
			create_square(obj, x1, x1, y1, y2, z1, z2,  0.0, 0.0, 0.0, 1.0, 0.0, 1.0, -1);
			create_square(obj, x1, x2, y2, y2, z1, z2,  0.0, 1.0, 1.0, 1.0, 0.0, 1.0, 1);
			create_square(obj, x1, x2, y1, y2, z2, z2,  0.0, 1.0, 0.0, 1.0, 1.0, 1.0, 1);
			create_x_L(obj, x2, y2, y, y1, z2, z, z1,   1.0, 1.0, pery, 0.0, 1.0, perz, 0.0, 1);
			create_y_L(obj, x1, x, x2, y1, z2, z, z1,   0.0, perx, 1.0, 0.0, 1.0, perz, 0.0, -1);
			create_z_L(obj, x1, x, x2, y2, y, y1, z1,   0.0, perx, 1.0, 1.0, pery, 0.0, 0.0, -1);
		}
		break;

	      case FX | FY | FZ:
		create_square(obj, x, x, y1, y, z1, z,  perx, perx, 0.0, pery, 0.0, perz, -1);
		create_square(obj, x1, x, y, y, z1, z,  0.0, perx, pery, pery, 0.0, perz, -1);
		create_square(obj, x1, x, y1, y, z, z,  0.0, perx, 0.0, pery, perz, perz, -1);

		if (Draw_Sides) {
			create_square(obj, x2, x2, y1, y2, z1, z2,  1.0, 1.0, 0.0, 1.0, 0.0, 1.0, 1);
			create_square(obj, x1, x2, y2, y2, z1, z2,  0.0, 1.0, 1.0, 1.0, 0.0, 1.0, 1);
			create_square(obj, x1, x2, y1, y2, z2, z2,  0.0, 1.0, 0.0, 1.0, 1.0, 1.0, 1);
			create_x_L(obj, x1, y2, y, y1, z2, z, z1,   0.0, 1.0, pery, 0.0, 1.0, perz, 0.0, -1);
			create_y_L(obj, x2, x, x1, y1, z2, z, z1,   1.0, perx, 0.0, 0.0, 1.0, perz, 0.0, -1);
			create_z_L(obj, x2, x, x1, y2, y, y1, z1,   1.0, perx, 0.0, 1.0, pery, 0.0, 0.0, -1);
		}
		break;
	}

	/* Create output */
	if (FLDset_nspace (out, 3) != 1) {
		ERR_RETURN("Error setting nspace");
	}
	if (FLDset_nnodes (out, object.nnodes) != 1) {
		ERR_RETURN("Error setting nnodes");
	}
	if (FLDset_coord(out, (float *)object.coord, 3*object.nnodes, OM_SET_ARRAY_COPY) != 1) {
		ERR_RETURN("Error setting coord");
	}
	if (FLDset_ncell_sets (out, 0) != 1) {
		ERR_RETURN("Error setting ncell_sets");
	}
	if (FLDadd_cell_set(out, "Quad") != 1) {
		ERR_RETURN("Error setting cell type");
	}
	if (FLDget_cell_set(out, 0, &cell_set) != 1) {
		ERR_RETURN("Error getting cell set");
	}
	if (FLDset_ncells(cell_set, object.ncells) != 1) {
		ERR_RETURN("Error setting ncells");
	}
	if (FLDset_node_connect(cell_set, object.connect, 4*object.ncells,
				OM_SET_ARRAY_COPY) != 1) {
		ERR_RETURN("Error setting cell connect list");
	}

	if (FLDset_node_data_ncomp (out, 2) != 1) {
		ERR_RETURN("Error setting nnode_data");
	}
	if (FLDset_node_data_comp (out, 0, 3, "uvw", "uvw") != 1) {
		ERR_RETURN("Error setting node component");
	}
	if (FLDset_node_data(out, 0, (char *)object.uvws, DTYPE_FLOAT, 
			     3*object.nnodes, OM_SET_ARRAY_COPY) != 1) {
		ERR_RETURN("Error setting node data");
	}
	if (FLDset_node_data_id(out, 0, 670) != 1) {
		ERR_RETURN("Error setting node data id");
	}
	if (FLDset_node_data_comp (out, 1, 3, "normals", "normals") != 1) {
		ERR_RETURN("Error setting node component");
	}
	if (FLDset_node_data(out, 1, (char *)object.norm, DTYPE_FLOAT, 
			     3*object.nnodes, OM_SET_ARRAY_COPY) != 1) {
		ERR_RETURN("Error setting node data");
	}
	if (FLDset_node_data_id(out, 1, 666) != 1) {
		ERR_RETURN("Error setting node data id");
	}
	ARRfree(dims);
	return(1);
}

static int create_square( OBJ *obj, 
	float x1, float x2, float y1, float y2, float z1, float z2,
	float u1, float u2, float v1, float v2, float w1, float w2, int N)
{
	float v[4][3], uvw[4][3], norm[4][3];
	int i, j;

	for(i=0; i<4; i++)
		norm[i][0] = norm[i][1] = norm[i][2] = 0.0;

	/* assign x coordinates */
	if ( y1 == y2 ) {
		v[0][0] = v[3][0] = x1;
		v[1][0] = v[2][0] = x2;
		uvw[0][0] = uvw[3][0] = u1;
		uvw[1][0] = uvw[2][0] = u2;
		for (i=0; i<4; i++)
			norm[i][1] = (float)N;
	}
	else {
		if (x1 == x2)
			for (i=0; i<4; i++)
				norm[i][0] = (float)N;
		if (z1 == z2)
			for (i=0; i<4; i++)
				norm[i][2] = (float)N;
		v[0][0] = v[1][0] = x1;
		v[2][0] = v[3][0] = x2;
		uvw[0][0] = uvw[1][0] = u1;
		uvw[2][0] = uvw[3][0] = u2;
	}

	/* assign y coordinates */
	v[0][1] = v[3][1] = y1;
	v[1][1] = v[2][1] = y2;
	uvw[0][1] = uvw[3][1] = v1;
	uvw[1][1] = uvw[2][1] = v2;

	/* assign z coordinates */
	v[0][2] = v[1][2] = z1;
	v[2][2] = v[3][2] = z2;
	uvw[0][2] = uvw[1][2] = w1;
	uvw[2][2] = uvw[3][2] = w2;

	for (i=0; i<4; i++) {
		for (j=0; j<3; j++) {
			obj->uvws[obj->nnodes][j] = uvw[i][j];
			obj->coord[obj->nnodes][j] = v[i][j];
			obj->norm[obj->nnodes][j] = norm[i][j];
		}
		obj->connect[obj->ncells*4+i] = obj->nnodes;
		(obj->nnodes)++;
	}
	(obj->ncells)++;
	return(1);
}

static int create_x_L(OBJ *obj, 
	float x, float y1, float y, float y2, float z1, float z, 
	float z2, float u, float v1, float v, float v2, float w1, 
	float w, float w2, int N)
{
	float verts[4][3], uvw[4][3], norm[4][3];
	int i, j;

	for(i=0; i<4; i++) {
		norm[i][0] = (float)N;
		norm[i][1] = norm[i][2] = 0.0;
		verts[i][0] = x;
		uvw[i][0] = u;
	}

	/* assign y coordinates */
	verts[0][1] = verts[3][1] = y1;
	verts[1][1] = verts[2][1] = y;
	uvw[0][1] = uvw[3][1] = v1;
	uvw[1][1] = uvw[2][1] = v;

	/* assign z coordinates */
	verts[0][2] = z1;
	verts[1][2] = z;
	verts[2][2] = verts[3][2] = z2;
	uvw[0][2] = w1;
	uvw[1][2] = w;
	uvw[2][2] = uvw[3][2] = w2;

	for (i=0; i<4; i++) {
		for (j=0; j<3; j++) {
			obj->uvws[obj->nnodes][j] = uvw[i][j];
			obj->coord[obj->nnodes][j] = verts[i][j];
			obj->norm[obj->nnodes][j] = norm[i][j];
		}
		obj->connect[obj->ncells*4+i] = obj->nnodes;
		(obj->nnodes)++;
	}
	(obj->ncells)++;

	/* assign y coordinates */
	verts[0][1] = y1;
	verts[1][1] = y;
	verts[2][1] = verts[3][1] = y2;
	uvw[0][1] = v1;
	uvw[1][1] = v;
	uvw[2][1] = uvw[3][1] = v2;

	/* assign z coordinates */
	verts[0][2] = verts[3][2] = z1;
	verts[1][2] = verts[2][2] = z;
	uvw[0][2] = uvw[3][2] = w1;
	uvw[1][2] = uvw[2][2] = w;

	for (i=0; i<4; i++) {
		for (j=0; j<3; j++) {
			obj->uvws[obj->nnodes][j] = uvw[i][j];
			obj->coord[obj->nnodes][j] = verts[i][j];
			obj->norm[obj->nnodes][j] = norm[i][j];
		}
		obj->connect[obj->ncells*4+i] = obj->nnodes;
		(obj->nnodes)++;
	}
	(obj->ncells)++;
	return(1);
}

static int create_y_L(OBJ *obj, 
	float x1, float x, float x2, float y, float z1, float z, 
	float z2, float u1, float u, float u2, float v, float w1, 
	float w, float w2, int N)
{
	float verts[4][3], uvw[4][3], norm[4][3];
	int i, j;

	for(i=0; i<4; i++) {
		norm[i][1] = (float)N;
		norm[i][0] = norm[i][2] = 0.0;
		verts[i][1] = y;
		uvw[i][1] = v;
	}

	/* assign x coordinates */
	verts[0][0] = verts[3][0] = x1;
	verts[1][0] = verts[2][0] = x;
	uvw[0][0] = uvw[3][0] = u1;
	uvw[1][0] = uvw[2][0] = u;

	/* assign z coordinates */
	verts[0][2] = z1;
	verts[1][2] = z;
	verts[2][2] = verts[3][2] = z2;
	uvw[0][2] = w1;
	uvw[1][2] = w;
	uvw[2][2] = uvw[3][2] = w2;

	for (i=0; i<4; i++) {
		for (j=0; j<3; j++) {
			obj->uvws[obj->nnodes][j] = uvw[i][j];
			obj->coord[obj->nnodes][j] = verts[i][j];
			obj->norm[obj->nnodes][j] = norm[i][j];
		}
		obj->connect[obj->ncells*4+i] = obj->nnodes;
		(obj->nnodes)++;
	}
	(obj->ncells)++;

	/* assign x coordinates */
	verts[0][0] = x1;
	verts[1][0] = x;
	verts[2][0] = verts[3][0] = x2;
	uvw[0][0] = u1;
	uvw[1][0] = u;
	uvw[2][0] = uvw[3][0] = u2;

	/* assign z coordinates */
	verts[0][2] = verts[3][2] = z1;
	verts[1][2] = verts[2][2] = z;
	uvw[0][2] = uvw[3][2] = w1;
	uvw[1][2] = uvw[2][2] = w;

	for (i=0; i<4; i++) {
		for (j=0; j<3; j++) {
			obj->uvws[obj->nnodes][j] = uvw[i][j];
			obj->coord[obj->nnodes][j] = verts[i][j];
			obj->norm[obj->nnodes][j] = norm[i][j];
		}
		obj->connect[obj->ncells*4+i] = obj->nnodes;
		(obj->nnodes)++;
	}
	(obj->ncells)++;
	return(1);
}

static int create_z_L(OBJ *obj, 
	float x1, float x, float x2, float y1, float y, float y2, 
	float z,  float u1, float u, float u2, 
	float v1, float v, float v2, float w, int N)
{
	float verts[4][3], uvw[4][3], norm[4][3];
	int i, j;

	for(i=0; i<4; i++) {
		norm[i][2] = (float)N;
		norm[i][0] = norm[i][1] = 0.0;
		verts[i][2] = z;
		uvw[i][2] = w;
	}

	/* assign y coordinates */
	verts[0][1] = verts[3][1] = y1;
	verts[1][1] = verts[2][1] = y;
	uvw[0][1] = uvw[3][1] = v1;
	uvw[1][1] = uvw[2][1] = v;

	/* assign x coordinates */
	verts[0][0] = x1;
	verts[1][0] = x;
	verts[2][0] = verts[3][0] = x2;
	uvw[0][0] = u1;
	uvw[1][0] = u;
	uvw[2][0] = uvw[3][0] = u2;

	for (i=0; i<4; i++) {
		for (j=0; j<3; j++) {
			obj->uvws[obj->nnodes][j] = uvw[i][j];
			obj->coord[obj->nnodes][j] = verts[i][j];
			obj->norm[obj->nnodes][j] = norm[i][j];
		}
		obj->connect[obj->ncells*4+i] = obj->nnodes;
		(obj->nnodes)++;
	}
	(obj->ncells)++;

	/* assign y coordinates */
	verts[0][1] = y1;
	verts[1][1] = y;
	verts[2][1] = verts[3][1] = y2;
	uvw[0][1] = v1;
	uvw[1][1] = v;
	uvw[2][1] = uvw[3][1] = v2;

	/* assign x coordinates */
	verts[0][0] = verts[3][0] = x1;
	verts[1][0] = verts[2][0] = x;
	uvw[0][0] = uvw[3][0] = u1;
	uvw[1][0] = uvw[2][0] = u;

	for (i=0; i<4; i++) {
		for (j=0; j<3; j++) {
			obj->uvws[obj->nnodes][j] = uvw[i][j];
			obj->coord[obj->nnodes][j] = verts[i][j];
			obj->norm[obj->nnodes][j] = norm[i][j];
		}
		obj->connect[obj->ncells*4+i] = obj->nnodes;
		(obj->nnodes)++;
	}
	(obj->ncells)++;
	return(1);
}

