/*
			Copyright (c) 1994 by
			Advanced Visual Systems Inc.
			All Rights Reserved

	This software comprises unpublished confidential information of
	Advanced Visual Systems Inc. and may not be used, copied or made
	available to anyone, except in accordance with the license
	under which it is furnished.

	This file is under Perforce control
	$Id: //depot/express/fcs70/modules/ext_face.c#1 $
*/

#define XP_WIDE_API	/* Use Wide APIs */

#include <avs/util.h>
#include <avs/err.h>
#include <avs/om.h>
#include <avs/fld.h>
#include <avs/arr.h>
#include <avs/dv_util.h>

#define ERR_RETURN(A) ERRerror("ext_face", 0, ERR_ORIG, A); return(0);
#define MAX_CELL_SETS 256
#define MAX_LABEL_SIZE  1024
#define STATUS_REPORT   256

static char *cell_name[]= {"", "Point", "Line", "Tri", "Quad", "", "Tri2", "", "Quad2"};

int FUNCext_face (OMobj_id in, OMobj_id out_sets, OMobj_id out);

int DVext_face_update(OMobj_id elem_id)
{
	OMobj_id in, out, out_sets;
	in = OMfind_subobj(elem_id, OMstr_to_name("in"), OM_OBJ_RD);
	out = OMfind_subobj(elem_id, OMstr_to_name("out_cells"), OM_OBJ_RW);
	out_sets = OMfind_subobj(elem_id, OMstr_to_name("out_sets"), OM_OBJ_RW);

	if (FUNCext_face(in, out_sets, out)) {
		return(1);
	}
	else return(0);
}

/* 64-bit porting. Only Modified Internally */
int FUNCext_face (OMobj_id in, OMobj_id out_sets, OMobj_id out)
{
	int   cs, os, nc, nsets, cell_nnodes, stat;
	xp_long ncells, nnodes;
	int   i, dtype, veclen, ndim, ncomp, null_flag;
	xp_long size, *node_list, *face_node_list;
	int   cell_nfaces, fnl_size;
	int   *face_nnodes, *face_corner_nnodes;
	int   out_nsets, out_cell_nnodes[MAX_CELL_SETS];
	xp_long out_ncells[MAX_CELL_SETS], *out_cells[MAX_CELL_SETS], 
                *out_lists[MAX_CELL_SETS], out_list_sizes[MAX_CELL_SETS];
	char  *cell_data, *out_cell_data;
	OMobj_id   cell_set, sets_id, out_set;
	char   label[MAX_LABEL_SIZE], units[MAX_LABEL_SIZE];
	double null_value, min, max;

	if (FLDset_ncell_sets (out_sets, 0) != 1) {
		ERR_RETURN("Error setting ncell_sets");
	}
	sets_id = OMfind_subobj(out, OMstr_to_name("cell_set"), OM_OBJ_RW);
	if (OMis_null_obj(sets_id)) {
		ERR_RETURN("cannot get cell sets");
	}
	OMset_obj_ref(sets_id, OMnull_obj, 0);

	if (FLDget_nnodes(in, &nnodes) != 1 || nnodes == 0) {
		return(0);
	}

	if (FLDget_ncell_sets(in, &nsets) != 1) {
		ERR_RETURN("cannot get nsets");
	}

	nc=0;

	for (cs=0; cs<nsets; cs++) {
		if (FLDget_cell_set(in, cs, &cell_set) != 1) {
			ERR_RETURN("cannot get cell set");
		}
		if (FLDget_cell_ndim(cell_set, &ndim) != 1) {
			ERR_RETURN("cannot get ndim");
		}

		if (ndim < 3) {
			stat = OMadd_obj_ref(sets_id, cell_set, 0);
			if (stat != 1) {
				ERR_RETURN("cannot reference cell set");
			}
			continue;
		}
		if (FLDget_ncells(cell_set, &ncells) != 1) {
			ERR_RETURN("cannot get ncells");
		}

		if (FLDget_cell_set_nnodes(cell_set,  &cell_nnodes) != 1) {
			ERR_RETURN("cannot get cell nnodes");
		}

		if (FLDget_node_connect(cell_set, &node_list, 
					&size, OM_GET_ARRAY_RD) != 1) {
			ERR_RETURN("cannot get cell connectivity");
		}

		if (FLDget_cell_faces(cell_set, &cell_nfaces, &face_nnodes, 
				      &face_corner_nnodes, &face_node_list, &fnl_size) != 1) {
			ERR_RETURN("cannot get cell nfaces");
		}
		if (FLDget_cell_data_ncomp (cell_set, &ncomp) != 1) {
			ncomp = 0;
		}
		/*----------------------------------*/
		/*   GET EXTERNAL faces             */
		/*----------------------------------*/
		OMpush_status_range((int)(cs*100/nsets),
				    (int)((cs+1)*100/nsets));

		stat = UTILexternal_faces(nnodes, ncells, cell_nnodes, node_list, 
					  cell_nfaces, face_nnodes, face_corner_nnodes, face_node_list,
					  MAX_CELL_SETS, &out_nsets, out_ncells, 
					  out_cell_nnodes, out_cells,
					  out_lists, out_list_sizes);
		OMpop_status_range();
		if (stat != 1) {
			ARRfree(node_list);
			ARRfree(face_nnodes);
			ARRfree(face_corner_nnodes);
			ARRfree(face_node_list);
			for (os=0; os<out_nsets; os++) {
				if (out_cells[os]) {
					ARRfree(out_cells[os]);
				}
			}
			if (FLDset_ncell_sets (out_sets, 0) != 1) {
				ERR_RETURN("Error setting ncell_sets");
			}
			ERR_RETURN("faces are not extracted");
		}
		ARRfree(node_list);
		ARRfree(face_nnodes);
		ARRfree(face_corner_nnodes);
		ARRfree(face_node_list);

		/*-----------------------------------*/
		/*  ADD new 2D cell sets             */
		/*-----------------------------------*/
		for (os=0; os<out_nsets; os++) {
			if (out_ncells[os] == 0 || out_list_sizes[os] == 0 || !out_lists[os]) {
				if (out_cells[os]) {
					ARRfree(out_cells[os]);
				}
				continue;
			}
			if (FLDadd_cell_set(out_sets, cell_name[out_cell_nnodes[os]]) != 1) {
				ERR_RETURN("Error setting cell type");
			}
			if (FLDget_cell_set(out_sets, nc, &out_set) != 1) {
				ERR_RETURN("Error getting cell set");
			}
			if (ncomp) {
				if (FLDset_cell_set(out_set, "Cell_Data") != 1) {
					ERR_RETURN("Error setting cell type");
				}
			}
			if (FLDget_cell_set(out_sets, nc++, &out_set) != 1) {
				ERR_RETURN("Error getting cell set");
			}
			if (FLDset_ncells(out_set, out_ncells[os]) != 1) {
				ERR_RETURN("Error setting ncells");
			}

			if (FLDset_node_connect(out_set, out_lists[os], out_list_sizes[os],
						    OM_SET_ARRAY_FREE) != 1) {
				ERR_RETURN("Error setting cell connect list");
			}
			if (ncomp) {
				if (FLDset_cell_data_ncomp (out_set, ncomp) != 1) {
					ERR_RETURN("Error setting cell data ncomp");
				}
			}
			for (i=0; i< ncomp; i++) {
				if (FLDget_cell_data_veclen(cell_set, i, &veclen) != 1) {
					ERR_RETURN("Error getting cell veclen");
				}
				stat = FLDget_cell_data_label(cell_set, i, label, MAX_LABEL_SIZE);
				if (stat < 0)
					return(stat);
				else if (stat == 0)
					strcpy(label, "");
				stat = FLDget_cell_data_units(cell_set, i, units, MAX_LABEL_SIZE);
				if (stat < 0)
					return(stat);
				else if (stat == 0)
					strcpy(units, "");
				if (FLDset_cell_data_comp (out_set, i, veclen, label, units) != 1) {
					ERR_RETURN("Error setting mode data component");
				}
				if (FLDget_cell_data(cell_set, i, &dtype, &cell_data, 
						     &size, OM_GET_ARRAY_RD) != 1) {
					ERR_RETURN("Error setting cell data");
				}
				if (FLDget_cell_data(out_set, i, &dtype, &out_cell_data, 
						     &size, OM_GET_ARRAY_WR) != 1) {
					ERR_RETURN("Error setting cell data");
				}
				UTILcopy_array_list(ncells, veclen, dtype, cell_data, out_ncells[os],
						    out_cells[os], out_cell_data);

				if (FLDget_cell_null_data(cell_set, i, &null_flag, (char *)&null_value) != 1) {
					ERR_RETURN("cannot get null data");
				}
				if (null_flag) {
					if (FLDset_cell_null_data(out_set, i, (char *)&null_value, dtype) != 1) {
						ERR_RETURN("Error setting null value");
					}
				}
				else {
					if (FLDset_cell_null_flag(out_set, i, 0) != 1) {
						ERR_RETURN("Error setting null flag");
					}
				}
				if (FLDget_cell_data_minmax(cell_set, i, 
							    (char *)&min, (char *)&max) != 1) {
					ERR_RETURN("Error getting node minmax");
				}
				if (FLDset_cell_data_minmax(out_set, i, 
							    (char *)&min, (char *)&max, dtype) != 1) {
					ERR_RETURN("Error setting cell minmax");
				}
				if (cell_data)
					ARRfree(cell_data);
				if (out_cell_data)
					ARRfree(out_cell_data);
			}
			stat = OMadd_obj_ref(sets_id, out_set, 0);
			if (stat != 1) {
				ERR_RETURN("cannot reference cell set");
			}

			if (out_cells[os]) {
				ARRfree(out_cells[os]);
			}
		}
	}
	return(1);
}

