/*
			Copyright (c) 1996 by
			Advanced Visual Systems Inc.
			All Rights Reserved

	This software comprises unpublished confidential information of
	Advanced Visual Systems Inc. and may not be used, copied or made
	available to anyone, except in accordance with the license
	under which it is furnished.

	This file is under Perforce control
	$Id: //depot/express/fcs70/modules/image/libavsx/iface.c#1 $
*/

#include <stdlib.h>

#include "avsx.h"

static void avsx_init (avsx_t *);
static int file_input (avsx_t *);
static int file_output (avsx_t *);

/*-------------------------------------------------------------------------*/
void *avsOpen(char *filename, int access)
{
  avsx_t *avsx;

  if (!(avsx = (avsx_t *) malloc(sizeof(avsx_t))))
  {
    ERRerror("avsOpen", 0, ERR_ORIG, "Failure allocating avsx struct");
    return(NULL);
  }

  avsx_init(avsx);

  if (access & DV_IMAGE_FILE_ACCESS_READ)
  {
    if ((avsx->fp = (FILE *) FILEfopen(filename, SIO_R_BIN)) == NULL)
    {
      ERRerror("avsOpen", 0, ERR_ORIG, "Cannot open file for input");
      free(avsx);
      return(NULL);
    }

    /* go read the file now */
    if (!file_input(avsx))
    {
      fclose(avsx->fp);
      free(avsx);
      return(NULL);
    }
  }
  else if (access & DV_IMAGE_FILE_ACCESS_WRITE)
  {
    if ((avsx->fp = (FILE *) FILEfopen(filename, SIO_W_BIN)) == NULL)
    {
      ERRerror("avsOpen", 0, ERR_ORIG, "Cannot open file for output");
      free(avsx);
      return(NULL);
    }
  }
  else
  {
    ERRerror("avsOpen", 0, ERR_ORIG, "File access type is invalid");
    free(avsx);
    return(NULL);
  }

  avsx->access = access;

  return((void *) avsx);
}

/*-------------------------------------------------------------------------*/
void avsClose(void *avsx)
{
  if (avsx)
  {
    avsx_t *xt = (avsx_t *) avsx;

    /* if writing a file do it on close */
    if (xt->access & DV_IMAGE_FILE_ACCESS_WRITE)
    {
      if (file_output(xt) == avsxFAILURE) {
        ERRerror("avsClose", 0, ERR_ORIG, "Error while writing AVS.x file");
      }

      /* set this NULL since the caller alloc'd it and should free it */
      xt->image = NULL;
    }

    if (xt->image != NULL)
      free(xt->image);

    if (xt->fp != NULL)
      fclose(xt->fp);

    free(xt);
  }
}

/*-------------------------------------------------------------------------*/
static void avsx_init(avsx_t *avsx)
{
  avsx->access = DV_IMAGE_FILE_ACCESS_UNSET;
  avsx->width = avsx->height = 0;
  avsx->colortype = DV_IMAGE_COLORTYPE_RGB;
  avsx->flip = 0;
  avsx->image = NULL;
}

/*----------------------  G E T  F U N C T I O N S  -----------------------*/

/*-------------------------------------------------------------------------*/
int avsGetWidth(void *avsx, int *width)
{
  avsx_t *xt = (avsx_t *) avsx;

  *width = xt->width;

  return(avsxSUCCESS);
}

/*-------------------------------------------------------------------------*/
int avsGetHeight(void *avsx, int *height)
{
  avsx_t *xt = (avsx_t *) avsx;

  *height = xt->height;

  return(avsxSUCCESS);
}

/*-------------------------------------------------------------------------*/
int avsGetARGBImage(void *avsx, unsigned char **data)
{
  avsx_t *xt = (avsx_t *) avsx;

  *data = xt->image;

  return(avsxSUCCESS);
}

/*-------------------------------------------------------------------------*/
static int file_input(avsx_t *avsx)
{
  byte header[8], *line_ptr;
  int line_inc, i, interrupt;
  float per_count;

  if (fread(header, sizeof(header), 1, avsx->fp) != 1)
  {
    ERRerror("avsRead", 0, ERR_ORIG, "Failure reading header");
    return(0);
  }

  avsx->width = (header[2] << 8) + header[3];
  avsx->height = (header[6] << 8) + header[7];

  /* some old image files on little endians may have the bytes reversed */
  if (avsx->width == 0 && avsx->height == 0)
  {
    avsx->width = (header[1] << 8) + header[0];
    avsx->height = (header[5] << 8) + header[4];
  }

  if (avsx->width <= 0 || avsx->width > 32768 ||
      avsx->height <= 0 || avsx->height > 32768)
  {
    ERRerror("avsRead", 2, ERR_ORIG, "Width %d or height %d unreasonable?",
	     avsx->width, avsx->height);
    return(0);
  }

  /* read image into temporary buffer in case we're interrupted (we want
   * the actual number of lines read)
   */
  if ((avsx->image = (byte *) malloc((avsx->width * avsx->height) << 2)) == NULL)
  {
    ERRerror("avsRead", 0, ERR_ORIG, "Failure allocating buffer");
    return(0);
  }

  line_inc = avsx->width << 2;
  per_count = 100.0 / (avsx->height - 1.0);
  interrupt = 0;

  line_ptr = avsx->image;
  for (i = 0; i < avsx->height; i++, line_ptr += line_inc)
  {
    if (!(i & 0xf))
      OMstatus_check((int)(i*per_count), NULL, &interrupt);

    if (interrupt || (fread(line_ptr, 1, line_inc, avsx->fp)) != line_inc)
      break;
  }

  if (i == 0 || (!interrupt && i < avsx->height))
    return(0);

  /* actual number of lines read */
  avsx->height = i;

  return(1);
}

/*----------------------  S E T  F U N C T I O N S  -----------------------*/

/*-------------------------------------------------------------------------*/
int avsSetWidth(void *avsx, int width)
{
  avsx_t *xt = (avsx_t *) avsx;

  if (width <= 0)
  {
    ERRerror("avsSetWidth", 1, ERR_ORIG, "Width (%d) is unreasonable?", width);
    return(avsxFAILURE);
  }

  xt->width = width;

  return(avsxSUCCESS);
}

/*-------------------------------------------------------------------------*/
int avsSetHeight(avsx, height)
void *avsx;
int height;
{
  avsx_t *xt = (avsx_t *) avsx;

  if (height <= 0)
  {
    ERRerror("avsSetHeight", 1, ERR_ORIG, "Height (%d) is unreasonable?", height);
    return(avsxFAILURE);
  }

  xt->height = height;

  return(avsxSUCCESS);
}

/*-------------------------------------------------------------------------*/
int avsSetFileType(void *avsx, int type)
{
  /* N/A, always binary */
  return(avsxSUCCESS);
}

/*-------------------------------------------------------------------------*/
int avsSetDepth(void *avsx, int depth)
{
  /* N/A, always 32 */
  return(avsxSUCCESS);
}

/*-------------------------------------------------------------------------*/
int avsSetColorType(void *avsx, int type)
{
  avsx_t *xt = (avsx_t *) avsx;

  if (type < 0 || type > DV_IMAGE_COLORTYPE_MAX)
  {
    ERRerror("avsSetColorType", 1, ERR_ORIG, "Type (%d) is invalid", type);
    return(avsxFAILURE);
  }

  xt->colortype = type;

  return(avsxSUCCESS);
}

/*-------------------------------------------------------------------------*/
int avsSetCompressionType(void *avsx, int type, int quality)
{
  /* N/A, always uncompressed */
  return(avsxSUCCESS);
}

/*-------------------------------------------------------------------------*/
int avsSetReductionType(void *avsx, int type)
{
  /* N/A, always ARGB */
  return(avsxSUCCESS);
}

/*-------------------------------------------------------------------------*/
int avsSetARGBImage(void *avsx, unsigned char *data, int flip)
{
  avsx_t *xt = (avsx_t *) avsx;

  xt->image = data;
  xt->flip = flip;

  return(avsxSUCCESS);
}

/*-------------------------------------------------------------------------*/
static int file_output(avsx_t *avsx)
{
  int dims[2];
  byte *src;
  int line_inc, i, interrupt;
  float per_count;

  if (!(avsx->access & DV_IMAGE_FILE_ACCESS_WRITE))
    return(avsxFAILURE);

  if (avsx->colortype == DV_IMAGE_COLORTYPE_GREY)
  {
    int grey;

    src = avsx->image;
    for (i = 0; i < avsx->width * avsx->height; i++)
    {
      src++;
      grey =  *src++ * 0.299;
      grey += *src++ * 0.587;
      grey += *src++ * 0.114;
      *--src = grey;  *--src = grey;  *--src = grey;
      src += 3;
    }
  }

#ifdef AVS_LITTLEENDIAN
  {
    byte *s = (byte *) &avsx->width + 3;
    byte *d = (byte *) dims;
    *d++ = *s--;  *d++ = *s--; *d++ = *s--;  *d++ = *s;
    s = (byte *) &avsx->height + 3;
    *d++ = *s--;  *d++ = *s--; *d++ = *s--;  *d = *s;
  }
#else
  dims[0] = avsx->width;  dims[1] = avsx->height;
#endif

  if (fwrite((char *)dims, 8, 1, avsx->fp) != 1)
    return(avsxFAILURE);

  line_inc = avsx->width << 2;
  per_count = 100.0 / (avsx->height - 1.0);
  interrupt = 0;

  /* now write it */
  if (avsx->flip)
  {
    src = avsx->image + (line_inc * (avsx->height - 1));
    for (i = 0; i < avsx->height; i++, src -= line_inc)
    {
      if (!(i & 0xf))
	OMstatus_check((int)(i*per_count), NULL, &interrupt);

      if (interrupt || (fwrite(src, 1, line_inc, avsx->fp) != line_inc))
	break;
    }
  }
  else
  {
    src = avsx->image;
    for (i = 0; i < avsx->height; i++, src += line_inc)
    {
      if (!(i & 0xf))
	OMstatus_check((int)(i*per_count), NULL, &interrupt);

      if (interrupt || (fwrite(src, 1, line_inc, avsx->fp) != line_inc))
	break;
    }
  }

  if (i != avsx->height && !interrupt)
    return(avsxFAILURE);

  return(avsxSUCCESS);
}
