/*
			Copyright (c) 1993 by
			Advanced Visual Systems Inc.
			All Rights Reserved

	This software comprises unpublished confidential information of
	Advanced Visual Systems Inc. and may not be used, copied or made
	available to anyone, except in accordance with the license
	under which it is furnished.

	This file is under Perforce control
	$Id: //depot/express/fcs70/modules/int_cell.c#1 $
*/

#define XP_WIDE_API	/* Use Wide APIs */

#include <avs/util.h>
#include <avs/err.h>
#include <avs/om.h>
#include <avs/fld.h>
#include <avs/arr.h>
#include <avs/mat.h>
#include <avs/dv_util.h>
#include <avs/data_utils.h>

#define ERR_RETURN(A) ERRerror("interp_cell_data", 0, ERR_ORIG, A); return(0);

#define MAX_LABEL_SIZE  1024
#define MAX_NAME_SIZE 1024
#define NPOINTS  1024

int FUNCinterp_cell_data(OMobj_id in, OMobj_id probe, int comp, OMobj_id out);

int DVinterp_cell_data_update(OMobj_id elem_id)
{
	OMobj_id in, out, probe, tmp;
	int   stat, comp;

	in = OMfind_subobj(elem_id, OMstr_to_name("in"), OM_OBJ_RD);
	probe = OMfind_subobj(elem_id, OMstr_to_name("probe"), OM_OBJ_RD);
	out = OMfind_subobj(elem_id, OMstr_to_name("out_nd"), OM_OBJ_RW);

	tmp = OMfind_subobj(elem_id, OMstr_to_name("component"), OM_OBJ_RD);
	stat = OMget_int_val(tmp, &comp);
	if (stat <= 0) {
		ERR_RETURN("cannot get component");
	}

	stat = FUNCinterp_cell_data(in, probe, comp, out);

	return(stat);
}

/* 64-bit porting. Only Modified Internally */
int FUNCinterp_cell_data(OMobj_id in, OMobj_id probe, int comp, OMobj_id out)
{
	int    stat, cs, *sets, cell_ncomp, nspace, first;
	int    dtype, in_dtype, null_flag, in_null_flag, out_null_flag;
	xp_long nnodes, nprobes, size;
	int    nsets, veclen, in_veclen, probe_nspace;
	xp_long count, nleft, nreq, dims[2], min_rng[2], max_rng[2];
	char   label[MAX_LABEL_SIZE], units[MAX_LABEL_SIZE];
	char   **cell_data, *node_data, *out_node_data;
	double null_value, dnull_value, rnull_value, min, max;
	float probe_xyz[3*NPOINTS], xform[4][4], *xfm, in_xform[4][4], *in_xfm, t_xform[4][4];
	OMobj_id cell_set;
	char  *block_table, *mesh_info;

	if (FLDget_nnodes(in, &nnodes) != 1) {
		ERR_RETURN("cannot get nnodes");
	}
	if (nnodes == 0) {
		/* Clean output and return */
		if (FLDset_nnodes (out, 0) != 1) {
			ERR_RETURN("Error setting nnodes");
		}
		if (FLDset_ncell_sets(out, 0) != 1) {
			ERR_RETURN("cannot set nsets");
		}
		if (FLDset_node_data_ncomp (out, 0) != 1) {
			ERR_RETURN("Error setting nnode_data");
		}
		return(1);
	}
	if (FLDget_nspace(in, &nspace) != 1) {
		ERR_RETURN("cannot get nspace");
	}
	if (FLDget_ncell_sets(in, &nsets) != 1) {
		return(0);
	}
	stat = FLDget_mesh_info(in, &mesh_info);
	if (!stat) {
		ERR_RETURN("cannot create field info");
	}
	stat = FLDinterp_init(in, mesh_info, &block_table);
	if (stat != 1) {
		FLDfree_mesh_info(mesh_info);
		ERR_RETURN("no block table created");
	}

	if (FLDget_nnodes(probe, &nprobes) != 1) {
		ERR_RETURN("cannot get nnodes for probe");
	}
	if (FLDget_nspace(probe, &probe_nspace) != 1) {
		ERR_RETURN("cannot get nspace for probe");
	}

	stat = FLDget_xform(probe, (float *)xform);
	if (stat < 0) {
		ERR_RETURN("cannot get xform for probe");
	}
	else if (stat == 0) {
		xfm = (float *)0;
	}
	else if (MATmat_is_identity((float *)xform, 4))
		xfm = (float *)0;
	else
		xfm = (float *)xform;
	stat = FLDget_xform(in, (float *)in_xform);
	if (stat < 0) {
		ERR_RETURN("cannot get xform for field");
	}
	else if (stat == 0) {
		in_xfm = (float *)0;
	}
	else if (MATmat_is_identity((float *)in_xform, 4))
		in_xfm = (float *)0;
	else
		in_xfm = (float *)in_xform;
	if (in_xfm) {
		MATmat4_inverse(t_xform, (Matr4 *)in_xfm);
		if (xfm)
			MATmat4_multiply((Matr4 *)xfm, t_xform, (Matr4 *)xfm);
		else
			xfm = (float *)t_xform;
	}

	if (FLDset_nnodes(out, nprobes) != 1) {
		ERR_RETURN("cannot set nnodes");
	}
	if (FLDset_node_data_ncomp (out, 1) != 1) {
		ERR_RETURN("Error setting nnode_data");
	}

	if (FLDget_cell_data_minmax_total(in, comp,
					  (char *)&min, (char *)&max) != 1) {
		ERR_RETURN("Error getting cell minmax");
	}

	sets = (int *)malloc(nsets*sizeof(int));
	cell_data = (char **)malloc(nsets*sizeof(char *));

	for (first=1,cs=0; cs<nsets; cs++) {
		if (FLDget_cell_set(in, cs, &cell_set) != 1) {
			ERR_RETURN("cannot get cell set");
		}
		if (FLDget_cell_data_ncomp(cell_set, &cell_ncomp) != 1) {
			ERR_RETURN("Error getting ncomp");
		}
		if (comp >= cell_ncomp) {
			ERRerror("interp_cell_data", 2, ERR_ORIG, "cell set %d has no data component %d, skipping", cs, comp);
			sets[cs] = -1;
			continue;
		}
		else
			sets[cs] = cs;

		if (FLDget_cell_data(cell_set, comp, &in_dtype, &cell_data[cs],
				     &size, OM_GET_ARRAY_RD) != 1) {
			ERR_RETURN("Error setting cell data");
		}

		if (FLDget_cell_null_data(cell_set, comp, &in_null_flag, (char *)&null_value) != 1) {
			ERR_RETURN("cannot get null data");
		}
		if (FLDget_cell_data_veclen(cell_set, comp, &in_veclen) != 1) {
			ERR_RETURN("Error getting cell veclen");
		}
		if (first) {
			first = 0;
			null_flag = in_null_flag;
			veclen = in_veclen;
			dtype = in_dtype;
			if (null_flag)
				UTILtype_to_double(&dnull_value, (char *)&null_value, dtype);
			stat = FLDget_cell_data_label(cell_set, comp, label, MAX_LABEL_SIZE);
			if (stat < 0)
				return(stat);
			else if (stat == 0)
				strcpy(label, "Interp Cell Data");
			stat = FLDget_cell_data_units(cell_set, comp, units, MAX_LABEL_SIZE);
			if (stat < 0)
				return(stat);
			else if (stat == 0)
				strcpy(units, "");
			if (FLDset_node_data_comp (out, 0, veclen, label, units) != 1) {
				ERR_RETURN("Error setting node data component");
			}
			if (FLDget_node_data(out, 0, &dtype, &node_data, &size, OM_GET_ARRAY_WR) != 1) {
				ERR_RETURN("Error getting node data for writing");
			}
		}
		else {
			if (in_null_flag) {
				null_flag = 1;
				UTILtype_to_double(&rnull_value, (char *)&null_value, dtype);
				if (rnull_value != dnull_value) {
					ERRerror("interp_cell_data", 2, ERR_ORIG, "cell set %d has different null data for component %d, skipping", cs, comp );
					sets[cs] = -1;
					continue;
				}
			}
			if (in_veclen != veclen) {
				ERRerror( "interp_cell_data", 2, ERR_ORIG, "cell set %d has different veclen for component %d, skipping", cs, comp );
				sets[cs] = -1;
				continue;
			}
			if (in_dtype != dtype) {
				ERRerror( "interp_cell_data", 2, ERR_ORIG, "cell set %d has different data type for component %d, skipping", cs, comp );
				sets[cs] = -1;
				continue;
			}
		}
	}

	stat = 1;
	count = 0;
	dims[0] = probe_nspace;
	dims[1] = nprobes;
	min_rng[0] = 0;
		max_rng[0] = probe_nspace;
	out_null_flag = 0;
	out_node_data = node_data;

	while (stat) {
		nleft = nprobes - count;
		if (nleft > 0) {
			if (nleft > NPOINTS)
				nreq = NPOINTS;
			else {
				nreq = nleft;
				stat = 0;
			}
		}
		else
			break;

		min_rng[1] = count;
		max_rng[1] = count + nreq;
		if (FLDget_sub_coord(probe, 2, dims, min_rng, max_rng, probe_xyz) != 1) {
			ERR_RETURN("cannot get coordinates for probe");
		}
		if (UTILinterpolate_cell_data(nreq, probe_xyz, probe_nspace, xfm,
					      mesh_info, block_table,
					      sets, dtype, veclen, cell_data,
					      (char *)&min, (char *)&max,
					      &null_flag, (char *)&null_value,
					      out_node_data) != 1) {
			ARRfree(node_data);
			if (FLDset_node_data_ncomp (out, 0) != 1) {
				ERR_RETURN("Error setting nnode_data");
			}
			FLDfree_mesh_info (mesh_info);
			FLDinterp_end(block_table);
			ERR_RETURN("interpolation is not performed");
		}

		count += nreq;
		out_node_data = node_data+veclen*count*DTYPEtype_size[dtype];
		if (null_flag == 1)
			out_null_flag = 1;
	}

	if (out_null_flag) {
		if (FLDset_node_null_data(out, 0, (char *)&null_value, dtype) != 1) {
			ERR_RETURN("Error setting null value");
		}
	}
	else {
		if (FLDset_node_null_flag(out, 0, 0) != 1) {
			ERR_RETURN("Error setting null flag");
		}
	}

	ARRfree(node_data);

	if (FLDset_node_data_minmax(out, 0, (char *)&min, (char *)&max, dtype) != 1) {
		ERR_RETURN("Error setting node minmax");
	}
	FLDfree_mesh_info(mesh_info);
	FLDinterp_end(block_table);
        free(sets);
	for (cs=0; cs<nsets; cs++) ARRfree(cell_data[cs]);
        free(cell_data);
	return(1);
}
