/*
			Copyright (c) 1994 by
			Advanced Visual Systems Inc.
			All Rights Reserved

	This software comprises unpublished confidential information of
	Advanced Visual Systems Inc. and may not be used, copied or made
	available to anyone, except in accordance with the license
	under which it is furnished.

	This file is under Perforce control
	$Id: //depot/express/fcs70/modules/iso.c#1 $
*/

#define XP_WIDE_API	/* Use Wide APIs */

#include <avs/util.h>
#include <avs/err.h>
#include <avs/om.h>
#include <avs/fld.h>
#include <avs/arr.h>
#include <avs/dv_util.h>

#define METHOD_SUCCESS 1
#define METHOD_FAILURE 0

#define ERR_RETURN(A) {ERRerror("iso", 0, ERR_ORIG, A); \
                       return METHOD_FAILURE;}

#define MAX_NAME_SIZE 1024

/* modules/celldata.c */
int FUNCcopy_cell_data (OMobj_id in, OMobj_id in_set, OMobj_id out_set,
                        xp_long ncells, xp_long *cell_list,
                        int ncomp, int *comp, xp_long start);

int FUNCiso (OMobj_id in_mesh, OMobj_id in, int comp, double level,
             OMobj_id out, OMobj_id param,
             int ncell_data, int *cell_data);

/* 64-bit porting. Only Modified Internally */
int DViso_update(OMobj_id elem_id)
{
    OMobj_id in, out, level_id, param;
    double  level;
    int stat, *cell_data, comp;
    xp_long size;

    in = OMfind_subobj(elem_id, OMstr_to_name("in"), OM_OBJ_RD);
    out = OMfind_subobj(elem_id, OMstr_to_name("out"), OM_OBJ_RW);
    param = OMfind_subobj(elem_id, OMstr_to_name("nparam"), OM_OBJ_RW);
    level_id = OMfind_subobj(elem_id, OMstr_to_name("level"), OM_OBJ_RD);
    OMget_real_val(level_id, &level);
    cell_data = (int *)OMret_name_array_ptr(elem_id, OMstr_to_name("cell_data"),
                                            OM_GET_ARRAY_RD, &size, NULL);
    if (size == 0)
        cell_data = NULL;
    if (cell_data == NULL)
        size = 0;

    /* OK if unset, default to component 0 */
    stat = OMget_name_int_val(elem_id, OMstr_to_name("component"), &comp);
    if (stat != OM_STAT_SUCCESS) comp = 0;

    stat = FUNCiso(in, in, comp, level, out, param,  (int)size, cell_data);
    
    if (cell_data)
        ARRfree(cell_data);
    return(stat);
}

#if 0
int
DV_ARRiso_update(OMobj_id elem_id, OMevent_mask event_mask, int seq_num)
{
    OMobj_id in_arr, param_arr, out_arr;
    xp_long i, num_fields;
    OMobj_id in, out, level_id, param;
    double level;
    int stat, *cell_data, comp;
    xp_long size;
    OMobj_id in_xform, out_xform;
    OMobj_name xform_name = OMstr_to_name("xform");

    in_arr  = OMfind_subobj(elem_id, OMstr_to_name("in"), OM_OBJ_RD);
    out_arr = OMfind_subobj(elem_id, OMstr_to_name("out"), OM_OBJ_RW);

    stat = OMget_array_size( in_arr, &num_fields );
    if( stat != OM_STAT_SUCCESS ) return METHOD_FAILURE;

    param_arr = OMfind_subobj(elem_id, OMstr_to_name("nparam"), OM_OBJ_RW);
    level_id = OMfind_subobj(elem_id, OMstr_to_name("level"), OM_OBJ_RD);

    OMget_real_val(level_id, &level);
    cell_data = (int *)OMret_name_array_ptr(elem_id, OMstr_to_name("cell_data"),
                                            OM_GET_ARRAY_RD, &size, NULL);
    if (size == 0)
        cell_data = NULL;
    if (cell_data == NULL)
        size = 0;

    /* OK if unset, default to component 0 */
    stat = OMget_name_int_val(elem_id, OMstr_to_name("component"), &comp);
    if (stat != OM_STAT_SUCCESS) comp = 0;

    stat = OMset_array_size( param_arr, num_fields );
    if( stat != OM_STAT_SUCCESS ) return METHOD_FAILURE;
    stat = OMset_array_size( out_arr, num_fields );
    if( stat != OM_STAT_SUCCESS ) return METHOD_FAILURE;

    for( i = 0; i < num_fields; ++i ) {
        stat = OMget_array_val( in_arr, i, &in, OM_OBJ_RD );
        if( stat != OM_STAT_SUCCESS ) continue;
        stat = OMget_array_val( param_arr, i, &param, OM_OBJ_RW );
        if( stat != OM_STAT_SUCCESS ) continue;
        stat = OMget_array_val( out_arr, i, &out, OM_OBJ_RW );
        if( stat != OM_STAT_SUCCESS ) continue;

        stat = FUNCiso(in, in, comp, level, out, param,  (int)size, cell_data);

#if 0
        if( stat != METHOD_SUCCESS ) {
            ERRerror( "iso", 1, ERR_ORIG,
                      "Error while processing field: %d", i );
        }
#endif

       /* This can be done easily in V, but its faster in C. */
       in_xform  = OMfind_subobj(in,  xform_name, OM_OBJ_RD);
       out_xform = OMfind_subobj(out, xform_name, OM_OBJ_RW);
       OMset_obj_ref( out_xform, in_xform, 0 );
    }

    if (cell_data)
        ARRfree(cell_data);
    return METHOD_SUCCESS;
}
#endif

/*
 * Splitting what is conceptually one field into two arguments
 * (in_mesh, in) allows us to avoid merges in some circumstances,
 * improving performance.
 */

/* 64-bit porting. Only Modified Internally */
int FUNCiso (OMobj_id in_mesh, OMobj_id in, int comp, double level,
             OMobj_id out, OMobj_id param,
             int ncell_data, int *cell_data)
{
	int   i, nspace, cell_ndim;
	xp_long nnodes, out_ncells_line, out_ncells_tri;
	int   data_type, veclen, ncomp;
	xp_long out_nnodes, size, conn_size_line, conn_size_tri;
	xp_long *out_nlist_line, *out_nlist_tri, *min_node_list, *max_node_list;
	int stat, null_flag;
	int   nset, set_count;
	xp_long **cell_ind;
	xp_long *ncells, count;
	char  name[MAX_NAME_SIZE];
	float  *out_coord, *t;
	char   *node_data;
	char   *mesh_info;
	double null_value;
	OMobj_id   cell_set, in_cell_set;

	out_nnodes = 0;
	out_nlist_line = out_nlist_tri = min_node_list = max_node_list = NULL;
	cell_ind = NULL;
	out_coord = t = NULL;
	node_data = NULL;

	if (FLDget_nnodes(in, &nnodes) != 1) {
		ERR_RETURN("cannot get nnodes");
	}
	if (FLDget_nspace (in_mesh, &nspace) != 1) {
		ERR_RETURN("Error getting nspace");
	}
	if (FLDget_node_data_ncomp(in, &ncomp) != 1) {
		ERR_RETURN("Error getting ncomp");
	}

	/********************************/
	/*   Free pre-allocated arrays  */
	/********************************/
	if (FLDset_nnodes (out, 0) != 1) {
		ERR_RETURN("Error setting nnodes");
	}
	if (FLDset_ncell_sets (out, 0) != 1) {
		ERR_RETURN("Error setting ncell_sets");
	}
	if (FLDset_long (param, "nnodes", 0) != 1) {
		ERR_RETURN("Error setting out nnodes");
	}
	min_node_list = max_node_list = NULL;
	if (FLDset_array_long (param, "min_node", 0, min_node_list,
			      OM_SET_ARRAY_FREE) != 1) {
		ERR_RETURN("Error setting min_node_list");
	}
	if (FLDset_array_long (param, "max_node", 0, max_node_list,
			      OM_SET_ARRAY_FREE) != 1) {
		ERR_RETURN("Error setting max_node_list");
	}
	if (FLDset_long (param, "nnodes", 0) != 1) {
		ERR_RETURN("Error setting out nnodes");
	}
	t = NULL;
	if (FLDset_array_float (param, "tpar", 0, t, OM_SET_ARRAY_FREE) != 1) {
		ERR_RETURN("Error setting tpar");
	}
	out_coord = NULL;
	if (FLDset_coord(out, out_coord, 0, OM_SET_ARRAY_FREE) != 1) {
		ERR_RETURN("Error setting out_coord");
	}


	if (nnodes == 0 || ncomp == 0) {
		return(1);
	}

	if (FLDget_node_data_veclen(in, comp, &veclen) !=1) {
		ERR_RETURN("Error getting veclen");
	}

	if (veclen != 1) {
		ERR_RETURN("component is not scalar, iso is not performed");
	}

	if (FLDget_node_data(in, comp, &data_type, &node_data,
			      &size, OM_GET_ARRAY_RD) != 1) {
		ERR_RETURN("cannot get node data");
	}
	if (FLDget_node_null_data(in, comp, &null_flag, (char *)&null_value) != 1) {
		ERR_RETURN("cannot get null data");
	}
	stat = FLDget_mesh_info(in_mesh, &mesh_info);
	if (!stat) {
		ERR_RETURN("cannot create cell table");
	}
	if (ncell_data) {
		if (FLDget_ncell_sets (in, &nset) != 1) {
			ERR_RETURN("Error getting ncell_sets");
		}
		ncells = (xp_long *)malloc(nset*sizeof(xp_long));
		cell_ind = (xp_long **)malloc(nset*sizeof(xp_long *));
	}
	else {
		cell_ind = NULL;
		ncells = NULL;
	}

	OMpush_status_range(0, 90);

	stat = UTILisosurf(mesh_info, node_data, data_type,
			   null_flag, (char *)&null_value, (double)level,
			   &out_nnodes, &min_node_list, &max_node_list,
			   &out_ncells_line, &out_nlist_line, &conn_size_line,
			   &out_ncells_tri, &out_nlist_tri, &conn_size_tri,
			   ncells, cell_ind);

	OMpop_status_range();

	if (stat != 1) {
		if (FLDset_ncell_sets (out, 0) != 1) {
			ERR_RETURN("Error setting ncell_sets");
		}
		if (node_data)
			ARRfree(node_data);
		FLDfree_mesh_info(mesh_info);
		ERR_RETURN("no isosurface created");
	}

	/*** OUTPUT FIELD ***/

	if (FLDset_nspace (out, nspace) != 1) {
		ERR_RETURN("Error setting nspace");
	}
	if (FLDset_nnodes (out, out_nnodes) != 1) {
		ERR_RETURN("Error setting nnodes");
	}
	if (FLDset_long (param, "nnodes", out_nnodes) != 1) {
		ERR_RETURN("Error setting out nnodes");
	}

	if (out_nnodes) {
		if (FLDget_coord(out, &out_coord, &size, OM_GET_ARRAY_WR) != 1) {
			ERR_RETURN("Error setting coordinate array");
		}
		if (FLDget_coord_units(in, name, MAX_NAME_SIZE) == 1) {
			if (FLDset_coord_units (out, name) != 1) {
				ERR_RETURN("Error setting units");
			}
		}
		if (FLDset_array_long (param, "min_node", out_nnodes,
				      min_node_list, OM_SET_ARRAY_FREE) != 1) {
			ERR_RETURN("Error setting min_node_list");
		}
		if (FLDset_array_long (param, "max_node", out_nnodes,
				      max_node_list, OM_SET_ARRAY_FREE) != 1) {
			ERR_RETURN("Error setting max_node_list");
		}
		if (FLDget_array_float (param, "tpar", &t, &size, OM_GET_ARRAY_WR) != 1) {
			ERR_RETURN("Error setting t array");
		}

		UTILget_param(out_nnodes, data_type, node_data,
			      level, min_node_list, max_node_list, t);

		OMstatus_check(95,NULL,&stat);

		if (stat) {
			if (FLDset_ncell_sets (out, 0) != 1) {
				ERR_RETURN("Error setting ncell_sets");
			}
			if (FLDset_long (param, "nnodes", 0) != 1) {
				ERR_RETURN("Error setting out nnodes");
			}
			if (t)
				ARRfree(t);
			if (out_coord)
				ARRfree(out_coord);
			if (node_data)
				ARRfree(node_data);
			if (min_node_list)
				ARRfree(min_node_list);
			if (max_node_list)
				ARRfree(max_node_list);
			FLDfree_mesh_info(mesh_info);
			ERR_RETURN("no isosurface created");
		}


		UTILeval_coord(mesh_info, out_nnodes,
			       min_node_list, max_node_list, t, out_coord);

		OMstatus_check(100,NULL,&stat);

		if (stat) {
			if (FLDset_ncell_sets (out, 0) != 1) {
				ERR_RETURN("Error setting ncell_sets");
			}
			if (FLDset_long (param, "nnodes", 0) != 1) {
				ERR_RETURN("Error setting out nnodes");
			}
			if (t)
				ARRfree(t);
			if (out_coord)
				ARRfree(out_coord);
			if (node_data)
				ARRfree(node_data);
			if (min_node_list)
				ARRfree(min_node_list);
			if (max_node_list)
				ARRfree(max_node_list);
			FLDfree_mesh_info(mesh_info);
			ERR_RETURN("no isosurface created");
		}
		if (t)
			ARRfree(t);
		if (out_coord)
			ARRfree(out_coord);
	}
	else {
		if (min_node_list)
			ARRfree(min_node_list);
		if (max_node_list)
			ARRfree(max_node_list);
	}
	set_count = 0;
	if (out_ncells_line) {
		if (FLDadd_cell_set(out, "Line") != 1) {
			ERR_RETURN("Error setting cell type");
		}
		if (FLDget_cell_set(out, set_count++, &cell_set) != 1) {
			ERR_RETURN("Error getting cell set");
		}
		if (FLDset_ncells(cell_set, out_ncells_line) != 1) {
			ERR_RETURN("Error setting ncells");
		}
		if (FLDset_node_connect(cell_set, out_nlist_line, conn_size_line,
					    OM_SET_ARRAY_FREE) != 1) {
			ERR_RETURN("Error setting cell connect list");
		}
		if (ncell_data) {
			if (FLDset_cell_set(cell_set, "Cell_Data_Set") != 1) {
				ERR_RETURN("Error setting cell type");
			}
			if (FLDset_cell_data_ncomp (cell_set, 0) != 1) {
				ERR_RETURN("Error setting cell data ncomp");
			}
			if (FLDset_cell_data_ncomp (cell_set, ncell_data) != 1) {
				ERR_RETURN("Error setting cell data ncomp");
			}
			for (count=0,i=0; i<nset; i++) {
				if (FLDget_cell_set(in, i, &in_cell_set) != 1) {
					ERR_RETURN("Error getting cell set");
				}
				if (FLDget_cell_ndim(in_cell_set,  &cell_ndim) != 1) {
					ERR_RETURN("cannot get cell ndim");
				}
				if (cell_ndim != 2)
					continue;
				stat = FUNCcopy_cell_data (in, in_cell_set, cell_set,
							   ncells[i], cell_ind[i],
							   ncell_data, cell_data, count);
				if (stat != 1) {
					if (FLDset_cell_data_ncomp (cell_set, 0) != 1) {
						ERR_RETURN("Error setting cell data ncomp");
					}
					for(count=0; count<(xp_long)nset; count++)
						if (cell_ind[count]) {
							ARRfree(cell_ind[count]);
							cell_ind[count] = NULL;
						}
					break;
				}
				count += ncells[i];
				if (cell_ind[i]) {
					ARRfree(cell_ind[i]);
					cell_ind[i] = NULL;
				}
			}
		}
	}
	else {
		if (out_nlist_line)
			ARRfree(out_nlist_line);
	}
	if (out_ncells_tri) {
		if (FLDadd_cell_set(out, "Tri") != 1) {
			ERR_RETURN("Error setting cell type");
		}
		if (FLDget_cell_set(out, set_count, &cell_set) != 1) {
			ERR_RETURN("Error getting cell set");
		}
		if (FLDset_ncells(cell_set, out_ncells_tri) != 1) {
			ERR_RETURN("Error setting ncells");
		}
		if (FLDset_node_connect(cell_set, out_nlist_tri, conn_size_tri,
					    OM_SET_ARRAY_FREE) != 1) {
			ERR_RETURN("Error setting cell connect list");
		}
		if (ncell_data) {
			if (FLDset_cell_set(cell_set, "Cell_Data_Set") != 1) {
				ERR_RETURN("Error setting cell type");
			}
			if (FLDset_cell_data_ncomp (cell_set, 0) != 1) {
				ERR_RETURN("Error setting cell data ncomp");
			}
			if (FLDset_cell_data_ncomp (cell_set, ncell_data) != 1) {
				ERR_RETURN("Error setting cell data ncomp");
			}
			for (count=0,i=0; i<nset; i++) {
				if (FLDget_cell_set(in, i, &in_cell_set) != 1) {
					ERR_RETURN("Error getting cell set");
				}
				if (FLDget_cell_ndim(in_cell_set,  &cell_ndim) != 1) {
					ERR_RETURN("cannot get cell ndim");
				}
				if (cell_ndim != 3)
					continue;
				stat = FUNCcopy_cell_data (in, in_cell_set, cell_set,
							   ncells[i], cell_ind[i],
							   ncell_data, cell_data, count);
				if (stat != 1) {
					if (FLDset_cell_data_ncomp (cell_set, 0) != 1) {
						ERR_RETURN("Error setting cell data ncomp");
					}
					for(count=0; count<(xp_long)nset; count++)
						if (cell_ind[count]) {
							ARRfree(cell_ind[count]);
							cell_ind[count] = NULL;
						}
					break;
				}
				count += ncells[i];
				if (cell_ind[i]) {
					ARRfree(cell_ind[i]);
					cell_ind[i] = NULL;
				}
			}
		}
	}
	else {
		if (out_nlist_tri)
			ARRfree(out_nlist_tri);
	}
	if (ncells)
		free(ncells);
	if (cell_ind) {
		if( !out_ncells_tri && !out_ncells_line ) {
			for(count=0; count<(xp_long)nset; count++)
				if (cell_ind[count])
					ARRfree(cell_ind[count]);
		}
		free(cell_ind);
	}
	if (node_data)
		ARRfree(node_data);

	FLDfree_mesh_info(mesh_info);

	return(1);
}
