/*
			Copyright (c) 1994 by
			Advanced Visual Systems Inc.
			All Rights Reserved

	This software comprises unpublished confidential information of
	Advanced Visual Systems Inc. and may not be used, copied or made
	available to anyone, except in accordance with the license
	under which it is furnished.

	This file is under Perforce control
	$Id: //depot/express/fcs70/modules/isoline.c#1 $
*/

#define XP_WIDE_API	/* Use Wide APIs */

#include <avs/util.h>
#include <avs/err.h>
#include <avs/om.h>
#include <avs/fld.h>
#include <avs/arr.h>
#include <avs/dv_util.h>
#include <avs/data_utils.h>

#define METHOD_SUCCESS 1
#define METHOD_FAILURE 0

#define ERR_RETURN(A) {ERRerror("isoline", 0, ERR_ORIG, A); \
                       return METHOD_FAILURE;}

#define MAX_NAME_SIZE 1024

int FUNCisoline (OMobj_id in, int comp, int nlev, float *levels, int color,
                 OMobj_id out);

int DVisoline_update(OMobj_id elem_id)
{
    OMobj_id in, out, level_id, col_id, comp_id;
    float  *level;
    int stat, type, comp, color;
    xp_long nlev;

    in = OMfind_subobj(elem_id, OMstr_to_name("in"), OM_OBJ_RD);
    out = OMfind_subobj(elem_id, OMstr_to_name("out"), OM_OBJ_RW);
    level_id = OMfind_subobj(elem_id, OMstr_to_name("level"), OM_OBJ_RD);
    col_id = OMfind_subobj(elem_id, OMstr_to_name("color"), OM_OBJ_RD);
    comp_id = OMfind_subobj(elem_id, OMstr_to_name("component"), OM_OBJ_RD);

    type = OM_TYPE_FLOAT;
    level = NULL;
    nlev = 0;

    stat = OMget_array_sz(level_id, &type, (char **)(&level),
                          &nlev, OM_GET_ARRAY_RD);
    if (stat != OM_STAT_SUCCESS || nlev == 0) {
        ERR_RETURN("Error getting level array");
    }
    OMget_int_val(comp_id, &comp);
    OMget_int_val(col_id, &color);

    if (FUNCisoline(in, comp, (int)nlev, level, color, out) == METHOD_SUCCESS ) {
        stat = METHOD_SUCCESS;
    }
    else stat = METHOD_FAILURE;
    if (level)
        ARRfree(level);
    return stat;
}


/* 64-bit porting. Only Modified Internally */
int
DV_ARRisoline_update(OMobj_id elem_id, OMevent_mask event_mask, int seq_num)
{
    OMobj_id in_arr, out_arr;
    xp_long i, num_fields;

    OMobj_id in, out, level_id, col_id, comp_id;
    OMobj_id src_xform_id, dst_xform_id;
    OMobj_name xform_name = OMstr_to_name("xform");
    float *level;
    int stat, type, comp, color;
    xp_long nlev;

    in_arr = OMfind_subobj(elem_id, OMstr_to_name("in"), OM_OBJ_RD);
    out_arr = OMfind_subobj(elem_id, OMstr_to_name("out"), OM_OBJ_RW);

    stat = OMget_array_size( in_arr, &num_fields );
    if( stat != OM_STAT_SUCCESS ) return METHOD_FAILURE;

    stat = OMset_array_size( out_arr, num_fields );
    if( stat != OM_STAT_SUCCESS ) return METHOD_FAILURE;

    level_id = OMfind_subobj(elem_id, OMstr_to_name("level"), OM_OBJ_RD);
    col_id = OMfind_subobj(elem_id, OMstr_to_name("color"), OM_OBJ_RD);
    comp_id = OMfind_subobj(elem_id, OMstr_to_name("component"), OM_OBJ_RD);

    type = OM_TYPE_FLOAT;
    level = NULL;
    nlev = 0;

    stat = OMget_array_sz(level_id, &type, (char **)(&level),
                          &nlev, OM_GET_ARRAY_RD);
    if (stat != OM_STAT_SUCCESS || nlev == 0) {
        ERR_RETURN("Error getting level array");
    }
    OMget_int_val(comp_id, &comp);
    OMget_int_val(col_id, &color);

    for( i = 0; i < num_fields; ++i ) {
        stat = OMget_array_val( in_arr, i, &in, OM_OBJ_RD );
        if( stat != OM_STAT_SUCCESS ) continue;
        stat = OMget_array_val( out_arr, i, &out, OM_OBJ_RW );
        if( stat != OM_STAT_SUCCESS ) continue;

        stat = FUNCisoline(in, comp, (int)nlev, level, color, out);
#if 0
        if( stat != METHOD_SUCCESS ) {
            ERRerror( "isoline", 1, ERR_ORIG,
                      "Error while processing field: %d", i );
        }
#endif

        /* This can be done easily in V, but its faster in C. */
        src_xform_id = OMfind_subobj(in,  xform_name, OM_OBJ_RD);
        dst_xform_id = OMfind_subobj(out, xform_name ,OM_OBJ_RW);
        OMset_obj_ref( dst_xform_id, src_xform_id, 0 );
    }

    if (level)
        ARRfree(level);
    return METHOD_SUCCESS;
}


/* 64-bit porting. Only Modified Internally */
int FUNCisoline (OMobj_id in, int comp, int nlev, float *levels, int color,
                 OMobj_id out)
{
	int   nspace, data_type, data_id;
	xp_long out_ncells, *out_nnodes;
	int   nl, nsets, veclen, ncomp;
	xp_long i, count, n_out_nnodes, size, conn_size;
	xp_long *out_nlist, **min_node_list, **max_node_list;
	int    stat, null_flag;
	char  label[MAX_NAME_SIZE], units[MAX_NAME_SIZE];
	float  *out_coord, *t;
	char   *node_data, *out_node_data;
	char   *mesh_info;
	double  null_value;
	OMobj_id   cell_set;

	out_nnodes = out_nlist = NULL;
	min_node_list = max_node_list = NULL;
	out_coord = t = NULL;
	node_data = out_node_data = NULL;

	if (FLDset_ncell_sets(out, 0) != 1) {
		ERR_RETURN("cannot set nsets");
	}
	if (FLDset_nnodes(out, 0) != 1) {
		ERR_RETURN("cannot set nnodes");
	}
	if (FLDget_node_data_ncomp(in, &ncomp) != 1) {
		ERR_RETURN("Error getting ncomp");
	}
	if (ncomp == 0) {
		if (FLDset_node_data_ncomp (out, 0) != 1) {
			ERR_RETURN("Error setting nnode_data");
		}
		/* ERR_RETURN("no data components, no isoline created"); */
		return METHOD_FAILURE;  /* return quietly */
	}
	if (nlev == 0) {
		if (FLDset_node_data_ncomp (out, 0) != 1) {
			ERR_RETURN("Error setting nnode_data");
		}
		ERR_RETURN("no contour level specified, no isoline created");
	}
	if (FLDget_node_data_veclen(in, comp, &veclen) !=1) {
		ERR_RETURN("Error getting veclen");
	}

	if (veclen != 1) {
		if (FLDset_node_data_ncomp (out, 0) != 1) {
			ERR_RETURN("Error setting nnode_data");
		}
		ERR_RETURN("first component must be scalar, no isoline created");
	}
	if (FLDget_nspace(in, &nspace) != 1) {
		ERR_RETURN("cannot get nspace");
	}
	if (FLDget_node_data(in, comp, &data_type, &node_data, 
			      &size, OM_GET_ARRAY_RD) != 1) {
		ERR_RETURN("cannot get node data");
	}
	if (FLDget_node_null_data(in, comp, &null_flag, (char *)&null_value) != 1) {
		ERR_RETURN("cannot get null data");
	}
	stat = FLDget_mesh_info(in, &mesh_info);
	if (!stat) {
		ERR_RETURN("cannot create cell table");
	}

	if (FLDset_ncell_sets (out, 0) != 1) {
		ERR_RETURN("Error setting ncell_sets");
	}

	out_nnodes    = (xp_long  *)malloc(nlev*sizeof(xp_long));
	min_node_list = (xp_long **)malloc(nlev*sizeof(xp_long*));
	max_node_list = (xp_long **)malloc(nlev*sizeof(xp_long*));
        if (out_nnodes    == NULL ||
            min_node_list == NULL ||
            max_node_list == NULL  ) {
          ERR_RETURN("cannot allocate node arrays");
        }
        memset( out_nnodes,    0, nlev*sizeof(xp_long)  );
        memset( min_node_list, 0, nlev*sizeof(xp_long*) );
        memset( max_node_list, 0, nlev*sizeof(xp_long*) );

	n_out_nnodes = 0;
	nsets = 0;
	for (nl=0; nl<nlev; nl++) {
		stat = UTILisoline(mesh_info, node_data, data_type, 
				   null_flag, (char *)&null_value, (double )levels[nl], 
				   &out_nnodes[nl], &min_node_list[nl], &max_node_list[nl],
				   &out_ncells, &out_nlist, &conn_size, NULL, NULL);
		if (stat != 1) {
			ERR_RETURN("No isoline created");
		}
		if (out_nnodes[nl]) {
			/* Cell Set */
			if (nl)
				for (i=0;i<conn_size;i++)
					out_nlist[i] += n_out_nnodes;
			if (FLDadd_cell_set(out, "Line") != 1) {
				ERR_RETURN("Error setting cell type");
			}
			if (FLDget_cell_set(out, nsets, &cell_set) != 1) {
				ERR_RETURN("Error getting cell set");
			}
			if (FLDset_ncells(cell_set, out_ncells) != 1) {
				ERR_RETURN("Error setting ncells");
			}
			if (FLDset_node_connect(cell_set, out_nlist, conn_size,
						OM_SET_ARRAY_FREE) != 1) {
				ERR_RETURN("Error setting cell connect list");
			}
			nsets++;
		}
		else {
			if (out_nlist)
				ARRfree(out_nlist);
		}
		n_out_nnodes += out_nnodes[nl];
	}

	if (FLDset_ncell_sets (out, nsets) != 1) {
		ERR_RETURN("Error setting ncell_sets");
	}

	/*** OUTPUT coord and data ***/
	if (FLDset_nspace (out, nspace) != 1) {
		ERR_RETURN("Error setting nspace");
	}
	if (FLDset_nnodes (out, n_out_nnodes) != 1) {
		ERR_RETURN("Error setting nnodes");
	}

	if (n_out_nnodes) {
		if (FLDget_coord_units(in, units, MAX_NAME_SIZE) == 1) {
			if (FLDset_coord_units (out, units) != 1) {
				ERR_RETURN("Error setting units");
			}
		}

		if (FLDget_coord(out, &out_coord, &size, OM_GET_ARRAY_WR) != 1) {
			ERR_RETURN("Error setting coordinate array");
		}

		for (count=0, nl=0; nl< nlev; nl++) {
			if (out_nnodes[nl]) {
				t = (float *)malloc(out_nnodes[nl]*sizeof(float));
				if (t==NULL) {
					ERR_RETURN("Error allocating t parameter");
				}

				UTILget_param(out_nnodes[nl], data_type, node_data,
					      levels[nl], min_node_list[nl], max_node_list[nl],
					      t);
				UTILeval_coord(mesh_info, out_nnodes[nl], 
					       min_node_list[nl], max_node_list[nl], 
					       t, out_coord+count);
				if (t)
					free(t);
				count += nspace*out_nnodes[nl];
			}
		}
		if (color) {
			if (FLDset_node_data_ncomp (out, 1) != 1) {
				ERR_RETURN("Error setting nnode_data");
			}
			if (FLDget_node_data_units(in, comp, units, MAX_NAME_SIZE) != 1) {
				strcpy(units, "");
			}
			if (FLDget_node_data_label(in, comp, label, MAX_NAME_SIZE) != 1) {
				strcpy(label, "");
			}
			if (FLDset_node_data_comp (out, 0, 1, label, units) != 1) {
				ERR_RETURN("Error setting node component");
			}

			if (FLDget_node_data_id(in, comp, &data_id) == 1)
				FLDset_node_data_id(out, 0, data_id);

			if (FLDget_node_data(out, 0, &data_type, &out_node_data, 
					     &size, OM_GET_ARRAY_WR) != 1) {
				ERR_RETURN("Error setting node data");
			}

			if (FLDcopy_node_minmax(in, out, comp, 0) != 1) {
				ERR_RETURN("Error copying node minmax");
			}
			if (FLDcopy_node_minmax_vec(in, out, comp, 0) != 1) {
				ERR_RETURN("Error copying node minmax");
			}

			if (FLDget_node_data_type(in, comp, &data_type) != 1) {
				ERR_RETURN("Error copying node minmax");
			}
			for (count=0, nl=0; nl<nlev; nl++) {
				UTILinit_array(out_node_data+count*DTYPEtype_size[data_type], 
					       out_nnodes[nl], 
					       (double)levels[nl], data_type);
				count += out_nnodes[nl];
			}
			if (out_node_data)
				ARRfree(out_node_data);
		}
		else {
			if (FLDset_node_data_ncomp (out, 0) != 1) {
				ERR_RETURN("Error setting nnode_data");
			}
		}
	}
	else {
		if (FLDset_ncell_sets (out, 0) != 1) {
			ERR_RETURN("Error setting ncell_sets");
		}
	}

	FLDfree_mesh_info (mesh_info);

	for(nl=0; nl<nlev; nl++) {
		if (min_node_list[nl])
			ARRfree(min_node_list[nl]);
		if (max_node_list[nl])
			ARRfree(max_node_list[nl]);
	}
	if (node_data)
		ARRfree(node_data);
	if (n_out_nnodes)
		ARRfree(out_coord);
	if (out_nnodes)
		free(out_nnodes);
	if (min_node_list)
		free(min_node_list);
	if (max_node_list)
		free(max_node_list);

	return(1);
}
