/*
			Copyright (c) 1993 by
			Advanced Visual Systems Inc.
			All Rights Reserved

	This software comprises unpublished confidential information of
	Advanced Visual Systems Inc. and may not be used, copied or made
	available to anyone, except in accordance with the license
	under which it is furnished.

	This file is under Perforce control
	$Id: //depot/express/fcs70/modules/mbstream.c#1 $
*/

#define XP_WIDE_API	/* Use Wide APIs */

#include <avs/util.h>
#include <avs/err.h>
#include <avs/om.h>
#include <avs/fld.h>
#include <avs/arr.h>
#include <avs/dv_util.h>
#include <avs/fld_info.h>
#include <avs/mat.h>

#define ERR_RETURN(A) ERRerror("stream_m_b", 0, ERR_ORIG, A); return(0);
#define CONN_BLOCK    1000
#define NODES_BLOCK   1000
#define MAX_NAME_SIZE 1024

int UTILtype_to_float(float *dst, char *src, int dtype);

int FUNCstream_multi_block (OMobj_id in, int nblocks, OMobj_id *fields,
                            OMobj_id probe, int order, int forward, int nseg,
                            int max_seg, double min_vel, int stream_comp,
                            int ribbons, double rib_width, double rib_angle,
                            OMobj_id out);

int DVmulti_block_find_block(OMobj_id in, int nblocks, OMobj_id *fields,
                             char **field_info, char **block_table,
                             float *xyz, int *block);

int DVstream_multi_block_update(OMobj_id elem_id)
{
	OMobj_id in, out, elm, probe, *block;
	int stat;
	double min_vel;
	int   i, nblocks, order, forward, backward, nseg, stream_comp, max_seg, ribbons;
	double rib_width, rib_angle;

	in = OMfind_subobj(elem_id, OMstr_to_name("in"), OM_OBJ_RD);
	out = OMfind_subobj(elem_id, OMstr_to_name("out"), OM_OBJ_RD);

	probe = OMfind_subobj(elem_id, OMstr_to_name("probe"), OM_OBJ_RD);

	elm = OMfind_subobj(elem_id, OMstr_to_name("order"), OM_OBJ_RD);
	OMget_int_val(elm, &order);

	elm = OMfind_subobj(elem_id, OMstr_to_name("forw_back"), OM_OBJ_RD);
	OMget_int_val(elm, &forward);

	elm = OMfind_subobj(elem_id, OMstr_to_name("backward"), OM_OBJ_RD);
	OMget_int_val(elm, &backward);

	elm = OMfind_subobj(elem_id, OMstr_to_name("nseg"), OM_OBJ_RD);
	OMget_int_val(elm, &nseg);

	elm = OMfind_subobj(elem_id, OMstr_to_name("max_seg"), OM_OBJ_RD);
	OMget_int_val(elm, &max_seg);

	elm = OMfind_subobj(elem_id, OMstr_to_name("stream_comp"), OM_OBJ_RD);
	OMget_int_val(elm, &stream_comp);

	elm = OMfind_subobj(elem_id, OMstr_to_name("min_vel"), OM_OBJ_RD);
	OMget_real_val(elm, &min_vel);

	elm = OMfind_subobj(elem_id, OMstr_to_name("ribbons"), OM_OBJ_RD);
	OMget_int_val(elm, &ribbons);
	if (ribbons) {
		elm = OMfind_subobj(elem_id, OMstr_to_name("rib_width"), OM_OBJ_RD);
		OMget_real_val(elm, &rib_width);
		elm = OMfind_subobj(elem_id, OMstr_to_name("rib_angle"), OM_OBJ_RD);
		OMget_real_val(elm, &rib_angle);
	}
	else {
		rib_width = 0;
		rib_angle = 0;
	}

	elm = OMfind_subobj(in, OMstr_to_name("nblocks"), OM_OBJ_RD);
	OMget_int_val(elm, &nblocks);

	if (nblocks == 0)
		return(0);

	block = (OMobj_id *)malloc(nblocks*sizeof(OMobj_id));

	elm = OMfind_subobj(in, OMstr_to_name("fields"), OM_OBJ_RD);
	for (i=0; i< nblocks; i++)
		if ((stat = OMget_array_val(elm, i, block+i, OM_OBJ_RD)) != 1)
			return(stat);


	stat = FUNCstream_multi_block(in, nblocks, block, probe, order, forward, nseg, max_seg, min_vel,
				      stream_comp, ribbons, rib_width, rib_angle, out);

	free(block);

	return(stat);
}


/* 64-bit porting. Only Modified Internally */
int FUNCstream_multi_block (OMobj_id in, int nblocks, OMobj_id *fields,
                            OMobj_id probe, int order, int forward, int nseg,
                            int max_seg, double min_vel, int stream_comp,
                            int ribbons, double rib_width, double rib_angle,
                            OMobj_id out)
{
	int   block;
	int   nspace, stat, nalloc, out_veclen, first, cur_nspace, cur_veclen;
	xp_long size, n, nprobes;
	int   veclen, i;
	float point[3], end_point[3], *probe_xyz, width_end, angle_end;
	float xform[4][4], *xfm, in_xform[4][4], *in_xfm, t_xform[4][4];
	char  *node_data;
	char  **block_table, **mesh_info;
	xp_long *cells;
	int   npoints, ln_npoints, probe_nspace, dtype, null_flag, dir;
	float *points, *rpoints, *uvws, *values, *times, *mags, *out_data, angle, width;
	xp_long out_nnodes, out_ncells, node_size, *connect, conn_size;
	char  label[MAX_NAME_SIZE], units[MAX_NAME_SIZE], cell_name[MAX_NAME_SIZE];
	double null_value, min, max;
	float dmin, dmax, dmin0, dmax0;
	OMobj_id   cell_set;
	int max_points;
	int interrupt;


	/********************************/
	/*   Free pre-allocated arrays  */
	/********************************/
	if (FLDset_ncell_sets (out, 0) != 1) {
		ERR_RETURN("Error setting ncell_sets");
	}
	out_data = points = NULL;
	if (FLDset_nnodes (out, 0) != 1) {
		ERR_RETURN("Error setting nnodes");
	}
	if (FLDset_coord(out, points, 0, OM_SET_ARRAY_FREE) != 1) {
		ERR_RETURN("Error setting coord");
	}
	if (FLDset_node_data_ncomp (out, 0) != 1) {
		ERR_RETURN("Error setting nnode_data");
	}
	/****************/
	/*   Get probe  */
	/****************/
	if (FLDget_nnodes(probe, &nprobes) != 1) {
		ERR_RETURN("cannot get nnodes for probe");
	}
	if (FLDget_nspace(probe, &probe_nspace) != 1) {
		ERR_RETURN("cannot get nspace for probe");
	}
	if (FLDget_coord(probe, &probe_xyz, &size, OM_GET_ARRAY_RD) != 1) {
		ERR_RETURN("cannot get coordinates for probe");
	}
	stat = FLDget_xform(probe, (float *)xform);
	if (stat < 0) {
		ERR_RETURN("cannot get xform for probe");
	}
	else if (stat == 0) {
		xfm = (float *)0;
	}
	else if (MATmat_is_identity((float *)xform, 4))
		xfm = (float *)0;
	else
		xfm = (float *)xform;

	mesh_info = (char **)malloc(nblocks*sizeof(char *));
	block_table = (char **)malloc(nblocks*sizeof(char *));

	/******************************************************/
	/*   Make shure that all blocks have compatible data  */
	/******************************************************/
	first = 1;
	for (block=0; block<nblocks; block++) {
		stat = FLDget_mesh_info(fields[block], &(mesh_info[block]));
		if (!stat) {
			ERR_RETURN("cannot create cell table");
		}

		OMpush_status_range(block*100/nblocks, (block+1)*100/nblocks);

		stat = FLDinterp_init(fields[block], mesh_info[block], &(block_table[block]));

		OMpop_status_range();
		if (stat != 1) {
			for (i=0; i<block; i++)
				FLDinterp_end(block_table[i]);
			ERR_RETURN("no block table created");
		}
		if (FLDget_node_data_veclen(fields[block], stream_comp, &veclen) != 1) {
			ERR_RETURN("Error getting veclen");
		}
		if (veclen < 2) {
			ERR_RETURN("cannot create streamlines for scalar");
		}
			if (FLDget_nspace(fields[block], &nspace) != 1) {
				ERR_RETURN("cannot get nnodes");
			}
		if (first) {
			stat = FLDget_xform(fields[block], (float *)in_xform);
			if (stat < 0) {
				ERR_RETURN("cannot get xform for field");
			}
			else if (stat == 0) {
				in_xfm = (float *)0;
			}
			else if (MATmat_is_identity((float *)in_xform, 4))
				in_xfm = (float *)0;
			else
				in_xfm = (float *)in_xform;
			if (in_xfm) {
				MATmat4_inverse(t_xform, (Matr4 *)in_xfm);
				if (xfm)
					MATmat4_multiply((Matr4 *)xfm, t_xform, (Matr4 *)xfm);
				else
					xfm = (float *)t_xform;
			}
			if (FLDget_node_data_type(fields[block], stream_comp, &dtype) != 1) {
				ERR_RETURN("Error copying node minmax");
			}
			if (FLDget_node_data_minmax(fields[block], stream_comp,
						    (char *)&min, (char *)&max) != 1) {
				ERR_RETURN("Error getting node minmax");
			}
			UTILtype_to_float(&dmax0, (char *)&max, dtype);
			UTILtype_to_float(&dmin0, (char *)&min, dtype);

			cur_veclen = veclen;
			cur_nspace = nspace;
			first = 0;
		}
		else {
			if (FLDget_node_data_type(fields[block], stream_comp, &dtype) != 1) {
				ERR_RETURN("Error copying node minmax");
			}
			if (FLDget_node_data_minmax(fields[block], stream_comp,
						    (char *)&min, (char *)&max) != 1) {
				ERR_RETURN("Error getting node minmax");
			}
			UTILtype_to_float(&dmax, (char *)&max, dtype);
			UTILtype_to_float(&dmin, (char *)&min, dtype);
			if (dmin < dmin0)
				dmin0 = dmin;
			if (dmax > dmax0)
				dmax0 = dmax;

			if (nspace != cur_nspace) {
				ERR_RETURN("cannot create streamlines for blocks with different nspace");
			}
			if (veclen != cur_veclen) {
				ERR_RETURN("cannot create streamlines for blocks with different veclen");
			}
		}
	}

	if (forward)
		dir = 1;
	else
		dir = -1;
	strcpy(units, "");
	strcpy(label, "");
	max_points = max_seg+1;
	out_nnodes = 0;
	out_ncells = 0;
	if (ribbons) {
		strcpy(cell_name, "Quad");
		nalloc = 2;
		conn_size = 4*(max_points-1);
		node_size = 2*max_points;
		out_veclen = 1;
		uvws = (float *)malloc(3*max_points*sizeof(float));
		times = (float *)malloc(max_points*sizeof(float));
		mags = (float *)malloc(max_points*sizeof(float));
		cells = (xp_long *)malloc(max_points*sizeof(xp_long));
		if (uvws == NULL || times == NULL || mags == NULL || cells == NULL) {
			ERR_RETURN("cannot allocate data");
		}
	}
	else {
		strcpy(cell_name, "Polyline");
		nalloc = 1;
		conn_size = CONN_BLOCK;
		node_size = max_points;
		out_veclen = veclen;
		uvws = (float *)0;
		times = (float *)0;
		mags = (float *)0;
		cells = (xp_long *)0;
	}
	connect = (xp_long *)ARRalloc(NULL, DTYPE_LONG, conn_size, NULL);
	points = (float *)ARRalloc(NULL, DTYPE_FLOAT, nspace*node_size, NULL);
	out_data = (float *)ARRalloc(NULL, DTYPE_FLOAT, out_veclen*node_size, NULL);
	values = (float *)malloc(veclen*max_points*sizeof(float));
	if (connect==NULL || points == NULL || out_data==NULL || values == NULL) {
		ERR_RETURN("cannot allocate data");
	}
	rpoints = points;
	for (n=0; n<nprobes; n++) {
		if (n & 0x1) {
			OMstatus_check((int)(100*n/nprobes),"streamlines_update",&interrupt);
			if (interrupt) {
				if (values)
					free(values);
				if (uvws)
					free(uvws);
				if (mags)
					free(mags);
				if (times)
					free(times);
				if (cells)
					free(cells);
				ARRfree(connect);
				ARRfree(points);
				ARRfree(out_data);
				ARRfree(node_data);
				ARRfree(probe_xyz);
				for (block=0; block<nblocks; block++) {
					FLDfree_mesh_info(mesh_info[block]);
					FLDinterp_end(block_table[block]);
				}
				free(mesh_info);
				free(block_table);
				ERR_RETURN("streamlines are not created");
			}
		}

		point[0] = point[1] = point[2] = 0.0;
		memcpy(point, probe_xyz+n*probe_nspace, probe_nspace*sizeof(float));
		if (xfm)
			MATvec3_mat4_multiply(point, (Matr4 *)xfm);

		block = -1;
		ln_npoints = 0;
		width = rib_width;
		angle = 3.14*rib_angle/180.0;
		first = 1;

		if (node_size < out_nnodes+nalloc*max_points) {
			int incr = nalloc*max_points;
			if( incr < NODES_BLOCK ) incr = NODES_BLOCK;
			node_size += incr;
			points = (float *)ARRrealloc(points, DTYPE_FLOAT,
						     nspace*node_size, NULL);
			out_data = (float *)ARRrealloc(out_data, DTYPE_FLOAT,
						       out_veclen*node_size, NULL);
			if (points == NULL || out_data == NULL) {
				ERR_RETURN("cannot allocate data");
			}
			rpoints = points+out_nnodes*nspace;
		}

		while (DVmulti_block_find_block(in, nblocks, fields,
						mesh_info, block_table, point, &block)) {

			if (FLDget_node_data(fields[block], stream_comp, &dtype, &node_data, &size, OM_GET_ARRAY_RD) != 1) {
				ERR_RETURN("cannot get node data");
			}
			if (FLDget_node_null_data(fields[block], stream_comp, &null_flag, (char *)&null_value) != 1) {
				ERR_RETURN("cannot get null data");
			}

			if (FLDget_node_data_units(fields[block], stream_comp, units, MAX_NAME_SIZE) != 1) {
				strcpy(units, "");
			}
			if (FLDget_node_data_label(fields[block], stream_comp, label, MAX_NAME_SIZE) != 1) {
				strcpy(label, "");
			}

			end_point[0] = end_point[1] = end_point[2] = 0.0;

			UTILstream_line(mesh_info[block], block_table[block],
					node_data, dtype, veclen, null_flag, (char *)&null_value,
					order, dir, nseg, min_vel, point,
					max_points-ln_npoints, &npoints, rpoints,
					values, ribbons, cells, uvws,
					mags, times, end_point);

			if (npoints>1) {
				memcpy(point, end_point, nspace*sizeof(float));
				if (ribbons) {
					stat = UTILribbon(mesh_info[block], npoints, nspace,
							  rpoints, values,
							  cells, uvws, mags, times,
							  veclen, node_data, dtype,
							  null_flag, (char *)&null_value,
							  width, angle, &width_end, &angle_end);
					if (!stat) {
						ERR_RETURN("Error creating ribbons");
					}
					width = width_end;
					angle = angle_end;
					/**********************************************/
					/**  Add one more to connect previous block   */
					/**********************************************/
					if (first == 0 && out_ncells != 0) {
						if (conn_size <= 4*(out_ncells+1-1)) {
							conn_size += 4*(max_points-1);
							connect = (xp_long *)ARRrealloc(connect, DTYPE_LONG,
										    conn_size, NULL);
							if (connect == NULL) {
								ERR_RETURN("cannot allocate connectivity");
							}
						}
						connect[out_ncells*4] = connect[(out_ncells-1)*4+1];
						connect[out_ncells*4+1] = out_nnodes;
						connect[out_ncells*4+2] = out_nnodes+npoints;
						connect[out_ncells*4+3] = connect[(out_ncells-1)*4+2];
						out_ncells++;
					}

					memcpy(out_data+out_veclen*out_nnodes, mags, npoints*out_veclen*sizeof(float));
					memcpy(out_data+out_veclen*(out_nnodes+npoints),
					       mags, npoints*out_veclen*sizeof(float));
					if (conn_size <= 4*(out_ncells+npoints-1)) {
						conn_size += 4*(max_points-1);
						connect = (xp_long *)ARRrealloc(connect, DTYPE_LONG,
									    conn_size, NULL);
						if (connect == NULL) {
							ERR_RETURN("cannot allocate connectivity");
						}
					}
					for (i=0; i<npoints-1; i++) {
						connect[out_ncells*4] = out_nnodes+i;
						connect[out_ncells*4+1] = out_nnodes+1+i;
						connect[out_ncells*4+2] = out_nnodes+1+npoints+i;
						connect[out_ncells*4+3] = out_nnodes+npoints+i;
						out_ncells++;
					}
					out_nnodes += 2*npoints;
					rpoints += 2*npoints*nspace;
				}
				else {
					memcpy(out_data+out_veclen*(out_nnodes+ln_npoints), values, npoints*out_veclen*sizeof(float));
					rpoints += npoints*nspace;
				}
				first = 0;
				ARRfree(node_data);
				ln_npoints += npoints;
			}
			else {
				ARRfree(node_data);
				break;
			}
			/* end while block found loop */
		}
		if (!ribbons && ln_npoints > 1) {
			if (conn_size <= 2*out_ncells) {
				conn_size += CONN_BLOCK;
				connect = (xp_long *)ARRrealloc(connect, DTYPE_LONG,
							    conn_size, NULL);
				if (connect == NULL) {
					ERR_RETURN("cannot allocate connectivity");
				}
			}
			connect[out_ncells*2] = out_nnodes;
			out_nnodes += ln_npoints;
			connect[out_ncells*2+1] = out_nnodes-1;
			out_ncells++;
		}
	}

	if (FLDset_nspace (out, nspace) != 1) {
		ERR_RETURN("Error setting nspace");
	}
	if (FLDset_nnodes (out, out_nnodes) != 1) {
		ERR_RETURN("Error setting nnodes");
	}

	/****  PUT VELOCITY into DATA  ***/

	if (FLDset_node_data_ncomp (out, 1) != 1) {
		ERR_RETURN("Error setting nnode_data");
	}
	if (FLDset_node_data_comp (out, 0, out_veclen, label, units) != 1) {
		ERR_RETURN("Error setting node component");
	}
	if (out_nnodes) {
		if (FLDset_node_data(out, 0, (char *)out_data, DTYPE_FLOAT,
				     out_veclen*out_nnodes, OM_SET_ARRAY_FREE) != 1) {
			ERR_RETURN("Error setting node data");
		}
		if (FLDset_node_data_minmax(out, 0, (char *)&dmin0, (char *)&dmax0, DTYPE_FLOAT) != 1) {
			ERR_RETURN("Error setting node minmax");
		}
	}

	if (FLDadd_cell_set(out, cell_name) != 1) {
		ERR_RETURN("Error setting cell type");
	}

	if (FLDget_cell_set(out, 0, &cell_set) != 1) {
		ERR_RETURN("Error getting cell set");
	}

	if (ribbons) {
		if (FLDset_ncells(cell_set, out_ncells) != 1) {
			ERR_RETURN("Error setting ncells");
		}
	}
	else {
		if (FLDset_npolys(cell_set, out_ncells) != 1) {
			ERR_RETURN("Error setting ncells");
		}
	}
	if (out_nnodes)
		if (FLDset_coord(out, points, nspace*out_nnodes, OM_SET_ARRAY_FREE) != 1) {
			ERR_RETURN("Error setting coord");
		}
	if (out_ncells) {
		if (ribbons) {
			if (FLDset_node_connect(cell_set, connect, 2*nalloc*out_ncells,
						OM_SET_ARRAY_FREE) != 1) {
				ERR_RETURN("Error setting cell connect list");
			}
		}
		else {
			if (FLDset_poly_connect(cell_set, connect, 2*nalloc*out_ncells,
						OM_SET_ARRAY_FREE) != 1) {
				ERR_RETURN("Error setting cell connect list");
			}
		}
	}
	if (values)
		free(values);
	if (uvws)
		free(uvws);
	if (mags)
		free(mags);
	if (times)
		free(times);
	if (cells)
		free(cells);
	for (i=0; i<nblocks; i++) {
		FLDfree_mesh_info(mesh_info[i]);
		FLDinterp_end(block_table[i]);
	}
	ARRfree(probe_xyz);
	free(mesh_info);
	free(block_table);
	return(1);
}

/* 64-bit porting. Only Modified Internally */
int DVmulti_block_find_block(OMobj_id in, int nblocks, OMobj_id *fields,
                             char **field_info, char **block_table,
                             float *xyz, int *block)
{
	xp_long cell;
	int inside, i;
	float uvw[3];
	Field_Table    *ct;
	PFI            find_cell_func;

	for (i=0; i<nblocks; i++) {
		ct = (Field_Table *)field_info[i];
		FIget_field_func(ct, FIELD_FUNC_find_cell, find_cell_func);

		if (!find_cell_func)
			return(0);

		inside = (*find_cell_func)(field_info[i], block_table[i], xyz, &cell, uvw);
		if (inside && *block != i) {
			*block = i;
			return(1);
		}
	}
	return(0);
}
