/*
			Copyright (c) 1994 by
			Advanced Visual Systems Inc.
			All Rights Reserved

	This software comprises unpublished confidential information of
	Advanced Visual Systems Inc. and may not be used, copied or made
	available to anyone, except in accordance with the license
	under which it is furnished.

	This file is under Perforce control
	$Id: //depot/express/fcs70/modules/orthoslc.c#1 $
*/

#define XP_WIDE_API	/* Use Wide APIs */

#include <avs/util.h>
#include <avs/err.h>
#include <avs/dtype.h>
#include <avs/om.h>
#include <avs/fld.h>
#include <avs/dv_util.h>
#include <avs/mat.h>

#define METHOD_SUCCESS 1
#define METHOD_FAILURE 0

#define ERR_RETURN(A) {ERRerror("orthoslice", 0, ERR_ORIG, A); \
                       return METHOD_FAILURE;}

#define MAX_NAME_SIZE 1024

#define STRUCTURED  1
#define UNIFORM     2
#define RECTILINEAR 3

static float IXFORM[16] = {1.0, 0.0, 0.0, 0.0,
                           0.0, 1.0, 0.0, 0.0,
                           0.0, 0.0, 1.0, 0.0,
                           0.0, 0.0, 0.0, 1.0};

static float YXFORM[16] = {1.0, 0.0, 0.0, 0.0,
                           0.0, 0.0, 1.0, 0.0,
                           0.0,-1.0, 0.0, 0.0,
                           0.0, 0.0, 0.0, 1.0};

static float XXFORM[16] = {0.0, 1.0, 0.0, 0.0,
                           0.0, 0.0, 1.0, 0.0,
                           1.0, 0.0, 0.0, 0.0,
                           0.0, 0.0, 0.0, 1.0};

static float ZXFORM[16] = {1.0, 0.0, 0.0, 0.0,
                           0.0, 1.0, 0.0, 0.0,
                           0.0, 0.0, 1.0, 0.0,
                           0.0, 0.0, 0.0, 1.0};

static float *XFORMS[]= {XXFORM, YXFORM, ZXFORM};

int FUNCorthoslice (OMobj_id in, int plane, int axis, OMobj_id out);
int FUNCorthoslice_unif (OMobj_id in, int plane, int axis, OMobj_id out);
int FUNCorthoslice_rect (OMobj_id in, int plane, int axis, OMobj_id out);

int DVorthoslice_update(OMobj_id elem)
{
    OMobj_id src_id, dst_id, axis_id, plane_id;
    int plane, axis;

    src_id = OMfind_subobj(elem,OMstr_to_name("in"),OM_OBJ_RD);
    axis_id = OMfind_subobj(elem,OMstr_to_name("axis"),OM_OBJ_RD);
    plane_id = OMfind_subobj(elem,OMstr_to_name("plane"),OM_OBJ_RD);
    dst_id = OMfind_subobj(elem,OMstr_to_name("out"),OM_OBJ_RW);

    if( OMget_int_val(plane_id,&plane) != OM_STAT_SUCCESS )
        ERR_RETURN( "Can't get plane" )
    if( OMget_int_val(axis_id,&axis) != OM_STAT_SUCCESS )
        ERR_RETURN( "Can't get axis" )

    if (FUNCorthoslice(src_id, plane, axis, dst_id) != METHOD_SUCCESS)
        return METHOD_FAILURE;

    return METHOD_SUCCESS;
}


int DVorthoslice_unif_update(OMobj_id elem)
{
    OMobj_id src_id, dst_id, axis_id, plane_id;
    int plane, axis;

    src_id = OMfind_subobj(elem,OMstr_to_name("in"),OM_OBJ_RD);
    axis_id = OMfind_subobj(elem,OMstr_to_name("axis"),OM_OBJ_RD);
    plane_id = OMfind_subobj(elem,OMstr_to_name("plane"),OM_OBJ_RD);
    dst_id = OMfind_subobj(elem,OMstr_to_name("out"),OM_OBJ_RW);

    if( OMget_int_val(plane_id,&plane) != OM_STAT_SUCCESS )
        ERR_RETURN( "Can't get plane" )
    if( OMget_int_val(axis_id,&axis)  != OM_STAT_SUCCESS )
        ERR_RETURN( "Can't get axis" )

    if (FUNCorthoslice_unif(src_id, plane, axis, dst_id) != METHOD_SUCCESS)
        return METHOD_FAILURE;

    return METHOD_SUCCESS;
}


int DVorthoslice_rect_update(OMobj_id elem)
{
    OMobj_id src_id, dst_id, axis_id, plane_id;
    int plane, axis;

    src_id = OMfind_subobj(elem,OMstr_to_name("in"),OM_OBJ_RD);
    axis_id = OMfind_subobj(elem,OMstr_to_name("axis"),OM_OBJ_RD);
    plane_id = OMfind_subobj(elem,OMstr_to_name("plane"),OM_OBJ_RD);
    dst_id = OMfind_subobj(elem,OMstr_to_name("out"),OM_OBJ_RW);

    if( OMget_int_val(plane_id, &plane) != OM_STAT_SUCCESS )
        ERR_RETURN( "Can't get plane" )
    if( OMget_int_val(axis_id, &axis) != OM_STAT_SUCCESS )
        ERR_RETURN( "Can't get axis" )

    if (FUNCorthoslice_rect(src_id, plane, axis, dst_id) != METHOD_SUCCESS)
        return METHOD_FAILURE;

    return METHOD_SUCCESS;
}

/* 64-bit porting. Only Modified Internally */
int
DV_ARRorthoslice_update(OMobj_id elem_id, OMevent_mask event_mask, int seq_num)
{
    OMobj_id src_arr_id, dst_arr_id;
    xp_long i, num_fields;

    OMobj_id src_id, dst_id, axis_id, plane_id;
    int plane, axis;
    int stat;

    src_arr_id = OMfind_subobj(elem_id,OMstr_to_name("in"),OM_OBJ_RD);
    dst_arr_id = OMfind_subobj(elem_id,OMstr_to_name("out"),OM_OBJ_RW);

    stat = OMget_array_size( src_arr_id, &num_fields );
    if( stat != OM_STAT_SUCCESS ) return METHOD_FAILURE;

    stat = OMset_array_size( dst_arr_id, num_fields );
    if( stat != OM_STAT_SUCCESS ) return METHOD_FAILURE;

    axis_id  = OMfind_subobj(elem_id,OMstr_to_name("axis"),OM_OBJ_RD);
    plane_id = OMfind_subobj(elem_id,OMstr_to_name("plane"),OM_OBJ_RD);

    if( OMget_int_val(plane_id, &plane) != OM_STAT_SUCCESS )
        ERR_RETURN( "Can't get plane" )
    if( OMget_int_val(axis_id,  &axis) != OM_STAT_SUCCESS )
        ERR_RETURN( "Can't get axis" )

    for( i = 0; i < num_fields; ++i ) {
        stat = OMget_array_val( src_arr_id, i, &src_id, OM_OBJ_RD );
        if( stat != OM_STAT_SUCCESS ) continue;
        stat = OMget_array_val( dst_arr_id, i, &dst_id, OM_OBJ_RW );
        if( stat != OM_STAT_SUCCESS ) continue;

        stat = FUNCorthoslice(src_id, plane, axis, dst_id);
#if 0
        if( stat != METHOD_SUCCESS ) {
            ERRerror( "orthoslice", 1, ERR_ORIG,
                      "Error while processing field: %d", i );
        }
#endif
    }

    return METHOD_SUCCESS;
}

/* 64-bit porting. Only Modified Internally */
int
DV_ARRorthoslice_unif_update(OMobj_id elem_id, OMevent_mask event_mask, int seq_num)
{
    OMobj_id src_arr_id, dst_arr_id;
    xp_long i, num_fields;

    OMobj_id src_id, dst_id, axis_id, plane_id;
    int plane, axis;
    int stat;

    src_arr_id = OMfind_subobj(elem_id,OMstr_to_name("in"),OM_OBJ_RD);
    dst_arr_id = OMfind_subobj(elem_id,OMstr_to_name("out"),OM_OBJ_RW);

    stat = OMget_array_size( src_arr_id, &num_fields );
    if( stat != OM_STAT_SUCCESS ) return METHOD_FAILURE;

    stat = OMset_array_size( dst_arr_id, num_fields );
    if( stat != OM_STAT_SUCCESS ) return METHOD_FAILURE;

    axis_id  = OMfind_subobj(elem_id,OMstr_to_name("axis"),OM_OBJ_RD);
    plane_id = OMfind_subobj(elem_id,OMstr_to_name("plane"),OM_OBJ_RD);

    if( OMget_int_val(plane_id, &plane) != OM_STAT_SUCCESS )
        ERR_RETURN( "Can't get plane" )
    if( OMget_int_val(axis_id,  &axis) != OM_STAT_SUCCESS )
        ERR_RETURN( "Can't get axis" )

    for( i = 0; i < num_fields; ++i ) {
        stat = OMget_array_val( src_arr_id, i, &src_id, OM_OBJ_RD );
        if( stat != OM_STAT_SUCCESS ) continue;
        stat = OMget_array_val( dst_arr_id, i, &dst_id, OM_OBJ_RW );
        if( stat != OM_STAT_SUCCESS ) continue;

        stat = FUNCorthoslice_unif(src_id, plane, axis, dst_id);
#if 0
        if( stat != METHOD_SUCCESS ) {
            ERRerror( "othoslice_unif", 1, ERR_ORIG,
                      "Error while processing field: %d", i );
        }
#endif
    }

    return METHOD_SUCCESS;
}

/* 64-bit porting. Only Modified Internally */
int
DV_ARRorthoslice_rect_update(OMobj_id elem_id, OMevent_mask event_mask, int seq_num)
{
    OMobj_id src_arr_id, dst_arr_id;
    xp_long i, num_fields;

    OMobj_id src_id, dst_id, axis_id, plane_id;
    int plane, axis;
    int stat;

    src_arr_id = OMfind_subobj(elem_id,OMstr_to_name("in"),OM_OBJ_RD);
    dst_arr_id = OMfind_subobj(elem_id,OMstr_to_name("out"),OM_OBJ_RW);

    stat = OMget_array_size( src_arr_id, &num_fields );
    if( stat != OM_STAT_SUCCESS ) return METHOD_FAILURE;

    stat = OMset_array_size( dst_arr_id, num_fields );
    if( stat != OM_STAT_SUCCESS ) return METHOD_FAILURE;

    axis_id  = OMfind_subobj(elem_id,OMstr_to_name("axis"),OM_OBJ_RD);
    plane_id = OMfind_subobj(elem_id,OMstr_to_name("plane"),OM_OBJ_RD);

    if( OMget_int_val(plane_id, &plane) != OM_STAT_SUCCESS )
        ERR_RETURN( "Can't get plane" )
    if( OMget_int_val(axis_id,  &axis) != OM_STAT_SUCCESS )
        ERR_RETURN( "Can't get axis" )

    for( i = 0; i < num_fields; ++i ) {
        stat = OMget_array_val( src_arr_id, i, &src_id, OM_OBJ_RD );
        if( stat != OM_STAT_SUCCESS ) continue;
        stat = OMget_array_val( dst_arr_id, i, &dst_id, OM_OBJ_RW );
        if( stat != OM_STAT_SUCCESS ) continue;

        stat = FUNCorthoslice_rect(src_id, plane, axis, dst_id);
#if 0
        if( stat != METHOD_SUCCESS ) {
            ERRerror( "orthoslice_rect", 1, ERR_ORIG,
                      "Error while processing field: %d", i );
        }
#endif
    }

    return METHOD_SUCCESS;
}

/* 64-bit porting. Only Modified Internally */
int FUNCorthoslice (OMobj_id in, int plane, int axis, OMobj_id out)
{
	int ndim, i, stat, comp, ncomp, null_flag, data_id;
	xp_long *dims, size;
	int type, in_type, veclen, nspace, out_nspace;
	float *xyz, *out_xyz, *out_xform, xfm[16];
	int  sind, ind[2];
	int out_ndim, dims_size;
	xp_long out_dims[3], min_rng[4], max_rng[4], rdims[4];
	char  units[MAX_NAME_SIZE], label[MAX_NAME_SIZE];
	char   *node_data;
	double null_value;

	if (FLDget_ndim (in, &ndim) != 1) {
		ERR_RETURN("Error getting ndim");
	}
	if (ndim < 2) {
		ERR_RETURN("Invalid ndim in the input field, must be greater than 1");
	}
	if (axis >= ndim || axis < 0) {
		sprintf(label, "invalid axis value %d, orthoslice is not created", axis);
		ERR_RETURN(label);
	}
	if (FLDget_dims (in, &dims, &dims_size) != 1) {
		ERR_RETURN("Error getting dims");
	}
	if (plane >= (int)dims[axis] || plane < 0) {
		ARRfree(dims); /* This could be added to many of the early returns */
		sprintf(label, "invalid plane value %d, orthoslice is not created", plane);
		ERR_RETURN(label);
	}
	if (FLDget_nspace (in, &nspace) != 1) {
		ERR_RETURN("Error getting nspace");
	}
	stat = FLDget_xform(in, xfm);
	if (stat < 0) {
		ERR_RETURN("cannot get xform for probe");
	}
	else if (stat == 0)
		memcpy(xfm, IXFORM, 16*sizeof(float));

	if (FLDget_coord(in, &xyz, &size, OM_GET_ARRAY_RD) != 1) {
		ERR_RETURN("cannot get coordinates ");
	}
	/*-----------------------------*/
	/*   OUT FIELD                 */
	/*-----------------------------*/

	out_ndim = ndim - 1;

	switch (axis) {
	      case 0:
		sind = 0;
		ind[0] = 1;
		ind[1] = 2;
		break;

	      case 1:
		sind = 1;
		ind[0] = 0;
		ind[1] = 2;
		break;

	      case 2:
		sind = 2;
		ind[0] = 0;
		ind[1] = 1;
		break;

	      default:
		return(0);
	}
	memcpy(rdims+1, dims, ndim*sizeof(xp_long));
	min_rng[sind+1] = plane;
	max_rng[sind+1] = plane+1;
	min_rng[ind[0]+1] = 0;
	min_rng[ind[1]+1] = 0;
	if (ind[0] < ndim)
		max_rng[ind[0]+1] = dims[ind[0]];
	if (ind[1] < ndim)
		max_rng[ind[1]+1] = dims[ind[1]];
	for (i=0; i<out_ndim; i++)
		out_dims[i] = dims[ind[i]];

	if (FLDset_ndim (out, out_ndim) != 1) {
		ERR_RETURN("Error setting ndim");
	}
	if (FLDset_dims (out, out_dims) != 1) {
		ERR_RETURN("Error setting dims");
	}
	if (FLDget_coord_units(in, units, MAX_NAME_SIZE) == 1) {
		if (FLDset_coord_units (out, units) != 1) {
			ERR_RETURN("Error setting units");
		}
	}
	out_nspace = nspace;
	if (FLDset_nspace (out, out_nspace) != 1) {
		ERR_RETURN("Error setting nspace");
	}

	if (FLDget_coord(out, &out_xyz, &size, OM_GET_ARRAY_WR) != 1) {
		ERR_RETURN("Error setting coordinate array");
	}
	rdims[0] = nspace;
	min_rng[0] = 0;
	max_rng[0] = nspace;
	if (FLDget_sub_coord(in, ndim+1, rdims,
			     min_rng, max_rng, out_xyz) != 1) {
		ERR_RETURN("Error getting sub coordinate array");
	}
	out_xform = xfm;

	if (FLDset_xform (out, out_xform) != 1) {
		ERR_RETURN("Error setting xform");
	}

	if (FLDget_node_data_ncomp(in, &ncomp) != 1) {
		ERR_RETURN("Error getting ncomp");
	}
	if (FLDset_node_data_ncomp (out, ncomp) != 1) {
		ERR_RETURN("Error setting nnode_data");
	}
	for (comp=0; comp<ncomp; comp++) {
		if (FLDget_node_data_units(in, comp, units, MAX_NAME_SIZE) != 1) {
			strcpy(units, "");
		}
		if (FLDget_node_data_label(in, comp, label, MAX_NAME_SIZE) != 1) {
			strcpy(label, "");
		}
		if (FLDget_node_data_veclen(in, comp, &veclen) != 1) {
			ERR_RETURN("Error getting veclen");
		}
		if (FLDset_node_data_comp (out, comp, veclen, label, units) != 1) {
			ERR_RETURN("Error setting node component");
		}

		if (FLDcopy_node_minmax(in, out, comp, comp) != 1) {
			ERR_RETURN("Error copying node minmax");
		}
		if (FLDcopy_node_minmax_vec(in, out, comp, comp) != 1) {
			ERR_RETURN("Error copying node minmax");
		}

		if (FLDget_node_data_id(in, comp, &data_id) == 1)
			FLDset_node_data_id(out, comp, data_id);

		if (FLDget_node_data_type(in, comp, &in_type) != 1) {
			ERR_RETURN("Error copying node minmax");
		}
		type = in_type;
		if (FLDget_node_data(out, comp, &type, &node_data,
				      &size, OM_GET_ARRAY_WR) != 1) {
			ERR_RETURN("Error setting node data");
		}
		rdims[0] = veclen;
		min_rng[0] = 0;
		max_rng[0] = veclen;
		if (FLDget_sub_node_data(in, comp, &type, ndim+1, rdims,
				     min_rng, max_rng, node_data) != 1) {
			ERR_RETURN("cannot get node data");
		}
		if (FLDget_node_null_data(in, comp, &null_flag, (char *)&null_value) != 1) {
			ERR_RETURN("cannot get null data");
		}
		if (null_flag) {
			if (FLDset_node_null_data(out, comp, (char *)&null_value, type) != 1) {
				ERR_RETURN("Error setting null value");
			}
		}
		else {
			if (FLDset_node_null_flag(out, comp, 0) != 1) {
				ERR_RETURN("Error setting null flag");
			}
		}
		ARRfree(node_data);
	}
	ARRfree(dims);
	ARRfree(xyz);
	ARRfree(out_xyz);
	return(1);
}

/* 64-bit porting. Only Modified Internally */
int FUNCorthoslice_unif (OMobj_id in, int plane, int axis, OMobj_id out)
{
	int ndim, i, stat, comp, ncomp, null_flag, data_id;
	xp_long *dims, size;
	int type, in_type, veclen, nspace, out_nspace;
	float *xyz, *out_xyz, *out_xform, xform[16], xfm1[16], xfm[16], dz;
	int  sind, ind[2];
	int out_ndim, dims_size;
	xp_long out_dims[3], min_rng[4], max_rng[4], rdims[4];
	char  units[MAX_NAME_SIZE], label[MAX_NAME_SIZE];
	char   *node_data;
	double null_value;

	if (FLDget_ndim (in, &ndim) != 1) {
		ERR_RETURN("Error getting ndim");
	}
	if (ndim < 2) {
		ERR_RETURN("Invalid ndim in the input field, must be greater than 1");
	}
	if (axis >= ndim || axis < 0) {
		sprintf(label, "invalid axis value %d, orthoslice is not created", axis);
		ERR_RETURN(label);
	}
	if (FLDget_dims (in, &dims, &dims_size) != 1) {
		ERR_RETURN("Error getting dims");
	}
	if (plane >= (int)dims[axis] || plane < 0) {
		ARRfree(dims);
		sprintf(label, "invalid plane value %d, orthoslice is not created", plane);
		ERR_RETURN(label);
	}
	if (FLDget_nspace (in, &nspace) != 1) {
		ERR_RETURN("Error getting nspace");
	}
	stat = FLDget_xform(in, xfm);
	if (stat < 0) {
		ERR_RETURN("cannot get xform for probe");
	}
	else if (stat == 0)
		memcpy(xfm, IXFORM, 16*sizeof(float));

	if (FLDget_points (in, &xyz, &size, OM_GET_ARRAY_RD) != 1) {
		ERR_RETURN("Error getting points");
	}
	/*-----------------------------*/
	/*   OUT FIELD                 */
	/*-----------------------------*/

	out_ndim = ndim - 1;

	switch (axis) {
	      case 0:
		sind = 0;
		ind[0] = 1;
		ind[1] = 2;
		break;

	      case 1:
		sind = 1;
		ind[0] = 0;
		ind[1] = 2;
		break;

	      case 2:
		sind = 2;
		ind[0] = 0;
		ind[1] = 1;
		break;

	      default:
		return(0);
	}
	memcpy(rdims+1, dims, ndim*sizeof(xp_long));
	min_rng[sind+1] = plane;
	max_rng[sind+1] = plane+1;
	min_rng[ind[0]+1] = 0;
	min_rng[ind[1]+1] = 0;
	if (ind[0] < ndim)
		max_rng[ind[0]+1] = dims[ind[0]];
	if (ind[1] < ndim)
		max_rng[ind[1]+1] = dims[ind[1]];
	for (i=0; i<out_ndim; i++)
		out_dims[i] = dims[ind[i]];

	if (FLDset_ndim (out, out_ndim) != 1) {
		ERR_RETURN("Error setting ndim");
	}
	if (FLDset_dims (out, out_dims) != 1) {
		ERR_RETURN("Error setting dims");
	}
	if (FLDget_coord_units(in, units, MAX_NAME_SIZE) == 1) {
		if (FLDset_coord_units (out, units) != 1) {
			ERR_RETURN("Error setting units");
		}
	}
	out_nspace = nspace-1;
	if (FLDset_nspace (out, out_nspace) != 1) {
		ERR_RETURN("Error setting nspace");
	}

	if (FLDset_npoints (out, 2) != 1) {
		ERR_RETURN("Error setting npoints");
	}
	if (FLDget_points (out, &out_xyz, &size, OM_GET_ARRAY_WR) != 1) {
		ERR_RETURN("Error getting points");
	}
	for (i=0; i<out_ndim; i++) {
		out_xyz[i] = xyz[ind[i]];
		out_xyz[i+out_nspace] = xyz[ind[i]+nspace];
	}
	if (dims[sind] != 1)
		dz = xyz[sind] + (xyz[sind+nspace] - xyz[sind])*plane/
			(dims[sind] - 1.);
	else
		dz = 0;

	memcpy(xfm1, XFORMS[axis], 16*sizeof(float));
	xfm1[12+sind] = dz;
	MATmat4_multiply((Matr4 *)xfm1, (Matr4 *)xfm, (Matr4 *)xform);
	out_xform = xform;

	if (FLDset_xform (out, out_xform) != 1) {
		ERR_RETURN("Error setting xform");
	}

	if (FLDget_node_data_ncomp(in, &ncomp) != 1) {
		ERR_RETURN("Error getting ncomp");
	}
	if (FLDset_node_data_ncomp (out, ncomp) != 1) {
		ERR_RETURN("Error setting nnode_data");
	}
	for (comp=0; comp<ncomp; comp++) {
		if (FLDget_node_data_units(in, comp, units, MAX_NAME_SIZE) != 1) {
			strcpy(units, "");
		}
		if (FLDget_node_data_label(in, comp, label, MAX_NAME_SIZE) != 1) {
			strcpy(label, "");
		}
		if (FLDget_node_data_veclen(in, comp, &veclen) != 1) {
			ERR_RETURN("Error getting veclen");
		}
		if (FLDset_node_data_comp (out, comp, veclen, label, units) != 1) {
			ERR_RETURN("Error setting node component");
		}

		if (FLDcopy_node_minmax(in, out, comp, comp) != 1) {
			ERR_RETURN("Error copying node minmax");
		}
		if (FLDcopy_node_minmax_vec(in, out, comp, comp) != 1) {
			ERR_RETURN("Error copying node minmax");
		}

		if (FLDget_node_data_id(in, comp, &data_id) == 1)
			FLDset_node_data_id(out, comp, data_id);

		if (FLDget_node_data_type(in, comp, &in_type) != 1) {
			ERR_RETURN("Error copying node minmax");
		}
		type = in_type;
		if (FLDget_node_data(out, comp, &type, &node_data,
				      &size, OM_GET_ARRAY_WR) != 1) {
			ERR_RETURN("Error setting node data");
		}
		rdims[0] = veclen;
		min_rng[0] = 0;
		max_rng[0] = veclen;
		if (FLDget_sub_node_data(in, comp, &type, ndim+1, rdims,
				     min_rng, max_rng, node_data) != 1) {
			ERR_RETURN("cannot get node data");
		}
		if (FLDget_node_null_data(in, comp, &null_flag, (char *)&null_value) != 1) {
			ERR_RETURN("cannot get null data");
		}
		if (null_flag) {
			if (FLDset_node_null_data(out, comp, (char *)&null_value, type) != 1) {
				ERR_RETURN("Error setting null value");
			}
		}
		else {
			if (FLDset_node_null_flag(out, comp, 0) != 1) {
				ERR_RETURN("Error setting null flag");
			}
		}
		ARRfree(node_data);
	}
	ARRfree(dims);
	ARRfree(xyz);
	ARRfree(out_xyz);
	return(1);
}

/* 64-bit porting. Only Modified Internally */
int FUNCorthoslice_rect (OMobj_id in, int plane, int axis, OMobj_id out)
{
	int ndim, i, j, stat, comp, ncomp, null_flag, data_id;
	xp_long *dims, size, n;
	int type, in_type, veclen, nspace, out_nspace;
	xp_long offset[3], out_offset[3];
	float *xyz, *out_xyz, *out_xform, xform[16], xfm1[16], xfm[16], dz;
	int  sind, ind[2];
	int out_ndim, dims_size;
	xp_long out_dims[3], min_rng[4], max_rng[4], rdims[4];
	char  units[MAX_NAME_SIZE], label[MAX_NAME_SIZE];
	char   *node_data;
	double null_value;

	if (FLDget_ndim (in, &ndim) != 1) {
		ERR_RETURN("Error getting ndim");
	}
	if (ndim < 2) {
		ERR_RETURN("Invalid ndim in the input field, must be greater than 1");
	}
	if (axis >= ndim || axis < 0) {
		sprintf(label, "invalid axis value %d, orthoslice is not created", axis);
		ERR_RETURN(label);
	}
	if (FLDget_dims (in, &dims, &dims_size) != 1) {
		ERR_RETURN("Error getting dims");
	}
	if (plane >= (int)dims[axis] || plane < 0) {
		ARRfree(dims);
		sprintf(label, "invalid plane value %d, orthoslice is not created", plane);
		ERR_RETURN(label);
	}
	if (FLDget_nspace (in, &nspace) != 1) {
		ERR_RETURN("Error getting nspace");
	}
	stat = FLDget_xform(in, xfm);
	if (stat < 0) {
		ERR_RETURN("cannot get xform for probe");
	}
	else if (stat == 0)
		memcpy(xfm, IXFORM, 16*sizeof(float));

	if (FLDget_points (in, &xyz, &size, OM_GET_ARRAY_RD) != 1) {
		ERR_RETURN("Error getting points");
	}
	/*-----------------------------*/
	/*   OUT FIELD                 */
	/*-----------------------------*/

	out_ndim = ndim - 1;

	switch (axis) {
	      case 0:
		sind = 0;
		ind[0] = 1;
		ind[1] = 2;
		break;

	      case 1:
		sind = 1;
		ind[0] = 0;
		ind[1] = 2;
		break;

	      case 2:
		sind = 2;
		ind[0] = 0;
		ind[1] = 1;
		break;

	      default:
		return(0);
	}
	memcpy(rdims+1, dims, ndim*sizeof(xp_long));
	min_rng[sind+1] = plane;
	max_rng[sind+1] = plane+1;
	min_rng[ind[0]+1] = 0;
	min_rng[ind[1]+1] = 0;
	if (ind[0] < ndim)
		max_rng[ind[0]+1] = dims[ind[0]];
	if (ind[1] < ndim)
		max_rng[ind[1]+1] = dims[ind[1]];
	for (i=0; i<out_ndim; i++)
		out_dims[i] = dims[ind[i]];

	if (FLDset_ndim (out, out_ndim) != 1) {
		ERR_RETURN("Error setting ndim");
	}
	if (FLDset_dims (out, out_dims) != 1) {
		ERR_RETURN("Error setting dims");
	}
	if (FLDget_coord_units(in, units, MAX_NAME_SIZE) == 1) {
		if (FLDset_coord_units (out, units) != 1) {
			ERR_RETURN("Error setting units");
		}
	}
	out_nspace = nspace-1;
	if (FLDset_nspace (out, out_nspace) != 1) {
		ERR_RETURN("Error setting nspace");
	}

	for (n=0,i=0; i<out_ndim; i++)
		n += out_dims[i];
	if (FLDset_npoints (out, (int)n) != 1) {
		ERR_RETURN("Error setting npoints");
	}
	if (FLDget_points (out, &out_xyz, &size, OM_GET_ARRAY_WR) != 1) {
		ERR_RETURN("Error getting points");
	}
	offset[0] = 0;
	out_offset[0] = 0;
	for (i=1; i<ndim; i++)
		offset[i] = offset[i-1] + dims[i-1]*nspace;
	for (i=1; i<out_ndim; i++)
		out_offset[i] = out_offset[i-1] + out_dims[i-1]*out_nspace;
	for (i=0; i<out_ndim; i++) {
		for (n=0; n<out_dims[i]; n++) {
			for (j=0; j<out_ndim; j++)
				out_xyz[out_offset[i]+n*out_nspace+j] =
					xyz[offset[ind[i]]+n*nspace+ind[j]];
		}
	}
	dz = xyz[offset[sind]+plane*nspace+sind];
	memcpy(xfm1, XFORMS[axis], 16*sizeof(float));
	xfm1[12+sind] = dz;
	MATmat4_multiply((Matr4 *)xfm1, (Matr4 *)xfm, (Matr4 *)xform);
	out_xform = xform;

	if (FLDset_xform (out, out_xform) != 1) {
		ERR_RETURN("Error setting xform");
	}

	if (FLDget_node_data_ncomp(in, &ncomp) != 1) {
		ERR_RETURN("Error getting ncomp");
	}
	if (FLDset_node_data_ncomp (out, ncomp) != 1) {
		ERR_RETURN("Error setting nnode_data");
	}
	for (comp=0; comp<ncomp; comp++) {
		if (FLDget_node_data_units(in, comp, units, MAX_NAME_SIZE) != 1) {
			strcpy(units, "");
		}
		if (FLDget_node_data_label(in, comp, label, MAX_NAME_SIZE) != 1) {
			strcpy(label, "");
		}
		if (FLDget_node_data_veclen(in, comp, &veclen) != 1) {
			ERR_RETURN("Error getting veclen");
		}
		if (FLDset_node_data_comp (out, comp, veclen, label, units) != 1) {
			ERR_RETURN("Error setting node component");
		}

		if (FLDcopy_node_minmax(in, out, comp, comp) != 1) {
			ERR_RETURN("Error copying node minmax");
		}
		if (FLDcopy_node_minmax_vec(in, out, comp, comp) != 1) {
			ERR_RETURN("Error copying node minmax");
		}

		if (FLDget_node_data_id(in, comp, &data_id) == 1)
			FLDset_node_data_id(out, comp, data_id);

		if (FLDget_node_data_type(in, comp, &in_type) != 1) {
			ERR_RETURN("Error copying node minmax");
		}
		type = in_type;
		if (FLDget_node_data(out, comp, &type, &node_data,
				      &size, OM_GET_ARRAY_WR) != 1) {
			ERR_RETURN("Error setting node data");
		}
		rdims[0] = veclen;
		min_rng[0] = 0;
		max_rng[0] = veclen;
		if (FLDget_sub_node_data(in, comp, &type, ndim+1, rdims,
				     min_rng, max_rng, node_data) != 1) {
			ERR_RETURN("cannot get node data");
		}
		if (FLDget_node_null_data(in, comp, &null_flag, (char *)&null_value) != 1) {
			ERR_RETURN("cannot get null data");
		}
		if (null_flag) {
			if (FLDset_node_null_data(out, comp, (char *)&null_value, type) != 1) {
				ERR_RETURN("Error setting null value");
			}
		}
		else {
			if (FLDset_node_null_flag(out, comp, 0) != 1) {
				ERR_RETURN("Error setting null flag");
			}
		}
		ARRfree(node_data);
	}
	ARRfree(dims);
	ARRfree(xyz);
	ARRfree(out_xyz);
	return(1);
}
