/*
			Copyright (c) 1994 by
			Advanced Visual Systems Inc.
			All Rights Reserved

	This software comprises unpublished confidential information of
	Advanced Visual Systems Inc. and may not be used, copied or made
	available to anyone, except in accordance with the license
	under which it is furnished.

	This file is under Perforce control
	$Id: //depot/express/fcs70/modules/pln_crop.c#1 $
*/

#define XP_WIDE_API	/* Use Wide APIs */

#include <avs/util.h>
#include <avs/err.h>
#include <avs/om.h>
#include <avs/fld.h>
#include <avs/arr.h>
#include <avs/math.h>
#include <avs/dv_util.h>

#define ERR_RETURN(A) ERRerror("plane_crop", 0, ERR_ORIG, A); return(0);
#define MAX_NAME_SIZE 1024
#define NPOINTS  1024

int FUNCplane_crop (OMobj_id in, int nplane, OMobj_id *plane, int in_out,
                    int and_or, OMobj_id out);

/* 64-bit porting. Only Modified Internally */
int DVplane_crop_update(OMobj_id elem_id)
{
	OMobj_id in, out, plane, *planes;
	int   stat, i, in_out, and_or;
	xp_long npln;

	in = OMfind_subobj(elem_id, OMstr_to_name("in"), OM_OBJ_RD);
	out = OMfind_subobj(elem_id, OMstr_to_name("out"), OM_OBJ_RW);
	plane = OMfind_subobj(elem_id, OMstr_to_name("plane"), OM_OBJ_RD);
	stat = OMget_array_size(plane, &npln);
	if (stat != 1 || npln == 0) {
		return(0);
	}
	planes = (OMobj_id *)malloc(npln*sizeof(OMobj_id));
	if (planes == NULL) {
		ERR_RETURN("cannot allocate plane ids");
	}
	for (i=0; i<(int)npln; i++) {
		if ((stat = OMget_array_val (plane, i, &planes[i], OM_OBJ_RD)) != 1) {
			ERR_RETURN("Error getting planes");
		}
	}
	FLDget_int(elem_id, "in_out", &in_out);
	FLDget_int(elem_id, "and_or", &and_or);

	stat = FUNCplane_crop(in, (int)npln, planes, in_out, and_or, out);
	free(planes);
	return(stat);
}

/* 64-bit porting. Only Modified Internally */
int FUNCplane_crop (OMobj_id in, int nplane, OMobj_id *plane, int in_out,
                    int and_or, OMobj_id out)
{
	int   i, j, comp, data_type, data_id, veclen, ncomp, null_flag, plane_nspace;
	xp_long n, nnodes, *out_ncells1,  size_w, *conn_size1, plane_nnodes; 
	xp_long **out_clist1, **out_nlist1;
	int   stat, nspace, nsets, ns;
	xp_long *old_nlist, old_nnodes, size;
	int   inside;
	xp_long nleft, count, nreq, dims[3], min_rng[3], max_rng[3];
	char  cell_name[MAX_NAME_SIZE], units[MAX_NAME_SIZE], label[MAX_NAME_SIZE];
	float  *pnts = NULL, *out_coord;
	char   *plane_data, *mesh_info;
	double null_value;
	float  bounds[3], norm[3], xfm[4][4], *in_xfm, in_xform[4][4], t_xform[4][4];
	float  dist, pxyz[3], xyz[3*NPOINTS], plane_norm[3], plane_dist;

	/* Clean output */
	if (FLDset_nnodes (out, 0) != 1) {
		ERR_RETURN("Error setting nnodes");
	}
	if (FLDset_node_data_ncomp (out, 0) != 1) {
		ERR_RETURN("Error setting nnode_data");
	}

	stat = FLDget_xform(in, (float *)in_xform);
	if (stat < 0) {
		ERR_RETURN("cannot get xform for field");
	}
	else if (stat == 0) {
		in_xfm = (float *)0;
	}
	else if (MATmat_is_identity((float *)in_xform, 4))
		in_xfm = (float *)0;
	else
		in_xfm = (float *)in_xform;
	if (in_xfm)
		MATmat4_inverse(t_xform, (Matr4 *)in_xfm);

	if (FLDget_nnodes(in, &nnodes) != 1) {
		ERR_RETURN("Error getting nnodes");
	}
	if (FLDget_nspace(in, &nspace) != 1) {
		ERR_RETURN("Error getting nspace");
	}
	if (FLDget_node_data_ncomp(in, &ncomp) != 1) {
		ERR_RETURN("Error getting ncomp");
	}
	stat = FLDget_mesh_info(in, &mesh_info);
	if (!stat) {
		ERR_RETURN("cannot create cell table");
	}
	plane_data = malloc(nnodes*sizeof(char));
	if (plane_data == NULL) {
		ERR_RETURN("cannot allocate plane data");
	}
	for (i=0; i<nplane; i++) {
		pxyz[0]=pxyz[1]=pxyz[2]=0.0;
		/*-----------------*/
		/*   GET PLANE     */
		/*-----------------*/
		if (FLDget_nnodes (plane[i], &plane_nnodes) != 1) {
			ERR_RETURN("Error getting nnodes for plane");
		}
		if (FLDget_nspace (plane[i], &plane_nspace) != 1) {
			ERR_RETURN("Error getting nspace for plane");
		}
		if (FLDget_points (plane[i], &pnts, &size, OM_GET_ARRAY_RD) != 1) {
			ERR_RETURN("Error getting points for plane");
		}
		stat = FLDget_xform(plane[i], (float *)xfm);
		if (stat < 0) {
			ERR_RETURN("cannot get xform for plane");
		}
		if (in_xfm)
			MATmat4_multiply(xfm, t_xform, xfm);
		/*------------------*/
		/* TRANSFORM PLANE  */
		/*------------------*/
		for (j=0; j<plane_nspace; j++)
			bounds[j] = pnts[j];
		ARRfree(pnts); pnts = NULL;
		for (; j<3; j++)
			bounds[j] = 0.0;
		norm[0] = 0.0; norm[1] = 0.0; norm[2] = 1.0;
		MATvec3_mat4_multiply(bounds, xfm);
		MATxform_vecs(1, (Matr3 *)norm, xfm, (Matr3 *)plane_norm);
		plane_dist = VEC_DOT(bounds, plane_norm);

		/*----------------------------------*/
		/*   compute plane distances        */
		/*----------------------------------*/
		stat = 1;
		count = 0;
		dims[0] = nspace;
		dims[1] = nnodes;
		min_rng[0] = 0;
		max_rng[0] = nspace;
		while (stat) {
			nleft = nnodes - count;
			if (nleft > 0) {
				if (nleft > NPOINTS)
					nreq = NPOINTS;
				else {
					nreq = nleft;
					stat = 0;
				}
			}
			else
				break;

			min_rng[1] = count;
			max_rng[1] = count + nreq;
			if (FLDget_sub_coord(in, 2, dims, min_rng, max_rng, xyz) != 1) {
				ERR_RETURN("cannot get coordinates for field");
			}
			if (i==0) {
				for (n=0; n<nreq; n++) {
					memcpy(pxyz,xyz+nspace*n,nspace*sizeof(float));
					dist = VEC_DOT(pxyz, plane_norm);
					if (dist > plane_dist)
						if (in_out)
							plane_data[count+n] = 1;
						else
							plane_data[count+n] = 0;
					else
						if (in_out)
							plane_data[count+n] = 0;
						else
							plane_data[count+n] = 1;
				}
			}
			else {
				for (n=0; n<nreq; n++) {
					memcpy(pxyz,xyz+nspace*n,nspace*sizeof(float));
					dist = VEC_DOT(pxyz, plane_norm);
					if (dist > plane_dist)
						if (in_out)
							inside = 1;
						else
							inside = 0;
					else
						if (in_out)
							inside = 0;
						else
							inside = 1;
					if (and_or)
						plane_data[count+n] &= inside;
					else
						plane_data[count+n] |= inside;
				}
			}

			count += nreq;
		}
	}

	if (FLDget_ncell_sets(in, &nsets) != 1) {
		ERR_RETURN("Error getting nsets");
	}
	/* Cell list, per cell set, after thresholding */
	out_clist1 = (xp_long **)calloc(nsets, sizeof(xp_long *));
	/* Number of cells, per cell set, after thresholding */
	out_ncells1 = (xp_long *)malloc(nsets*sizeof(xp_long));
	/* Connection list, per cell set, after thresholding */
	out_nlist1 = (xp_long **)malloc(nsets*sizeof(xp_long *));
	/* Size of connection list, per cell set, after thresholding */
	conn_size1 = (xp_long *)malloc(nsets*sizeof(xp_long));
	if (out_clist1 == NULL || out_ncells1 == NULL ||
            out_nlist1 == NULL || conn_size1 == NULL) {
		ERR_RETURN("cannot allocate cell lists");
	}

	memset(&null_value, 0, sizeof(double));
	if (FLDset_ncell_sets(out, 0) != 1) {
		ERR_RETURN("cannot set nsets");
	}

	stat = UTILthresh_null(mesh_info, plane_data, DTYPE_BYTE,
			       (char *)&null_value,
			       &old_nnodes, &old_nlist,
			       nsets, out_clist1, out_ncells1,
			       out_nlist1, conn_size1);
	if (!stat) {
		ERR_RETURN("cannot thresh_null cells");
	}
	free(plane_data);

	/*** OUTPUT MESH (and CELL DATA) ***/
	for (ns=0,i=0; i<nsets; i++) {
		if (out_ncells1[i] != 0) {
			OMobj_id in_cell_set, out_cell_set;
			int cell_ncomp;
			if (FLDget_cell_set(in, i, &in_cell_set) != 1) {
				ERR_RETURN("cannot get cell set");
			}
			if (FLDget_cell_set_name(in_cell_set, cell_name, MAX_NAME_SIZE) != 1) {
				ERR_RETURN("Error getting set name");
			}
			if (FLDadd_cell_set(out, cell_name) != 1) {
				ERR_RETURN("Error adding out cell set");
			}
			if (FLDget_cell_set(out, ns++, &out_cell_set) != 1) {
				ERR_RETURN("Error getting out cell set");
			}
			if (FLDset_ncells(out_cell_set, out_ncells1[i]) != 1) {
				ERR_RETURN("Error setting ncells");
			}
			if (FLDset_node_connect(out_cell_set, out_nlist1[i], conn_size1[i],
						OM_SET_ARRAY_FREE) != 1) {
				ERR_RETURN("Error setting cell connect list");
			}
			/* Deal with cell data, if any */
			if (out_clist1[i] == NULL)
				cell_ncomp = 0;
			if (FLDget_cell_data_ncomp(in_cell_set, &cell_ncomp) != 1) {
				cell_ncomp = 0;
			}
			/* Use FUNCcopy_cell_data instead ? */
			if (cell_ncomp != 0) {
				char   *cell_data, *out_cell_data;
				if (FLDset_cell_set(out_cell_set, "Cell_Data_Set") != 1) {
					ERR_RETURN("Error setting cell type");
				}
				if (FLDset_cell_data_ncomp (out_cell_set, 0) != 1) {
					ERR_RETURN("Error setting ncell_data");
				}
				if (FLDset_cell_data_ncomp (out_cell_set, cell_ncomp) != 1) {
					ERR_RETURN("Error setting ncell_data");
				}
				for (comp=0; comp<cell_ncomp; comp++) {
					if (FLDget_cell_data_units(in_cell_set, comp, units, MAX_NAME_SIZE) != 1)
						strcpy(units, "");
					if (FLDget_cell_data_label(in_cell_set, comp, label, MAX_NAME_SIZE) != 1)
						strcpy(label, "");
					if (FLDget_cell_data_veclen(in_cell_set, comp, &veclen) != 1) {
						ERR_RETURN("Error getting cell veclen");
					}
					if (FLDset_cell_data_comp (out_cell_set, comp, veclen, label, units) != 1) {
						ERR_RETURN("Error setting cell component");
					}
                                	if (FLDget_cell_data_id(in_cell_set, comp, &data_id) == 1)
						FLDset_cell_data_id(out_cell_set, comp, data_id);
					if (FLDcopy_cell_minmax(in_cell_set, out_cell_set, comp, comp) != 1) {
						ERR_RETURN("Error copying cell minmax");
					}
					if (FLDcopy_cell_minmax_vec(in_cell_set, out_cell_set, comp, comp) != 1) {
						ERR_RETURN("Error copying cell minmax vec");
					}
					cell_data = out_cell_data = NULL;
					if (FLDget_cell_data(in_cell_set, comp, &data_type, &cell_data,
							     &size_w, OM_GET_ARRAY_RD) != 1) {
						ERR_RETURN("cannot get cell data");
					}
					if (FLDget_cell_data(out_cell_set, comp, &data_type, &out_cell_data,
							     &size_w, OM_GET_ARRAY_WR) != 1) {
						ERR_RETURN("Error setting cell data");
					}
					size_w = UTILdata_list_struct(NULL, data_type, veclen, cell_data,
								 out_ncells1[i], out_clist1[i], out_cell_data);
					ARRfree(cell_data);
					ARRfree(out_cell_data);

					if (FLDget_cell_null_data(in_cell_set, comp, &null_flag, (char *)&null_value) != 1) {
						ERR_RETURN("cannot get cell null data");
					}
					if (null_flag) {
						if (FLDset_cell_null_data(out_cell_set, comp, (char *)&null_value, data_type) != 1) {
							ERR_RETURN("Error setting cell null value");
						}
					}
					else {
						if (FLDset_cell_null_flag(out_cell_set, comp, 0) != 1) {
							ERR_RETURN("Error setting cell null flag");
						}
					}
				}	/* loop over components */
			}	/* if cell_list and ncomp != 0 */
			if (out_clist1[i]) ARRfree(out_clist1[i]);
		}
	}
	if (FLDset_nnodes (out, old_nnodes) != 1) {
		ERR_RETURN("Error setting nnodes");
	}
	if (FLDget_coord_units(in, units, MAX_NAME_SIZE) == 1) {
		if (FLDset_coord_units (out, units) != 1) {
			ERR_RETURN("Error setting units");
		}
	}
	if (FLDget_nspace (in, &nspace) != 1) {
		ERR_RETURN("Error getting nspace");
	}
	if (FLDset_nspace (out, nspace) != 1) {
		ERR_RETURN("Error setting nspace");
	}
	if (FLDget_coord(out, &out_coord, &size_w, OM_GET_ARRAY_WR) != 1) {
		ERR_RETURN("Error setting coordinate array");
	}
	if (old_nnodes) {
		UTILget_coord_list(mesh_info, old_nnodes, old_nlist, out_coord);
	}

	/** OUT NODE DATA **/
	if (FLDset_node_data_ncomp (out, ncomp) != 1) {
		ERR_RETURN("Error setting nnode_data");
	}
	if (old_nnodes) {
		char   *node_data, *out_node_data;
		for (comp=0; comp<ncomp; comp++) {
			if (FLDget_node_data_units(in, comp, units, MAX_NAME_SIZE) != 1) {
				strcpy(units, "");
			}
			if (FLDget_node_data_label(in, comp, label, MAX_NAME_SIZE) != 1) {
				strcpy(label, "");
			}
			if (FLDget_node_data_veclen(in, comp, &veclen) != 1) {
				ERR_RETURN("Error getting veclen");
			}
			if (FLDset_node_data_comp (out, comp, veclen, label, units) != 1) {
				ERR_RETURN("Error setting node component");
			}
			if (FLDcopy_node_minmax(in, out, comp, comp) != 1) {
				ERR_RETURN("Error copying node minmax");
			}
			if (FLDcopy_node_minmax_vec(in, out, comp, comp) != 1) {
				ERR_RETURN("Error copying node minmax");
			}

			if (FLDget_node_data_id(in, comp, &data_id) == 1)
				FLDset_node_data_id(out, comp, data_id);

			if (FLDget_node_data(in, comp, &data_type, &node_data,
					     &size_w, OM_GET_ARRAY_RD) != 1) {
				ERR_RETURN("cannot get node data");
			}

			if (FLDget_node_data(out, comp, &data_type, &out_node_data,
					     &size_w, OM_GET_ARRAY_WR) != 1) {
				ERR_RETURN("Error setting node data");
			}
			size_w = UTILget_data_list(mesh_info, data_type, veclen, node_data,
						 old_nnodes, old_nlist, out_node_data);
			ARRfree(node_data);
			ARRfree(out_node_data);
			if (FLDget_node_null_data(in, comp, &null_flag, (char *)&null_value) != 1) {
				ERR_RETURN("cannot get null data");
			}
			if (null_flag) {
				if (FLDset_node_null_data(out, comp, (char *)&null_value, data_type) != 1) {
					ERR_RETURN("Error setting null value");
				}
			}
			else {
				if (FLDset_node_null_flag(out, comp, 0) != 1) {
					ERR_RETURN("Error setting null flag");
				}
			}
		}
	}
	FLDfree_mesh_info(mesh_info);
	if (old_nnodes)
		ARRfree(out_coord);

	free(out_clist1);
	free(out_ncells1);
	free(out_nlist1);
	free(conn_size1);
	if (old_nlist)
		free(old_nlist);
	return(1);
}
