/*
			Copyright (c) 1994 by
			Advanced Visual Systems Inc.
			All Rights Reserved

	This software comprises unpublished confidential information of
	Advanced Visual Systems Inc. and may not be used, copied or made
	available to anyone, except in accordance with the license
	under which it is furnished.

	This file is under Perforce control
	$Id: //depot/express/fcs70/modules/pln_dist.c#1 $
*/

#define XP_WIDE_API	/* Use Wide APIs */

#include <avs/util.h>
#include <avs/err.h>
#include <avs/om.h>
#include <avs/fld.h>
#include <avs/math.h>
#include <avs/arr.h>
#include <avs/mat.h>
#include <avs/om_att.h>

#define METHOD_SUCCESS 1
#define METHOD_FAILURE 0

#define ERR_RETURN(A) {ERRerror("plane_dist", 0, ERR_ORIG, A); \
                       return METHOD_FAILURE;}
#define EPS  0.0001

int FUNCplane_dist (OMobj_id in, OMobj_id plane, OMobj_id out, float *dist);

int DVplane_distance_update(OMobj_id elem_id)
{
    OMobj_id in, out, plane;
    float dist;

    in = OMfind_subobj(elem_id, OMstr_to_name("in"), OM_OBJ_RD);
    out = OMfind_subobj(elem_id, OMstr_to_name("out"), OM_OBJ_RW);
    plane = OMfind_subobj(elem_id, OMstr_to_name("plane"), OM_OBJ_RD);

    if (FUNCplane_dist(in, plane, out, &dist) != METHOD_SUCCESS ) {
        return METHOD_FAILURE;
    }

    if (OMset_name_real_val (elem_id, OMstr_to_name("dist"), (double)dist) != 1) {
        ERR_RETURN("cannot set dist value");
    }
    return METHOD_SUCCESS;
}

/* 64-bit porting. Only Modified Internally */
int
DV_ARRplane_distance_update(OMobj_id elem_id, OMevent_mask event_mask, int seq_num)
{
    OMobj_id in_arr, out_arr;
    xp_long i, num_fields;
    int stat;

    OMobj_id in, out, plane, dist_id;
    float dist;

    in_arr  = OMfind_subobj(elem_id, OMstr_to_name("in"), OM_OBJ_RD);
    out_arr = OMfind_subobj(elem_id, OMstr_to_name("out"), OM_OBJ_RW);
    plane   = OMfind_subobj(elem_id, OMstr_to_name("plane"), OM_OBJ_RD);
    dist_id = OMfind_subobj(elem_id, OMstr_to_name("dist"), OM_OBJ_RW);

    stat = OMget_array_size( in_arr, &num_fields );
    if( stat != OM_STAT_SUCCESS ) return METHOD_FAILURE;

    stat = OMset_array_size( out_arr, num_fields );
    if( stat != OM_STAT_SUCCESS ) return METHOD_FAILURE;

    for( i = 0; i < num_fields; ++i ) {
        stat = OMget_array_val( in_arr,  i, &in,  OM_OBJ_RD );
        if( stat != OM_STAT_SUCCESS ) continue;
        stat = OMget_array_val( out_arr, i, &out, OM_OBJ_RW );
        if( stat != OM_STAT_SUCCESS ) continue;

        stat = FUNCplane_dist(in, plane, out, &dist);
#if 0
        if( stat != METHOD_SUCCESS ) {
            ERRerror( "plane_distance", 1, ERR_ORIG,
                      "Error while processing field: %d", i );
        }
#endif

        /* It should be always the same value, so just set it once */
        if (i == 0 && OMset_real_val(dist_id, dist) != 1) {
            ERR_RETURN("cannot set dist value");
        }
    }

    return METHOD_SUCCESS;
}

/* 64-bit porting. Only Modified Internally */
int FUNCplane_dist (OMobj_id in, OMobj_id plane, OMobj_id out, float *dist)
{
	xp_long i_w, nnodes, size;
	int   i, type, recomp;
	int   nspace, plane_nspace, stat;
	float  *coord, *pnts;
	float  bounds[3], norm[3], norm_xfm[3], *norm_x;
	float  xfm[4][4], *in_xfm, in_xform[4][4];
	float  cross[3], l;
	float  *node_dist, plane_dist;
	int    in_seq, out_seq;
	OMobj_id pln_id;

	/*-----------------*/
	/*   GET PLANE     */
	/*-----------------*/
	if (FLDget_nspace (plane, &plane_nspace) != 1) {
		ERR_RETURN("Error getting nspace");
	}
	if (FLDget_points (plane, &pnts, &size, OM_GET_ARRAY_RD) != 1) {
		ERR_RETURN("Error getting points");
	}
	stat = FLDget_xform(plane, (float *)xfm);
	if (stat < 0) {
		ERR_RETURN("cannot get xform for probe");
	}
	/*------------------*/
	/* TRANSFORM PLANE  */
	/*------------------*/
	stat = FLDget_xform(in, (float *)in_xform);
	if (stat < 0) {
		ERR_RETURN("cannot get xform for field");
	}
	else if (stat == 0) {
		in_xfm = (float *)0;
	}
	else if (MATmat_is_identity((float *)in_xform, 4))
		in_xfm = (float *)0;
	else
		in_xfm = (float *)in_xform;

	for (i=0; i<plane_nspace; i++)
		bounds[i] = pnts[i];

	ARRfree(pnts);

	for (; i<3; i++)
		bounds[i] = 0.0;
	norm[0] = 0.0; norm[1] = 0.0; norm[2] = 1.0;

	MATvec3_mat4_multiply(bounds, xfm);

	MATxform_vecs(1, (Matr3 *)norm, xfm, (Matr3 *)norm_xfm);

	l = VEC_LEN(norm_xfm);
	VEC_NORMALIZE(norm_xfm,l);

	plane_dist = VEC_DOT(bounds, norm_xfm);

	*dist = plane_dist;

	stat = FLDget_array_float (out, "plane_norm", &norm_x, &size, OM_GET_ARRAY_RD);
	if (stat < 1) {
		if (FLDadd_float_array(out, "plane_norm", 3) != 1) {
			ERR_RETURN("cannot create normal array");
		}
		/*
		 * Prevent this array from being saved.
		 * Eventually we might be able to determine this
		 * automatically but for now, it must be set by hand.
		 */
		pln_id = OMfind_subobj(out, OMstr_to_name("plane_norm"),
					OM_OBJ_RW);
		OMset_obj_atts(pln_id, OM_atts_nosave);
		norm_x = NULL;
	}
	out_seq = FLDget_subelem_seq(out, "Node_Data");
	in_seq = FLDget_subelem_seq(in, "Mesh");
	if (in_seq > out_seq)
		recomp = 1;
	else {
		if (stat < 1)
			recomp = 1;
		else {
			VEC_CROSS(cross, norm_xfm, norm_x);
			l = VEC_LEN(cross);
			if (l < EPS) {
				/* CFS PR 23822.  Catch the case where the new
				 * normal is 180 different than the old normal
				 * the cross test gives the wrong result.
				 */
				l = VEC_DOT(norm_xfm, norm_x);
				if( l > 0 ) recomp = 0;
				else recomp = 1;
			}
			else
				recomp = 1;
		}
	}
	if (recomp) {
		if (FLDget_nnodes(in, &nnodes) != 1) {
			ERR_RETURN("cannot get nnodes");
		}
		if (FLDget_nspace(in, &nspace) != 1) {
			ERR_RETURN("cannot get nnodes");
		}
		if (FLDget_coord(in, &coord, &size, OM_GET_ARRAY_RD) != 1) {
			ERR_RETURN("cannot get coordinates");
		}

		if (FLDset_nnodes (out, nnodes) != 1) {
			ERR_RETURN("Error setting nnodes");
		}
		type = OM_TYPE_FLOAT;
		if (FLDget_node_data(out, 0, &type, (char **)&node_dist,
				      &size, OM_GET_ARRAY_WR) != 1) {
			ERR_RETURN("cannot get node data");
		}
		cross[0]=cross[1]=cross[2]=0.0;
		for (i_w=0; i_w<nnodes; i_w++) {
			memcpy(cross, coord+nspace*i_w, nspace*sizeof(float));
			if (in_xfm)
				MATvec3_mat4_multiply(cross, (Matr4 *)in_xfm);
			node_dist[i_w] = VEC_DOT(cross, norm_xfm);
		}
		if (FLDset_array_float (out, "plane_norm", 3, norm_xfm, OM_SET_ARRAY_COPY) != 1) {
			ERR_RETURN("cannot set normal array");
		}
		if (coord) ARRfree(coord);
		if (node_dist) ARRfree(node_dist);
	}
	if (norm_x)
	    ARRfree(norm_x);
	return(1);
}
