/*
			Copyright (c) 2003 by
			Advanced Visual Systems Inc.
			All Rights Reserved

	This software comprises unpublished confidential information of
	Advanced Visual Systems Inc. and may not be used, copied or made
	available to anyone, except in accordance with the license
	under which it is furnished.

	This file is under Perforce control
	$Id: //depot/express/fcs70/modules/rd_dxf.c#1 $
*/

#define XP_WIDE_API	/* Use Wide APIs */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>  /* time_t */

#include <avs/f_utils.h>
#include <avs/err.h>
#include <avs/om.h>
#include <avs/fld.h>

#define METHOD_SUCCESS 1
#define METHOD_FAILURE 0

#define LINE_LEN 256

#define ERR_RETURN(A) {ERRerror("read_dxf", 0, ERR_ORIG, A); \
                       return(METHOD_FAILURE);}

#ifndef GD_COLOR_DATA_ID
#define GD_COLOR_DATA_ID 667
#endif


#define DXF_DEFAULT_COLOR 7

/* Default color table */
static float dxf_colors[10][3] = {
    { 0.0, 0.0, 0.0 },	/* black */
    { 1.0, 0.0, 0.0 },	/* red */
    { 1.0, 1.0, 0.0 },	/* yellow */
    { 0.0, 1.0, 0.0 },	/* green */
    { 0.0, 1.0, 1.0 },	/* cyan */
    { 0.0, 0.0, 1.0 },	/* blue */
    { 1.0, 0.0, 1.0 },	/* magenta */
    { 1.0, 1.0, 1.0 },	/* white (index 7, the default color) */
    { 0.5, 0.5, 0.5 },	/* dark gray */
    { 0.75, 0.75, 0.75 }	/* light gray */
};

static int FUNCread_dxf( FILE *fp, OMobj_id fld );

int
DVread_dxf_update( OMobj_id mod_id, OMevent_mask mask, int seq_num )
{
    OMobj_id file_id, fld_id;
    FILE *fp = NULL;
    char *temp_str = NULL, *filename = NULL, *ext = NULL;
    char file_buf[AVS_PATH_MAX];

    /* in */
    file_id = OMfind_subobj( mod_id, OMstr_to_name("filename"), OM_OBJ_RD );

    /* out */
    fld_id = OMfind_subobj( mod_id, OMstr_to_name("outFld"), OM_OBJ_RW );

    if( OMget_str_val(file_id, &temp_str, 0) != OM_STAT_SUCCESS )
        ERR_RETURN( "Can't get filename." );

    filename = FILEmap_variables( temp_str, file_buf );

    if( temp_str ) { free (temp_str); temp_str = NULL; }

    if( filename != NULL && filename[0] != 0 )
        fp = FILEfopen( filename, SIO_R_BIN );

    if( fp == NULL ) {
        ERRerror( "read_dxf", 1, ERR_ORIG, "Can't open data file: %s",
                  filename );
        return METHOD_FAILURE;
    }

    ext = FILEget_file_extension( filename );
    ext += 1;	/* skip over '.' */

    /* Basic (shared) field setup */
    FLDset_nnodes( fld_id, 0 );
    FLDset_ncell_sets( fld_id, 0 );
    FLDset_nspace( fld_id, 3 );

    if( !strcmp( ext, "DXF" ) || !strcmp( ext, "dxf" ) ) {
        FUNCread_dxf( fp, fld_id );
    }
    else {
        ERRerror( "read_dxf", 1, ERR_ORIG,
                  "Unknown file extension: %s", ext );
    }

    fclose( fp );

    return METHOD_SUCCESS;
}


/*
 * Data structures and methods to hold the verticies and
 * connection lists and more.
 */

/*
 * The point of all these vertex data structures is to catch
 * verticies that are shared between the individual cells.
 *
 * In order to handle a large number of verticies without a O(n^2)
 * slowdown, this code uses randomized binary search trees
 * to implement a O(n*log(n)) alorgithm to speed the process.
 */

typedef struct _Vertex {
    float xyz[3];
    xp_long index;	/* Does double duty as the search priority */

    /* Overhead to implement the searching algorithm */
    /* int priority; */
    struct _Vertex *left;
    struct _Vertex *right;
    struct _Vertex *parent;
} Vertex;

static void vertex_init( Vertex *v, float *xyz )
{
    if( xyz != NULL ) {
        v->xyz[0] = xyz[0];
        v->xyz[1] = xyz[1];
        v->xyz[2] = xyz[2];
    }

    v->index = rand();
    v->left   = NULL;
    v->right  = NULL;
    v->parent = NULL;
}

static void vertex_tree_free( Vertex *v )
{
    Vertex *tmp;
    while( v != NULL ) {
        if( v->left )
            vertex_tree_free( v->left );
        tmp = v->right;
        free( v );
        v = tmp;	/* OK if NULL, thats the end of the loop */
    }
}

static void vertex_tree_rotate_right( Vertex *y )
{
    Vertex *x = y->left;
    Vertex *p = y->parent;

    y->left = x->right;

    if( y->left != NULL )
        y->left->parent = y;

    if( p != NULL ) {	/* NULL when y is (old) root */
        if( p->right == y )
            p->right = x;
        else
            p->left = x;
    }

    x->parent = p;
    x->right  = y;
    y->parent = x;	/* x might be new root */
}

static void vertex_tree_rotate_left( Vertex *x )
{
    Vertex *y = x->right;
    Vertex *p = x->parent;

    x->right = y->left;

    if( x->right != NULL )
        x->right->parent = x;

    if( p != NULL ) {	/* NULL when x is (old) root */
        if( p->left == x )
            p->left = y;
        else
            p->right = y;
    }

    y->parent = p;
    y->left   = x;
    x->parent = y;	/* y might be new root */
}

static void vertex_tree_bubble_up( Vertex *v )
{
    Vertex *p;
    while( 1 ) {
        p = v->parent;
        if( p != NULL && v->index < p->index ) {
            if( p->left == v )
                vertex_tree_rotate_right( p );
            else
                vertex_tree_rotate_left( p );
        }
        else break;
    }
}

static int vertex_compare( float *xyz1, float *xyz2 )
{
    if( xyz1[0] < xyz2[0] ) return -1;
    if( xyz1[0] > xyz2[0] ) return  1;

    if( xyz1[1] < xyz2[1] ) return -1;
    if( xyz1[1] > xyz2[1] ) return  1;

    if( xyz1[2] < xyz2[2] ) return -1;
    if( xyz1[2] > xyz2[2] ) return  1;

    return  0;
}

static Vertex* vertex_tree_add( Vertex *v, float *xyz )
{
    int compare;
    Vertex* p;

    while( v != NULL ) {	/* not expecting this test to fail */
        compare = vertex_compare( v->xyz, xyz );
        if( compare == 0 ) {
            /* Exact hit ... we've found what we are looking for */
            return v;
        }
        else if( compare > 0 ) {
            /* current is bigger then the new, so look left */
            p = v;
            v = v->left;
        }
        else if( compare < 0 ) {
            /* current is smaller than the new, so look right */
            p = v;
            v = v->right;
        }
        if( v == NULL ) {
            /* No child to search, so create a new one */
            v = malloc( sizeof(Vertex) );
            vertex_init( v, xyz );
            if( compare > 0 ) p->left  = v;
            else              p->right = v;
            v->parent = p;
            vertex_tree_bubble_up( v );
            /* After rebalancing the tree, return the new node */
            return v;
        }
    } /* while */

    return NULL;	/* not expecting this */
}

/*
 * v_t_count and v_t_coords need to be in exact sync in the
 * order that they visit the nodes.
 */
/* 64-bit porting. Directly Modified */  
static void vertex_tree_count( Vertex *v, xp_long * count )
{
    v->index = *count;	/* used for node connect lists */
    *count += 1;	/* count myself */

    if( v->left )
        vertex_tree_count( v->left, count );
    if( v->right )
        vertex_tree_count( v->right, count );
}
/* 64-bit porting. Only Modified Internally */ 
static void vertex_tree_coords( Vertex *v, float *coords )
{
    xp_long where = v->index*3;
    coords[where+0] = v->xyz[0];
    coords[where+1] = v->xyz[1];
    coords[where+2] = v->xyz[2];

    if( v->left )
        vertex_tree_coords( v->left, coords );
    if( v->right )
        vertex_tree_coords( v->right, coords );
}

/*
 * Vertex Set disconnected ... we need to connect the triangles,
 * finding duplicate points as we go along.
 */

typedef struct _Vset_d {
    Vertex *root;
} Vset_d;

static void vset_d_init( Vset_d *vset )
{
    vset->root = NULL;
}

static void vset_d_free( Vset_d *vset )
{
    if( vset != NULL && vset->root != NULL )
        vertex_tree_free( vset->root );
}

/*
 * Return the total number of nodes we have collected and
 * additionally fill in the node index counters that will
 * be needed to construct the node connect lists.
 */
/* 64-bit porting. Directly Modified */ 
static xp_long vset_d_count( Vset_d *vset )
{
    if( vset != NULL && vset->root != NULL ) {
        xp_long count = 0;
        vertex_tree_count( vset->root, &count );
        return count;
    }
    else return 0;
}

/* Fill in the coord list you get from FLDget_coords */
static void vset_d_coords( Vset_d *vset, float *coords )
{
    if( vset != NULL && vset->root != NULL ) {
        vertex_tree_coords( vset->root, coords );
    }
}

static Vertex *vset_d_add( Vset_d *vset, float xyz[3] )
{
    if( vset->root == NULL ) {
        vset->root = malloc( sizeof(Vertex) );
        vertex_init( vset->root, xyz );
        return vset->root;
    }
    else {
        Vertex *ret = vertex_tree_add( vset->root, xyz );
        /* There is a chance the new guy is the new root. */
        if( ret->parent == NULL ) vset->root = ret;
        return ret;
    }
}

/*
 * Data structures to handle cell sets and their connection lists.
 */

static const int cellset_chunk = 2048;

/* Match values from v/fld.v */
#define CELL_TYPE_INVALID -1
#define CELL_TYPE_POINT 1
#define CELL_TYPE_LINE 2
#define CELL_TYPE_TRI  4
#define CELL_TYPE_QUAD 5

typedef struct {
    int cell_type;
    xp_long ncells;
    xp_long nnodes;
    char *name;

    float *colors;	/* rgb (3 floats) color per cell */
    xp_long colors_alloc;

    /* Pointers to vert objects, *not* integer connection indicies. */
    Vertex **vertex_conn;
    xp_long vertex_alloc;
} Cellset;

static void cellset_init( Cellset *cset, const char *name )
{
    cset->cell_type = CELL_TYPE_INVALID;
    cset->ncells = 0;
    cset->nnodes = 0;
    if( name != NULL && name[0] != 0 )
        cset->name = strdup( name );
    else
        cset->name = NULL;
    cset->colors_alloc = 0;
    cset->colors       = NULL;
    cset->vertex_alloc = cellset_chunk;
    cset->vertex_conn  = malloc( cset->vertex_alloc * sizeof(Vertex *) );
}

static void cellset_free( Cellset *cset )
{
    if( cset->name ) { free( cset->name );  cset->name = NULL; }
    if( cset->vertex_conn ) { free( cset->vertex_conn ); cset->vertex_conn = NULL; }
    if( cset->colors ) { free( cset->colors ); cset->colors = NULL; }
}

static void cellset_add_point(
    Cellset *cset, Vertex *v1 )
{
    if( (cset->ncells+1) >= cset->vertex_alloc ) {
        cset->vertex_alloc += cellset_chunk;
        cset->vertex_conn = realloc( cset->vertex_conn,
                                     cset->vertex_alloc * sizeof( Vertex *) );
    }

    /*
     * The pointers should still be valid as we shuffle around
     * the search tree.
     */
    cset->vertex_conn[(cset->ncells*2)+0] = v1;

    cset->ncells += 1;
    cset->nnodes += 1;
}

static void cellset_add_line(
    Cellset *cset, Vertex *v1, Vertex *v2 )
{
    if( (cset->ncells+1)*2 >= cset->vertex_alloc ) {
        cset->vertex_alloc += cellset_chunk;
        cset->vertex_conn = realloc( cset->vertex_conn,
                                     cset->vertex_alloc * sizeof( Vertex *) );
    }

    /* The pointers should still be valid as we shuffle around
     * the search tree.
     */
    cset->vertex_conn[(cset->ncells*2)+0] = v1;
    cset->vertex_conn[(cset->ncells*2)+1] = v2;

    cset->ncells += 1;
    cset->nnodes += 2;
}

static void cellset_add_tri(
    Cellset *cset, Vertex *v1, Vertex *v2, Vertex *v3 )
{
    if( (cset->ncells+1)*3 >= cset->vertex_alloc ) {
        cset->vertex_alloc += cellset_chunk;
        cset->vertex_conn = realloc( cset->vertex_conn,
                                     cset->vertex_alloc * sizeof( Vertex *) );
    }

    /* The pointers should still be valid as we shuffle around
     * the search tree.
     */
    cset->vertex_conn[(cset->ncells*3)+0] = v1;
    cset->vertex_conn[(cset->ncells*3)+1] = v2;
    cset->vertex_conn[(cset->ncells*3)+2] = v3;

    cset->ncells += 1;
    cset->nnodes += 3;
}

static void cellset_add_quad(
    Cellset *cset, Vertex *v1, Vertex *v2, Vertex *v3, Vertex *v4 )
{
    if( (cset->ncells+1)*4 >= cset->vertex_alloc ) {
        cset->vertex_alloc += cellset_chunk;
        cset->vertex_conn = realloc( cset->vertex_conn,
                                     cset->vertex_alloc * sizeof( Vertex *) );
    }

    /* The pointers should still be valid as we shuffle around
     * the search tree.
     */
    cset->vertex_conn[(cset->ncells*4)+0] = v1;
    cset->vertex_conn[(cset->ncells*4)+1] = v2;
    cset->vertex_conn[(cset->ncells*4)+2] = v3;
    cset->vertex_conn[(cset->ncells*4)+3] = v4;

    cset->ncells += 1;
    cset->nnodes += 4;
}

/* Special, to help with POLYLINE */
static void cellset_add_vertex( Cellset *cset, Vertex *v )
{
    if( (cset->nnodes+1)*3 >= cset->vertex_alloc ) {
        cset->vertex_alloc += cellset_chunk;
        cset->vertex_conn = realloc( cset->vertex_conn,
                                     cset->vertex_alloc * sizeof(Vertex *) );
    }

    cset->vertex_conn[cset->nnodes] = v;
    cset->nnodes += 1;
    /* Don't increment cell counter */
}

/* 64-bit porting. Only Modified Internally */
static void cellset_add_colors( Cellset *cset, float *rgb )
{
    /* Assuming a cell has just been added. */
    xp_long where = cset->ncells-1;

    if( cset->colors_alloc == 0 ) {
        cset->colors_alloc = cellset_chunk;
        cset->colors = malloc( cset->colors_alloc * sizeof( float *) );
    }
    else if( (where+1)*3 >= cset->colors_alloc ) {
        cset->colors_alloc += cellset_chunk;
        cset->colors = realloc( cset->colors, cset->colors_alloc * sizeof(float) );
    }

    cset->colors[(where*3)+0] = rgb[0];
    cset->colors[(where*3)+1] = rgb[1];
    cset->colors[(where*3)+2] = rgb[2];
}

/* Fill in the connection list you get from FLDget_node_connect */
/* 64-bit porting. Directly Modified */
static void cellset_conn( Cellset *cset, xp_long *conn_list )
{
    xp_long i;
    for( i = 0; i < cset->nnodes; ++i ) {
        conn_list[i] = cset->vertex_conn[i]->index;
    }
}

/*
 * No node data, so we can regard this as a pure cell set field.
 * This does very little other than help manage multiple cell sets.
 */
typedef struct {
    int ncsets;
    Cellset *list;
} Mesh;

static void mesh_init( Mesh *mesh )
{
    mesh->list = NULL;
    mesh->ncsets = 0;
}

static int mesh_ncsets( Mesh *mesh )
{
    return mesh->ncsets;
}

static Cellset *mesh_get( Mesh *mesh, int index )
{
    return &(mesh->list[index]);
}

static Cellset *mesh_add( Mesh *mesh, const char *name )
{
    Cellset *cset = NULL;
    if( mesh == NULL ) return NULL;

    if( mesh->ncsets == 0 ) {
        mesh->list = malloc( sizeof( Cellset ) );
    }
    else {
        /* Not very efficient, but there shouldn't be many cellsets */
        mesh->list = realloc( mesh->list, (mesh->ncsets+1) * sizeof( Cellset ) );
    }
    cset = &(mesh->list[mesh->ncsets]);
    cellset_init( cset, name );
    mesh->ncsets += 1;
    /* Warning! this pointer can be invalidated by the next realloc. */
    return cset;
}

static void mesh_free( Mesh *mesh )
{
    if( mesh->list ) {
        int i;
        for( i = 0; i < mesh->ncsets; ++i ) {
            cellset_free( &(mesh->list[i]) );
        }
        free( mesh->list );
    }
}

#define VAL_TYPE_INT  DTYPE_INT
#define VAL_TYPE_REAL DTYPE_DOUBLE
#define VAL_TYPE_STR  DTYPE_STRING

/*
 * A few parsing utilities.
 */

static int trim_crlf( char *p )
{
    if( p == NULL )
        return 0;
    else {
        int len = (int)strlen( p );
        char *pe;
        while( len != 0 ) {
            pe = p+len-1;	/* point at end of string */
            if( pe[0] == 0xa || pe[0] == 0xd ) {
                pe[0] = 0;	/* Stomp on CR/LF */
                len--;
            }
            else break;
        }
        return len;
    }
}

typedef struct _dxf_pair {
    int group_code;
    int val_type;
    /* These last three can be thought of as a union -
     * only one will be valid at a time.
     */
    int   val_int;
    float val_real;
    char  val_str[LINE_LEN];
} dxf_pair;

/*
 * Read a pair of lines out of the dxf file.
 *
 * The lines are ALWAYS in pairs.
 *   line 1: int group code.
 *   line 2: int, float, or string value.
 */
static int read_pair( FILE *fp, dxf_pair *pair )
{
    static char line_buf1[LINE_LEN], line_buf2[LINE_LEN];
    int num_scanned, val_type, group_code;

    pair->group_code  = -1;
    pair->val_type    = -1;
    pair->val_int     = 0;
    pair->val_real    = 0.0;
    pair->val_str[0]  = 0;

 read_line1:
    if( fgets( line_buf1, LINE_LEN, fp ) == NULL ) {
        /* End of file */
        return -1;
    }

    /* Skip blank lines (actually, don't expect these in DXF) */
    if( line_buf1[0] == 0 )   goto read_line1;
    if( line_buf1[0] == 0xa ) goto read_line1;	/* LF */
    if( line_buf1[0] == 0xd ) goto read_line1;	/* CR */

    num_scanned = sscanf( line_buf1, "%d", &group_code );
    if( num_scanned != 1 )
        return -1;

    if( 0 <= group_code && group_code <= 9 )
        val_type = VAL_TYPE_STR;
    else if( 10 <= group_code && group_code <= 59 )
        val_type = VAL_TYPE_REAL;
    else if( 60 <= group_code && group_code <= 79 )
        val_type = VAL_TYPE_INT;
    else if( 210 <= group_code && group_code <= 239 )
        val_type = VAL_TYPE_REAL;
    else if( 999 == group_code )
        val_type = VAL_TYPE_STR;

    pair->group_code  = group_code;
    pair->val_type    = val_type;

/* read_line2: */
    if( fgets( line_buf2, LINE_LEN, fp ) == NULL ) {
        /* End of file */
        return -1;
    }

    if( val_type == VAL_TYPE_INT ) {
        int val_int;
        num_scanned = sscanf( line_buf2, "%d", &val_int );
        if( num_scanned == 1 )
            pair->val_int = val_int;
        else return -1;
    }
    else if( val_type == VAL_TYPE_REAL ) {
        float val_real;
        num_scanned = sscanf( line_buf2, "%f", &val_real );
        if( num_scanned == 1 )
            pair->val_real = val_real;
        else return -1;
    }
    else if( val_type == VAL_TYPE_STR ) {
        trim_crlf( line_buf2 );	/* Strip CR/LF from end */
        strcpy( pair->val_str, line_buf2 );
    }

#if 0	/* Ack, I've seen at least one file where a blank line
         * here meant empty string, so the next line was a group code.
         * In other words, its not safe the skip blank lines here.
         */
    if( line_buf2[0] == 0 )   goto read_line2;
    if( line_buf2[0] == 0xa ) goto read_line2;	/* LF */
    if( line_buf2[0] == 0xd ) goto read_line2;	/* CR */
#endif

    return 1;
}

/*
 * Read ASCII DXF files.  Only a subset of the format is supported.
 * LINE, 3DLINE, 3DFACE, POINT, and POLYLINE/VERTEX are supported.
 * It does polymeshes and polyfaces, but closed polymeshes are not handled
 * correctly. Limited color handling - it knows about only the predefined
 * colors 0 - 9. No support for BLOCK, INSERT or LAYER.  The latter would
 * be feasible to add and would improve the color handling at least somewhat.
 *
 * One cool feature is that it features efficient discovery of
 * duplicate verticies, using logic copied from the triangle reader.
 */

#define SECTION_HEADER 0
#define SECTION_TABLES 1
#define SECTION_BLOCKS 2
#define SECTION_ENTITIES 3

#define ENT_TYPE_INVALID -1
#define ENT_TYPE_POINT 0
#define ENT_TYPE_LINE  1
#define ENT_TYPE_FACE  2
#define ENT_TYPE_POLY  3

/* 64-bit porting. Only Modified Internally */
static int FUNCread_dxf( FILE *fp, OMobj_id fld_id )
{
    int i, j;
    xp_long i_w;

    int use_colors = 0;
    int color_index = DXF_DEFAULT_COLOR;	/* 7 (white) is the default */

    int section = SECTION_HEADER;
    dxf_pair pair;

    /* char obj_name[LINE_LEN]; */
    float xyz1[3], xyz2[3], xyz3[3], xyz4[4];
    xp_long conn[4]  = { 0, 0, 0, 0 };
    float rgb[3] = { 1.0, 1.0, 1.0 };

    OMobj_id cset_id;
    xp_long fld_nodes = 0;

    Vset_d vset;
    Mesh mesh;

    Cellset *cset_point = NULL;
    Cellset *cset_line = NULL;
    Cellset *cset_tri  = NULL;
    Cellset *cset_quad = NULL;
    Cellset *cset_poly = NULL;

    vset_d_init( &vset );
    mesh_init( &mesh );

    /* "Gotcha!" The realloc inside of mesh_add makes the returned
     * pointers unreliable, so add them all, then get the pointers.
     */
    mesh_add( &mesh, "point" );
    mesh_add( &mesh, "line" );
    mesh_add( &mesh, "tri"  );
    mesh_add( &mesh, "quad" );
    cset_point = mesh_get( &mesh, 0 );
    cset_line = mesh_get( &mesh, 1 );
    cset_tri  = mesh_get( &mesh, 2  );
    cset_quad = mesh_get( &mesh, 3 );

    /* This one is special, I'm just using it as a temporary buffer
     * to help deal with the various flavors of POLYLINE.
     */
    cset_poly = malloc( sizeof(Cellset) );
    cellset_init( cset_poly, NULL );

    cset_point->cell_type = CELL_TYPE_POINT;
    cset_line->cell_type = CELL_TYPE_LINE;
    cset_tri->cell_type  = CELL_TYPE_TRI;
    cset_quad->cell_type = CELL_TYPE_QUAD;

    /* obj_name[0] = 0; */

    while( 1 ) {

        if( read_pair( fp, &pair ) < 0 ) {
            /* End of file or another error */
            break;
        }

        if( pair.group_code == 2 ) {
            /*
             * More tests are possible here for robustness, specifically
             * the previous pair must be (0, SECTION)
             */
            if( strncmp( pair.val_str, "TABLES", 6 ) == 0 )
                section = SECTION_TABLES;
            if( strncmp( pair.val_str, "BLOCKS", 6 ) == 0 )
                section = SECTION_BLOCKS;
            if( strncmp( pair.val_str, "ENTITIES", 8 ) == 0 )
                section = SECTION_ENTITIES;
        }

        if( section == SECTION_ENTITIES ) {
            static int entity_type  = ENT_TYPE_INVALID;
            static int vertex_count = 0;
            static int poly_flags = 0, vertex_flags = 0;
            static int poly_N = 0, poly_M = 0;

            if( pair.group_code == 0 ) {
                /*
                 * Generally, this is the start of a new entity.
                 * Since you usually don't have pairs that explicitly mark
                 * the end of an entity, this also means the end of the
                 * previous entity.  One important exception to this rule
                 * is that a (0, SEQEND) pair ends a POLYLINE.
                 */
                /*
                 * First, finish off the current open entity.
                 */
                if( entity_type == ENT_TYPE_POINT ) {
                    cellset_add_point( cset_point,
                                       vset_d_add( &vset, xyz1 ) );
                    cellset_add_colors( cset_point, rgb );
                    entity_type = ENT_TYPE_INVALID;
                }
                else if( entity_type == ENT_TYPE_LINE ) {
                    cellset_add_line( cset_line,
                                      vset_d_add( &vset, xyz1 ),
                                      vset_d_add( &vset, xyz2 ) );
                    cellset_add_colors( cset_line, rgb );
                    entity_type = ENT_TYPE_INVALID;
                }
                else if( entity_type == ENT_TYPE_FACE ) {
                    if( xyz3[0] == xyz4[0] &&
                        xyz3[1] == xyz4[1] &&
                        xyz3[2] == xyz4[2] )
                    {
                        /* Last two points identical, thus its a tri */
                        cellset_add_tri( cset_tri,
                                         vset_d_add( &vset, xyz1 ),
                                         vset_d_add( &vset, xyz2 ),
                                         vset_d_add( &vset, xyz3 ) );
                        cellset_add_colors( cset_tri, rgb );
                        entity_type = ENT_TYPE_INVALID;
                    }
                    else {
                        cellset_add_quad( cset_quad,
                                          vset_d_add( &vset, xyz1 ),
                                          vset_d_add( &vset, xyz2 ),
                                          vset_d_add( &vset, xyz3 ),
                                          vset_d_add( &vset, xyz4 ) );
                        cellset_add_colors( cset_quad, rgb );
                        entity_type = ENT_TYPE_INVALID;
                    }
                }
                else if( entity_type == ENT_TYPE_POLY && vertex_count > 0 ) {
                    /* Finish off current VERTEX */
                    if( (poly_flags&64) &&
                        (vertex_flags&128) && !(vertex_flags&64) ) {
                        /* This is crucial; its a dummy vertex that has
                         * connection information in its 71,72,73,74 codes.
                         */
                        void *v1 = NULL, *v2 = NULL, *v3 = NULL;
                        if( 0 < conn[0] && conn[0] < cset_poly->nnodes )
                            v1 = cset_poly->vertex_conn[conn[0]];
                        if( 0 < conn[1] && conn[1] < cset_poly->nnodes )
                            v2 = cset_poly->vertex_conn[conn[1]];
                        if( 0 < conn[2] && conn[2] < cset_poly->nnodes )
                            v3 = cset_poly->vertex_conn[conn[2]];

                        if( conn[2] == conn[3] ) {	/* Its a Tri */
                            if( v1 && v2 && v3 ) {
                                cellset_add_tri( cset_tri, v1, v2, v3 );
                                cellset_add_colors( cset_tri, rgb );
                            }
                        }
                        else {	/* Its a Quad */
                            void *v4 = NULL;
                            if( 0 < conn[3] && conn[3] < cset_poly->nnodes )
                                v4 = cset_poly->vertex_conn[conn[3]];
                            if( v1 && v2 && v3 && v4 ) {
                                cellset_add_quad( cset_quad, v1, v2, v3, v4 );
                                cellset_add_colors( cset_quad, rgb );
                            }
                        }
                    }
                    else {
                        /* Normal vertex */
                        cellset_add_vertex( cset_poly,
                                            vset_d_add( &vset, xyz1 ) );
                    }
                }

                /*
                 * The special POLYLINE end marker.
                 */
                if( entity_type == ENT_TYPE_POLY && vertex_count > 0 &&
                    strncmp( pair.val_str, "SEQEND", 6 ) == 0 ) {
                    /*
                     * End of a vertex sequence.
                     *
                     * Lots of fun here.  We might have a POLYLINE that
                     * really is a polyline.  The duplicate-vertex removal
                     * logic means we don't have control over the exact order
                     * of the verticies, so we cannot use FLD.Polyline.
                     * Thus we use plain-old FLD.Line cell set.
                     * Or we might have a POLYLINE that really a mesh
                     * made up of quads.  Or it might be the "polyface"
                     * variety of POLYLINE that is just another way to specify
                     * Quads and/or Tris but with explicit connection
                     * information (helps with the duplicate vertex issue).
                     */
                    if( poly_flags&16 ) {
                        /* Polymesh case */
                        for( i = 0; i < poly_M-1; ++i ) {
                            for( j = 0; j < poly_N-1; ++j ) {
                                int i1, i2, i3, i4;

                                /* Not quite sure of the best order */
                                i1 = j + (i*poly_N);
                                i2 = j + (i*poly_N) +     poly_N;
                                i3 = j + (i*poly_N) + 1 + poly_N;
                                i4 = j + (i*poly_N) + 1;
#if 0
                                fprintf( stderr, "V %d %d %d %d\n",
                                         i1, i2, i3, i4 );
#endif
                                cellset_add_quad( cset_quad,
                                    cset_poly->vertex_conn[i1],
                                    cset_poly->vertex_conn[i2],
                                    cset_poly->vertex_conn[i3],
                                    cset_poly->vertex_conn[i4] );
                                cellset_add_colors( cset_quad, rgb );
                            }
                        }
                    }
                    else if( poly_flags&64 ) {
                        /* Polyface case.  This is a no-op, because the
                         * faces get added in by one, as we see the dummy
                         * VERTEXs with connection info.
                         */
                    }
                    else {
                        /* polyline */
                        for( i_w = 1; i_w < cset_poly->nnodes; ++i_w ) {
                            void *v1 = cset_poly->vertex_conn[i_w-1];
                            void *v2 = cset_poly->vertex_conn[i_w];
                            cellset_add_line( cset_line, v1, v2 );
                            cellset_add_colors( cset_line, rgb );
                        }
                        if( poly_flags&1 ) {	/* closed polyline */
                            void *v1 = cset_poly->vertex_conn[i_w-1];
                            void *v2 = cset_poly->vertex_conn[0];
                            cellset_add_line( cset_line, v1, v2 );
                            cellset_add_colors( cset_line, rgb );
                        }
                    }

                    /* reset for the next poly */
                    cset_poly->nnodes = 0;
                    cset_poly->ncells = 0;
                    vertex_count = 0;
                }

                /*
                 * Look for the start of various entities that
                 * we recognize.
                 */
                if( strncmp( pair.val_str, "LINE", 4 ) == 0 ||
                    strncmp( pair.val_str, "3DLINE", 6 ) == 0 ) {
                    entity_type = ENT_TYPE_LINE;
                }
                else if( strncmp( pair.val_str, "3DFACE", 6 ) == 0 ) {
                    entity_type = ENT_TYPE_FACE;
                }
                else if( strncmp( pair.val_str, "POINT", 5 ) == 0 ) {
                    entity_type = ENT_TYPE_POINT;
                }
                else if( strncmp( pair.val_str, "POLYLINE", 8 ) == 0 ) {
                    /*
                     * case 1:  It really is a "polyline".
                     * case 2:  It is a M x N quad mesh.
                     * case 3:  "polyface" (bitval 64 is on)
                     */
                    entity_type = ENT_TYPE_POLY;
                    vertex_count = 0;
                }
                else if( strncmp( pair.val_str, "VERTEX", 6 ) == 0 ) {
                    vertex_count += 1;
                }

                /*
                 * TODOs:
                 *
                 * How do TRACE and SOLID differ from 3DFACE ?
                 * "These entities are planar in nature.  All points
                 *  are expressed in Entity coordinates."
                 */
            } /* group_code == 0 */
            else if( pair.group_code == 2 ) {
                /* a name  of some sort, often a block */
                /* fprintf( stderr, "Name %s\n", pair.val_str ); */
            }
            else if( pair.group_code == 8 ) {
                /* layer name */
                /* fprintf( stderr, "Layer %s\n", pair.val_str ); */
            }
            else if( 10 <= pair.group_code && pair.group_code <= 33 ) {
                /* Read in a X,Y, or Z coordinate */
                int index = (pair.group_code/10) - 1;	/* x,y, or z */
                if( pair.group_code%10 == 0 )
                    xyz1[index] = pair.val_real;
                else if( pair.group_code%10 == 1 )
                    xyz2[index] = pair.val_real;
                else if( pair.group_code%10 == 2 )
                    xyz3[index] = pair.val_real;
                else if( pair.group_code%10 == 3 )
                    xyz4[index] = pair.val_real;
            }
            else if( pair.group_code == 62 ) {
                /* color index */
                color_index = pair.val_int;

                if( color_index < 0 || 9 < color_index ) {
                    color_index = DXF_DEFAULT_COLOR;
                }
                if( color_index != DXF_DEFAULT_COLOR ) {
                    /* Save space in the field by not writing out color info
                     * if we never see anything but the default color.
                     */
                    use_colors = 1;
                }
                rgb[0] = dxf_colors[color_index][0];
                rgb[1] = dxf_colors[color_index][1];
                rgb[2] = dxf_colors[color_index][2];
            }
            else if( pair.group_code == 70 ) {
                /* flags (bitfield). Exact meaning depends on context ... */

                /* polygon
                 *  1 - if polyline closed, if polygon closed in M direction
                 *  8 - polyline (default?)
                 * 16 - polygons
                 * 32 - if polygon closed in N direction
                 * 64 - Is a polyface mesh
                 */
                if( entity_type == ENT_TYPE_POLY && vertex_count == 0 ) {
                    poly_flags = pair.val_int;
                }
                /* vertex (sometimes comes after the xyz)
                 *  1 - extra vertex created by curve fitting
                 * 32 - part of polyline
                 * 64 - part of 3D mesh
                 * 128 - polyface mesh vertex
                 */
                else if( entity_type == ENT_TYPE_POLY && vertex_count > 0 ) {
                    vertex_flags = pair.val_int;
                }
            }
            /* Inside a VERTEX, 71,72,73,74 mean polyface index */
            else if( pair.group_code == 71 && entity_type == ENT_TYPE_POLY ) {
                /* If polyface, "the number of vertices in the mesh" */
                if( poly_flags&16 && vertex_count == 0 ) {
                    poly_M = pair.val_int;
                }
                else if( poly_flags&64 && vertex_count == 0 ) {
                    /* fprintf( stderr, "Polyface num verts %d\n", pair.val_int ); */
                }
                else if( poly_flags&64 && vertex_count > 0 ) {
                    /* face index (1 based) */
                    if( pair.val_int < 0 )
                        conn[0] = -pair.val_int - 1;
                    else
                        conn[0] = pair.val_int - 1;
                }
            }
            else if( pair.group_code == 72 && entity_type == ENT_TYPE_POLY ) {
                /* If polyface, "the number faces" */
                if( poly_flags&16 && vertex_count == 0 ) {
                    poly_N = pair.val_int - 1;
                }
                else if( poly_flags&64 && vertex_count == 0 ) {
                    /* fprintf( stderr, "Polyface num faces %d\n", pair.val_int ); */
                }
                else if( poly_flags&64 && vertex_count > 0 ) {
                    if( pair.val_int < 0 )
                        conn[1] = -pair.val_int - 1;
                    else
                        conn[1] = pair.val_int - 1;
                }
            }
            else if( pair.group_code == 73 && entity_type == ENT_TYPE_POLY ) {
                if( poly_flags&64 && vertex_count > 0 ) {
                    if( pair.val_int < 0 )
                        conn[2] = -pair.val_int - 1;
                    else
                        conn[2] = pair.val_int - 1;
                    /* This is a trick to help distinguish Tris from Quads */
                    conn[3] = conn[2];
                }
            }
            else if( pair.group_code == 74 && entity_type == ENT_TYPE_POLY ) {
                if( poly_flags&64 && vertex_count > 0 ) {
                    if( pair.val_int < 0 )
                        conn[3] = -pair.val_int - 1;
                    else
                        conn[3] = pair.val_int - 1;
                }
            }
        }

    } /* while (loop over lines in file) */

    /* The file has been read, now put the info in the field. */

    /* This overwrites the priorities with the index counters, but
     * the priorities are no longer needed. (makes debugging harder ... )
     */
    fld_nodes = vset_d_count( &vset );
    FLDset_nnodes( fld_id, fld_nodes );

    /* Field coordinate list */
    {
        float *fld_coords;
        xp_long size;
        FLDget_coord( fld_id, &fld_coords, &size, OM_GET_ARRAY_RW );
        if( fld_coords != NULL ) {
            vset_d_coords( &vset, fld_coords );
            ARRfree( fld_coords );
        }
    }

    /* Each cell set */
    for( i = 0, j = 0; i < mesh_ncsets( &mesh ); ++i ) {
        Cellset *cset;
        xp_long *conn_list;
        xp_long size;

        cset = mesh_get( &mesh, i );

#if 0
        fprintf( stderr, "CS:%d, type:%d, cells: %ld, nodes: %ld\n",
                 i, cset->cell_type, cset->ncells, cset->nnodes );
#endif

        if( cset->ncells == 0 ) {
            cellset_free( cset );
            continue;
        }

        /* Create field cell set */

        if( cset->cell_type == CELL_TYPE_LINE )
            FLDadd_cell_set( fld_id, "Line" );
        else if( cset->cell_type == CELL_TYPE_TRI )
            FLDadd_cell_set( fld_id, "Tri" );
        else if( cset->cell_type == CELL_TYPE_QUAD )
            FLDadd_cell_set( fld_id, "Quad" );
        else {
            cellset_free( cset );
            continue;	/* Should not happen */
        }

        j += 1;	/* Count cell sets actually created */

        FLDget_cell_set( fld_id, j-1, &cset_id );

        FLDset_ncells( cset_id, cset->ncells );

        /* Cell set connection list */
        FLDget_node_connect( cset_id, &conn_list, &size, OM_GET_ARRAY_RW );
        if( conn_list != NULL ) {
            cellset_conn( cset, conn_list );
            ARRfree( conn_list );
        }

        /* Cell set name (optional) */
        if( cset->name != NULL ) {
            FLDset_cell_set_user_name( cset_id, cset->name );
        }

        /* Cell data */
        if( use_colors != 0 && cset->colors != NULL ) {
            float *cell_data = NULL;
            int type = DTYPE_FLOAT;

            FLDset_cell_data_ncomp( cset_id, 1 );
            FLDset_cell_data_veclen( cset_id, 0, 3 );
            FLDset_cell_data_id( cset_id, 0, GD_COLOR_DATA_ID );
            FLDset_cell_data_type( cset_id, 0, DTYPE_FLOAT );

            FLDget_cell_data( cset_id, 0, &type, (char **)&cell_data,
                              &size, OM_GET_ARRAY_RW );
            if( cell_data != NULL ) {
                memcpy( cell_data, cset->colors,
                        cset->ncells * 3 * sizeof(float) );
                ARRfree( cell_data );
            }
        }
        else FLDset_cell_data_ncomp( cset_id, 0 );
    }

    vset_d_free( &vset );
    cellset_free( cset_poly );	/* Never added to the mesh */
    free( cset_poly );
    mesh_free( &mesh );		/* Also takes care of child cellsets */

    return METHOD_SUCCESS;
}
