/*
			Copyright (c) 1994 by
			Advanced Visual Systems Inc.
			All Rights Reserved

	This software comprises unpublished confidential information of
	Advanced Visual Systems Inc. and may not be used, copied or made
	available to anyone, except in accordance with the license
	under which it is furnished.

	This file is under Perforce control
	$Id: //depot/express/fcs70/modules/rd_image.c#1 $
*/

#define XP_WIDE_API	/* Use Wide APIs */

#include <stdio.h>
#include <string.h>

#include <avs/config.h> /* pick up NO_DL_LOAD on some platforms */
#include <avs/util.h>
#include <avs/err.h>
#include <avs/om.h>
#include <avs/fld.h>
#include <avs/gd.h>
#include <avs/f_utils.h>

#include "image.h"

/* protos for funcs private to this file */
static int FUNCset_image_field (int, int, unsigned char **, OMobj_id);
static void FUNCmessage (const char *);
static int FUNCread_image (void *, char *, int, OMobj_id, int *, int *);
static int FUNCread_wrapup (const char *, byte *,
                            void *(*)(), void *, void *);

#define ERR_RETURN(A) {ERRerror( "DVread image", 0, ERR_ORIG, A ); return(0);}

#ifndef NO_DL_LOAD

/* for dynamic, loaded at runtime, libs make and use an array of info strings
 * containing the library and function names: these get looked up
 */

/* indices into reader info tables below */
#define LIBNAME         0
#define OPEN_FUNC       1
#define CLOSE_FUNC      2
#define GETWIDTH_FUNC   3
#define GETHEIGHT_FUNC  4
#define GETARGB_FUNC    5

static const char *avs_info[] =
{
#ifdef MSDOS
  "libavsx",
#else
  "libavsx." DLLIB_SUFFIX,
#endif
  "avsOpen", "avsClose", "avsGetWidth", "avsGetHeight", "avsGetARGBImage"
};

static const char *bmp_info[] =
{
#ifdef MSDOS
  "libbmp",
#else
  "libbmp." DLLIB_SUFFIX,
#endif
  "bmpOpen", "bmpClose", "bmpGetWidth", "bmpGetHeight", "bmpGetARGBImage"
};

static const char *gif_info[] =
{
#ifdef MSDOS
  "libgif",
#else
  "libgif." DLLIB_SUFFIX,
#endif
  "GIFOpen", "GIFClose", "GIFGetWidth", "GIFGetHeight", "GIFGetARGBImage"
};

static const char *jpeg_info[] =
{
#ifdef MSDOS
  "libjpeg",
#else
  "libjpeg." DLLIB_SUFFIX,
#endif
  "jpegOpen", "jpegClose", "jpegGetWidth", "jpegGetHeight", "jpegGetARGBImage"
};

static const char *pbm_info[] =
{
#ifdef MSDOS
  "libpbm",
#else
  "libpbm." DLLIB_SUFFIX,
#endif
  "pbmOpen", "pbmClose", "pbmGetWidth", "pbmGetHeight", "pbmGetARGBImage"
};

static const char *sgiRGB_info[] =
{
#ifdef MSDOS
  "libsgiim",
#else
  "libsgiim." DLLIB_SUFFIX,
#endif
  "sgiRGBOpen", "sgiRGBClose", "sgiRGBGetWidth", "sgiRGBGetHeight", "sgiRGBGetARGBImage"
};

static const char *sunras_info[] =
{
#ifdef MSDOS
  "libsunrs",
#else
  "libsunrs." DLLIB_SUFFIX,
#endif
  "SunRasOpen", "SunRasClose", "SunRasGetWidth", "SunRasGetHeight", "SunRasGetARGBImage"
};

static const char *tiff_info[] =
{
#ifdef MSDOS
  "libtif",
#else
  "libtif." DLLIB_SUFFIX,
#endif
  "TIFFBegin", "TIFFEnd", "TIFFGetWidth", "TIFFGetHeight", "TIFFGetARGBImage"
};

#else

void *NULLfunc_voidp() { return((void *) NULL); }
void NULLfunc_void() { return; }
int NULLfunc_int() { return(0); }

static rd_funcs_t avs_funcs = { avsOpen, avsClose,
			     avsGetWidth, avsGetHeight, avsGetARGBImage };

static rd_funcs_t bmp_funcs = { bmpOpen, bmpClose,
			     bmpGetWidth, bmpGetHeight, bmpGetARGBImage };

static rd_funcs_t gif_funcs = { GIFOpen, GIFClose,
			     GIFGetWidth, GIFGetHeight, GIFGetARGBImage };

static rd_funcs_t jpeg_funcs = { jpegOpen, jpegClose,
			      jpegGetWidth, jpegGetHeight, jpegGetARGBImage };

static rd_funcs_t pbm_funcs = { pbmOpen, pbmClose,
			     pbmGetWidth, pbmGetHeight, pbmGetARGBImage };

static rd_funcs_t sgi_funcs = { sgiRGBOpen, sgiRGBClose,
			     sgiRGBGetWidth, sgiRGBGetHeight,
			     sgiRGBGetARGBImage };

static rd_funcs_t sun_funcs = { SunRasOpen, SunRasClose,
			     SunRasGetWidth, SunRasGetHeight,
			     SunRasGetARGBImage };

static rd_funcs_t tiff_funcs = { TIFFBegin, TIFFEnd,
			      TIFFGetWidth, TIFFGetHeight, TIFFGetARGBImage };

#endif

/* The order of these MUST match the UI radiobox order */
DV_image_read_map_t FUNCimage_readers_map[DV_IMAGE_READERS_COUNT] = {
  { DV_IMAGE_FORMAT_AVSX,		/* 0 */
#ifndef NO_DL_LOAD
    avs_info
#else
    &avs_funcs
#endif
  },
  { DV_IMAGE_FORMAT_BMP,		/* 1 */
#ifndef NO_DL_LOAD
    bmp_info
#else
    &bmp_funcs
#endif
  },
  { DV_IMAGE_FORMAT_GIF,		/* 2 */
#ifndef NO_DL_LOAD
    gif_info
#else
    &gif_funcs
#endif
  },
  { DV_IMAGE_FORMAT_JPEG,		/* 3 */
#ifndef NO_DL_LOAD
    jpeg_info
#else
    &jpeg_funcs
#endif
  },
  { DV_IMAGE_FORMAT_PBM,		/* 4 */
#ifndef NO_DL_LOAD
    pbm_info
#else
    &pbm_funcs
#endif
  },
  { DV_IMAGE_FORMAT_SGIRGB,		/* 5 */
#ifndef NO_DL_LOAD
    sgiRGB_info
#else
    &sgi_funcs
#endif
  },
  { DV_IMAGE_FORMAT_SUNRAS,		/* 6 */
#ifndef NO_DL_LOAD
    sunras_info
#else
    &sun_funcs
#endif
  },
  { DV_IMAGE_FORMAT_TIFF,		/* 7 */
#ifndef NO_DL_LOAD
    tiff_info
#else
    &tiff_funcs
#endif
  }
};

/*----------------------------------------------------------------------
 * the public entry point (update method in v/dv.v)
 */
int DVread_image_update(OMobj_id elem_id)
{
  OMobj_id file_id, info_id, out;
  char *filename = NULL;
  int rb_val, from_format, flip, format, result;

  file_id = OMfind_subobj(elem_id, OMstr_to_name("filename"), OM_OBJ_RD);
  if (OMis_null_obj(file_id) ||
      OMget_str_val(file_id, &filename, 0) != OM_STAT_SUCCESS)
    return(0);

  if (OMget_name_int_val(elem_id,  OMstr_to_name("flip"), &flip)
							!= OM_STAT_SUCCESS)
    flip = 1;
  if (OMget_name_int_val(elem_id,  OMstr_to_name("format"), &rb_val)
							!= OM_STAT_SUCCESS)
    rb_val = 0;

  out = OMfind_subobj(elem_id, OMstr_to_name("out"), OM_OBJ_RW);

  /* map format UIradiobox val to internal def */
  if (rb_val < 0 || rb_val > DV_IMAGE_READERS_COUNT)
    format = DV_IMAGE_FORMAT_UNKNOWN;
  else if (rb_val == 0)
    format = DV_IMAGE_FORMAT_AUTO;
  else
    /* This is where the UIradiobox and the reader_map must match up. */
    format = FUNCimage_readers_map[rb_val-1].dv_val;

  /* verify that we can handle this format and that, if the user is being
   * explicit, that the format s/he has selected matches that of the file
   * itself; this way the library code doesn't have to be responsible for
   * checking the format
   */
  from_format = FUNCget_image_format(filename);

  if (format == DV_IMAGE_FORMAT_AUTO)
  {
    /* OK, consider this verified */
    format = from_format;
  }
  else if (format != from_format)
  {
    /* explicit format does not match that in the file */
    char name[16], fname[16];

    FUNCget_image_format_name(format, name);
    FUNCget_image_format_name(from_format, fname);
    ERRerror( "DVread_image", 2, ERR_ORIG,
	      "Not read: format selected is '%s', but file is of format '%s'?",
	      name, fname );
    return(0);
  }

  /* now go read the file of particular format */
  result = 0;
  switch (format)
  {
    case DV_IMAGE_FORMAT_AVSX:
    case DV_IMAGE_FORMAT_BMP:
    case DV_IMAGE_FORMAT_GIF:
    case DV_IMAGE_FORMAT_JPEG:
    case DV_IMAGE_FORMAT_PBM:
    case DV_IMAGE_FORMAT_SGIRGB:
    case DV_IMAGE_FORMAT_SUNRAS:
    case DV_IMAGE_FORMAT_TIFF:
    {
      int i, width, height;

      /* find the info corresponding to the format number */
      for (i = 0; i < DV_IMAGE_READERS_COUNT; ++i)
      {
	if (format == FUNCimage_readers_map[i].dv_val)
	{
	  result = FUNCread_image((void *)FUNCimage_readers_map[i].info,
                                  filename, flip, out,
				  &width, &height);
	  break;
	}
      }
      if (result)
      {
	char info[128];

	/* give some info about what was read */
	info_id = OMfind_subobj(elem_id, OMstr_to_name("info"), OM_OBJ_RW);

	FUNCget_image_format_name(format, info);
	sprintf(info, "%s %dx%d", info, width, height);

	OMset_str_val(info_id, info);
      }
      break;
    }

    case DV_IMAGE_FORMAT_NOFILE:
      ERRerror( "DVread_image", 1, ERR_ORIG, "Cannot open file: %s", filename);
      break;

    case DV_IMAGE_FORMAT_UNKNOWN:
      ERRerror( "DVread_image", 1, ERR_ORIG, "File is of unknown format: %s",
	        filename );
      break;

    default:
      ERRerror( "DVread_image", 0, ERR_ORIG, "File format is not supported");
      break;
  }

  if (filename)
    free(filename);

  return(result);
}

/*----------------------------------------------------------------------
 * attempts to determine the format from the file itself
 */
int FUNCget_image_format(const char *filename)
{
  FILE *fp;
  char *extension;
  unsigned char file_id[8];

  /* first open the file for existence check */
  if ((fp = (FILE *) FILEfopen(filename, SIO_R_BIN)) == NULL)
    return(DV_IMAGE_FORMAT_NOFILE);

  /* all others we try and id by their header info... */
  if ((fread(file_id, 8, 1, fp)) != 1)
  {
    /* not much we can do with < 8 bytes */
    fclose(fp);
    return(DV_IMAGE_FORMAT_UNKNOWN);
  }

  fclose(fp);

  /* try to id from the file first, then if the filename has a .x/X extension
   * guess that it's an AVS .x file, else call it unknown
   */
  if (!strncmp((char *) file_id, "GIF87a", 6) ||
      !strncmp((char *) file_id, "GIF89a", 6))
    return(DV_IMAGE_FORMAT_GIF);
  else if (file_id[0] == 0xff && file_id[1] == 0xd8 && file_id[2] == 0xff)
    return(DV_IMAGE_FORMAT_JPEG);
  else if (file_id[0] == 'P' && file_id[1] >= '1' && file_id[1] <= '6')
    return(DV_IMAGE_FORMAT_PBM);
  else if ((file_id[0] == 0x01 && file_id[1] == 0xda) ||
	   (file_id[0] == 0xda && file_id[1] == 0x01))
    return(DV_IMAGE_FORMAT_SGIRGB);
  else if (file_id[0] == 0x59 && (file_id[1] & 0x7f) == 0x26 &&
	   file_id[2] == 0x6a && (file_id[3] & 0x7f) == 0x15)
    return(DV_IMAGE_FORMAT_SUNRAS);
  else if ((file_id[0] == 'M' && file_id[1] == 'M') ||
	   (file_id[0] == 'I' && file_id[1] == 'I'))
    return(DV_IMAGE_FORMAT_TIFF);

  else if (file_id[0] == '%' && file_id[1] == '!')
    return(DV_IMAGE_FORMAT_PS);
  else if (file_id[0] == 'B' && file_id[1] == 'M')
    return(DV_IMAGE_FORMAT_BMP);
  else if (file_id[0] == 0x52 && file_id[1] == 0xcc)
    return(DV_IMAGE_FORMAT_UTAH);
  else if (file_id[0] == 0x1f && file_id[1] == 0x9d)
    return(DV_IMAGE_FORMAT_Z);
  else if (file_id[0] == 0x1f && file_id[1] == 0x8b)
    return(DV_IMAGE_FORMAT_GZIP);

  else if ((extension = strrchr(filename, '.')) &&
	   (!strcmp(&extension[1], "x") || !strcmp(&extension[1], "X")))
    return(DV_IMAGE_FORMAT_AVSX);

  else
    return(DV_IMAGE_FORMAT_UNKNOWN);
}

/*----------------------------------------------------------------------
 * returns the ASCII equiv of an integer format
 */
int FUNCget_image_format_name(int format, char *name)
{
  int result = 1;
  switch (format)
  {
    case DV_IMAGE_FORMAT_AVSX:
      sprintf(name, "%s", "AVSX");
      break;
    case DV_IMAGE_FORMAT_GIF:
      sprintf(name, "%s", "GIF");
      break;
    case DV_IMAGE_FORMAT_JPEG:
      sprintf(name, "%s", "JPEG");
      break;
    case DV_IMAGE_FORMAT_PBM:
      sprintf(name, "%s", "PBM");
      break;
    case DV_IMAGE_FORMAT_SUNRAS:
      sprintf(name, "%s", "Sun Raster");
      break;
    case DV_IMAGE_FORMAT_SGIRGB:
      sprintf(name, "%s", "SGI Image");
      break;
    case DV_IMAGE_FORMAT_TIFF:
      sprintf(name, "%s", "TIFF");
      break;
    case DV_IMAGE_FORMAT_PS:
      sprintf(name, "%s", "PostScript");
      break;
    case DV_IMAGE_FORMAT_BMP:
      sprintf(name, "%s", "BMP");
      break;
    case DV_IMAGE_FORMAT_UTAH:
      sprintf(name, "%s", "Utah RLE");
      break;
    case DV_IMAGE_FORMAT_Z:
      sprintf(name, "%s", "compressed");
      break;
    case DV_IMAGE_FORMAT_GZIP:
      sprintf(name, "%s", "gzip compressed");
      break;
    default:
      sprintf(name, "%s", "<unknown>");
      result = 0;
      break;
  }
  return(result);
}

/*----------------------------------------------------------------------
 * composes a field of things image
 */
/* 64-bit porting. Only Modified Internally */
static int FUNCset_image_field(int w, int h, unsigned char **data, 
                               OMobj_id out)
{
  xp_long dims[2], size;
  int type;
  float *points;

  dims[0] = w;  dims[1] = h;

  if (FLDset_rspace(out, 2) != 1)
    ERR_RETURN("Error setting rspace");
  if (FLDset_nnodes (out, w*h) != 1)
    ERR_RETURN("Error setting nnodes");
  if (FLDset_dims(out, dims) != 1)
    ERR_RETURN("Error setting dims");
  if (FLDget_points(out, &points, &size, OM_GET_ARRAY_WR) != 1)
    ERR_RETURN("Error getting points");

  points[0] = points[1] = 0.0;
  points[2] = (float) (w - 1);
  points[3] = (float) (h - 1);

  if (FLDset_node_data_ncomp(out, 1) != 1)
    ERR_RETURN("Error setting node ncomps");
  if (FLDset_node_data_id(out, 0, GD_RGB_DATA_ID) != 1)
    ERR_RETURN("Error setting comp id");
  if (FLDset_node_data_veclen(out, 0, 4) != 1)
    ERR_RETURN("Error setting comp veclen");

  type = DTYPE_BYTE;
  if (FLDget_node_data(out, 0, &type, (char **) data, &size,
		       OM_GET_ARRAY_WR) != 1)
    ERR_RETURN("Error getting node data array");

  ARRfree(points);

  return(1);
}

/*----------------------------------------------------------------------
 * pops up a dialog with a message
 */
static void FUNCmessage(const char *msg)
{
#if 0
  static int mbox = 0;
  static OMobj_id msg_id;

  if (!mbox)
  {
    msg_id = OMcreate_obj_from_path("UImessageBox", "Read Image", OMinst_obj);
    mbox = 1;
  }

  if (!(OMis_null_obj(msg_id)))
  {
    OMset_name_str_val(msg_id, OMstr_to_name("message"), msg);
    OMset_name_int_val(msg_id, OMstr_to_name("visible"), 1);
  }
#endif

  ERRerror( "DVread image", 0, ERR_ORIG, msg );

  return;
}

/*----------------------------------------------------------------------
 * generic image reader; "NO_DL_LOAD" means we don't do the dynamic library/dll
 * open and lookup but just call the funcs from an assumed-to-be filled-in
 * table
 */
static int FUNCread_image(void *access, char *filename, int flip, 
                          OMobj_id out, int *width, int *height)
{
  byte *node_data = NULL, *data = NULL;
  int w, h, line_inc;
  void *priv = NULL, *(*fptr)() = NULL, *(*close)() = NULL, *handle = NULL;
#ifndef NO_DL_LOAD
  char **info = (char **) access;
#else
  rd_funcs_t *funcs = (rd_funcs_t *) access;
#endif

  *width = *height = 0;
  if (!filename || strlen(filename) == 0)
    return(0);

  /* open the lib */
#ifndef NO_DL_LOAD

  if (info[LIBNAME] != NULL)
    DL_OPEN(info[LIBNAME], DL_BIND, handle);
  else
    return(FUNCread_wrapup("No library name to load?",
			   node_data, close, priv, handle));

  if (handle == NULL)
  {
    char err[512];
#ifdef MSDOS
    const char *libpath = "PATH";
    DWORD last_err = GetLastError();
#elif defined(hpux) || defined(__hpux)
    const char *libpath = "SHLIB_PATH";
#elif defined(__APPLE__) || defined(__MACH__)
    const char *libpath = "DYLD_LIBRARY_PATH";
#else
    const char *libpath = "LD_LIBRARY_PATH";
#endif

    sprintf(err, "Cannot open library '%s'.  Most likely cause is an unset or incorrectly set '%s' environment variable: please see the System Prerequisites Document.  The system error message is:\n'%s'\n", info[LIBNAME], libpath, DL_ERROR());

#ifdef MSDOS
    sprintf(err, "%s and GetLastError number is %d", err, last_err);
#endif

    return(FUNCread_wrapup(err, node_data, close, priv, handle));
  }
#endif

  /* lookup the Open function */
#ifndef NO_DL_LOAD
  DL_LOOKUP(handle, info[OPEN_FUNC], void *(*)(), fptr);
  if (fptr == NULL)
    return(FUNCread_wrapup("Cannot find Open function in library",
			   node_data, close, priv, handle));
#else
  fptr = (void *(*)()) funcs->Open;
#endif

  /* and open the file */
  if (!(priv = (*fptr)(filename, DV_IMAGE_FILE_ACCESS_READ)))
    return(FUNCread_wrapup("Cannot open file",
			   node_data, close, priv, handle));

  /* lookup the Close function now for err return */
#ifndef NO_DL_LOAD
  DL_LOOKUP(handle, info[CLOSE_FUNC], void *(*)(), close);
  if (close == NULL)
    return(FUNCread_wrapup("Cannot find Close function in library",
			   node_data, close, priv, handle));
#else
  close = (void *(*)()) funcs->Close;
#endif

  /* lookup the GetWidth function */
#ifndef NO_DL_LOAD
  DL_LOOKUP(handle, info[GETWIDTH_FUNC], void *(*)(), fptr);
  if (fptr == NULL)
    return(FUNCread_wrapup("Cannot find GetWidth function in library",
			   node_data, close, priv, handle));
#else
  fptr = (void *(*)()) funcs->GetWidth;
#endif

  /* and get the width */
  (*fptr)(priv, &w);

  /* lookup the GetHeight function */
#ifndef NO_DL_LOAD
  DL_LOOKUP(handle, info[GETHEIGHT_FUNC], void *(*)(), fptr);
  if (fptr == NULL)
    return(FUNCread_wrapup("Cannot find GetHeight function in library",
			   node_data, close, priv, handle));
#else
  fptr = (void *(*)()) funcs->GetHeight;
#endif

  /* and get the height */
  (*fptr)(priv, &h);

  /* lookup the GetARGBImage function */
#ifndef NO_DL_LOAD
  DL_LOOKUP(handle, info[GETARGB_FUNC], void *(*)(), fptr);
  if (fptr == NULL)
    return(FUNCread_wrapup("Cannot find GetARGBImage function in library",
			   node_data, close, priv, handle));
#else
  fptr = (void *(*)()) funcs->GetARGBImage;
#endif

  /* and get the image data */
  if (!((*fptr)(priv, &data)))
    return(FUNCread_wrapup("Failure accessing ARGB data",
			   node_data, close, priv, handle));

  /* setup the field data info */
  if (!FUNCset_image_field(w, h, &node_data, out))
    return(FUNCread_wrapup("Failure setting field data",
			   node_data, close, priv, handle));

  /* copy data to field */
  line_inc = w << 2;
  if (flip)
    memcpy(node_data, data, h * line_inc);
  else
  {
    int i;
    byte *line_ptr = data;
    byte *ndata = node_data + ((h - 1) * line_inc);
    for (i = 0; i < h; i++, line_ptr += line_inc, ndata -= line_inc)
      memcpy(ndata, line_ptr, line_inc);
  }

  *width = w;  *height = h;

  FUNCread_wrapup(NULL, node_data, close, priv, handle);

  return(1);
}

/*----------------------------------------------------------------------
 * what to do at the end of (or on error when) reading the file
 */
static int FUNCread_wrapup(const char *msg, byte *data,
                           void *(*close)(), void *priv, void *handle)
{
  /* close the file */
  if (close && priv)
   (*close)(priv);

  /* free the field data */
  if (data)
    ARRfree(data);

#ifndef NO_DL_LOAD
  /* close the library */
  if (handle)
    DL_CLOSE(handle);
#endif

  /* output message and return 0 if error */
  if (msg)
  {
    FUNCmessage(msg);
    return(0);
  }
  else
    return(1);
}
