/*
			Copyright (c) 2001 by
			Advanced Visual Systems Inc.
			All Rights Reserved

	This software comprises unpublished confidential information of
	Advanced Visual Systems Inc. and may not be used, copied or made
	available to anyone, except in accordance with the license
	under which it is furnished.

	This file is under Perforce control
	$Id: //depot/express/fcs70/modules/rd_img2vol.c#1 $
*/

#define XP_WIDE_API	/* Use Wide APIs */

#include <stdio.h>
#include <string.h>
#include <sys/types.h>  /* time_t */

#include <avs/config.h> /* pick up NO_DL_LOAD on some platforms */
#include <avs/util.h>
#include <avs/err.h>
#include <avs/f_utils.h>
#include <avs/arr.h>
#include <avs/om.h>
#include <avs/fld.h>
#include <avs/gd_def.h>	/* pick up GD_RGB_DATA_ID */

#include "image.h"

/* protos for funcs private to this file */
static int FUNCcheck_image_formats(const char *, int, int, int);

static int FUNCread_volume (char *, int, int, int, int, OMobj_id);
static int FUNCget_image_funcs(void *);
static int FUNCset_volume_field (int, int, int, int, OMobj_id);
static int FUNCread_image_dims (char *, int *, int *);
static int FUNCread_image (char *, int, int, int, void *);
static int FUNCread_wrapup (const char *, void *);


#define ERR_RETURN(A) {\
    ERRerror( "read_img2vol", 0, ERR_ORIG, A ); \
    if( filename != NULL ) free( filename ); \
    return 0; \
}

#ifndef NO_DL_LOAD
/* for dynamic, loaded at runtime, libs make and use an array of info strings
 * containing the library and function names: these get looked up
 */

/* indices into reader info tables (FUNCimage_reader_map) */
#define LIBNAME         0
#define OPEN_FUNC       1
#define CLOSE_FUNC      2
#define GETWIDTH_FUNC   3
#define GETHEIGHT_FUNC  4
#define GETARGB_FUNC    5
#endif

/*
 * Definitions for types of output formats.
 */
#define NUM_OUT_FORMATS 7

#define  ARGB_OUT_FORMAT 0
#define ALPHA_OUT_FORMAT 1
#define   RED_OUT_FORMAT 2
#define GREEN_OUT_FORMAT 3
#define  BLUE_OUT_FORMAT 4
#define  LUMI_OUT_FORMAT 5
#define  LUMI_OUT_FORMAT_BYTE 6


int
DVread_img2vol_update(OMobj_id elem_id, OMevent_mask event_mask, int seq_num)
{
  OMobj_id file_id;
  char *filename = NULL;
  int rb_val, format, result;
  int start, end, out_format;

  file_id = OMfind_subobj(elem_id, OMstr_to_name("filename"), OM_OBJ_RD);
  if (OMis_null_obj(file_id) || OMget_str_val(file_id, &filename, 0) != OM_STAT_SUCCESS)
    return(0);

  if (strchr(filename, '%')==NULL)
    ERR_RETURN("Filename does not contain a format specifier");

  if (OMget_name_int_val(elem_id, OMstr_to_name("format"), &rb_val) != OM_STAT_SUCCESS)
    rb_val = 0;

  /* map format UIradiobox val to internal def */
  if (rb_val < 0 || rb_val > DV_IMAGE_READERS_COUNT)
    format = DV_IMAGE_FORMAT_UNKNOWN;
  else if (rb_val == 0)
    format = DV_IMAGE_FORMAT_AUTO;
  else
    /* This is where the UIradiobox and the reader_map must match up. */
    format = FUNCimage_readers_map[rb_val-1].dv_val;


  if (OMget_name_int_val(elem_id, OMstr_to_name("start"), &start) != OM_STAT_SUCCESS)
    ERR_RETURN("Image start index is not valid.");

  if (OMget_name_int_val(elem_id, OMstr_to_name("end"), &end) != OM_STAT_SUCCESS)
    ERR_RETURN("Image end index is not valid.");

  if (start>=end)
    ERR_RETURN("Starting image index must be greater than ending image index.");

  if (OMget_name_int_val(elem_id, OMstr_to_name("out_format"), &rb_val) != OM_STAT_SUCCESS)
    rb_val = LUMI_OUT_FORMAT;

  /* map format UIradiobox val to internal def */
  if (rb_val < 0 || rb_val >= NUM_OUT_FORMATS)
    out_format = ARGB_OUT_FORMAT;
  else
    out_format = rb_val;


  /* Check all files to ensure that they exist and that they are all of the
   * same format.  If we haven't explicitly stated a format then store the
   * returned format.
   */
  format = FUNCcheck_image_formats(filename, start, end, format);


  /* now go read the file of particular format */
  result = 0;
  switch (format)
  {
    case DV_IMAGE_FORMAT_AVSX:
    case DV_IMAGE_FORMAT_BMP:
    case DV_IMAGE_FORMAT_GIF:
    case DV_IMAGE_FORMAT_JPEG:
    case DV_IMAGE_FORMAT_PBM:
    case DV_IMAGE_FORMAT_SGIRGB:
    case DV_IMAGE_FORMAT_SUNRAS:
    case DV_IMAGE_FORMAT_TIFF:
    {
      result = FUNCread_volume(filename, format, start, end, out_format, elem_id);
      break;
    }

    case DV_IMAGE_FORMAT_NOFILE: break;

    case DV_IMAGE_FORMAT_UNKNOWN: break;

    case DV_IMAGE_FORMAT_MISMATCH: break;

    default:
      ERRerror("read_img2vol", 0, ERR_ORIG, "File format is not supported");
      break;
  }

  if (filename)
    free(filename);

  return(result);
}


/*-----------------------------------------------------------------------------
 * checks the image files to ensure that they exist and
 * are all of the same type/format
 */
static int
FUNCcheck_image_formats(const char *filename, int start, int end, int format)
{
  char filename_buf[AVS_PATH_MAX];
  int i, from_format;

  OMpush_status_range(0, 10);
  OMstatus_check(0, "Checking Image files", NULL);

  for (i=start; i<=end; i++) {
    sprintf(filename_buf, filename, i);

    from_format = FUNCget_image_format(filename_buf);

    if (from_format == DV_IMAGE_FORMAT_NOFILE) {
      ERRerror("read_img2vol", 1, ERR_ORIG, "Cannot open file: %s",
               filename_buf);
      format = DV_IMAGE_FORMAT_NOFILE;
      break;
    }
    else if (from_format == DV_IMAGE_FORMAT_UNKNOWN) {
      ERRerror("read_img2vol", 1, ERR_ORIG, "File is of unknown format: %s",
               filename_buf);
      format = DV_IMAGE_FORMAT_UNKNOWN;
      break;
    }

    if (format == DV_IMAGE_FORMAT_AUTO) {
      /* Assume all files are of this type */
      format = from_format;
    }
    else if (format != from_format) {
      ERRerror("read_img2vol", 1, ERR_ORIG, "File is of differing type: %s",
               filename_buf);
      format = DV_IMAGE_FORMAT_MISMATCH;
      break;
    }
  }

  OMstatus_check(100, "Checking Image files", NULL);
  OMpop_status_range();

  return(format);
}


/*---------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------*/


/*
 * Data pointers used to cope with freeing the data easily on error
 */
static void *node_data;

/*
 * Function pointers used to decouple accessing functions from whether
 * they are in a shared or static library.
 */
static void *(*funcOpen)() = NULL;
static void *(*funcClose)() = NULL;
static void *(*funcGetWidth)() = NULL;
static void *(*funcGetHeight)() = NULL;
static void *(*funcGetARGBImage)() = NULL;


/*
 * Handle for shared library if we need to use it.
 */
static void *handle = NULL;


/*----------------------------------------------------------------------
 * builds the specified images into a volume by reading each one
 * and adding it to a volume field.
 */
static int FUNCread_volume(char* filename, int format, int start, int end,
                           int out_format, OMobj_id elem_id)
{
  char filename_buf[AVS_PATH_MAX];
  char info[128];
  int i, width, height, num_slices, nnodes, image_size;
  OMobj_id out_id, info_id;

  unsigned char *node_data_b;
  float *node_data_f;

  num_slices = end - start + 1;
  node_data = NULL;

  /* find the info corresponding to the format number */
  for (i = 0; i < DV_IMAGE_READERS_COUNT; ++i)
  {
    if (format == FUNCimage_readers_map[i].dv_val)
    {
      if (!FUNCget_image_funcs((void *)FUNCimage_readers_map[i].info))
        return(0);
      else
        break;
    }
  }

  /* get dims of first image */
  sprintf(filename_buf, filename, start);

  if (!FUNCread_image_dims(filename_buf, &width, &height))
    return(0);

  /* sometimes this can be a pretty big number ... */
  nnodes = width*height*num_slices;
  image_size = width*height;

  if (out_format==ARGB_OUT_FORMAT) {
    node_data_b = (unsigned char *)ARRalloc(NULL, DTYPE_BYTE, 4*nnodes, NULL);
    if( node_data_b == NULL ) {
        ERRerror("read_img2vol", 1, ERR_ORIG, 
                 "Unable to allocate output field's node data (%d bytes)",
                 4*nnodes*DTYPEtype_size[DTYPE_BYTE] );
        return 0;
    }
    node_data = node_data_b;
    image_size = image_size * 4;
  }
  else if (out_format==LUMI_OUT_FORMAT) {
    node_data_f = (float *)ARRalloc(NULL, DTYPE_FLOAT, nnodes, NULL);
    if( node_data_f == NULL ) {
        ERRerror("read_img2vol", 1, ERR_ORIG, 
                 "Unable to allocate output field's node data (%d bytes)", 
                 nnodes*DTYPEtype_size[DTYPE_FLOAT] );
        return 0;
    }
    node_data = node_data_f;
  }
  else {
    node_data_b = (unsigned char *)ARRalloc(NULL, DTYPE_BYTE, nnodes, NULL);
    if( node_data_b == NULL ) {
        ERRerror("read_img2vol", 1, ERR_ORIG, 
                 "Unable to allocate output field's node data (%d bytes)",
                 nnodes*DTYPEtype_size[DTYPE_BYTE] );
        return 0;
    }
    node_data = node_data_b;
  }

  OMpush_status_range(10, 100);

  /* read images into volume field */
  for (i=start; i<=end; i++) {
    OMstatus_check((i-start)*100/num_slices, "Reading Image files", NULL);
    sprintf(filename_buf, filename, i);

    if (out_format==LUMI_OUT_FORMAT) {
      if (!FUNCread_image(filename_buf, width, height, out_format, node_data_f)) {
        OMpop_status_range();
        return(0);
      }
      node_data_f += image_size;
    }
    else {
      if (!FUNCread_image(filename_buf, width, height, out_format, node_data_b)) {
        OMpop_status_range();
        return(0);
      }
      node_data_b += image_size;
    }

  }

  OMstatus_check(100, "Reading Image files", NULL);
  OMpop_status_range();

  /* get volume data output reference */
  out_id = OMfind_subobj(elem_id, OMstr_to_name("out"), OM_OBJ_RW);

  /* set the output volume field */
  if (!FUNCset_volume_field(width, height, num_slices, out_format, out_id))
    return(0);

#ifndef NO_DL_LOAD
  /* close the library */
  if (handle)
    DL_CLOSE(handle);
#endif

  /* give some info about what was read */
  info_id = OMfind_subobj(elem_id, OMstr_to_name("info"), OM_OBJ_RW);

  FUNCget_image_format_name(format, info);
  sprintf(info, "%s %dx%dx%d", info, width, height, num_slices);

  OMset_str_val(info_id, info);

  return(1);
}


/*----------------------------------------------------------------------
 * given an access variable loads the appropriate shared library if
 * necessary and then sets the function pointers.
 * "NO_DL_LOAD" means we don't do the dynamic library/dll open and
 * lookup but just set the function pointers from an assumed-to-be
 * filled-in table
 */
static int FUNCget_image_funcs(void *access)
{
#ifndef NO_DL_LOAD
  char **info = (char **) access;
#else
  rd_funcs_t *funcs = (rd_funcs_t *) access;
#endif

  /* open the lib */
#ifndef NO_DL_LOAD

  if (info[LIBNAME] != NULL)
    DL_OPEN(info[LIBNAME], DL_BIND, handle);
  else
    return(FUNCread_wrapup("No library name to load?", NULL));

  if (handle == NULL)
  {
    char err[512];
#ifdef MSDOS
    const char *libpath = "PATH";
    DWORD last_err = GetLastError();
#elif defined(hpux) || defined(__hpux)
    const char *libpath = "SHLIB_PATH";
#elif defined(__APPLE__) || defined(__MACH__)
    const char *libpath = "DYLD_LIBRARY_PATH";
#else
    const char *libpath = "LD_LIBRARY_PATH";
#endif

    sprintf(err, "Cannot open library '%s'.  Most likely cause is an unset or incorrectly set '%s' environment variable: please see the System Prerequisites Document.  The system error message is:\n'%s'\n", info[LIBNAME], libpath, DL_ERROR());

#ifdef MSDOS
    sprintf(err, "%s and GetLastError number is %d", err, last_err);
#endif

    return(FUNCread_wrapup(err, NULL));
  }
#endif /* NO_DL_LOAD */


  /* lookup the Image access functions */
#ifndef NO_DL_LOAD
  DL_LOOKUP(handle, info[OPEN_FUNC], void *(*)(), funcOpen);
  if (funcOpen == NULL)
    return(FUNCread_wrapup("Cannot find Open function in library", NULL));

  DL_LOOKUP(handle, info[CLOSE_FUNC], void *(*)(), funcClose);
  if (funcClose == NULL)
    return(FUNCread_wrapup("Cannot find Close function in library", NULL));

  DL_LOOKUP(handle, info[GETWIDTH_FUNC], void *(*)(), funcGetWidth);
  if (funcGetWidth == NULL)
    return(FUNCread_wrapup("Cannot find GetWidth function in library", NULL));

  DL_LOOKUP(handle, info[GETHEIGHT_FUNC], void *(*)(), funcGetHeight);
  if (funcGetHeight == NULL)
    return(FUNCread_wrapup("Cannot find GetHeight function in library", NULL));

  DL_LOOKUP(handle, info[GETARGB_FUNC], void *(*)(), funcGetARGBImage);
  if (funcGetARGBImage == NULL)
    return(FUNCread_wrapup("Cannot find GetARGBImage function in library", NULL));
#else
  funcOpen = (void *(*)()) funcs->Open;
  funcClose = (void *(*)()) funcs->Close;
  funcGetWidth = (void *(*)()) funcs->GetWidth;
  funcGetHeight = (void *(*)()) funcs->GetHeight;
  funcGetARGBImage = (void *(*)()) funcs->GetARGBImage;
#endif  /* NO_DL_LOAD */

  return(1);
}



/*---------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------*/


/*----------------------------------------------------------------------
 * composes a volume field
 */
/* 64-bit porting. Only Modified Internally */
static int
FUNCset_volume_field(int w, int h, int s, int out_format, OMobj_id out)
{
  xp_long dims[3], size, nnodes;
  float *points;

  /* Clean previous output.  Fixes CFS PR 30780 - if the output
   * format changes, you might need to clear the old node data id.
   */
  if (FLDset_node_data_ncomp(out, 0) != 1)
    return(FUNCread_wrapup("Error setting node ncomps", NULL));

  dims[0] = w;  dims[1] = h;  dims[2] = s;

  nnodes = w*h*s;

  /* Should always be determined by the dimensions
  if (FLDset_nnodes (out, nnodes) != 1)
    return(FUNCread_wrapup("Error setting nnodes", NULL));
  */
  if (FLDset_dims(out, dims) != 1)
    return(FUNCread_wrapup("Error setting dims", NULL));
  if ( (FLDget_points(out, &points, &size, OM_GET_ARRAY_WR) != 1) || (size!=6) )
    return(FUNCread_wrapup("Error getting points", NULL));

  points[0] = points[1] = points[2] = 0.0;
  points[3] = (float) (w - 1);
  points[4] = (float) (h - 1);
  points[5] = (float) (s - 1);

  ARRfree(points);

  if (FLDset_node_data_ncomp(out, 1) != 1)
    return(FUNCread_wrapup("Error setting node ncomps", NULL));

  if (out_format == ARGB_OUT_FORMAT)
  {
    if (FLDset_node_data_veclen(out, 0, 4) != 1)
      return(FUNCread_wrapup("Error setting comp veclen", NULL));

    FLDset_node_data_id(out, 0, GD_RGB_DATA_ID);

    if (FLDset_node_data(out, 0, node_data, DTYPE_BYTE, nnodes*4,
                         OM_SET_ARRAY_FREE) != 1)
      return(FUNCread_wrapup("Error setting node data array", NULL));
  }
  else if (out_format==LUMI_OUT_FORMAT)
  {
    if (FLDset_node_data_veclen(out, 0, 1) != 1)
      return(FUNCread_wrapup("Error setting comp veclen", NULL));

    if (FLDset_node_data(out, 0, node_data, DTYPE_FLOAT, nnodes,
                         OM_SET_ARRAY_FREE) != 1)
      return(FUNCread_wrapup("Error setting node data array", NULL));
  }
  else
  {
    if (FLDset_node_data_veclen(out, 0, 1) != 1)
      return(FUNCread_wrapup("Error setting comp veclen", NULL));

    if (FLDset_node_data(out, 0, node_data, DTYPE_BYTE, nnodes,
                         OM_SET_ARRAY_FREE) != 1)
      return(FUNCread_wrapup("Error setting node data array", NULL));
  }

  return(1);
}


/*----------------------------------------------------------------------
 * gets width and height from specified image
 */
static int
FUNCread_image_dims(char *filename, int *width, int *height)
{
  int w, h;
  void *priv = NULL;

  *width = *height = 0;
  if (!filename || strlen(filename) == 0)
    return(0);

  /* open the file */
  if (!(priv = (*funcOpen)(filename, DV_IMAGE_FILE_ACCESS_READ)))
    return(FUNCread_wrapup("Cannot open file", priv));

  /* get the width */
  (*funcGetWidth)(priv, &w);

  /* get the height */
  (*funcGetHeight)(priv, &h);

  /* store the dimensions */
  *width = w;  *height = h;

  /* close the file */
  (*funcClose)(priv);

  return(1);
}


/*----------------------------------------------------------------------
 * generic single image reader
 */
static int
FUNCread_image(char *filename, int width, int height, int out_format,
               void* image_data)
{
  unsigned char *data = NULL;
  int w, h;
  void *priv = NULL;
  char err[512];

  if (!filename || strlen(filename) == 0)
    return(0);

  /* open the file */
  if (!(priv = (*funcOpen)(filename, DV_IMAGE_FILE_ACCESS_READ))) {
    sprintf(err, "Cannot open file: %s", filename);
    return(FUNCread_wrapup(err, priv));
  }

  /* get the width */
  (*funcGetWidth)(priv, &w);

  /* get the height */
  (*funcGetHeight)(priv, &h);

  if ((w!=width) || (h!=height)) {
    sprintf(err, "File is of differing size: %s", filename);
    return(FUNCread_wrapup(err, priv));
  }

  /* get the image data */
  if (!((*funcGetARGBImage)(priv, &data)))
    return(FUNCread_wrapup("Failure accessing ARGB data", priv));


  /* copy data to field */
  if (out_format == ARGB_OUT_FORMAT)
  {
    unsigned char *image_data_b = (unsigned char*)image_data;
    memcpy(image_data_b, data, w*h*4);
  }
  else if (out_format == LUMI_OUT_FORMAT)
  {
    int i,j;
    float *image_data_f = (float*)image_data;

    for (i=0, j=0; i<w*h; i++, j+=4) {
      image_data_f[i] = ( ((float)data[j+AVS_RED_BYTE]  *0.299) +
                          ((float)data[j+AVS_GREEN_BYTE]*0.587) +
                          ((float)data[j+AVS_BLUE_BYTE] *0.114) ) / 255.;
    }

  }
  else if (out_format == LUMI_OUT_FORMAT_BYTE)
  {
    int i,j;
    unsigned char *image_data_b = (unsigned char*)image_data;
    float pixel_f;
    int pixel_i;

    for (i=0, j=0; i<w*h; i++, j+=4) {
      /* This can be silly when the image was gray-scale to begin
         with, but our image reader interface goes not have a
         getGrayScaleImage function.
      */
      pixel_f         = ( ((float)data[j+AVS_RED_BYTE]  *0.299) +
                          ((float)data[j+AVS_GREEN_BYTE]*0.587) +
                          ((float)data[j+AVS_BLUE_BYTE] *0.114) );      
      /* round */
      pixel_i = pixel_f + 0.5;
      /* clip */
      if( pixel_i < 0 ) pixel_i = 0;
      else if( pixel_i > 255 ) pixel_i = 255;

      image_data_b[i] = pixel_i;
    }

  }
  else
  {
    int i,j;
    unsigned char *image_data_b = (unsigned char*)image_data;

    switch(out_format) {
      case ALPHA_OUT_FORMAT: j = AVS_ALPHA_BYTE; break;
      case   RED_OUT_FORMAT: j = AVS_RED_BYTE;   break;
      case GREEN_OUT_FORMAT: j = AVS_GREEN_BYTE; break;
      case  BLUE_OUT_FORMAT: j = AVS_BLUE_BYTE;  break;
    }

    /* Copy over a single channel from the image */

    for (i=0; i<w*h; i++, j+=4) {
      image_data_b[i] = data[j];
    }
  }


  /* close the file */
  (*funcClose)(priv);

  return(1);
}



/*----------------------------------------------------------------------
 * what to do at the end of reading files or when we get a critical error
 */
static int FUNCread_wrapup(const char *msg, void *priv)
{
  /* close the file */
  if (funcClose && priv)
   (*funcClose)(priv);

  /* free the field data */
  if (node_data!=NULL)
    ARRfree(node_data);

#ifndef NO_DL_LOAD
  /* close the library */
  if (handle)
    DL_CLOSE(handle);
#endif

  /* output message and return 0 if error */
  if (msg) {
    ERRerror("read_img2vol", 0, ERR_ORIG, msg);
    return(0);
  }
  else
    return(1);
}
