/*
			Copyright (c) 2003 by
			Advanced Visual Systems Inc.
			All Rights Reserved

	This software comprises unpublished confidential information of
	Advanced Visual Systems Inc. and may not be used, copied or made
	available to anyone, except in accordance with the license
	under which it is furnished.

	This file is under Perforce control
	$Id: //depot/express/fcs70/modules/rd_poly.c#1 $
*/

#define XP_WIDE_API	/* Use Wide APIs */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>  /* time_t */

#include <avs/f_utils.h>
#include <avs/err.h>
#include <avs/om.h>
#include <avs/fld.h>

#define METHOD_SUCCESS 1
#define METHOD_FAILURE 0

#define LINE_LEN 1024

#ifndef GD_COLOR_DATA_ID
#define GD_COLOR_DATA_ID 667
#endif

#define ERR_RETURN(A) {ERRerror("read_polygon", 0, ERR_ORIG, A); \
                       return(METHOD_FAILURE);}

static int FUNCread_polygon_OBJ( FILE *fp, OMobj_id fld_id, const char *filename );

static int FUNCread_polygon_PLY( FILE *fp, OMobj_id fld_id );

static int FUNCread_polygon_VTK( FILE *fp, OMobj_id fld_id );

int
DVread_polygon_update(OMobj_id mod_id, OMevent_mask mask, int seq_num)
{
    OMobj_id file_id, fld_id;
    FILE *fp = NULL;
    char *temp_str = NULL, *filename = NULL, *ext = NULL;
    char file_buf[AVS_PATH_MAX];

    /* in */
    file_id = OMfind_subobj( mod_id, OMstr_to_name("filename"), OM_OBJ_RD );

    /* out */
    fld_id = OMfind_subobj( mod_id, OMstr_to_name("outFld"), OM_OBJ_RW );

    if( OMget_str_val(file_id, &temp_str, 0) != OM_STAT_SUCCESS )
        ERR_RETURN( "Can't get filename." );

    filename = FILEmap_variables( temp_str, file_buf );

    if( temp_str ) { free (temp_str); temp_str = NULL; }

    if( filename != NULL && filename[0] != 0 )
        fp = FILEfopen( filename, SIO_R_BIN );

    if( fp == NULL ) {
        ERRerror( "read_polygon", 1, ERR_ORIG, "Can't open data file: %s",
                  filename );
        return METHOD_FAILURE;
    }

    ext = FILEget_file_extension( filename );
    ext += 1;	/* skip over '.' */

    /* Basic (shared) field setup */
    FLDset_nnodes( fld_id, 0 );
    FLDset_ncell_sets( fld_id, 0 );
    /* Fixed by the V code */
    /* FLDset_nspace( fld_id, 3 ); */

    if( !strcmp( ext, "OBJ" ) || !strcmp( ext, "obj" ) ) {
        FUNCread_polygon_OBJ( fp, fld_id, filename );
    }
    else if( !strcmp( ext, "PLY" ) || !strcmp( ext, "ply" ) ) {
        FUNCread_polygon_PLY( fp, fld_id );
    }
    else if( !strcmp( ext, "VTK" ) || !strcmp( ext, "vtk" ) ) {
        FUNCread_polygon_VTK( fp, fld_id );
    }
    else {
        ERRerror( "read_polygon", 1, ERR_ORIG,
                  "Unknown file extension: %s", ext );
    }

    fclose( fp );

    return METHOD_SUCCESS;
}

/*
 * Shared infrastructure for the polygon readers.
 * Data structures and methods to hold the verticies and
 * polygon connection lists.
 */

typedef struct _Vertex {
    float xyz[3];
} Vertex;

static void vertex_init( Vertex *v, float *xyz )
{
    if( xyz != NULL ) {
        v->xyz[0] = xyz[0];
        v->xyz[1] = xyz[1];
        v->xyz[2] = xyz[2];
    }
}

static const int vset_chunk = 2048;

/* Connected verticies ... the data file supplies the connection information.
 * We need to keep the original order that the verticies came in, so
 * that the connection information stays valid.  This could easily be
 * "Vlist", but I'm keeping the terminology parallel with the triangle reader.
 */
typedef struct _Vset_c {
    Vertex **list;	/* Use pointers to make the realloc faster */
    xp_long list_alloc;
    xp_long nnodes;
} Vset_c;

static void vset_c_init( Vset_c *vset )
{
    vset->list = NULL;
    vset->list_alloc = 0;
    vset->nnodes = 0;
}

/* 64-bit porting. Only Modified Internally */ 
static void vset_c_free( Vset_c *vset )
{
    xp_long i;
    if( vset->list ) {
        for( i = 0; i < vset->nnodes; ++i ) {
            free( vset->list[i] );
        }
        free( vset->list );
    }
}

/* Fill in the coord list you get from FLDget_coords */
/* 64-bit porting. Only Modified Internally */ 
static void vset_c_coords( Vset_c *vset, float *coords )
{
    xp_long i;
    for( i = 0; i < vset->nnodes; ++i ) {
        Vertex *v = vset->list[i];
        coords[(i*3)+0] = v->xyz[0];
        coords[(i*3)+1] = v->xyz[1];
        coords[(i*3)+2] = v->xyz[2];
    }
}

/* 64-bit porting. Directly Modified */ 
static xp_long vset_c_count( Vset_c *vset )
{
    return vset->nnodes;
}

static void vset_c_add( Vset_c *vset, float xyz[3] )
{
    Vertex *v = malloc( sizeof(Vertex) );
    vertex_init( v, xyz );

    if( 0 == vset->list_alloc ) {
        vset->list_alloc = vset_chunk;
        vset->list = malloc( vset->list_alloc * sizeof(Vertex *) );
    }
    else if( vset->nnodes >= vset->list_alloc ) {
        vset->list_alloc += vset_chunk;
        vset->list = realloc( vset->list, vset->list_alloc * sizeof(Vertex *) );
    }
    vset->list[vset->nnodes] = v;
    vset->nnodes += 1;
}

/* Data structures to handle cell sets and their connection lists.
 * The OBJ format can have multiple 'groups', which are mapped
 * to multple cell sets, so extra infrastucture is needed to
 * manage the cell set information.
 */

static const int cellset_chunk = 2048;

typedef struct {
    xp_long ncells;
    xp_long nnodes;		/* help keep track of incremental adds */
    char *name;

    float *colors;	/* rgb (3 floats) color per cell */
    xp_long colors_alloc;

    /* Standard (for Express) integer connection indicies.  We can
     * do this because the formats covered here provide explicit
     * connection information (compare with triangle reader).
     */
    xp_long *conn;
    xp_long conn_alloc;

    int *pnodes;
    xp_long pnodes_alloc;
} Cellset;

/* 64-bit porting. Only Modified Internally */ 
static void cellset_init( Cellset *cset, const char *name )
{
    cset->ncells = 0;
    cset->nnodes = 0;

    if( name != NULL && name[0] != 0 )
        cset->name = strdup( name );
    else
        cset->name = NULL;

    cset->colors_alloc = 0;
    cset->colors = NULL;

    cset->conn_alloc = cellset_chunk;
    cset->conn = malloc( cset->conn_alloc * sizeof(xp_long) );

    cset->pnodes_alloc = cellset_chunk;
    cset->pnodes = malloc( cset->pnodes_alloc * sizeof(xp_long) );
}

static void cellset_free( Cellset *cset )
{
    if( cset->name ) free( cset->name );
    if( cset->colors ) free( cset->colors );
    if( cset->conn ) free( cset->conn );
    if( cset->pnodes ) free( cset->pnodes );
}

/* 64-bit porting. Directly Modified */ 
static void
cellset_add_poly( Cellset *cset, int nverts, xp_long *verts, float *rgb )
{
    int i;

    if( cset->nnodes+nverts > cset->conn_alloc ) {
        cset->conn_alloc += cellset_chunk;
        cset->conn = realloc( cset->conn, cset->conn_alloc * sizeof(xp_long) );
    }
    if( cset->ncells+1 > cset->pnodes_alloc ) {
        cset->pnodes_alloc += cellset_chunk;
        cset->pnodes = realloc( cset->pnodes, cset->pnodes_alloc * sizeof(xp_long) );
    }
    if( cset->colors_alloc == 0 ) {
        cset->colors_alloc = cellset_chunk;
        cset->colors = malloc( cset->colors_alloc * sizeof( float *) );
    }
    else if( (cset->ncells+1)*3 >= cset->colors_alloc ) {
        cset->colors_alloc += cellset_chunk;
        cset->colors = realloc( cset->colors, cset->colors_alloc * sizeof(float) );
    }

    for( i = 0; i < nverts; ++i ) {
        cset->conn[cset->nnodes+i] = verts[i];
    }
    cset->pnodes[cset->ncells] = nverts;
    if( rgb != NULL ) {
        xp_long where = cset->ncells;
        cset->colors[(where*3)+0] = rgb[0];
        cset->colors[(where*3)+1] = rgb[1];
        cset->colors[(where*3)+2] = rgb[2];
    }
    cset->ncells += 1;
    cset->nnodes += nverts;
}

/* Fill in the connection list you get from FLDget_poly_connect */
/* 64-bit porting. Directly Modified */ 
static void cellset_conn( Cellset *cset, xp_long *conn_list )
{
    xp_long i;
    for( i = 0; i < cset->nnodes; ++i ) {
        conn_list[i] = cset->conn[i];	/* memcpy? */
    }
}

/* Fill in the connection list you get from FLDget_poly_nnodes */
/* 64-bit porting. Only Modified Internally */ 
static void cellset_pnodes( Cellset *cset, int *pnode_list )
{
    xp_long i;
    for( i = 0; i < cset->ncells; ++i ) {
        pnode_list[i] = cset->pnodes[i];	/* memcpy? */
    }
}

/*
 * No node data, so we can regard this as a pure cell set field.
 * This does very little other than help manage multiple cell sets.
 */
typedef struct {
    int ncsets;
    Cellset *list;
} Mesh;

static void mesh_init( Mesh *mesh )
{
    mesh->list = NULL;
    mesh->ncsets = 0;
}

static int mesh_ncsets( Mesh *mesh )
{
    return mesh->ncsets;
}

static Cellset *mesh_get( Mesh *mesh, int index )
{
    return &(mesh->list[index]);
}

static Cellset *mesh_add( Mesh *mesh, const char *name )
{
    Cellset *cset = NULL;
    if( mesh == NULL ) return NULL;

    /* A special bit of trickiness to deal with empty groups,
     * which happen when 'g' commands are added in the middle
     * of the 'v' commands in OBJ files. (Why? essentially as comments?)
     */
    if( mesh->ncsets > 0 ) {
        if( mesh->list[mesh->ncsets-1].ncells == 0 )
            return &(mesh->list[mesh->ncsets-1]);
    }

    if( mesh->ncsets == 0 ) {
        mesh->list = malloc( sizeof( Cellset ) );
    }
    else {
        /* Not very efficient, but there shouldn't be many cellsets */
        mesh->list = realloc( mesh->list, (mesh->ncsets+1) * sizeof( Cellset ) );
    }
    cset = &(mesh->list[mesh->ncsets]);
    cellset_init( cset, name );
    mesh->ncsets += 1;
    /* Warning! this pointer can be invalidated by the next realloc. */
    return cset;
}

static void mesh_free( Mesh *mesh )
{
    if( mesh->list ) {
        int i;
        for( i = 0; i < mesh->ncsets; ++i ) {
            cellset_free( &(mesh->list[i]) );
        }
        free( mesh->list );
    }
}

/*
 * This following "material" code is specific to OBJ files.
 */

typedef struct _Material {
    char *name;
    float rgb[3];
} Material;

static const int material_chunk = 64;

typedef struct _MaterialLib {
    Material **list;
    xp_long list_alloc;
    xp_long nmats;
} MaterialLib;


static void mtllib_init( MaterialLib *mtllib )
{
    mtllib->list = NULL;
    mtllib->list_alloc = 0;
    mtllib->nmats = 0;
}

/* 64-bit porting. Only Modified Internally */ 
static void mtllib_free( MaterialLib *mtllib )
{
    if( mtllib != NULL && mtllib->list != NULL ) {
        xp_long i;
        for( i = 0; i < mtllib->nmats; ++i ) {
            Material *mat = mtllib->list[i];
            if( mat != NULL ) {
                if( mat->name != NULL ) free( mat->name );
                free( mat );
            }
        }
        free( mtllib->list );
    }
}

static void mtllib_read( MaterialLib *mtllib, const char *filename )
{
    char line_buf[LINE_LEN], *p;
    int num_scanned;
    Material *mat = NULL;
    FILE *fp;

    if( filename == NULL || filename[0] == 0 ) return ;

    fp = FILEfopen( filename, SIO_R_BIN );
    if( fp == NULL ) return;

    while( 1 ) {
        /* Read a line out of the file */
        if( fgets( line_buf, LINE_LEN, fp ) == NULL ) {
            /* End of file */
            break;
        }
        p = line_buf;
        /* Skip leading spaces and tabs */
        while( p[0] == 0x20 || p[0] == 0x9 ) p++;
        /* Skip blank lines (CR/LF) and comments (#) */
        if( p[0] == 0 || p[0] == 0xa || p[0] == 0xd ) continue;
        if( p[0] == 0x23 ) continue;

        if( strncmp( p, "newmtl", 6 ) == 0 ) {
            char name[LINE_LEN];
            if( mat && mat->name ) free( mat->name );
            num_scanned = sscanf( p, "newmtl %s", name );
            if( num_scanned == 1 ) {
                if( mat == NULL ) mat = malloc( sizeof(Material) );
                mat->name = strdup( name );
                mat->rgb[0] = 1.0;
                mat->rgb[1] = 1.0;
                mat->rgb[2] = 1.0;
            }
        }
        else if( mat != NULL && strncmp( p, "Kd", 2 ) == 0 ) {
            float *rgb = mat->rgb;
            num_scanned = sscanf( p, "Kd %f %f %f",
                                  rgb, rgb+1, rgb+2 );
            if( num_scanned == 3 ) {
#if 0
                printf( "Kd %f %f %f\n", rgb[0], rgb[1], rgb[2] );
#endif
                if( 0 == mtllib->list_alloc ) {
                    mtllib->list_alloc = material_chunk;
                    mtllib->list = malloc( mtllib->list_alloc * sizeof(Material *) );
                }
                else if( mtllib->nmats+1 > mtllib->list_alloc ) {
                    mtllib->list_alloc += material_chunk;
                    mtllib->list = realloc( mtllib->list, mtllib->list_alloc * sizeof(Material *) );
                }
                mtllib->list[mtllib->nmats] = mat;
                mtllib->nmats += 1;
                mat = NULL;
            }
        }
    } /* while */

    fclose( fp );
}

/* 64-bit porting. Only Modified Internally */ 
static int mtllib_get_rgb( MaterialLib *mtllib, const char *mtl, float *rgb )
{
    /* defensive programming overkill ? */
    if( mtllib != NULL && mtl != NULL &&
        rgb != NULL && mtllib->list != NULL ) {
        xp_long i;
        for( i = 0; i < mtllib->nmats; ++i ) {
            Material *mat = mtllib->list[i];
            if( mat != NULL && mat->name != NULL ) {
                if( strcmp( mat->name, mtl ) == 0 ) {
                    rgb[0] = mat->rgb[0];
                    rgb[1] = mat->rgb[1];
                    rgb[2] = mat->rgb[2];
                    return 1;
                }
            }
        }
        return 0;
    }
    return 0;
}

/*
 * A few parsing utilities.
 */

static int trim_crlf( char *p )
{
    if( p == NULL )
        return 0;
    else {
        int len = (int)strlen( p );
        char *pe;
        while( len != 0 ) {
            pe = p+len-1;	/* point at end of string */
            if( pe[0] == 0xa || pe[0] == 0xd ) {
                pe[0] = 0;	/* Stomp on CR/LF */
                len--;
            }
            else break;
        }
        return len;
    }
}

static char *
skip_until_white( char *p )
{
    while( p[0] != 0x20 && p[0] != 0x9 &&
           p[0] != 0xa  && p[0] != 0xd &&
           p[0] != 0 )
        p++;
    return p;
}

static char *
skip_over_white( char *p )
{
    while( (p[0] == 0x20 || p[0] == 0x9 ||
            p[0] == 0xa  || p[0] == 0xd ) &&
            p[0] != 0 )
        p++;
    return p;
}

/*
 * Reader for the Wavefront Advanced Visualizer OBJ format.
 *
 * This reader accepts only a limited subset of the OBJ format.
 * It processes v, f, and g commands.  It can read f commands that
 * include normal and texture information, but the normal/texture
 * information is not used.  This reader can process usemtl and mtllib
 * commands, but only uses the diffuse color ('Kd').  
 */
/* 64-bit porting. Only Modified Internally */ 
static int
FUNCread_polygon_OBJ( FILE *fp, OMobj_id fld_id, const char *filename )
{
    char line_buf[LINE_LEN], *p;
    int i, num_scanned;
    int use_colors = 0;

    char obj_name[LINE_LEN];
    float xyz[3], rgb[3] = { 1.0, 1.0, 1.0 };

    xp_long fld_nodes = 0;

    Vset_c vset;
    Mesh mesh;
    Cellset *cset = NULL;
    MaterialLib mtllib;

    vset_c_init( &vset );
    mesh_init( &mesh );
    mtllib_init( &mtllib );

    while( 1 ) {
        /* Read a line out of the file */
        if( fgets( line_buf, LINE_LEN, fp ) == NULL ) {
            /* End of file */
            break;
        }
        p = line_buf;
        /* Skip spaces and tabs */
        while( p[0] == 0x20 || p[0] == 0x9 ) p++;

        /* Skip blank lines (CR/LF) and comments (#) */
        if( p[0] == 0 || p[0] == 0xa || p[0] == 0xd ) continue;
        if( p[0] == 0x23 ) continue;

        /* Vertex */
        if( p[0] == 'v' ) {
            num_scanned = sscanf( p, "v %f %f %f", xyz, xyz+1, xyz+2 );
            if( num_scanned == 3 ) {
                vset_c_add( &vset, xyz );
            }
        }
        /* Face */
        if( p[0] == 'f' ) {
#define MAX_SCANNED 256
            xp_long f1[MAX_SCANNED];

            /* Not using a simple all-in one sscanf, so it can handle faces
             * of form ...
             *     f a/b/c a/b/c a/b/c
             */
            p++;	/* skip over 'f' */
            num_scanned = 0;
            while( num_scanned < MAX_SCANNED &&
                   p[0] != 0xa && p[0] != 0xd && p[0] != 0 ) {
                int scanned, tmp;
                /* Skip leading white space */
                while( p[0] == 0x20 || p[0] == 0x9 ) p++;

                scanned = sscanf( p, "%d", &tmp );
                if( scanned == 1 ) {
                    f1[num_scanned] = tmp;
                    num_scanned++;
                }
                else if( scanned == EOF )
                    break;
                /* if scanned == 0 ?? */

                /* Skip over the a/b/c stuff until we see white space */
                p = skip_until_white( p );
            }

            /* negative number is an offset from the last vertex seen.
             * -1 seems to mean the last one (not 0).
             */
            if( 3 <= num_scanned ) {

                if( cset == NULL ) {
                    /* In case there is no 'g' command. */
                    cset = mesh_add( &mesh, NULL );
                }

                for( i = 0; i < num_scanned; ++i ) {
                  if( f1[i] < 0 ) f1[i] = vset.nnodes - f1[0];
                  else            f1[i] -= 1;
                }

                cellset_add_poly( cset, num_scanned, f1, rgb );
            }
            else {
                static int once = 0;
                if( once == 0 ) {
                    fprintf( stderr, "Warning: this reader does not handle lines or points\n" );
                    once = 1;
                }
            }
        }
        /* Group */
        else if( p[0] == 'g' ) {
            trim_crlf( p );
            p++;	/* skip over 'g' */
            /* Skip whitespace after 'g' */
            while( p[0] == 0x20 || p[0] == 0x9 ) p++;
            obj_name[0] = 0;
            strncpy( obj_name, p, LINE_LEN );

            cset = mesh_add( &mesh, obj_name );
        }
        /* Use material */
        else if( p[0] == 'u' && strncmp( p, "usemtl", 6 ) == 0 ) {
            /* Should be a name found in the 'material library' */
            num_scanned = sscanf( p, "usemtl %s", obj_name );

            /* if no mtlib, default to datafilename.mtl ? */
            if( mtllib_get_rgb( &mtllib, obj_name, rgb ) != 0 ) {
                /* OK, have a new rgb */
                use_colors = 1;
            }
        }
        /* Material library */
        else if( p[0] == 'm' && strncmp( p, "mtllib", 6 ) == 0 ) {
            /* Is it possible that there be more than one per file? */
            mtllib_free( &mtllib );
            mtllib_init( &mtllib );

            /* Attempt to read the file */
            num_scanned = sscanf( p, "mtllib %s", obj_name );
#if 0
            printf( "mtllib %s\n", obj_name );
#endif
            if( strlen( obj_name ) > 0 ) {
                if( FILEis_path_absolute( obj_name ) ) {
                    mtllib_read( &mtllib, obj_name );
                }
                else {
                    char mtl_name[LINE_LEN];
                    char *dir = FILEget_dir_name( filename, 0 );
                    if( dir && strlen( dir ) > 0 ) {
                        FILEconcat_dir_and_file( dir, obj_name, mtl_name );
                        mtllib_read( &mtllib, mtl_name );
                    }
                    else {
                        mtllib_read( &mtllib, obj_name );
                    }
                }
            }
        }
    } /* while */

    /* The file has been read, now put the info in the field. */

    fld_nodes = vset_c_count( &vset );
    FLDset_nnodes( fld_id, fld_nodes );

    /* Field coordinate list */
    {
        float *fld_coords;
        xp_long size;
        FLDget_coord( fld_id, &fld_coords, &size, OM_GET_ARRAY_RW );
        if( fld_coords != NULL ) {
            vset_c_coords( &vset, fld_coords );
            ARRfree( fld_coords );
        }
    }

    FLDset_ncell_sets( fld_id, mesh_ncsets( &mesh ) );

    /* Each cell set */
    for( i = 0; i < mesh_ncsets( &mesh ); ++i ) {
        OMobj_id cset_id;
        xp_long *conn_list;
        int *pnode_list;
        xp_long size;

        cset = mesh_get( &mesh, i );

        FLDget_cell_set( fld_id, i, &cset_id );
        FLDset_npolys( cset_id, cset->ncells );

        /* (Poly) Cell set connection list */
        conn_list = (xp_long *)ARRalloc(NULL, DTYPE_LONG, cset->nnodes, NULL);
        if( conn_list != NULL ) {
            cellset_conn( cset, conn_list );
            /* Need to to a 'set' because the size of the array is
             * not determined by the V definitions of polyhedron.
             * Thus, 'get'-ing the array gets garbage.
             */
            FLDset_poly_connect( cset_id, conn_list, cset->nnodes,
                                 OM_SET_ARRAY_FREE );
            conn_list = NULL;	/* OM owns array now */
        }

        /* size should be same as npolys */
        FLDget_poly_nnodes( cset_id, &pnode_list, &size, OM_GET_ARRAY_RW );
        if( pnode_list != NULL ) {
            cellset_pnodes( cset, pnode_list );
            ARRfree( pnode_list );
        }

        /* Cell set name (optional) */
        if( cset->name != NULL ) {
            FLDset_cell_set_user_name( cset_id, cset->name );
        }

        /* Cell data */
        if( use_colors != 0 && cset->colors != NULL ) {
            float *cell_data = NULL;
            int type = DTYPE_FLOAT;

            FLDset_cell_data_ncomp( cset_id, 1 );
            FLDset_cell_data_veclen( cset_id, 0, 3 );
            FLDset_cell_data_id( cset_id, 0, GD_COLOR_DATA_ID );
            FLDset_cell_data_type( cset_id, 0, DTYPE_FLOAT );

            FLDget_cell_data( cset_id, 0, &type, (char **)&cell_data,
                              &size, OM_GET_ARRAY_RW );
            if( cell_data != NULL ) {
                memcpy( cell_data, cset->colors,
                        cset->ncells * 3 * sizeof(float) );
                ARRfree( cell_data );
            }
        }
        else FLDset_cell_data_ncomp( cset_id, 0 );
    }	/* loop over cell sets */

    vset_c_free( &vset );
    mesh_free( &mesh );
    mtllib_free( &mtllib );

    return 1;
}


/*
 * Reader for the PLY format, also known as the Stanford triangle format.
 *
 * This reader accepts only a limited subset of the PLY format.
 * The files must be ASCII format and only it understands basic
 * properties such as vertex x,y,z and face vertex indicies.
 * The most obvious enhancement would be to read face rgb, since
 * we already have the infrastructure to handle cell colors.
 * It wouldn't be all that hard to also do vertex colors, but neither
 * the shared infrastructure nor the V code (no node data) support this.
 */
/* 64-bit porting. Only Modified Internally */
static int
FUNCread_polygon_PLY( FILE *fp, OMobj_id fld_id )
{
    char line_buf[LINE_LEN], *p;
    int i, num_scanned, tmp;
    enum { HEADER, VERTEX, FACE } phase;

    float xyz[3];
    /* float rgb[3] = { 1.0, 1.0, 1.0 }; */

    xp_long fld_nodes = 0;
    int fld_faces = 0;

    Vset_c vset;
    Mesh mesh;
    Cellset *cset = NULL;

    vset_c_init( &vset );
    mesh_init( &mesh );
    /* No reason to break up into multiple cell sets. */
    cset = mesh_add( &mesh, NULL );

    if( (fgets( line_buf, LINE_LEN, fp ) == NULL) ||
        (strncmp( line_buf, "ply", 3 ) != 0) ) {
        ERRerror("read_polygon_PLY", 0, ERR_ORIG,
                 "Not a PLY file" );
        return 0;
    }
    phase = HEADER;

    while( 1 ) {
        /* Read a line out of the file */
        if( fgets( line_buf, LINE_LEN, fp ) == NULL ) {
            /* End of file */
            break;
        }
        p = line_buf;
        /* Skip spaces and tabs */
        while( p[0] == 0x20 || p[0] == 0x9 ) p++;

        /* Skip blank lines (CR/LF) */
        if( p[0] == 0 || p[0] == 0xa || p[0] == 0xd ) continue;

        if( phase == HEADER ) {
            if( strncmp( p, "end_header", 10 ) == 0 ) {
                phase = VERTEX;
            }
            else if( strncmp( p, "element vertex", 14 ) == 0 ) {
                num_scanned = sscanf( p+14, "%d", &tmp );
                if( num_scanned == 1 ) fld_nodes = tmp;
            }
            else if( strncmp( p, "element face", 12 ) == 0 ) {
                num_scanned = sscanf( p+12, "%d", &tmp );
                if( num_scanned == 1 ) fld_faces = tmp;
            }
            else if( strncmp( p, "format", 6 ) == 0 ) {
                /* check for ascii format */
                p = skip_over_white( p+6 );
                if( strncmp( p, "ascii", 5 ) != 0 ) {
                    ERRerror("read_polygon_PLY", 0, ERR_ORIG,
                             "Cannot read binary PLY file" );
                    return 0;
                }
            }
        }
        else if( phase == VERTEX ) {
            num_scanned = sscanf( p, "%f %f %f", xyz, xyz+1, xyz+2 );
            if( num_scanned == 3 ) {
                vset_c_add( &vset, xyz );
            }
            if( vset.nnodes == fld_nodes ) {
                phase = FACE;
            }
        }
        else if( phase == FACE ) {
#undef MAX_SCANNED
#define MAX_SCANNED 256
            xp_long f1[MAX_SCANNED];

            num_scanned = sscanf( p, "%d", &tmp );
            p = skip_until_white( p );	/* skip over the number */
            if( num_scanned == 1 ) {
                int nverts = tmp;	/* verts in this polygon */
                num_scanned = 0;
                for( i = 0; i < nverts; i++ ) {
                    p = skip_over_white( p );
                    if( 1 == sscanf( p, "%d", &tmp ) ) {
                        p = skip_until_white( p );	/* skip past number */
                        f1[num_scanned] = tmp;
                        num_scanned++;
                    }
                    else break;
                }
                /* After normal termination of loop */
                cellset_add_poly( cset, num_scanned, f1, NULL );
            }
            else break;
        }

    } /* while */

    /* The file has been read, now put the info in the field. */

    fld_nodes = vset_c_count( &vset );
    FLDset_nnodes( fld_id, fld_nodes );

    /* Field coordinate list */
    {
        float *fld_coords;
        xp_long size;
        FLDget_coord( fld_id, &fld_coords, &size, OM_GET_ARRAY_RW );
        if( fld_coords != NULL ) {
            vset_c_coords( &vset, fld_coords );
            ARRfree( fld_coords );
        }
    }

    FLDset_ncell_sets( fld_id, mesh_ncsets( &mesh ) );

    /* Each cell set */
    for( i = 0; i < mesh_ncsets( &mesh ); ++i ) {
        OMobj_id cset_id;
        xp_long *conn_list;
        int *pnode_list;
        xp_long size;

        cset = mesh_get( &mesh, i );

        if( FLDget_cell_set( fld_id, i, &cset_id ) != 1 )
            continue;
        if( FLDset_npolys( cset_id, cset->ncells ) != 1 )
            continue;

        /* (Poly) Cell set connection list */
        conn_list = (xp_long *)ARRalloc(NULL, DTYPE_LONG, cset->nnodes, NULL);
        if( conn_list != NULL ) {
            cellset_conn( cset, conn_list );
            /* Need to to a 'set' because the size of the array is
             * not determined by the V definitions of polyhedron.
             * Thus, 'get'-ing the array gets garbage.
             */
            FLDset_poly_connect( cset_id, conn_list, cset->nnodes,
                                 OM_SET_ARRAY_FREE );
            conn_list = NULL;	/* OM owns array now */
        }

        /* size should be same as npolys */
        FLDget_poly_nnodes( cset_id, &pnode_list, &size, OM_GET_ARRAY_RW );
        if( pnode_list != NULL ) {
            cellset_pnodes( cset, pnode_list );
            ARRfree( pnode_list );
        }

        /* Cell data */
        FLDset_cell_data_ncomp( cset_id, 0 );
    }	/* loop over cell sets */

    vset_c_free( &vset );
    mesh_free( &mesh );

    return 1;
}

/*
 * Reads ASCII VTK files, with the following restrictions.
 *
 * The dataset must be either POLYDATA or UNSTRUCTURED_GRID.
 * If its a POLYDATA dataset, then the points must be followed by
 * either LINES or POLYGONS.  The POINTS must be broken up into lines
 * no longer than 1024 characters each.  The face index information for each
 * line or polygon must be on a single line (usually the case but not always).
 * If its a UNSTRUCTURED_GRID dataset, then only cells of type line, tri,
 * polygon, pixel, and quad.
 *
 * The two most obvious enhancements would be to add cell type checking and
 * add an ability to read at least simple cell data.
 */
/* 64-bit porting. Only Modified Internally */ 
static int
FUNCread_polygon_VTK( FILE *fp, OMobj_id fld_id )
{
    char line_buf[LINE_LEN], *p;
    int i, num_scanned, tmp;

    /* This phase stuff is getting a little messy ... */
    enum { HEADER, MESH, DATA }  phase;
    enum { MESH_UNKNOWN, MESH_POINTS, MESH_POLYGONS,
           MESH_LINES, MESH_CELLS } phase_mesh;

    float xyz[3];

    xp_long fld_nodes = 0, fld_faces = 0;

    Vset_c vset;
    Mesh mesh;
    Cellset *cset = NULL;

    vset_c_init( &vset );
    mesh_init( &mesh );
    /* No reason to break up into multiple cell sets,
     * so just create one up front.
     */
    cset = mesh_add( &mesh, NULL );

    if( (fgets( line_buf, LINE_LEN, fp ) == NULL) ||
        (strncmp( line_buf, "# vtk DataFile", 14 ) != 0) ) {
        ERRerror("read_polygon_VTK", 0, ERR_ORIG,
                 "Not a VTK file" );
        return 0;
    }
    phase = HEADER;

    while( 1 ) {
        /* Read a line out of the file */
        if( fgets( line_buf, LINE_LEN, fp ) == NULL ) {
            /* End of file */
            break;
        }
        p = line_buf;

        /* Skip leading spaces and tabs */
        while( p[0] == 0x20 || p[0] == 0x9 ) p++;
        /* Skip blank lines (CR/LF) */
        if( p[0] == 0 || p[0] == 0xa || p[0] == 0xd ) continue;

        if( phase == HEADER ) {
            if( strncmp( p, "DATASET", 7 ) == 0 ) {
                p = skip_over_white( p+7 );
                if( strncmp( p, "POLYDATA", 8 ) == 0 ||
                    strncmp( p, "UNSTRUCTURED_GRID", 17 ) == 0 ) {
                    phase = MESH;
                    phase_mesh = MESH_UNKNOWN;
                }
                else {
                    ERRerror("read_polygon_VTK", 0, ERR_ORIG,
                             "Can only read datasets of type POLYDATA and UNSTRUCTURED_GRID" );
                }
            }
            else if( strncmp( p, "BINARY", 6 ) == 0 ) {
                /* Whoops, ASCII only */
                ERRerror("read_polygon_VTK", 0, ERR_ORIG,
                         "Cannot read BINARY files" );
                return 0;
            }
        }
        else if( phase == MESH ) {
            if( phase_mesh == MESH_UNKNOWN &&
                strncmp( p, "POINTS", 6 ) == 0 ) {
                num_scanned = sscanf( p+6, "%d", &tmp );
                if( num_scanned == 1 ) {
                    fld_nodes = tmp;
                    phase_mesh = MESH_POINTS;
                }
            }
            else if( phase_mesh == MESH_UNKNOWN &&
                     strncmp( p, "POLYGONS", 8 ) == 0 ) {
                num_scanned = sscanf( p+8, "%d", &tmp );
                if( num_scanned == 1 ) {
                    fld_faces = tmp;
                    phase_mesh = MESH_POLYGONS;
                }
            }
            else if( phase_mesh == MESH_UNKNOWN &&
                     strncmp( p, "LINES", 5 ) == 0 ) {
                num_scanned = sscanf( p+5, "%d", &tmp );
                if( num_scanned == 1 ) {
                    fld_faces = tmp;
                    phase_mesh = MESH_LINES;
                }
            }
            else if( phase_mesh == MESH_UNKNOWN &&
                     strncmp( p, "CELLS", 5 ) == 0 ) {
                /*
                 * Restriction: the cell types *must* be one of:
                 * 3 (line), 5 (triangle), 7 (polygon), 8 (pixel), 9 (quad).
                 * Anything else will not work.  Currently this
                 * reader is not checking the cell types.
                 */
                num_scanned = sscanf( p+5, "%d", &tmp );
                if( num_scanned == 1 ) {
                    fld_faces = tmp;
                    phase_mesh = MESH_CELLS;
                }
            }
            else if( phase_mesh == MESH_POINTS ) {
                /* Need the 'while' loop because there might be multiple
                 * points on one line.  At least one dataset I've seen
                 * has *all* the points on one line, which this reader cannot
                 * cope with, unless I make the major change of abandoning
                 * the line-by-line technique for reading in the file.
                 */
                while( 1 ) {
                    num_scanned = sscanf( p, "%f %f %f", xyz, xyz+1, xyz+2 );
                    if( num_scanned == 3 ) {
                        vset_c_add( &vset, xyz );
                        /* Yuck!  Is the %n arguement to sscanf portable? */
                        p = skip_until_white( p );	/* skip over x */
                        p = skip_over_white( p );
                        p = skip_until_white( p );	/* skip over y */
                        p = skip_over_white( p );
                        p = skip_until_white( p );	/* skip over z */
                        p = skip_over_white( p );
                    }
                    else break;	/* EOL: break out of inner while */

                    if( vset.nnodes == fld_nodes ) {
                        /* Read all the points I expect */
                        phase_mesh = MESH_UNKNOWN;
                    }
                }
            }
            else if( phase_mesh == MESH_POLYGONS ||
                     phase_mesh == MESH_LINES ||
                     phase_mesh == MESH_CELLS ) {
#undef MAX_SCANNED
#define MAX_SCANNED 1024
                xp_long f1[MAX_SCANNED];

                num_scanned = sscanf( p, "%d", &tmp );
                p = skip_until_white( p );	/* skip over the number */
                if( num_scanned == 1 ) {
                    int nverts = tmp;	/* verts in this polygon */
                    num_scanned = 0;
                    for( i = 0; i < nverts; i++ ) {
                        p = skip_over_white( p );
                        if( 1 == sscanf( p, "%d", &tmp ) ) {
                            p = skip_until_white( p );	/* skip past number */
                            f1[num_scanned] = tmp;
                            num_scanned++;

                            /* Some whacky logic to break up connected
                             * polylines into disjoint pairs.
                             *
                             * The issue could be avoided by a more flexible
                             * reader that would switch between FLD.Polyhedron
                             * and FLD.Polyline cell set types as needed, but
                             * by design, I'm keeping this reader simple.
                             *
                             * [Actually, Polylines are hard to use in a
                             *  reader like this, because then verts in each
                             *  polyline segment must be consecutive - and
                             *  we can't guarantee that.]
                             */
                            if( phase_mesh == MESH_LINES && nverts != 2 &&
                                num_scanned == 2 && i != nverts-1 ) {
                                cellset_add_poly( cset, 2, f1, NULL );
                                num_scanned = 1;
                                f1[0] = f1[1];
                            }
                        }
                        else break;
                    }
                    /* After normal termination of loop */
                    cellset_add_poly( cset, num_scanned, f1, NULL );

                    if( cset->ncells == fld_faces ) {
                        /* Read all the cells I expect */
                        phase_mesh = MESH_UNKNOWN;
                    }
                }
                else break;
            }
        }
    } /* while */

    /* The file has been read, now put the info in the field. */

    fld_nodes = vset_c_count( &vset );
    FLDset_nnodes( fld_id, fld_nodes );

    /* Field coordinate list */
    {
        float *fld_coords;
        xp_long size;
        FLDget_coord( fld_id, &fld_coords, &size, OM_GET_ARRAY_RW );
        if( fld_coords != NULL ) {
            vset_c_coords( &vset, fld_coords );
            ARRfree( fld_coords );
        }
    }

    FLDset_ncell_sets( fld_id, mesh_ncsets( &mesh ) );

    /* Each cell set */
    for( i = 0; i < mesh_ncsets( &mesh ); ++i ) {
        OMobj_id cset_id;
        xp_long *conn_list;
        int *pnode_list;
        xp_long size;

        cset = mesh_get( &mesh, i );

        FLDget_cell_set( fld_id, i, &cset_id );
        FLDset_npolys( cset_id, cset->ncells );

        /* (Poly) Cell set connection list */
        conn_list = (xp_long *)ARRalloc(NULL, DTYPE_LONG, cset->nnodes, NULL);
        if( conn_list != NULL ) {
            cellset_conn( cset, conn_list );
            /* Need to do a 'set' because the size of the array is
             * not determined by the V definitions of polyhedron.
             * Thus, 'get'-ing the array gets garbage.
             */
            FLDset_poly_connect( cset_id, conn_list, cset->nnodes,
                                 OM_SET_ARRAY_FREE );
            conn_list = NULL;	/* OM owns array now */
        }

        /* size shoud be same as npolys */
        FLDget_poly_nnodes( cset_id, &pnode_list, &size, OM_GET_ARRAY_RW );
        if( pnode_list != NULL ) {
            cellset_pnodes( cset, pnode_list );
            ARRfree( pnode_list );
        }

        /* no Cell data */
        FLDset_cell_data_ncomp( cset_id, 0 );
    }	/* loop over cell sets */

    vset_c_free( &vset );
    mesh_free( &mesh );

    return 1;
}
