/*
			Copyright (c) 1993 by
			Advanced Visual Systems Inc.
			All Rights Reserved

	This software comprises unpublished confidential information of
	Advanced Visual Systems Inc. and may not be used, copied or made
	available to anyone, except in accordance with the license
	under which it is furnished.

	This file is under Perforce control
	$Id: //depot/express/fcs70/modules/rd_ucd.c#1 $
*/
/*-----------------------------------------------------*
 *                                                     *
 *               ****  read_ucd  ****                  *
 *                                                     *
 * read a file (ascii or binary) in the ucd format.    *
 * if the file is binary then it is assumed it was     *
 * created using the ucd_write module.                 *
 *                                                     *
 *-----------------------------------------------------*/

#define XP_WIDE_API	/* Use Wide APIs */

#include <stdio.h>
#include <sys/types.h>  /* time_t */

#include <avs/util.h>
#include <avs/f_utils.h>
#include <avs/err.h>
#include <avs/om.h>
#include <avs/math.h>
#include <avs/dtype.h>
#include <avs/fld.h>


#define BUFFER_SIZE  200
#define UCD_ASCII_FILE 3
#define UCD_BINARY_3_0 1
#define UCD_BINARY_3_5 2

#define UCD_MAX_NODE_PER_CELL 20
#define UCD_LABEL_LEN 1024
#define UCD_NUM_CELL_TYPES   8

#define ERR_RETURN(A) ERRerror("read_ucd", 0, ERR_ORIG, A); return(0);
#define MAXLINE 1024
#define MAX_MATER  64
#define MAX_CELL   16
#define BLOCK_SIZE 1024

typedef struct Cells_ {
	char cell_name[128];
	int nnodes;
	int nmat;
	int mat_count;
	int *mat;
	xp_long *ncells;
	xp_long **cell_list;
	xp_long *csize;
} Cells;

typedef struct _ctype_struct_ascii {
	xp_long id;
	int mat, n, cell_type;
} Ctype_A;

typedef struct _ctype_struct_binary {
	int id, mat, n, cell_type;
} Ctype_B;

typedef struct _ntype_struct {
	xp_long n, node_list[UCD_MAX_NODE_PER_CELL];
} Ntype;

typedef struct _model_stats_ascii {
	char *node_data_labels, *cell_data_labels, *model_data_labels,
	     *node_data_units, *cell_data_units, *model_data_units;

	xp_long num_nodes, num_cells, num_nlist_nodes;
	int num_node_data, num_cell_data, num_model_data,
	    *node_active_list, *cell_active_list, *model_active_list,
	    num_node_comp, *node_comp_list, num_cell_comp, *cell_comp_list,
	    num_model_comp, *model_comp_list, 
	    mid_edge_flag, material_flag;
  } Mtype_A;

typedef struct _model_stats_binary {
	char *node_data_labels, *cell_data_labels, *model_data_labels,
	     *node_data_units, *cell_data_units, *model_data_units;

	int num_nodes, num_cells, num_node_data, num_cell_data, num_model_data,
	    *node_active_list, *cell_active_list, *model_active_list,
	    num_node_comp, *node_comp_list, num_cell_comp, *cell_comp_list,
	    num_model_comp, *model_comp_list, num_nlist_nodes, 
	    mid_edge_flag, material_flag;
  } Mtype_B;
  
static const char *set_names [MAX_CELL]={"Point", "Line", "Tri", "Quad", "Tet", "Pyr",
				  "Prism", "Hex",
				   "",     "Line2", "Tri2", "Quad2", "Tet2", "Pyr2",
				  "Prism2", "Hex2"};
static int  set_nnodes [MAX_CELL]={1,2,3,4,4,5,6,8,1,3,6,8,10,13,15,20};

static int  UCD_num_nodes[] = {1, 2, 3, 4, 4, 5, 6, 8 };

int FUNCread_ucd(char *file_name, OMobj_id out);
int FUNCcheck_ucd_file(char *file_name);

static int read_data(FILE *fp, xp_long num_data, float *data, float *mini_data, 
                     float *maxi_data, char *data_labels, char *units, 
                     char *string, int *num_comp, int *comp_list,
                     int data_veclen);

static int read_ascii(char *file_name, Ctype_A **pcells, Ntype **pcell_nlists, 
                      Mtype_A *model, float **coord, 
                      float **pnode_data, float **pcell_data, 
                      float **pmodel_data, float **pmin_node_data, 
                      float **pmax_node_data, float **pmin_cell_data, 
                      float **pmax_cell_data, float **pmin_model_data,
                      float **pmax_model_data, xp_long **node_id);

static int read_binary3_0(char *file_name, Ctype_B **pcells, int **pcell_nlists, 
                          Mtype_B *model, float **coord,
                          float **pnode_data, float **pcell_data, 
                          float **pmodel_data, float **pmin_node_data, 
                          float **pmax_node_data, float **pmin_cell_data, 
                          float **pmax_cell_data, float **pmin_model_data,
                          float **pmax_model_data, int **node_id);

static int read_binary3_5(char *file_name, Ctype_B **pcells, int **pcell_nlists, 
                          Mtype_B *model, float **coord,
                          float **pnode_data, float **pcell_data, 
                          float **pmodel_data, float **pmin_node_data, 
                          float **pmax_node_data, float **pmin_cell_data, 
                          float **pmax_cell_data, float **pmin_model_data,
                          float **pmax_model_data, int **node_id);

static int get_token(char **tline, void *data, const char *control);

static int get_label(char *string, char delimiter, int number, char *label);
static void ucd_qsorti(xp_long *values, xp_long *indexes, xp_long num_entries);
static void ucd_qsi(xp_long *values, xp_long *indexes, xp_long left, xp_long right);
static xp_long ucd_bsearchi(xp_long v, xp_long *list, xp_long n);

int DVread_ucd_update(OMobj_id elem_id)
{
    OMobj_id filename, out;
    char *str;
    int Result;

    str = NULL;
    filename = OMfind_subobj(elem_id, OMstr_to_name("filename"), OM_OBJ_RD);
    OMget_str_val(filename, &str, 0);
    out = OMfind_subobj(elem_id, OMstr_to_name("out"), OM_OBJ_RW);

    Result = FUNCread_ucd(str, out);

    if (str) free(str);

    return(Result);
}


/*--------------------------------------------------------------------
 *      Read UCD module
 *--------------------------------------------------------------------*/
/* 64-bit porting. Only Modified Internally */
int FUNCread_ucd(char *file_name, OMobj_id out)
{
	char data_labels[UCD_LABEL_LEN], cell_labels[UCD_LABEL_LEN],
	     node_labels[UCD_LABEL_LEN], node_units[UCD_LABEL_LEN], 
	     cell_units[UCD_LABEL_LEN],	data_units[UCD_LABEL_LEN];

	float *out_data;

	int file_version, *node_id_b = NULL, mat_id, i, j, n, type, veclen, prop_size;
	xp_long *node_id_a = NULL, j_w, n_w, cell, size, offset, cell_tsize;

	Ctype_A *cells_a = NULL;
	Ctype_B *cells_b = NULL;

	float *coord, *node_data, *cell_data, *model_data, 
        *min_node_data, *max_node_data, *min_cell_data,
        *max_cell_data, *min_model_data, *max_model_data;

	xp_long num_nodes, num_cells;
	int num_node_data, num_cell_data, num_model_data, *bcell_nlists, material_flag;
	int *node_active_list, *cell_active_list, *model_active_list,
	num_node_comp, *node_comp_list, num_cell_comp, *cell_comp_list,
	num_model_comp, *model_comp_list;

	Mtype_A model_stats_a;
    Mtype_B model_stats_b;
    
	Ntype *cell_nlists;

	char   label[MAXLINE], units[MAXLINE];
	OMobj_id cell_set;
	Cells  out_cells[MAX_CELL];
	int    cs, ncell_sets, comp, nc, found, cell_nnodes;
	xp_long set_ncells, *conn_list, *list = NULL;
	int    ctype, mat;
	xp_long ind;
	float  *props, *min = NULL, *max = NULL;


	/**************
	 ***  body  ***
	 **************/

	if (!file_name)
		return(0);

	cell_nlists = NULL;
	bcell_nlists = NULL;
	model_stats_a.mid_edge_flag =0;
    model_stats_b.mid_edge_flag =0;
    
	/*  check to see if file is binary or ascii.  */

	file_version = FUNCcheck_ucd_file(file_name);
	if (file_version == 0) {
		ERRerror("read ucd", 1, ERR_ORIG,
			 "Error in read_ucd: can't read file:\n%s", file_name);
		return (0);
	}

    if (file_version == UCD_ASCII_FILE) {
		model_stats_a.node_data_labels = NULL;
		model_stats_a.cell_data_labels = NULL;
		model_stats_a.model_data_labels = NULL;

		model_stats_a.node_data_units = NULL;
		model_stats_a.cell_data_units = NULL;
		model_stats_a.model_data_units = NULL;

		model_stats_a.node_active_list = NULL;
		model_stats_a.cell_active_list = NULL;
		model_stats_a.model_active_list = NULL;

		model_stats_a.node_comp_list = NULL;
		model_stats_a.cell_comp_list = NULL;
		model_stats_a.model_comp_list = NULL;
    }
    else {
		model_stats_b.node_data_labels = NULL;
		model_stats_b.cell_data_labels = NULL;
		model_stats_b.model_data_labels = NULL;

		model_stats_b.node_data_units = NULL;
		model_stats_b.cell_data_units = NULL;
		model_stats_b.model_data_units = NULL;

		model_stats_b.node_active_list = NULL;
		model_stats_b.cell_active_list = NULL;
		model_stats_b.model_active_list = NULL;

		model_stats_b.node_comp_list = NULL;
		model_stats_b.cell_comp_list = NULL;
		model_stats_b.model_comp_list = NULL;
    }

	if (FLDset_ncell_sets (out, 0) != 1) {
		ERR_RETURN("Error setting ncell_sets");
	}
	if (FLDset_nnodes (out, 0) != 1) {
		ERR_RETURN("Error setting nnodes")
	}
	if (FLDset_node_data_ncomp (out, 0) != 1) {
		ERR_RETURN("Error setting nnode_data");
	}


	if (file_version == UCD_ASCII_FILE) {
		if (!read_ascii (file_name, &cells_a, &cell_nlists,
				 &model_stats_a, &coord,
				 &node_data, &cell_data, &model_data,
				 &min_node_data, &max_node_data,
				 &min_cell_data, &max_cell_data,
				 &min_model_data, &max_model_data,
				 &node_id_a)) {
			ERRerror("read ucd", 1, ERR_ORIG,
				 "Error in read_ucd: can't read file:\n%s",
                                 file_name);
			return (0);
		}
		if (model_stats_a.model_data_labels)
			strcpy(data_labels,
			       model_stats_a.model_data_labels);
		if (model_stats_a.cell_data_labels)
			strcpy(cell_labels,
			       model_stats_a.cell_data_labels);
		if (model_stats_a.node_data_labels)
			strcpy(node_labels,
			       model_stats_a.node_data_labels);
		if (model_stats_a.model_data_units)
			strcpy(data_units,
			       model_stats_a.model_data_units);
		if (model_stats_a.cell_data_units)
			strcpy(cell_units, model_stats_a.cell_data_units);
		if (model_stats_a.node_data_units)
			strcpy(node_units, model_stats_a.node_data_units);

		num_nodes = model_stats_a.num_nodes;
		num_cells = model_stats_a.num_cells;
		num_node_data = model_stats_a.num_node_data;
		num_cell_data = model_stats_a.num_cell_data;
		num_model_data = model_stats_a.num_model_data;
        
        material_flag = model_stats_a.material_flag;
        
		node_active_list = model_stats_a.node_active_list;
		cell_active_list = model_stats_a.cell_active_list;
		model_active_list = model_stats_a.model_active_list;

		node_comp_list = model_stats_a.node_comp_list;
		cell_comp_list = model_stats_a.cell_comp_list;
		model_comp_list = model_stats_a.model_comp_list;

		num_node_comp = model_stats_a.num_node_comp;
		num_cell_comp = model_stats_a.num_cell_comp;
		num_model_comp = model_stats_a.num_model_comp;

		cell_tsize = model_stats_a.num_nlist_nodes;
	}
	else if (file_version == UCD_BINARY_3_0) {
		if (!read_binary3_0 (file_name, &cells_b, &bcell_nlists,
				     &model_stats_b, &coord,
				     &node_data, &cell_data,
				     &model_data, &min_node_data,
				     &max_node_data, &min_cell_data,
				     &max_cell_data, &min_model_data,
				     &max_model_data, &node_id_b)) {
			ERRerror("read ucd", 1, ERR_ORIG,
				 "Error in read_ucd: can't read bin file.\n%s",
                                 file_name);
			return (0);
		}
		if (model_stats_b.model_data_labels)
			strcpy(data_labels,
			       model_stats_b.model_data_labels);
		if (model_stats_b.cell_data_labels)
			strcpy(cell_labels,
			       model_stats_b.cell_data_labels);
		if (model_stats_b.node_data_labels)
			strcpy(node_labels,
			       model_stats_b.node_data_labels);
		if (model_stats_b.model_data_units)
			strcpy(data_units,
			       model_stats_b.model_data_units);
		if (model_stats_b.cell_data_units)
			strcpy(cell_units, model_stats_b.cell_data_units);
		if (model_stats_b.node_data_units)
			strcpy(node_units, model_stats_b.node_data_units);

		num_nodes = model_stats_b.num_nodes;
		num_cells = model_stats_b.num_cells;
		num_node_data = model_stats_b.num_node_data;
		num_cell_data = model_stats_b.num_cell_data;
		num_model_data = model_stats_b.num_model_data;

        material_flag = model_stats_a.material_flag;

		node_active_list = model_stats_b.node_active_list;
		cell_active_list = model_stats_b.cell_active_list;
		model_active_list = model_stats_b.model_active_list;

		node_comp_list = model_stats_b.node_comp_list;
		cell_comp_list = model_stats_b.cell_comp_list;
		model_comp_list = model_stats_b.model_comp_list;

		num_node_comp = model_stats_b.num_node_comp;
		num_cell_comp = model_stats_b.num_cell_comp;
		num_model_comp = model_stats_b.num_model_comp;

		cell_tsize = model_stats_b.num_nlist_nodes;
	}
	else if (file_version == UCD_BINARY_3_5) {
		if (!read_binary3_5 (file_name, &cells_b, &bcell_nlists,
				     &model_stats_b, &coord,
				     &node_data, &cell_data,
				     &model_data, &min_node_data,
				     &max_node_data, &min_cell_data,
				     &max_cell_data, &min_model_data,
				     &max_model_data, &node_id_b)) {
			ERRerror("read ucd", 1, ERR_ORIG,
				 "Error in read_ucd: can't read bin file.\n%s",
                                 file_name);
			return (0);
		}
		if (model_stats_b.model_data_labels)
			strcpy(data_labels,
			       model_stats_b.model_data_labels);
		if (model_stats_b.cell_data_labels)
			strcpy(cell_labels,
			       model_stats_b.cell_data_labels);
		if (model_stats_b.node_data_labels)
			strcpy(node_labels,
			       model_stats_b.node_data_labels);
		if (model_stats_b.model_data_units)
			strcpy(data_units,
			       model_stats_b.model_data_units);
		if (model_stats_b.cell_data_units)
			strcpy(cell_units, model_stats_b.cell_data_units);
		if (model_stats_b.node_data_units)
			strcpy(node_units, model_stats_b.node_data_units);

		num_nodes = model_stats_b.num_nodes;
		num_cells = model_stats_b.num_cells;
		num_node_data = model_stats_b.num_node_data;
		num_cell_data = model_stats_b.num_cell_data;
		num_model_data = model_stats_b.num_model_data;

        material_flag = model_stats_b.material_flag;
        
		node_active_list = model_stats_b.node_active_list;
		cell_active_list = model_stats_b.cell_active_list;
		model_active_list = model_stats_b.model_active_list;

		node_comp_list = model_stats_b.node_comp_list;
		cell_comp_list = model_stats_b.cell_comp_list;
		model_comp_list = model_stats_b.model_comp_list;

		num_node_comp = model_stats_b.num_node_comp;
		num_cell_comp = model_stats_b.num_cell_comp;
		num_model_comp = model_stats_b.num_model_comp;

		cell_tsize = model_stats_b.num_nlist_nodes;

	}
	else {
		ERRerror("read ucd", 0, ERR_ORIG,
                         "Error in read_ucd: unknown file format.");
		return (0);
	}

    if(node_id_a)
       free(node_id_a);
    if(node_id_b)
	   free(node_id_b);

	if (FLDset_nnodes (out, num_nodes) != 1) {
		ERR_RETURN("Error setting nnodes")
	}

	if (FLDset_nspace (out, 3) != 1) {
		ERR_RETURN("Error setting nspace")
	}
	strcpy(data_units, "");
	if (FLDset_coord_units (out, data_units) != 1) {
		ERR_RETURN("Error setting coord units")
	}

	if (FLDset_coord(out, coord, 3*num_nodes, OM_SET_ARRAY_FREE) != 1) {
		ERR_RETURN("Error setting coordinate array")
	}


	/*  store cell type, topology.  */

	list = (xp_long *)malloc(num_cells*sizeof(xp_long));

	if (bcell_nlists) {  /*  if read in as binary file.  */
		for (cell = 0, j_w = 0; cell < num_cells; cell++, j_w += n_w) {
			n_w = cells_b[cell].n;
			list[cell] = j_w;
		}
	}


	/* Init cell structure */
	for (i=0; i<MAX_CELL; i++) {
		strcpy(out_cells[i].cell_name, set_names[i]);
		out_cells[i].nnodes = set_nnodes[i];
		out_cells[i].nmat = 0;
		out_cells[i].mat_count = 1;
		out_cells[i].mat = (int *)malloc(out_cells[i].mat_count * sizeof(int));
		out_cells[i].ncells = (xp_long *)malloc(out_cells[i].mat_count * sizeof(xp_long));
		out_cells[i].cell_list = (xp_long **)malloc(out_cells[i].mat_count * sizeof(xp_long *));
		out_cells[i].csize = (xp_long *)malloc(out_cells[i].mat_count * sizeof(xp_long));
		for (j=0; j<out_cells[i].mat_count; j++) {
			out_cells[i].csize[j] = 0;
			out_cells[i].ncells[j]=0;
			out_cells[i].cell_list[j]= (xp_long *)0;
		}
	}
	/*** READ CELL INFO ***/
	if (material_flag) {
		for (cell = 0; cell < num_cells; cell++) {
			if (file_version == UCD_ASCII_FILE) {
			   ctype = cells_a[cell].cell_type;
			   if (cells_a[cell].n > UCD_num_nodes[cells_a[cell].cell_type])
			      ctype += 8;

			   mat = cells_a[cell].mat;
		    }
		    else {
			   ctype = cells_b[cell].cell_type;
			   if (cells_b[cell].n > UCD_num_nodes[cells_b[cell].cell_type])
			      ctype += 8;

			   mat = cells_b[cell].mat;
			}
			for (found=0, mat_id =0, i=0; i<out_cells[ctype].nmat; i++){
				if (out_cells[ctype].mat[i] == mat) {
					found =1;
					mat_id = i;
				}
			}
			if (!found) {
				if (out_cells[ctype].nmat >= out_cells[ctype].mat_count) {
					out_cells[ctype].mat_count += MAX_MATER;
					out_cells[ctype].mat = (int *)realloc(out_cells[ctype].mat,
									 out_cells[ctype].mat_count * sizeof(int));
					out_cells[ctype].ncells = (xp_long *)realloc(out_cells[ctype].ncells,
									    out_cells[ctype].mat_count * sizeof(xp_long));
					out_cells[ctype].cell_list = (xp_long **)realloc(out_cells[ctype].cell_list,
										out_cells[ctype].mat_count * sizeof(xp_long *));
					out_cells[ctype].csize = (xp_long *)realloc(out_cells[ctype].csize,
									   out_cells[ctype].mat_count * sizeof(xp_long));
					for (j=out_cells[ctype].nmat; j<out_cells[ctype].mat_count; j++) {
						out_cells[ctype].csize[j] = 0;
						out_cells[ctype].ncells[j]=0;
						out_cells[ctype].cell_list[j]= (xp_long *)0;
					}
				}

				ind = out_cells[ctype].nmat;
				out_cells[ctype].mat[ind] = mat;
				mat_id = out_cells[ctype].nmat;
				out_cells[ctype].nmat = out_cells[ctype].nmat+1;
			}

			ind = out_cells[ctype].ncells[mat_id];
			if (ind >= out_cells[ctype].csize[mat_id]) {
				if (out_cells[ctype].csize[mat_id] == 0) {
					out_cells[ctype].csize[mat_id] += BLOCK_SIZE;
					out_cells[ctype].cell_list[mat_id] =
						(xp_long *)malloc(out_cells[ctype].csize[mat_id]*sizeof(xp_long));
				}
				else {
					out_cells[ctype].csize[mat_id] += BLOCK_SIZE;
					out_cells[ctype].cell_list[mat_id] =
						(xp_long *)realloc(out_cells[ctype].cell_list[mat_id],
							       out_cells[ctype].csize[mat_id]*sizeof(xp_long));
				}
			}
			out_cells[ctype].cell_list[mat_id][ind] = cell;
			out_cells[ctype].ncells[mat_id] = out_cells[ctype].ncells[mat_id]+1;
		}
	}
	else {
		for (cell = 0; cell < num_cells; cell++) {
			if (file_version == UCD_ASCII_FILE) {
			   ctype = cells_a[cell].cell_type;
			   if (cells_a[cell].n > UCD_num_nodes[cells_a[cell].cell_type])
			      ctype += 8;
		    }
		    else {
			   ctype = cells_b[cell].cell_type;
			   if (cells_b[cell].n > UCD_num_nodes[cells_b[cell].cell_type])
			      ctype += 8;				
		    }
			ind = out_cells[ctype].ncells[0];
			if (ind >= out_cells[ctype].csize[0]) {
				if (out_cells[ctype].csize[0] == 0) {
					out_cells[ctype].csize[0] += BLOCK_SIZE;
					out_cells[ctype]. cell_list[0] =
						(xp_long *)malloc(out_cells[ctype].csize[0]*sizeof(xp_long));
				}
				else {
					out_cells[ctype].csize[0] += BLOCK_SIZE;
					out_cells[ctype].cell_list[0] =
						(xp_long *)realloc(out_cells[ctype].cell_list[0],
							       out_cells[ctype].csize[0]*sizeof(xp_long));
				}
			}
			out_cells[ctype].nmat = 1;
			out_cells[ctype].cell_list[0][ind] = cell;
			out_cells[ctype].ncells[0] = out_cells[ctype].ncells[0]+1;
		}
	}

	for (ncell_sets=0, i=0; i<MAX_CELL; i++)
		for (j=0; j<out_cells[i].nmat; j++)
			if (out_cells[i].ncells[j] > 0)
				ncell_sets++;

	cell_set = OMfind_subobj(out, OMstr_to_name("cell_set"), OM_OBJ_RW);
	OMreset_obj(cell_set,0);

	if (FLDset_ncell_sets (out, ncell_sets) != 1) {
		ERR_RETURN("Error setting ncell_sets");
	}


	type = DTYPE_FLOAT;
	if (num_node_data) {
		if (FLDset_node_data_ncomp (out, num_node_comp) != 1) {
			ERR_RETURN("Error setting nnode_data");
		}
		for (offset=0, i=0; i< num_node_comp; i++) {
			if (!get_label(node_labels, '.', i, label))
				sprintf(label, "node_data_%d",i);

			if (!get_label(node_units, '.', i, units))
				strcpy(units, " ");

			veclen = node_comp_list[i];
			if (FLDset_node_data_comp (out, i, veclen, label,
						   units) != 1) {
				ERR_RETURN("Error setting node component");
			}

			if (FLDget_node_data(out, i, &type,
					     (char **)&(out_data), &size,
					     OM_GET_ARRAY_WR) != 1) {
				ERR_RETURN("Error getting node data for writing");
			}
			for (n_w=0; n_w<num_nodes; n_w++)  {
				memcpy(out_data+n_w*veclen,
				       node_data+offset+n_w*veclen,
				       veclen*sizeof(float));
			}
			offset += veclen*num_nodes;
			ARRfree(out_data);
		}
		free(node_data);
	}
	else {
		if (FLDset_node_data_ncomp (out, 0) != 1) {
			ERR_RETURN("Error setting nnode_data");
		}
	}

	if (num_cell_data) {
		min = (float *)malloc(num_cell_data*sizeof(float));
		max = (float *)malloc(num_cell_data*sizeof(float));
		for (offset=0, comp=0; comp < num_cell_comp; comp++)  {
			veclen = cell_comp_list[comp];
			if (veclen == 1) {
				min[comp] = cell_data[offset];
				max[comp] = cell_data[offset];
				for (n_w=1; n_w<num_cells; n_w++)  {
					if (min[comp] > cell_data[offset+n_w])
						min[comp] = cell_data[offset+n_w];
					if (max[comp] < cell_data[offset+n_w])
						max[comp] = cell_data[offset+n_w];
				}
			}
			offset += veclen*num_cells;
		}
	}
	/*  store cell data.  */
	for (cs=0, i=0; i<MAX_CELL; i++) {
		for (j=0; j<out_cells[i].nmat; j++) {
			if (out_cells[i].ncells[j] == 0)
				continue;
			if (FLDget_cell_set(out, cs, &cell_set) != 1) {
				ERR_RETURN("Error getting out cell set");
			}
			if (FLDset_cell_set(cell_set, out_cells[i].cell_name) != 1) {
				ERR_RETURN("Error setting cell type");
			}
			if (FLDset_ncells(cell_set, out_cells[i].ncells[j]) != 1) {
				ERR_RETURN("Error setting ncells");
			}
			if (FLDget_cell_set_nnodes(cell_set, &cell_nnodes) != 1) {
				ERR_RETURN("Error setting cell nnodes");
			}
			if (FLDget_node_connect(cell_set, &conn_list, &size,
						OM_GET_ARRAY_WR) != 1) {
				ERR_RETURN("Error getting cell connect list for writing");
			}
			if (num_cell_data) {
				if (FLDset_cell_data_ncomp (cell_set, num_cell_comp) != 1) {
					ERR_RETURN("Error setting ncell_data");
				}
				for (offset=0, comp=0; comp < num_cell_comp; comp++) {
					if (!get_label(cell_labels, '.', comp, label))
						sprintf(label, "cell_data_%d",i);

					if (!get_label(cell_units, '.', comp, units))
						strcpy(units, " ");
					veclen = cell_comp_list[comp];
					if (FLDset_cell_data_comp (cell_set, comp, veclen,
								   label, units) != 1) {
						ERR_RETURN("Error setting cell component");
					}

					if (FLDget_cell_data(cell_set, comp, &type,
							     (char **)&(out_data), &size,
							     OM_GET_ARRAY_WR) != 1) {
						ERR_RETURN("Error getting cell data for writing");
					}
					for (n_w=0; n_w<out_cells[i].ncells[j]; n_w++)  {
						memcpy(out_data+n_w*veclen,
						       cell_data+offset+
							out_cells[i].cell_list[j][n_w]*veclen,
						       veclen*sizeof(float));
					}
					if (veclen == 1) {
						if (FLDset_cell_data_minmax(cell_set, comp, 
									    (char *)&min[comp], (char *)&max[comp], type) != 1) {
							ERR_RETURN("Error setting cell data min max");
						}
					}
					offset += veclen*num_cells;
					ARRfree(out_data);
				}
			}
			set_ncells = out_cells[i].ncells[j];
			for (cell=0, nc=0; cell<set_ncells; cell++) {
				ind = out_cells[i].cell_list[j][cell];

				if (bcell_nlists)   /* if read in as binary file. */
					for (n=0; n<cell_nnodes; n++)
						conn_list[nc++] = bcell_nlists[list[ind]+n];
				else
					for (n=0; n<cell_nnodes; n++)
						conn_list[nc++] = cell_nlists[ind].node_list[n];
			}
			if (material_flag) {
				if (FLDset_cell_nprops (cell_set, 1) != 1) {
					ERR_RETURN("Error setting cell nprops");
				}
				if (FLDget_cell_props(cell_set, (float **)&props,
						      &prop_size, OM_GET_ARRAY_WR) != 1) {
					ERR_RETURN("Error getting cell props for writing");
				}
				props[0] = (float)out_cells[i].mat[j];
				ARRfree(props);
			}
			cs++;
			ARRfree(conn_list);
		}
	}

	/* Free cell structure */
	for (i=0; i<MAX_CELL; i++) {
		for (j=0; j<out_cells[i].mat_count; j++)
			if (out_cells[i].csize[j])
				free(out_cells[i].cell_list[j]);
		free(out_cells[i].mat);
		free(out_cells[i].ncells);
		free(out_cells[i].cell_list);
		free(out_cells[i].csize);
	}

	if (num_node_data) {
		free (min_node_data);
		free (max_node_data);
	}

	if (num_cell_data) {
		if(min)
			free(min);
		if (max)
			free(max);

		free (cell_data);
		free (min_cell_data);
		free (max_cell_data);
	}

	if (num_model_data) {
		free (model_data);
		free (min_model_data);
		free (max_model_data);
	}

	if (list)
		free(list);

    if (file_version == UCD_ASCII_FILE) {
		
		if(cells_a)
		   free (cells_a);

		if (cell_nlists)
			free (cell_nlists);


		if (model_stats_a.node_data_labels )
			free (model_stats_a.node_data_labels );
		if (model_stats_a.cell_data_labels )
			free (model_stats_a.cell_data_labels );
		if (model_stats_a.model_data_labels )
			free (model_stats_a.model_data_labels );

		if (model_stats_a.node_data_units )
			free (model_stats_a.node_data_units );
		if (model_stats_a.cell_data_units )
			free (model_stats_a.cell_data_units );
		if (model_stats_a.model_data_units )
			free (model_stats_a.model_data_units );

		if (model_stats_a.node_active_list )
			free (model_stats_a.node_active_list );
		if (model_stats_a.cell_active_list )
			free (model_stats_a.cell_active_list );
		if (model_stats_a.model_active_list )
			free (model_stats_a.model_active_list );

		if (model_stats_a.node_comp_list )
			free (model_stats_a.node_comp_list );
		if (model_stats_a.cell_comp_list )
			free (model_stats_a.cell_comp_list );
		if (model_stats_a.model_comp_list )
			free (model_stats_a.model_comp_list );
    }
    else {

		if(cells_b)
		   free (cells_b);

		if (bcell_nlists)
			free (bcell_nlists);

		if (model_stats_b.node_data_labels )
			free (model_stats_b.node_data_labels );
		if (model_stats_b.cell_data_labels )
			free (model_stats_b.cell_data_labels );
		if (model_stats_b.model_data_labels )
			free (model_stats_b.model_data_labels );

		if (model_stats_b.node_data_units )
			free (model_stats_b.node_data_units );
		if (model_stats_b.cell_data_units )
			free (model_stats_b.cell_data_units );
		if (model_stats_b.model_data_units )
			free (model_stats_b.model_data_units );

		if (model_stats_b.node_active_list )
			free (model_stats_b.node_active_list );
		if (model_stats_b.cell_active_list )
			free (model_stats_b.cell_active_list );
		if (model_stats_b.model_active_list )
			free (model_stats_b.model_active_list );
		
		if (model_stats_b.node_comp_list )
			free (model_stats_b.node_comp_list );
		if (model_stats_b.cell_comp_list )
			free (model_stats_b.cell_comp_list );
		if (model_stats_b.model_comp_list )
			free (model_stats_b.model_comp_list );
    }

	return(1);
}


/*-----------------------------------------------------*
 *                                                     *
 *            ****  read_data  ****                    *
 *                                                     *
 * read a block of data from an ascii file.            *
 *-----------------------------------------------------*/
/* 64-bit porting. Directly Modified */
static int
read_data(FILE *fp, xp_long num_data, float *data, float *mini_data, 
          float *maxi_data, char *data_labels, char *units, char *string,
          int *num_comp, int *comp_list, int data_veclen)
{
	char line[BUFFER_SIZE];

	float value;

	int num_vals, j, i, n, m, sum;
	xp_long offset, node, id;

	/**************
	 ***  body  ***
	 **************/

	/*  read the number of components and their dimension.  */

	fscanf (fp, "%d", num_comp);

	for (i = 0; i < *num_comp; i++)
		fscanf (fp, "%d", &comp_list[i]);

	sum = 0;
	for (i = 0; i < *num_comp; i++)
		sum += comp_list[i];
	if (sum != data_veclen) {
		ERRerror("read ucd", 0, ERR_ORIG, 
                         "Error in read_ucd: bad number of components.");
		return (0);
	}

	fgets (line, BUFFER_SIZE, fp);

	/*  read the component names and units.  */

	for (i = 0, n = 0, m = 0; i < *num_comp; i++) {
		fgets (line, BUFFER_SIZE, fp);
		if (i == 0) strcpy (string, line);

		for (j = 0; line[j] == ' '; j++);

		for (; line[j] != ','; j++) {
			if (n >= UCD_LABEL_LEN) {
				ERRerror("read ucd", 0, ERR_ORIG, "Error in read_ucd: labels string is too long.");
				return(0);
			}
			data_labels[n++] = line[j];
		}

		for (++j; line[j] == ' '; j++);

		for (; line[j] != '\n'; j++) {
			if (m >= UCD_LABEL_LEN) {
				ERRerror("read ucd", 0, ERR_ORIG, "Error in read_ucd: units string is too long.");
				return(0);
			}
			units[m++] = line[j];
		}

		if (i != *num_comp - 1) {
			if (n >= UCD_LABEL_LEN) {
				ERRerror("read ucd", 0, ERR_ORIG, "Error in read_ucd: labels string is too long.");
				return(0);
			}
			data_labels[n++] = '.';
			if (m >= UCD_LABEL_LEN) {
				ERRerror("read ucd", 0, ERR_ORIG, "Error in read_ucd: units string is too long.");
				return(0);
			}
			units[m++] = '.';
		}
	}

	if (n >= UCD_LABEL_LEN) {
		ERRerror("read ucd", 0, ERR_ORIG, "Error in read_ucd: labels string is too long.");
		return(0);
	}
	if (m >= UCD_LABEL_LEN) {
		ERRerror("read ucd", 0, ERR_ORIG, "Error in read_ucd: units string is too long.");
		return(0);
	}
	data_labels[n] = '\0';
	units[m] = '\0';

	for (i = 0; i < *num_comp; i++)
		mini_data[i] = maxi_data[i] = 0.0;

	for (node = 0; node < num_data; node++) {
		fscanf (fp, "%ld", &id);
		offset = 0;

		for (i = 0; i < *num_comp; i++) {
			num_vals = comp_list[i];

			for (j = 0; j < num_vals; j++) {
				fscanf (fp, "%f",  &value);
				data[node * num_vals + offset + j] = value;
			}


			/*  only set max/min for scalar data.  */

			if (num_vals == 1) {
				if (node == 0)
					mini_data[i] = maxi_data[i] = value;
				else {
					if (mini_data[i] > value)
					    mini_data[i] = value;
					if (maxi_data[i] < value)
					    maxi_data[i] = value;
				}
			}

			offset += num_data * num_vals;
		}
	}
	return(1);
}


/*-----------------------------------------------------*
 *                                                     *
 *               ****  check_file  ****                *
 *                                                     *
 * check to see if the file to read is binary or not.  *
 *-----------------------------------------------------*/
int
FUNCcheck_ucd_file(char *file_name)
{
	int c;
	FILE *fp;

	/**************
	 ***  body  ***
	 **************/

	if (!(fp = FILEfopen (file_name, SIO_R_BIN))) {
		ERRerror("read ucd", 1, ERR_ORIG,
                         "Error in read_ucd: can't read file:\n%s",
			  file_name);
		return (0);
	}

	c = fgetc(fp);
	if (c == EOF) {
		fclose (fp);
		return (0);
	}
	else if (c == 0x5) {
		fclose (fp);
		return (UCD_BINARY_3_0);
	}
	else if (c == 0x7) {
		fclose (fp);
		return (UCD_BINARY_3_5);
	}
	else {
		fclose (fp);
		return (UCD_ASCII_FILE);
	}
}

/*-----------------------------------------------------*
 *                                                     *
 *               ****  read_ascii  ****                *
 *                                                     *
 * read a ucd ascii file.                              *
 *-----------------------------------------------------*/
/* 64-bit porting. Directly Modified */
static int
read_ascii(char *file_name, Ctype_A **pcells, Ntype **pcell_nlists, 
           Mtype_A *model, float **coord, 
           float **pnode_data, float **pcell_data, float **pmodel_data, 
           float **pmin_node_data, float **pmax_node_data, 
           float **pmin_cell_data, float **pmax_cell_data,
           float **pmin_model_data, float **pmax_model_data, xp_long **node_id)
{
	char *tl, string[UCD_LABEL_LEN], line[BUFFER_SIZE], 
	cell_type[UCD_NUM_CELL_TYPES];

	Ctype_A *cells;

	FILE *fp;

	float *node_data, *cell_data, *model_data, x, y, z,
        *min_node_data, *max_node_data, *min_cell_data, *max_cell_data,
        *min_model_data, *max_model_data;

	xp_long node,  cell, num_nodes, num_cells, k, *node_ids, *node_ind, num_nlist_nodes;
	int num_node_data, num_cell_data, num_model_data, i, n, sum, ucd_cell_type;

	Ntype *cell_nlists;

	static const char *cell_list[] = {"pt", "line", "tri", "quad", "tet",
					    "pyr", "prism", "hex"};

	/**************
	 ***  body  ***
	 **************/

	model->node_data_labels = (char *) malloc(UCD_LABEL_LEN * sizeof(char));
	model->cell_data_labels = (char *) malloc(UCD_LABEL_LEN * sizeof(char));
	model->model_data_labels = (char *)malloc(UCD_LABEL_LEN * sizeof(char));

	model->node_data_units = (char *) malloc(UCD_LABEL_LEN * sizeof(char));
	model->cell_data_units = (char *) malloc(UCD_LABEL_LEN * sizeof(char));
	model->model_data_units = (char *) malloc(UCD_LABEL_LEN * sizeof(char));

	strcpy(model->node_data_labels, "");
	strcpy(model->cell_data_labels, "");
	strcpy(model->model_data_labels, "");
	strcpy(model->node_data_units, "");
	strcpy(model->cell_data_units, "");
	strcpy(model->model_data_units, "");

	node_data = cell_data = model_data = NULL;

	if (!(fp = FILEfopen (file_name, SIO_R_TXT)))
		return (0);

	fgets (line, BUFFER_SIZE, fp);

	/*  skip comments.  */

	while (line[0] == '#')
		fgets (line, BUFFER_SIZE, fp);

	/*  read model stats.  */

	sscanf (line, "%ld %ld %d %d %d", &num_nodes, &num_cells, &num_node_data,
		&num_cell_data, &num_model_data);


	cells = NULL;
	cell_nlists = NULL;
	cells = (Ctype_A *)malloc(sizeof(Ctype_A) * num_cells);
	cell_nlists = (Ntype *)malloc(sizeof(Ntype) * num_cells);
	if (cells == NULL || cell_nlists == NULL) {
		ERRerror("read ucd", 0, ERR_ORIG, "Error in read_ucd: can't alloc cells ids.");
		fclose (fp);
		return (0);
	}
	num_nlist_nodes = 0;

	*node_id = node_ids = node_ind = NULL;
	*node_id = (xp_long *)malloc(sizeof(xp_long) * num_nodes);
	node_ids = (xp_long *)malloc(sizeof(xp_long) * num_nodes);
	node_ind = (xp_long *)malloc(sizeof(xp_long) * num_nodes);
	if (*node_id == NULL || node_ids == NULL || node_ind == NULL) {
		ERRerror("read ucd", 0, ERR_ORIG, "Error in read_ucd: can't alloc node ids.");
		fclose (fp);
		return (0);
	}

	/*  read nodal coordinates.  */
	*coord = (float *)ARRalloc(NULL, DTYPE_FLOAT, num_nodes*3, NULL);

	for (node = 0; node < num_nodes; node++) {
		fscanf (fp, "%ld %f %f %f", *node_id + node, &x, &y, &z);
		(*coord)[node*3] = x;
		(*coord)[node*3+1] = y;
		(*coord)[node*3+2] = z;
		node_ids[node] = *(*node_id + node);
		node_ind[node] = node;
	}

	/* sort node id's */
	ucd_qsorti(node_ids, node_ind, num_nodes);

	/*  read cell type, topology.  */

	fgets (line, BUFFER_SIZE, fp);

	model->mid_edge_flag = 0;
	model->material_flag = 0;
	for (cell = 0; cell < num_cells; cell++) {
		fgets (line, BUFFER_SIZE, fp);
		tl = line;
		n = 0;

		get_token (&tl, &cells[cell].id, "%ld");
		get_token (&tl, &cells[cell].mat, "%d");

		i = get_token (&tl, cell_type, "%s");
		cell_type[i] = '\0';

		for (ucd_cell_type = 0; ucd_cell_type < UCD_NUM_CELL_TYPES;
		     ucd_cell_type++)
			if (!strcmp(cell_list[ucd_cell_type], cell_type))
				break;

		if (ucd_cell_type < UCD_NUM_CELL_TYPES)
			cells[cell].cell_type = ucd_cell_type;
		else {
			ERRerror("read ucd", 0, ERR_ORIG, "Error in read_ucd: unknown cell type.");
			cells[cell].cell_type = 0;
		}
		while (get_token(&tl, &node, "%ld")){
			k = ucd_bsearchi(node, node_ids, num_nodes);
			cell_nlists[cell].node_list[n++] = node_ind[k];
		}
		cell_nlists[cell].n = cells[cell].n = n;
		num_nlist_nodes += n;
		if (n > UCD_num_nodes[ucd_cell_type])
			model->mid_edge_flag = 1;
		if (cells[cell].mat != 0 ||  /* CFS PR 25643 */
		    cells[cell].mat != cells[0].mat)
			model->material_flag = 1;
	}
	free (node_ids);
	free (node_ind);


	/*  read nodal data.  */

	model->node_active_list = model->node_comp_list = NULL;
	node_data = min_node_data = max_node_data =NULL;
	model->cell_active_list = model->cell_comp_list = NULL;
	cell_data = min_cell_data = max_cell_data =NULL;
	model->model_active_list = model->model_comp_list = NULL;
	model_data = min_model_data = max_model_data =NULL;
	if (num_node_data) {
		model->node_active_list = (int *)
					  malloc(num_node_data *sizeof(int));
		model->node_comp_list = (int *)
					malloc(num_node_data *sizeof(int));
		node_data = (float *)
			    malloc(sizeof(float) * num_nodes * num_node_data);
		min_node_data = (float *)malloc(sizeof(float) * num_node_data);
		max_node_data = (float *)malloc(sizeof(float) * num_node_data);
		if (model->node_active_list == NULL ||
		    model->node_comp_list == NULL || node_data == NULL ||
		    min_node_data == NULL || max_node_data == NULL ) {
			ERRerror("read ucd", 0, ERR_ORIG, "Error in read_ucd: can't alloc node data.");
			fclose (fp);
			return (0);
		}
		if (!read_data (fp, num_nodes, node_data, min_node_data,
				max_node_data, model->node_data_labels,
				model->node_data_units, string,
				&model->num_node_comp, model->node_comp_list,
				num_node_data )) {
			fclose (fp);
			return(0);
		}

		sum = 0;
		for (i = 0; i < model->num_node_comp; i++) {
			sum += model->node_comp_list[i];
			model->node_active_list[i] = 0;
		}
		if (sum != num_node_data) {
			ERRerror("read ucd", 0, ERR_ORIG, "Error in read_ucd: bad number of nodal components.");
			fclose (fp);
			return (0);
		}
		model->node_active_list[0] = 1;
	}


	/*  read cell data.  */

	if (num_cell_data) {
		model->cell_active_list = (int *)
					  malloc(num_cell_data *sizeof(int));
		model->cell_comp_list = (int *)
					malloc(num_cell_data *sizeof(int));
		cell_data = (float *)
			    malloc(sizeof(float) * num_cells * num_cell_data);
		min_cell_data = (float *)malloc(sizeof(float) * num_cell_data);
		max_cell_data = (float *)malloc(sizeof(float) * num_cell_data);

		if (model->cell_active_list == NULL ||
		    model->cell_comp_list == NULL || cell_data == NULL ||
		    min_cell_data == NULL || max_cell_data == NULL ) {
			ERRerror("read ucd", 0, ERR_ORIG, "Error in read_ucd: can't alloc cell data.");
			fclose (fp);
			return (0);
		}
		if (!read_data (fp, num_cells, cell_data, min_cell_data,
				max_cell_data, model->cell_data_labels,
				model->cell_data_units, string,
				&model->num_cell_comp, model->cell_comp_list,
				num_cell_data)) {
			fclose (fp);
			return(0);
		}

		sum = 0;
		for (i = 0; i < model->num_cell_comp; i++) {
			sum += model->cell_comp_list[i];
			model->cell_active_list[i] = 0;
		}
		if (sum != num_cell_data) {
			ERRerror("read ucd", 0, ERR_ORIG, "Error in read_ucd: bad number of cell components.");
			fclose (fp);
			return (0);
		}
		model->cell_active_list[0] = 1;

	}


	/*  read and store model data.  */

	if (num_model_data) {
		/* gnat 7932: read-ahead for the num_comps and then put
		 * the fp back for read_data() to read it again!
		 */
		xp_long offset = ftell(fp);
		int num_comp;
		fscanf (fp, "%d", &num_comp);
		fseek(fp, offset, 0);

		model->model_active_list = (int *)
					   malloc(num_model_data *sizeof(int));
		model->model_comp_list = (int *)
					 malloc(num_model_data *sizeof(int));
		model_data = (float *)malloc(sizeof(float) * num_model_data * num_comp);
		min_model_data =(float *)malloc(sizeof(float) * num_model_data);
		max_model_data =(float *)malloc(sizeof(float) * num_model_data);

		if (model->model_active_list == NULL ||
		    model->model_comp_list == NULL || model_data == NULL ||
		    min_model_data == NULL || max_model_data == NULL ) {
			ERRerror("read ucd", 0, ERR_ORIG, "Error in read_ucd: can't alloc model data.");
			fclose (fp);
			return (0);
		}
		if (!read_data (fp, num_model_data, model_data, min_model_data,
				max_model_data, model->model_data_labels,
				model->model_data_units, string,
				&model->num_model_comp, model->model_comp_list,
				num_model_data)) {
			fclose (fp);
			return(0);
		}
	}

	*pcells = cells;
	*pnode_data = node_data;
	*pcell_data = cell_data;
	*pcell_nlists = cell_nlists;
	*pmodel_data = model_data;

	model->num_nodes = num_nodes;
	model->num_cells = num_cells;
	model->num_nlist_nodes = num_nlist_nodes;
	model->num_node_data = num_node_data;
	model->num_cell_data = num_cell_data;
	model->num_model_data = num_model_data;

	*pmax_node_data = max_node_data;
	*pmin_node_data = min_node_data;
	*pmax_cell_data = max_cell_data;
	*pmin_cell_data = min_cell_data;
	*pmax_model_data = max_model_data;
	*pmin_model_data = min_model_data;

	fclose (fp);

	return(1);
}


/*-----------------------------------------------------*
 *                                                     *
 *               ****  read_binary 3.0 *               *
 *                                                     *
 * read a ucd binary file.   3.0 version               *
 *-----------------------------------------------------*/
/* 64-bit porting. Only Modified Internally */
static int
read_binary3_0(char *file_name, Ctype_B **pcells, int **pcell_nlists, 
               Mtype_B *model, float **coord, 
               float **pnode_data, float **pcell_data, float **pmodel_data, 
               float **pmin_node_data, float **pmax_node_data, 
               float **pmin_cell_data, float **pmax_cell_data, 
               float **pmin_model_data, float **pmax_model_data,
               int **node_id)
{
	char 	magic;

	Ctype_B *cells;

	FILE *fp;

	float *xc, *node_data, *cell_data, *model_data,
        *min_node_data, *max_node_data, *min_cell_data, *max_cell_data,
        *min_model_data, *max_model_data;

	int node, cell, num_nodes, num_cells, num_node_data, 
	num_cell_data, num_model_data, i,
	*cell_nlists, num_nlist_nodes;

	/**************
	 ***  body  ***
	 **************/

	model->node_data_labels = (char *) malloc(100 * sizeof(char));
	model->cell_data_labels = (char *) malloc(100 * sizeof(char));
	model->model_data_labels = (char *) malloc(100 * sizeof(char));

	model->node_data_units = (char *) malloc(100 * sizeof(char));
	model->cell_data_units = (char *) malloc(100 * sizeof(char));
	model->model_data_units = (char *) malloc(100 * sizeof(char));

	model->node_active_list = (int *) malloc(20 *sizeof(int));
	model->cell_active_list = (int *) malloc(20 *sizeof(int));
	model->model_active_list = (int *) malloc(20 *sizeof(int));

	model->node_comp_list = (int *) malloc(20 *sizeof(int));
	model->cell_comp_list = (int *) malloc(20 *sizeof(int));
	model->model_comp_list = (int *) malloc(20 *sizeof(int));

	node_data = cell_data = model_data = NULL;

	if (!(fp = FILEfopen (file_name, SIO_R_BIN)))
		return (0);

	fread (&magic, sizeof(char), 1, fp);

	fread (model->node_data_labels, sizeof(char), 100, fp);
	fread (model->cell_data_labels, sizeof(char), 100, fp);
	fread (model->model_data_labels, sizeof(char), 100, fp);
	fread (model->node_data_units, sizeof(char), 100, fp);
	fread (model->cell_data_units, sizeof(char), 100, fp);
	fread (model->model_data_units, sizeof(char), 100, fp);

	fread (&model->num_nodes, sizeof(int), 1, fp);
	fread (&model->num_cells, sizeof(int), 1, fp);	
	fread (&model->num_node_data, sizeof(int), 1, fp);
	fread (&model->num_cell_data, sizeof(int), 1, fp);
	fread (&model->num_model_data, sizeof(int), 1, fp);

	fread (model->node_active_list, sizeof(int), 20, fp);
	fread (model->cell_active_list, sizeof(int), 20, fp);
	fread (model->model_active_list, sizeof(int), 20, fp);

	fread (&model->num_node_comp, sizeof(int), 1, fp);
	fread (model->node_comp_list, sizeof(int), 20, fp);
	fread (&model->num_cell_comp, sizeof(int), 1, fp);
	fread (model->cell_comp_list, sizeof(int), 20, fp);
	fread (&model->num_model_comp, sizeof(int), 1, fp);
	fread (model->model_comp_list, sizeof(int), 20, fp);	
	fread (&model->num_nlist_nodes, sizeof(int), 1, fp);

	num_nodes = model->num_nodes;
	num_cells = model->num_cells;
	num_node_data = model->num_node_data;
	num_cell_data = model->num_cell_data;
	num_model_data = model->num_model_data;
	num_nlist_nodes = model->num_nlist_nodes;

	*node_id = NULL;
	*node_id = (int *)malloc(sizeof(int) * num_nodes);
	if (*node_id == NULL) {
		ERRerror("read ucd", 0, ERR_ORIG, "Error in read_ucd: can't alloc node ids.");
		return (0);
	}
	for (node = 0; node < num_nodes; node++)
		*(*node_id + node) = node+1;
	cells = NULL, cell_nlists = NULL;
	cells = (Ctype_B *)malloc(sizeof(Ctype_B) * num_cells);
	cell_nlists = (int *)malloc(sizeof(int) * num_nlist_nodes);
	if (cells == NULL || cell_nlists == NULL) {
		ERRerror("read ucd", 0, ERR_ORIG, "Error in read_ucd: can't alloc cells.");
		fclose (fp);
		return (0);
	}

	fread  (cells, sizeof(Ctype_B), num_cells, fp);

	model->mid_edge_flag = 0;
	model->material_flag = 0;
	for (cell=0; cell < num_cells; cell++) {
		if (cells[cell].n > UCD_num_nodes[cells[cell].cell_type]) {
			model->mid_edge_flag = 1;
			break;
		}
	}
	for (cell=1; cell < num_cells; cell++) {
		if (cells[cell].mat != cells[0].mat ) {
			model->material_flag = 1;
			break;
		}
	}

	fread  (cell_nlists, sizeof(int), num_nlist_nodes, fp);
	for (i = 0; i < num_nlist_nodes; i++)
		cell_nlists[i]--;

	xc =NULL;
	xc = (float *)malloc(sizeof(float) * num_nodes);
	if (xc == NULL) {
		ERRerror("read ucd", 0, ERR_ORIG, "Error in read_ucd: can't alloc coordinates.");
		fclose (fp);
		return (0);
	}

	*coord = (float *)ARRalloc(NULL, DTYPE_FLOAT, num_nodes*3, NULL);
	fread (xc, sizeof(float), num_nodes, fp);

	for (i=0; i<num_nodes; i++)
		(*coord)[3*i] = xc[i];

	fread (xc, sizeof(float), num_nodes, fp);
	for (i=0; i<num_nodes; i++)
		(*coord)[3*i+1] = xc[i];

	fread (xc, sizeof(float), num_nodes, fp);
	for (i=0; i<num_nodes; i++)
		(*coord)[3*i+2] = xc[i];
	free(xc);

	node_data = NULL, min_node_data = NULL, max_node_data = NULL;
	if (num_node_data) {
		node_data = (float *)
			    malloc(sizeof(float) * num_nodes * num_node_data);
		min_node_data = (float *)malloc(sizeof(float) * num_node_data);
		max_node_data = (float *)malloc(sizeof(float) * num_node_data);

		if (node_data == NULL || min_node_data == NULL ||
		    max_node_data == NULL) {
			ERRerror("read ucd", 0, ERR_ORIG, "Error in read_ucd: can't alloc node data.");
			fclose (fp);
			return (0);
		}
		fread (min_node_data, sizeof(float), num_node_data, fp);
		fread (max_node_data, sizeof(float), num_node_data, fp);
		fread (node_data, sizeof(float),
		       num_node_data * num_nodes, fp);
	}

	cell_data = NULL, min_cell_data = NULL, max_cell_data = NULL;
	if (num_cell_data) {
		cell_data = (float *)
			    malloc(sizeof(float) * num_cells * num_cell_data);
		min_cell_data = (float *)malloc(sizeof(float) * num_cell_data);
		max_cell_data = (float *)malloc(sizeof(float) * num_cell_data);

		if (cell_data == NULL || min_cell_data == NULL ||
		    max_cell_data == NULL) {
			ERRerror("read ucd", 0, ERR_ORIG, "Error in read_ucd: can't alloc cell data.");
			fclose (fp);
			return (0);
		}
		fread (min_cell_data, sizeof(float), num_cell_data, fp);
		fread (max_cell_data, sizeof(float), num_cell_data, fp);
		fread (cell_data, sizeof(float),
		       num_cell_data * num_cells, fp);
	}

	model_data = NULL, min_model_data = NULL, max_model_data = NULL;
	if (num_model_data) {
		model_data = (float *)malloc(sizeof(float) * num_model_data);
		min_model_data =(float *)malloc(sizeof(float) * num_model_data);
		max_model_data =(float *)malloc(sizeof(float) * num_model_data);

		if (model_data == NULL || min_model_data == NULL ||
		    max_model_data == NULL) {
			ERRerror("read ucd", 0, ERR_ORIG, "Error in read_ucd: can't alloc model data.");
			fclose (fp);
			return (0);
		}
		fread (min_model_data, sizeof(float), num_model_data, fp);
		fread (max_model_data, sizeof(float), num_model_data, fp);
		fread (model_data, sizeof(float), num_model_data, fp);
	}

	*pcells = cells;
	*pcell_nlists = cell_nlists;
	*pnode_data = node_data;
	*pcell_data = cell_data;
	*pmodel_data = model_data;

	model->num_nodes = num_nodes;
	model->num_cells = num_cells;
	model->num_node_data = num_node_data;
	model->num_cell_data = num_cell_data;
	model->num_model_data = num_model_data;

	*pmax_node_data = max_node_data;
	*pmin_node_data = min_node_data;
	*pmax_cell_data = max_cell_data;
	*pmin_cell_data = min_cell_data;
	*pmax_model_data = max_model_data;
	*pmin_model_data = min_model_data;

	fclose (fp);

	return(1);
}

/*-----------------------------------------------------*
 *                                                     *
 *               ****  read_binary  3.5                *
 *                                                     *
 * read a ucd binary file.        3.5 version          *
 *-----------------------------------------------------*/
/* 64-bit porting. Only Modified Internally */
static int
read_binary3_5(char *file_name, Ctype_B **pcells, int **pcell_nlists, 
               Mtype_B *model, float **coord, 
               float **pnode_data, float **pcell_data, float **pmodel_data, 
               float **pmin_node_data, float **pmax_node_data, 
               float **pmin_cell_data, float **pmax_cell_data, 
               float **pmin_model_data, float **pmax_model_data,
               int **node_id)
{
	char 	magic;

	Ctype_B *cells;

	FILE *fp;

	float *xc, *node_data, *cell_data, *model_data,
        *min_node_data, *max_node_data, *min_cell_data, *max_cell_data,
        *min_model_data, *max_model_data;

	int node, cell, num_nodes, num_cells, num_node_data, 
	num_cell_data, num_model_data, i, *cell_nlists, num_nlist_nodes;

	/**************
	 ***  body  ***
	 **************/

	node_data = cell_data = model_data = NULL;

	if (!(fp = FILEfopen (file_name, SIO_R_BIN)))
		return (0);

	fread (&magic, sizeof(char), 1, fp);
	fread (&model->num_nodes, sizeof(int), 1, fp);
	fread (&model->num_cells, sizeof(int), 1, fp);
	fread (&model->num_node_data, sizeof(int), 1, fp);
	fread (&model->num_cell_data, sizeof(int), 1, fp);
	fread (&model->num_model_data, sizeof(int), 1, fp);
	fread (&model->num_nlist_nodes, sizeof(int), 1, fp);
	num_nodes = model->num_nodes;
	num_cells = model->num_cells;
	num_node_data = model->num_node_data;
	num_cell_data = model->num_cell_data;
	num_model_data = model->num_model_data;
	num_nlist_nodes = model->num_nlist_nodes;

	*node_id = NULL;
	*node_id = (int *)malloc(sizeof(int) * num_nodes);
	if (*node_id == NULL) {
		ERRerror("read ucd", 0, ERR_ORIG, "Error in read_ucd: can't alloc node ids.");
		return (0);
	}
	for (node = 0; node < num_nodes; node++)
		*(*node_id + node) = node+1;
	cells = NULL, cell_nlists = NULL;
	cells = (Ctype_B *)malloc(sizeof(Ctype_B) * num_cells);
	cell_nlists = (int *)malloc(sizeof(int) * num_nlist_nodes);
	if (cells == NULL || cell_nlists == NULL) {
		ERRerror("read ucd", 0, ERR_ORIG, "Error in read_ucd: can't alloc cells.");
		fclose (fp);
		return (0);
	}

	fread  (cells, sizeof(Ctype_B), num_cells, fp);
	fread  (cell_nlists, sizeof(int), num_nlist_nodes, fp);

	model->mid_edge_flag = 0;
	model->material_flag = 0;
	for (cell=0; cell < num_cells; cell++) {
		if (cells[cell].n > UCD_num_nodes[cells[cell].cell_type]) {
			model->mid_edge_flag = 1;
			break;
		}
	}
	for (cell=1; cell < num_cells; cell++) {
		if (cells[cell].mat != 0 ||  /* CFS PR 25643 */
		    cells[cell].mat != cells[0].mat ) {
			model->material_flag = 1;
			break;
		}
	}

	for (i = 0; i < num_nlist_nodes; i++)
		cell_nlists[i]--;

	xc =NULL;
	xc = (float *)malloc(sizeof(float) * num_nodes);
	if (xc == NULL) {
		ERRerror("read ucd", 0, ERR_ORIG, "Error in read_ucd: can't alloc coordinates.");
		fclose (fp);
		return (0);
	}
	*coord = (float *)ARRalloc(NULL, DTYPE_FLOAT, num_nodes*3, NULL);
	fread (xc, sizeof(float), num_nodes, fp);

	for (i=0; i<num_nodes; i++)
		(*coord)[3*i] = xc[i];

	fread (xc, sizeof(float), num_nodes, fp);
	for (i=0; i<num_nodes; i++)
		(*coord)[3*i+1] = xc[i];

	fread (xc, sizeof(float), num_nodes, fp);
	for (i=0; i<num_nodes; i++)
		(*coord)[3*i+2] = xc[i];
	free(xc);

	node_data = NULL, min_node_data = NULL, max_node_data = NULL;
	if (num_node_data) {
		model->node_data_labels = (char *) malloc(UCD_LABEL_LEN 
					  * sizeof(char));
		model->node_data_units = (char *) malloc(UCD_LABEL_LEN 
					 * sizeof(char));
		model->node_active_list = (int *) malloc( num_node_data 
					  * sizeof(int));
		model->node_comp_list = (int *) malloc( num_node_data 
					* sizeof(int));

		fread (model->node_data_labels, sizeof(char),
		       UCD_LABEL_LEN, fp);
		fread (model->node_data_units, sizeof(char),
		       UCD_LABEL_LEN, fp);
		fread (&model->num_node_comp, sizeof(int), 1, fp);
		fread (model->node_comp_list, sizeof(int), num_node_data, fp);

		node_data = (float *)malloc(sizeof(float) * num_nodes 
					    * num_node_data);
		min_node_data = (float *)malloc(sizeof(float) * num_node_data);
		max_node_data = (float *)malloc(sizeof(float) * num_node_data);

		if (node_data == NULL || min_node_data == NULL ||
		    max_node_data == NULL) {
			ERRerror("read ucd", 0, ERR_ORIG, "Error in read_ucd: can't alloc node data.");
			fclose (fp);
			return (0);
		}
		fread (min_node_data, sizeof(float), num_node_data, fp);
		fread (max_node_data, sizeof(float), num_node_data, fp);
		fread (node_data, sizeof(float),
		num_node_data * num_nodes, fp);
		fread (model->node_active_list, sizeof(int),
                       num_node_data, fp);
	}

	cell_data = NULL, min_cell_data = NULL, max_cell_data = NULL;
	if (num_cell_data) {
		model->cell_data_labels = (char *) malloc(UCD_LABEL_LEN 
							  * sizeof(char));
		model->cell_data_units = (char *) malloc(UCD_LABEL_LEN 
							 * sizeof(char));
		model->cell_active_list = (int *) malloc(num_cell_data 
							 * sizeof(int));
		model->cell_comp_list = (int *) malloc(num_cell_data 
						       * sizeof(int));

		fread (model->cell_data_labels, sizeof(char),
		       UCD_LABEL_LEN, fp);
		fread (model->cell_data_units, sizeof(char),
		       UCD_LABEL_LEN, fp);
		fread (&model->num_cell_comp, sizeof(int), 1, fp);
		fread (model->cell_comp_list, sizeof(int), num_cell_data, fp);

		cell_data = (float *)malloc(sizeof(float) * num_cells 
					    * num_cell_data);
		min_cell_data = (float *)malloc(sizeof(float) * num_cell_data);
		max_cell_data = (float *)malloc(sizeof(float) * num_cell_data);

		if (cell_data == NULL || min_cell_data == NULL ||
		    max_cell_data == NULL) {
			ERRerror("read ucd", 0, ERR_ORIG, "Error in read_ucd: can't alloc cell data.");
			fclose (fp);
			return (0);
		}
		fread (min_cell_data, sizeof(float), num_cell_data, fp);
		fread (max_cell_data, sizeof(float), num_cell_data, fp);
		fread (cell_data, sizeof(float),
                       num_cell_data * num_cells, fp);
		fread (model->cell_active_list, sizeof(int),
		       num_cell_data, fp);
	}

	model_data = NULL, min_model_data = NULL, max_model_data = NULL;
	if (num_model_data) {
		model->model_data_labels = (char *) malloc(UCD_LABEL_LEN 
							   * sizeof(char));
		model->model_data_units = (char *) malloc(UCD_LABEL_LEN 
							  * sizeof(char));
		model->model_active_list = (int *) malloc(num_model_data 
							  * sizeof(int));
		model->model_comp_list = (int *) malloc(num_model_data 
							* sizeof(int));

		fread (model->model_data_labels, sizeof(char),
		       UCD_LABEL_LEN, fp);
		fread (model->model_data_units, sizeof(char),
		       UCD_LABEL_LEN, fp);
		fread ((&model->num_model_comp), sizeof(int), 1, fp);
		fread (model->model_comp_list, sizeof(int),
		       num_model_data, fp);

		model_data = (float *)malloc(sizeof(float) * num_model_data);
		min_model_data =(float *)malloc(sizeof(float) * num_model_data);
		max_model_data =(float *)malloc(sizeof(float) * num_model_data);

		if (model_data == NULL || min_model_data == NULL ||
		    max_model_data == NULL) {
			ERRerror("read ucd", 0, ERR_ORIG,
				 "Error in read_ucd: can't alloc model data.");
			fclose (fp);
			return (0);
		}
		fread (min_model_data, sizeof(float), num_model_data, fp);
		fread (max_model_data, sizeof(float), num_model_data, fp);
		fread (model_data, sizeof(float), num_model_data, fp);
		fread (model->model_active_list, sizeof(int),
		       num_model_data, fp);
	}

	*pcells = cells;
	*pcell_nlists = cell_nlists;
	*pnode_data = node_data;
	*pcell_data = cell_data;
	*pmodel_data = model_data;

	model->num_nodes = num_nodes;
	model->num_cells = num_cells;
	model->num_node_data = num_node_data;
	model->num_cell_data = num_cell_data;
	model->num_model_data = num_model_data;

	*pmax_node_data = max_node_data;
	*pmin_node_data = min_node_data;
	*pmax_cell_data = max_cell_data;
	*pmin_cell_data = min_cell_data;
	*pmax_model_data = max_model_data;
	*pmin_model_data = min_model_data;

	fclose (fp);

	return(1);
}


/*-----------------------------------------------------*
 *                                                     *
 *               ****  get_token  ****                 *
 *                                                     *
 * returns the next value in tline[] separated by a    *
 * blank.                                              *
 *-----------------------------------------------------*/
static int get_token(char **tline, void *data, const char *control)
{
	char *line, token[80], c;

	int ntc;

	/**************
	 ***  body  ***
	 **************/

	c = 0;
	line = *tline;

	for (ntc = 0; c != '\n'; line++) {
		if (((c  = *line) != ' ') && (c != '\t') && (c != '\f') && (c != '\n'))
			token[ntc++] = c;
		else if (ntc != 0) {
			token[ntc] = '\0';
			sscanf (token, control, data);
			*tline = line;
			break;
		}
	}

	return (ntc);
}

static int  get_label (char *string, char delimiter, int number, 
                       char *label)
{
	int   i, j, k, len;
	char  current;

	/* check to make sure that structure is not NULL */
	if (string == NULL)
		return (0);

	/* search for the appropriate label */
	k = 0;
	len = (int)strlen (string);
	for (i = 0; i <= number; i++) {
		current = string[k++];
		j = 0;
		while (current != delimiter) {

			/* build the label character by character */
			label[j++] = current;
			current = string[k++];

			/* the last character was found */
			if (k > len) {
				/* the nth label was not found, where n = number */
				if (i < number) {
					return (0);
				}  /* end if */
				current = delimiter;
			}  /* end if */

		}  /* end while */
		label[j] = '\0';
	}  /* end for */

	/* a valid label was found */
	return (1);
}

/*-----------------------------------------------------*
 *                                                     *
 *           ****  ucd_qsorti  ****                     *
 *                                                     *
 *-----------------------------------------------------*/

/* 64-bit porting. Directly Modified */
static void ucd_qsorti (xp_long *values, xp_long *indexes, xp_long num_entries)
{
	xp_long *x;
	xp_long *ix, mid;

	register xp_long i, j, k;

	/**************
	 ***  body  ***
	 **************/

	mid = (num_entries - 1) / 2;
	if (mid == 0) mid = 1;

	x = (xp_long *)malloc(mid *sizeof(xp_long));
	ix = (xp_long *)malloc(mid * sizeof(xp_long));

	/*$dir no_recurrence */
	for (i = 0; i < mid; i++) {
		x[i] = values[i];
		ix[i] = indexes[i];
	}

	/*$dir fork */
	ucd_qsi (x, ix, 0, mid-1);
	ucd_qsi (values, indexes, mid, num_entries-1);
	/*$dir join */

	i = 0; j = mid;

	while ((i < mid) && (j < num_entries)) {
		if (values[j] < x[i]) {
			values[i + j - mid] = values[j];
			indexes[i + j - mid] = indexes[j];
			j++;
		}
		else {
			values[i + j - mid] = x[i];
			indexes[i + j - mid] = ix[i];
			i++;
		}
	}

	if (i < mid) {
		/*$dir no_recurrence */
		for (k = i; k < mid; k++) {
			values[k + j - mid] = x[k];
			indexes[k + j - mid] = ix[k];
		}
	}

	free(x);
	free(ix);
}


/*-----------------------------------------------------*
 *                                                     *
 *           ****  ucd_qsi  ****                       *
 *                                                     *
 *-----------------------------------------------------*/

/* 64-bit porting. Directly Modified */
static void ucd_qsi (xp_long *values, xp_long *indexes, xp_long left, xp_long right)
{
	register xp_long i, j;

	xp_long x, y;

	xp_long z;

	/**************
	 ***  body  ***
	 **************/

	x = values[(left + right) / 2];
	i = left;
	j = right;

	do {
		while (values[i] < x && i < right) i++;
		while (x < values[j] && j > left) j--;

		if (i <= j) {
			y = values[i];
			values[i] = values[j];
			values[j] = y;
			z = indexes[i];
			indexes[i] = indexes[j];
			indexes[j] = z;
			i++;
			j--;
		}
	} while (i <= j);

	if (left < j) ucd_qsi (values, indexes, left, j);
	if (i < right) ucd_qsi (values, indexes, i, right);
}


/*-----------------------------------------------------*
 *                                                     *
 *               ****  ucd_bsearchi  ****              *
 *                                                     *
 *-----------------------------------------------------*/

/* 64-bit porting. Directly Modified */
static xp_long ucd_bsearchi (xp_long v, xp_long *list, xp_long n)
{
	xp_long found, top, bot, indexer;

	/**************
	 ***  body  ***
	 **************/

	n = n - 1;

	if (v <= list[0])
		return (0);

	if (v >= list[n])
		return (n);

	found = 0;
	top = 0;
	bot = n;

	while (!found) {
		indexer = (top + bot) / 2;

		if (v > list[indexer])
			top = indexer;
		else if (v < list[indexer])
			bot = indexer;
		else
			return (indexer);

		if ((bot - top) < 2)
			if (fabs(list[bot] - v + 0.0) < fabs(v - list[top] + 0.0))
				return (bot);
			else
				return (top);
	}
	return (0);
}
