/*
			Copyright (c) 1995 by
			Advanced Visual Systems Inc.
			All Rights Reserved

	This software comprises unpublished confidential information of
	Advanced Visual Systems Inc. and may not be used, copied or made
	available to anyone, except in accordance with the license
	under which it is furnished.

	This file is under Perforce control
	$Id: //depot/express/fcs70/modules/shrink.c#1 $
*/

#define XP_WIDE_API	/* Use Wide APIs */

#include <avs/util.h>
#include <avs/err.h>
#include <avs/om.h>
#include <avs/fld.h>
#include <avs/arr.h>
#include <avs/dv_util.h>

#define ERR_RETURN(A) ERRerror("DVseparate_cells_update",1,ERR_ORIG, A); return(0);
#define MAX_NAME_LEN 256
#define ALLOC_BLOCK 1024

/* 64-bit porting. Only Modified Internally */
int DVseparate_cells_update(OMobj_id mod_id)
{
	OMobj_id in, out, in_set, out_set;
	int shrink;
	double scale;
	int nsets, set, cell_nnodes, nspace, nnode_data, null_flag;
	xp_long ncells, nnodes, *node_conn;
	int i, j, n, veclen, dtype, data_id;
	xp_long i_w, size, out_nnodes, *ind=NULL, alloc_size = ALLOC_BLOCK;
	int count, poly_flag, *poly_nnodes;
	xp_long npoly, *poly_conn;
	float *in_coord, *out_coord, center[3];
	double null_value;
	char *in_data, *out_data, name[MAX_NAME_LEN], units[MAX_NAME_LEN];


	in = OMfind_subobj(mod_id, OMstr_to_name("in"), OM_OBJ_RD);
	out = OMfind_subobj(mod_id, OMstr_to_name("out"), OM_OBJ_RW);

	if (OMget_name_int_val(mod_id, OMstr_to_name("shrink"), &shrink) != 1)
		shrink = 0;

	if (OMget_name_real_val(mod_id, OMstr_to_name("scale"), &scale) != 1)
		scale = 1;
	/* 
	 * clean the output
	 */
	if (FLDset_nnodes(out, 0) != 1) {
		ERR_RETURN("cannot get nnodes");
	}
	if (FLDset_ncell_sets (out, 0) != 1) {
		ERR_RETURN("Error setting ncell_sets");
	}
	FLDset_node_data_ncomp(out, 0);

	if (FLDget_nnodes(in, &nnodes) != 1 || nnodes == 0) {
		/* 
		 * this could be a case when module disconnected 
		 * clean the output and return
		 */
		return(0);
	}
	if (FLDget_nspace (in, &nspace) != 1) {
		ERR_RETURN("Error getting nspace");
	}
	if (FLDset_nspace (out, nspace) != 1) {
		ERR_RETURN("Error setting nspace");
	}

	if (FLDget_node_data_ncomp(in, &nnode_data) != 1)
		nnode_data = 0;

	if (FLDset_node_data_ncomp(out, nnode_data) != 1) {
		ERR_RETURN("Error setting ncomp");
	}

	if (FLDget_ncell_sets(in, &nsets) != 1) {
		ERR_RETURN("cannot get nsets");
	}
	if (FLDset_ncell_sets(out, nsets) != 1) {
		ERR_RETURN("cannot get nsets");
	}

	ind = (xp_long *)malloc(alloc_size*sizeof(xp_long));
	if (ind == NULL) {
		ERR_RETURN("cannot allocate index array");
	}


	for (out_nnodes=0, set=0; set<nsets; set++) {
		if (FLDget_cell_set(in, set, &in_set) != 1) {
			ERR_RETURN("Error getting cell set");
		}
		if (FLDget_cell_set(out, set, &out_set) != 1) {
			ERR_RETURN("Error getting cell set");
		}

		/*--------------------------------------------------*/
		/*  Make out cell data to be the same as in set      */
		/*--------------------------------------------------*/
		if (FLDcopy_subelem_ref(out_set, in_set) != 1) {
			ERR_RETURN("Error setting field reference");
		}
		if (FLDget_cell_set(out, set, &out_set) != 1) {
			ERR_RETURN("Error getting cell set");
		}

		if (FLDget_cell_set_name(in_set, name, MAX_NAME_LEN) != 1) {
			ERR_RETURN("Error getting cell set name");
		}
		if (FLDset_cell_set(out_set, name) != 1) {
			ERR_RETURN("Error setting cell set");
		}
		if (FLDget_cell_set(out, set, &out_set) != 1) {
			ERR_RETURN("Error getting cell set");
		}
		if (FLDget_poly_flag(out_set, &poly_flag) != 1) {
			poly_flag = 0;
		}

		if (poly_flag == 0) {
			if (FLDget_ncells(out_set, &ncells) != 1) {
				ERR_RETURN("cannot get ncells");
			}
			if (FLDget_cell_set_nnodes(out_set,  &cell_nnodes) != 1) {
				ERR_RETURN("cannot get cell nnodes");
			}
			if (out_nnodes+ncells*cell_nnodes >= alloc_size) {
				alloc_size += (ncells*cell_nnodes/ALLOC_BLOCK+1)*ALLOC_BLOCK;
				ind = (xp_long *)realloc(ind, alloc_size*sizeof(xp_long));
				if (ind == NULL) {
					ERR_RETURN("cannot allocate index array");
				}
			}
			if (FLDget_node_connect(in_set, &node_conn, &size, OM_GET_ARRAY_RD) != 1) {
				ERR_RETURN("cannot get cell connectivity");
			}
			size = ncells*cell_nnodes;
			for (i_w=0; i_w<size; i_w++)
				ind[out_nnodes+i_w] = node_conn[i_w];
			ARRfree(node_conn);

			if (FLDget_node_connect(out_set, &node_conn, &size, OM_GET_ARRAY_WR) != 1) {
				ERR_RETURN("cannot set cell connectivity");
			}
			size = ncells*cell_nnodes;
			for (i_w=0; i_w<size; i_w++)
				node_conn[i_w] = out_nnodes+i_w;

			out_nnodes += ncells*cell_nnodes;
			ARRfree(node_conn);
		}
		else {
			if (FLDget_npolys(out_set, &npoly) != 1) {
				ERR_RETURN("cannot get ncells");
			}
			if (FLDget_poly_nnodes(in_set, &poly_nnodes, &size, OM_GET_ARRAY_RD) != 1) {
				ERR_RETURN("cannot get poly_nnodes");
			}
			if (FLDget_poly_connect(in_set, &poly_conn, &size, OM_GET_ARRAY_RD) != 1) {
				ERR_RETURN("cannot get poly connectivity");
			}
			node_conn = (xp_long *)ARRalloc(NULL, DTYPE_LONG, size, NULL);

			for (count=0,i_w=0; i_w<npoly; i_w++) {
				if (poly_nnodes[i_w] == 2) {  /* polyline or polytri */
					cell_nnodes = (int)(poly_conn[2*i_w+1]-poly_conn[2*i_w]+1);
				}
				else {
					cell_nnodes = poly_nnodes[i_w];
				}
				if (out_nnodes+cell_nnodes >= alloc_size) {
					alloc_size += (cell_nnodes/ALLOC_BLOCK+1)*ALLOC_BLOCK;
					ind = (xp_long *)realloc(ind, alloc_size*sizeof(xp_long));
					if (ind == NULL) {
						ERR_RETURN("cannot allocate index array");
					}
				}
				if (poly_nnodes[i_w] == 2) {  /* polyline or polytri */
					for (j=0; j<cell_nnodes; j++)
						ind[out_nnodes+j] = poly_conn[2*i_w]+j;
					node_conn[2*i_w] = out_nnodes;
					node_conn[2*i_w+1] = out_nnodes+cell_nnodes-1;
				}
				else {
					for (j=0; j<cell_nnodes; j++) {
						ind[out_nnodes+j] = poly_conn[count];
						node_conn[count++] = out_nnodes+j;
					}
				}
				out_nnodes += cell_nnodes;
			}
			if (FLDset_poly_connect(out_set, node_conn, count, OM_SET_ARRAY_FREE) != 1) {
				ERR_RETURN("cannot set poly connectivity");
			}
			ARRfree(poly_conn);
			ARRfree(poly_nnodes);
		}
	}
	if (FLDset_nnodes(out, out_nnodes) != 1) {
		ERR_RETURN("Error setting nnodes");
	}

	if (FLDget_coord(in, &in_coord, &size, OM_GET_ARRAY_RD) != 1) {
		ERR_RETURN("Error setting coordinate array");
	}

	if (FLDget_coord(out, &out_coord, &size, OM_GET_ARRAY_WR) != 1) {
		ERR_RETURN("Error setting coordinate array");
	}

	UTILcopy_array_list(nnodes, nspace, DTYPE_FLOAT, (char *)in_coord, out_nnodes,
			    ind, (char *)out_coord);
	if (shrink) {
		for (set=0; set<nsets; set++) {
			if (FLDget_cell_set(out, set, &out_set) != 1) {
				ERR_RETURN("Error getting cell set");
			}
			if (FLDget_poly_flag(out_set, &poly_flag) != 1) {
				poly_flag = 0;
			}

			if (poly_flag == 0) {
				if (FLDget_ncells(out_set, &ncells) != 1) {
					ERR_RETURN("cannot get ncells");
				}
				if (FLDget_cell_set_nnodes(out_set,  &cell_nnodes) != 1) {
					ERR_RETURN("cannot get cell nnodes");
				}
				if (FLDget_node_connect(out_set, &node_conn, &size, OM_GET_ARRAY_RD) != 1) {
					ERR_RETURN("cannot get cell connectivity");
				}
				for (i_w=0; i_w<ncells; i_w++) {
					for (j=0; j<nspace; j++)
						center[j] = 0.0;
					for (n=0; n<cell_nnodes; n++)
						for (j=0; j<nspace; j++)
							center[j] += out_coord[node_conn[i_w*cell_nnodes+n]*nspace+j];
					for (j=0; j<nspace; j++)
						center[j] = center[j]/cell_nnodes;
					for (n=0; n<cell_nnodes; n++)
						for (j=0; j<nspace; j++)
							out_coord[node_conn[i_w*cell_nnodes+n]*nspace+j] = center[j]+
								(scale)*(out_coord[node_conn[i_w*cell_nnodes+n]*nspace+j]-center[j]);
				}
				ARRfree(node_conn);
			}
			else {
				if (FLDget_npolys(out_set, &npoly) != 1) {
					ERR_RETURN("cannot get ncells");
				}
				if (FLDget_poly_nnodes(out_set, &poly_nnodes, &size, OM_GET_ARRAY_RD) != 1) {
					ERR_RETURN("cannot get poly_nnodes");
				}
				if (FLDget_poly_connect(out_set, &poly_conn, &size, OM_GET_ARRAY_RD) != 1) {
					ERR_RETURN("cannot get poly connectivity");
				}
				for (count=0,i_w=0; i_w<npoly; i_w++) {
					for (j=0; j<nspace; j++)
						center[j] = 0.0;
					if (poly_nnodes[i_w] == 2) {  /* polyline or polytri */
						cell_nnodes = (int)(poly_conn[2*i_w+1]-poly_conn[2*i_w]+1);
						for (n=0; n<cell_nnodes; n++)
							for (j=0; j<nspace; j++)
								center[j] += out_coord[(poly_conn[i_w*2]+n)*nspace+j];
						for (j=0; j<nspace; j++)
							center[j] = center[j]/cell_nnodes;
						for (n=0; n<cell_nnodes; n++)
							for (j=0; j<nspace; j++)
								out_coord[(poly_conn[i_w*2]+n)*nspace+j] = center[j]+
									(scale)*(out_coord[(poly_conn[i_w*2]+n)*nspace+j]-center[j]);
					}
					else {
						cell_nnodes = poly_nnodes[i_w];
						for (n=0; n<cell_nnodes; n++)
							for (j=0; j<nspace; j++)
								center[j] += out_coord[poly_conn[count+n]*nspace+j];
						for (j=0; j<nspace; j++)
							center[j] = center[j]/cell_nnodes;
						for (n=0; n<cell_nnodes; n++)
							for (j=0; j<nspace; j++)
								out_coord[poly_conn[count+n]*nspace+j] = center[j]+
									(scale)*(out_coord[poly_conn[count+n]*nspace+j]-center[j]);
						count += cell_nnodes;
					}
				}
				ARRfree(poly_conn);
				ARRfree(poly_nnodes);
			}
		}
	}
	for (i=0; i<nnode_data; i++) {
		if (FLDget_node_data_veclen(in, i, &veclen) != 1) {
			ERR_RETURN("Error getting veclen");
		}
		if (FLDget_node_data_units(in, i, units, MAX_NAME_LEN) != 1) {
			strcpy(units, "");
		}
		if (FLDget_node_data_label(in, i, name, MAX_NAME_LEN) != 1) {
			strcpy(name, "");
		}
		if (FLDset_node_data_comp (out, i, veclen, name, units) != 1) {
			ERR_RETURN("Error setting node component");
		}
		if (FLDget_node_data(in, i, &dtype, &in_data,
				     &size, OM_GET_ARRAY_RD) != 1) {
			ERR_RETURN("cannot get node data");
		}
		if (FLDget_node_data(out, i, &dtype, &out_data,
				     &size, OM_GET_ARRAY_WR) != 1) {
			ERR_RETURN("cannot get node data");
		}
		UTILcopy_array_list(nnodes, veclen, dtype, in_data, out_nnodes,
				    ind, out_data);

		if (FLDcopy_node_minmax(in, out, i, i) != 1) {
			ERR_RETURN("Error copying node minmax");
		}
		if (FLDcopy_node_minmax_vec(in, out, i, i) != 1) {
			ERR_RETURN("Error copying node minmax");
		}
		if (FLDget_node_data_id(in, i, &data_id) == 1)
			FLDset_node_data_id(out, i, data_id);

		if (FLDget_node_null_data(in, i, &null_flag, (char *)&null_value) != 1) {
			ERR_RETURN("cannot get null data");
		}
		if (null_flag) {
			if (FLDset_node_null_data(out, i, (char *)&null_value, dtype) != 1) {
				ERR_RETURN("Error setting null value");
			}
		}
		else {
			if (FLDset_node_null_flag(out, i, 0) != 1) {
				ERR_RETURN("Error setting null flag");
			}
		}
		ARRfree(in_data);
		ARRfree(out_data);
	}
	ARRfree(in_coord);
	ARRfree(out_coord);
	free(ind);
	return(1);
}
