/*
			Copyright (c) 1993 by
			Advanced Visual Systems Inc.
			All Rights Reserved

	This software comprises unpublished confidential information of
	Advanced Visual Systems Inc. and may not be used, copied or made
	available to anyone, except in accordance with the license
	under which it is furnished.

	This file is under Perforce control
	$Id: //depot/express/fcs70/modules/tbl_scat.c#1 $
*/

#define XP_WIDE_API	/* Use Wide APIs */

#include <avs/util.h>
#include <avs/err.h>
#include <avs/dtype.h>
#include <avs/data_utils.h>
#include <avs/om.h>
#include <avs/fld.h>
#include <avs/arr.h>
#include <avs/gd_def.h>

#define ERR_RETURN(A) ERRerror("DVtable_to_scat", 0, ERR_ORIG, A); return(0);
#define MAX_NAME_SIZE 1024
#define MAX_NULL_LEN  64

#define  IJK_TO_INDEX(ndim,dims,ijk,ind) \
{ \
	 int iii; \
	 xp_long ddd; \
	 for (ind=ijk[0],ddd=dims[0],iii=1; iii<ndim; iii++) { \
		 ind += ddd*ijk[iii]; \
		 ddd *= dims[iii]; \
	 } \
}

typedef struct _DBdata {
	int type;
	xp_long size;
	xp_long u_size;
	int null_flag;
	char *null_value;
	char *values;
	char *u_values;
	char **strings;
	char **u_strings;
	char *name;
} DBdata;

static int get_table_data(OMobj_id obj, DBdata *data, int discrete);
static int free_table_data(DBdata *data);
static xp_long get_unique_index(DBdata *data, xp_long ind);

static const char *xyz_names[] = {"x_name", "y_name", "z_name"};


/* 64-bit porting. Only Modified Internally */
int DVtable_to_scat_update(OMobj_id elem_id, OMevent_mask event_mask, int seq_num)
{
	OMobj_id  xyz_sel_id, xyz_id, data_sel_id, data_id, out, val_id, tmp_id;
	int *xyz_axis, *xyz_set, *data,nspace, nnode_data;
	xp_long  nnodes, ndata, *node_list;
	int j, k, stat, discrete[3], dtype, comp;
	xp_long i, nvals, size;
	int valid;
	int cell_type;
	double val;
	char  *label, units[MAX_NAME_SIZE];
	float *out_xyz;
	char **out_data;
	int  *oper;
	DBdata xyz_data[3], *data_data;


	xyz_sel_id = OMfind_subobj(elem_id, OMstr_to_name("xyz_select"), OM_OBJ_RD);

	xyz_id = OMfind_subobj(elem_id, OMstr_to_name("field_xyz"), OM_OBJ_RD);

	data_sel_id = OMfind_subobj(elem_id, OMstr_to_name("data_select"), OM_OBJ_RD);

	data_id = OMfind_subobj(elem_id, OMstr_to_name("field_data"), OM_OBJ_RD);

	if (OMget_name_int_val(elem_id, OMstr_to_name("cell_type"), &cell_type) != 1)
		cell_type = 0;

	out = OMfind_subobj(elem_id, OMstr_to_name("out"), OM_OBJ_RW);

	xyz_axis = (int *)OMret_name_array_ptr(xyz_sel_id, OMstr_to_name("xyz_axis"),
					       OM_GET_ARRAY_RD, &size, NULL);
	if (xyz_axis == NULL)
		return(0);
	xyz_set = (int *)OMret_name_array_ptr(xyz_sel_id, OMstr_to_name("xyz_set"),
					       OM_GET_ARRAY_RD, &size, NULL);
	if (xyz_set == NULL)
		return(0);
	if (OMget_name_int_val(xyz_sel_id, OMstr_to_name("ndim"), &nspace) != 1) {
		return(0);
	}

	data = (int *)OMret_name_array_ptr(data_sel_id, OMstr_to_name("data"),
					       OM_GET_ARRAY_RD, &ndata, NULL);
	if (data == NULL)
		ndata = 0;

	if (nspace == 0) {
		return(0);
	}
	if (FLDset_ncell_sets (out, 0) != 1) {
		ERR_RETURN("Error setting ncell_sets");
	}

	if (FLDset_node_data_ncomp (out, 0) != 1) {
		ERR_RETURN("Error setting nnode_data");
	}

	if (FLDset_nspace(out, nspace) != 1) {
		ERR_RETURN("cannot set nspace");
	}
	for (k=0,j=0,i=0; i<3; i++) {
		if (xyz_set[i] == 0 || xyz_axis[i] < 0)
			continue;
		if (OMget_array_val(xyz_id, xyz_axis[i], &val_id, OM_OBJ_RD) != 1) {
			ERR_RETURN("cannot get field_xyz");
		}
		if (OMget_name_long_val(val_id, OMstr_to_name("nvals"), &nvals) != 1) {
			ERR_RETURN("cannot get nvals");
		}
		if (nvals == 0) {
			if (!(event_mask & OM_EVENT_INST)) {
				if (OMget_name_str_val(val_id, OMstr_to_name("name"), &label, 0) != 1) {
					ERR_RETURN("cannot get column name");
				}
				ERRerror("DVtable_to_scat", 1, ERR_ORIG, 
					 " column %s doesn't have valid data for coordinates. Ignored", label);
			}
			continue;
		}

		if (k==0) {
			nnodes = nvals;
			if (FLDset_nnodes(out, nnodes) != 1) {
				ERR_RETURN("cannot get nnodes");
			}
			if (FLDget_coord(out, &out_xyz, &size, OM_GET_ARRAY_RW) != 1) {
				ERR_RETURN("Error setting coordinate array");
			}
			k=1;
		}
		if (OMget_name_int_val(val_id, OMstr_to_name("discrete"), &discrete[j]) != 1) {
			ERR_RETURN("cannot get discrete value");
		}

		tmp_id = OMfind_subobj(val_id, OMstr_to_name("values"), OM_OBJ_RD);
		if (OMget_data_type(tmp_id, &dtype) != 1) {
			ERR_RETURN("cannot get values type");
		}

		if (dtype == DTYPE_STRING && !discrete[j]) {
			if (OMget_name_str_val(val_id, OMstr_to_name("name"), &label, 0) != 1) {
				ERR_RETURN("cannot get name");
			}
			ERRerror("DVtable_to_scat", 1, ERR_ORIG, 
				 " column %s has must be discrete type. Set to discrete", label);
			discrete[j] = 1;
		}
		stat = get_table_data(val_id, xyz_data+j, discrete[j]);

		if (stat != 1) {
			ERR_RETURN("cannot get values");
		}
		/* output labels */
		if (OMset_name_str_val(elem_id, OMstr_to_name(xyz_names[j]),xyz_data[j].name) != 1) {
			ERR_RETURN("cannot set xyz name");
		}
		j++;
		if (j == nspace)
			break;
	}

	if (j == 0) {
		if (!(event_mask & OM_EVENT_INST)) {
			ERR_RETURN("No valid data for field axis, field is not created");
		}
		else {
			return(0);
		}
	}

	if (cell_type == 0) {
		if (FLDadd_cell_set(out, "Point") != 1) {
			ERR_RETURN("Error setting cell type");
		}
	}
	else {
		if (FLDadd_cell_set(out, "Polyline") != 1) {
			ERR_RETURN("Error setting cell type");
		}
	}
	if (FLDget_cell_set(out, 0, &tmp_id) != 1) {
		ERR_RETURN("Error getting cell set");
	}
	if (FLDset_ncells(tmp_id, nnodes) != 1) {
		ERR_RETURN("Error setting ncells");
	}
	if (cell_type == 0) {
		if (FLDget_node_connect(tmp_id, &node_list, 
					&size, OM_GET_ARRAY_RW) != 1) {
			ERR_RETURN("cannot get cell connectivity");
		}
		for (i=0; i<nnodes; i++)
			node_list[i] = i;
		}
	else {
		FLDset_npolys(tmp_id, 1);
		if (FLDget_poly_connect(tmp_id, &node_list,
					&size, OM_GET_ARRAY_RW) != 1) {
			ERR_RETURN("cannot get cell connectivity");
		}
		node_list[0] = 0;
		node_list[1] = nnodes-1;
	}
	ARRfree(node_list);

	data_data = (DBdata *)malloc(ndata*sizeof(DBdata));
	oper = malloc(ndata*sizeof(int));
	out_data = malloc(ndata*sizeof(char *));

	for (j=0, i=0; i< ndata; i++) {
		if (OMget_array_val(data_id, data[i], &val_id, OM_OBJ_RD) != 1) {
			ERR_RETURN("cannot get field_xyz");
		}
		if (OMget_name_int_val(val_id, OMstr_to_name("valid"), &valid) != 1) {
			ERR_RETURN("cannot get valid");
		}
		if (!valid) {
			if (OMget_name_str_val(val_id, OMstr_to_name("name"), &label, 0) != 1) {
				ERR_RETURN("cannot get name");
			}
			ERRerror("DVtable_to_scat", 1, ERR_ORIG, " column %s has invalid field data type. Ignored", label);
			continue;
		}

		tmp_id = OMfind_subobj(val_id, OMstr_to_name("values"), OM_OBJ_RD);
		if (OMget_data_type(tmp_id, &dtype) != 1) {
			ERR_RETURN("cannot get values type");
		}
		if (dtype == DTYPE_STRING) {
			if (OMget_name_str_val(val_id, OMstr_to_name("name"), &label, 0) != 1) {
				ERR_RETURN("cannot get name");
			}
			ERRerror("DVtable_to_scat", 1, ERR_ORIG, " column %s has invalid field data type. Ignored", label);
			continue;
		}

		if (OMget_name_int_val(val_id, OMstr_to_name("operation"), &oper[j]) != 1) {
			ERR_RETURN("cannot get operation value");
		}
		stat = get_table_data(val_id, data_data+j, 0);

		if (stat != 1) {
			ERR_RETURN("cannot get values");
		}
		j++;
	}
	nnode_data = j;

	if (FLDset_node_data_ncomp (out, nnode_data) != 1) {
		ERR_RETURN("Error setting nnode_data");
	}

	for (comp=0; comp<nnode_data; comp++) {
		strcpy(units,"");
		if (FLDset_node_data_comp (out, comp, 1, data_data[comp].name, units) != 1) {
			ERR_RETURN("Error setting node component");
		}
		dtype = data_data[comp].type;
		FLDset_node_data_type(out, comp, dtype);
		if (FLDget_node_data(out, comp, &dtype, &(out_data[comp]), 
				     &size, OM_GET_ARRAY_WR) != 1) {
			ERR_RETURN("cannot get node data");
		}
		if (oper[comp]) {
			if (FLDset_node_data_id(out, comp, GD_RADIUS_DATA_ID) != 1) {
				ERR_RETURN("cannot set node data id");
			}
		}
		if (data_data[comp].null_flag) {
			UTILdouble_to_type(&val, *(double *)data_data[comp].null_value, dtype);
			if (FLDset_node_null_data(out, comp, (char *)&val, dtype) != 1) {
				ERR_RETURN("Error setting null value");
			}
		}
		else {
			if (FLDset_node_null_flag(out, comp, 0) != 1) {
				ERR_RETURN("Error setting null flag");
			}
		}
	}
	for (i=0; i<nvals; i++) {
		for (j=0; j<nspace; j++) {
			dtype = xyz_data[j].type;
			if (discrete[j]) {
				out_xyz[i*nspace+j] = get_unique_index(xyz_data+j, i);
			}
			else {
				UTILtype_to_double(&val, xyz_data[j].values+i*DTYPEtype_size[dtype],
						   dtype);
				out_xyz[i*nspace+j] = val;
			}
		}
		for (comp=0; comp<nnode_data; comp++) {
			dtype = data_data[comp].type;
			size = DTYPEtype_size[dtype];
			memcpy(out_data[comp]+i*size, data_data[comp].values+i*size, size);
		}
	}
	if (data) ARRfree(data);
	ARRfree(xyz_axis);
	ARRfree(xyz_set);
	ARRfree(out_xyz);
	for (comp=0; comp<nspace; comp++) {
		free_table_data(xyz_data+comp);
	}
	for (comp=0; comp<nnode_data; comp++) {
		free_table_data(data_data+comp);
		if (out_data[comp])
			ARRfree(out_data[comp]);
	}
	free(out_data);
	free(oper);
	free(data_data);
	return(1);
}

/* 64-bit porting. Only Modified Internally */
static int get_table_data(OMobj_id obj, DBdata *data, int discrete)
{
	OMobj_id id;
	xp_long i, j, size;
	int found;

	id = OMfind_subobj(obj, OMstr_to_name("values"), OM_OBJ_RD);
	if (OMget_data_type(id, &(data->type)) != 1) {
		ERR_RETURN("cannot get values type");
	}
	if (OMget_name_int_val(obj, OMstr_to_name("null_flag"), &(data->null_flag)) != 1) {
		data->null_flag = 0;
	}
	if (OMget_name_str_val(obj, OMstr_to_name("name"), &(data->name), 0) != 1) {
		ERR_RETURN("cannot get name");
	}
	data->u_size = 0;
	data->values = NULL;
	data->u_values = NULL;
	if (discrete) {
		data->size = 0;
		switch (data->type) {
		      case DTYPE_BYTE:
		      case DTYPE_CHAR:
		{
			char *values;
			char *dvalues;

			if (data->null_flag) {
				data->null_value = malloc(sizeof(double));
				if (OMget_name_real_val(obj, OMstr_to_name("null_value"), 
							(double *)(data->null_value)) != 1) {
					ERR_RETURN("cannot get null_value");
				}
			}
			data->values = OMret_name_array_ptr(obj, OMstr_to_name("values"),
							      OM_GET_ARRAY_RD, &size, NULL);
			if (size)
				data->u_values = ARRalloc(NULL, DTYPE_BYTE, size, NULL);

			data->size = size;
			values = (char *)data->values;
			dvalues = (char *)data->u_values;

			for (i=0; i<size; i++) {
				for (found=0,j=0; j<data->u_size; j++)
					if (values[i] == dvalues[j]) {
						found = 1;
						break;
					}
				if (!found) {
					dvalues[data->u_size] = values[i];
					(data->u_size)++;
				}
			}
		}
			break;

		      case DTYPE_SHORT:
		{
			short *values;
			short *dvalues;

			if (data->null_flag) {
				data->null_value = malloc(sizeof(double));
				if (OMget_name_real_val(obj, OMstr_to_name("null_value"), (double *)(data->null_value)) != 1) {
					ERR_RETURN("cannot get null_value");
				}
			}
			data->values = OMret_name_array_ptr(obj, OMstr_to_name("values"),
							      OM_GET_ARRAY_RD, &size, NULL);
			if (size)
				data->u_values = ARRalloc(NULL, DTYPE_SHORT, size, NULL);

			data->size = size;
			values = (short *)data->values;
			dvalues = (short *)data->u_values;

			for (i=0; i<size; i++) {
				for (found=0,j=0; j<data->u_size; j++)
					if (values[i] == dvalues[j]) {
						found = 1;
						break;
					}
				if (!found) {
					dvalues[data->u_size] = values[i];
					(data->u_size)++;
				}
			}

		}
			break;

		      case DTYPE_INT:
		{
			int *values;
			int *dvalues;

			if (data->null_flag) {
				data->null_value = malloc(sizeof(double));
				if (OMget_name_real_val(obj, OMstr_to_name("null_value"), (double *)(data->null_value)) != 1) {
					ERR_RETURN("cannot get null_value");
				}
			}
			data->values = OMret_name_array_ptr(obj, OMstr_to_name("values"),
							      OM_GET_ARRAY_RD, &size, NULL);
			if (size)
				data->u_values = ARRalloc(NULL, DTYPE_INT, size, NULL);

			data->size = size;
			values = (int *)data->values;
			dvalues = (int *)data->u_values;

			for (i=0; i<size; i++) {
				for (found=0,j=0; j<data->u_size; j++)
					if (values[i] == dvalues[j]) {
						found = 1;
						break;
					}
				if (!found) {
					dvalues[data->u_size] = values[i];
					(data->u_size)++;
				}
			}
		}
			break;

		      case DTYPE_LONG:
		{
			xp_long *values;
			xp_long *dvalues;

			if (data->null_flag) {
				data->null_value = malloc(sizeof(double));
				if (OMget_name_real_val(obj, OMstr_to_name("null_value"), (double *)(data->null_value)) != 1) {
					ERR_RETURN("cannot get null_value");
				}
			}
			data->values = OMret_name_array_ptr(obj, OMstr_to_name("values"),
							      OM_GET_ARRAY_RD, &size, NULL);
			if (size)
				data->u_values = ARRalloc(NULL, DTYPE_LONG, size, NULL);

			data->size = size;
			values = (xp_long *)data->values;
			dvalues = (xp_long *)data->u_values;

			for (i=0; i<size; i++) {
				for (found=0,j=0; j<data->u_size; j++)
					if (values[i] == dvalues[j]) {
						found = 1;
						break;
					}
				if (!found) {
					dvalues[data->u_size] = values[i];
					(data->u_size)++;
				}
			}
		}
			break;

		      case DTYPE_FLOAT:
		{
			float *values;
			float *dvalues;

			if (data->null_flag) {
				data->null_value = malloc(sizeof(double));
				if (OMget_name_real_val(obj, OMstr_to_name("null_value"), (double *)(data->null_value)) != 1) {
					ERR_RETURN("cannot get null_value");
				}
			}

			data->values = OMret_name_array_ptr(obj, OMstr_to_name("values"),
							      OM_GET_ARRAY_RD, &size, NULL);
			if (size)
				data->u_values = ARRalloc(NULL, DTYPE_FLOAT, size, NULL);

			data->size = size;
			values = (float *)data->values;
			dvalues = (float *)data->u_values;

			for (i=0; i<size; i++) {
				for (found=0,j=0; j<data->u_size; j++)
					if (values[i] == dvalues[j]) {
						found = 1;
						break;
					}
				if (!found) {
					dvalues[data->u_size] = values[i];
					(data->u_size)++;
				}
			}
		}
			break;

		      case DTYPE_DOUBLE:
		{
			double *values;
			double *dvalues;

			if (data->null_flag) {
				data->null_value = malloc(sizeof(double));
				if (OMget_name_real_val(obj, OMstr_to_name("null_value"), (double *)(data->null_value)) != 1) {
					ERR_RETURN("cannot get null_value");
				}
			}
			data->values = OMret_name_array_ptr(obj, OMstr_to_name("values"),
							      OM_GET_ARRAY_RD, &size, NULL);
			if (size)
				data->u_values = ARRalloc(NULL, DTYPE_DOUBLE, size, NULL);


			data->size = size;
			values = (double *)data->values;
			dvalues = (double *)data->u_values;

			for (i=0; i<size; i++) {
				for (found=0,j=0; j<data->u_size; j++)
					if (values[i] == dvalues[j]) {
						found = 1;
						break;
					}
				if (!found) {
					dvalues[data->u_size] = values[i];
					(data->u_size)++;
				}
			}
		}
			break;

		      case DTYPE_STRING:
			id = OMfind_subobj(obj, OMstr_to_name("values"), OM_OBJ_RD);
			if (OMget_array_size(id, &size) != 1 || size == 0) {
				data->size = 0;
				return(0);
			}
			data->size = size;
			data->strings = (char **)malloc(data->size*sizeof(char *));
			data->u_strings = (char **)malloc(data->size*sizeof(char *));

			for (i=0; i<size; i++) {
				if (OMget_str_array_val(id, i, &(data->strings[i]), 0) != 1) {
					ERR_RETURN("cannot get string data");
				}
				for (found = 0, j=0; j<data->u_size; j++)
					if (strcmp(data->strings[i], data->u_strings[j]) == 0) {
						found = 1;
						break;
					}
				if (!found) {
					data->u_strings[data->u_size] = strdup(data->strings[i]);
					(data->u_size)++;
				}
			}
			if (data->null_flag) {
				if (OMget_name_str_val(obj, OMstr_to_name("null_value"), 
						       &(data->null_value), 0) != 1)  {
					ERR_RETURN("cannot get null_value");
				}
			}
			break;

		      default:
			data->type = DTYPE_UNSET;
			break;
		}
	}
	else {
		switch (data->type) {
		      case DTYPE_BYTE:
		      case DTYPE_CHAR:
		      case DTYPE_SHORT:
		      case DTYPE_INT:
		      case DTYPE_LONG:
		      case DTYPE_FLOAT:
		      case DTYPE_DOUBLE:
			data->size = 0;
			data->values = OMret_name_array_ptr(obj, OMstr_to_name("values"),
							    OM_GET_ARRAY_RD, &size, NULL);
			data->size = size;
			if (data->null_flag) {
				data->null_value = malloc(sizeof(double));
				if (OMget_name_real_val(obj, OMstr_to_name("null_value"), (double *)(data->null_value)) != 1) {
					ERR_RETURN("cannot get null_value");
				}
			}
			break;

		      case DTYPE_STRING:
			id = OMfind_subobj(obj, OMstr_to_name("values"), OM_OBJ_RD);
			if (OMget_array_size(id, &size) != 1 || data->size == 0)
				return(0);
			data->size = size;
			data->strings = (char **)malloc(data->size*sizeof(char *));
			for (i=0; i<(xp_long)data->size; i++)
				if (OMget_str_array_val(id, i, &(data->strings[i]), 0) != 1) {
					ERR_RETURN("cannot get string data");
				}
			if (data->null_flag) {
				if (OMget_name_str_val(obj, OMstr_to_name("null_value"), 
						       &(data->null_value), 0) != 1)  {
					ERR_RETURN("cannot get null_value");
				}
			}
			break;

		      default:
			data->type = DTYPE_UNSET;
			break;
		}
	}
	if (data->size == 0)
		return(0);
	else
		return(1);
}

/* 64-bit porting. Only Modified Internally */
static int free_table_data(DBdata *data)
{
	xp_long i;

	if (data->null_flag)
		free(data->null_value);
	if (data->name)
		free(data->name);
	switch (data->type) {
	      case DTYPE_BYTE:
	      case DTYPE_CHAR:
	      case DTYPE_SHORT:
	      case DTYPE_INT:
	      case DTYPE_LONG:
	      case DTYPE_FLOAT:
	      case DTYPE_DOUBLE:
		if (data->values && data->size)
			ARRfree(data->values);
		if (data->u_values && data->u_size)
			ARRfree(data->u_values);
		break;

	      case DTYPE_STRING:
		for (i=0; i<data->size; i++)
			free(data->strings[i]);
		free(data->strings);
		for (i=0; i<data->u_size; i++)
			free(data->u_strings[i]);
		free(data->u_strings);
		break;

	      default:
		break;
	}
	return(1);
}

/* 64-bit porting. Directly Modified */
static xp_long get_unique_index(DBdata *data, xp_long ind)
{
	xp_long i;

	if (data->size == 0)
		return(0);

	switch (data->type) {
	      case DTYPE_BYTE:
	      case DTYPE_CHAR:
	{
		char *values;
		char *dvalues;

		values = (char *)data->values;
		dvalues = (char *)data->u_values;

		for (i=0; i<data->u_size; i++)
			if (values[ind] == dvalues[i])
				return(i);
	}
		break;

	      case DTYPE_SHORT:
	{
		short *values;
		short *dvalues;

		values = (short *)data->values;
		dvalues = (short *)data->u_values;

		for (i=0; i<data->u_size; i++)
			if (values[ind] == dvalues[i])
				return(i);
	}
		break;

	      case DTYPE_INT:
	{
		int *values;
		int *dvalues;

		values = (int *)data->values;
		dvalues = (int *)data->u_values;

		for (i=0; i<data->u_size; i++)
			if (values[ind] == dvalues[i])
				return(i);
	}
		break;

	      case DTYPE_LONG:
	{
		xp_long *values;
		xp_long *dvalues;

		values = (xp_long *)data->values;
		dvalues = (xp_long *)data->u_values;

		for (i=0; i<data->u_size; i++)
			if (values[ind] == dvalues[i])
				return(i);
	}
		break;

	      case DTYPE_FLOAT:
	{
		float *values;
		float *dvalues;

		values = (float *)data->values;
		dvalues = (float *)data->u_values;

		for (i=0; i<data->u_size; i++)
			if (values[ind] == dvalues[i])
				return(i);
	}
		break;

	      case DTYPE_DOUBLE:
	{
		double *values;
		double *dvalues;

		values = (double *)data->values;
		dvalues = (double *)data->u_values;

		for (i=0; i<data->u_size; i++)
			if (values[ind] == dvalues[i])
				return(i);
	}
		break;

	      case DTYPE_STRING:
		for (i=0; i<data->u_size; i++)
			if (strcmp(data->strings[ind], data->u_strings[i]) == 0)
				return(i);
		break;

	      default:
		return (-1);
	}
	return(-1);
}

