/*
			Copyright (c) 1993 by
			Advanced Visual Systems Inc.
			All Rights Reserved

	This software comprises unpublished confidential information of
	Advanced Visual Systems Inc. and may not be used, copied or made
	available to anyone, except in accordance with the license
	under which it is furnished.

	This file is under Perforce control
	$Id: //depot/express/fcs70/modules/tbl_unif.c#1 $
*/

#define XP_WIDE_API	/* Use Wide APIs */

#include <avs/util.h>
#include <avs/err.h>
#include <avs/om.h>
#include <avs/dtype.h>
#include <avs/fld.h>
#include <avs/arr.h>
#include <avs/math.h>
#include <avs/data_utils.h>

#define ERR_RETURN(A) ERRerror("DVtable_to_unif", 0, ERR_ORIG, A); return(0);
#define MAX_NAME_SIZE 1024
#define MAX_NULL_LEN  64

#define  IJK_TO_INDEX(ndim,dims,ijk,ind) \
{ \
	 int iii; \
	 xp_long ddd; \
	 for (ind=ijk[0],ddd=dims[0],iii=1; iii<ndim; iii++) { \
		 ind += ddd*ijk[iii]; \
		 ddd *= dims[iii]; \
	 } \
}

typedef struct _DBdata {
	int type;
	xp_long size;
	xp_long u_size;
	int null_flag;
	char *null_value;
	char *values;
	char *u_values;
	char **strings;
	char **u_strings;
	char *name;
} DBdata;

static int get_table_data(OMobj_id obj, DBdata *data, int discrete);
static int free_table_data(DBdata *data);
static xp_long get_unique_index(DBdata *data, xp_long ind);
static int get_unique_label(DBdata *data, xp_long ind, char *label);
static int get_label(double value, int type, char *label);

static char *xyz_labels[] = {"x_labels", "y_labels", "z_labels"};
static char *xyz_names[] = {"x_name", "y_name", "z_name"};


/* 64-bit porting. Only Modified Internally */
int DVtable_to_unif_update(OMobj_id elem_id, OMevent_mask event_mask, int seq_num)
{
	OMobj_id  xyz_sel_id, xyz_id, data_sel_id, data_id, out, val_id, tmp_id;
	int *xyz_axis, *xyz_set, *data, ndim, nspace, nnode_data;
	xp_long dims[3], ijk[3], nnodes, ndata;
	int j, stat, discrete[3], ind_n, dtype, comp;
	xp_long i, k, nvals, ind, size, count;
	int valid, norm, label_step, set_labels, count_records=0;
	double min[3], max[3], bin_size[3], null_value, *out_null_value, point, val;
	char  *label, units[MAX_NAME_SIZE];
	float *out_xyz;
	char **out_data;
	int  *oper, **sum;
	DBdata xyz_data[3], *data_data;


	xyz_sel_id = OMfind_subobj(elem_id, OMstr_to_name("xyz_select"), OM_OBJ_RD);

	xyz_id = OMfind_subobj(elem_id, OMstr_to_name("field_xyz"), OM_OBJ_RD);

	data_sel_id = OMfind_subobj(elem_id, OMstr_to_name("data_select"), OM_OBJ_RD);

	data_id = OMfind_subobj(elem_id, OMstr_to_name("field_data"), OM_OBJ_RD);

	out = OMfind_subobj(elem_id, OMstr_to_name("out"), OM_OBJ_RW);

	xyz_axis = (int *)OMret_name_array_ptr(xyz_sel_id, OMstr_to_name("xyz_axis"),
					       OM_GET_ARRAY_RD, &size, NULL);
	if (xyz_axis == NULL)
		return(0);

	xyz_set = (int *)OMret_name_array_ptr(xyz_sel_id, OMstr_to_name("xyz_set"),
					       OM_GET_ARRAY_RD, &size, NULL);
	if (xyz_set == NULL)
		return(0);

	if (OMget_name_int_val(xyz_sel_id, OMstr_to_name("ndim"), &ndim) != 1) {
		return(0);
	}

	data = (int *)OMret_name_array_ptr(data_sel_id, OMstr_to_name("data"),
					       OM_GET_ARRAY_RD, &ndata, NULL);
	if (data == NULL)
		ndata = 0;

	if (ndim == 0) {
		return(0);
	}
	nspace = ndim;

	if (FLDset_ndim(out, ndim) != 1) {
		ERR_RETURN("cannot set ndim");
	}
	if (FLDset_nspace(out, nspace) != 1) {
		ERR_RETURN("cannot set nspace");
	}
	if (FLDget_points (out, &out_xyz, &size, OM_GET_ARRAY_RW) != 1) {
		ERR_RETURN("Error getting points");
	}

	if (FLDset_node_data_ncomp (out, 0) != 1) {
		ERR_RETURN("Error setting nnode_data");
	}

	for (i=0; i<3; i++) {
		tmp_id = OMfind_subobj(elem_id, OMstr_to_name(xyz_labels[i]), OM_OBJ_RW);
		OMset_array_size(tmp_id, 0);
		OMset_obj_val(tmp_id, OMnull_obj, 0);
	}
	for (j=0,i=0; i<3; i++) {
		if (xyz_set[i] == 0 || xyz_axis[i] < 0)
			continue;
		if (OMget_array_val(xyz_id, xyz_axis[i], &val_id, OM_OBJ_RD) != 1) {
			ERR_RETURN("cannot get field_xyz");
		}
		if (OMget_name_int_val(val_id, OMstr_to_name("discrete"), &discrete[j]) != 1) {
			ERR_RETURN("cannot get discrete value");
		}

		if (OMget_name_long_val(val_id, OMstr_to_name("nvals"), &nvals) != 1) {
			ERR_RETURN("cannot get nvals");
		}
		if (nvals == 0) {
			if (!(event_mask & OM_EVENT_INST)) {
				if (OMget_name_str_val(val_id, OMstr_to_name("name"), &label, 0) != 1) {
					ERR_RETURN("cannot get name");
				}
				ERRerror("DVtable_to_unif", 1, ERR_ORIG,
					 " column %s doesn't have valid data for coordinates. Ignored", label);
			}
			continue;
		}
		tmp_id = OMfind_subobj(val_id, OMstr_to_name("values"), OM_OBJ_RD);
		if (OMget_data_type(tmp_id, &dtype) != 1) {
			ERR_RETURN("cannot get values type");
		}

		if (OMget_name_int_val(val_id, OMstr_to_name("normalize"), &norm) != 1) {
			ERR_RETURN("cannot get normalize");
		}

		if (OMget_name_int_val(val_id, OMstr_to_name("set_labels"), &set_labels) != 1) {
			ERR_RETURN("cannot get set_labels value");
		}

		if (dtype == DTYPE_STRING && !discrete[j]) {
			if (OMget_name_str_val(val_id, OMstr_to_name("name"), &label, 0) != 1) {
				ERR_RETURN("cannot get name");
			}
			ERRerror("DVtable_to_unif", 1, ERR_ORIG,
				 " column %s must be discrete type. Set to discrete", label);
			discrete[j] = 1;
		}
		stat = get_table_data(val_id, xyz_data+j, discrete[j]);

		if (stat != 1) {
			if (OMget_name_str_val(val_id, OMstr_to_name("name"), &label, 0) != 1) {
				ERR_RETURN("cannot get name");
			}
			ERRerror("DVtable_to_unif", 1, ERR_ORIG,
				 " column %s doesn't have valid data for coordinates. Ignored", label);
			continue;
		}
		if (discrete[j]) {
			dims[j] = xyz_data[j].u_size;
			if (norm && dims[j] > 1) {
				out_xyz[j] = 0;
				out_xyz[nspace+j] = 1;
			}
			else {
				out_xyz[j] = 0;
				out_xyz[nspace+j] = xyz_data[j].u_size-1;
			}
			/* output labels */
			if (OMset_name_str_val(elem_id, OMstr_to_name(xyz_names[j]),xyz_data[j].name) != 1) {
				ERR_RETURN("cannot set xyz name");
			}
			if (set_labels) {
				if (OMget_name_int_val(val_id, OMstr_to_name("label_step"), &label_step) != 1) {
					ERR_RETURN("cannot get label_step");
				}
				if (OMget_name_int_val(val_id, OMstr_to_name("index_labels"), &ind_n) != 1) {
					ERR_RETURN("cannot get index_labels value");
				}
				tmp_id = OMfind_subobj(elem_id, OMstr_to_name(xyz_labels[j]), OM_OBJ_RW);
				if (OMset_array_size(tmp_id, (dims[j]-1)/label_step+1) != 1) {
					ERR_RETURN("cannot set labels size");
				}
				for (count=0,k=0; k<dims[j]; k+=label_step) {
					if (ind_n)
						sprintf(units, "%ld", k+1);
					else {
						stat = get_unique_label(xyz_data+j, k, units);
						if (stat != 1)
							strcpy(units," ");
					}
					if (OMset_str_array_val(tmp_id, count++, units) != 1) {
						ERR_RETURN("cannot set labels");
					}
				}
			}
		}
		else {
			OMget_name_real_val(val_id, OMstr_to_name("min"), &min[j]);
			OMget_name_real_val(val_id, OMstr_to_name("max"), &max[j]);
			OMget_name_real_val(val_id, OMstr_to_name("bin_size"), &bin_size[j]);
			if (bin_size[j] != 0.0)
				dims[j] = ceil((max[j]-min[j])/bin_size[j]);
			else
				dims[j] = 1;
			if (norm) {
				out_xyz[j] = 0;
				out_xyz[nspace+j] = 1;
			}
			else {
				out_xyz[j] = min[j]+bin_size[j]/2;
				out_xyz[nspace+j] = out_xyz[j]+bin_size[j]*(dims[j]-1);
			}
			/* output labels */
			if (OMset_name_str_val(elem_id, OMstr_to_name(xyz_names[j]),xyz_data[j].name) != 1) {
				ERR_RETURN("cannot set xyz name");
			}
			if (set_labels)  {
				if (OMget_name_int_val(val_id, OMstr_to_name("index_labels"), &ind_n) != 1) {
					ERR_RETURN("cannot get index_labels value");
				}
				tmp_id = OMfind_subobj(elem_id, OMstr_to_name(xyz_labels[j]), OM_OBJ_RW);
				if (OMget_name_int_val(val_id, OMstr_to_name("label_step"), &label_step) != 1) {
					ERR_RETURN("cannot get normalize");
				}
				if (OMset_array_size(tmp_id, (dims[j]-1)/label_step+1) != 1) {
					ERR_RETURN("cannot set labels size");
				}
				for (count=0,k=0; k<dims[j]; k+=label_step) {
					if (ind_n)
						sprintf(units, "%ld", k+1);
					else {
						val = min[j]+bin_size[j]*(k+0.5);
						stat = get_label(val, dtype, units);
						if (stat != 1)
							strcpy(units," ");
					}
					if (OMset_str_array_val(tmp_id, count++, units) != 1) {
						ERR_RETURN("cannot set label");
					}
				}
			}
		}

		j++;
		if (j == ndim)
			break;
	}
	if (j == 0) {
		if (!(event_mask & OM_EVENT_INST)) {
			ERR_RETURN("No valid data for field axis, field is not created");
		}
		else {
			return(0);
		}
	}

	if (FLDset_dims(out, dims) != 1) {
		ERR_RETURN("cannot get nnodes");
	}
	if (FLDget_nnodes(out, &nnodes) != 1) {
		ERR_RETURN("cannot get nnodes");
	}

	data_data = (DBdata *)malloc(ndata*sizeof(DBdata));
	oper = (int *)malloc(ndata*sizeof(int));
	out_data = (char **)malloc(ndata*sizeof(char *));
	sum = (int **)malloc(ndata*sizeof(int *));
	out_null_value = (double *)malloc(ndata*sizeof(double));

	for (j=0, i=0; i< ndata; i++) {
		if (OMget_array_val(data_id, data[i], &val_id, OM_OBJ_RD) != 1) {
			ERR_RETURN("cannot get field_xyz");
		}
		if (OMget_name_int_val(val_id, OMstr_to_name("valid"), &valid) != 1) {
			ERR_RETURN("cannot get valid");
		}
		if (!valid) {
			if (OMget_name_str_val(val_id, OMstr_to_name("name"), &label, 0) != 1) {
				ERR_RETURN("cannot get name");
			}
			ERRerror("DVtable_to_unif", 1, ERR_ORIG, " column %s has invalid field data type. Ignored", label);
			continue;
		}
		tmp_id = OMfind_subobj(val_id, OMstr_to_name("values"), OM_OBJ_RD);
		if (OMget_data_type(tmp_id, &dtype) != 1) {
			ERR_RETURN("cannot get values type");
		}
		if (dtype == DTYPE_STRING) {
			if (OMget_name_str_val(val_id, OMstr_to_name("name"), &label, 0) != 1) {
				ERR_RETURN("cannot get name");
			}
			ERRerror("DVtable_to_unif", 1, ERR_ORIG, " column %s has invalid field data type. Ignored", label);
			continue;
		}
		stat = 0;
		if (i==0)
			if (OMget_name_int_val(val_id, OMstr_to_name("count_records"), &stat) != 1)
				stat = 0;
		if (stat) {
			data_data[j].null_value = (char *)malloc(sizeof(double));
			count_records = stat;
			data_data[j].size = 0;
			data_data[j].u_size = 0;
			*((double *)(data_data[j].null_value)) = 0;
			data_data[j].type = DTYPE_INT;
			data_data[j].values = NULL;
			data_data[j].u_values = NULL;
			data_data[j].null_flag = 0;
			data_data[j].name = (char *)malloc(32*sizeof(char));
			strcpy(data_data[j].name, "Records Count");
			oper[j] = 0;
			out_null_value[j] = 0;
		}
		else {
			if (OMget_name_int_val(val_id, OMstr_to_name("operation"), &oper[j]) != 1) {
				ERR_RETURN("cannot get operation value");
			}
			stat = get_table_data(val_id, data_data+j, 0);

			if (stat != 1) {
				if (OMget_name_str_val(val_id, OMstr_to_name("name"), &label, 0) != 1) {
					ERR_RETURN("cannot get name");
				}
				ERRerror("DVtable_to_unif", 1, ERR_ORIG, " column %s has invalid field data. Ignored", label);
				continue;
			}
			if (oper[j] == 0)
				out_null_value[j] = 0;
			else if (data_data[j].null_flag == 0) {
				OMget_name_real_val(val_id, OMstr_to_name("max"), &null_value);
				out_null_value[j] = null_value+1;
			}
			else
				out_null_value[j] = *((double *)(data_data[j].null_value));
		}
		if (oper[j]) {
			sum[j] = (int *)ARRalloc(NULL, DTYPE_INT, nnodes, NULL);
			memset(sum[j], 0, nnodes*sizeof(int));
		}
		else
			sum[j] = NULL;
		j++;
	}
	nnode_data = j;

	if (FLDset_node_data_ncomp (out, nnode_data) != 1) {
		ERR_RETURN("Error setting nnode_data");
	}

	for (comp=0; comp<nnode_data; comp++) {
		strcpy(units,"");
		if (FLDset_node_data_comp (out, comp, 1, data_data[comp].name, units) != 1) {
			ERR_RETURN("Error setting node component");
		}
		dtype = data_data[comp].type;
		FLDset_node_data_type(out, comp, dtype);
		if (FLDget_node_data(out, comp, &dtype, &(out_data[comp]),
				     &size, OM_GET_ARRAY_WR) != 1) {
			ERR_RETURN("cannot get node data");
		}
		val = 0;
		UTILinit_array(out_data[comp], nnodes, val, dtype);

		UTILdouble_to_type(&val, out_null_value[comp], dtype);
		if (FLDset_node_null_data(out, comp, (char *)&val, dtype) != 1) {
			ERR_RETURN("Error setting null value");
		}
	}
	for (i=0; i<nvals; i++) {
		for (stat = 1,j=0; j<ndim; j++) {
			if (discrete[j]) {
				ijk[j] = get_unique_index(xyz_data+j, i);
				if (ijk[j] < 0) {
					stat = 0;
					break;
				}
			}
			else {
				dtype = xyz_data[j].type;
				UTILtype_to_double(&point, xyz_data[j].values+i*DTYPEtype_size[dtype],
						   dtype);
				if (bin_size[j] != 0.0)
					point = (point-min[j])/bin_size[j];
				else
					point = 0.0;

				if (point == dims[j])
					ijk[j] = dims[j]-1;
				else
					ijk[j] = point;
				if (ijk[j] < 0 || ijk[j] > dims[j]-1) {
					stat = 0;
					break;
				}
			}
		}
		if (stat == 0)
			continue;

		IJK_TO_INDEX(ndim,dims,ijk,ind);
		if (ind >= nnodes) {
			ERR_RETURN("index outside array range");
		}
		for (comp=0; comp<nnode_data; comp++) {
			dtype = data_data[comp].type;
			size = DTYPEtype_size[dtype];
			if (comp == 0 && count_records) {
				*((int *)(out_data[comp]+ind*size)) += 1;
				continue;
			}
			if (oper[comp])
				sum[comp][ind] += 1;

			UTILtype_to_double(&point, data_data[comp].values+i*size, dtype);

			if (data_data[comp].null_flag &&
			    (point == *((double *)(data_data[comp].null_value))))
				UTILdouble_to_type(out_data[comp]+ind*size,
						   out_null_value[comp], dtype);
			else {
				UTILtype_to_double(&val, out_data[comp]+ind*size, dtype);
				val += point;
					UTILdouble_to_type(out_data[comp]+ind*size, val, dtype);
			}
		}
	}
	for (comp=0; comp<nnode_data; comp++) {
		dtype = data_data[comp].type;
		size = DTYPEtype_size[dtype];
		if (oper[comp]) {
			for (i=0; i<nnodes; i++) {
				UTILtype_to_double(&val, out_data[comp]+i*size, dtype);
				if  (sum[comp][i] != 0) {
					val = val/sum[comp][i];
					UTILdouble_to_type((out_data[comp]+i*size), val, dtype);
				}
				else
					UTILdouble_to_type((out_data[comp]+i*size), out_null_value[comp], dtype);
			}
		}
	}


	if (xyz_axis != NULL)	ARRfree(xyz_axis);
	if (xyz_set != NULL) 	ARRfree(xyz_set);
	if (data != NULL)  	ARRfree(data);
	if (out_xyz != NULL)  	ARRfree(out_xyz);

  	for (comp=0; comp<ndim; comp++) {
		free_table_data(xyz_data+comp);
	}
	for (comp=0; comp<nnode_data; comp++) {
		free_table_data(data_data+comp);
		if (out_data[comp])
			if (out_data[comp] != NULL) ARRfree(out_data[comp]);
		if (sum[comp])
			if (sum[comp] != NULL) ARRfree(sum[comp]);
	}

	free((char *)out_data);
	free((char *)out_null_value);
	free((char *)sum);
	free((char *)oper);
	free((char *)data_data);
	return(1);
}

/* 64-bit porting. Only Modified Internally */
static int get_table_data(OMobj_id obj, DBdata *data, int discrete)
{
	OMobj_id id;
	xp_long i, j, size;
	int found;

	id = OMfind_subobj(obj, OMstr_to_name("values"), OM_OBJ_RD);
	if (OMget_data_type(id, &(data->type)) != 1) {
		ERR_RETURN("cannot get values type");
	}
	if (OMget_name_int_val(obj, OMstr_to_name("null_flag"), &(data->null_flag)) != 1) {
		data->null_flag = 0;
	}
	if (OMget_name_str_val(obj, OMstr_to_name("name"), &(data->name), 0) != 1) {
		ERR_RETURN("cannot get name");
	}
	data->u_size = 0;
	data->values = NULL;
	data->u_values = NULL;
	if (discrete) {
		data->size = 0;
		switch (data->type) {
		      case DTYPE_BYTE:
		      case DTYPE_CHAR:
		{
			char *values;
			char *dvalues;

			if (data->null_flag) {
				data->null_value = (char *)malloc(sizeof(double));
				if (OMget_name_real_val(obj, OMstr_to_name("null_value"),
							(double *)(data->null_value)) != 1) {
					ERR_RETURN("cannot get null_value");
				}
			}
			data->values = OMret_name_array_ptr(obj, OMstr_to_name("values"),
							    OM_GET_ARRAY_RD, &size, NULL);
			if (size)
				data->u_values = ARRalloc(NULL, DTYPE_BYTE, size, NULL);

			data->size = size;
			values = (char *)data->values;
			dvalues = (char *)data->u_values;

			for (i=0; i<size; i++) {
				for (found=0,j=0; j<data->u_size; j++)
					if (values[i] == dvalues[j]) {
						found = 1;
						break;
					}
				if (!found) {
					dvalues[data->u_size] = values[i];
					(data->u_size)++;
				}
			}
		}
			break;

		      case DTYPE_SHORT:
		{
			short *values;
			short *dvalues;

			if (data->null_flag) {
				data->null_value = (char *)malloc(sizeof(double));
				if (OMget_name_real_val(obj, OMstr_to_name("null_value"), (double *)(data->null_value)) != 1) {
					ERR_RETURN("cannot get null_value");
				}
			}
			data->values = OMret_name_array_ptr(obj, OMstr_to_name("values"),
							      OM_GET_ARRAY_RD, &size, NULL);
			if (size)
				data->u_values = ARRalloc(NULL, DTYPE_SHORT, size, NULL);

			data->size = size;
			values = (short *)data->values;
			dvalues = (short *)data->u_values;

			for (i=0; i<size; i++) {
				for (found=0,j=0; j<data->u_size; j++)
					if (values[i] == dvalues[j]) {
						found = 1;
						break;
					}
				if (!found) {
					dvalues[data->u_size] = values[i];
					(data->u_size)++;
				}
			}

		}
			break;

		      case DTYPE_INT:
		{
			int *values;
			int *dvalues;

			if (data->null_flag) {
				data->null_value = (char *)malloc(sizeof(double));
				if (OMget_name_real_val(obj, OMstr_to_name("null_value"), (double *)(data->null_value)) != 1) {
					ERR_RETURN("cannot get null_value");
				}
			}
			data->values = OMret_name_array_ptr(obj, OMstr_to_name("values"),
							      OM_GET_ARRAY_RD, &size, NULL);
			if (size)
				data->u_values = ARRalloc(NULL, DTYPE_INT, size, NULL);

			data->size = size;
			values = (int *)data->values;
			dvalues = (int *)data->u_values;

			for (i=0; i<size; i++) {
				for (found=0,j=0; j<data->u_size; j++)
					if (values[i] == dvalues[j]) {
						found = 1;
						break;
					}
				if (!found) {
					dvalues[data->u_size] = values[i];
					(data->u_size)++;
				}
			}
		}
			break;

		      case DTYPE_LONG:
		{
			xp_long *values;
			xp_long *dvalues;

			if (data->null_flag) {
				data->null_value = (char *)malloc(sizeof(double));
				if (OMget_name_real_val(obj, OMstr_to_name("null_value"), (double *)(data->null_value)) != 1) {
					ERR_RETURN("cannot get null_value");
				}
			}
			data->values = OMret_name_array_ptr(obj, OMstr_to_name("values"),
							      OM_GET_ARRAY_RD, &size, NULL);
			if (size)
				data->u_values = ARRalloc(NULL, DTYPE_LONG, size, NULL);

			data->size = size;
			values = (xp_long *)data->values;
			dvalues = (xp_long *)data->u_values;

			for (i=0; i<size; i++) {
				for (found=0,j=0; j<data->u_size; j++)
					if (values[i] == dvalues[j]) {
						found = 1;
						break;
					}
				if (!found) {
					dvalues[data->u_size] = values[i];
					(data->u_size)++;
				}
			}
		}
			break;

		      case DTYPE_FLOAT:
		{
			float *values;
			float *dvalues;

			if (data->null_flag) {
				data->null_value = (char *)malloc(sizeof(double));
				if (OMget_name_real_val(obj, OMstr_to_name("null_value"), (double *)(data->null_value)) != 1) {
					ERR_RETURN("cannot get null_value");
				}
			}

			data->values = OMret_name_array_ptr(obj, OMstr_to_name("values"),
							      OM_GET_ARRAY_RD, &size, NULL);
			if (size)
				data->u_values = ARRalloc(NULL, DTYPE_FLOAT, size, NULL);

			data->size = size;
			values = (float *)data->values;
			dvalues = (float *)data->u_values;

			for (i=0; i<size; i++) {
				for (found=0,j=0; j<data->u_size; j++)
					if (values[i] == dvalues[j]) {
						found = 1;
						break;
					}
				if (!found) {
					dvalues[data->u_size] = values[i];
					(data->u_size)++;
				}
			}
		}
			break;

		      case DTYPE_DOUBLE:
		{
			double *values;
			double *dvalues;

			if (data->null_flag) {
				data->null_value = (char *)malloc(sizeof(double));
				if (OMget_name_real_val(obj, OMstr_to_name("null_value"), (double *)(data->null_value)) != 1) {
					ERR_RETURN("cannot get null_value");
				}
			}
			data->values = OMret_name_array_ptr(obj, OMstr_to_name("values"),
							      OM_GET_ARRAY_RD, &size, NULL);
			if (size)
				data->u_values = ARRalloc(NULL, DTYPE_DOUBLE, size, NULL);

			data->size = size;
			values = (double *)data->values;
			dvalues = (double *)data->u_values;

			for (i=0; i<size; i++) {
				for (found=0,j=0; j<data->u_size; j++)
					if (values[i] == dvalues[j]) {
						found = 1;
						break;
					}
				if (!found) {
					dvalues[data->u_size] = values[i];
					(data->u_size)++;
				}
			}
		}
			break;

		      case DTYPE_STRING:
			id = OMfind_subobj(obj, OMstr_to_name("values"), OM_OBJ_RD);
			if (OMget_array_size(id, &size) != 1 || size == 0) {
				data->size = 0;
				return(0);
			}
			data->size = size;
			data->strings = (char **)malloc(data->size*sizeof(char *));
			data->u_strings = (char **)malloc(data->size*sizeof(char *));

			for (i=0; i<size; i++) {
				if (OMget_str_array_val(id, i, &(data->strings[i]), 0) != 1) {
					ERR_RETURN("cannot get string data");
				}
				for (found = 0, j=0; j<data->u_size; j++)
					if (strcmp(data->strings[i], data->u_strings[j]) == 0) {
						found = 1;
						break;
					}
				if (!found) {
					data->u_strings[data->u_size] = strdup(data->strings[i]);
					(data->u_size)++;
				}
			}
			if (data->null_flag) {
				if (OMget_name_str_val(obj, OMstr_to_name("null_value"),
						       &(data->null_value), 0) != 1)  {
					ERR_RETURN("cannot get null_value");
				}
			}
			break;

		      default:
			data->type = DTYPE_UNSET;
			break;
		}
	}
	else {
		switch (data->type) {
		      case DTYPE_BYTE:
		      case DTYPE_CHAR:
		      case DTYPE_SHORT:
		      case DTYPE_INT:
		      case DTYPE_LONG:
		      case DTYPE_FLOAT:
		      case DTYPE_DOUBLE:
			data->size = 0;
			data->values = OMret_name_array_ptr(obj, OMstr_to_name("values"),
							    OM_GET_ARRAY_RD, &size, NULL);
			data->size = size;
			if (data->null_flag) {
				data->null_value = (char *)malloc(sizeof(double));
				if (OMget_name_real_val(obj, OMstr_to_name("null_value"), (double *)(data->null_value)) != 1) {
					ERR_RETURN("cannot get null_value");
				}
			}
			break;

		      case DTYPE_STRING:
			id = OMfind_subobj(obj, OMstr_to_name("values"), OM_OBJ_RD);
			if (OMget_array_size(id, &size) != 1 || size == 0)
				return(0);
			data->size = size;
			data->strings = (char **)malloc(data->size*sizeof(char *));
			for (i=0; i<data->size; i++)
				if (OMget_str_array_val(id, i, &(data->strings[i]), 0) != 1) {
					ERR_RETURN("cannot get string data");
				}
			if (data->null_flag) {
				if (OMget_name_str_val(obj, OMstr_to_name("null_value"),
						       &(data->null_value), 0) != 1)  {
					ERR_RETURN("cannot get null_value");
				}
			}
			break;

		      default:
			data->type = DTYPE_UNSET;
			break;
		}
	}
	if (data->size == 0)
		return(0);
	else
		return(1);
}

/* 64-bit porting. Only Modified Internally */
static int free_table_data(DBdata *data)
{
	xp_long i;

	if (data->null_flag)
		free(data->null_value);
	if (data->name)
		free(data->name);
	switch (data->type) {
	      case DTYPE_BYTE:
	      case DTYPE_CHAR:
	      case DTYPE_SHORT:
	      case DTYPE_INT:
	      case DTYPE_LONG:
	      case DTYPE_FLOAT:
	      case DTYPE_DOUBLE:
		if (data->values && data->size)
			ARRfree(data->values);
		if (data->u_values && data->u_size)
			ARRfree(data->u_values);
		break;

	      case DTYPE_STRING:
		for (i=0; i<data->size; i++)
			free(data->strings[i]);
		free(data->strings);
		for (i=0; i<data->u_size; i++)
			free(data->u_strings[i]);
		free(data->u_strings);
		break;

	      default:
		break;
	}
	return(1);
}

/* 64-bit porting. Directly Modified */
static xp_long get_unique_index(DBdata *data, xp_long ind)
{
	xp_long i;

	if (data->size == 0)
		return(-1);

	switch (data->type) {
	      case DTYPE_BYTE:
	      case DTYPE_CHAR:
	{
		char *values;
		char *dvalues;

		values = (char *)data->values;
		dvalues = (char *)data->u_values;

		for (i=0; i<data->u_size; i++)
			if (values[ind] == dvalues[i])
				return(i);
	}
		break;

	      case DTYPE_SHORT:
	{
		short *values;
		short *dvalues;

		values = (short *)data->values;
		dvalues = (short *)data->u_values;

		for (i=0; i<data->u_size; i++)
			if (values[ind] == dvalues[i])
				return(i);
	}
		break;

	      case DTYPE_INT:
	{
		int *values;
		int *dvalues;

		values = (int *)data->values;
		dvalues = (int *)data->u_values;

		for (i=0; i<data->u_size; i++)
			if (values[ind] == dvalues[i])
				return(i);
	}
		break;

	      case DTYPE_LONG:
	{
		xp_long *values;
		xp_long *dvalues;

		values = (xp_long *)data->values;
		dvalues = (xp_long *)data->u_values;

		for (i=0; i<data->u_size; i++)
			if (values[ind] == dvalues[i])
				return(i);
	}
		break;

	      case DTYPE_FLOAT:
	{
		float *values;
		float *dvalues;

		values = (float *)data->values;
		dvalues = (float *)data->u_values;

		for (i=0; i<data->u_size; i++)
			if (values[ind] == dvalues[i])
				return(i);
	}
		break;

	      case DTYPE_DOUBLE:
	{
		double *values;
		double *dvalues;

		values = (double *)data->values;
		dvalues = (double *)data->u_values;

		for (i=0; i<data->u_size; i++)
			if (values[ind] == dvalues[i])
				return(i);
	}
		break;

	      case DTYPE_STRING:
		for (i=0; i<data->u_size; i++)
			if (strcmp(data->strings[ind], data->u_strings[i]) == 0)
				return(i);
		break;

	      default:
		return (-1);
	}
	return(-1);
}


/* 64-bit porting. Directly Modified */
static int get_unique_label(DBdata *data, xp_long ind, char *label)
{
	if (data->u_size == 0)
		return(0);

	switch (data->type) {
	      case DTYPE_BYTE:
	      case DTYPE_CHAR:
	{
		char *dvalues;

		dvalues = (char *)data->u_values;

		sprintf(label, "%c", dvalues[ind]);
	}
		break;

	      case DTYPE_SHORT:
	{
		short *dvalues;

		dvalues = (short *)data->u_values;

		sprintf(label, "%d", (int)(dvalues[ind]));
	}
		break;

	      case DTYPE_INT:
	{
		int *dvalues;

		dvalues = (int *)data->u_values;

		sprintf(label, "%d", dvalues[ind]);
	}
		break;

	      case DTYPE_LONG:
	{
		xp_long *dvalues;

		dvalues = (xp_long *)data->u_values;

		sprintf(label, "%ld", dvalues[ind]);
	}
		break;

	      case DTYPE_FLOAT:
	{
		float *dvalues;

		dvalues = (float *)data->u_values;

		sprintf(label, "%f", dvalues[ind]);
	}
		break;

	      case DTYPE_DOUBLE:
	{
		double *dvalues;

		dvalues = (double *)data->u_values;

		sprintf(label, "%f", (float)(dvalues[ind]));
	}
		break;

	      case DTYPE_STRING:
		strcpy(label, data->u_strings[ind]);
		break;

	      default:
		return (-1);
	}
	return(1);
}

/* 64-bit porting. Only Modified Internally */
static int get_label(double value, int type, char *label)
{
	switch (type) {
	      case DTYPE_BYTE:
	      case DTYPE_CHAR:
		sprintf(label, "%c", (char)value);
		break;

	      case DTYPE_SHORT:
	      case DTYPE_INT:
		sprintf(label, "%d", (int)value);
		break;

	      case DTYPE_LONG:
		sprintf(label, "%ld", (xp_long)value);
		break;

	      case DTYPE_FLOAT:
	      case DTYPE_DOUBLE:
		sprintf(label, "%f", (float)value);
		break;

	      case DTYPE_STRING:
		return(-1);

	      default:
		return (-1);
	}
	return(1);
}
