/*
			Copyright (c) 2004 by
			Advanced Visual Systems Inc.
			All Rights Reserved

	This software comprises unpublished confidential information of
	Advanced Visual Systems Inc. and may not be used, copied or made
	available to anyone, except in accordance with the license
	under which it is furnished.

	This file is under Perforce control
	$Id: //depot/express/fcs70/modules/thresh0cell.c#1 $
*/
#define XP_WIDE_API
#include <avs/util.h>
#include <avs/err.h>
#include <avs/om.h>
#include <avs/fld.h>
#include <avs/arr.h>
#include <avs/dv_util.h>

#define METHOD_SUCCESS 1
#define METHOD_FAILURE 0

#define ERR_RETURN(A) {ERRerror("thresh_null_cells", 0, ERR_ORIG, A); \
                       return(METHOD_FAILURE);}

#define MAX_NAME_SIZE 1024

int FUNCthresh_null_cells (OMobj_id, int, OMobj_id, int *);

int DVthresh_null_cells_update(OMobj_id elem_id)
{
    OMobj_id in, out, out_link, comp_id;
    int stat, comp, copy_in;

    in = OMfind_subobj(elem_id, OMstr_to_name("in"), OM_OBJ_RD);
    out = OMfind_subobj(elem_id, OMstr_to_name("out_thresh"), OM_OBJ_RW);
    out_link = OMfind_subobj(elem_id, OMstr_to_name("out"), OM_OBJ_RW);

    comp_id = OMfind_subobj(elem_id, OMstr_to_name("thresh_component"),
                            OM_OBJ_RW);
    OMget_int_val(comp_id, &comp);

    FUNCthresh_null_cells(in, comp, out, &copy_in);
    if (copy_in) {
        stat = OMset_obj_ref(out_link, in, 0);
        if (stat != OM_STAT_SUCCESS) {
            ERR_RETURN("Error setting out reference");
        }
    }
    else {
        stat = OMset_obj_ref(out_link, out, 0);
        if (stat != OM_STAT_SUCCESS) {
            ERR_RETURN("Error setting out reference");
        }
    }
    return METHOD_SUCCESS;
}

int FUNCthresh_null_cells (OMobj_id in, int thr_comp, OMobj_id out, int *copy_in)
{
    xp_long nnodes, old_ncells, out_ncells, size, out_conn_size;
    xp_long *out_conn_list, *cell_map;
    int   data_type, data_id, veclen, ncomp, comp, null_flag;
    int   stat, nspace, nsets, set, ns=0;
    char  cell_name[MAX_NAME_SIZE], units[MAX_NAME_SIZE], label[MAX_NAME_SIZE];
    float *out_coord;
    char  *cell_data, *mesh_info;
    double null_value;
    OMobj_id in_set, cell_set;

    /* Skip over links and do a bit of sanity testing. */
    if (OMget_obj_val(in, &in) != OM_STAT_SUCCESS) {
        ERR_RETURN("Invalid in field");
    }
    if (FLDget_nnodes(in, &nnodes) != 1) {
        ERR_RETURN("Error getting nnodes");
    }
    if (FLDset_nnodes (out, 0) != 1) {
        ERR_RETURN("Error setting nnodes");
    }
    nsets = 0;
    if (FLDget_ncell_sets(in, &nsets) != 1) {
        ERR_RETURN("Error getting nsets");
    }
    /* We will be adding cell sets one by one */
    if (FLDset_ncell_sets(out, 0) != 1) {
        ERR_RETURN("cannot set nsets");
    }

    if (FLDget_nspace (in, &nspace) != 1) {
        ERR_RETURN("Error setting nspace");
    }
    if (FLDset_nspace (out, nspace) != 1) {
        ERR_RETURN("Error setting nspace");
    }

    /* Its hard to clean up from a failed FLDget_mesh_info call,
     * so do a "test probe" of the call that is most likely to fail.
     */
    if (nsets > 0) {
        if (FLDget_cell_set(in, 0, &in_set) < 1) {
            ERR_RETURN("cannot get cell set");
        }
    }
    mesh_info = NULL;
    stat = FLDget_mesh_info(in, &mesh_info);
    if (!stat) {
        ERR_RETURN("cannot create info table");
    }

    /* Loop over cell sets */

    for(set=0; set<nsets; set++) {
        if (FLDget_cell_set(in, set, &in_set) != 1) {
            ERR_RETURN("Error getting cell set");
        }
        if (FLDget_ncells(in_set, &old_ncells) != 1) {
            ERR_RETURN("Error getting ncells");
        }
        if (FLDget_cell_data_ncomp(in_set, &ncomp) != 1) {
            ERR_RETURN("Error getting cell set ncomp");
        }
        if (thr_comp >= ncomp) {
            ERR_RETURN("bad component number, threshold is not performed");
        }
        if (ncomp>0) {
            if (FLDget_cell_data_veclen(in_set, thr_comp, &veclen) !=1) {
                ERR_RETURN("Error getting veclen");
            }
            if (veclen != 1) {
                ERR_RETURN("component is not scalar, threshold is not performed");
            }
            if (FLDget_cell_null_data(in_set, thr_comp, &null_flag, (char *)&null_value) != 1) {
                ERR_RETURN("cannot get null data");
            }
            /* Make it OK for some sets to have null_flag == 0 ? */
            if (null_flag == 0) {
                *copy_in = 1;
                if (FLDset_nnodes (out, 0) != 1) {
                    ERR_RETURN("Error setting nnodes");
                }
                if (FLDset_ncell_sets(out, 0) != 1) {
                    ERR_RETURN("cannot set nsets");
                }
                return(1);
            }
            else {
                *copy_in = 0;
            }
            if (FLDget_cell_data(in_set, thr_comp, &data_type, &cell_data,
                                 &size, OM_GET_ARRAY_RD) != 1) {
                ERR_RETURN("cannot get cell data");
            }
        }

        stat = UTILthresh_null_from_cells( mesh_info, set,
                   cell_data, data_type,
                   (char *)&null_value,
                   &cell_map, &out_ncells,
                   &out_conn_list, &out_conn_size );

        if (cell_data) {
		ARRfree(cell_data);
                cell_data = NULL;
	}

        if (!stat) {
            ERR_RETURN("cannot thresh_null_from_cells");
        }

        if (FLDget_cell_set_name(in_set, cell_name, MAX_NAME_SIZE) != 1) {
            ERR_RETURN("Error getting set name");
        }
        if (FLDadd_cell_set(out, cell_name) != 1) {
            ERR_RETURN("Error adding cell set");
        }
        if (FLDget_cell_set(out, ns++, &cell_set) != 1) {
            ERR_RETURN("Error getting out cell set");
        }
        if (FLDset_ncells(cell_set, out_ncells) != 1) {
            ERR_RETURN("Error setting ncells");
        }
        if (FLDset_node_connect(cell_set, out_conn_list, out_conn_size,
                                OM_SET_ARRAY_FREE) != 1) {
            ERR_RETURN("Error setting node connect list");
        }
        out_conn_list = NULL;
        if (FLDset_cell_data_ncomp(cell_set, ncomp) != 1) {
            ERR_RETURN("Error setting cell ncomps");
        }

        /* Loop over cell data components */

        for (comp=0; comp<ncomp; comp++) {

            char   *in_cell_data, *out_cell_data;

            if (FLDget_cell_data_units(in_set, comp, units, MAX_NAME_SIZE) != 1) {
                strcpy(units, "");
            }
            if (FLDget_cell_data_label(in_set, comp, label, MAX_NAME_SIZE) != 1) {
                strcpy(label, "");
            }
            if (FLDget_cell_data_veclen(in_set, comp, &veclen) != 1) {
                ERR_RETURN("Error getting veclen");
            }
            if (FLDset_cell_data_comp (cell_set, comp, veclen, label, units) != 1) {
                ERR_RETURN("Error setting cell component");
            }
            if (FLDcopy_cell_minmax(in_set, cell_set, comp, comp) != 1) {
                ERR_RETURN("Error copying node minmax");
            }
            if (FLDcopy_cell_minmax_vec(in_set, cell_set, comp, comp) != 1) {
                ERR_RETURN("Error copying node minmax");
            }
            if (FLDget_cell_data_id(in_set, comp, &data_id) == 1)
                FLDset_cell_data_id(cell_set, comp, data_id);

            if (FLDget_cell_data(in_set, comp, &data_type, &in_cell_data,
                                 &size, OM_GET_ARRAY_RD) != 1) {
                ERR_RETURN("Error getting cell data");
            }
            if (FLDget_cell_data(cell_set, comp, &data_type, &out_cell_data,
                                 &size, OM_GET_ARRAY_WR) != 1) {
                ERR_RETURN("Error getting cell data for writing");
            }
            /* Can't use UTILget_data_list because it uses the field table
             * ("mesh_info") to look at the nnodes.
             */
            {
                int i, n;
                int bytes_per_cell = veclen * DTYPEtype_size[data_type];
                for( i = 0; i < old_ncells; ++i ) {
                    n = cell_map[i];
                    /* n == -1 means skip cell. */
                    if( n >= 0 ) {
                        memcpy( out_cell_data + (n * bytes_per_cell),
                                in_cell_data  + (i * bytes_per_cell),
                                bytes_per_cell );
                    }
                }
            }

            if (FLDget_cell_null_data(in_set, comp, &null_flag, (char *)&null_value) != 1) {
                ERR_RETURN("cannot get null data");
            }
            if (null_flag) {
                if (FLDset_cell_null_data(cell_set, comp, (char *)&null_value, data_type) != 1) {
                    ERR_RETURN("Error setting null value");
                }
            }
            else {
                if (FLDset_node_null_flag(cell_set, comp, 0) != 1) {
                    ERR_RETURN("Error setting null flag");
                }
            }
            if (in_cell_data)  ARRfree(in_cell_data);
            if (out_cell_data) ARRfree(out_cell_data);

        } /* Loop over cell data components */

    } /* Loop over sets */


    /*** OUTPUT FIELD ***/

    if (FLDset_nnodes (out, nnodes) != 1) {
        ERR_RETURN("Error setting nnodes");
    }
    if (FLDget_coord_units(in, units, MAX_NAME_SIZE) == 1) {
        if (FLDset_coord_units (out, units) != 1) {
            ERR_RETURN("Error setting units");
        }
    }

    if (FLDget_coord(in, &out_coord, &size, OM_GET_ARRAY_RD) != 1) {
        ERR_RETURN("Error getting coordinate array");
    }
    if (FLDset_coord(out, out_coord, size, OM_SET_ARRAY_FREE) != 1) {
        ERR_RETURN("Error setting coordinate array");
    }
    out_coord = NULL;

    FLDfree_mesh_info(mesh_info);
    if (cell_map)
        free(cell_map);
    return(1);
}
