/*
			Copyright (c) 1994 by
			Advanced Visual Systems Inc.
			All Rights Reserved

	This software comprises unpublished confidential information of
	Advanced Visual Systems Inc. and may not be used, copied or made
	available to anyone, except in accordance with the license
	under which it is furnished.

	This file is under Perforce control
	$Id: //depot/express/fcs70/modules/wr_ucd.c#1 $
*/
/*-----------------------------------------------------*
 *                                                     *
 *             ****  write ucd module  ****            *
 *                                                     *
 *                                                     *
 *-----------------------------------------------------*/

#define XP_WIDE_API	/* Use Wide APIs */

#include <stdio.h>

#include <avs/util.h>
#include <avs/err.h>
#include <avs/om.h>
#include <avs/dtype.h>
#include <avs/fld.h>
#include <avs/mat.h>
#include <avs/f_utils.h>

#define UCD_NTYPES		8
#define UCD_NUM_CELL_TYPES	8

#define UCD_POINT		0
#define UCD_LINE		1
#define	UCD_TRIANGLE		2
#define UCD_QUADRILATERAL	3
#define UCD_TETRAHEDRON		4
#define UCD_PYRAMID		5
#define UCD_PRISM		6
#define UCD_HEXAHEDRON		7

#define UCD_MAX_NODE_PER_CELL    20

#define MAX_CELL   8

#define ERR_RETURN(A) {ERRerror("write_ucd", 0, ERR_ORIG, A); \
                      if (fp) \
		          fclose (fp);\
                      return(0);}
#define UCD_LABEL_LEN 1024

typedef struct _ctype_struct {
	int id, mat, n, cell_type;
} Ctype;



static const char *set_names[]={"Point", "Line", "Tri", "Quad", "Tet", "Pyr",
			  "Prism", "Hex"};
static int  ucd_types[] = {UCD_POINT, UCD_LINE, UCD_TRIANGLE,
			   UCD_QUADRILATERAL, UCD_TETRAHEDRON,
		           UCD_PYRAMID, UCD_PRISM, UCD_HEXAHEDRON};
static const char *cell_type[] = {"pt", "line", "tri", "quad", "tet",
			    "pyr", "prism", "hex"};


int FUNCwrite_ucd(OMobj_id in, const char *file_name, int format);

int DVwrite_ucd_update(OMobj_id elem_id)
{
    OMobj_id filename, in, form;
    char *str = NULL;
    int format = 0, Result;

    in = OMfind_subobj(elem_id, OMstr_to_name("in"), OM_OBJ_RW);

    filename = OMfind_subobj(elem_id, OMstr_to_name("filename"),
			      OM_OBJ_RD);
    OMget_str_val(filename, &str, 0);

    form = OMfind_subobj(elem_id, OMstr_to_name("format"),
			 OM_OBJ_RD);
    OMget_int_val(form, &format);

    Result = FUNCwrite_ucd(in, str, format);

    if (str) free(str);

    return(Result);
}


/* 64-bit porting. Only Modified Internally */
int FUNCwrite_ucd(OMobj_id in, const char *file_name, int format)
{
	char node_data_labels[UCD_LABEL_LEN], cell_data_labels[UCD_LABEL_LEN],
	node_data_units[UCD_LABEL_LEN], cell_data_units[UCD_LABEL_LEN];
	char data_labels[UCD_LABEL_LEN], data_units[UCD_LABEL_LEN];

	FILE *fp;

	xp_long cell, node, *node_list, num_nlist_nodes;
	int i, veclen, mat_id, num_cell_data, num_node_data, num_model_data, n, j, k;

	int num_vals;

	const char magic = 7;

	xp_long num_nodes, num_cells;
	int *node_comp_list = NULL, *cell_comp_list = NULL;
	int *node_active_list = NULL, *cell_active_list = NULL;

	OMobj_id cell_set;
	xp_long count, ncells, *off, size;
	int cs, nprop, ucd_cell_type, dtype, stat, nsets, prop_size;
	int ncomp, node_ncomp, cell_ncomp, nspace, cell_nnodes;
	float *coord, xyz3[3], *xfm, xform[4][4], *props, **data, **ndata, **cdata;


	/**************
	 ***  body  ***
	 **************/

	if (format) { /* ascii */
		if (!(fp = FILEfopen (file_name, SIO_W_TXT))) {
			ERR_RETURN(" can't open file.");
		}
	}
	else {        /* binary */
		if (!(fp = FILEfopen (file_name, SIO_W_BIN))) {
			ERR_RETURN(" can't open file.");
		}
	}

	stat = FLDget_xform(in, (float *)xform);
	if (stat < 0) {
		ERR_RETURN("cannot get xform for ucd");
	}
	else if (stat == 0) {
		xfm = (float *)0;
	}
	else if (MATmat_is_identity((float *)xform, 4))
		xfm = (float *)0;
	else
		xfm = (float *)xform;

	if (FLDget_nnodes (in, &num_nodes) != 1) {
		ERR_RETURN("Error getting nnodes");
	}
	if (FLDget_nspace (in, &nspace) != 1) {
		ERR_RETURN("Error getting nspace");
	}
	if (FLDget_coord(in,(float **)&coord, &size, OM_GET_ARRAY_RD) != 1) {
		ERR_RETURN("Error getting coordinates");
	}

	if (FLDget_node_data_ncomp(in, &node_ncomp) != 1)
		ncomp = 0;

	for (num_node_data = 0, n=0; n<node_ncomp; n++) {
		if (FLDget_node_data_veclen(in, n, &veclen) != 1) {
			ERR_RETURN("Error getting veclen");
		}
		num_node_data += veclen;
	}

	if (FLDget_ncell_sets(in, &nsets) != 1) {
		ERR_RETURN("cannot get nsets");
	}

	num_cell_data = 0;

	for (num_cells=0, num_nlist_nodes=0, cs=0; cs<nsets; cs++) {
 		if (FLDget_cell_set(in, cs, &cell_set) != 1) {
			ERR_RETURN("cannot get cell set");
		}
		if (FLDget_ncells(cell_set, &ncells) != 1) {
			ERR_RETURN("cannot get ncells");
		}
		num_cells += ncells;

		if (FLDget_cell_set_nnodes(cell_set,  &cell_nnodes) != 1) {
			ERR_RETURN("cannot get cell nnodes");
		}
		num_nlist_nodes += ncells*cell_nnodes;

		if (FLDget_cell_data_ncomp (cell_set, &ncomp) != 1) {
			ncomp = 0;
		}

		if (cs == 0) {
			for (n=0; n<ncomp; n++) {
				if (FLDget_cell_data_veclen(cell_set, n,
							    &veclen) != 1) {
					ERR_RETURN("Error getting veclen");
				}
				num_cell_data += veclen;
			}
			cell_comp_list = (int *)malloc(sizeof(int) *
						       num_cell_data);
			cell_ncomp = ncomp;
			for (n=0; n<ncomp; n++) {
				if (FLDget_cell_data_veclen(cell_set, n,
							    &veclen) != 1) {
					ERR_RETURN("Error getting veclen");
				}
				cell_comp_list[n] = veclen;
			}
		}
		else {
			if (ncomp != cell_ncomp) {
				ERR_RETURN("cell sets with different data, cannot convert");
			}
		}
		if (FLDget_cell_nprops (cell_set, &nprop) != 1) {
			nprop = 0;
		}
	}

	num_model_data = 0;


	if (format) {                    /* "ASCII" */
		/* 
		 * Write ASCII header 
		 */
		fprintf(fp, "# AVS UCD File\n");
		fprintf(fp, "#\n");
		fprintf(fp, "# num_nodes num_cells num_node_data num_cell_data num_model_data\n");
		fprintf(fp, "# nodal coordinates\n");
		fprintf(fp, "# cell type, topology\n");

		fprintf(fp, "%ld %ld %d %d %d\n", 
			num_nodes, 
			num_cells, 
			num_node_data, 
			num_cell_data, 
			num_model_data);
		/* 
		 * Write nodes
		 */
		for (node = 0; node < num_nodes; node++) {
			xyz3[0]=xyz3[1]=xyz3[2]=0.0;
			memcpy((void *)xyz3, (void *)(coord+nspace*node),
			       nspace*sizeof(float));
			if (xfm)
				MATvec3_mat4_multiply(xyz3, (Matr4 *)xfm);

			fprintf(fp, "%ld %f %f %f\n", node, 
				xyz3[0], xyz3[1], xyz3[2]);
		}
		ARRfree(coord);

		/*
		 * Write cells
		 */
		for (count=0,cs=0; cs<nsets; cs++) {
			if (FLDget_cell_set(in, cs, &cell_set) != 1) {
				ERR_RETURN("cannot get cell set");
			}
			if (FLDget_cell_set_name(cell_set, data_labels, UCD_LABEL_LEN) != 1) {
				ERR_RETURN("cannot get cell name");
			}
			ucd_cell_type = -1;
			for (i=0; i< MAX_CELL; i++) {
				if (strcmp(data_labels, set_names[i]) == 0) {
					ucd_cell_type = ucd_types[i];
					break;
				}
			}
			if (ucd_cell_type < 0) {
				ERR_RETURN("unknown cell type, cannot convert");
			}

			if (FLDget_ncells(cell_set, &ncells) != 1) {
				ERR_RETURN("cannot get ncells");
			}

			if (FLDget_cell_set_nnodes(cell_set,  &cell_nnodes) != 1) {
				ERR_RETURN("cannot get cell nnodes");
			}
			if (FLDget_node_connect(cell_set, &node_list, 
						&size, OM_GET_ARRAY_RD) != 1) {
				ERR_RETURN("cannot get cell connectivity");
			}
			if (FLDget_cell_nprops (cell_set, &nprop) != 1) {
				nprop = 0;
			}
			if (nprop) {
				if (FLDget_cell_props(cell_set, (float **)&props,
						      &prop_size, OM_GET_ARRAY_RD) != 1) {
					ERR_RETURN("Error getting cell props");
				}
				mat_id = (int)props[0];
			}
			else
				mat_id = 0;
			for (cell = 0; cell < ncells; cell++) {
				fprintf(fp, "%ld %d %s", count, mat_id, 
					cell_type[ucd_cell_type]);

				for (j = 0; j < cell_nnodes; ++j)
					fprintf(fp, " %ld", node_list[cell*cell_nnodes+j]);
				fprintf(fp, "\n");
				count++;
			}
		}
		/*
		 *  Write node data
		 */
		if (num_node_data) {
			node_comp_list = (int *)malloc(sizeof(int) *
						       num_node_data);

			fprintf(fp, "%d", node_ncomp);
			for (i=0; i<node_ncomp; i++) {
				if (FLDget_node_data_veclen(in, i, &veclen) != 1) {
					ERR_RETURN("Error getting veclen");
				}
				node_comp_list[i] = veclen;
			}
			for (i = 0; i < node_ncomp; ++i)
				fprintf(fp, " %d", node_comp_list[i]);
			fprintf(fp, "\n");

			data = (float **)malloc(node_ncomp*sizeof(float *));
			dtype = DTYPE_FLOAT;

			for (i=0; i<node_ncomp; i++) {
				if (FLDget_node_data_label(in, i, data_labels,
							   UCD_LABEL_LEN) != 1) {
					sprintf(data_labels, "data_%d", i);
				}
				if (FLDget_node_data_units(in, i, data_units,
							   UCD_LABEL_LEN) != 1) {
					strcpy(data_units," ");
				}

				fprintf(fp, "%s, %s\n", data_labels,
					data_units);
				if (FLDget_typed_node_data(in, i, &dtype,
							   (char **)&(data[i]), &size,
							   OM_GET_ARRAY_RD) != 1) {
					ERR_RETURN("Error getting node_data");
				}
			}

			for (node = 0; node < num_nodes; ++node) {
				fprintf(fp, "%ld", node);

				for (j = 0; j < node_ncomp; ++j) {
					num_vals = node_comp_list[j];
					for (k = 0; k < num_vals; ++k) {
						fprintf(fp, " %f", 
							data[j][node * num_vals + k]);
					}
				}
				fprintf(fp, "\n");
			}
			for (i=0; i<node_ncomp; i++) {
				ARRfree(data[i]);
			}
			free(data);

		}

		/*
		 *  Write cell data
		 */
		if (num_cell_data) {
			fprintf(fp, "%d", cell_ncomp);
			for (i = 0; i < cell_ncomp; ++i)
				fprintf(fp, " %d", cell_comp_list[i]);
			fprintf(fp, "\n");

			data = (float **)malloc(cell_ncomp*sizeof(float *));
			dtype = DTYPE_FLOAT;

			if (FLDget_cell_set(in, 0, &cell_set) != 1) {
				ERR_RETURN("cannot get cell set");
			}
			for (i=0; i<cell_ncomp; i++) {
				if (FLDget_cell_data_label(cell_set, i, data_labels,
							   UCD_LABEL_LEN) != 1) {
					sprintf(data_labels, "data_%d", i);
				}
				if (FLDget_cell_data_units(cell_set, i, data_units,
							   UCD_LABEL_LEN) != 1) {
					strcpy(data_units," ");
				}

				fprintf(fp, "%s, %s\n", data_labels,
					data_units);
			}
			for (count=0,cs=0; cs<nsets; cs++) {
				if (FLDget_cell_set(in, cs, &cell_set) != 1) {
					ERR_RETURN("cannot get cell set");
				}
				if (FLDget_ncells(cell_set, &ncells) != 1) {
					ERR_RETURN("cannot get ncells");
				}
				for (i=0; i<cell_ncomp; i++) {
					if (FLDget_typed_cell_data(cell_set, i, &dtype,
								   (char **)&(data[i]), &size,
								   OM_GET_ARRAY_RD) != 1) {
						ERR_RETURN("Error getting cell_data");
					}
				}
				for (cell = 0; cell < ncells; ++cell) {
					fprintf(fp, "%ld", count++);

					for (j = 0; j < cell_ncomp; ++j) {
						num_vals = cell_comp_list[j];
						for (k = 0; k < num_vals; ++k) {
							fprintf(fp, " %f", 
								data[j][cell * num_vals + k]);
						}
					}
					fprintf(fp, "\n");
				}
				for (i=0; i<cell_ncomp; i++) {
					ARRfree(data[i]);
					data[i] = NULL;
				}
			}
			free(data);
		}

	}
	else {
		/*
		 *  Write binary file
		 */
		float *node_data = NULL, *cell_data = NULL;

		float *min_node_data = NULL, *max_node_data = NULL;
		float *min_cell_data = NULL, *max_cell_data = NULL;

		Ctype *cells = NULL;
		int *cell_nlists = NULL;

		int num_nodes_n, num_cells_n, num_nlist_nodes_n;

		/* 64-bit changes. Truncated to 32-bit as binary UCD format doesn't support 64-bit datasets */
		num_nodes_n = (int)num_nodes;
		num_cells_n = (int)num_cells;
		num_nlist_nodes_n = (int)num_nlist_nodes;

		/*  compute list of node lists.  */
		cells = (Ctype *)malloc(sizeof(Ctype) * num_cells_n);
		cell_nlists = (int *)malloc(sizeof(int) * num_nlist_nodes_n);
		if (cell_nlists == NULL || cells == NULL) {
			ERR_RETURN("malloc failed for cells");
		}
		for (n=0, count=0,cs=0; cs<nsets; cs++) {
			if (FLDget_cell_set(in, cs, &cell_set) != 1) {
				ERR_RETURN("cannot get cell set");
			}
			if (FLDget_cell_set_name(cell_set, data_labels, UCD_LABEL_LEN) != 1) {
				ERR_RETURN("cannot get cell name");
			}
			ucd_cell_type = -1;
			for (i=0; i< MAX_CELL; i++) {
				if (strcmp(data_labels, set_names[i]) == 0) {
					ucd_cell_type = ucd_types[i];
					break;
				}
			}
			if (ucd_cell_type < 0) {
				ERR_RETURN("unknown cell type, cannot convert");
			}

			if (FLDget_ncells(cell_set, &ncells) != 1) {
				ERR_RETURN("cannot get ncells");
			}

			if (FLDget_cell_set_nnodes(cell_set,  &cell_nnodes) != 1) {
				ERR_RETURN("cannot get cell nnodes");
			}
			if (FLDget_node_connect(cell_set, &node_list, 
						&size, OM_GET_ARRAY_RD) != 1) {
				ERR_RETURN("cannot get cell connectivity");
			}
			if (FLDget_cell_nprops (cell_set, &nprop) != 1) {
				nprop = 0;
			}
			if (nprop) {
				if (FLDget_cell_props(cell_set, (float **)&props,
						      &prop_size, OM_GET_ARRAY_RD) != 1) {
					ERR_RETURN("Error getting cell props");
				}
				mat_id = (int)props[0];
			}
			else
				mat_id = 0;
			for (cell = 0; cell < ncells; cell++) {
				cells[n].id = n;
				cells[n].mat = mat_id;
				cells[n].cell_type = ucd_cell_type;
				cells[n++].n = cell_nnodes;
				for (i = 0; i < cell_nnodes; i++) /* stupid 1 based */
					cell_nlists[count++] = (int)node_list[cell*cell_nnodes+i]+1;
			}
		}

		ndata = NULL;

		if (num_node_data) {
			ndata = (float **)malloc(node_ncomp*sizeof(float *));
			node_active_list = (int *)malloc(sizeof(int) *
							 num_node_data);
			node_comp_list = (int *)malloc(sizeof(int) *
						       num_node_data);

			if (node_active_list == NULL || node_comp_list == NULL) {
				ERR_RETURN( "can't alloc node list.");
			}
			min_node_data = (float *)malloc(sizeof(float) *
							num_node_data);
			max_node_data = (float *)malloc(sizeof(float) *
							num_node_data);
			/* Note that num_node_data != node_ncomp */
			for (i=0; i<num_node_data; i++) {
				node_comp_list[i] = 0;
				node_active_list[i] = 0;
				min_node_data[i] = 0.0;
				max_node_data[i] = 0.0;
			}
			strcpy(node_data_labels, "");
			strcpy(node_data_units, "");
			dtype = DTYPE_FLOAT;
			for (i=0; i<node_ncomp; i++) {
				if (FLDget_node_data_veclen(in, i, &veclen) != 1) {
					ERR_RETURN("Error getting veclen");
				}
				node_comp_list[i] = veclen;
				if (FLDget_node_data_label(in, i, data_labels,
							   UCD_LABEL_LEN) != 1) {
					sprintf(data_labels, "data_%d", i);
				}
				if (FLDget_node_data_units(in, i, data_units,
							   UCD_LABEL_LEN) != 1) {
					strcpy(data_units," ");
				}

				if (FLDget_typed_node_data(in, i, &dtype,
							   (char **)&(ndata[i]), &size,
							   OM_GET_ARRAY_RD) != 1) {
					ERR_RETURN("Error getting node_data");
				}
				strcat(node_data_labels, data_labels);
				strcat(node_data_units, data_units);
				if (i != (node_ncomp-1)) {
					strcat(node_data_labels, ".");
					strcat(node_data_units, ".");
				}
			}
		}

		cdata = NULL;
		off = NULL;

		if (num_cell_data) {
			cdata = (float **)malloc(cell_ncomp*sizeof(float *));
			off = (xp_long *)malloc(cell_ncomp*sizeof(xp_long));
			cell_active_list = (int *)malloc(sizeof(int) *
							 num_cell_data);
			cell_comp_list = (int *)malloc(sizeof(int) *
						       num_cell_data);

			if (cell_active_list == NULL || cell_comp_list == NULL) {
				ERR_RETURN( "can't alloc cell list.");
			}
			min_cell_data = (float *)malloc(sizeof(float) *
							num_cell_data);
			max_cell_data = (float *)malloc(sizeof(float) *
							num_cell_data);
			/* Note that num_cell_data != cell_ncomp */
			for (i=0; i<num_cell_data; i++) {
				cell_comp_list[i] = 0;
				cell_active_list[i] = 0;
				min_cell_data[i] = 0.0;
				max_cell_data[i] = 0.0;
			}
			strcpy(cell_data_labels, "");
			strcpy(cell_data_units, "");
			dtype = DTYPE_FLOAT;
			if (FLDget_cell_set(in, 0, &cell_set) != 1) {
				ERR_RETURN("cannot get cell set");
			}
			for (i=0; i<cell_ncomp; i++) {
				if (FLDget_cell_data_veclen(cell_set, i, &veclen) != 1) {
					ERR_RETURN("Error getting veclen");
				}
				cell_comp_list[i] = veclen;
				if (FLDget_cell_data_label(cell_set, i, data_labels,
							   UCD_LABEL_LEN) != 1) {
					sprintf(data_labels, "data_%d", i);
				}
				if (FLDget_cell_data_units(cell_set, i, data_units,
							   UCD_LABEL_LEN) != 1) {
					strcpy(data_units," ");
				}
				strcat(cell_data_labels, data_labels);
				strcat(cell_data_units, data_units);
				if (i != (cell_ncomp-1)) {
					strcat(cell_data_labels, ".");
					strcat(cell_data_units, ".");
				}
				off[i] = 0;
				cdata[i] = NULL;
				cdata[i] = (float *)malloc(num_cells_n*veclen*sizeof(float));
				if (cdata[i] == NULL) {
					ERR_RETURN( "can't alloc cell data.");
				}
			}
			for (count=0,cs=0; cs<nsets; cs++) {
				if (FLDget_cell_set(in, cs, &cell_set) != 1) {
					ERR_RETURN("cannot get cell set");
				}
				if (FLDget_ncells(cell_set, &ncells) != 1) {
					ERR_RETURN("cannot get ncells");
				}
				for (i=0; i<cell_ncomp; i++) {
					if (FLDget_typed_cell_data(cell_set, i, &dtype,
								   (char **)&cell_data, &size,
								   OM_GET_ARRAY_RD) != 1) {
						ERR_RETURN("Error getting cell_data");
					}
					memcpy(cdata[i]+off[i], cell_data, cell_comp_list[i]*ncells*sizeof(float));
					off[i] += cell_comp_list[i]*ncells;
					ARRfree(cell_data);
					cell_data = NULL;
				}
			}
		}
		/*  store information into some structures for 
		    binary write.  */

		fwrite (&magic, sizeof(char), 1, fp);

		fwrite (&num_nodes_n, sizeof(int), 1, fp);
		fwrite (&num_cells_n, sizeof(int), 1, fp);
		fwrite (&num_node_data, sizeof(int), 1, fp);
		fwrite (&num_cell_data, sizeof(int), 1, fp);
		fwrite (&num_model_data, sizeof(int), 1, fp);

		fwrite (&num_nlist_nodes_n, sizeof(int), 1, fp);
		fwrite (cells, sizeof(Ctype), num_cells_n, fp);
		fwrite (cell_nlists, sizeof(int), num_nlist_nodes_n, fp);

		node_data = NULL;
		node_data = (float *)malloc(num_nodes_n*sizeof(float));
		if (node_data == NULL) {
			ERR_RETURN( "can't alloc coord.");
		}
		for (i=0; i<num_nodes_n; i++)
			node_data[i] = coord[i*nspace];
		fwrite (node_data, sizeof(float), num_nodes_n, fp);
		if (nspace > 1)
			for (i=0; i<num_nodes_n; i++)
				node_data[i] = coord[i*nspace+1];
		else
			for (i=0; i<num_nodes_n; i++)
				node_data[i] = 0.0;
		fwrite (node_data, sizeof(float), num_nodes_n, fp);
		if (nspace > 2)
			for (i=0; i<num_nodes_n; i++)
				node_data[i] = coord[i*nspace+2];
		else
			for (i=0; i<num_nodes_n; i++)
				node_data[i] = 0.0;
		fwrite (node_data, sizeof(float), num_nodes_n, fp);

		free(node_data);

		if (num_node_data) {
			fwrite (node_data_labels, sizeof(char),
				UCD_LABEL_LEN, fp);
			fwrite (node_data_units, sizeof(char),
				UCD_LABEL_LEN, fp);
			fwrite (&node_ncomp, sizeof(int),
				1, fp);
			fwrite (node_comp_list, sizeof(int),
				num_node_data, fp);
			fwrite (min_node_data, sizeof(float),
				num_node_data, fp);
			fwrite (max_node_data, sizeof(float),
				num_node_data, fp);
			for (i=0; i<node_ncomp; i++) {
				fwrite (ndata[i], sizeof(float),
					num_nodes_n * node_comp_list[i], fp);
				ARRfree(ndata[i]);
			}
			fwrite (node_active_list, sizeof(int),
				num_node_data, fp);
		}

		if (num_cell_data) {
			fwrite (cell_data_labels, sizeof(char),
				UCD_LABEL_LEN, fp);
			fwrite (cell_data_units, sizeof(char),
				UCD_LABEL_LEN, fp);
			fwrite (&cell_ncomp, sizeof(int),
				1, fp);
			fwrite (cell_comp_list, sizeof(int),
				num_cell_data, fp);
			fwrite (min_cell_data, sizeof(float),
				num_cell_data, fp);
			fwrite (max_cell_data, sizeof(float),
				num_cell_data, fp);
			for (i=0; i<cell_ncomp; i++) {
				fwrite (cdata[i], sizeof(float),
					num_cells_n * cell_comp_list[i], fp);
				free(cdata[i]);
			}
			fwrite (cell_active_list, sizeof(int),
				num_cell_data, fp);
		}

		if (node_active_list)
			free (node_active_list);
		if (cell_active_list)
			free (cell_active_list);

		if (node_comp_list)
			free (node_comp_list);
		if (cell_comp_list)
			free (cell_comp_list);

		if (min_node_data)
			free (min_node_data);
		if (max_node_data)
			free (max_node_data);
		if (min_cell_data)
			free (min_cell_data);
		if (max_cell_data)
			free (max_cell_data);

		if (ndata)
			free(ndata);
		if (cdata)
			free(cdata);
		if (off)
			free(off);
		free (cells);
		free (cell_nlists);
	}

	fclose (fp);
	return(1);
}
