//			Copyright (c) 1994 by
//			Advanced Visual Systems Inc.
//			All Rights Reserved
//
//	This software comprises unpublished confidential information of
//	Advanced Visual Systems Inc. and may not be used, copied or made
//	available to anyone, except in accordance with the license
//	under which it is furnished.
//
//	This file is under Perforce control
//	$Id: //depot/express/fcs70/motif_ui/layout.cxx#1 $
//

#include <stdio.h>
#include <stdlib.h>
#include <X11/Xlib.h>
#include <Xm/Frame.h>
#include <Xm/Scale.h>
#include <Xm/Form.h>

#define XP_WIDE_API	/* Use Wide APIs */
#include <avs/event_x.h>
#include "ui/layout.h"
#include "ui_gen.h"
#include "ui/dpyMgr.h"

// Global constants
const int LAYOUT_HANDLE_SIZE = 7;
const int BUFSIZE = 512; // Max object path string characters

//const String EDIT_OBJ_PATH_STRING = "NetworkEditor.ObjectEditor.editControl.edit_obj_path";
//const String DISP_OBJ_PATH_STRING = "NetworkEditor.ObjectEditor.editControl.disp_obj_path";
static UtString EDIT_OBJ_PATH_STRING;
static UtString DISP_OBJ_PATH_STRING;
static Boolean draw_last = False;
static int specialCase;

// Initialize static members of UIlayout class
int             UIlayout::gridX = 4;
int             UIlayout::gridY = 4;
Widget          UIlayout::moveWidget = 0;
UIlayoutState   UIlayout::state = LAYOUT_STATE_NORMAL;
UIlayoutObjectType UIlayout::potentialActionState = LAYOUT_OBJECT_MOVE;
int             UIlayout::editEnable = 0;
int             UIlayout::snapToGrid = 1;
Dimension       UIlayout::last_motion_width = 0;
Dimension       UIlayout::last_motion_height = 0;
Position        UIlayout::last_motion_x = 0;
Position        UIlayout::last_motion_y = 0;
Cursor          UIlayout::moveCursor = 0;
Cursor          UIlayout::resizeTCursor = 0;
Cursor          UIlayout::resizeBCursor = 0;
Cursor          UIlayout::resizeRCursor = 0;
Cursor          UIlayout::resizeLCursor = 0;
Cursor          UIlayout::resizeTLCursor = 0;
Cursor          UIlayout::resizeTRCursor = 0;
Cursor          UIlayout::resizeBLCursor = 0;
Cursor          UIlayout::resizeBRCursor = 0;
GC              UIlayout::expose_gc = 0;
Window          UIlayout::root = 0;
Widget          UIlayout::destParent = 0;
GC              UIlayout::layout_gc = 0;
Boolean         UIlayout::changeX = False;
Boolean         UIlayout::changeY = False;
Boolean         UIlayout::resizeWidth = False;
Boolean         UIlayout::resizeHeight = False;
Position        UIlayout::select_x = 0;
Position        UIlayout::select_y = 0;
Time            UIlayout::moveReleaseTime = 0;

////////////////////////////////////////////////////////////////////////
// DEBUG STUFF
#ifdef DEBUG_LAYOUT
char *stateNames[] = {
        "LAYOUT_STATE_NORMAL",
        "LAYOUT_STATE_MOVE",
        "LAYOUT_STATE_RESIZE",
        "LAYOUT_STATE_PENDING_NORMAL"
};

ostream &
operator <<(ostream &o, UIlayoutState s)
{
        o << stateNames[s];
        return o;
}
#endif

//////////////////////////////////////////////////////////////
// Inline functions

#ifndef DEBUG_LAYOUT
#define INLINE_FUNC inline
#else
#define INLINE_FUNC
#endif


INLINE_FUNC void
copyXAnyEvent(XAnyEvent *to, XAnyEvent *from )
{
    to->type = from->type;
    to->serial = from->serial;
    to->send_event = from->send_event;
    to->display = from->display;
    to->window = from->window;
}

INLINE_FUNC void
UIlayout::setState( UIlayoutState s )
{
#ifdef DEBUG_LAYOUT
    cout << ">>>>>>>> Layout state set to: " << s << endl;
#endif
    state = s;
}

/////////////////////////////////////////////////////////////////////////
// Class ctor's dtor's

UIlayout::UIlayout()
{
    // Get default display context pointer
    const UIdpyContext &dpyCtx = UIgetDisplayContext(getenv("DISPLAY"));
    if( dpyCtx.display() ) {
        // Create move and resize cursors.
        moveCursor = XCreateFontCursor(dpyCtx.display(), 52 );
        resizeTCursor = XCreateFontCursor(dpyCtx.display(), 138 );
        resizeBCursor = XCreateFontCursor(dpyCtx.display(), 16 );
        resizeRCursor = XCreateFontCursor(dpyCtx.display(), 96 );
        resizeLCursor = XCreateFontCursor(dpyCtx.display(), 70 );
        resizeTLCursor = XCreateFontCursor(dpyCtx.display(), 134 );
        resizeTRCursor = XCreateFontCursor(dpyCtx.display(), 136 );
        resizeBLCursor = XCreateFontCursor(dpyCtx.display(), 12 );
        resizeBRCursor = XCreateFontCursor(dpyCtx.display(), 14 );
    }
}

UIlayout::~UIlayout()
{
    // Get default display context pointer
    const UIdpyContext &dpyCtx = UIgetDisplayContext(getenv("DISPLAY"));
    if( dpyCtx.display() ) {
        // Destroy the move and resize cursors
        XFreeCursor(dpyCtx.display(), moveCursor);
        XFreeCursor(dpyCtx.display(), resizeTCursor);
        XFreeCursor(dpyCtx.display(), resizeBCursor);
        XFreeCursor(dpyCtx.display(), resizeRCursor);
        XFreeCursor(dpyCtx.display(), resizeLCursor);
        XFreeCursor(dpyCtx.display(), resizeTLCursor);
        XFreeCursor(dpyCtx.display(), resizeTRCursor);
        XFreeCursor(dpyCtx.display(), resizeBLCursor);
        XFreeCursor(dpyCtx.display(), resizeBRCursor);
    }
}


//////////////////////////////////////////////////////////////////////
// This callback redraws the layout editor frame around the widget
// window after the widget has been redrawn from an expose event.
//
void
UIlayout::overlayWindowExposeHandler(Widget, XPointer, XExposeEvent *)
{
    if( UIisBeingDestroyed( moveWidget ) ) return;
    // Redraw outline

    switch (UIlayout::state) {
        case LAYOUT_STATE_NORMAL:
        case LAYOUT_STATE_MOVE:
        case LAYOUT_STATE_RESIZE:
//          break;
        case LAYOUT_STATE_PENDING_NORMAL:
        {
            UIlayout::drawLayoutObject(moveWidget,
                                       XtWindow(moveWidget), expose_gc,
                                       0, 0,
                                       last_motion_width, last_motion_height,
                                       LAYOUT_OBJECT_RESIZE,
                                       LAYOUT_OBJECT_EXPOSE);
        }
        break;
    }
}

///////////////////////////////////////////////////////////////////////////////

Position
UIlayout::findParentZeroX(Widget w, Widget Parent)
{
    Position  x,y;
    int       X,Y;
    Window    returnedWindow;

    UIgeom::getWidgetPos(XtParent(w),x, y);

    XTranslateCoordinates(XtDisplay(w),
                          XtWindow(Parent),root,
                          x, y,
                          &X,&Y,
                          &returnedWindow);

    return (Position)X;
}

///////////////////////////////////////////////////////////////////////////////

Position
UIlayout::findParentZeroY(Widget w, Widget Parent)
{
    Position      x,y;
    int           X,Y;
    int           Unused;
    Window        returnedWindow;
    Window        child;
    Window        newChild;
    Widget        newParent;
    unsigned int  keys_buttons;

    UIgeom::getWidgetPos(XtParent(w),x,y);

    XTranslateCoordinates(XtDisplay(w),
                          XtWindow(Parent),root,
                          x,y,
                          &X,&Y,
                          &returnedWindow);

    newChild = root;

    while (newChild != 0) {
        child = newChild;
        XQueryPointer(XtDisplay(w),
                      child, &root,
                      &newChild,
                      &Unused, &Unused,
                      &Unused, &Unused,
                      &keys_buttons);
    }

    newParent = XtWindowToWidget(XtDisplay(w),child);

    // If returnedWindow is a plain X11 window and not
    // associated with a widget, newParent will be NULL
    // Otherwise, use newParent as destParent for gridding
    // purposes.

    if (newParent != NULL) {
        destParent = newParent;
    }

    return (Position)Y;
}


void
UIlayout::groupObjects()
{
    WidgetList    children;
    Cardinal      numChildren;
    int           i;

    if (!UIlayout::editEnable) return;

    if (UIlayout::state == LAYOUT_STATE_NORMAL) return;

    if (XtIsComposite(moveWidget) ||
        XtIsSubclass(moveWidget,xmFrameWidgetClass)) {

        XtVaGetValues(moveWidget,
                      XmNchildren,    &children,
                      XmNnumChildren, &numChildren,
                      NULL);

        for (i=0; i < numChildren; i++) {
            UIlayout::removeDirectEventHandlers(children[i]);
            UIlayout::addIndirectEventHandlers(children[i], moveWidget);
        }
    }
}


void
UIlayout::ungroupObjects()
{
    WidgetList     children;
    Cardinal       numChildren;
    int            i;

    if (!UIlayout::editEnable) return;

    if (UIlayout::state == LAYOUT_STATE_NORMAL) return;

    if (XtIsComposite(moveWidget) ||
        XtIsSubclass(moveWidget,xmFrameWidgetClass)) {

        XtVaGetValues(moveWidget,
                      XmNchildren,    &children,
                      XmNnumChildren, &numChildren,
                      NULL);

        for (i=0; i < numChildren; i++) {
            UIlayout::removeIndirectEventHandlers(children[i], moveWidget);
        }
        addDirectEventHandlers(moveWidget);
    }
}

void
UIlayout::cleanupOnObjectDestroy( Widget w )
{
    UIlayout::state = LAYOUT_STATE_NORMAL;
    if (layout_gc)
        XFreeGC(XtDisplay(w),layout_gc);
    layout_gc = NULL;
    moveWidget = 0;
}

///////////////////////////////////////////////////////////////////
// The indirectXXX event handlers are set on all children
// widgets of a composite object, ex. UIscrolledWindow which has a
// workAreaWidget, scrollbar widgets, clipWindow widget, etc.
// This handlers causes all buttonevents to be sent to the widget
// registered.

void
UIlayout::indirectButtonPressCB(Widget w, Widget hWnd,
                                XButtonEvent *event, Boolean *contEvent )
{
    Position   x,y;
    XButtonEvent  newEvent;

    // copy the original event struture members that
    // we'll need to a new event structure and then
    // modify x and y.

    copyXAnyEvent( (XAnyEvent *)&newEvent, (XAnyEvent *)event );

    newEvent.root = event->root;
    newEvent.subwindow = event->subwindow;
    newEvent.button = event->button;
    newEvent.x = event->x;
    newEvent.y = event->y;
    newEvent.x_root = event->x_root;
    newEvent.y_root = event->y_root;
    newEvent.time = event->time;

    while (w != hWnd) {
        UIgeom::getWidgetPos(w,x,y);
        newEvent.x += x;
        newEvent.y += y;
        w = XtParent(w);
    }

    UIlayout::buttonPressCB(hWnd,NULL,&newEvent,contEvent);
}

void
UIlayout::indirectButtonReleaseCB(Widget w, Widget hWnd,
                                  XButtonEvent *event, Boolean *contEvent)
{
    Position   x,y;
    XButtonEvent  newEvent;

    // copy the original event struture members that
    // we'll need to a new event structure and then
    // modify x and y.

    copyXAnyEvent( (XAnyEvent *)&newEvent, (XAnyEvent *)event );

    newEvent.root = event->root;
    newEvent.subwindow = event->subwindow;
    newEvent.button = event->button;
    newEvent.x = event->x;
    newEvent.y = event->y;
    newEvent.x_root = event->x_root;
    newEvent.y_root = event->y_root;
    newEvent.time = event->time;

    while (w != hWnd) {
        UIgeom::getWidgetPos(w,x,y);
        newEvent.x += x;
        newEvent.y += y;
        w = XtParent(w);
    }

    UIlayout::buttonReleaseCB(hWnd,NULL,&newEvent,contEvent);
}

void
UIlayout::indirectButtonMotionCB(Widget w, Widget hWnd, XMotionEvent *event)
{
    Position   x,y;
    XMotionEvent  newEvent;

    // copy the original event struture members that
    // we'll need to a new event structure and then
    // modify x and y.

    copyXAnyEvent( (XAnyEvent *)&newEvent, (XAnyEvent *)event );

    newEvent.root = event->root;
    newEvent.subwindow = event->subwindow;
    newEvent.x = event->x;
    newEvent.y = event->y;
    newEvent.x_root = event->x_root;
    newEvent.y_root = event->y_root;
    newEvent.time = event->time;
    newEvent.state = event->state;
    newEvent.is_hint = event->is_hint;
    newEvent.same_screen = event->same_screen;

    while (w != hWnd) {
        UIgeom::getWidgetPos(w,x,y);
        newEvent.x += x;
        newEvent.y += y;
        w = XtParent(w);
    }

    UIlayout::buttonMotionCB(hWnd, NULL, &newEvent);
}

void
UIlayout::indirectPointerMotionCB(Widget w, Widget hWnd, XMotionEvent *event)
{
    Position   x,y;
    XMotionEvent  newEvent;

    // copy the original event struture members that
    // we'll need to a new event structure and then
    // modify x and y.

    copyXAnyEvent( (XAnyEvent *)&newEvent, (XAnyEvent *)event );
    newEvent.x = event->x;
    newEvent.y = event->y;
    newEvent.x_root = event->x_root;
    newEvent.y_root = event->y_root;
    newEvent.time = event->time;
    newEvent.state = event->state;

    while (w != hWnd) {
        UIgeom::getWidgetPos(w,x,y);
        newEvent.x += x;
        newEvent.y += y;
        w = XtParent(w);
    }

    UIlayout::pointerMotionCB(hWnd, NULL, &newEvent);
}

void
UIlayout::addIndirectEventHandlers(Widget hWnd, Widget SendTo)
{
    WidgetList  children;
    Cardinal    numChildren;
    int         i;
    OMobj_id    ObjId;

    if (UIisBeingDestroyed(hWnd)) return;

    if (XtIsWidget(hWnd)) {

        ObjId = UIwidgetToObjId(hWnd);

        if (hWnd == UIobjIdToWidget(ObjId)) {
            ObjId = UIwidgetToObjId(hWnd);
            OMpush_ctx(ObjId, OM_STATE_USR, OM_CTX_TOSS_NOTIFIES, 0);
            SetIntVal(ObjId, "groupToParent",1);
            OMpop_ctx(ObjId);
        }

        // Add child widget button press event handlers
        XtInsertEventHandler(hWnd,
                             ButtonPressMask,
                             False,
                             (XtEventHandler)UIlayout::indirectButtonPressCB,
                             (XtPointer)SendTo,
                             XtListTail);

        XtInsertEventHandler(hWnd,
                             ButtonReleaseMask,
                             False,
                             (XtEventHandler)UIlayout::indirectButtonReleaseCB,
                             (XtPointer)SendTo,
                             XtListTail);

        XtAddEventHandler(hWnd,
                          Button2MotionMask,
                          False,
                          (XtEventHandler)UIlayout::indirectButtonMotionCB,
                          (XtPointer)SendTo);

        XtAddEventHandler(hWnd,
                          PointerMotionMask,
                          False,
                          (XtEventHandler)UIlayout::indirectPointerMotionCB,
                          (XtPointer)SendTo);

        if (XtIsComposite(hWnd) ||
            XtIsSubclass(hWnd,xmFrameWidgetClass) ||
            XtIsSubclass(hWnd,xmFormWidgetClass) ||
            XtIsSubclass(hWnd,xmScaleWidgetClass)) {

            XtVaGetValues(hWnd,
                          XmNchildren,    &children,
                          XmNnumChildren, &numChildren,
                          NULL);

            for (i=0; i < numChildren; i++) {
                UIlayout::addIndirectEventHandlers(children[i],SendTo);
            }
        }
    }
}

void
UIlayout::removeIndirectEventHandlers(Widget hWnd, Widget SendTo)
{
    WidgetList  children;
    Cardinal    numChildren;
    int         i;

    if (XtIsWidget(hWnd)) {

        // Remove child widget button press event handlers
        XtRemoveEventHandler(hWnd,
                             ButtonPressMask,
                             False,
                             (XtEventHandler)UIlayout::indirectButtonPressCB,
                             (XtPointer)SendTo);

        XtRemoveEventHandler(hWnd,
                             ButtonReleaseMask,
                             False,
                             (XtEventHandler)UIlayout::indirectButtonReleaseCB,
                             (XtPointer)SendTo);

        XtRemoveEventHandler(hWnd,
                             Button2MotionMask,
                             False,
                             (XtEventHandler)UIlayout::indirectButtonMotionCB,
                             (XtPointer)SendTo);

        if (XtIsComposite(hWnd) ||
            XtIsSubclass(hWnd,xmFrameWidgetClass) ||
            XtIsSubclass(hWnd,xmFormWidgetClass) ||
            XtIsSubclass(hWnd,xmScaleWidgetClass)) {

            XtVaGetValues(hWnd,
                          XmNchildren,    &children,
                          XmNnumChildren, &numChildren,
                          NULL);

            for (i=0; i < numChildren; i++) {
                UIlayout::removeIndirectEventHandlers(children[i],SendTo);
            }
        }
    }
}

void
UIlayout::addDirectEventHandlers(Widget hWnd)
{
    WidgetList  children;
    Cardinal    numChildren;
    int         i;
    UIwindow    *UIclass;

    if (!XtIsComposite(hWnd) &&
        !XtIsSubclass(hWnd,xmFrameWidgetClass)) return;

    XtVaGetValues(hWnd,
                  XmNchildren,    &children,
                  XmNnumChildren, &numChildren,
                  NULL);

    for (i=0 ; i < numChildren; i++) {
        if (XtIsComposite(children[i]) ||
            XtIsSubclass(hWnd,xmScaleWidgetClass) ||
            XtIsSubclass(hWnd,xmFrameWidgetClass)) {

            addDirectEventHandlers(children[i]);
        }

        // Check if the child is a widget, and if the widget is a
        // part of a composite widget, see if we're the main widget
        // of that composite.
        if (children[i] != NULL &&
            XtIsWidget(children[i]) &&
            children[i] == UIobjIdToWidget(UIwidgetToObjId(children[i])))
        {
            UIclass = UIwidgetToClass(children[i]);

            if (UIclass != NULL) {
                UIclass->addLayoutEditorEventHandlers(children[i]);

                if (children[i] == UIobjIdToWidget(UIwidgetToObjId(children[i])))
                {
                    OMpush_ctx(UIclass->ObjId, OM_STATE_USR,
                               OM_CTX_TOSS_NOTIFIES, 0);

                    SetIntVal(UIclass->ObjId, "groupToParent",0);
                    OMpop_ctx(UIclass->ObjId);
                }
            }
        }
    }
}

void
UIlayout::removeDirectEventHandlers(Widget hWnd)
{
    WidgetList  children;
    Cardinal    numChildren;
    int         i;

    if (XtIsWidget(hWnd)) {
        // Remove child widget button press event handlers
        XtRemoveEventHandler(hWnd,XtAllEvents,True,
                             (XtEventHandler) UIlayout::keyPressCB,NULL);

        XtRemoveEventHandler(hWnd,XtAllEvents,True,
                             (XtEventHandler) UIlayout::buttonPressCB,
                             NULL);

        XtRemoveEventHandler(hWnd,XtAllEvents,True,
                             (XtEventHandler) UIlayout::buttonReleaseCB,NULL);

        XtRemoveEventHandler(hWnd,XtAllEvents,True,
                             (XtEventHandler) UIlayout::buttonMotionCB,
                             NULL);

        if (XtIsComposite(hWnd) ||
            XtIsSubclass(hWnd,xmFrameWidgetClass) ||
            XtIsSubclass(hWnd,xmScaleWidgetClass)) {

            XtVaGetValues(hWnd,
                          XmNchildren,    &children,
                          XmNnumChildren, &numChildren,
                          NULL);

            for (i=0; i < numChildren; i++) {
                UIlayout::removeDirectEventHandlers(children[i]);
                UIlayout::removeIndirectEventHandlers(children[i], hWnd);
            }
        }
    }
}

void
UIlayout::keyPressCB(Widget, XtPointer, XKeyEvent *event)
{
    char       buffer;
    KeySym     keysym;
    UIwindow   *Win;

    if (!UIlayout::editEnable) return;

    if (UIlayout::state == LAYOUT_STATE_NORMAL) return;

    XLookupString(event,&buffer,1,&keysym,NULL);

    switch(keysym) {
        case 65288:  // backspace
        case 65535:  // delete
        {
            UIlayout::drawLayoutObject(moveWidget, root, layout_gc,
                                       last_motion_x,
                                       last_motion_y,
                                       last_motion_width,
                                       last_motion_height,
                                       LAYOUT_OBJECT_MOVE,
                                       LAYOUT_OBJECT_OFF);

            UIlayout::state = LAYOUT_STATE_NORMAL;

            XtVaGetValues(moveWidget,XmNuserData,&Win,NULL);

            OMdestroy_obj(Win->ObjId,0);
        }
        break;
    }
}

///////////////////////////////////////////////////////////////////////////////

void
UIlayout::buttonPressCB(Widget w, XtPointer, XButtonEvent *event,
                        Boolean *contEvent)
{
    // First action is button 2 press and return
    // This sets the UIlayout state to EDIT on this widget.
    if( !editEnable ) {
        switch (state) {
            case LAYOUT_STATE_NORMAL:
               break;
            case LAYOUT_STATE_PENDING_NORMAL:
            case LAYOUT_STATE_MOVE:
            case LAYOUT_STATE_RESIZE:
                cleanUpLayoutState();
                break;
        }
        return;
    }

    if( event->button == Button2 ) {
        switch (state ) {
            case LAYOUT_STATE_NORMAL:
            {
                moveWidget = w;
                destParent = XtParent(w);

                // Block other event handlers
                *contEvent = False;

                setState(LAYOUT_STATE_NORMAL);
            }
            break;

            case LAYOUT_STATE_PENDING_NORMAL:
            {
                if( moveWidget == w ) { // Inside of current move object
                                        // if cursor over resize handle
                    if( potentialActionState == LAYOUT_OBJECT_RESIZE ) {
                        Position x, y;
                        Dimension width, height;
                        UIgeom::getWidgetPosSize(w, x, y, width, height);

                        changeX = (event->x <= LAYOUT_HANDLE_SIZE) ? True : False;
                        changeY = (event->y <= LAYOUT_HANDLE_SIZE) ? True : False;

                        resizeWidth = (event->x >= width - LAYOUT_HANDLE_SIZE ||
                                       event->x <= LAYOUT_HANDLE_SIZE ) ? True : False;

                        resizeHeight = (event->y >= height - LAYOUT_HANDLE_SIZE ||
                                        event->y <= LAYOUT_HANDLE_SIZE) ? True : False;

                        specialCase = 0;

                        // Special case for UIsliders
                        unsigned char orientation;
                        if (XtIsSubclass(w,xmScaleWidgetClass)) {
                            XtVaGetValues(w,XmNorientation,&orientation,NULL);

                            if (orientation == XmHORIZONTAL) {
                                if (resizeHeight)
                                    specialCase = 1;
                                resizeHeight = 0;

                                if (changeY)
                                    specialCase = 1;
                                changeY = 0;
                            }
                            else {
                                if (resizeWidth)
                                    specialCase = 1;
                                resizeWidth = 0;

                                if (changeX)
                                    specialCase = 1;
                                changeX = 0;
                            }
                        }
                        // Special case for UIoptionMenu
                        UIwindow *winClass = UIwidgetToClass( moveWidget );
                        if( winClass && (winClass->classType() == "UIoptionMenu") ) {
                            resizeHeight = resizeWidth = changeX = changeY = 0;
                            specialCase = 1;
                        }
                        select_x = event->x;
                        select_y = event->y;
                        draw_last = False;

                        // Block other event handlers
                        *contEvent = False;

                        setState(LAYOUT_STATE_RESIZE);
                    }
                    else if( potentialActionState == LAYOUT_OBJECT_MOVE ) {
                        select_x = event->x;
                        select_y = event->y;
                        draw_last = False;

                        // Block other event handlers
                        *contEvent = False;

                        setState(LAYOUT_STATE_MOVE);
                    }
                }
                else { // If outside, cleanup and return to normal state
                    cleanUpLayoutState();
                }
            }
            break;

            case LAYOUT_STATE_MOVE:
            case LAYOUT_STATE_RESIZE:
                break;
        }
    }
}

///////////////////////////////////////////////////////////////////////////
//
void
UIlayout::cleanUpLayoutState()
{
    if( moveWidget ) {
        XtRemoveEventHandler(moveWidget,
                             ExposureMask | SubstructureNotifyMask | StructureNotifyMask,
                             False,
                             (XtEventHandler)UIlayout::overlayWindowExposeHandler,
                             NULL);

        // Clear any set cursors
        XUndefineCursor(XtDisplay(moveWidget),XtWindow(moveWidget));

        // Attempt to restore UIcursor
        UIwindow *win = UIwidgetToClass( moveWidget );
        // Get the "cursor" Id
        OMobj_id cursorId = OMfind_subobj(win->ObjId, OMstr_to_name("cursor"),
                                          OM_OBJ_RW);
        // Don't do anything if "attach_cursor" is the activating object
        OMobj_id cursorOLId;
        if( OMget_obj_val(cursorId, &cursorOLId ) == OM_STAT_SUCCESS ) {
            // Overloaded cursor object
            UIcursor *cursor =
                (UIcursor *)((OMXgroup *)OMret_omx_ptr(cursorOLId, 0))->ret_omx_ptr("UIcursor");
            if(cursor) cursor->update(0, -1); //Force update on cursor
        }

        if (layout_gc) // Free layout GC
            XFreeGC(XtDisplay(moveWidget),layout_gc);
        XtUnmanageChild( moveWidget );
        XtManageChild( moveWidget );
    }
    // Reinitialize the static vars
    layout_gc = NULL;
    moveWidget = 0;
    potentialActionState = LAYOUT_OBJECT_MOVE;
    last_motion_width = last_motion_height = 0;
    last_motion_x = last_motion_y = select_x = select_y = 0;
    setState(LAYOUT_STATE_NORMAL);
}


///////////////////////////////////////////////////////////////////////////////
//
void
UIlayout::buttonReleaseCB(Widget w, XtPointer, XButtonEvent *event,
                          Boolean *contEvent)
{
    if( event->button == Button2 ) {
        switch (state ) {
            case LAYOUT_STATE_NORMAL:
            {
                if( moveWidget == w ) {
                    // Get the UI object and check its local edit mode flag
                    UIwindow *win = UIwidgetToClass( moveWidget );
                    int edit_val;
                    OMobj_id layoutId = FindSubObj( win->ObjId, "layout" );
                    if( (GetIntVal( layoutId, "edit", &edit_val) == OM_STAT_SUCCESS) &&
                        (edit_val == 1) ) {

                        // set path to ObjectEditor edited object;
                        OMobj_id editObjPathId =
                            OMfind_str_subobj( OMroot_obj,
                                               EDIT_OBJ_PATH_STRING.c_str(),
                                               OM_OBJ_RW);
                        if( !OMis_null_obj(editObjPathId) ) {
                            char buf[BUFSIZE];
//                            cout << OMret_obj_path(win->ObjId, buf, BUFSIZE ) << endl;
                            OMret_obj_path_to(win->ObjId, OMroot_obj,
                                              buf, BUFSIZE, 0);
                            OMset_str_val(editObjPathId, buf);
                        }

                        // set path in ObjectEditor displayed object;
                        OMobj_id dispObjPathId =
                            OMfind_str_subobj( OMroot_obj,
                                               DISP_OBJ_PATH_STRING.c_str(),
                                               OM_OBJ_RW);
                        if( !OMis_null_obj(dispObjPathId) ) {
                            char buf[BUFSIZE];
//                            cout << OMret_obj_path(win->ObjId, buf, BUFSIZE ) << endl;
                            OMret_obj_path_to(win->ObjId, OMroot_obj,
                                              buf, BUFSIZE, 0);
                            OMset_str_val(dispObjPathId, buf);
                        }

                        // Add other layout editor handlers here !
                        XtInsertEventHandler(moveWidget,
                                             ExposureMask | SubstructureNotifyMask | StructureNotifyMask, False,
                                             (XtEventHandler)UIlayout::overlayWindowExposeHandler,NULL, XtListTail);

                        // Initialize the the drawing components
                        Window parent, *children;
                        unsigned int nchildren;
                        XQueryTree(XtDisplay(w), XtWindow(w), &root,
                                   &parent, &children, &nchildren);
                        XtFree((char *)children);
                        Pixel bg, fg;
                        XtVaGetValues (w, XmNbackground, &bg,
                                          XmNforeground, &fg, NULL);

                        // Create layout GC
                        XtGCMask value_mask;
                        value_mask = GCFunction | GCSubwindowMode | GCLineWidth | GCForeground;
                        XGCValues values;
                        values.function = GXxor;
                        values.subwindow_mode = IncludeInferiors;
                        values.line_width = 1;
                        values.foreground = fg ^ bg;
                        layout_gc = XCreateGC(XtDisplay(w), root,
                                              value_mask, &values);

                        // Create GC for refreshing button face on Expose events.
                        value_mask = GCFunction | GCSubwindowMode | GCLineWidth | GCForeground;
                        values.function = GXcopy;
                        values.foreground = fg;
                        expose_gc = XCreateGC(XtDisplay(w), root,
                                              value_mask, &values);

                        UIgeom::getWidgetSize(w, last_motion_width, last_motion_height);

                        last_motion_x = event->x_root - event->x;
                        last_motion_y = event->y_root - event->y;

                        XRaiseWindow(XtDisplay(w),XtWindow(w));

                        // Draw the layout editor frame and handles
                        UIlayout::drawLayoutObject(
                            w, XtWindow(moveWidget), layout_gc,
                            0, 0,
                            last_motion_width, last_motion_height,
                            LAYOUT_OBJECT_RESIZE, LAYOUT_OBJECT_ON);

                        // Set the initial cursor type
                        XDefineCursor( XtDisplay(moveWidget),
                                       XtWindow(moveWidget), moveCursor );

                        // Block other event handlers
                        *contEvent = False;

                        setState(LAYOUT_STATE_PENDING_NORMAL);
                    }
                    else {
                        setState(LAYOUT_STATE_NORMAL);
                        moveWidget = 0;
                    }
                }
            }
            break;

            case LAYOUT_STATE_PENDING_NORMAL:
            {
                if( moveWidget != w ) { // Reset things to normal
                    XtRemoveEventHandler(
                        moveWidget,
                        ExposureMask | SubstructureNotifyMask | StructureNotifyMask, False,
                        (XtEventHandler)UIlayout::overlayWindowExposeHandler,
                        NULL);

                    if( layout_gc )
                        XFreeGC(XtDisplay(moveWidget),layout_gc);
                    layout_gc = NULL;

                    // Remove the move cursor
                    XUndefineCursor(XtDisplay(moveWidget),
                                    XtWindow(moveWidget));

                    // Attempt to restore UIcursor
                    UIwindow *win = UIwidgetToClass( moveWidget );
                    // Get the "cursor" Id
                    OMobj_id cursorId = OMfind_subobj(win->ObjId,
                                                      OMstr_to_name("cursor"),
                                                      OM_OBJ_RW);
                    // Don't do anything if "attach_cursor" is the activating object
                    OMobj_id cursorOLId;
                    if( OMget_obj_val(cursorId, &cursorOLId ) == OM_STAT_SUCCESS ) { // Overloaded cursor object
                        UIcursor *cursor =
                            (UIcursor *)((OMXgroup *)OMret_omx_ptr(cursorOLId, 0))->ret_omx_ptr("UIcursor");
                        if(cursor) cursor->update(0, -1); //Force update on cursor
                    }

                    XtUnmanageChild(moveWidget);
                    XtManageChild(moveWidget);

                    // Reinitialize the static vars
                    last_motion_width = last_motion_height = 0;
                    last_motion_x = last_motion_y = select_x = select_y = 0;

                    // Block other event handlers
                    *contEvent = False;

                    setState(LAYOUT_STATE_NORMAL);
                }
            }
            break;

            case LAYOUT_STATE_MOVE:
            {
                // Reposition object if changed position with the layout editor
                // Undraw layout frame
                if( draw_last ) {
                    UIlayout::drawLayoutObject(
                        moveWidget, root, layout_gc,
                        last_motion_x, last_motion_y,
                        last_motion_width, last_motion_height,
                        LAYOUT_OBJECT_MOVE, LAYOUT_OBJECT_OFF);
                }

                // Get the curr position and size of the move widget and parent
                Position x, y;
                Dimension width, height, parent_width, parent_height;
                UIgeom::getWidgetPosSize(moveWidget, x, y, width, height);
                UIgeom::getWidgetSize(XtParent(moveWidget),
                                      parent_width, parent_height);

                // Translate the coordinates from the root
                // to the underlying parent in order to sync up positions
                int X, Y;
                Window returnedWindow;
                XTranslateCoordinates(XtDisplay(w),
                                      root, XtWindow(XtParent(moveWidget)),
                                      last_motion_x, last_motion_y,
                                      &X, &Y,
                                      &returnedWindow);

                XtWidgetGeometry   request;
                request.request_mode = CWX | CWY;
                request.x = X;
                request.y = Y;

                if (request.x >= 0 && request.y >= 0 &&
                    request.x + width <= parent_width &&
                    request.y + height <= parent_height &&
                    (request.x != x || request.y != y)) {

                    request.x = UIlayout::gridMapX(request.x,0);
                    request.y = UIlayout::gridMapY(request.y,0);

                    if( UIwidgetToClass(moveWidget)->classType() != "UIshell" ) {
                        UIupdateOMRequestedWindowPos(moveWidget,
                                                     request.x,request.y,
                                                     OM_CTX_ALL_NOTIFIES);
                    }
                }

                // Need this to set up for possible reparenting after move
                moveReleaseTime = event->time;

//                XtUnmanageChild(moveWidget);
//                XtManageChild(moveWidget);
                // Block other event handlers
                *contEvent = False;

                setState(LAYOUT_STATE_PENDING_NORMAL);
            }
            break;

            case LAYOUT_STATE_RESIZE:
            {
                UIlayout::drawLayoutObject(
                    moveWidget, root, layout_gc,
                    0, 0,
                    last_motion_width, last_motion_height,
                    LAYOUT_OBJECT_RESIZE, LAYOUT_OBJECT_OFF);

                Position x, y;
                Dimension width, height;
                UIgeom::getWidgetPosSize(moveWidget, x, y, width, height);
                XtWidgetGeometry   request;
                Window returnedWindow;
                request.request_mode = CWX | CWY | CWWidth | CWHeight;

                // Translate coordinates to parent to get 1-to-1 correspondence
                // with layout object
                int retX, retY;
                XTranslateCoordinates(XtDisplay(w),
                                      root, XtWindow(XtParent(moveWidget)),
                                      last_motion_x, last_motion_y,
                                      &retX, &retY,
                                      &returnedWindow);

                Dimension Width, Height;
                Width  = request.width  = last_motion_width;
                Height = request.height = last_motion_height;

                if( UIwidgetToClass(moveWidget)->classType() != "UIshell" ) {
                    if (retX != x || retY != y ||
                        request.width != width || request.height != height) {
                        UIupdateOMRequestedWindowPosSize(moveWidget,
                                                         retX, retY,
                                                         Width-1, Height-1,
                                                         OM_CTX_ALL_NOTIFIES);
                    }
                }

                // Block other event handlers
                *contEvent = False;

                setState(LAYOUT_STATE_PENDING_NORMAL);
            }
            break;
        }
    }
}


///////////////////////////////////////////////////////////////////////////////
//
void
UIlayout::pointerMotionCB(Widget w, XtPointer, XMotionEvent *event)
{
    /*
     * This call works in conjunction with PointerMotionHintMask.
     * We get the position of the cursor and the mask at the current time.
     * Since these values are more recent than the ones in the event, we
     * update the event with these values.
     *
     * Only if the pointer moves after this call will we get another motion
     * event.
     */
    Window dum_w;
    int x, y, dummy;
    unsigned int mask;
    XQueryPointer(XtDisplay(w), XtWindow(w), &dum_w, &dum_w,
                  &dummy, &dummy, &x, &y, &mask);
    event->x = x;
    event->y = y;
    event->state = mask;

    // Checks for pointer crossing into the resize handles,
    // reset cursor to reflect appropriate action possiblities
    // Don't do anything if we're in the middle of a move action
    if( !moveWidget ) return;
    switch (state ) {
        case LAYOUT_STATE_NORMAL:
            break;

        case LAYOUT_STATE_PENDING_NORMAL:
        case LAYOUT_STATE_MOVE:
        case LAYOUT_STATE_RESIZE:
        {
            // Determine the postion of the pointer relative to the resize handles
            Position x,y;
            Dimension width, height;
            Boolean _resizeWidth, _resizeHeight;
            Boolean _changeX, _changeY;

            _resizeWidth = _resizeHeight = _changeX = _changeY = False; // Init

            UIgeom::getWidgetPosSize(moveWidget, x, y, width, height);

            if(event->x <= LAYOUT_HANDLE_SIZE) _changeX = True;
            if(event->y <= LAYOUT_HANDLE_SIZE) _changeY = True;

            if (event->x >= width - LAYOUT_HANDLE_SIZE ||
                event->x <= LAYOUT_HANDLE_SIZE )
                _resizeWidth = True;

            if (event->y >= height - LAYOUT_HANDLE_SIZE ||
                event->y <= LAYOUT_HANDLE_SIZE)
                _resizeHeight = True;

            // We only want to change the cursor on a change of boundary,
            // not each time the event handler is called.
            if( (w == moveWidget ) &&
                (state == LAYOUT_STATE_PENDING_NORMAL) ) {

//cout << "_resizeWidth= " << (char *)(_resizeWidth ? "True" : "False") ;
//cout << " _resizeHeight= " << (char *)(_resizeHeight ? "True" : "False");
//cout << " _changeX= " << (char *)(_changeX ? "True" : "False");
//cout << " _changeY= " << (char *)(_changeY ? "True" : "False") << endl;

                if ( _resizeWidth || _resizeHeight ) {
                    if( _resizeWidth && !_resizeHeight ) {
                        // Set horizontal cursors
                        if( _changeX )
                            XDefineCursor(XtDisplay(moveWidget),
                                          XtWindow(moveWidget), resizeLCursor);
                        else
                            XDefineCursor(XtDisplay(moveWidget),
                                          XtWindow(moveWidget), resizeRCursor);
                    }
                    else if( _resizeHeight && !_resizeWidth ) {
                        // Set vertical cursors
                        if( _changeY )
                            XDefineCursor(XtDisplay(moveWidget),
                                          XtWindow(moveWidget), resizeTCursor);
                        else
                            XDefineCursor(XtDisplay(moveWidget),
                                          XtWindow(moveWidget), resizeBCursor);
                    }
                    else if( _resizeHeight && _resizeWidth &&
                             !_changeX && !_changeY ) {
                        // Set bottom right cursor
                        XDefineCursor(XtDisplay(moveWidget),
                                      XtWindow(moveWidget), resizeBRCursor);
                    }
                    else if( _resizeHeight && _changeY ) {
                        // Set top right cursor
                        XDefineCursor(XtDisplay(moveWidget),
                                      XtWindow(moveWidget), resizeTRCursor);
                    }
                    else if( _changeX && _resizeWidth ) {
                        // Set bottom left cursor
                        XDefineCursor(XtDisplay(moveWidget),
                                      XtWindow(moveWidget), resizeBLCursor);
                    }
                    potentialActionState = LAYOUT_OBJECT_RESIZE;
                }
                if( _changeX && _changeY ) {
                    // Set top left cursor
                    XDefineCursor(XtDisplay(moveWidget),
                                  XtWindow(moveWidget), resizeTLCursor);
                    potentialActionState = LAYOUT_OBJECT_RESIZE;
                }
                else if( !_resizeWidth && !_resizeHeight ) {
                    // Set move cursor
                    XDefineCursor(XtDisplay(moveWidget),
                                  XtWindow(moveWidget), moveCursor);
                    potentialActionState = LAYOUT_OBJECT_MOVE;
                }
            }
        }
        break;
    }
}


///////////////////////////////////////////////////////////////////////////////
//
void
UIlayout::buttonMotionCB(Widget w, XtPointer, XMotionEvent *event)
{
    /*
     * This call works in conjunction with PointerMotionHintMask.
     * We get the position of the cursor and the mask at the current time.
     * Since these values are more recent than the ones in the event, we
     * update the event with these values.
     *
     * Only if the pointer moves after this call will we get another motion
     * event.
     */
    Window dum_w;
    int x, y, dummy;
    unsigned int mask;
    XQueryPointer(XtDisplay(w), XtWindow(w), &dum_w, &dum_w,
                  &dummy, &dummy, &x, &y, &mask);
    event->x = x;
    event->y = y;
    event->state = mask;

    if( event->state == Button2Mask ) {
        switch (state ) {
            case LAYOUT_STATE_NORMAL:
                break;
            case LAYOUT_STATE_PENDING_NORMAL:
                break;

            case LAYOUT_STATE_MOVE:
            {        // Calculate new move
                Dimension width, height;
                Position x_floor = findParentZeroX(moveWidget, destParent);
                Position y_floor = findParentZeroY(moveWidget, destParent);
                Position curr_motion_x = gridMapX(event->x_root - select_x,
                                                  x_floor);
                Position curr_motion_y = gridMapY(event->y_root - select_y,
                                                  y_floor);
                UIgeom::getWidgetSize(moveWidget, width, height);
                // Undraw editor frame
                // flag to prevent spurious redraw after move completed
                if( draw_last ) {
                    drawLayoutObject(moveWidget, root, layout_gc,
                                     last_motion_x,last_motion_y,
                                     last_motion_width,last_motion_height,
                                     LAYOUT_OBJECT_MOVE, LAYOUT_OBJECT_OFF);
                }
                // Redraw editor frame in new position
                drawLayoutObject(moveWidget, root, layout_gc,
                                 curr_motion_x, curr_motion_y,
                                 width, height,
                                 LAYOUT_OBJECT_MOVE, LAYOUT_OBJECT_ON);

                // Remember position for next time
                last_motion_x = curr_motion_x;
                last_motion_y = curr_motion_y;
                last_motion_width = width;
                last_motion_height = height;
                draw_last = True;
            }
            break;

            case LAYOUT_STATE_RESIZE:
            {

#ifdef DEBUG_LAYOUT
cout << "UIlayout::buttonMotionCB: widget= " << XtName(w) << endl;
cout << "\tevent->type= " << event->type << endl;
cout << "\tevent->window= " << event->window << endl;
cout << "\tevent->subwindow= " << event->subwindow << endl;
cout << "\tevent->x= " << event->x << endl;
cout << "\tevent->y= " << event->y << endl;
#endif

                if( specialCase == 1 ) return;

                Dimension width, height;
                Position x, y;
                UIgeom::getWidgetPosSize(moveWidget, x, y, width, height);

                Position new_x, new_y;
                Dimension new_width, new_height;

                if (changeX)
                    new_x = event->x_root - select_x;
                else
                    new_x = event->x_root - event->x;

                if (changeY)
                    new_y = event->y_root - select_y;
                else
                    new_y = event->y_root - event->y;

                if (resizeWidth) {
                    if (changeX)
                        new_width = width + select_x - event->x;
                    else
                        new_width = width + event->x - select_x;
                }
                else
                    new_width = width;

                if (resizeHeight) {
                    if (changeY)
                        new_height = height + select_y - event->y;
                    else
                        new_height = height + event->y - select_y;
                }
                else
                    new_height = height;

#ifdef DEBUG_LAYOUT
cout << "\tnew_x= " << new_x << " new_y= " << new_y << endl;
cout << "\tnew_height= " << new_height << " new_width= " << new_width << endl;
#endif

                 if (new_height > 0 && new_width > 0) {
                     // Don't let widget resize below parent 0,0

                     new_x = UIlayout::gridMapX(new_x,0 /*x_floor*/);
                     new_y = UIlayout::gridMapY(new_y,0 /*y_floor*/);

                     // This is an odd one: need to subtract 1 from width and
                     // height, because gridding makes size increase before
                     // position in certain cases.
                     new_width  = UIlayout::gridMapX(new_width-1,1);
                     new_height = UIlayout::gridMapY(new_height-1,1);

                     // Catch any errors in negative widths or heights here.
                     // Value gets real big when crossing over...
                     if (new_width > 4  && new_width < 2048 &&
                         new_height > 4 && new_height < 2048) {
                         if( draw_last ) {
                             UIlayout::drawLayoutObject(
                                 moveWidget, root, layout_gc,
                                 last_motion_x, last_motion_y,
                                 last_motion_width, last_motion_height,
                                 LAYOUT_OBJECT_RESIZE, LAYOUT_OBJECT_OFF);
                         }
#ifdef DEBUG_LAYOUT
cout << "\tdraw: new_x= " << new_x << " new_y= " << new_y << endl;
cout << "\tdraw: new_height= " << new_height << " new_width= " << new_width << endl;
cout << endl;
#endif
                         UIlayout::drawLayoutObject(
                             moveWidget, root, layout_gc,
                             new_x, new_y,
                             new_width, new_height,
                             LAYOUT_OBJECT_RESIZE, LAYOUT_OBJECT_ON);

                         last_motion_x = new_x;
                         last_motion_y = new_y;
                         last_motion_width = new_width;
                         last_motion_height = new_height;
                         draw_last = True;
                     }
                 }
                 break;
            }
        }
    }
}


//////////////////////////////////////////////////////////////////////
//
void
UIlayout::cmdListEnterNotifyCB(Widget, Widget hWnd, XCrossingEvent *event)
{
    OMobj_id           SubId;
    UIwindow           *Parent;
    UIwindow           *Win;
    XtWidgetGeometry   request;
    OMobj_id           OptionObjId;
    String             Str;

    if (!UIlayout::editEnable) return;

    destParent = hWnd;

    if (moveWidget == 0) return;

    if (event->time - moveReleaseTime < 50) {
        if (checkForReparent(hWnd, &request, &Parent, &Win,
                             event->x_root, event->y_root,
                             event->x, event->y)) {

            SubId = OMfind_subobj(Win->ObjId,OMstr_to_name("label"),OM_OBJ_RD);
            if (OMget_str_val(SubId, &Str,0) == OM_STAT_SUCCESS) {

                if (strcmp(Str,"UIoption") == 0) {

                    OptionObjId =  UIcreate_obj_from_str("UIoption",
                                                         "UIoption",
                                                         NULL);

                    SubId = OMfind_subobj(Parent->ObjId,
                                          OMstr_to_name("cmdList"),
                                          OM_OBJ_RW);

                    if (OMis_null_obj(SubId))
                        fprintf( stderr, "[UIlayout] Error searching for cmdList.\n" );

                    if (OMset_array_val(SubId, OM_ARRAY_APPEND,
                                        OptionObjId) != OM_STAT_SUCCESS)
                        fprintf( stderr, "[UIlayout] Error appending to array.\n" );

                    OMpush_ctx(OptionObjId,
                               OM_STATE_PROG,
                               OM_CTX_ALL_NOTIFIES,
                               0);

                    OMclose_obj(OptionObjId);
                    OMpop_ctx(OptionObjId);

                    moveWidget = 0;
                }
                else {
                    if (UIisOptionBox(Win->ObjId)) {
                        UIerrorHandler(XmDIALOG_ERROR, Win->ObjId, "A UIoptionBox may only accept children of type UIoption.");
                    }
                    else {
                        UIerrorHandler(XmDIALOG_ERROR, Win->ObjId, "A UIradioBox may only accept children of type UIoption.");
                    }
                }
            }
            else {
                UIupdateOMRequestedWindowPos(moveWidget,
                                             UIlayout::gridMapX(request.x,0),
                                             UIlayout::gridMapY(request.y,0),
                                             OM_CTX_ALL_NOTIFIES);

                SubId = OMfind_subobj(Win->ObjId, OMstr_to_name("parent"),
                                      OM_OBJ_RW);
                OMset_obj_ref(SubId, Parent->ObjId, 0);
                moveWidget = 0;
            }
        }
    }
}

///////////////////////////////////////////////////////////////////////////////
//
void
UIlayout::enterNotifyCB(Widget w, Widget reg, XCrossingEvent *event)
{
    // This event handler is registered on all container widgets
    // when a enter event occurs the handler should check the
    // state and

    OMobj_id           SubId;
    UIwindow           *Win,*Parent;
    XtWidgetGeometry   request;

    if (!UIlayout::editEnable) return;

    // Don't do anything if called on itself
    if( !moveWidget || (moveWidget == reg) ) return;

    // match the window we're over with the event->window
    Window child_win, root_win;
    int root_x, root_y, win_x, win_y;
    unsigned int keys;
    if( XQueryPointer(event->display, event->window,
                      &root_win, &child_win, &root_x, &root_y,
                      &win_x, &win_y, &keys )) {
        // Pointer on same screen, proceed
        if( child_win ) { // Ok have valid window
            if( child_win != event->window ) {
                return;
            }
        }
    }

    XtVaGetValues(w, XmNuserData, &Win, NULL);

    if (event->time - moveReleaseTime < 50) {

        if (checkForReparent(w, &request, &Parent, &Win,
                             event->x_root, event->y_root,
                             event->x, event->y)) {

            Position xpos, ypos;
            UIupdateOMRequestedWindowPos(moveWidget,
                                        xpos = UIlayout::gridMapX(request.x,0),
                                        ypos = UIlayout::gridMapY(request.y,0),
                                        OM_CTX_TOSS_NOTIFIES);

            SubId = OMfind_subobj(Win->ObjId, OMstr_to_name("parent"),
                                  OM_OBJ_RW);

            if( Parent ) {
                // This actually kicks off the reparenting!
                OMset_obj_ref(SubId, Parent->ObjId, 0);
            }
            moveWidget = 0;
        }
    }
}


///////////////////////////////////////////////////////////////////////////////
// Checks to see if the move of the widget requires a reparenting to
// another widget. Returns 1 if required and sets the address of the
// Win and Parent UI classes in the arg list.
//
int
UIlayout::checkForReparent(Widget newParent,
                 XtWidgetGeometry *request,
                 UIwindow **Parent,
                 UIwindow **Win,
                 int, int, int, int)
{
    if (!UIlayout::editEnable) return 0;

    if (moveWidget == 0) return(0);

    destParent = newParent;

    if (destParent != moveWidget &&
        destParent != XtParent(moveWidget) &&
        XtParent(destParent) != moveWidget &&
        !UIisAncestorOf(destParent, moveWidget)) {

        // Set the Win and Parent pointer arguments to the address of
        // the respective UI classes
        XtVaGetValues(moveWidget,XmNuserData, Win,  NULL);
        XtVaGetValues(newParent, XmNuserData, Parent,NULL);

        // need to translate from root to parent widget coords.
        Window returnedWindow;
        int retx, rety;
        XTranslateCoordinates(XtDisplay(newParent),
                              root,XtWindow(newParent),
                              last_motion_x,last_motion_y,
                              &retx, &rety,
                              &returnedWindow);

        request->request_mode = CWX | CWY;
        request->x = (Position)retx;
        request->y = (Position)rety;

        return(1);
    }

    else return(0);
}

///////////////////////////////////////////////////////////////////////////////
// Adjust the desired x position to snap to the nearest grid point
//
Position
UIlayout::gridMapX(Position in, Position floor)
{
    // Adjust the desired x position to snap to the nearest grid point
    Position out = in;

    if (snapToGrid)
        out = ((Position)( (in-floor+(gridX/2)) / gridX * gridX )+floor);

    // If a resize, bound value by 0,0 of window (in root coords.)
    if (out < floor && UIlayout::state == LAYOUT_STATE_RESIZE &&
        !UIisWorkAreaChild(XtParent(moveWidget)))
        out = floor;

    return out;
}

///////////////////////////////////////////////////////////////////////////////
// Adjust the desired y position to snap to the nearest grid point
//
Position
UIlayout::gridMapY(Position in, Position floor)
{
    // Adjust the desired y position to snap to the nearest grid point
    Position out = in;

    if (snapToGrid)
        out = ((Position)( (in-floor+(gridY/2)) / gridY * gridY )+floor);

    // If a resize, bound value by 0,0 of window (in root coords.)
    if (out < floor && UIlayout::state == LAYOUT_STATE_RESIZE &&
        !UIisWorkAreaChild(XtParent(moveWidget)))
        out = floor;

    return out;
}

///////////////////////////////////////////////////////////////////////////////
//
// DrawLayoutObject() draws the object that is manipulated by
// the user when in Layout Mode.  The object is drawn in XOR
// mode.  To prevent stray lines, the first position is cached.
// The next time the object is drawn, the cached coordinates are
// used to undraw.
//
///////////////////////////////////////////////////////////////////////////////
void
UIlayout::drawLayoutObject(Widget w_in, Window drawto_in, GC gc_in,
                           int x_in, int y_in,
                           int width_in, int height_in,
                           UIlayoutObjectType how_in, UIlayoutObjectState on)
{
    int x, y, width, height, how;
    Window drawto;
    Widget w;
    GC gc;
    static int x_old, y_old, width_old, height_old, how_old;
    static Widget w_old;
    static Window drawto_old;
    static GC gc_old;

    if (UIisBeingDestroyed(w_in)) return;

    x_in = x_in + 2;
    y_in = y_in + 2;
    width_in = width_in - 5;
    height_in = height_in - 5;

    switch(on) {

        case LAYOUT_OBJECT_EXPOSE:
        {
            x = x_in;
            y = y_in;
            width = width_in;
            height = height_in;
            how = how_in;
            drawto = drawto_in;
            w = w_in;
            gc = gc_in;
        }
        break;

        case LAYOUT_OBJECT_ON:
        {
            x = x_old = x_in;
            y = y_old = y_in;
            width = width_old = width_in;
            height = height_old = height_in;
            how = how_old = how_in;
            drawto = drawto_old = drawto_in;
            w = w_old = w_in;
            gc = gc_old = gc_in;
        }
        break;

        case LAYOUT_OBJECT_OFF:
        {
            x = x_old;
            y = y_old;
            width = width_old;
            height = height_old;
            how = how_old;
            drawto = drawto_old;
            w = w_old;
            gc = layout_gc;
        }
        break;
    }

    // Draw the bounding box
    XDrawRectangle(XtDisplay(w), drawto, gc, x, y, width, height);

    // Draw a crosshair
    XDrawLine(XtDisplay(w), drawto, gc,
              x+(width/2)-LAYOUT_HANDLE_SIZE, y+(height/2),
              x+(width/2)+LAYOUT_HANDLE_SIZE, y+(height/2));

    XDrawLine(XtDisplay(w), drawto, gc,
              x+(width/2), y+(height/2)-LAYOUT_HANDLE_SIZE,
              x+(width/2), y+(height/2)+LAYOUT_HANDLE_SIZE);

    if (how == LAYOUT_OBJECT_RESIZE) {
        // Draw the corner handles
        XFillRectangle(XtDisplay(w), drawto, gc,
                       x+1,
                       y+1,
                       LAYOUT_HANDLE_SIZE,
                       LAYOUT_HANDLE_SIZE);
        XFillRectangle(XtDisplay(w), drawto, gc,
                       x+width-LAYOUT_HANDLE_SIZE,
                       y+height-LAYOUT_HANDLE_SIZE,
                       LAYOUT_HANDLE_SIZE,
                       LAYOUT_HANDLE_SIZE);
        XFillRectangle(XtDisplay(w), drawto, gc,
                       x+1,
                       y+height-LAYOUT_HANDLE_SIZE,
                       LAYOUT_HANDLE_SIZE,
                       LAYOUT_HANDLE_SIZE);
        XFillRectangle(XtDisplay(w), drawto, gc,
                       x+width-LAYOUT_HANDLE_SIZE,
                       y+1,
                       LAYOUT_HANDLE_SIZE,
                       LAYOUT_HANDLE_SIZE);

        // Draw the side handles
        XFillRectangle(XtDisplay(w), drawto, gc,
                       x+(width/2)-LAYOUT_HANDLE_SIZE/2,
                       y+1,
                       LAYOUT_HANDLE_SIZE,
                       LAYOUT_HANDLE_SIZE);
        XFillRectangle(XtDisplay(w), drawto, gc,
                       x+(width/2)-LAYOUT_HANDLE_SIZE/2,
                       y+height-LAYOUT_HANDLE_SIZE,
                       LAYOUT_HANDLE_SIZE,
                       LAYOUT_HANDLE_SIZE);

        XFillRectangle(XtDisplay(w), drawto, gc,
                       x+1,
                       y+(height/2)-LAYOUT_HANDLE_SIZE/2,
                       LAYOUT_HANDLE_SIZE,
                       LAYOUT_HANDLE_SIZE);
        XFillRectangle(XtDisplay(w), drawto, gc,
                       x+width-LAYOUT_HANDLE_SIZE,
                       y+(height/2)-LAYOUT_HANDLE_SIZE/2,
                       LAYOUT_HANDLE_SIZE,
                       LAYOUT_HANDLE_SIZE);
    }

}

/////////////////////////////////////////////////////////////////////////
// UIlayoutEditState class methods. NOTE: the class is autogenerate
// via the OMX facility in Express. This implemetation must track
// changes to the UI.v code for this class.
/////////////////////////////////////////////////////////////////////////
int
UIlayoutEditState::init(OMevent_mask event_mask, int seq_num)
{
   // return 1 for success
   return update( event_mask, seq_num);
}
int
UIlayoutEditState::update(OMevent_mask, int seq_num)
{
    // Get platform specific class pointer
    UIlayout *layout = (UIlayout *)ret_class_ptr("UIlayout");

    // Turn layout editing on or off
    layout->editEnable = (int)isActive;

    // If layout editing is turned off, check to see if
    // an outline was left around a selected object
    if (layout->editEnable == 0) {
        if (layout->state == LAYOUT_STATE_PENDING_NORMAL) {
            // Undraw outline
            layout->drawLayoutObject(layout->moveWidget, layout->root,
                                     layout->layout_gc,
                                     layout->last_motion_x,
                                     layout->last_motion_y,
                                     layout->last_motion_width,
                                     layout->last_motion_height,
                                     LAYOUT_OBJECT_RESIZE,
                                     LAYOUT_OBJECT_OFF);

            // Clear any set cursors
            XUndefineCursor(XtDisplay(layout->moveWidget),
                            XtWindow(layout->moveWidget));
            // Attempt to restore UIcursor
            UIwindow *win = UIwidgetToClass( layout->moveWidget );
            // Get the "cursor" Id
            OMobj_id cursorId = OMfind_subobj(win->ObjId,
                                              OMstr_to_name("cursor"),
                                              OM_OBJ_RW);
            // Don't do anything if "attach_cursor" is the activating object
            OMobj_id cursorOLId;
            if( OMget_obj_val(cursorId, &cursorOLId ) == OM_STAT_SUCCESS ) {
                // Overloaded cursor object
                UIcursor *cursor =
                    (UIcursor *)((OMXgroup *)OMret_omx_ptr(cursorOLId, 0))->ret_omx_ptr("UIcursor");
                if(cursor) cursor->update(0, -1); //Force update on cursor
            }

            if (layout->layout_gc) // Free layout GC
                XFreeGC(XtDisplay(layout->moveWidget),layout->layout_gc);
            XtUnmanageChild( layout->moveWidget );
            XtManageChild( layout->moveWidget );

            // Restore state
            layout->state = LAYOUT_STATE_NORMAL;
            layout->layout_gc = NULL;
            layout->moveWidget = 0;
        }
    }

    // Set snapToGrid
    if( snapToGrid.changed(seq_num) )
        layout->snapToGrid = (int)snapToGrid;
    // Group objects
    if( group.changed(seq_num) ) {
        layout->groupObjects();
    }
    // Ungroup objects
    if( ungroup.changed(seq_num) ) {
        layout->ungroupObjects();
    }

    // Update grid values
    if( layout->snapToGrid ) {
        if( xGridSize.changed(seq_num) ) {
            layout->gridX = xGridSize;
        }

        // Update grid values
        if( yGridSize.changed(seq_num) ) {
            layout->gridY = yGridSize;
        }
    }

//    //
//    // Kludgey for now: when entering Layout Mode, check environment
//    // variables for gridding.  This should be set via GUI!
//    //
//    if (snapToGrid) {
//        char *envp;
//        int grid;
//        if ((envp = getenv("XP_LAYOUT_GRID_X")) != NULL) {
//            if (sscanf(envp, "%d", &grid))
//                layout->gridX = grid;
//        }
//        if ((envp = getenv("XP_LAYOUT_GRID_Y")) != NULL) {
//            if (sscanf(envp, "%d", &grid))
//                layout->gridY = grid;
//        }
//    }

    // Set the Object Editor path strings in the UIlayout class.
    EDIT_OBJ_PATH_STRING = (char  *)edit_obj_path;
    DISP_OBJ_PATH_STRING = (char  *)disp_obj_path;

    // return 1 for success
    return(1);
}
