/*
                Copyright (c) 1999 by
                Advanced Visual Systems Inc.
                All Rights Reserved

        This software comprises unpublished confidential information of
        Advanced Visual Systems Inc. and may not be used, copied or made
        available to anyone, except in accordance with the license
        under which it is furnished.

        This file is under Perforce control
        $Id: //depot/express/fcs70/reduct/dataSurfRed.h#1 $
*/

//============================================================================
// dataSurfaceRed.h
// surface decimation class based on Schroeder et al SIGGRAPH92
//============================================================================

#ifndef __dataSurfRed_h
#define __dataSurfRed_h

#include <avs/port.h>

#include "dsrLinks.h"

#define DSR_LARGE_FLOAT 1.0e29
#define DSR_LARGE_INTEGER 2147483647 // 2^31 - 1

#define DSR_MAX_TRIS_PER_NODE 64
#define DSR_DDWEIGHT 0.5

// Special structures for building loops
typedef struct _dsrNode
{
  xp_long id;
  float   x[3];
  int     deRefs; //monitor memory requirements; new only when necessary
  int     newRefs;
} dsrNode, *dsrNodePtr;

typedef struct _dsrTri
{
  xp_long id;
  float   area;
  float   n[3];
  xp_long nodes[3];
} dsrTri, *dsrTriPtr;


// Special classes for manipulating data
class dsrNodeArray
{
public:
  dsrNodeArray() {this->MaxId = -1; this->Array = new dsrNode[DSR_MAX_TRIS_PER_NODE+1];};
  ~dsrNodeArray() {delete [] this->Array;};

  int GetNumberOfNodes() {return this->MaxId + 1;};
  void InsertNextNode(dsrNode& v) {this->MaxId++; this->Array[this->MaxId] = v;};
  dsrNode& GetNode(int i) {return this->Array[i];};
  void Reset() {this->MaxId = -1;};

  dsrNode *Array;  // pointer to data
  int MaxId;             // maximum index inserted thus far
};

class dsrTriArray
{
public:
  dsrTriArray() {this->MaxId = -1; this->Array = new dsrTri[DSR_MAX_TRIS_PER_NODE+1];};
  ~dsrTriArray() {delete [] this->Array;};

  int GetNumberOfTriangles() {return this->MaxId + 1;};
  void InsertNextTriangle(dsrTri& t) {this->MaxId++; this->Array[this->MaxId] = t;};
  dsrTri& GetTriangle(int i) {return this->Array[i];};
  void Reset() {this->MaxId = -1;};

  dsrTri *Array;  // pointer to data
  int MaxId;            // maximum index inserted thus far
};


class dataSurfaceRed
{
public:
  dataSurfaceRed();
  ~dataSurfaceRed();

  // Set the decimation error bounds. Expressed as a fraction of the longest
  // side of the input datas bounding box.
  void SetInitSurfTol (float _arg)
  {
    if (InitSurfTol != _arg)
      InitSurfTol = (_arg<0.0f?0.0f:(_arg>1.0f?1.0f:_arg));
  };

  float GetInitSurfTol () { return InitSurfTol; };


  // Set the value of the increment by which to increase the decimation
  // error after each iteration.
  void SetSurfTolInc (float _arg)
  {
    if (SurfTolInc != _arg)
      SurfTolInc = (_arg<0.0f?0.0f:(_arg>1.0f?1.0f:_arg));
  };

  float GetSurfTolInc () { return SurfTolInc; };

  // Specify the desired reduction in the total number of polygons. Because
  // of various constraints, this level of reduction may not be realizable.
  void SetTargetReduction (float _arg)
  {
    if (TargetReduction != _arg)
      TargetReduction = (_arg<0.0f?0.0f:(_arg>1.0f?1.0f:_arg));
  };

  float GetTargetReduction () { return TargetReduction; };


  // Specify the maximum number of iterations to attempt. If decimation target
  // is reached first, this value will not be reached.
  void SetMaximumIterations (int _arg)
  {
   if (MaximumIterations != _arg)
     MaximumIterations = (_arg<0?0:(_arg>DSR_LARGE_INTEGER?10:_arg));
  };

  int GetMaximumIterations () { return MaximumIterations; };


  // Specify the mesh feature angles.
  void SetInitialEdgeAngle (float _arg)
  {
    if (InitialEdgeAngle != _arg)
      InitialEdgeAngle = (_arg<0.0f?0.0f:(_arg>180.0f?180.0f:_arg));
  };

  float GetInitialEdgeAngle () { return InitialEdgeAngle; };


  // Set/Get the angle by which to increase feature angle over each iteration.
  void SetEdgeAngleIncrement (float _arg)
  {
    if (EdgeAngleIncrement != _arg)
      EdgeAngleIncrement = (_arg<0.0f?0.0f:(_arg>180.0f?180.0f:_arg));
  };

  float GetEdgeAngleIncrement () { return EdgeAngleIncrement; };


  // Set the largest permissible feature angle.
  void SetMaximumEdgeAngle (float _arg)
  {
    if (MaximumEdgeAngle != _arg)
      MaximumEdgeAngle = (_arg<0.0f?0.0f:(_arg>180.0f?180.0f:_arg));
  };

  float GetMaximumEdgeAngle () { return MaximumEdgeAngle; };


  // Turn on/off the preservation of feature edges.
  void SetPreserveEdges (int _arg)
  {
    if (PreserveEdges != _arg)
      PreserveEdges = _arg;
  };

  int GetPreserveEdges () { return PreserveEdges; };

  void SetDataDependence(int _arg)  {
    if(DataDependence != _arg)
      DataDependence = (_arg<0?0:(_arg>2?2:_arg));
  };

  int GetDataDependence() { return DataDependence;}

  void SetDDWeight( float _arg)
  {
    if(DDWeight = _arg)
      DDWeight = (_arg<0.0f?0.0f:(_arg>1.0f?1.0f:_arg));
  };

  float GetDDWeight() { return DDWeight;}

  void DebugOn() { Debug=1; };
  void DebugOff() { Debug=0; };

  // set input mesh
  void SetMesh(xp_long _nnodes, float *_nodes, float *_bounds, xp_long _ntris, xp_long *_tris)
  {
    this->nnodes_in=_nnodes;
    this->nodes_in=_nodes;
    this->bounds=_bounds;
    this->ntris_in=_ntris;
    this->tris_in=_tris;
  };

  // retrieve output mesh
  void GetMesh(xp_long &_nnodes, float *&_nodes, xp_long &_ntris, xp_long *&_tris)
  {
    _nnodes=this->nnodes_out;
    _nodes=this->nodes_out;
    _ntris=this->ntris_out;
    _tris=this->tris_out;
  };

  // set input data
  void SetDDData(float *_data, int _vecLen)
  {
    if(_data && _vecLen)
    {
      this->data_in=_data;
      this->vecLen=_vecLen;
    }
    else {
      this->data_in=NULL;
      this->vecLen=0; // vecLen=0 means no data present
    }
  };

  // retrieve node mapping
  void GetNodeMapping(xp_long *&node_mapping)
  {
    node_mapping=this->node_mapping;
  };

  // run the surface reduction method
  void Execute();

private:
  // inputs
  xp_long nnodes_in;
  float *nodes_in;
  float *bounds;
  xp_long ntris_in;
  xp_long *tris_in;
  float *data_in;
  int vecLen;

  // outputs
  xp_long nnodes_out;
  float *nodes_out;
  xp_long ntris_out;
  xp_long *tris_out;
  xp_long *node_mapping;

  // parameters
  float InitialEdgeAngle; // dihedral angle constraint
  float EdgeAngleIncrement;
  float MaximumEdgeAngle;
  int PreserveEdges; // do/dont worry about feature edges
  int DataDependence;
  float DDWeight;
  float InitSurfTol; // decimation error in fraction of bounding box
  float SurfTolInc; // each iteration will bump error this amount
  float TargetReduction; //target reduction of mesh (fraction)
  int MaximumIterations; // maximum number of passes over data

  // internal
  float Pt[3], Normal[3]; // least squares plane point & normal
  float Angle, Distance; // current feature angle and distance
  float CosAngle; // Cosine of dihedral angle

  float Tolerance; // Intersection tolerance
  int ContinueTriangulating; // Stops recursive tri. if necessary

  float *X; //coordinates of current point
  float *NodeError, Error, LEFactor, MinEdgeError; // support error computation
  unsigned char Debug;

  xp_long *Tris; // copy of tris_in that will be trashed
  dsrLinks *Links; // links between pts and tris
  dsrNodeArray *LoopNodes; // loop of nodes around point
  dsrTriArray *LoopTris; // loop of triangles around point
  //  dsrNodeArray NodeArray;
  //dsrTriArray TriangleArray;

  // member functions
  int BuildLoop(xp_long ptId, int nTris, xp_long *tris);
  void EvaluateLoop(int& vtype, dsrNodePtr fedges[]);
  void SplitLoop(dsrNodePtr fedges[2], int numNodes,
                 dsrNodePtr *nodes, int& n1, dsrNodePtr *l1,
                 int& n2, dsrNodePtr *l2);
  int CanSplitLoop(dsrNodePtr fedges[2], int numNodes,
                   dsrNodePtr nodes[], int& n1, dsrNodePtr l1[],
                   int& n2, dsrNodePtr l2[], float& ar);
  void Triangulate(int numNodes, dsrNodePtr nodes[]);
  int CalcError(xp_long ptId);
  float CalcDataGrad(xp_long ptId);  // data dependence
  float CalcDataCurv(xp_long ptId);  // data dependence

  int IsEdge(xp_long p1, xp_long p2);
  int IsTriangle(xp_long v1, xp_long v2, xp_long v3);
  int GetTriEdgeNeighbours(xp_long triId, xp_long p1, xp_long p2, xp_long *triIds);
  void ReplaceTri(xp_long triId, xp_long *newNodes);
};

#endif
