//			Copyright (c) 1993 by
//			Advanced Visual Systems Inc.
//			All Rights Reserved
//
//	This software comprises unpublished confidential information of
//	Advanced Visual Systems Inc. and may not be used, copied or made
//	available to anyone, except in accordance with the license
//	under which it is furnished.
//
//	This file is under Perforce control
//	$Id: //depot/express/fcs70/cxxutl/string.cxx#1 $
//

// Source file for UtString class

//----------------------------------------------------------------------

#include <stdlib.h>
#include <string.h>
#include <ctype.h>

#ifdef XP_STD_IOSTREAM
#include <iostream>
#else
#include <iostream.h>
#endif

#include <avs/cxxutl/string.hxx>

//----------------------------------------------------------------------
UtString::UtString()
//----------------------------------------------------------------------
{
    _str = 0;
    _size = 0;
    _capacity = 0;
}

//----------------------------------------------------------------------
UtString::UtString(const char *string)
//----------------------------------------------------------------------
{
    _str = 0;
    if (!string) {
        _size = 0;
        _capacity = 0;
    }
    else {
        _size = ::strlen(string);
        _capacity = _size;
        if( _capacity > 0 ) {
            _str = new char[_capacity + 1];
            ::strcpy(_str, string);
            _str[_size] = 0;
        }
    }
}

//----------------------------------------------------------------------
UtString::UtString(const char* string, size_t numChar)
//----------------------------------------------------------------------
{
    _str = 0;
    if (!string) {
        _size = 0;
        _capacity = 0;
    }
    else {
        _size = numChar;
        _capacity = _size;
        if( _capacity > 0 ) {
            _str = new char[_capacity + 1];
            ::strncpy(_str, string, _size);
            _str[_size] = 0;
        }
    }
}

//----------------------------------------------------------------------
UtString::UtString(
    const UtString &s)
//----------------------------------------------------------------------
{
    _size     = s._size;
    _capacity = _size;
    _str = 0;

    if (_capacity > 0)
        _str = new char[_capacity + 1];

    if (_size == 0 || s._str == 0)
        return;

    ::memcpy( _str, s._str, _size );
    _str[_size] = 0;
}

//----------------------------------------------------------------------
UtString::UtString(
    const UtString &s, size_t pos, size_t len )
//----------------------------------------------------------------------
{
    if( (len > 0) && (pos < s._size) ) {
        _size = (len < (s._size - pos) ) ? len : (s._size - pos);
        _capacity = _size;
        _str = new char[_capacity + 1];
        ::memcpy( _str, s._str+pos, _size );
        _str[_size] = 0;

    }
    else {
        _str = 0;
        _size = 0;
        _capacity = 0;
    }
}

//----------------------------------------------------------------------
UtString::UtString(size_t numChar, char c)
//----------------------------------------------------------------------
{
    _size = numChar;
    _capacity = _size;
    _str = new char[_capacity + 1];
    for( size_t i = 0; i < _size; ++i ) _str[i] = c;
    _str[_size] = 0;
}

//----------------------------------------------------------------------
UtString::~UtString()
//----------------------------------------------------------------------
{
    delete [] _str;
    _str = 0;
    _size = 0;
    _capacity =0;
}

//----------------------------------------------------------------------
UtString& UtString::operator=(
    const UtString &s)
//----------------------------------------------------------------------
{
    if (this == &s) return *this;

    if (s._size == 0 || s._str  == 0) {
        // Alternatively, just set size to 0
        delete [] _str;
        _str = 0;
        _size = 0;
        _capacity = 0;
        return *this;
    }

    _size = s._size;
    if( _size > _capacity ) {
        _capacity = _size;
        delete [] _str;
        _str = new char[_capacity + 1];
    }

    ::memcpy( _str, s._str, _size );
    _str[_size] = 0;
    return *this;
}

//----------------------------------------------------------------------
UtString& UtString::operator=(
    const char* s)
//----------------------------------------------------------------------
{
    if (s == 0) {
        // Alternatively, just set size to 0
        delete [] _str;
        _str = 0;
        _size = 0;
        _capacity = 0;
        return *this;
    }

    _size = ::strlen( s );
    if( _size > _capacity ) {
        _capacity = _size;
        delete [] _str;
        _str = new char[_capacity + 1];
    }

    // Some special-case logic for zero length but non-NULL strings
    if( _size == 0 ) {
        if( _str ) _str[0] = 0;
    }
    else
        ::strcpy( _str, s );

    return *this;
}

//----------------------------------------------------------------------
UtString&
UtString::operator+=(
    const UtString &s)
//----------------------------------------------------------------------
{
    size_t old_size = _size;

    _size += s._size;
    if ( !_size )
        return *this;

    if( _size > _capacity ) {
        _capacity = _size;
        char *p = new char[_capacity+1];
        if (_str) {
            ::memcpy(p, _str, old_size);
            ::memcpy(p+old_size, s._str, s._size);
        }
        else {
            ::memcpy(p, s._str, s._size);
        }
        delete [] _str;
        _str = p;
    }
    else
        ::memcpy(_str+old_size, s._str, s._size);

    _str[_size] = 0;
    return *this;
}

//----------------------------------------------------------------------
UtString&
UtString::operator+=(
    const char* s)
//----------------------------------------------------------------------
{
    size_t old_size = _size;

    if( !s ) return *this;

    size_t xtra_size = ::strlen(s);
    _size += xtra_size;

    if ( _size == 0 )
        return *this;

    if( _size > _capacity ) {
        _capacity = _size;
        char *p = new char[_capacity+1];
        if (_str) {
            ::memcpy(p, _str, old_size);
            ::memcpy(p+old_size, s, xtra_size);
        }
        else {
            ::strcpy(p,s);
        }
        delete [] _str;
        _str = p;
    }
    else if( xtra_size > 0 )
        ::memcpy(_str+old_size, s, xtra_size);

    _str[_size] = 0;
    return *this;
}

//----------------------------------------------------------------------
// Append a single character.
//----------------------------------------------------------------------
UtString& UtString::append(char c)
//----------------------------------------------------------------------
{
    ++_size;
    if( _size > _capacity ) {
        _capacity = _size;
        char * p = new char[ _capacity + 1 ];
        ::memcpy( p, _str, _size-1 );
        delete [] _str;
        _str = p;
    }

    _str[_size-1] = c;
    _str[_size] = 0;

    return *this;
}

//----------------------------------------------------------------------
UtString UtString::substr(
    size_t pos,
    size_t cnt)
//----------------------------------------------------------------------
{
    // Illegal or meaningless args return empty
    if (pos >= _size) {
        return UtString();
    }
    // If not enough characters to satisfy request cut it down
    if (cnt + pos > _size ) {
        cnt = _size - pos;
    }
    // Use a constructor to eliminate redundant copy constructor.
    // This also allows the destructor to take care of cleanup.
    return UtString(*this, pos, cnt);
}


void UtString::reserve( size_t new_capacity )
{
    if( new_capacity > _capacity ) {
        _capacity = new_capacity;
        char *p = new char[_capacity + 1];
        if( _str ) ::memcpy( p, _str, _size );
        delete [] _str;
        _str = p;
        _str[_size] = 0;
    }
}


// used in cxxutl/enum.cxx, ag/color.cxx

//----------------------------------------------------------------------
// remove all space and tabs from string
//----------------------------------------------------------------------
UtString &UtString::collapse()
//----------------------------------------------------------------------
{
    if (_str) {
        size_t old_size = _size;
        size_t i;
        char *tmpptr = _str;
        char *strptr = _str;

        for( i = 0; i < old_size; ++i ) {
            // if not blank and tab then copy this character
            if (*strptr != ' ' && *strptr != 9) {
                *tmpptr = *strptr;
                tmpptr++;
            }
            else
                --_size;
            strptr++;
        }

        _str[_size] = 0;
    }

    return *this;
}

// used in ag/lstyle.cxx

//----------------------------------------------------------------------
// remove all leading and trailing spaces and tabs from string
//----------------------------------------------------------------------
UtString &UtString::trim()
//----------------------------------------------------------------------
{
    if (_str) {
        size_t old_size = _size;
        size_t i;
        char *tmpptr = _str;
        char *strptr = _str;
        int leading = 1;

        /* Strip leading whitespace */
        for( i = 0; i < old_size; ++i ) {
            // if not blank and tab then copy this character
            if( !leading || (*strptr != ' ' && *strptr != 9) ) {
                *tmpptr = *strptr;
                tmpptr++;
                leading = 0;
            }
            else
                --_size;
            strptr++;
        }

        /* Strip trailing whitespace ... no need to copy */
        while( 1 ) {
            char c = _str[_size-1];
            if( c == ' ' || c == 9 ) --_size;
            else break;
        }

        _str[_size] = 0;
    }
    return *this;
}


// used in cxxutl/enum.cxx, ag/color.cxx

//----------------------------------------------------------------------
// convert string to lower case
//----------------------------------------------------------------------
UtString& UtString::lower()
//----------------------------------------------------------------------
{
    size_t i;
    char *strptr = _str;
    if (strptr) {
        for( i = 0; i < _size; ++i ) {
            *strptr = tolower( *strptr );
            strptr++;
        }
    }
    return *this;
}


//----------------------------------------------------------------------
// convert string to upper case
//----------------------------------------------------------------------
UtString& UtString::upper()
//----------------------------------------------------------------------
{
    size_t i;
    char *strptr = _str;
    if (strptr) {
        for( i = 0; i < _size; ++i ) {
            *strptr = toupper( *strptr );
            strptr++;
        }
    }
    return *this;
}


//----------------------------------------------------------------------
void UtString::alloc_copy(
    size_t new_capacity )
//----------------------------------------------------------------------
{
    char * new_str = 0;
    if( new_capacity > 0 )
        new_str = new char[ new_capacity ];

    ::memcpy(new_str, _str, _size);
    new_str[_size] = 0;

    delete [] _str;

    _capacity = new_capacity;
    _str      = new_str;
    _size     = 0;
}


//----------------------------------------------------------------------
void UtString::alloc_no_copy(
    size_t new_capacity )
//----------------------------------------------------------------------
{
    char * new_str = 0;
    if( new_capacity > 0 ) {
        new_str = new char[ new_capacity ];
        new_str[0] = 0;
    }

    delete [] _str;

    _capacity = new_capacity;
    _str      = new_str;
    _size     = 0;
}


//----------------------------------------------------------------------
bool operator==(
    const UtString& s1,
    const UtString& s2)
//----------------------------------------------------------------------
{
    if (s1.length() != s2.length()) return 0;
    for( size_t i = 0; i < s1.length(); ++i ) {
        if( s1[i] != s2[i] )
            return 0;
    }
    return 1;
}

//----------------------------------------------------------------------
bool operator==(
    const UtString& s1,
    const char * s2)
//----------------------------------------------------------------------
{
    if (s2 == 0) {
        if (s1.empty())
            return 1;       // Could go either way on this one.
        else
            return 0;
    }
    size_t i;
    for( i = 0; i < s1.length(); ++i ) {
        if( s2[i] == 0 )    // If we've reached the end of the c_str, without
            return 0;       // reaching the end of the UtString then false
        if( s1[i] != s2[i] )
            return 0;
    }
    if( s2[i] != 0 )    // If we have _not_ reached the end of the c_str when
        return 0;       // we've reached the end of the UtString then false.
    return 1;
}


//----------------------------------------------------------------------
bool operator!=(
    const UtString& s1,
    const UtString& s2)
//----------------------------------------------------------------------
{
    if (s1.length() != s2.length()) return 1;
    for( size_t i = 0; i < s1.length(); ++i ) {
        if( s1[i] != s2[i] )
            return 1;
    }
    return 0;
}

//----------------------------------------------------------------------
bool operator!=(
    const UtString& s1,
    const char *s2)
//----------------------------------------------------------------------
{
    if (s2 == 0) {
        if (s1.empty())
            return 0;
        else
            return 1;
    }
    size_t i;
    for( i = 0; i < s1.length(); ++i ) {
        if( s2[i] == 0 )
            return 1;
        if( s1[i] != s2[i] )
            return 1;
    }
    if( s2[i] != 0 )
        return 1;
    return 0;
}


//----------------------------------------------------------------------
bool operator<(
    const UtString& s1,
    const UtString& s2)
//----------------------------------------------------------------------
{
    size_t i;
    for( i = 0; (i < s1.length()) && (i < s2.length()); ++i ) {
        if( s1[i] < s2[i] )
            return 1;
        if( s1[i] > s2[i] )
            return 0;
    }
    return( s1.length() < s2.length() );
}

//----------------------------------------------------------------------
bool operator<(
    const UtString& s1,
    const char * s2)
//----------------------------------------------------------------------
{
    if (s2 == (char *)0) return 0;

    size_t i;
    for( i = 0; (i < s1.length()) && (s2[i] != 0); ++i ) {
        if( s1[i] < s2[i] )
            return 1;
        if( s1[i] > s2[i] )
            return 0;
    }
    if( (s1.length() == i) && (s2[i] == 0) )
        return 0;   // Strings are equal, thus return false
    if( (s1.length() == i) )
        return 1;   // s1 is shorter

    return 0;   // s2 is shorter
}

//----------------------------------------------------------------------
bool operator<(
    const char * s1,
    const UtString& s2)
//----------------------------------------------------------------------
{
    if (s1 == (char *)0) return 1;

    size_t i;
    for( i = 0; (s1[i] !=0) && (i < s2.length()); ++i ) {
        if( s1[i] < s2[i] )
            return 1;
        if( s1[i] > s2[i] )
            return 0;
    }
    if( (s1[i] == 0) && (s2.length() == i) )
        return 0;   // Strings are equal, thus return false
    if( s1[i] == 0 )
        return 1;   // s1 is shorter

    return 0;   // s2 is shorter
}

//----------------------------------------------------------------------
bool operator>(
    const UtString& s1,
    const UtString& s2)
//----------------------------------------------------------------------
{
    if (s1.length() == 0) return 0;
    if (s2.length() == 0) return 1;
    size_t i;
    for( i = 0; (i < s1.length()) && (i < s2.length()); ++i ) {
        if( s1[i] > s2[i] )
            return 1;
        if( s1[i] < s2[i] )
            return 0;
    }
    return( s1.length() > s2.length() );
}

//----------------------------------------------------------------------
bool operator>(
    const UtString& s1,
    const char * s2)
//----------------------------------------------------------------------
{
    if (s2 == (char *)0) return 1;

    size_t i;
    for( i = 0; (i < s1.length()) && (s2[i] != 0); ++i ) {
        if( s1[i] > s2[i] )
            return 1;
        if( s1[i] < s2[i] )
            return 0;
    }
    if( (s1.length() == i) && (s2[i] == 0) )
        return 0;   // Strings are equal, thus return false
    if( (s1.length() == i) )
        return 0;   // s1 is shorter

    return 1;   // s2 is shorter
}

//----------------------------------------------------------------------
bool operator>(
    const char * s1,
    const UtString& s2)
//----------------------------------------------------------------------
{
    if (s1 == (char *)0) return 0;

    size_t i;
    for( i = 0; (s1[i] != 0) && (i < s2.length()); ++i ) {
        if( s1[i] > s2[i] )
            return 1;
        if( s1[i] < s2[i] )
            return 0;
    }
    if( (s1[i] == 0) && (s2.length() == i) )
        return 0;   // Strings are equal, thus return false
    if( (s1[i] == 0) )
        return 0;   // s1 is shorter

    return 1;   // s2 is shorter
}

//----------------------------------------------------------------------
UtString
operator+(
    const UtString &s1,
    const UtString &s2)
//----------------------------------------------------------------------
{
    UtString result = s1;
    result += s2;
    return result;
}

//----------------------------------------------------------------------
UtString operator+(
    const UtString& s1,
    const char* s2)
//----------------------------------------------------------------------
{
    UtString result( s1 );
    result += s2;
    return result;
}

#ifdef XP_STD_IOSTREAM
#include <iomanip>
using std::setw;
#else
#include <iomanip.h>
#endif

//----------------------------------------------------------------------
istream& operator>>(
    istream &io,
    UtString &s)
//----------------------------------------------------------------------
{
    const int UtString_size = 1024;
    char inBuf[ UtString_size ];
    io >> setw( UtString_size-1 ) >> inBuf;
    s = inBuf; // UtString::operator=(char*);
    return io;
}

//----------------------------------------------------------------------
ostream& operator<<(
    ostream &os,
    const UtString &s)
//----------------------------------------------------------------------
{
    if (s.empty()) return os;
    return os << s.c_str();
}

