/*
    		Copyright (c) 1994 by
      		Advanced Visual Systems Inc.
      		All Rights Reserved

      This software comprises unpublished confidential information of
      Advanced Visual Systems Inc. and may not be used, copied or made
      available to anyone, except in accordance with the license
      under which it is furnished.

      This file is under Perforce control
      $Id: //depot/express/fcs70/gd/roi.c#1 $
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#define XP_WIDE_API	/* Use Wide APIs */
#include <avs/util.h>
#include <avs/om.h>
#include <avs/om_att.h>		/* needed for OM_prop_rdonly usage */
#include <avs/math.h>
#include <avs/fld.h>
#include <avs/gd.h>
#include "sw/swx.h"

/* Local structure for edit mesh module */
typedef struct _EditMeshData {
   GDview *view;
   GDcamera *camera;
   OMobj_id pobj_id;
   int mode;			/* 0 = edit point, 1 = edit prim. */
   int draw_mode;		/* 0 = copy, 1 = XOR */
   int immed;			/* 0 = update on confirm, 1 = update on button up. */
   float color[3];		/* hilight color */
   GDpick_data pick_data;
   float fullxform[4][4];
   xp_long nnodes;
   int nspace, cell_type;
   int coord_alloced;
   float *coord_array;
   xp_long coord_size;
   int conn_alloced;
   xp_long *conn_array;
   xp_long conn_size;
   int nnode_alloced;
   int *nnode_array;
   xp_long nnode_size;
   xp_long ncells;
   int connected;		/* 1 if polyline is a closed region. */
   xp_long vert_size;
   xp_long nverts;
   FLOAT3 *verts, *tverts;
   SHORT2 *sverts;
   int prim_moved;
   int move_state;
   SHORT2 *ssverts, *esverts;
   FLOAT2 *w2verts;
   FLOAT3 *w3verts;
   int state, x, y;
   int prevx, prevy;
} EditMeshData;

/* Buffers for ROI drawing */
typedef struct _ROIData {
  GDview *view;
  GDcamera *camera;
  int mode, draw_mode, option;
  int color_flag;
  float color[3];
  /* Data for drawing ROIs */
  xp_long npoints, pointbuf_size;
  short *pointbuf;
  int line_flag;
  xp_long nlines, linebuf_size;
  short *linebuf;
  int box_flag;
  xp_long nbox, boxbuf_size;
  short *boxbuf;
  int free_flag;
  xp_long nfree, freebuf_size, nverts;
  xp_long curr_size, curr_idx;
  short *freebuf;
  short *currptr;
} ROIData;

/* Local structure for draw cursor 2d module */
typedef struct _DrawCursorData {
   GDview *view;
   GDcamera *camera;
   GDobject *object;
   int width, height;		/* width and height of the view */
   float color[3];		/* cursor color */
   int x, y;
   int xs, xe;			/* X start/end positions */
   int ys, ye;			/* Y start/end positions */
   int prev_x, prev_y;
   int prev_xs, prev_ys;
   int prev_xe, prev_ye;
   int state;
   int immed;
   int mode;			/* 0=full screen 1=cross */
   int size;			/* size of cross */
   int thickness;		/* line thickness of cursor */
} DrawCursorData;

/* Room for 200 shorts ROIData buffer */
#define FREEBUF_INCR	200

/* Half the width of the point hilite square. */
#define HILITE_SIZE	3

/* Used to determine if we are close enough to
   redraw the primitive.
*/
#define DELTA_SIZE	5

#define ROI_POINT	0
#define ROI_LINE	1
#define ROI_BOX		2
#define ROI_POLYLINE	3
#define ROI_POLYGON	4

#define DELETE_CELL_SET	1
#define DELETE_CELL	2
#define MODIFY_CELL	3

/* Private functions: map utilities */
static void GDmap_ss_box_to_points (ROIData *, short *);
static void GDmap2d_write_mesh (OMobj_id, int, ROIData *, xp_long, float *, char *);
static void GDmap2d_write_mesh_cdata (OMobj_id, ROIData *, xp_long);

/* Private functions: ROI utilities */
static int ROIget_inputs (OMobj_id, ROIData *);
static void ROIget_color (OMobj_id, ROIData *);
static void ROIsetup (GDview *, ROIData *);
static void ROIredraw (ROIData *, GDview *);
static void ROIclear (ROIData *, int);
static int ROIread (OMobj_id, ROIData *);
static int ROIwrite (OMobj_id, ROIData *, int);

static void ROIpoint_addpt (ROIData *, int, int);
static void ROIpoint_cont (ROIData *, GDview *, int, int, int);
static void ROIpoint_click (ROIData *, GDview *, int, int);
static void ROIpoint_draw (ROIData *, GDview *, int);

static void ROIline_addpt (ROIData *, int, int);
static void ROIline_endpt (ROIData *, GDview *, int, int);
static void ROIline_cont (ROIData *, GDview *, int, int, int);
static void ROIline_click (ROIData *, GDview *, int, int);

static void ROIbox_addpt (ROIData *, int, int);
static void ROIbox_endpt (ROIData *, GDview *, int, int);
static void ROIbox_cont (ROIData *, GDview *, int, int, int);
static void ROIbox_click (ROIData *, GDview *, int, int);

static void ROIfree_addpt (ROIData *, int, int, int);
static void ROIfree_cont (ROIData *, GDview *, int, int, int);
static void ROIfree_click (ROIData *, GDview *, int, int);

/* Private functions: Draw cursor utilities */
static int GDdraw2d_cursor_get_inputs (OMobj_id, DrawCursorData *);
static void GDdraw2d_write_point (OMobj_id, float *);

/* Private functions: Edit Mesh utilities */
static int GDedit_mesh_get_inputs (OMobj_id, EditMeshData *);
static int GDedit_mesh_get_pickinfo (OMobj_id, EditMeshData *);
static int GDedit_mesh_get_fieldinfo (OMobj_id, EditMeshData *, int, int);
static void GDedit_mesh_free_fieldinfo (OMobj_id, EditMeshData *);
static void GDedit_mesh_hilite (OMobj_id, EditMeshData *);
static void GDedit_mesh_modify (OMobj_id, EditMeshData *);
static void GDedit_mesh_setup (OMobj_id, EditMeshData *);
static void GDedit_mesh_updfield (OMobj_id, EditMeshData *);
static void GDedit_mesh_realloc (EditMeshData *, xp_long);
static void GDedit_mesh_delfield (OMobj_id, EditMeshData *);


/* ROI modules:
   GDroi2d_cont_xxx  - Draw 2D ROI with UItwoPoint.
   GDroi2d_click_xxx - Draw 2D ROI with UIonePoint.
   GDdraw2d_cursor_xxx - Draw 2D cursor and map point back to world space.
   GDmap2d_ss_xxx    - Map 2D ROI from screem space to world space.
*/
int GDroi2d_cont_create(OMobj_id elem_id)
{
   OMobj_id ptr_id;
   ROIData *buf;

   /* Allocate data and initialize it. */
   ALLOCN(buf, ROIData, 1, "can't allocate ROIData");

   /* find the local ptr element and save the local structure */
   ptr_id = OMfind_subobj(elem_id, OMstr_to_name("local_ptr"), OM_OBJ_RW);
   if (OMis_null_obj(ptr_id))
      return(0);
   else OMset_ptr_val(ptr_id, buf, 0);
   return(1);
}

int GDroi2d_cont_delete(OMobj_id elem_id)
{
   OMobj_id ptr_id;
   ROIData *buf;

   /* find the local ptr element and save the local structure */
   ptr_id = OMfind_subobj(elem_id, OMstr_to_name("local_ptr"), OM_OBJ_RW);
   if (OMis_null_obj(ptr_id))
      return(0);
   if (OMget_ptr_val(ptr_id, (void **)&buf, 0) != OM_STAT_SUCCESS)
      return(0);

   /* Clear all buffers used for drawing */
   ROIclear(buf, 1);
   free(buf);

   GDclear_local(elem_id);
   return(1);
}

/* We should get activations when the view changes,
   the state changes or done is updated.
   When the view changes, we refresh, When state changes
   we process the event. When done changes, we write the
   buffers to the framework.
*/
int GDroi2d_cont_update(OMobj_id elem_id)
{
   OMobj_id ptr_id, seq_id, state_id;
   int seq, redraw = 0, clear = 0, immed = 0;
   int state, x, y;
   ROIData *buf;
   GDview *view;

   /* find the local ptr element and get the ROIData structure */
   ptr_id = OMfind_subobj(elem_id, OMstr_to_name("local_ptr"), OM_OBJ_RW);
   if (OMis_null_obj(ptr_id))
      return(0);
   if (OMget_ptr_val(ptr_id, (void **)&buf, 0) != OM_STAT_SUCCESS)
      return(0);

   /* Every time the update function runs the sequence number goes up. This
      means the sequence number of the element itself will be the same
      as the sequence number of the update function after the update function
      executes.  We can use this fact to determine if any of the sub-elements
      has changed since the last time we ran.  If any of the elements have a
      sequence number higher than this, we know that the element has been
      modified since the last execution.
   */
   seq_id = OMfind_subobj(elem_id, OMstr_to_name("upd_func"), OM_OBJ_RD);
   if (OMis_null_obj(seq_id))
      return(0);
   seq = OMget_obj_seq(seq_id, OMnull_obj, 0);

   /* Check to see if the user has requested that we
      write the contents of the buffer to the framework.
   */
   ptr_id = OMfind_subobj(elem_id, OMstr_to_name("done"), OM_OBJ_RW);
   if (OMis_null_obj(ptr_id))
      return(0);
   if (OMget_obj_seq(ptr_id, OMnull_obj, OM_SEQ_VAL) > seq) {
      ROIwrite(elem_id, buf, 0);
      ROIclear(buf, 1);
      clear = 1;
   }

   /* Check to see if the user has requested that we
      clear the contents of the buffer.
   */
   ptr_id = OMfind_subobj(elem_id, OMstr_to_name("clear"), OM_OBJ_RW);
   if (OMis_null_obj(ptr_id))
      return(0);
   if (OMget_obj_seq(ptr_id, OMnull_obj, OM_SEQ_VAL) > seq) {
      ROIclear(buf, 1);
      /* ROIwrite(elem_id, buf, 0); */
      clear = 1;
   }

   /* Check to see if the user has requested that we
      redraw the contents of the buffer.
   */
   ptr_id = OMfind_subobj(elem_id, OMstr_to_name("redraw"), OM_OBJ_RW);
   if (OMis_null_obj(ptr_id))
      return(0);
   if (OMget_obj_seq(ptr_id, OMnull_obj, OM_SEQ_VAL) > seq) {
      /* Make sure the buffer has stuff to draw in it. */
      if (buf->npoints || buf->nlines || buf->nbox || buf->nfree)
         redraw = 1;
      /* Else, just return */
      else return(1);
   }

   state_id = OMfind_subobj(elem_id, OMstr_to_name("state"), OM_OBJ_RW);
   if (OMis_null_obj(state_id))
      return(0);

   /* Make sure we have a good activation. If we got activated
      because of done, it is likely that state will not have changed.
   */
   if (seq > OMget_obj_seq(state_id, OMnull_obj, OM_SEQ_VAL))
      state = 0;
   else {
      OMget_int_val(state_id, &state);

      /* Get xy screen space position that is associated with interactor. */
      if (!GDget_int_val(elem_id, "x", &x))
         return(0);
      if (!GDget_int_val(elem_id, "y", &y))
         return(0);
   }

   /* Deal with other inputs to module. */
   if (!ROIget_inputs(elem_id, buf))
      return(0);
   ROIget_color(elem_id, buf);

   /* ROI get inputs has retrieved the view and camera for us. */
   view = buf->view;

   /* If we are in copy mode, we need to restore the contents
      of the window before we begin drawing.
      0 = copy
      1 = xor
   */
   view->status = GDview_status(view);
   if (!buf->draw_mode) {
      /* COPY mode - Copy the back buffer to the
	 front buffer (ie. the window) to refresh
	 the contents of the view.
      */
      GDview_call_func(view, "view_refresh", 0);

      /* If clear has been requested we can quit
	 after we have refreshed the buffer.
      */
      if (clear)
	 return(1);

      /* Do setup after we swap since it will set
	 the drawable to the window. we want the
	 swap to pay attention to what the double
	 buffer flag actually says.
      */
      ROIsetup(view, buf);

      /* If mode is append or if redraw has been specified -
         draw all existing lines, boxes and freehands.
      */
      if (redraw || buf->mode) {
         GDstate_set_draw_mode(view, 0);
	 ROIredraw(buf, view);
      }
      else ROIclear(buf, 0);
   }
   else {
      /* XOR mode */
      if (clear) {
         GDview_call_func(view, "view_refresh", 0);
	 return(1);
      }
      else if (redraw) {
         GDview_call_func(view, "view_refresh", 0);

	 /* setup after swap for reason stated above. */
         ROIsetup(view, buf);
         /* If mode is append - draw all existing points, lines, boxes and freehands */
         if (redraw || buf->mode) {
            GDstate_set_draw_mode(view, 0);
	    ROIredraw(buf, view);
         }
	 else ROIclear(buf, 0);
      }
      else ROIsetup(view, buf);
   }

   if (!redraw) {
      switch (buf->option) {
         case ROI_POINT:	/* draw point */
	    ROIpoint_cont(buf, view, state, x, y);
	    break;
         case ROI_LINE:		/* draw line */
	    ROIline_cont(buf, view, state, x, y);
	    break;
         case ROI_BOX:		/* draw box */
	    ROIbox_cont(buf, view, state, x, y);
	    break;
         case ROI_POLYLINE:	/* draw polyline */
         case ROI_POLYGON:	/* draw polygon */
	    ROIfree_cont(buf, view, state, x, y);
	    break;
         default:
	    ERRerror("GDroi2d_cont_update", 0, ERR_ORIG, "Unsupported option");
	    break;
      }
   }
   GDstack_pop(view);

   /* Check to see if the user has requested that we
      write the contents of the buffer to the framework
      immediately or wait for an explicit done indication.
   */
   ptr_id = OMfind_subobj(elem_id, OMstr_to_name("immed"), OM_OBJ_RW);
   if (OMis_null_obj(ptr_id))
      return(0);
   if (OMget_int_val(ptr_id, &immed) == OM_STAT_SUCCESS) {
      if (immed && state == 3) {
         ROIwrite(elem_id, buf, 0);
         ROIclear(buf, 1);
      }
   }
   return(1);
}

int GDroi2d_click_create(OMobj_id elem_id)
{
   OMobj_id ptr_id;
   ROIData *buf;

   /* Allocate data and initialize it. */
   ALLOCN(buf, ROIData, 1, "can't allocate ROIData");

   /* find the local ptr element and save the local structure */
   ptr_id = OMfind_subobj(elem_id, OMstr_to_name("local_ptr"), OM_OBJ_RW);
   if (OMis_null_obj(ptr_id))
      return(0);
   else OMset_ptr_val(ptr_id, buf, 0);
   return(1);
}

int GDroi2d_click_delete(OMobj_id elem_id)
{
   OMobj_id ptr_id;
   ROIData *buf;

   /* find the local ptr element and save the local structure */
   ptr_id = OMfind_subobj(elem_id, OMstr_to_name("local_ptr"), OM_OBJ_RW);
   if (OMis_null_obj(ptr_id))
      return(0);
   if (OMget_ptr_val(ptr_id, (void **)&buf, 0) != OM_STAT_SUCCESS)
      return(0);

   /* Clear all buffers used for drawing */
   ROIclear(buf, 1);
   free(buf);

   GDclear_local(elem_id);
   return(1);
}

int GDroi2d_click_update(OMobj_id elem_id)
{
   OMobj_id ptr_id, seq_id, add_id, erase_id, close_id;
   int seq, redraw = 0, clear = 0, immed = 0;
   int add = 0, erase = 0, close = 0;
   int x, y;
   GDview *view;
   ROIData *buf;

   /* find the local ptr element and get the ROIData structure */
   ptr_id = OMfind_subobj(elem_id, OMstr_to_name("local_ptr"), OM_OBJ_RW);
   if (OMis_null_obj(ptr_id))
      return(0);
   if (OMget_ptr_val(ptr_id, (void **)&buf, 0) != OM_STAT_SUCCESS)
      return(0);

   /* Every time the update function runs the sequence number goes up. This
      means the sequence number of the element itself will be the same
      as the sequence number of the update function after the update function
      executes.  We can use this fact to determine if any of the sub-elements
      has changed since the last time we ran.  If any of the elements have a
      sequence number higher than this, we know that the element has been
      modified since the last execution.
   */
   seq_id = OMfind_subobj(elem_id, OMstr_to_name("upd_func"), OM_OBJ_RD);
   if (OMis_null_obj(seq_id))
      return(0);
   seq = OMget_obj_seq(seq_id, OMnull_obj, 0);

   /* Check to see if the user has requested that we
      write the contents of the buffer to the framework.
   */
   ptr_id = OMfind_subobj(elem_id, OMstr_to_name("done"), OM_OBJ_RW);
   if (OMis_null_obj(ptr_id))
      return(0);
   if (OMget_obj_seq(ptr_id, OMnull_obj, OM_SEQ_VAL) > seq) {
      ROIwrite(elem_id, buf, 1);
      ROIclear(buf, 1);
      clear = 1;
   }

   /* Check to see if the user has requested that we
      clear the contents of the buffer.
   */
   ptr_id = OMfind_subobj(elem_id, OMstr_to_name("clear"), OM_OBJ_RW);
   if (OMis_null_obj(ptr_id))
      return(0);
   if (OMget_obj_seq(ptr_id, OMnull_obj, OM_SEQ_VAL) > seq) {
      ROIclear(buf, 1);
      /* ROIwrite(elem_id, buf, 0); */
      clear = 1;
   }

   /* Check to see if the user has requested that we
      redraw the contents of the buffer.
   */
   ptr_id = OMfind_subobj(elem_id, OMstr_to_name("redraw"), OM_OBJ_RW);
   if (OMis_null_obj(ptr_id))
      return(0);
   if (OMget_obj_seq(ptr_id, OMnull_obj, OM_SEQ_VAL) > seq) {
      /* Make sure the buffer has stuff to draw in it - make sure
	 to include a test on nverts which is the polyline/polygon
	 in progress.
      */
      if (buf->npoints || buf->nlines || buf->nbox || buf->nfree || buf->nverts)
         redraw = 1;
      /* Else, just return */
      else return(1);
   }

   /* Three different inputs: add point, erase point and close. */
   add_id = OMfind_subobj(elem_id, OMstr_to_name("add"), OM_OBJ_RW);
   if (OMis_null_obj(add_id))
      return(0);
   if (OMget_obj_seq(add_id, OMnull_obj, OM_SEQ_VAL) > seq) {
      add = 1;

      /* Get xy screen space position that is associated with interactor. */
      if (!GDget_int_val(elem_id, "x", &x))
         return(0);
      if (!GDget_int_val(elem_id, "y", &y))
         return(0);
   }

   erase_id = OMfind_subobj(elem_id, OMstr_to_name("erase"), OM_OBJ_RW);
   if (OMis_null_obj(erase_id))
      return(0);
   if (OMget_obj_seq(erase_id, OMnull_obj, OM_SEQ_VAL) > seq)
      erase = 1;

   close_id = OMfind_subobj(elem_id, OMstr_to_name("close"), OM_OBJ_RW);
   if (OMis_null_obj(close_id))
      return(0);
   if (OMget_obj_seq(close_id, OMnull_obj, OM_SEQ_VAL) > seq)
      close = 1;

   /* Make sure we have a good activation. */
   if (!add && !erase && !close && !redraw && !clear)
      return(1);

   /* Deal with other inputs to module. */
   if (!ROIget_inputs(elem_id, buf))
      return(1);
   ROIget_color(elem_id, buf);

   /* ROI get inputs has retrieved the view and camera for us. */
   view = buf->view;

   /* COPY mode drawing - Copy the back buffer to the
      front buffer (ie. the window) to refresh
      the contents of the view.
   */
   view->status = GDview_status(view);
   GDview_call_func(view, "view_refresh", 0);

   /* If a clear has been requested, we can quit after
      we refresh the view since there is nothing to draw.
   */
   if (clear)
      return(1);

   /* Do setup after we swap since it will set
      the drawable to the window. we want the
      swap to pay attention to what the double
      buffer flag actually says.
   */
   ROIsetup(view, buf);

   /* If mode is append - draw all existing points, lines, boxes and freehands */
   if (buf->mode || redraw) {
      GDstate_set_draw_mode(view, 0);
      ROIredraw(buf, view);
      /* Redraw any polyline/polygon in progress */
      if (buf->nverts > 1)
         GD2d_draw_polyline(view->State, buf->nverts, (SHORT2 *)&buf->currptr[1], NULL, NULL, 0);
   }
   else ROIclear(buf, 0);

   /* If we have not been activated on a redraw, process the
      add, erase or close event.
   */
   if (!redraw) {
      switch (buf->option) {
	 case ROI_POINT:	/* draw point */
	    if (add)
	       ROIpoint_click(buf, view, x, y);
            else if (erase)
	       if (buf->npoints)
		  buf->npoints--;
               if (buf->npoints)
		  ROIpoint_draw(buf, view, 1);
            break;
         case ROI_LINE:		/* draw line */
	    if (add)
	       ROIline_click(buf, view, x, y);
	    else if (erase) {
	       if (buf->nlines)
	          buf->nlines--;
               if (buf->nlines)
                  GD2d_draw_lines(view->State, buf->nlines*2, (SHORT2 *)buf->linebuf, NULL, NULL);
	    }
	    if (close)
	       buf->line_flag = 0;
	    break;
         case ROI_BOX:		/* draw box */
	    if (add)
	       ROIbox_click(buf, view, x, y);
	    else if (erase) {
	       if (buf->nbox)
	          buf->nbox--;
	       if (buf->nbox)
	          GD2d_draw_rects(view->State, buf->nbox*2, (SHORT2 *)buf->boxbuf, NULL);
	    }
	    if (close)
	       buf->box_flag = 0;
	    break;
         case ROI_POLYLINE:	/* draw polyline */
         case ROI_POLYGON:	/* draw polygon */
	    if (add) {
	       ROIfree_click(buf, view, x, y);
	       if (buf->nverts > 1)
	          GD2d_draw_polyline(view->State, buf->nverts, (SHORT2 *)&buf->currptr[1], NULL, NULL, 0);
            }
	    else if (erase) {
	       /* Only allow erase if we have an open region. */
	       if (buf->free_flag) {
	          if (buf->nverts) {
	             buf->nverts--;
	             buf->curr_size -= 2;	/* back off size when erasing a point. */
	          }
	          /* Need at least two points to draw a line. */
	          if (buf->nverts > 1)
	             GD2d_draw_polyline(view->State, buf->nverts, (SHORT2 *)&buf->currptr[1], NULL, NULL, 0);

	          /* If we may have erased all the points in the region
	             so clear the started flag and back off the size pointer
	             since the next point will start a new region.
	          */
	          if (buf->nverts == 0) {
	             buf->free_flag = 0;
	             buf->curr_size--;
	          }
	       }
	    }
	    if (close) {
	       /* Only allow close if we have an open region. */
	       if (buf->free_flag) {
	          buf->free_flag = 0;
	          /* Need at least 3 points to form a valid polygon - that is
	             two lines - we will add 1 to make at least a triangle.
	          */
		  if (buf->option == ROI_POLYGON && buf->nverts > 2) {
                     buf->currptr[buf->nverts*2+1] = buf->currptr[1];
                     buf->currptr[buf->nverts*2+2] = buf->currptr[2];
	             buf->curr_size += 2;
		     buf->nverts++;
	             buf->currptr[0] = buf->nverts;
	             buf->nfree++;
	             GD2d_draw_polyline(view->State, buf->nverts, (SHORT2 *)&buf->currptr[1], NULL, NULL, 0);
	          }
		  else if (buf->option == ROI_POLYLINE && buf->nverts > 1) {
                     buf->currptr[0] = buf->nverts;
                     buf->nfree++;
	             GD2d_draw_polyline(view->State, buf->nverts, (SHORT2 *)&buf->currptr[1], NULL, NULL, 0);
		  }
	          /* If we have less verts than make a valid region, reset the buffer
	             to eliminate those points.
	          */
	          else if (buf->nverts > 0) {
	             /* back all the way off - including spot reserved for
		        number of verts. the start of the next region will
		        allocate this space again.
                     */
	             buf->curr_size -= (buf->nverts * 2) + 1;
	             buf->nverts = 0;
	          }
               }
            }
	    break;
         default:
	    ERRerror("GDroi2d_click_update", 0, ERR_ORIG, "Unsupported option");
	    break;
      }
   }
   GDstack_pop(view);

   /* Check to see if the user has requested that we
      write the contents of the buffer to the framework
      immediately or wait for an explicit done indication.

      This is a little tricky since we add points one at a time.
      Depending on the type of primitive we are creating - it is
      done after different numbers of points have been added.
   */
   ptr_id = OMfind_subobj(elem_id, OMstr_to_name("immed"), OM_OBJ_RW);
   if (OMis_null_obj(ptr_id))
      return(0);
   if (OMget_int_val(ptr_id, &immed) == OM_STAT_SUCCESS) {
      if (immed) {
         if (ROIwrite(elem_id, buf, 0))
            ROIclear(buf, 1);
      }
   }
   return(1);
}

int GDdraw2d_cursor_create(OMobj_id elem_id)
{
   OMobj_id ptr_id;
   DrawCursorData *cursorData;

   /* Allocate data and initialize it. */
   ALLOCN(cursorData, DrawCursorData, 1, "can't allocate DrawCursorData");

   /* find the local ptr element and save the local structure */
   ptr_id = OMfind_subobj(elem_id, OMstr_to_name("local_ptr"), OM_OBJ_RW);
   if (OMis_null_obj(ptr_id))
      return(0);
   else OMset_ptr_val(ptr_id, cursorData, 0);
   return(1);
}

int GDdraw2d_cursor_delete(OMobj_id elem_id)
{
   OMobj_id ptr_id;
   DrawCursorData *cursorData;

   /* find the local ptr element and save the local structure */
   ptr_id = OMfind_subobj(elem_id, OMstr_to_name("local_ptr"), OM_OBJ_RW);
   if (OMis_null_obj(ptr_id))
      return(0);
   if (OMget_ptr_val(ptr_id, (void **)&cursorData, 0) != OM_STAT_SUCCESS)
      return(0);

   free(cursorData);

   GDclear_local(elem_id);
   return(1);
}

int GDdraw2d_cursor_update(OMobj_id elem_id)
{
   OMpfi state_func;
   OMobj_id ptr_id;
   DrawCursorData *cursorData;
   short sspoint[2];
   short lines[16];
   float mat[4][4], invmat[4][4], wpoint[2];

   /* find the local ptr element and get the DrawCursorData structure */
   ptr_id = OMfind_subobj(elem_id, OMstr_to_name("local_ptr"), OM_OBJ_RW);
   if (OMis_null_obj(ptr_id))
      return(0);
   if (OMget_ptr_val(ptr_id, (void **)&cursorData, 0) != OM_STAT_SUCCESS)
      return(0);

   /* Get view & camera related inputs. If we can't get
      the inputs successfully, quit before we get in
      trouble trying to use them.
   */
   if (!GDdraw2d_cursor_get_inputs(elem_id, cursorData))
      return(1);

   /* Push start of rendering marker onto the stack so we can
      restore original attributes when we are done.
   */
   GDstack_push(cursorData->view);

   /* Set the color to be used for drawing the cursor */
   GDstate_set_color(cursorData->view, cursorData->color);
   /* Set the line width for drawing the cursor */
   GDstate_set_line_width(cursorData->view, cursorData->thickness);

   /* Currently allowed modes are 0 = copy and non-zero = XOR.
      This needs to be improved so we have defines for each draw mode.
   */
   GDstate_set_draw_mode(cursorData->view, 1);

   /* Use the front buffer (ie. window) as the
      drawble. This allows our drawing to be visible even if
      we are in double buffer mode.
   */
   if (GDstate_get_func(cursorData->view, "state_set_drawable", &state_func))
      (*state_func)(cursorData->view, 1);

   if (cursorData->mode == 0) {
      cursorData->xs = 0;
      cursorData->xe = cursorData->width;
      cursorData->ys = 0;
      cursorData->ye = cursorData->height;
   }
   else {
      cursorData->xs = cursorData->x - cursorData->size;
      cursorData->xe = cursorData->x + cursorData->size;
      cursorData->ys = cursorData->y - cursorData->size;
      cursorData->ye = cursorData->y + cursorData->size;
   }

   switch (cursorData->state) {
      case 1:           /* start event */
         /* draw full screen cursor */
         lines[0] = cursorData->x;
         lines[1] = cursorData->ys;
         lines[2] = cursorData->x;
         lines[3] = cursorData->ye;
         lines[4] = cursorData->xs;
         lines[5] = cursorData->y;
         lines[6] = cursorData->xe;
         lines[7] = cursorData->y;
         /* This is the screen space entry point. There is also an
            entry for passing world space coords that will be
            transformed to screen space.  The additional
            entries on the call are for an array of facet colors
            and an array of vertex colors respectively. The vertex
            colors are averaged.
         */
         GD2d_draw_lines(cursorData->view->State, 4, (SHORT2 *)lines, NULL, NULL);
         break;
      case 2:           /* run event */
         /* erase line at previous position */
         lines[0] = cursorData->prev_x;
         lines[1] = cursorData->prev_ys;
         lines[2] = cursorData->prev_x;
         lines[3] = cursorData->prev_ye;
         lines[4] = cursorData->prev_xs;
         lines[5] = cursorData->prev_y;
         lines[6] = cursorData->prev_xe;
         lines[7] = cursorData->prev_y;
         /* draw line at new position */
         lines[8] = cursorData->x;
         lines[9] = cursorData->ys;
         lines[10] = cursorData->x;
         lines[11] = cursorData->ye;
         lines[12] = cursorData->xs;
         lines[13] = cursorData->y;
         lines[14] = cursorData->xe;
         lines[15] = cursorData->y;
         /* This is the screen space entry point. */
         GD2d_draw_lines(cursorData->view->State, 8, (SHORT2 *)lines, NULL, NULL);
         break;
      case 3:           /* stop event */
         /* erase line at previous position */
         lines[0] = cursorData->prev_x;
         lines[1] = cursorData->prev_ys;
         lines[2] = cursorData->prev_x;
         lines[3] = cursorData->prev_ye;
         lines[4] = cursorData->prev_xs;
         lines[5] = cursorData->prev_y;
         lines[6] = cursorData->prev_xe;
         lines[7] = cursorData->prev_y;
         /* This is the screen space entry point. */
         GD2d_draw_lines(cursorData->view->State, 4, (SHORT2 *)lines, NULL, NULL);
         break;
      default:          /* unknow event */
	 ERRerror("GDdraw_cursor_update", 0, ERR_ORIG, "Unsupported event");
         break;
   }
   /* save previous positions for erasure on other events */
   cursorData->prev_x = cursorData->x;
   cursorData->prev_y = cursorData->y;
   cursorData->prev_xs = cursorData->xs;
   cursorData->prev_ys = cursorData->ys;
   cursorData->prev_xe = cursorData->xe;
   cursorData->prev_ye = cursorData->ye;

   GDstack_pop(cursorData->view);

   /* If it is not immediate mode and the state is
      not the end event, don't map the point back to
      world space.
   */
   if (!cursorData->immed) {
      if (cursorData->state != 3)
         return(1);
   }

   /* We have drawn the cursor, now determine if we need to
      map the point back to world space and create an output
      mesh.
      The minimum required is that we have a camera attached
      as input. Optionally, we can also have an object.
   */
   if (cursorData->camera) {
      /* Get the full matrix associated with the object. This
         includes the camera matrix and all object matricies
         including the object's.
      */
      GDmap_get_matrix(cursorData->view, cursorData->camera,
		cursorData->object, &mat[0][0], 1);
      /* Take the inverse of the matrix. */
      MATmat4_inverse(invmat, mat);

      sspoint[0] = cursorData->x;
      sspoint[1] = cursorData->y;
      GDmap_ss_to_2dworld(cursorData->camera, &invmat[0][0], 1, &sspoint[0],
	&wpoint[0]);
      GDdraw2d_write_point(elem_id, wpoint);
   }
   return(1);
}

/* Map screen space coordinates back to model space
   given a view, camera and optional object. The other
   input is a buffer containing screen space lines, boxes
   and freehand regions.
*/
/* 64-bit porting. Only Modified Internally */
int GDmap2d_ss_update(OMobj_id elem_id)
{
   OMobj_id view_id, cam_id, obj_id, tmp_id;
   GDview *view;
   GDcamera *camera;
   GDobject *object;
   float mat[4][4], invmat[4][4];
   ROIData buf;
   int seq, mode;
   xp_long i, npoints;
   float *lpoints, *lptr;	/* Buffer for lines */
   short *bpoints;		/* Buffer for box points */
   short *bufptr;		/* pointer into freehand buffer */
   char *user_name;		/* user name for cell set */

   /* Get the sequence number of the update function so
      we can determine what has actually changed.
   */
   tmp_id = OMfind_subobj(elem_id, OMstr_to_name("upd_func"), OM_OBJ_RD);
   if (OMis_null_obj(tmp_id))
      return(0);
   seq = OMget_obj_seq(tmp_id, OMnull_obj, 0);

   /* Read the information from the ROI buffer.
      Need to ARRfree the data that I have read.
      If we can't read the buffer - for example, the
      input buffer has been disconnected, clear the
      output mesh.
   */
   if (!ROIread(elem_id, &buf)) {
      tmp_id = OMfind_subobj(elem_id, OMstr_to_name("out_mesh"), OM_OBJ_RW);
      if (OMis_null_obj(tmp_id))
         return(0);
      UTILclear_mesh(tmp_id);
      return(1);
   }

   ROIget_color(elem_id, &buf);

   /* Get the mode flag. A value of 0 means create a new
      field from the input buffer. A value of 1 means to
      append to the existing field.
   */
   tmp_id = OMfind_subobj(elem_id, OMstr_to_name("mode"), OM_OBJ_RD);
   if (OMis_null_obj(tmp_id))
      return(0);
   if (OMget_int_val(tmp_id, &mode) != OM_STAT_SUCCESS)
      mode = 0;

   /* If the user has requested a clear, clear the output field.
      Check this first in case the input buffer is empty, we want
      to make sure this request is honored.
   */
   tmp_id = OMfind_subobj(elem_id, OMstr_to_name("clear"), OM_OBJ_RD);
   if (OMis_null_obj(tmp_id))
      return(0);
   if (OMget_obj_seq(tmp_id, OMnull_obj, OM_SEQ_VAL) > seq) {
      tmp_id = OMfind_subobj(elem_id, OMstr_to_name("out_mesh"), OM_OBJ_RW);
      if (OMis_null_obj(tmp_id))
         return(0);
      UTILclear_mesh(tmp_id);
      if (buf.nfree)
        ARRfree(buf.freebuf);
      return(1);
   }

   /* Make sure there is something to map to world space. */
   if (!(buf.npoints || buf.nlines || buf.nbox || buf.nfree)) {
      /* There isn't and the mode is NOT append, clear the
         output field and return.
      */
      if (!mode) {
         tmp_id = OMfind_subobj(elem_id, OMstr_to_name("out_mesh"), OM_OBJ_RW);
         if (OMis_null_obj(tmp_id))
            return(0);

	 UTILclear_mesh(tmp_id);
         return(1);
      }
      /* There isn't and the mode is append, so return.
	 This will keep anybody downsteam from getting
	 activated when the user does a clear buffer on the
	 ROI being built.
      */
      else return(1);
   }

   tmp_id = OMfind_subobj(elem_id, OMstr_to_name("view_in"), OM_OBJ_RD);
   if (OMis_null_obj(tmp_id))
      return(0);
   if (OMget_obj_val(tmp_id, &view_id) != OM_STAT_SUCCESS)
      return(0);
   view = (GDview *)GDget_local(view_id, (OMpfi)GDview_init);
   if (view == NULL || view->Winfo == NULL || view->State == NULL)
      return(0);

   tmp_id = OMfind_subobj(elem_id, OMstr_to_name("cam_in"), OM_OBJ_RD);
   if (OMis_null_obj(tmp_id))
      return(0);
   if (OMget_obj_val(tmp_id, &cam_id) != OM_STAT_SUCCESS) {
      ERRerror("GDmap2d_ss_update", 0, ERR_ORIG, "Can't create output without camera");
      return(0);
   }
   camera = (GDcamera *)GDget_local(cam_id, (OMpfi)GDcamera_create);
   if (!camera)
      return(0);

   tmp_id = OMfind_subobj(elem_id, OMstr_to_name("obj_in"), OM_OBJ_RD);
   if (OMis_null_obj(tmp_id))
      return(0);

   /* The object is optional. If we don't have one the points
      are mapped back to the camera's space.
   */
   if (OMget_obj_val(tmp_id, &obj_id) != OM_STAT_SUCCESS)
      object = NULL;
   else object = (GDobject *)GDget_local(obj_id, (OMpfi)GDobject_create);

   /* Get the user name. This name will be written as the user
      name in the cell set.
   */
   tmp_id = OMfind_subobj(elem_id, OMstr_to_name("name"), OM_OBJ_RD);
   if (OMis_null_obj(tmp_id))
      user_name = NULL;
   else {
      if (OMget_str_val(tmp_id, &user_name, 0) != OM_STAT_SUCCESS)
         user_name = NULL;
      else {
	 /* If we have the empty string make sure we don't
	    write it to the output field.
         */
	 if (user_name[0] == '\0') {
	    free(user_name);
	    user_name = NULL;
	 }
      }
   }

   /* Get the full matrix associated with the object. This
      includes the camera matrix and all object matricies
      including the object's.
   */
   GDmap_get_matrix(view, camera, object, &mat[0][0], 1);
   /* Take the inverse of the matrix. */
   MATmat4_inverse(invmat, mat);

   /* Allocate enough space for all output points.
      Each point has 1 point. Each line has 2 points.
      Each box has 5 points - we currently convert it to
      a polyline.
   */
   npoints = buf.npoints + (buf.nlines << 1) + (buf.nbox * 5);
   bufptr = &buf.freebuf[0];
   for (i=0; i<buf.nfree; i++) {
      npoints += bufptr[0];
      bufptr += (bufptr[0] << 1) + 1;
   }

   /* If we somehow managed to get activated but have nothing to
      output to the field, exit
   */
   if (!npoints)
      return(0);

   /* We now have the total number or points. Double that
      since each point has a x and y.
   */
   ALLOCN(lpoints, float, npoints * 10, "can't alloc point buffer");
   if (buf.nbox)
      ALLOCN(bpoints, short, buf.nbox * 10, "can't alloc box point buffer");

   /* Map the points to world space. */
   lptr = &lpoints[0];
   if (buf.npoints)
      GDmap_ss_to_2dworld(camera, &invmat[0][0], buf.npoints,
	buf.pointbuf, lptr);

   /* Map the lines to world space. */
   lptr += buf.npoints << 1;
   if (buf.nlines)
      GDmap_ss_to_2dworld(camera, &invmat[0][0], buf.nlines*2,
	buf.linebuf, lptr);

   /* Map the boxes to world space. */
   lptr += buf.nlines << 2;
   if (buf.nbox) {
      GDmap_ss_box_to_points(&buf, bpoints);
      GDmap_ss_to_2dworld(camera, &invmat[0][0], buf.nbox*5,
	bpoints, lptr);
   }

   /* Map the freehand regions to world space. */
   lptr += buf.nbox * 10;

   if (buf.nfree) {
      /* Each line has two verts with x and y. */
      bufptr = &buf.freebuf[0];
      for (i=0; i<buf.nfree; i++) {
         GDmap_ss_to_2dworld(camera, &invmat[0][0], bufptr[0],
		&bufptr[1], lptr);
         lptr += bufptr[0] << 1;
         bufptr += (bufptr[0] << 1) + 1;
      }
   }
   GDmap2d_write_mesh(elem_id, mode, &buf, npoints, lpoints, user_name);

   /* Need to free up the data we got during ROIread. */
   free(lpoints);
   if (buf.nbox)
      free(bpoints);
   if (buf.npoints)
      ARRfree(buf.pointbuf);
   if (buf.nlines)
      ARRfree(buf.linebuf);
   if (buf.nbox)
      ARRfree(buf.boxbuf);
   if (buf.nfree)
      ARRfree(buf.freebuf);
   if (user_name)
      free(user_name);
   return(1);
}

int GDedit_mesh_create(OMobj_id elem_id)
{
   OMobj_id ptr_id;
   EditMeshData *buf;

   /* Allocate data and initialize it. */
   ALLOCN(buf, EditMeshData, 1, "can't allocate EditMeshData");

   /* Make the initial allocation of all the vert arrays */
   buf->vert_size = 10;
   ALLOCN(buf->verts, FLOAT3, buf->vert_size, "can't allocate verts");
   ALLOCN(buf->tverts, FLOAT3, buf->vert_size, "can't allocate verts");
   ALLOCN(buf->sverts, SHORT2, buf->vert_size, "can't allocate verts");
   ALLOCN(buf->ssverts, SHORT2, buf->vert_size, "can't allocate verts");
   ALLOCN(buf->esverts, SHORT2, buf->vert_size, "can't allocate verts");
   ALLOCN(buf->w2verts, FLOAT2, buf->vert_size, "can't allocate verts");
   ALLOCN(buf->w3verts, FLOAT3, buf->vert_size, "can't allocate verts");

   /* find the local ptr element and save the local structure */
   ptr_id = OMfind_subobj(elem_id, OMstr_to_name("local_ptr"), OM_OBJ_RW);
   if (OMis_null_obj(ptr_id))
      return(0);
   else OMset_ptr_val(ptr_id, buf, 0);
   return(1);
}

int GDedit_mesh_delete(OMobj_id elem_id)
{
   OMobj_id ptr_id;
   EditMeshData *buf;

   /* find the local ptr element and save the local structure */
   ptr_id = OMfind_subobj(elem_id, OMstr_to_name("local_ptr"), OM_OBJ_RW);
   if (OMis_null_obj(ptr_id))
      return(0);
   if (OMget_ptr_val(ptr_id, (void **)&buf, 0) != OM_STAT_SUCCESS)
      return(0);

   /* Free all the vert arrays */
   free(buf->verts);
   free(buf->tverts);
   free(buf->sverts);
   free(buf->ssverts);
   free(buf->esverts);
   free(buf->w2verts);
   free(buf->w3verts);

   /* Free any references we have to field data. */
   GDedit_mesh_free_fieldinfo(elem_id, buf);

   /* Clear all buffers used for drawing */
   free(buf);

   GDclear_local(elem_id);
   return(1);
}

int GDedit_mesh_update(OMobj_id elem_id)
{
   EditMeshData *buf;
   OMobj_id tmp_id, ptr_id, seq_id, state_id, done_id, remove_id;
   int seq, mode = 0, immed = 0;

   /* find the local ptr element and get the EditMeshData structure */
   ptr_id = OMfind_subobj(elem_id, OMstr_to_name("local_ptr"), OM_OBJ_RW);
   if (OMis_null_obj(ptr_id))
      return(0);
   if (OMget_ptr_val(ptr_id, (void **)&buf, 0) != OM_STAT_SUCCESS)
      return(0);

   /* Every time the update function runs the sequence number goes up. This
      means the sequence number of the element itself will be the same
      as the sequence number of the update function after the update function
      executes.  We can use this fact to determine if any of the sub-elements
      has changed since the last time we ran.  If any of the elements have a
      sequence number higher than this, we know that the element has been
      modified since the last execution.
   */
   seq_id = OMfind_subobj(elem_id, OMstr_to_name("upd_func"), OM_OBJ_RD);
   if (OMis_null_obj(seq_id))
      return(0);
   seq = OMget_obj_seq(seq_id, OMnull_obj, 0);

   /* Get view & camera related inputs. If we can't get
      the inputs successfully, quit before we get in
      trouble trying to use them.
   */
   if (!GDedit_mesh_get_inputs(elem_id, buf))
      return(1);

   /* Determine if we are to update immediately on
      the button up event or should we wait for an
      explicit confirmation.
   */
   if (!GDget_int_val(elem_id, "immed", &immed))
      immed = 0;
   buf->immed = immed;

   /* There are several paths for the execution of the module:
      picked_object changed or state changed.
      If the picked object changed,, we need to hilight the
      primitive selected.
      If the state changed, we need to move the selected
      primitive or point.
      If done has changed, we write the moved point(s) back to
      the field.
      If remove has changed, we remove the selected point/prim/cell.
   */
   if (!GDget_refer_db(elem_id, "picked_obj", &buf->pobj_id))
      return(0);
   if (OMget_obj_seq(buf->pobj_id, OMnull_obj, OM_SEQ_VAL) > seq) {
      /* Get the mode flag. A value of 0 means edit points, a value
         of 1 means edit primitives, a value of 2 means edit cells.
	 Do this at the start of the edit sequence so we don't end up
	 with the mode changing inside the sequence.
      */
      tmp_id = OMfind_subobj(elem_id, OMstr_to_name("mode"), OM_OBJ_RD);
      if (!OMis_null_obj(tmp_id)) {
         if (OMget_int_val(tmp_id, &mode) != OM_STAT_SUCCESS)
            mode = 0;
      }
      buf->mode = mode;

      /* If we have moved a primitive and we are in XOR mode, we
	 are about to move another primitive without having
	 confirmed the edit on the previous one. This will only
	 happen if we are NOT in immediate mode. We really should
	 just erase the previous point/prim/cell that was highlighted
	 but this is a lot easier.
      */
      if (buf->prim_moved && (buf->draw_mode == 1)) {
         buf->view->status = GDview_status(buf->view);
         GDview_call_func(buf->view, "view_refresh", 0);
      }

      /* Clear the primitive moved flag if we just picked an object. */
      buf->prim_moved = 0;

      /* Get the pickinfo first, then the fieldinfo. The second
	 routine has a dependency on the first in certain cases.
      */
      if (!GDedit_mesh_get_pickinfo(elem_id, buf))
	 return(1);
      /* We only need to the field data out for read to hilite. */
      if (!GDedit_mesh_get_fieldinfo(elem_id, buf, OM_GET_ARRAY_RD, OM_GET_ARRAY_RD))
	 return(1);
      GDedit_mesh_hilite(elem_id, buf);
   }

   /* Now check to see if the state has changed. It is possible that
      both the picked object and state change at the same time, if
      the pick is generated with a UItwoPoint. Handling this case
      gives use the ability to pick/hilite/move with one smooth motion.
   */
   state_id = OMfind_subobj(elem_id, OMstr_to_name("state"), OM_OBJ_RD);
   if (OMis_null_obj(state_id))
      return(0);
   if (OMget_obj_seq(state_id, OMnull_obj, OM_SEQ_VAL) > seq) {
      OMget_int_val(state_id, &buf->state);

      /* Get xy screen space position that is associated with interactor. */
      if (!GDget_int_val(elem_id, "x", &buf->x))
         return(0);
      if (!GDget_int_val(elem_id, "y", &buf->y))
         return(0);
      GDedit_mesh_modify(elem_id, buf);
      /* if the state is now button up - update the mesh */
      if (buf->state == 3) {
	 if (buf->immed)
	    GDedit_mesh_updfield(elem_id, buf);
      }
   }
   else {
      done_id = OMfind_subobj(elem_id, OMstr_to_name("done"), OM_OBJ_RD);
      if (OMis_null_obj(done_id))
         return(0);
      if (OMget_obj_seq(done_id, OMnull_obj, OM_SEQ_VAL) > seq) {
	  GDedit_mesh_updfield(elem_id, buf);
      }
      else {
         remove_id = OMfind_subobj(elem_id, OMstr_to_name("remove"), OM_OBJ_RD);
         if (OMis_null_obj(remove_id))
            return(0);
         if (OMget_obj_seq(remove_id, OMnull_obj, OM_SEQ_VAL) > seq) {
	    GDedit_mesh_delfield(elem_id, buf);
         }
      }
   }
   return(1);
}

/***************************/
/* Map utilities - Public. */
/***************************/

/* Maps 2D screen space points to 2D world space. */
/* 64-bit porting. Directly Modified */
void GDmap_ss_to_2dworld(GDcamera *camera, float *xform,
                         xp_long nss, short *sspts, float *points)
		/* nss - number of points - NOT primitives */
{
   float *tpoints;
   xp_long i;

   ALLOCN(tpoints, float, nss*2, "can't alloc line buffer");

   /* The first step is to correct the y positions which
      are from the windowing system which has 0,0 in the
      upper left to world space where 0,0 is in the center
      and xmax, ymax is in the upper left.
      While doing this we also transform the points from
      their screen space values to a -1,1 range.
   */
   for (i=0; i<nss; i++) {
      tpoints[i*2] = -1.0 + ((float)(sspts[i*2] - camera->vp_xmin) /
	(camera->vp_xmax - camera->vp_xmin)) * 2.0;
      tpoints[i*2+1] = -1.0 + ((float)(camera->vp_ymax - sspts[i*2+1] -
	camera->vp_ymin) / (camera->vp_ymax - camera->vp_ymin)) * 2.0;
   }
   MATxform_verts2(nss, (Matr2*)tpoints, (Matr4*)xform, (Matr2*)points);

   /* Free the temporary points array. */
   free(tpoints);
}

/* Maps 2D screen space points to 3D world space. */
/* 64-bit porting. Directly Modified */
void GDmap_ss_to_3dworld(GDcamera *camera, float *xform, xp_long nss, short *sspts,
                         float *opoints, float *points, int nspace)
	/* nss - number of points - NOT primitives */
	/* opoints - original points - contain Z values */
{

   float *tpoints;
   xp_long i;

   ALLOCN(tpoints, float, nss*3, "can't alloc line buffer");

   /* The first step is to correct the y positions which
      are from the windowing system which has 0,0 in the
      upper left to world space where 0,0 is in the center
      and xmax, ymax is in the upper left.
      While doing this we also transform the points from
      their screen space values to a -1,1 range.
   */
   for (i=0; i<nss; i++) {
      tpoints[i*3] = -1.0 + ((float)(sspts[i*2] - camera->vp_xmin) /
	(camera->vp_xmax - camera->vp_xmin)) * 2.0;
      tpoints[i*3+1] = -1.0 + ((float)(camera->vp_ymax - sspts[i*2+1] -
	camera->vp_ymin) / (camera->vp_ymax - camera->vp_ymin)) * 2.0;
      /* Use the Z value from the original point */
      tpoints[i*3+2] = opoints[i*3+2];
   }
   if (nspace == 2)
      MATxform_verts3to2(nss, (Matr3*)tpoints, (Matr4*)xform, (Matr2*)points, 1);
   else if (nspace == 3)
      MATxform_verts3(nss, (Matr3*)tpoints, (Matr4*)xform, (Matr3*)points);

   /* Free the temporary points array. */
   free(tpoints);

}

void GDmap_get_matrix(GDview *view, GDcamera *camera, GDobject *object,
                      float *xform, int flag)
	/* flag - if 1=include object's matrix */
{
   int i;

   MATmat4_identity((Matr4*)xform);

   /* The view and camera are required inputs. */
   if (!view || !camera) {
      ERRerror("GDmap_get_matrix", 0, ERR_ORIG,
               "View and camera are required");
      return;
   }

   /* Push start of rendering marker onto the stack so we can
      restore original attributes when we are done.
   */
   GDstack_push(view);
   /* Set up the camera and object matricies in the state
      so we can later get the full matrix used to transform
      from world to screen space. We need the inverse of this
      to transform our screen space position to a region
      in world space.
   */
   GDcamera_set_attrs(view, camera, 0);

   /* We need to build the matrix that includes all the
      objects from the one specified back upto and including top
      in order to do this correctly.  The object hierarchy
      is traversed building up fullmat as we go until we find
      the object specified. This matrix that has been built
      at this point is returned.

      This procedure may not always work. The case that causes
      a problem is the one where an object has more than 1 parent.
      We will only return the correct matrix in this case if
      the object we are looking for is the first one we find.
      One potential way to handle this is to use the pick path
      in the objects pick information. However, this will only
      work if we use the picked object from the view. Otherwise,
      we have no guarentee that the pick information has been
      written.
   */
   for (i=0; i<camera->nobjs; i++) {
      if (GDmap2d_obj_concat(view, camera->objs[i], object, xform, flag))
         break;
   }
   GDstack_pop(view);
}

/* obj1 is the object to be "rendered".
   obj2 is the object we are looking for.
*/
int GDmap2d_obj_concat(GDview *view, GDobject *obj1, GDobject *obj2,
                       float *xform, int flag)
	/* flag - if =1 include object's matrix */
{
   GDstate *state = view->State;
   GDxform *lxform= obj1->Xform;
   int i, retval = 0;
   OMpfi state_func;
   GDobject *tmp_obj;

   GDstack_push(view);

   /* If we don't have an object to look for - quit now. */
   if (!obj2) {
      if (GDstate_get_func(view, "state_update_matrix", &state_func))
         (*state_func)(view);
      memcpy(xform, state->fullmat, 16*sizeof(float));
      retval = 1;
   }
   else {
      /* If we have found the object we are looking for
         return the matrix at this point and stop traversing
         the object hierarchy. Don't include th object that
	 we are looking for.
      */
      if (!flag && obj1 == obj2) {
         memcpy(xform, state->fullmat, 16*sizeof(float));
         retval = 1;
      }
      else {
         if (lxform) {
            if (GDstate_get_func(view, "state_concat_obj_matrix", &state_func))
				{
					tmp_obj = view->State->obj;
					view->State->obj = obj1;
               (*state_func)(view, obj1->Xform->matrix);
					view->State->obj = tmp_obj;
				}
			}
         if (GDstate_get_func(view, "state_update_matrix", &state_func))
			{
				tmp_obj = view->State->obj;
				view->State->obj = obj1;
            (*state_func)(view);
				view->State->obj = tmp_obj;
			}

         /* If we have found the object we are looking for
            return the matrix at this point and stop traversing
            the object hierarchy.
         */
         if (obj1 == obj2) {
            memcpy(xform, state->fullmat, 16*sizeof(float));
            retval = 1;
         }
	 else {
            /* process any children that exist */
            if (obj1->nchildren) {
               for (i=0; i<obj1->nchildren; i++) {
                  if (GDmap2d_obj_concat(view, obj1->children[i], obj2,
			xform, flag)) {
                     retval = 1;
	             break;
                  }
               }
            }
         }
      }
   }
   GDstack_pop(view);
   return(retval);
}


/*****************************************/
/* Map utilities - Private to this file. */
/*****************************************/

/* 64-bit porting. Only Modified Internally */
static void GDmap_ss_box_to_points(ROIData *buf, short *pts)
{
   xp_long i;
   short x1, y1, x2, y2;
   short *sptr;

   sptr = pts;
   for (i=0; i<buf->nbox*4; i+=4) {
      /* Sort the points so x1,y1 is the lower
         left and x2,y2 is the upper right.
      */
      if (buf->boxbuf[i] <= buf->boxbuf[i+2]) {
         x1 = buf->boxbuf[i];
         x2 = buf->boxbuf[i+2];
      }
      else {
         x1 = buf->boxbuf[i+2];
         x2 = buf->boxbuf[i];
      }
      if (buf->boxbuf[i+1] <= buf->boxbuf[i+3]) {
         y1 = buf->boxbuf[i+1];
         y2 = buf->boxbuf[i+3];
      }
      else {
         y1 = buf->boxbuf[i+3];
         y2 = buf->boxbuf[i+1];
      }

      /* Now convert the box to a polyline. */
      *sptr++ = x1;	/* lower left */
      *sptr++ = y1;
      *sptr++ = x2;	/* lower right */
      *sptr++ = y1;
      *sptr++ = x2;	/* upper right */
      *sptr++ = y2;
      *sptr++ = x1;	/* upper left */
      *sptr++ = y2;
      *sptr++ = x1;	/* lower left - close box */
      *sptr++ = y1;
   }
}

/* Write the mesh.
   Depending on the mode, we will either create a "new" field
   from the input buffer (mode = 0). Or we will append the
   input buffer to the field (mode = 1).
*/
/* 64-bit porting. Directly Modified */
static void GDmap2d_write_mesh(OMobj_id elem_id, int mode, ROIData *buf,
                               xp_long npoints, float *points, char *name)
	/* mode - 0 = clear, 1 = append */
{
   OMobj_id mesh_id, cset_id;
   /* float mat[4][4]; */
   float *coord = NULL, *new_coord;
   xp_long coord_size, new_coord_size;
   short *bufptr;
   int ncell_sets;
   xp_long conn_idx, cur_points;
   xp_long i;
   int j;
   xp_long psize, lsize, bsize, fsize;
   xp_long *pconn, *pconn_ptr;
   xp_long *lconn, *lconn_ptr;
   xp_long *bconn, *bconn_ptr;
   xp_long *fconn, *fconn_ptr;

   /* Paranoia check - we shouldn't even come in here unless
      we have points to write to the field.
   */
   if (npoints == 0)
      return;

   mesh_id = OMfind_subobj(elem_id, OMstr_to_name("out_mesh"), OM_OBJ_RW);
   if (OMis_null_obj(mesh_id)) {
      ERRerror("GDmap2d_write_mesh", 0, ERR_ORIG, "Can't find output mesh");
      return;
   }

   /* Check mode: If 0 produce new field. If 1 append to
      existing field.
   */
   if (!mode) {
      /* Set up the mesh. This is the grid plus cells.
         Set up the grid: we need nspace, the number of
         nodes and the coordinates.
      */
      if (FLDset_nspace(mesh_id, 2) != 1) {
         ERRerror("GDmap2d_write_mesh", 0, ERR_ORIG, "Can't write nspace");
         return;
      }
      if (FLDset_nnodes(mesh_id, npoints) != 1) {
         ERRerror("GDmap2d_write_mesh", 0, ERR_ORIG, "Can't write nnodes");
         return;
      }
      if (FLDget_coord(mesh_id, (float **)&coord, &coord_size, OM_GET_ARRAY_WR) != 1) {
         ERRerror("GDmap2d_write_mesh", 0, ERR_ORIG, "Can't get coordinates");
         return;
      }
      /* Copy the buffer we have filled up to the array the OM has
	 allocated for us.
      */
      memcpy(coord, points, npoints*2*sizeof(float));
      /* Make sure we generate an activation */
      ARRfree(coord);

      /* Set the xform in the mesh to the identity.
	 The xform is tied to the input xform so we
	 do NOT want to set it !!!
      MATmat4_identity(mat);
      FLDset_xform(mesh_id, mat);
      */

      if (FLDset_ncell_sets(mesh_id, 0) != 1) {
         ERRerror("GDmap2d_write_mesh", 0, ERR_ORIG, "Can't set number of cell sets");
	 return;
      }

      /* Set the number of cell sets to 0 and set the connectivity
	 index to zero.
      */
      ncell_sets = 0;
      conn_idx = 0;
   }
   /* Else append to the cell sets that already exist. */
   else {
      /* Get the existing number of nodes. If we fail
	 take it to mean that we don't have a field yet.
      */
      if ((FLDget_nnodes(mesh_id, &cur_points) != 1)  || cur_points == 0) {
	 cur_points = 0;
         if (FLDset_nspace(mesh_id, 2) != 1) {
            ERRerror("GDmap2d_write_mesh", 0, ERR_ORIG, "Can't write nspace");
            return;
         }
         if (FLDset_ncell_sets(mesh_id, 0) != 1) {
            ERRerror("GDmap2d_write_mesh", 0, ERR_ORIG, "Can't set number of cell sets");
	    return;
         }

         /* Set the xform in the mesh to the identity.
	    The xform is tied to the input xform so we
	    do NOT want to set it !!!
         MATmat4_identity(mat);
         FLDset_xform(mesh_id, mat);
	 */
      }
      else {
         /* Get the existing coordinates. */
         if (FLDget_coord(mesh_id, (float **)&coord, &coord_size, OM_GET_ARRAY_RD) != 1) {
            ERRerror("GDmap2d_write_mesh", 0, ERR_ORIG, "Can't get coordinates");
            return;
         }
      }

      if (FLDset_nnodes(mesh_id, cur_points + npoints) != 1) {
         ERRerror("GDmap2d_write_mesh", 0, ERR_ORIG, "Can't write nnodes");
         return;
      }
      if (FLDget_coord(mesh_id, (float **)&new_coord, &new_coord_size,
     		 OM_GET_ARRAY_WR) != 1) {
         ERRerror("GDmap2d_write_mesh", 0, ERR_ORIG, "Can't get coordinates");
         return;
      }
      /* Copy the buffer we have filled up to the array the OM has
	 allocated for us. First, copy the existing points if there
	 are any, then copy the new points.
      */
      if (cur_points)
         memcpy(new_coord, coord, cur_points*2*sizeof(float));
      memcpy(&new_coord[cur_points*2], points, npoints*2*sizeof(float));
      /* Make sure we generate an activation */
      ARRfree(new_coord);
      if (coord) ARRfree(coord);

      /* Get the existing number of cell sets and set the connectivity
	 index to the current number of points.
      */
      if (FLDget_ncell_sets(mesh_id, &ncell_sets) != 1)
	 ncell_sets = 0;
      conn_idx = cur_points;
   }

   /* Set up the cell sets: We need to add the cell set with the
      type, set the number of cells and the connectivity.
   */
   if (buf->npoints) {
      if (FLDadd_cell_set(mesh_id, "Point") != 1) {
         ERRerror("GDmap2d_write_mesh", 0, ERR_ORIG, "Can't add point cell set");
         return;
      }
      if (FLDget_cell_set(mesh_id, ncell_sets++, &cset_id) != 1) {
         ERRerror("GDmap2d_write_mesh", 0, ERR_ORIG, "Can't get point cell set");
         return;
      }
      if (name) {
         if (FLDset_cell_set_user_name(cset_id, name) != 1) {
            ERRerror("GDmap2d_write_mesh", 0, ERR_ORIG, "Can't set user cell set name");
            return;
         }
      }
      if (FLDset_ncells(cset_id, buf->npoints) != 1) {
         ERRerror("GDmap2d_write_mesh", 0, ERR_ORIG, "Can't set number of points");
         return;
      }

      if (FLDget_node_connect(cset_id, &pconn, &psize, OM_GET_ARRAY_WR) != 1) {
         ERRerror("GDmap2d_write_mesh", 0, ERR_ORIG, "Can't get point cell set connectivity");
         return;
      }
      /* Build the connectivity */
      pconn_ptr = pconn;
      for (i=0; i<buf->npoints; i++)
	 *pconn_ptr++ = conn_idx + i;
      ARRfree(pconn);
      conn_idx += buf->npoints;

      /* Check to see if we are to set the cell colors */
      if (buf->color_flag) {
         if (FLDset_cell_set(cset_id, "Cell_Data_Set") != 1) {
            ERRerror("GDmap2d_write_mesh", 0, ERR_ORIG, "Can't add cell data");
            return;
         }
         GDmap2d_write_mesh_cdata(cset_id, buf, buf->npoints);
      }
   }
   if (buf->nlines) {
      if (FLDadd_cell_set(mesh_id, "Line") != 1) {
         ERRerror("GDmap2d_write_mesh", 0, ERR_ORIG, "Can't add line cell set");
         return;
      }
      if (FLDget_cell_set(mesh_id, ncell_sets++, &cset_id) != 1) {
         ERRerror("GDmap2d_write_mesh", 0, ERR_ORIG, "Can't get line cell set");
         return;
      }
      if (name) {
         if (FLDset_cell_set_user_name(cset_id, name) != 1) {
            ERRerror("GDmap2d_write_mesh", 0, ERR_ORIG, "Can't set user cell set name");
            return;
         }
      }
      if (FLDset_ncells(cset_id, buf->nlines) != 1) {
         ERRerror("GDmap2d_write_mesh", 0, ERR_ORIG, "Can't set number of lines");
         return;
      }

      if (FLDget_node_connect(cset_id, &lconn, &lsize, OM_GET_ARRAY_WR) != 1) {
         ERRerror("GDmap2d_write_mesh", 0, ERR_ORIG, "Can't get line cell set connectivity");
         return;
      }
      /* Build the connectivity */
      lconn_ptr = lconn;
      for (i=0; i<buf->nlines; i++) {
	 *lconn_ptr++ = conn_idx + (i * 2);
	 *lconn_ptr++ = conn_idx + (i * 2 + 1);
      }
      ARRfree(lconn);
      conn_idx += buf->nlines * 2;

      /* Check to see if we are to set the cell colors */
      if (buf->color_flag) {
         if (FLDset_cell_set(cset_id, "Cell_Data_Set") != 1) {
            ERRerror("GDmap2d_write_mesh", 0, ERR_ORIG, "Can't add cell data");
            return;
         }
         GDmap2d_write_mesh_cdata(cset_id, buf, buf->nlines);
      }
   }
   if (buf->nbox) {
      if (FLDadd_cell_set(mesh_id, "Polyline") != 1) {
         ERRerror("GDmap2d_write_mesh", 0, ERR_ORIG, "Can't add box cell set");
         return;
      }
      if (FLDget_cell_set(mesh_id, ncell_sets++, &cset_id) != 1) {
         ERRerror("GDmap2d_write_mesh", 0, ERR_ORIG, "Can't add box cell set");
         return;
      }
      if (name) {
         if (FLDset_cell_set_user_name(cset_id, name) != 1) {
            ERRerror("GDmap2d_write_mesh", 0, ERR_ORIG, "Can't set user cell set name");
            return;
         }
      }
      if (FLDset_npolys(cset_id, buf->nbox) != 1) {
         ERRerror("GDmap2d_write_mesh", 0, ERR_ORIG, "Can't set number of boxes");
         return;
      }

      if (FLDget_poly_connect(cset_id, &bconn, &bsize, OM_GET_ARRAY_WR) != 1) {
         ERRerror("GDmap2d_write_mesh", 0, ERR_ORIG, "Can't get box cell set connectivity");
         return;
      }
      /* Build the connectivity. Each polyline for a box is 5 points. */
      j = 0;
      bconn_ptr = bconn;
      for (i=0; i<buf->nbox; i++) {
	 *bconn_ptr++ = conn_idx + j;
	 j += 4;
	 *bconn_ptr++ = conn_idx + j;
	 j++;
      }
      ARRfree(bconn);
      conn_idx += buf->nbox * 5;

      /* Check to see if we are to set the cell colors */
      if (buf->color_flag) {
         if (FLDset_cell_set(cset_id, "Cell_Data_Set_Poly") != 1) {
            ERRerror("GDmap2d_write_mesh", 0, ERR_ORIG, "Can't add cell data");
            return;
         }
         GDmap2d_write_mesh_cdata(cset_id, buf, buf->nbox);
      }
   }
   if (buf->nfree) {
      if (FLDadd_cell_set(mesh_id, "Polyline") != 1) {
         ERRerror("GDmap2d_write_mesh", 0, ERR_ORIG, "Can't add freehand cell set");
         return;
      }
      if (FLDget_cell_set(mesh_id, ncell_sets++, &cset_id) != 1) {
         ERRerror("GDmap2d_write_mesh", 0, ERR_ORIG, "Can't get freehand cell set");
         return;
      }
      if (name) {
         if (FLDset_cell_set_user_name(cset_id, name) != 1) {
            ERRerror("GDmap2d_write_mesh", 0, ERR_ORIG, "Can't set user cell set name");
            return;
         }
      }
      if (FLDset_npolys(cset_id, buf->nfree) != 1) {
         ERRerror("GDmap2d_write_mesh", 0, ERR_ORIG, "Can't set number of freehand regions");
         return;
      }

      if (FLDget_poly_connect(cset_id, &fconn, &fsize, OM_GET_ARRAY_WR) != 1) {
         ERRerror("GDmap2d_write_mesh", 0, ERR_ORIG, "Can't get freehand cell set connectivity");
         return;
      }
      /* Build the connectivity */
      j = 0;
      bufptr = &buf->freebuf[0];
      fconn_ptr = fconn;
      for (i=0; i<buf->nfree; i++) {
	 *fconn_ptr++ = conn_idx + j;
	 j += bufptr[0];
	 *fconn_ptr++ = conn_idx + j - 1;
         bufptr += (bufptr[0] << 1) + 1;
      }
      ARRfree(fconn);

      /* Check to see if we are to set the cell colors */
      if (buf->color_flag) {
         if (FLDset_cell_set(cset_id, "Cell_Data_Set_Poly") != 1) {
            ERRerror("GDmap2d_write_mesh", 0, ERR_ORIG, "Can't add cell data");
            return;
         }
         GDmap2d_write_mesh_cdata(cset_id, buf, buf->nfree);
      }
   }
}

/* Write out cell data for each of the cells. The data
   is written as the special RGB float color data.
*/
/* 64-bit porting. Directly Modified */
static void GDmap2d_write_mesh_cdata(OMobj_id cell_set_id, ROIData *buf, xp_long n)
{
   xp_long i, size;
   int type;
   float *cdata, *cdata_ptr;

   /* Set up the cell data */
   if (FLDset_cell_data_ncomp(cell_set_id, 1) != 1) {
      ERRerror("GDmap2d_write_mesh_cdata", 0, ERR_ORIG,
               "Can't write cell data ncomp");
      return;
   }
   if (FLDset_cell_data_veclen(cell_set_id, 0, 3) != 1) {
      ERRerror("GDmap2d_write_mesh_cdata", 0, ERR_ORIG,
               "Can't write cell data veclen");
      return;
   }

   if (FLDset_cell_data_id(cell_set_id, 0, GD_COLOR_DATA_ID) != 1) {
      ERRerror("GDmap2d_write_mesh_cdata", 0, ERR_ORIG,
               "Can't write cell data id");
      return;
   }
   /* Have the OM manage the allocation for us. We then
      just need to fill the data in. This will also be more
      efficient since if we fill in a local array, a copy will
      then have to be made.
   */
   type = DTYPE_FLOAT;
   if (FLDget_cell_data(cell_set_id, 0, &type, (char **)&cdata,
	&size, OM_GET_ARRAY_WR) != 1) {
      ERRerror("GDmap2d_write_mesh_cdata", 0, ERR_ORIG,
               "Can't get buffer for cell data");
      return;
   }
   cdata_ptr = cdata;
   for (i=0; i<n; i++) {
      VEC_COPY(cdata_ptr, buf->color);
      cdata_ptr += 3;
   }
   ARRfree(cdata);
}

/*****************************************/
/* ROI utilities - Private to this file. */
/*****************************************/
static int ROIget_inputs(OMobj_id elem_id, ROIData *buf)
{
   OMobj_id ptr_id, tmp_id;

   ptr_id = OMfind_subobj(elem_id, OMstr_to_name("view_in"), OM_OBJ_RW);
   if (OMis_null_obj(ptr_id))
      return(0);
   if (OMget_obj_val(ptr_id, &tmp_id) != OM_STAT_SUCCESS)
      return(0);
   /* get pointer to local view structure so we can find
      the state and winfo.
   */
   buf->view = (GDview *)GDget_local(tmp_id, (OMpfi)GDview_init);
   if (buf->view == NULL || buf->view->Winfo == NULL || buf->view->State == NULL)
      return(0);

   ptr_id = OMfind_subobj(tmp_id, OMstr_to_name("picked_camera"), OM_OBJ_RD);
   if (OMis_null_obj(ptr_id))
      return(0);
   if (OMget_obj_val(ptr_id, &tmp_id) != OM_STAT_SUCCESS)
      return(0);
   buf->camera = (GDcamera *)GDget_local(tmp_id, (OMpfi)GDcamera_create);
   if (!buf->camera)
      return(0);

   /* If mode = 0, clear all lines/boxes/freehand buffers.
      Else if mode = 1, append to the buffers.
   */
   if (!GDget_int_val(elem_id, "mode", &buf->mode))
      buf->mode = 0;

   if (!GDget_int_val(elem_id, "option", &buf->option))
      buf->option = 0;

   /* click draw doesn't have this value - so set it to 0 */
   if (!GDget_int_val(elem_id, "draw_mode", &buf->draw_mode))
      buf->draw_mode = 0;

   return(1);
}

static void ROIget_color(OMobj_id elem_id, ROIData *buf)
{
   if (!GDget_int_val(elem_id, "color", &buf->color_flag))
      buf->color_flag = 0;

   /* Get the color to use. */
   if (!GDget_float_val(elem_id, "red", &buf->color[0]))
      buf->color[0] = 1.0;
   if (!GDget_float_val(elem_id, "green", &buf->color[1]))
      buf->color[1] = 1.0;
   if (!GDget_float_val(elem_id, "blue", &buf->color[2]))
      buf->color[2] = 1.0;
}

static void ROIsetup(GDview *view, ROIData *buf)
{
   OMpfi state_func;

   /* Push start of rendering marker onto the stack so we can
      restore original attributes when we are done.
   */
   GDstack_push(view);

   /* Set the color to be used for drawing */
   GDstate_set_color(view, buf->color);

   /* Tell GD to use the front buffer (ie. window) as the
      drawble. This allows our drawing to be visible even if
      we are in double buffer mode.
   */
   if (GDstate_get_func(view, "state_set_drawable", &state_func))
      (*state_func)(view, 1);
}

/* 64-bit porting. Only Modified Internally */
static void ROIredraw(ROIData *buf, GDview *view)
{
   short *bufptr;
   xp_long i;

   /* With points we only redraw if we are NOT in point mode. In
      that case later processing for the activation will cause
      us to redraw the points.
   */
   if (buf->npoints && (buf->option != ROI_POINT))
      ROIpoint_draw(buf, view, 1);
   if (buf->nlines)
      GD2d_draw_lines(view->State, buf->nlines*2, (SHORT2 *)buf->linebuf, NULL, NULL);
   if (buf->nbox)
      GD2d_draw_rects(view->State, buf->nbox*2, (SHORT2 *)buf->boxbuf, NULL);
   if (buf->nfree) {
      bufptr = &buf->freebuf[0];
      for (i=0; i<buf->nfree; i++) {
         GD2d_draw_polyline(view->State, bufptr[0], (SHORT2 *)&bufptr[1], NULL, NULL, 0);
         bufptr += (bufptr[0] << 1) + 1;
      }
   }
}

/* Clear all the buffer used for drawing but DO NOT free
   the actual buffer. Due to the modules usage - leave
   this up to the caller.
*/
static void ROIclear(ROIData *buf, int flag)
	/* flag - if 1, clear all. if 0, leave existing stuff alone */
{
   if (flag) {
      if (buf->npoints) {
         free(buf->pointbuf);
         buf->pointbuf = NULL;
	 buf->pointbuf_size = 0;
         buf->npoints = 0;
      }
      if (buf->nlines || buf->line_flag) {
         free(buf->linebuf);
         buf->linebuf = NULL;
	 buf->linebuf_size = 0;
         buf->nlines = 0;
         buf->line_flag = 0;
      }
      if (buf->nbox || buf->box_flag) {
         free(buf->boxbuf);
         buf->boxbuf = NULL;
	 buf->boxbuf_size = 0;
         buf->nbox = 0;
         buf->box_flag = 0;
      }
      if (buf->nfree || buf->free_flag) {
         free(buf->freebuf);
         buf->nverts = 0;
         buf->freebuf = NULL;
         buf->freebuf_size = 0;
         buf->curr_size = 0;
         buf->currptr = NULL;
         buf->nfree = 0;
         buf->free_flag = 0;
      }
   }
   else {
      /* Clears existing "finished" things. Leaves in progress stuff
	 alone - as indicated by xxx_flag's. This will be used if
	 we are not in append mode and we are drawing a new region.
      */
      if (buf->npoints) {
         free(buf->pointbuf);
         buf->pointbuf = NULL;
	 buf->pointbuf_size = 0;
         buf->npoints = 0;
      }
      if (buf->nlines) {
         free(buf->linebuf);
         buf->linebuf = NULL;
	 buf->linebuf_size = 0;
         buf->nlines = 0;
      }
      if (buf->nbox) {
         free(buf->boxbuf);
         buf->boxbuf = NULL;
	 buf->boxbuf_size = 0;
         buf->nbox = 0;
      }
      if (buf->nfree) {
         free(buf->freebuf);
         buf->nverts = 0;
         buf->freebuf = NULL;
         buf->freebuf_size = 0;
         buf->curr_size = 0;
         buf->currptr = NULL;
         buf->nfree = 0;
      }

   }
}

/* 64-bit porting. Only Modified Internally */
static int ROIread(OMobj_id elem_id, ROIData *buf)
{
   OMobj_id buf_id, tmp_id;
   int type;
   xp_long size;

   tmp_id = OMfind_subobj(elem_id, OMstr_to_name("in_buf"), OM_OBJ_RW);
   if (OMis_null_obj(tmp_id))
      return(0);
   if (OMget_obj_val(tmp_id, &buf_id) != OM_STAT_SUCCESS)
      return(0);

   /* Read the point buffer. */
   tmp_id = OMfind_subobj(buf_id, OMstr_to_name("npoints"), OM_OBJ_RW);
   if (OMis_null_obj(tmp_id))
      return(0);
   if (OMget_long_val(tmp_id, &buf->npoints) == OM_STAT_SUCCESS) {
      if (buf->npoints) {
         tmp_id = OMfind_subobj(buf_id, OMstr_to_name("pointbuf"), OM_OBJ_RW);
         if (OMis_null_obj(tmp_id))
            return(0);
         size = 0;
	 type = OM_TYPE_SHORT;
	 buf->pointbuf = 0;
         if (OMget_array_sz(tmp_id, &type, (char **)&buf->pointbuf,
	   &size, OM_GET_ARRAY_RD) != OM_STAT_SUCCESS)
            buf->npoints = 0;
      }
   }
   else buf->npoints= 0;

   /* Read the line buffer. */
   tmp_id = OMfind_subobj(buf_id, OMstr_to_name("nlines"), OM_OBJ_RW);
   if (OMis_null_obj(tmp_id))
      return(0);
   if (OMget_long_val(tmp_id, &buf->nlines) == OM_STAT_SUCCESS) {
      if (buf->nlines) {
         tmp_id = OMfind_subobj(buf_id, OMstr_to_name("linebuf"), OM_OBJ_RW);
         if (OMis_null_obj(tmp_id))
            return(0);
         size = 0;
	 type = OM_TYPE_SHORT;
	 buf->linebuf = 0;
         if (OMget_array_sz(tmp_id, &type, (char **)&buf->linebuf,
	   &size, OM_GET_ARRAY_RD) != OM_STAT_SUCCESS)
	    buf->nlines = 0;
      }
   }
   else buf->nlines = 0;

   /* Read the box buffer. */
   tmp_id = OMfind_subobj(buf_id, OMstr_to_name("nbox"), OM_OBJ_RW);
   if (OMis_null_obj(tmp_id))
      return(0);
   if (OMget_long_val(tmp_id, &buf->nbox) == OM_STAT_SUCCESS) {
      if (buf->nbox) {
         tmp_id = OMfind_subobj(buf_id, OMstr_to_name("boxbuf"), OM_OBJ_RW);
         if (OMis_null_obj(tmp_id))
            return(0);
         size = 0;
	 type = OM_TYPE_SHORT;
	 buf->boxbuf = 0;
         if (OMget_array_sz(tmp_id, &type, (char **)&buf->boxbuf,
	   &size, OM_GET_ARRAY_RD) != OM_STAT_SUCCESS)
            buf->nbox = 0;
      }
   }
   else buf->nbox = 0;

   /* Read the freehand buffer. */
   tmp_id = OMfind_subobj(buf_id, OMstr_to_name("nfree"), OM_OBJ_RW);
   if (OMis_null_obj(tmp_id))
      return(0);
   if (OMget_long_val(tmp_id, &buf->nfree) == OM_STAT_SUCCESS) {
      if (buf->nfree) {
         tmp_id = OMfind_subobj(buf_id, OMstr_to_name("freebuf"), OM_OBJ_RW);
         if (OMis_null_obj(tmp_id))
            return(0);
         size = 0;
	 type = OM_TYPE_SHORT;
	 buf->freebuf = 0;
         if (OMget_array_sz(tmp_id, &type, (char **)&buf->freebuf,
	   &size, OM_GET_ARRAY_RD) != OM_STAT_SUCCESS)
            buf->nfree = 0;
         else buf->curr_size = size;
      }
   }
   else buf->nfree= 0;

   return(1);
}

/* 64-bit porting. Only Modified Internally */
static int ROIwrite(OMobj_id elem_id, ROIData *buf, int close)
	/* close - 0 if don't auto-close, 1 if auto-close*/
{
   OMobj_id buf_id, tmp_id;
   int tmp_int, stat = 0;
   xp_long tmp_long;

   buf_id = OMfind_subobj(elem_id, OMstr_to_name("out_buf"), OM_OBJ_RW);
   if (OMis_null_obj(buf_id))
      return(0);

   /* Write out the point buffer. */
   tmp_id = OMfind_subobj(buf_id, OMstr_to_name("npoints"), OM_OBJ_RW);
   if (OMis_null_obj(tmp_id))
      return(0);
   /* Don't set the number if it is the same. This will
      avoid activations that aren't necessary. If get
      fails, it is because it is not set yet (it better be !!).
   */
   if (OMget_long_val(tmp_id, &tmp_long) != OM_STAT_SUCCESS)
      OMset_long_val(tmp_id, buf->npoints);
   else if (tmp_long != buf->npoints)
      OMset_long_val(tmp_id, buf->npoints);

   if (buf->npoints) {
      tmp_id = OMfind_subobj(buf_id, OMstr_to_name("pointbuf"), OM_OBJ_RW);
      if (OMis_null_obj(tmp_id))
         return(0);
      OMset_array(tmp_id, OM_TYPE_SHORT, (char *)buf->pointbuf,
	buf->npoints*2, OM_SET_ARRAY_COPY);
      stat = 1;
   }

   /* Write out the line buffer. */
   tmp_id = OMfind_subobj(buf_id, OMstr_to_name("nlines"), OM_OBJ_RW);
   if (OMis_null_obj(tmp_id))
      return(0);
   /* Don't set the number if it is the same. This will
      avoid activations that aren't necessary. If get
      fails, it is because it is not set yet (it better be !!).
   */
   if (OMget_long_val(tmp_id, &tmp_long) != OM_STAT_SUCCESS)
      OMset_long_val(tmp_id, buf->nlines);
   else if (tmp_long != buf->nlines)
      OMset_long_val(tmp_id, buf->nlines);

   if (buf->nlines) {
      tmp_id = OMfind_subobj(buf_id, OMstr_to_name("linebuf"), OM_OBJ_RW);
      if (OMis_null_obj(tmp_id))
         return(0);
      OMset_array(tmp_id, OM_TYPE_SHORT, (char *)buf->linebuf,
	buf->nlines*4, OM_SET_ARRAY_COPY);
      stat = 1;
   }

   /* Write out the box buffer. */
   tmp_id = OMfind_subobj(buf_id, OMstr_to_name("nbox"), OM_OBJ_RW);
   if (OMis_null_obj(tmp_id))
      return(0);
   /* Don't set the number if it is the same. This will
      avoid activations that aren't necessary.
   */
   if (OMget_long_val(tmp_id, &tmp_long) != OM_STAT_SUCCESS)
      OMset_long_val(tmp_id, buf->nbox);
   else if (tmp_long != buf->nbox)
      OMset_long_val(tmp_id, buf->nbox);

   if (buf->nbox) {
      tmp_id = OMfind_subobj(buf_id, OMstr_to_name("boxbuf"), OM_OBJ_RW);
      if (OMis_null_obj(tmp_id))
         return(0);
      OMset_array(tmp_id, OM_TYPE_SHORT, (char *)buf->boxbuf,
	buf->nbox*4, OM_SET_ARRAY_COPY);
      stat = 1;
   }

   /* Write out the freehand buffer. */
   tmp_id = OMfind_subobj(buf_id, OMstr_to_name("nfree"), OM_OBJ_RW);
   if (OMis_null_obj(tmp_id))
      return(0);

   /* If we have a region that was started but not yet closed.
      Perform the auto-close now for the user when we go to
      write the data out to the buffer.
      Only do this if we are asked to, else we
      may clear an ROI in progress if this routine was
      called from the click module.
   */
   if (buf->free_flag && close) {
      buf->free_flag = 0;
      /* Need at least 3 points to form a valid polygon -
	 that is two lines - we will add 1 to make at least a triangle.
      */
      if (buf->option == ROI_POLYGON && buf->nverts > 2) {
         buf->currptr[buf->nverts*2+1] = buf->currptr[1];
         buf->currptr[buf->nverts*2+2] = buf->currptr[2];
	 buf->curr_size += 2;
	 buf->nverts++;
	 buf->currptr[0] = buf->nverts;
	 buf->nfree++;
      }
      else if (buf->option == ROI_POLYLINE && buf->nverts > 1) {
	 buf->currptr[0] = buf->nverts;
	 buf->nfree++;
      }
      /* If we have less verts than make a valid region, reset the buffer
	 to eliminate those points.
      */
      else if (buf->nverts > 0) {
         /* back all the way off - including spot reserved for
            number of verts. the start of the next region will
            allocate this space again.
         */
         buf->curr_size -= (buf->nverts * 2) + 1;
         buf->nverts = 0;
      }
   }

   /* Don't set the number if it is the same. This will
      avoid activations that aren't necessary.
   */
   if (OMget_long_val(tmp_id, &tmp_long) != OM_STAT_SUCCESS)
      OMset_long_val(tmp_id, buf->nfree);
   else if (tmp_long != buf->nfree)
      OMset_long_val(tmp_id, buf->nfree);

   if (buf->nfree) {
      tmp_id = OMfind_subobj(buf_id, OMstr_to_name("fbuf_size"), OM_OBJ_RW);
      if (OMis_null_obj(tmp_id))
         return(0);
      OMset_long_val(tmp_id, buf->curr_size);
      tmp_id = OMfind_subobj(buf_id, OMstr_to_name("freebuf"), OM_OBJ_RW);
      if (OMis_null_obj(tmp_id))
         return(0);
      OMset_array(tmp_id, OM_TYPE_SHORT, (char *)buf->freebuf,
	buf->curr_size, OM_SET_ARRAY_COPY);
      stat = 1;
   }
   return(stat);
}

static void ROIpoint_addpt(ROIData *buf, int x, int y)
{
   /* Deal with point buffer allocation */
   if (!buf->npoints) {
      /* In case we get here are we have allocated previously. This
	 can happen if we don't get an end event.
      */
      if (!buf->pointbuf_size) {
         buf->pointbuf_size = 10;
         ALLOCN(buf->pointbuf, short, buf->pointbuf_size * 2, "can't alloc pointbuf");
      }
   }
   else if (buf->npoints== buf->pointbuf_size) {
      buf->pointbuf_size += 10;
      REALLOC(buf->pointbuf, short, buf->pointbuf_size * 4, "can't alloc pointbuf");
   }

   /* Add point to buffer */
   buf->pointbuf[buf->npoints*2] = x;
   buf->pointbuf[buf->npoints*2+1] = y;
   buf->npoints++;
}

static void ROIpoint_cont(ROIData *buf, GDview *view, int state, int x, int y)
{
   switch (state) {
      case 1:		/* begin event */
         GDstate_set_draw_mode(view, buf->draw_mode);
         ROIpoint_addpt(buf, x, y);
	 if (buf->draw_mode)
	    ROIpoint_draw(buf, view, 0);
         else ROIpoint_draw(buf, view, 1);
	 break;
      case 2:		/* motion event */
         GDstate_set_draw_mode(view, buf->draw_mode);
         ROIpoint_addpt(buf, x, y);
	 if (buf->draw_mode)
	    ROIpoint_draw(buf, view, 0);
         else ROIpoint_draw(buf, view, 1);
	 break;
      case 3:		/* stop event */
         GDstate_set_draw_mode(view, buf->draw_mode);
         ROIpoint_addpt(buf, x, y);
         ROIpoint_draw(buf, view, 1);
	 break;
      default:
         ERRerror("ROIpoint", 0, ERR_ORIG, "Unknown state");
	 break;
   }
}

static void ROIpoint_click(ROIData *buf, GDview *view, int x, int y)
{
   /* always draw with copy mode. */
   GDstate_set_draw_mode(view, 0);
   ROIpoint_addpt(buf, x, y);

   /* Just draw the last point we added */
   ROIpoint_draw(buf, view, 0);
}

/* 64-bit porting. Only Modified Internally */
static void ROIpoint_draw(ROIData *buf, GDview *view, int flag)
	/* flag - 0=draw last, 1=draw all */
{
   xp_long i;
   short rectbuf[4];

   if (!buf->npoints)
      return;

   if (flag) {
      /* We could be more efficient from a rendering standpoint if
	 we converted all the points up front and made one call
	 to render them.
      */
      for (i=0; i<buf->npoints; i++) {
         /* Draw a 10 pixel square to hilite the selected point. */
         rectbuf[0] = buf->pointbuf[i*2] - HILITE_SIZE;
         rectbuf[1] = buf->pointbuf[i*2+1] - HILITE_SIZE;
         rectbuf[2] = buf->pointbuf[i*2] + HILITE_SIZE;
         rectbuf[3] = buf->pointbuf[i*2+1] + HILITE_SIZE;
         GD2d_draw_fillrects(view->State, 2, (SHORT2 *)rectbuf, NULL);
      }
   }
   else {
      /* Calculate index into point buffer. */
      i = (buf->npoints * 2) - 2;
      /* Draw a 10 pixel square to hilite the selected point. */
      rectbuf[0] = buf->pointbuf[i] - HILITE_SIZE;
      rectbuf[1] = buf->pointbuf[i+1] - HILITE_SIZE;
      rectbuf[2] = buf->pointbuf[i] + HILITE_SIZE;
      rectbuf[3] = buf->pointbuf[i+1] + HILITE_SIZE;
      GD2d_draw_fillrects(view->State, 2, (SHORT2 *)rectbuf, NULL);
   }
}

static void ROIline_addpt(ROIData *buf, int x, int y)
{
   /* Deal with line buffer allocation */
   if (!buf->nlines) {
      /* In case we get here are we have allocated previously. This
	 can happen if we don't get an end event.
      */
      if (!buf->linebuf_size) {
         buf->linebuf_size = 10;
         ALLOCN(buf->linebuf, short, buf->linebuf_size * 4, "can't alloc linebuf");
      }
   }
   else if (buf->nlines == buf->linebuf_size) {
      buf->linebuf_size += 10;
      REALLOC(buf->linebuf, short, buf->linebuf_size * 4, "can't alloc linebuf");
   }

   /* indicate start of a new line */
   buf->line_flag = 1;
   /* Setup start point of line */
   buf->linebuf[buf->nlines*4] = x;
   buf->linebuf[buf->nlines*4+1] = y;
}

static void ROIline_endpt(ROIData *buf, GDview *view, int x, int y)
{
   /* Setup end point of line & draw it */
   buf->line_flag = 0;

   /* Protect against getting an end event without
      having started a line.
   */
   if (buf->linebuf) {
      buf->linebuf[buf->nlines*4+2] = x;
      buf->linebuf[buf->nlines*4+3] = y;
      GD2d_draw_lines(view->State, 2, (SHORT2 *)&buf->linebuf[buf->nlines*4], NULL, NULL);
   }
}

static void ROIline_cont(ROIData *buf, GDview *view, int state, int x, int y)
{
   switch (state) {
      case 1:		/* begin event */
         ROIline_addpt(buf, x, y);
	 break;
      case 2:		/* motion event */
	 /* This is the screen space entry point. There is also an
	    entry for passing world space coords that will be
	    transformed to screen space.  The additional
	    entries on the call are for an array of facet colors
	    and an array of vertex colors respectively. The vertex
	    colors are averaged.
         */
         GDstate_set_draw_mode(view, buf->draw_mode);

	 /* If we have made a line before, we need to erase the
	    old one before drawing the new one.
         */
	 if (buf->draw_mode) {
	    if (!buf->line_flag)
	       GD2d_draw_lines(view->State, 2, (SHORT2 *)&buf->linebuf[buf->nlines*4], NULL, NULL);
	    else buf->line_flag = 0;
	 }

	 /* Setup end point of line & draw it */
	 ROIline_endpt(buf, view, x, y);
	 break;
      case 3:		/* stop event */
         GDstate_set_draw_mode(view, buf->draw_mode);

	 /* If we have made a line before, we need to erase the
	    old one before drawing the new one.
         */
	 if (buf->draw_mode) {
	    if (!buf->line_flag)
	       GD2d_draw_lines(view->State, 2, (SHORT2 *)&buf->linebuf[buf->nlines*4], NULL, NULL);

	    /* Draw the final line in copy mode */
            GDstate_set_draw_mode(view, 0);
	 }

	 /* Setup end point of line & draw it */
	 ROIline_endpt(buf, view, x, y);
         /* Increment the number of lines we have. */
	 if (buf->linebuf)
            buf->nlines++;
	 break;
      default:
         ERRerror("ROIline", 0, ERR_ORIG, "Unknown state");
	 break;
   }
}

static void ROIline_click(ROIData *buf, GDview *view, int x, int y)
{
   if (!buf->line_flag)
      ROIline_addpt(buf, x, y);
   else {
      /* always draw with copy mode. */
      GDstate_set_draw_mode(view, 0);
      ROIline_endpt(buf, view, x, y);
      if (buf->linebuf)
         buf->nlines++;
   }
}

static void ROIbox_addpt(ROIData *buf, int x, int y)
{
   /* Deal with box buffer allocation */
   if (!buf->nbox) {
      /* In case we get here are we have allocated previously. This
	 can happen if we don't get an end event.
      */
      if (!buf->boxbuf_size) {
         buf->boxbuf_size = 10;
         ALLOCN(buf->boxbuf, short, buf->boxbuf_size * 4, "can't alloc boxbuf");
      }
   }
   else if (buf->nbox == buf->boxbuf_size) {
      buf->boxbuf_size += 10;
      REALLOC(buf->boxbuf, short, buf->boxbuf_size * 4, "can't alloc boxbuf");
   }

   /* indicate start of a new box */
   buf->box_flag = 1;
   /* Setup start point of box */
   buf->boxbuf[buf->nbox*4] = x;
   buf->boxbuf[buf->nbox*4+1] = y;
}

static void ROIbox_endpt(ROIData *buf, GDview *view, int x, int y)
{
   buf->box_flag = 0;

   /* Protect against getting an end event without
      having started a box.
   */
   if (buf->boxbuf) {
      /* setup new box & draw it */
      buf->boxbuf[buf->nbox*4+2] = x;
      buf->boxbuf[buf->nbox*4+3] = y;
      GD2d_draw_rects(view->State, 2, (SHORT2 *)&buf->boxbuf[buf->nbox*4], NULL);
   }
}

static void ROIbox_cont(ROIData *buf, GDview *view, int state, int x, int y)
{
   switch (state) {
      case 1:		/* begin event */
	 ROIbox_addpt(buf, x, y);
	 break;
      case 2:		/* motion event */
	 /* Use xor mode while moving the box around */
         GDstate_set_draw_mode(view, buf->draw_mode);
	 /* Use dashed lines while moving the box around */
         GDstate_set_line_style(view, 1);

	 /* If we have made a line before, we need to erase the
	    old one before drawing the new one.
         */
	 if (buf->draw_mode) {
	    if (!buf->box_flag)
	       GD2d_draw_rects(view->State, 2, (SHORT2 *)&buf->boxbuf[buf->nbox*4], NULL);
	    else buf->box_flag = 0;
	 }

	 /* setup new box & draw it */
	 ROIbox_endpt(buf, view, x, y);
	 break;
      case 3:		/* stop event */
	 /* Use xor mode while moving the line around */
         GDstate_set_draw_mode(view, buf->draw_mode);
	 /* Use dashed lines while moving the box around */
         GDstate_set_line_style(view, 1);

	 if (buf->draw_mode) {
	    GD2d_draw_rects(view->State, 2, (SHORT2 *)&buf->boxbuf[buf->nbox*4], NULL);
	    /* Use copy mode to draw final box */
            GDstate_set_draw_mode(view, 0);
	 }

	 /* Use solid lines to draw final box */
         GDstate_set_line_style(view, 0);
	 ROIbox_endpt(buf, view, x, y);
	 if (buf->boxbuf)
	    buf->nbox++;

	 break;
      default:
         ERRerror("ROIbox", 0, ERR_ORIG, "Unknown state");
	 break;
   }
}

static void ROIbox_click(ROIData *buf, GDview *view, int x, int y)
{
   if (!buf->box_flag)
      ROIbox_addpt(buf, x, y);
   else {
      /* always draw with copy mode and solid lines. */
      GDstate_set_draw_mode(view, 0);
      GDstate_set_line_style(view, 0);
      ROIbox_endpt(buf, view, x, y);
      if (buf->boxbuf)
         buf->nbox++;
   }
}

static void ROIfree_addpt(ROIData *buf, int x, int y, int flag)
     /* flag - 0 = start region, 1 = add to region */
{
   /* Deal with buffer allocation. */
   if (!buf->freebuf_size) {
      buf->freebuf_size = FREEBUF_INCR;
      ALLOCN(buf->freebuf, short, buf->freebuf_size, "can't alloc freebuf");
   }
   else {
      /* Every time we add a point - check to see if we have enough.
	 Make sure we can accomodate a new start region: space for
	 nverts and one vert - xy.
      */
      if (buf->curr_size > buf->freebuf_size - 3) {
         buf->freebuf_size += FREEBUF_INCR;
         REALLOC(buf->freebuf, short, buf->freebuf_size, "can't alloc freebuf");
	 buf->currptr = &buf->freebuf[buf->curr_idx];
      }
   }

   /* Setup start point of free buffer */
   buf->free_flag = 1;

   /* If it is a start region:
      reset nverts to 0
      setup the currptr to point to the start of the buffer.
      increment the size to save room for the nverts.
   */
   if (!flag) {
      buf->nverts = 0;		/* reset number of verts */
      buf->currptr = &buf->freebuf[buf->curr_size];
      buf->curr_idx = buf->curr_size;		/* save index at start of region. */
      buf->curr_size++;
   }

   /* Protect against getting an end event without
      having started a line.
   */
   if (buf->freebuf && buf->currptr) {
      /* Make sure the point is not the same as the
	 last one. This is more likely to happen in
	 the continuous case where we get a rapid
	 sequence of points.
      */
      if (buf->nverts == 0 ||
	 (!(buf->currptr[(buf->nverts-1)*2+1] == x &&
	  buf->currptr[(buf->nverts-1)*2+2] == y))) {
         /* Add point to the region & increment the number of points. */
         buf->currptr[buf->nverts*2+1] = x;
         buf->currptr[buf->nverts*2+2] = y;
         buf->nverts++;
         buf->curr_size += 2;		/* room for xy coordinates */
      }
   }
}

static void ROIfree_cont(ROIData *buf, GDview *view, int state, int x, int y)
{
   switch (state) {
      case 1:		/* begin event */
	 ROIfree_addpt(buf, x, y, 0);
	 break;
      case 2:		/* motion event */
      case 3:		/* end event */
         GDstate_set_draw_mode(view, buf->draw_mode);

	 /* If we have made a line before, we need to erase the
	    old one before drawing the new one.
         */
	 if (buf->draw_mode) {
	    if (!buf->free_flag)
	       GD2d_draw_polyline(view->State, buf->nverts, (SHORT2 *)&buf->currptr[1], NULL, NULL, 0);
	    else buf->free_flag = 0;
	    if (state == 3)
               GDstate_set_draw_mode(view, 0);
	 }

	 ROIfree_addpt(buf, x, y, 1);
	 /* Protect against end events when we haven't gotten a start. */
	 if (buf->freebuf && buf->currptr) {
	    if (state == 3) {
	       if (buf->option == ROI_POLYGON) {
	          /* Copy first point to last point. */
                  buf->currptr[buf->nverts*2+1] = buf->currptr[1];
                  buf->currptr[buf->nverts*2+2] = buf->currptr[2];
	          buf->curr_size += 2;
	          buf->nverts++;
               }
	       buf->currptr[0] = buf->nverts;
	       buf->free_flag = 0;
	       buf->nfree++;
            }
	    /* Do the drawing after we have potentially closed the region. */
	    GD2d_draw_polyline(view->State, buf->nverts, (SHORT2 *)&buf->currptr[1], NULL, NULL, 0);
         }
	 break;
      default:
         ERRerror("ROIfreehand", 0, ERR_ORIG, "Unknown state");
	 break;
   }
}

static void ROIfree_click(ROIData *buf, GDview *view, int x, int y)
{
   if (!buf->free_flag)
      ROIfree_addpt(buf, x, y, 0);
   else {
      /* always draw with copy mode and solid lines. */
      GDstate_set_draw_mode(view, 0);
      GDstate_set_line_style(view, 0);
      ROIfree_addpt(buf, x, y, 1);
   }
}

/*************************************************/
/* Draw cursor utilities - Private to this file. */
/*************************************************/

static int GDdraw2d_cursor_get_inputs(OMobj_id elem_id,
                                      DrawCursorData *cursorData)
{
   OMobj_id ptr_id, tmp_id;

   /* Before doing anything else, make sure the view is setup
      appropriately. Get pointer to local view structure so we
      can find the state and winfo.
   */
   ptr_id = OMfind_subobj(elem_id, OMstr_to_name("view_in"), OM_OBJ_RW);
   if (OMis_null_obj(ptr_id))
      return(0);
   if (OMget_obj_val(ptr_id, &tmp_id) != OM_STAT_SUCCESS)
      return(0);
   cursorData->view = (GDview *)GDget_local(tmp_id, (OMpfi)GDview_init);
   if (cursorData->view == NULL)
      return(0);

   /* Get view width and height. We should provide a routine
      to access this info (or a macro).
   */
   if (cursorData->view->Winfo == NULL || cursorData->view->State == NULL)
      return(0);
   cursorData->width = cursorData->view->Winfo->w;
   cursorData->height= cursorData->view->Winfo->h;

   ptr_id = OMfind_subobj(elem_id, OMstr_to_name("cam_in"), OM_OBJ_RW);
   if (OMis_null_obj(ptr_id))
      return(0);
   if (OMget_obj_val(ptr_id, &tmp_id) != OM_STAT_SUCCESS)
      return(0);
   cursorData->camera = (GDcamera *)GDget_local(tmp_id, (OMpfi)GDcamera_create);
   if (!cursorData->camera) {
      ERRerror("GDdraw2d_cursor_get_inputs", 0, ERR_ORIG, "Can't get camera information");
      return(0);
   }

   /* The object is optional. If we don't have one but we have
      a camera, the point is mapped to camera space.
   */
   ptr_id = OMfind_subobj(elem_id, OMstr_to_name("obj_in"), OM_OBJ_RD);
   if (OMis_null_obj(ptr_id))
      return(0);
   if (OMget_obj_val(ptr_id, &tmp_id) != OM_STAT_SUCCESS)
      cursorData->object = NULL;
   else cursorData->object = (GDobject *)GDget_local(tmp_id, (OMpfi)GDobject_create);

   /* Get state. Indicates start, run, stop from interactor. */
   if (!GDget_int_val(elem_id, "state", &cursorData->state))
      return(0);
   if (!GDget_int_val(elem_id, "x", &cursorData->x))
      return(0);
   if (!GDget_int_val(elem_id, "y", &cursorData->y))
      return(0);
   if (!GDget_int_val(elem_id, "immed", &cursorData->immed))
      cursorData->immed = 0;

   if (!GDget_int_val(elem_id, "mode", &cursorData->mode))
      cursorData->mode = 0;
   if (!GDget_int_val(elem_id, "size", &cursorData->size))
      cursorData->size = 10;
   if (!GDget_int_val(elem_id, "thickness", &cursorData->thickness))
      cursorData->thickness = 0;

   /* Get the color to use. */
   if (!GDget_float_val(elem_id, "red", &cursorData->color[0]))
      cursorData->color[0] = 1.0;
   if (!GDget_float_val(elem_id, "green", &cursorData->color[1]))
      cursorData->color[1] = 1.0;
   if (!GDget_float_val(elem_id, "blue", &cursorData->color[2]))
      cursorData->color[2] = 1.0;

   return(1);
}

/* 64-bit porting. Only Modified Internally */
static void GDdraw2d_write_point(OMobj_id elem_id, float *points)
{
   OMobj_id mesh_id, cset_id;
   float mat[4][4];
   xp_long psize, *pconn;

   mesh_id = OMfind_subobj(elem_id, OMstr_to_name("out_mesh"), OM_OBJ_RW);
   if (OMis_null_obj(mesh_id))
      return;

   /* Set up the mesh. This is the grid plus cells.
      Set up the grid: we need nspace, the number of
      nodes and the coordinates.
   */
   if (FLDset_nspace(mesh_id, 2) != 1) {
      ERRerror("GDdraw2d_write_points", 0, ERR_ORIG, "Can't write nspace");
      return;
   }
   /* We always have just 1 point. */
   if (FLDset_nnodes(mesh_id, 1) != 1) {
      ERRerror("GDdraw2d_write_points", 0, ERR_ORIG, "Can't write nnodes");
      return;
   }
   /* mode is OM_SET_ARRAY_COPY- meaning we will ask the
      framework to make a copy of the data. This is ok
      since it is just 1 point that we have in local storage.
   */
   if (FLDset_coord(mesh_id, points, 2, OM_SET_ARRAY_COPY) != 1) {
      ERRerror("GDdraw2d_write_points", 0, ERR_ORIG, "Can't write coordinates");
      return;
   }

   /* Set the xform in the mesh to the identity. */
   MATmat4_identity(mat);
   FLDset_xform(mesh_id, (float *)mat);

   /* Need to reset the cell sets back to the default type.
      Else the cell set render method may be wrong if the cell
      set is now a different type.
   */
   cset_id = OMfind_subobj(mesh_id, OMstr_to_name("cell_set"), OM_OBJ_RW);
   OMreset_obj(cset_id, 0);

   /* Set up the cells: We need the number of cell sets and
      the cell sets. In the cell sets, we need the number of
      cells (ie. primitives), the "type" and the connectivity.
   */
   if (FLDset_ncell_sets(mesh_id, 1) != 1) {
      ERRerror("GDdraw2d_write_points", 0, ERR_ORIG, "Can't write ncell sets");
      return;
   }
   if (FLDget_cell_set(mesh_id, 0, &cset_id) != 1) {
      ERRerror("GDdraw2d_write_points", 0, ERR_ORIG, "Can't get cell set id");
      return;
   }
   if (FLDset_cell_set(cset_id, "Point") != 1) {
      ERRerror("GDdraw2d_write_points", 0, ERR_ORIG, "Can't set cell type");
      return;
   }
   if (FLDset_ncells(cset_id, 1) != 1) {
      ERRerror("GDdraw2d_write_points", 0, ERR_ORIG, "Can't set number of points");
      return;
   }
   if (FLDget_node_connect(cset_id, &pconn, &psize, OM_GET_ARRAY_WR) != 1) {
      ERRerror("GDdraw2d_write_points", 0, ERR_ORIG, "Can't get cell set connectivity");
      return;
   }
   /* Build the connectivity */
   pconn[0] = 0;
   ARRfree(pconn);
}

/***********************************************/
/* Edit Mesh utilities - Private to this file. */
/***********************************************/

static int GDedit_mesh_get_inputs(OMobj_id elem_id, EditMeshData *buf)
{
   OMobj_id ptr_id, tmp_id;

   /* Before doing anything else, make sure the view is setup
      appropriately. Get pointer to local view structure so we
      can find the state and winfo.
   */
   ptr_id = OMfind_subobj(elem_id, OMstr_to_name("view_in"), OM_OBJ_RW);
   if (OMis_null_obj(ptr_id))
      return(0);
   if (OMget_obj_val(ptr_id, &tmp_id) != OM_STAT_SUCCESS)
      return(0);
   buf->view = (GDview *)GDget_local(tmp_id, (OMpfi)GDview_init);
   if (buf->view == NULL)
      return(0);

   /* Setup stuff to draw the cursor */
   if (buf->view->Winfo == NULL || buf->view->State == NULL)
      return(0);

   ptr_id = OMfind_subobj(elem_id, OMstr_to_name("cam_in"), OM_OBJ_RW);
   if (OMis_null_obj(ptr_id))
      return(0);
   if (OMget_obj_val(ptr_id, &tmp_id) != OM_STAT_SUCCESS)
      return(0);
   buf->camera = (GDcamera *)GDget_local(tmp_id, (OMpfi)GDcamera_create);
   if (!buf->camera) {
      ERRerror("GDedit_mesh_get_inputs", 0, ERR_ORIG, "Can't get camera_information");
      return(0);
   }

   /* Get the color to use. */
   if (!GDget_float_val(elem_id, "red", &buf->color[0]))
      buf->color[0] = 1.0;
   if (!GDget_float_val(elem_id, "green", &buf->color[1]))
      buf->color[1] = 1.0;
   if (!GDget_float_val(elem_id, "blue", &buf->color[2]))
      buf->color[2] = 1.0;

   /* Get the mode to draw with - either copy or XOR */
   if (!GDget_int_val(elem_id, "draw_mode", &buf->draw_mode))
      buf->draw_mode = 0;

   return(1);
}

static int GDedit_mesh_get_pickinfo(OMobj_id elem_id, EditMeshData *buf)
{
   OMobj_id pick_info_id, pick_data_id;
   int npicked;
   float objxform[4][4], viewxform[4][4];

   /* Get pick info id from the picked object */
   if (!GDget_refer_db(buf->pobj_id, "pick_info", &pick_info_id))
      return(0);
   GDpick_info_get_npicked(pick_info_id, &npicked);
   /* quit if nothing has been picked */
   if (!npicked)
      return(0);
   /* Get pick data id from pick info */
   if (!GDget_refer_db(pick_info_id, "pick_data", &pick_data_id))
      return(0);

   /* Get the pickdata for the first primitive picked. Only one
      primitive should have been picked. It should be the front
      most primitive. */
   GDpick_info_get_pickdata(pick_data_id, 0, &buf->pick_data);

   /* Get various transformation matricies from picked object
      the objxform and viewxform are currently only written when
      an object is actually picked. this means they don't
      get written when top is picked.
   */
   GDpick_info_get_objxform(pick_info_id, &objxform[0][0]);
   GDpick_info_get_viewxform(pick_info_id, &viewxform[0][0]);

   /* Get transformation matricies to use later.
      fullxform is from view to object.
      objxform is from top to object, including object.
   */
   MATmat4_multiply(objxform, viewxform, buf->fullxform);

   return(1);
}

/* 64-bit porting. Only Modified Internally */
static int GDedit_mesh_get_fieldinfo(OMobj_id elem_id, EditMeshData *buf,
                                     int coord_mode, int conn_mode)
{
   xp_long starti, endi;
   xp_long *dims;
   int dims_size, cell_nnodes;
   int is_rdonly;
   OMobj_id tmp_id, tmp1_id, conn_id, coord_id;

   /* See if we can get dims - if we can it is a
      uniform field. We can't edit uniform fields
      since they don't have an explicit coordinate
      array.
   */
   if (FLDget_dims(buf->pick_data.field_id, &dims, &dims_size) == 1) {
      ARRfree(dims);
      return(0);
   }

   /* Get information from the field. */
   if ((FLDget_nnodes(buf->pick_data.field_id, &buf->nnodes) != 1))
      return(0);
   if (buf->nnodes == 0)
      return(0);

   if ((FLDget_nspace(buf->pick_data.field_id, &buf->nspace) != 1))
      return(0);
   if (FLDget_cell_type(buf->pick_data.data_id, &buf->cell_type) != 1)
      return(0);
   /* Get the number of cells in the cell set.
      If the cell type is polyline (3), polytri (10) or polyhedron (19), we
      have to use a different interface to get the number of cells.
   */
   if (buf->cell_type == 3 || buf->cell_type == 10 || buf->cell_type == 19) {
      if (FLDget_npolys(buf->pick_data.data_id, &buf->ncells) != 1)
         return(0);
   }
   else {
      if (FLDget_ncells(buf->pick_data.data_id, &buf->ncells) != 1)
         return(0);
      if (FLDget_cell_set_nnodes(buf->pick_data.data_id, &cell_nnodes) != 1)
         return(0);
   }
   if (buf->ncells == 0)
      return(0);

   /* If we are going to ask for the data RW, make sure that it
      is NOT RDONLY else we will fail miserably. In this case, the
      OM breaks a connection that exists, allocates an array of the
      same size but fills it will zeros !!

      Do this for both the coordinates and the connectivity.
   */
   if (coord_mode == OM_GET_ARRAY_RW) {
      is_rdonly = 0;
      tmp_id = OMfind_subobj(buf->pick_data.field_id, OMstr_to_name("coordinates"), OM_OBJ_RW);
      tmp1_id = OMfind_subobj(tmp_id, OMstr_to_name("values"), OM_OBJ_RW);
      if (!OMis_null_obj(tmp1_id)) {
         if (OMget_obj_val(tmp1_id, &coord_id) == OM_STAT_SUCCESS) {
            if (OMget_obj_iprop_mode(coord_id, OM_prop_rdonly, &is_rdonly, 0) && is_rdonly == 1) {
               ERRverror("GDedit_mesh_get_fieldinfo", ERR_ORIG, "Coordinate array is read only");
               return(0);
            }
         }
      }
   }
   if (conn_mode == OM_GET_ARRAY_RW) {
      is_rdonly = 0;
      tmp_id = OMfind_subobj(buf->pick_data.data_id, OMstr_to_name("node_connect_list"), OM_OBJ_RW);
      if (!OMis_null_obj(tmp_id)) {
         if (OMget_obj_val(tmp_id, &conn_id) == OM_STAT_SUCCESS) {
            if (OMget_obj_iprop_mode(conn_id, OM_prop_rdonly, &is_rdonly, 0) && is_rdonly == 1) {
               ERRverror("GDedit_mesh_get_fieldinfo", ERR_ORIG, "Connectivity array is read only");
               return(0);
            }
         }
      }
   }

   /* Get the data read/write since we may later update
      the coordinates/connectivity.
   */
   if (FLDget_coord(buf->pick_data.field_id, (float **)&buf->coord_array,
	&buf->coord_size, coord_mode) != 1)
      return(0);
   buf->coord_alloced = 1;
   /* Get the cell connectivity.
      If the cell type is polyline (3), polytri (10) or polyhedron (19), we
      have to use a different interface to get the number of cells.
   */
   if (buf->cell_type == 3 || buf->cell_type == 10 || buf->cell_type == 19) {
      if (FLDget_poly_connect(buf->pick_data.data_id, &buf->conn_array,
	&buf->conn_size, conn_mode) != 1)
         return(0);
   }
   else {
      if (FLDget_node_connect(buf->pick_data.data_id, &buf->conn_array,
	&buf->conn_size, conn_mode) != 1)
         return(0);
      /* Sanity check size - there are cases where builtin OM function are
	 used where this will fail.
      */
      if (buf->ncells * cell_nnodes != buf->conn_size)
	 return(0);
   }
   buf->conn_alloced = 1;

   /* Get the nnodes array if we have a polyhedron cell set. */
   if (buf->cell_type == 19) {
      if (FLDget_poly_nnodes(buf->pick_data.data_id, &buf->nnode_array,
	&buf->nnode_size, conn_mode) != 1)
         return(0);
      buf->nnode_alloced = 1;
   }

   /* check to see if the polyline is connected so we can
      keep it that way when we edit it. This relies on the
      fact that the pickinfo is already setup.
   */
   buf->connected = 0;
   if (buf->cell_type == 3) {
      /* Get the start/end point indicies into the coord array. */
      starti = buf->conn_array[buf->pick_data.conni] * buf->nspace;
      endi = buf->conn_array[buf->pick_data.conni+1] * buf->nspace;;
      if ((buf->coord_array[starti] == buf->coord_array[endi]) &&
          (buf->coord_array[starti+1] == buf->coord_array[endi+1]))
	 buf->connected = 1;
   }

   return(1);
}

static void GDedit_mesh_free_fieldinfo(OMobj_id elem_id, EditMeshData *buf)
{
   /* Release any reference we have to field data */
   if (buf->coord_alloced) {
      ARRfree(buf->coord_array);
      buf->coord_alloced = 0;
   }
   if (buf->conn_alloced) {
      ARRfree(buf->conn_array);
      buf->conn_alloced = 0;
   }
   if (buf->nnode_alloced) {
      ARRfree(buf->nnode_array);
      buf->nnode_alloced = 0;
   }
}

/* Highlight the point/primitive/cell that has been selected */
/* 64-bit porting. Only Modified Internally */
static void GDedit_mesh_hilite(OMobj_id elem_id, EditMeshData *buf)
{
   GDview *view = buf->view;
   float *vert_ptr;
   xp_long i, vert_idx, conni;
   short rectbuf[4];

   GDedit_mesh_setup(elem_id, buf);

   /* Edit point */
   if (buf->mode == 0) {
      /* In all cases, we need to find the point that was picked
	 and hilite it. For all cell types, the pick data verti
	 points right at the point that we want to highlight.
      */
      buf->nverts = 1;
      vert_ptr = &buf->coord_array[buf->pick_data.verti*buf->nspace];
      buf->verts[0][0] = *vert_ptr++;
      buf->verts[0][1] = *vert_ptr++;
      if (buf->nspace == 3)
         buf->verts[0][2] = *vert_ptr++;
      else if (buf->nspace == 2)
         buf->verts[0][2] = 0.0;

      MATxform_verts3(1, buf->verts, buf->fullxform, buf->tverts);
      GDsw_2d_project_verts3d(1, buf->tverts, buf->sverts,
		buf->camera->vp_xmin, buf->camera->vp_ymin,
	 	buf->camera->vp_xmax - buf->camera->vp_xmin,
		buf->camera->vp_ymax - buf->camera->vp_ymin);

      /* Draw a 10 pixel square to hilite the selected point. */
      rectbuf[0] = buf->sverts[0][0] - HILITE_SIZE;
      rectbuf[1] = buf->sverts[0][1] - HILITE_SIZE;
      rectbuf[2] = buf->sverts[0][0] + HILITE_SIZE;
      rectbuf[3] = buf->sverts[0][1] + HILITE_SIZE;
      GD2d_draw_fillrects(view->State, 2, (SHORT2 *)rectbuf, NULL);
   }
   /* 1 = Edit primitive, 2 = Edit cell. */
   else if (buf->mode == 1 || buf->mode == 2) {
      /* Specific processing for the type of cell set we have. */
      switch(buf->cell_type) {
         case 1:	/* Point */
	    /* For point cell sets, the connectivity index points right
	       at the vert for the point we want to hilite.
            */
            buf->nverts = 1;
	    vert_idx = buf->conn_array[buf->pick_data.conni];
	    vert_ptr = &buf->coord_array[vert_idx*buf->nspace];
            buf->verts[0][0] = *vert_ptr++;
            buf->verts[0][1] = *vert_ptr++;
            if (buf->nspace == 3)
               buf->verts[0][2] = *vert_ptr++;
            else if (buf->nspace == 2)
               buf->verts[0][2] = 0.0;
            MATxform_verts3(1, buf->verts, buf->fullxform, buf->tverts);
            GDsw_2d_project_verts3d(1, buf->tverts, buf->sverts,
		buf->camera->vp_xmin, buf->camera->vp_ymin,
	 	buf->camera->vp_xmax - buf->camera->vp_xmin,
		buf->camera->vp_ymax - buf->camera->vp_ymin);

	    /* Draw a 10 pixel square to hilite the selected point. */
            rectbuf[0] = buf->sverts[0][0] - HILITE_SIZE;
            rectbuf[1] = buf->sverts[0][1] - HILITE_SIZE;
            rectbuf[2] = buf->sverts[0][0] + HILITE_SIZE;
            rectbuf[3] = buf->sverts[0][1] + HILITE_SIZE;
            GD2d_draw_fillrects(view->State, 2, (SHORT2 *)rectbuf, NULL);
	    break;
         case 2:	/* Line */
	    /* For line cell sets, a connectivity is two indicies.
	       To find the correct start vert for the line, we
	       take the primitive and multiply by two. This gives
	       us the start vert index in the connectivity array. From
	       the connectivity array, we can get the indicies into the
	       coordinate array for each vert.
            */
            buf->nverts = 2;
	    conni = buf->pick_data.primi * 2;
	    /* Start point. */
	    vert_idx = buf->conn_array[conni++];
	    vert_ptr = &buf->coord_array[vert_idx*buf->nspace];
            buf->verts[0][0] = *vert_ptr++;
            buf->verts[0][1] = *vert_ptr++;
            if (buf->nspace == 3)
               buf->verts[0][2] = *vert_ptr++;
            else if (buf->nspace == 2)
               buf->verts[0][2] = 0.0;
	    /* End point. */
	    vert_idx = buf->conn_array[conni];
	    vert_ptr = &buf->coord_array[vert_idx*buf->nspace];
            buf->verts[1][0] = *vert_ptr++;
            buf->verts[1][1] = *vert_ptr++;
            if (buf->nspace == 3)
               buf->verts[1][2] = *vert_ptr++;
            else if (buf->nspace == 2)
               buf->verts[1][2] = 0.0;

            MATxform_verts3(2, buf->verts, buf->fullxform, buf->tverts);
            GDsw_2d_project_verts3d(2, buf->tverts, buf->sverts,
		buf->camera->vp_xmin, buf->camera->vp_ymin,
	 	buf->camera->vp_xmax - buf->camera->vp_xmin,
		buf->camera->vp_ymax - buf->camera->vp_ymin);
	    GD2d_draw_lines(view->State, 2, (SHORT2 *)buf->sverts, NULL, NULL);
	    break;
         case 3:	/* Polyline */
	    if (buf->mode == 1) {
	       /* For polyline cell sets get the index into the coord
	          array as follows: get the start of the line by
	          indexing into the connectivity array. Add the primitive
	          index to get to the start of the line.
               */
               buf->nverts = 2;
	       vert_idx = buf->conn_array[buf->pick_data.conni] + buf->pick_data.primi;
            }
            else {
	       /* For polyline cell sets, get the index into the coord
	          array as follows: get the start of the line by
	          indexing into the connectivity array.
               */
	       vert_idx = buf->conn_array[buf->pick_data.conni];
	       buf->nverts = buf->conn_array[buf->pick_data.conni+1] - vert_idx + 1;
	       if (buf->nverts > buf->vert_size)
	          GDedit_mesh_realloc(buf, buf->nverts);
            }
	    vert_ptr = &buf->coord_array[vert_idx*buf->nspace];
	    /* xform verts using the Z coord */
	    if (buf->nspace == 3)
	       memcpy(buf->verts, vert_ptr, buf->nverts*3*sizeof(float));
	    else if (buf->nspace == 2) {
	       for (i=0; i<buf->nverts; i++) {
	          buf->verts[i][0] = *vert_ptr++;
	          buf->verts[i][1] = *vert_ptr++;
	          buf->verts[i][2] = 0.0;
	       }
	    }
            MATxform_verts3(buf->nverts, buf->verts, buf->fullxform, buf->tverts);
            GDsw_2d_project_verts3d(buf->nverts, buf->tverts, buf->sverts,
		buf->camera->vp_xmin, buf->camera->vp_ymin,
	 	buf->camera->vp_xmax - buf->camera->vp_xmin,
		buf->camera->vp_ymax - buf->camera->vp_ymin);
	    GD2d_draw_polyline(view->State, buf->nverts, (SHORT2 *)buf->sverts, NULL, NULL, 0);
	    break;
         case 4:	/* Tri */
	    /* For tri cell sets, a connectivity is three indicies.
	       To find the correct start vert for the triangle, we
	       take the primitive and multiply by three. This gives
	       us the start vert index in the connectivity array. From
	       the connectivity array, we can get the indicies into the
	       coordinate array for each vert.
            */
            buf->nverts = 3;
	    conni = buf->pick_data.primi * 3;
	    for (i=0; i<buf->nverts; i++) {
	       vert_idx = buf->conn_array[conni++];
	       vert_ptr = &buf->coord_array[vert_idx*buf->nspace];
               buf->verts[i][0] = *vert_ptr++;
               buf->verts[i][1] = *vert_ptr++;
               if (buf->nspace == 3)
                  buf->verts[i][2] = *vert_ptr++;
               else if (buf->nspace == 2)
                  buf->verts[i][2] = 0.0;
            }
            MATxform_verts3(buf->nverts, buf->verts, buf->fullxform, buf->tverts);
            GDsw_2d_project_verts3d(buf->nverts, buf->tverts, buf->sverts,
		buf->camera->vp_xmin, buf->camera->vp_ymin,
	 	buf->camera->vp_xmax - buf->camera->vp_xmin,
		buf->camera->vp_ymax - buf->camera->vp_ymin);
	    GD2d_draw_polytri(view->State, buf->nverts, (SHORT2 *)buf->sverts, NULL, NULL);
	    break;
         case 5:	/* Quad */
	    /* For quad cell sets, a connectivity is four indicies.
	       To find the correct start vert for the quad, we
	       take the primitive and multiply by four. This gives
	       us the start vert index in the connectivity array. From
	       the connectivity array, we can get the indicies into the
	       coordinate array for each vert. We need to access the
	       verts in a 0,1,3,2 order in order to make a 2 triangle
	       strip out of the verts.
            */
            buf->nverts = 4;
	    conni = buf->pick_data.primi * 4;
	    /* First point. */
	    vert_idx = buf->conn_array[conni];
	    vert_ptr = &buf->coord_array[vert_idx*buf->nspace];
            buf->verts[0][0] = *vert_ptr++;
            buf->verts[0][1] = *vert_ptr++;
            if (buf->nspace == 3)
               buf->verts[0][2] = *vert_ptr++;
            else if (buf->nspace == 2)
               buf->verts[0][2] = 0.0;
	    /* Second point. */
	    vert_idx = buf->conn_array[conni+1];
	    vert_ptr = &buf->coord_array[vert_idx*buf->nspace];
            buf->verts[1][0] = *vert_ptr++;
            buf->verts[1][1] = *vert_ptr++;
            if (buf->nspace == 3)
               buf->verts[1][2] = *vert_ptr++;
            else if (buf->nspace == 2)
               buf->verts[1][2] = 0.0;
	    /* Third point. */
	    vert_idx = buf->conn_array[conni+3];
	    vert_ptr = &buf->coord_array[vert_idx*buf->nspace];
            buf->verts[2][0] = *vert_ptr++;
            buf->verts[2][1] = *vert_ptr++;
            if (buf->nspace == 3)
               buf->verts[2][2] = *vert_ptr++;
            else if (buf->nspace == 2)
               buf->verts[2][2] = 0.0;
	    /* Fourth point. */
	    vert_idx = buf->conn_array[conni+2];
	    vert_ptr = &buf->coord_array[vert_idx*buf->nspace];
            buf->verts[3][0] = *vert_ptr++;
            buf->verts[3][1] = *vert_ptr++;
            if (buf->nspace == 3)
               buf->verts[3][2] = *vert_ptr++;
            else if (buf->nspace == 2)
               buf->verts[3][2] = 0.0;
            MATxform_verts3(buf->nverts, buf->verts, buf->fullxform, buf->tverts);
            GDsw_2d_project_verts3d(buf->nverts, buf->tverts, buf->sverts,
		buf->camera->vp_xmin, buf->camera->vp_ymin,
	 	buf->camera->vp_xmax - buf->camera->vp_xmin,
		buf->camera->vp_ymax - buf->camera->vp_ymin);
	    GD2d_draw_polytri(view->State, buf->nverts, (SHORT2 *)buf->sverts, NULL, NULL);
	    break;
         case 10:	/* Polytri */
	    if (buf->mode == 1) {
	       /* For polytri cell sets get the index into the coord
	          array as follows: get the start of the tri strip by
	          indexing into the connectivity array. Add the primitive
	          index to get to the start of the triangle.
               */
               buf->nverts = 3;
	       vert_idx = buf->conn_array[buf->pick_data.conni] + buf->pick_data.primi;
            }
            else {
	       /* Get to the start of the tri strip and the length
		  of the tri strip.
               */
	       vert_idx = buf->conn_array[buf->pick_data.conni];
	       buf->nverts = buf->conn_array[buf->pick_data.conni+1] - vert_idx + 1;
	       if (buf->nverts > buf->vert_size)
	          GDedit_mesh_realloc(buf, buf->nverts);
            }

	    vert_ptr = &buf->coord_array[vert_idx*buf->nspace];
	    /* xform verts using the Z coord */
	    if (buf->nspace == 3)
	       memcpy(buf->verts, vert_ptr, buf->nverts*sizeof(FLOAT3));
	    else if (buf->nspace == 2) {
	       for (i=0; i<buf->nverts; i++) {
	          buf->verts[i][0] = *vert_ptr++;
	          buf->verts[i][1] = *vert_ptr++;
	          buf->verts[i][2] = 0.0;
	       }
	    }
            MATxform_verts3(buf->nverts, buf->verts, buf->fullxform, buf->tverts);
            GDsw_2d_project_verts3d(buf->nverts, buf->tverts, buf->sverts,
		buf->camera->vp_xmin, buf->camera->vp_ymin,
	 	buf->camera->vp_xmax - buf->camera->vp_xmin,
		buf->camera->vp_ymax - buf->camera->vp_ymin);
	    GD2d_draw_polytri(view->State, buf->nverts, (SHORT2 *)buf->sverts, NULL, NULL);
	    break;
         case 19:	/* Polyhedron */
	    /* Get to the start of the polygon and the length
	       of the polygon.
            */
	    buf->nverts = buf->nnode_array[buf->pick_data.primi];
	    if (buf->nverts > buf->vert_size)
	       GDedit_mesh_realloc(buf, buf->nverts);

	    conni = buf->pick_data.conni;
	    for (i=0; i<buf->nverts; i++) {
	       vert_idx = buf->conn_array[conni++];
	       vert_ptr = &buf->coord_array[vert_idx*buf->nspace];
               buf->verts[i][0] = *vert_ptr++;
               buf->verts[i][1] = *vert_ptr++;
               if (buf->nspace == 3)
                  buf->verts[i][2] = *vert_ptr++;
               else if (buf->nspace == 2)
                  buf->verts[i][2] = 0.0;
            }

            MATxform_verts3(buf->nverts, buf->verts, buf->fullxform, buf->tverts);
            GDsw_2d_project_verts3d(buf->nverts, buf->tverts, buf->sverts,
		buf->camera->vp_xmin, buf->camera->vp_ymin,
	 	buf->camera->vp_xmax - buf->camera->vp_xmin,
		buf->camera->vp_ymax - buf->camera->vp_ymin);
	    GD2d_draw_polygon(view->State, buf->nverts, buf->sverts, NULL, NULL, 1);
	    break;
         default:
            ERRerror("GDedit_mesh_update", 0, ERR_ORIG, "Unknown/unsupported cell set type");
	    break;
      }
   }

   /* Save a copy of the screen space verts to be used to start
      an interactive move sequence.
   */
   memcpy(buf->ssverts, buf->sverts, buf->nverts*sizeof(SHORT2));
   GDstack_pop(view);

   /* Free any references we have to the field data */
   GDedit_mesh_free_fieldinfo(elem_id, buf);
}

/* 64-bit porting. Only Modified Internally */
static void GDedit_mesh_modify(OMobj_id elem_id, EditMeshData *buf)
{
   GDview *view = buf->view;
   xp_long i;
   int deltax = 0, deltay = 0;
   short rectbuf[4];

   switch (buf->state) {
      case 1:
	 buf->move_state = 1;
	 buf->prevx = buf->x;
	 buf->prevy = buf->y;
	 /* Every time we pick an object, the screen space
	    verts of the primitive are calculated and saved
	    in ssverts. These are used to start an interactive
	    sequence. When the sequence is done ssverts is updated
	    with the current position so when another sequence
	    starts we can pick up where we left off.
         */
         memcpy(buf->esverts, buf->ssverts, buf->nverts*sizeof(SHORT2));
	 break;
      case 2:
	 if (buf->move_state) {
	    deltax = buf->x - buf->prevx;
	    deltay = buf->y - buf->prevy;
	    /* Only move if we've gone a reasonable distance,
	       this allows us to select and stay highlighted
	       even if we are connected to a UItwoPoint.
            */
	    if ((UTABS(deltax) > DELTA_SIZE) ||
		(UTABS(deltay) > DELTA_SIZE)) {
	       buf->prevx = buf->x;
	       buf->prevy = buf->y;
	       buf->move_state = 2;
            }
            else {
	       deltax = 0;
	       deltay = 0;
            }
         }
	 break;
      case 3:
	 if (buf->move_state == 2) {
	    deltax = buf->x - buf->prevx;
	    deltay = buf->y - buf->prevy;
	    buf->move_state = 3;
         }
	 break;
      default:
	 break;
   }

   /* if we have moved at all, re-render the primitive. */
   if (deltax || deltay) {
      GDedit_mesh_setup(elem_id, buf);

      /* Edit point */
      if (buf->mode == 0) {
         if (buf->draw_mode != 0) {
	    /* We are using XOR mode so we need to erase the
	       previous drawing if any.
            */
            rectbuf[0] = buf->esverts[0][0] - HILITE_SIZE;
            rectbuf[1] = buf->esverts[0][1] - HILITE_SIZE;
            rectbuf[2] = buf->esverts[0][0] + HILITE_SIZE;
            rectbuf[3] = buf->esverts[0][1] + HILITE_SIZE;
            GD2d_draw_fillrects(view->State, 2, (SHORT2 *)rectbuf, NULL);
	 }

         buf->esverts[0][0] += deltax;
         buf->esverts[0][1] += deltay;

	 /* Draw a 10 pixel square to hilite the selected point. */
         rectbuf[0] = buf->esverts[0][0] - HILITE_SIZE;
         rectbuf[1] = buf->esverts[0][1] - HILITE_SIZE;
         rectbuf[2] = buf->esverts[0][0] + HILITE_SIZE;
         rectbuf[3] = buf->esverts[0][1] + HILITE_SIZE;
         GD2d_draw_fillrects(view->State, 2, (SHORT2 *)rectbuf, NULL);
      }
      /* 1 = Edit primitive, 2 = Edit cell */
      else if (buf->mode == 1 || buf->mode == 2) {
         if (buf->draw_mode != 0) {
	    /* We are using XOR mode so we need to erase the
	       previous drawing if any.
            */
            /* Specific processing for the type of cell set we have. */
            switch(buf->cell_type) {
               case 1:	/* Point */
	          if (buf->nverts == 1) {
	             /* Draw a 10 pixel square to hilite the selected point. */
                     rectbuf[0] = buf->esverts[0][0] - HILITE_SIZE;
                     rectbuf[1] = buf->esverts[0][1] - HILITE_SIZE;
                     rectbuf[2] = buf->esverts[0][0] + HILITE_SIZE;
                     rectbuf[3] = buf->esverts[0][1] + HILITE_SIZE;
                     GD2d_draw_fillrects(view->State, 2, (SHORT2 *)rectbuf, NULL);
                  }
	          else GD2d_draw_points(view->State, buf->nverts, (SHORT2 *)buf->esverts, NULL, NULL);
	          break;
               case 2:	/* Line */
	          GD2d_draw_lines(view->State, buf->nverts, (SHORT2 *)buf->esverts, NULL, NULL);
	          break;
               case 3:	/* Polyline */
	          GD2d_draw_polyline(view->State, buf->nverts, (SHORT2 *)buf->esverts, NULL, NULL, 0);
	          break;
               case 4:	/* Tri */
               case 5:	/* Quad */
               case 10:	/* Polytri */
	          GD2d_draw_polytri(view->State, buf->nverts, (SHORT2 *)buf->esverts, NULL, NULL);
	          break;
               default:
                  ERRerror("GDedit_mesh_modify", 0, ERR_ORIG, "Unknown/unsupported cell set type");
	          break;
            }
         }

	 for (i=0; i<buf->nverts; i++) {
	    buf->esverts[i][0] += deltax;
	    buf->esverts[i][1] += deltay;
	 }
         /* Specific processing for the type of cell set we have. */
         switch(buf->cell_type) {
            case 1:	/* Point */
	       if (buf->nverts == 1) {
	          /* Draw a 10 pixel square to hilite the selected point. */
                  rectbuf[0] = buf->esverts[0][0] - HILITE_SIZE;
                  rectbuf[1] = buf->esverts[0][1] - HILITE_SIZE;
                  rectbuf[2] = buf->esverts[0][0] + HILITE_SIZE;
                  rectbuf[3] = buf->esverts[0][1] + HILITE_SIZE;
                  GD2d_draw_fillrects(view->State, 2, (SHORT2 *)rectbuf, NULL);
               }
	       else GD2d_draw_points(view->State, buf->nverts, (SHORT2 *)buf->esverts, NULL, NULL);
	       break;
            case 2:	/* Line */
	       GD2d_draw_lines(view->State, buf->nverts, (SHORT2 *)buf->esverts, NULL, NULL);
	       break;
            case 3:	/* Polyline */
	       GD2d_draw_polyline(view->State, buf->nverts, (SHORT2 *)buf->esverts, NULL, NULL, 0);
	       break;
            case 4:	/* Tri */
            case 5:	/* Quad */
            case 10:	/* Polytri */
	       GD2d_draw_polytri(view->State, buf->nverts, (SHORT2 *)buf->esverts, NULL, NULL);
	       break;
            default:
               ERRerror("GDedit_mesh_modify", 0, ERR_ORIG, "Unknown/unsupported cell set type");
	       break;
         }
      }
      GDstack_pop(view);
   }

   /* Save where we left off so we can pick
      up again later in another sequence.
      Also mark the primitive as moved so if we
      get a done activation we know what to do.
   */
   if (buf->move_state == 3) {
      memcpy(buf->ssverts, buf->esverts, buf->nverts*sizeof(SHORT2));
      buf->prim_moved = 1;
      buf->move_state = 0;
   }
}

static void GDedit_mesh_setup(OMobj_id elem_id, EditMeshData *buf)
{
   GDview *view = buf->view;
   OMpfi state_func;

   view->status = GDview_status(view);

   /* If the draw mode is copy */
   if (buf->draw_mode == 0) {
      /* Refresh the contents of the screen before
         drawing on top of it.
      */
      GDview_call_func(view, "view_refresh", 0);
      GDstack_push(view);

      /* Currently allowed modes are 0 = copy and non-zero = XOR. */
      GDstate_set_draw_mode(view, GD_PROPS_DRAW_COPY);
   }
   /* The draw mode is XOR and we are modifying the mesh */
   else {
      GDstack_push(view);

      /* Currently allowed modes are 0 = copy and non-zero = XOR. */
      GDstate_set_draw_mode(view, GD_PROPS_DRAW_XOR);
   }

   /* Set the user defineable hilight color */
   GDstate_set_color(view, buf->color);

   /* Use the front buffer (ie. window) as the
      drawble. This allows our drawing to be visible even if
      we are in double buffer mode.
   */
   if (GDstate_get_func(view, "state_set_drawable", &state_func))
      (*state_func)(view, 1);
}

/* 64-bit porting. Only Modified Internally */
static void GDedit_mesh_updfield(OMobj_id elem_id, EditMeshData *buf)
{
   float mat[4][4], invmat[4][4], *vert_ptr;
   xp_long vert_idx, i, conni;
   GDobject *object;
   float *wverts;

   /* If we have a field and the primitive we have
      picked has been moved we can update it in
      the field.
   */
   if (buf->prim_moved) {
      /* Get the field info we need to modify the mesh. We need
         the data out read-write in this case.
      */
      if (!GDedit_mesh_get_fieldinfo(elem_id, buf, OM_GET_ARRAY_RW, OM_GET_ARRAY_RD))
         return;

      object = (GDobject *)GDget_local(buf->pobj_id, (OMpfi)GDobject_create);
      if (!object)
         return;

      /* Problem if we use GDget_map_matrix routine is
	 that the above matrix get may fail if we have
	 multiple parents for 1 object - there isn't
	 a unique path back to the object.
      */
      MATmat4_copy(mat, buf->fullxform);
      /* There are two cases here: if we are editing a 3D field, we
	 need the inverse of a 4x4, if we are editing a 2D field,
	 we need the inverse of a 3x3. To get the 3x3 inverse, we
	 just clear the 3rd row and 3rd column in the 4x4.
      */
      if (buf->nspace == 2) {
         mat[2][0] = 0.0; mat[2][1] = 0.0; mat[2][3] = 0.0;
         mat[0][2] = 0.0; mat[1][2] = 0.0; mat[3][2] = 0.0;
         mat[2][2] = 1.0;
      }
      MATmat4_inverse(invmat, mat);

      /* Edit point */
      if (buf->mode == 0) {
         if (buf->nspace == 3)
            wverts = &buf->w3verts[0][0];
         else if (buf->nspace == 2)
            wverts = &buf->w2verts[0][0];
         GDmap_ss_to_3dworld(buf->camera, &invmat[0][0], 1,
		&buf->esverts[0][0], &buf->tverts[0][0], wverts, buf->nspace);

         vert_ptr = &buf->coord_array[buf->pick_data.verti*buf->nspace];
	 *vert_ptr++ = wverts[0];
	 *vert_ptr++ = wverts[1];
	 if (buf->nspace == 3)
	    *vert_ptr++ = wverts[2];

         /* Let's check to see if the polyline is a closed region.
	    We need to keep it that way.
         */
	 if (buf->connected) {
	    /* Start point edited - update end point */
	    if (buf->pick_data.verti == buf->conn_array[buf->pick_data.conni]) {
	       vert_idx = buf->conn_array[buf->pick_data.conni+1];
	       vert_ptr = &buf->coord_array[vert_idx*buf->nspace];
	       *vert_ptr++ = wverts[0];
	       *vert_ptr++ = wverts[1];
	       if (buf->nspace == 3)
	          *vert_ptr++ = wverts[2];
	    }
            /* End point edited - update start point */
	    else if (buf->pick_data.verti == buf->conn_array[buf->pick_data.conni+1]) {
	       vert_idx = buf->conn_array[buf->pick_data.conni];
	       vert_ptr = &buf->coord_array[vert_idx*buf->nspace];
	       *vert_ptr++ = wverts[0];
	       *vert_ptr++ = wverts[1];
	       if (buf->nspace == 3)
	          *vert_ptr++ = wverts[2];
            }
         }
      }
      /* 1 = Edit primitive, 2 = Edit cell */
      else if (buf->mode == 1 || buf->mode == 2) {
         /* Specific processing for the type of cell set we have. */
         switch(buf->cell_type) {
            case 1:	/* Point */
               if (buf->nspace == 3)
                  wverts = &buf->w3verts[0][0];
               else if (buf->nspace == 2)
                  wverts = &buf->w2verts[0][0];
               GDmap_ss_to_3dworld(buf->camera, &invmat[0][0], 1, &buf->esverts[0][0],
			&buf->tverts[0][0], wverts, buf->nspace);

	       vert_idx = buf->conn_array[buf->pick_data.conni];
	       vert_ptr = &buf->coord_array[vert_idx*buf->nspace];
	       *vert_ptr++ = wverts[0];
	       *vert_ptr++ = wverts[1];
	       if (buf->nspace == 3)
		  *vert_ptr = wverts[2];
	       break;
            case 2:	/* Line */
               if (buf->nspace == 3)
                  wverts = &buf->w3verts[0][0];
               else if (buf->nspace == 2)
                  wverts = &buf->w2verts[0][0];
               GDmap_ss_to_3dworld(buf->camera, &invmat[0][0], 2, &buf->esverts[0][0],
			&buf->tverts[0][0], wverts, buf->nspace);

	       conni = buf->pick_data.primi * 2;
               /* Start point. */
	       vert_idx = buf->conn_array[conni++];
	       vert_ptr = &buf->coord_array[vert_idx*buf->nspace];
	       *vert_ptr++ = wverts[0];
	       *vert_ptr++ = wverts[1];
	       if (buf->nspace == 3)
		  *vert_ptr = wverts[2];
               /* End point. */
	       vert_idx = buf->conn_array[conni];
	       vert_ptr = &buf->coord_array[vert_idx*buf->nspace];
	       *vert_ptr++ = wverts[buf->nspace];
	       *vert_ptr++ = wverts[buf->nspace+1];
	       if (buf->nspace == 3)
		  *vert_ptr = wverts[buf->nspace+2];
	       break;
            case 3:	/* Polyline */
               if (buf->nspace == 3)
                  wverts = &buf->w3verts[0][0];
               else if (buf->nspace == 2)
                  wverts = &buf->w2verts[0][0];
               GDmap_ss_to_3dworld(buf->camera, &invmat[0][0], buf->nverts, &buf->esverts[0][0],
			&buf->tverts[0][0], wverts, buf->nspace);

               if (buf->mode == 1 )
	          vert_idx = buf->conn_array[buf->pick_data.conni] + buf->pick_data.primi;
	       else vert_idx = buf->conn_array[buf->pick_data.conni];
	       vert_ptr = &buf->coord_array[vert_idx*buf->nspace];
	       for (i=0; i<buf->nverts; i++) {
	          *vert_ptr++ = wverts[i*buf->nspace];
	          *vert_ptr++ = wverts[i*buf->nspace+1];
	          if (buf->nspace == 3)
	             *vert_ptr++ = wverts[i*buf->nspace+2];
	       }

	       if (buf->mode == 1) {
	          /* Let's check to see if the polyline is a closed region.
	             We need to keep it that way.
                  */
	          if (buf->connected) {
	             /* Start point edited - update end point */
	             if (vert_idx == buf->conn_array[buf->pick_data.conni]) {
		        vert_idx = buf->conn_array[buf->pick_data.conni+1];
		        vert_ptr = &buf->coord_array[vert_idx*buf->nspace];
	                *vert_ptr++ = wverts[0];
	                *vert_ptr++ = wverts[1];
			if (buf->nspace == 3)
	                   *vert_ptr++ = wverts[2];
	             }
                     /* End point edited - update start point */
	             else if ((vert_idx + 1) == buf->conn_array[buf->pick_data.conni+1]) {
		        vert_idx = buf->conn_array[buf->pick_data.conni];
		        vert_ptr = &buf->coord_array[vert_idx*buf->nspace];
	                *vert_ptr++ = wverts[buf->nspace];
	                *vert_ptr++ = wverts[buf->nspace+1];
			if (buf->nspace == 3)
	                   *vert_ptr++ = wverts[buf->nspace+2];
	             }
	          }
               }
	       break;
            case 4:	/* Tri */
               if (buf->nspace == 3)
                  wverts = &buf->w3verts[0][0];
               else if (buf->nspace == 2)
                  wverts = &buf->w2verts[0][0];
               GDmap_ss_to_3dworld(buf->camera, &invmat[0][0], buf->nverts, &buf->esverts[0][0],
			&buf->tverts[0][0], wverts, buf->nspace);

	       conni = buf->pick_data.primi * 3;
               for (i=0; i<buf->nverts; i++) {
	          vert_idx = buf->conn_array[conni++];
	          vert_ptr = &buf->coord_array[vert_idx*buf->nspace];
	          *vert_ptr++ = wverts[i*buf->nspace];
	          *vert_ptr++ = wverts[i*buf->nspace+1];
	          if (buf->nspace == 3)
	             *vert_ptr++ = wverts[i*buf->nspace+2];
               }
	       break;
            case 5:	/* Quad */
               if (buf->nspace == 3)
                  wverts = &buf->w3verts[0][0];
               else if (buf->nspace == 2)
                  wverts = &buf->w2verts[0][0];
               GDmap_ss_to_3dworld(buf->camera, &invmat[0][0], buf->nverts, &buf->esverts[0][0],
			&buf->tverts[0][0], wverts, buf->nspace);

	       conni = buf->pick_data.primi * 4;

	       /* First point. */
	       vert_idx = buf->conn_array[conni];
	       vert_ptr = &buf->coord_array[vert_idx*buf->nspace];
	       *vert_ptr++ = wverts[0];
	       *vert_ptr++ = wverts[1];
	       if (buf->nspace == 3)
		  *vert_ptr++ = wverts[2];
	       /* Second point. */
	       vert_idx = buf->conn_array[conni+1];
	       vert_ptr = &buf->coord_array[vert_idx*buf->nspace];
	       *vert_ptr++ = wverts[buf->nspace];
	       *vert_ptr++ = wverts[buf->nspace+1];
	       if (buf->nspace == 3)
		  *vert_ptr++ = wverts[buf->nspace+2];
	       /* Third point. */
	       vert_idx = buf->conn_array[conni+3];
	       vert_ptr = &buf->coord_array[vert_idx*buf->nspace];
	       *vert_ptr++ = wverts[2*buf->nspace];
	       *vert_ptr++ = wverts[2*buf->nspace+1];
	       if (buf->nspace == 3)
		  *vert_ptr++ = wverts[2*buf->nspace+2];
	       /* Fourth point. */
	       vert_idx = buf->conn_array[conni+2];
	       vert_ptr = &buf->coord_array[vert_idx*buf->nspace];
	       *vert_ptr++ = wverts[3*buf->nspace];
	       *vert_ptr++ = wverts[3*buf->nspace+1];
	       if (buf->nspace == 3)
		  *vert_ptr++ = wverts[3*buf->nspace+2];
	       break;
            case 10:	/* Polytri */
               if (buf->nspace == 3)
                  wverts = &buf->w3verts[0][0];
               else if (buf->nspace == 2)
                  wverts = &buf->w2verts[0][0];
               GDmap_ss_to_3dworld(buf->camera, &invmat[0][0], buf->nverts, &buf->esverts[0][0],
			&buf->tverts[0][0], wverts, buf->nspace);

               if (buf->mode == 1 )
	          vert_idx = buf->conn_array[buf->pick_data.conni] + buf->pick_data.primi;
	       else vert_idx = buf->conn_array[buf->pick_data.conni];
	       vert_ptr = &buf->coord_array[vert_idx*buf->nspace];
	       for (i=0; i<buf->nverts; i++) {
	          *vert_ptr++ = wverts[i*buf->nspace];
	          *vert_ptr++ = wverts[i*buf->nspace+1];
	          if (buf->nspace == 3)
	             *vert_ptr++ = wverts[i*buf->nspace+2];
	       }
	       break;
            default:
               ERRerror("GDedit_mesh_updfield", 0, ERR_ORIG, "Unknown/unsupported cell set type");
	       break;
         }
      }

      /* Release any reference we have to field data. This will
	 cause a set array to take place and anyone who is
	 attached to the field will find out about the changes.
      */
      GDedit_mesh_free_fieldinfo(elem_id, buf);
      buf->prim_moved = 0;
   }
}

/* 64-bit porting. Directly Modified */
static void GDedit_mesh_realloc(EditMeshData *buf, xp_long size)
{
   /* Reallocation of all the vert arrays */
   buf->vert_size = size;

   REALLOC(buf->verts, FLOAT3, buf->vert_size, "can't allocate verts");
   REALLOC(buf->tverts, FLOAT3, buf->vert_size, "can't allocate verts");
   REALLOC(buf->sverts, SHORT2, buf->vert_size, "can't allocate verts");
   REALLOC(buf->ssverts, SHORT2, buf->vert_size, "can't allocate verts");
   REALLOC(buf->esverts, SHORT2, buf->vert_size, "can't allocate verts");
   REALLOC(buf->w2verts, FLOAT2, buf->vert_size, "can't allocate verts");
   REALLOC(buf->w3verts, FLOAT3, buf->vert_size, "can't allocate verts");
}

/* Delete a point/primitive/cell from the field.

   Valid values for the deleted flag are:
      1 = delete cell set (DELETE_CELL_SET)
      2 = delete cell (DELETE_CELL)
      3 = modified cell (MODIFY_CELL)
*/
/* 64-bit porting. Only Modified Internally */
static void GDedit_mesh_delfield(OMobj_id elem_id, EditMeshData *buf)
{
   xp_long i, j, size;
   xp_long conni, verti1, verti2;
   int deleted = 0, ncell_sets, found;
   int type, ncomp;
   OMobj_id cset_id;

   /* Get the field info we need to modify the mesh. We need
      the data out read-write in this case.
   */
   if (!GDedit_mesh_get_fieldinfo(elem_id, buf, OM_GET_ARRAY_RW, OM_GET_ARRAY_RW))
      return;

   /* If we have a field, this means we have picked and
      hilighted something. We can delete the thing
      we have selected.
   */
   if (buf->coord_alloced && buf->conn_alloced) {
      /* Edit point */
      if (buf->mode == 0) {
         /* Specific processing for the type of cell set we have. */
         switch(buf->cell_type) {
            case 1:	/* Point */
	       /* We need to move the coordinates to remove the
		  one that is being deleted. We move all the coords
		  after the one being deleted up one points worth.
               */
	       conni = buf->pick_data.primi;
	       verti1 = buf->conn_array[conni];
	       size = (buf->nnodes - (verti1 + 1)) * buf->nspace;
	       /* Avoid copy if we are deleting the last line */
	       if (size)
	          memcpy(&buf->coord_array[verti1*buf->nspace],
			 &buf->coord_array[(verti1+1)*buf->nspace],
			 size*sizeof(float));

	       /* Setup the number of verts we deleted. For a point
		  cell set, this is always 1. This is needed at the
		  end of the routine when nnode and the connectivity
		  arrays for other cell sets are updated.
               */
	       size = 1;
	       /* If we have only 1 cell - delete the cell set.
		  Else we need to move and adjust the connectivity
		  indicies for all the lines following the one
		  that is being removed.
               */
	       if (buf->ncells == 1)
		  /* Delay the actual delete of the cell set until
		     we have updated the connectivity arrays of other
		     cell sets affected by the delete, else we can't
		     do it properly.
                  */
		  deleted = DELETE_CELL_SET;
               else {
		  for (i=buf->pick_data.primi; i<buf->ncells-1; i++) {
		     buf->conn_array[conni] = buf->conn_array[conni+1] - 1;
		     conni++;
		  }
		  deleted = DELETE_CELL;
	       }
	       break;
            case 3:	/* Polyline */
	       /* Delete point in polyline.
	          We need up all the coords in the polyline
		  after the point that we are deleting.
               */
	       conni = buf->pick_data.conni;
	       /* If the size of the polyline is 1 segment,
		  delete the whole thing since what will be left
		  will only be a point.
               */
	       if (buf->conn_array[conni+1] - buf->conn_array[conni] < 2) {
	          verti1 = buf->conn_array[conni];
	          verti2 = buf->conn_array[conni+1] + 1;
		  if (buf->ncells == 1)
		     deleted = DELETE_CELL_SET;
                  else deleted = DELETE_CELL;
	       }
	       else {
	          verti1 = buf->conn_array[conni] + buf->pick_data.primi;
	          verti2 = verti1 + 1;
		  deleted = MODIFY_CELL;
               }
	       size = (buf->nnodes - verti2) * buf->nspace;
	       /* Avoid copy if we are deleting the last line segment. */
	       if (size)
	          memcpy(&buf->coord_array[verti1*buf->nspace],
		    &buf->coord_array[verti2*buf->nspace],
		    size*sizeof(float));

	       /* Setup the number of verts we deleted. For a point
	          in a polyline this is always 1 unless the cell is
		  being deleted, in this case size is 2.
		  This is needed at the end of the routine when nnode
		  and the connectivity arrays for other cell sets are updated.
               */
	       if (deleted == DELETE_CELL)
	          size = 2;
               else size = 1;

	       /* We need to move and adjust the connectivity
		  indicies for all the lines following the one
		  that is being removed.
               */
	       if (deleted == MODIFY_CELL) {
		  for (i=conni+1; i<buf->conn_size; i++)
		     buf->conn_array[i] -= size;
               }
	       else if (deleted == DELETE_CELL) {
		  for (i=conni/2; i<buf->ncells-1; i++) {
		     buf->conn_array[conni] = buf->conn_array[conni+2] - size;
		     buf->conn_array[conni+1] = buf->conn_array[conni+3] - size;
		     conni += 2;
		  }
	       }
	       break;
            case 2:	/* Line */
            case 4:	/* Tri */
            case 5:	/* Quad */
            case 10:	/* Polytri */
               ERRerror("GDedit_mesh_delfield", 0, ERR_ORIG, "Can't delete a point in this cell type");
	       break;
            default:
	       break;
         }
      }
      /* 1 = Edit primitive, 2 = Edit cell */
      else if (buf->mode == 1 || buf->mode == 2) {
         /* Specific processing for the type of cell set we have. */
         switch(buf->cell_type) {
            case 1:	/* Point */
	       /* We need to move the coordinates to remove the
		  one that is being deleted. We move all the coords
		  after the one being deleted up one points worth.
               */
	       conni = buf->pick_data.primi;
	       verti1 = buf->conn_array[conni];
	       size = (buf->nnodes - (verti1 + 1)) * buf->nspace;
	       /* Avoid copy if we are deleting the last line */
	       if (size)
	          memcpy(&buf->coord_array[verti1*buf->nspace],
			 &buf->coord_array[(verti1+1)*buf->nspace],
			 size*sizeof(float));

	       /* Setup the number of verts we deleted. For a point
		  cell set, this is always 1. This is needed at the
		  end of the routine when nnode and the connectivity
		  arrays for other cell sets are updated.
               */
	       size = 1;
	       /* If we have only 1 cell - delete the cell set.
		  Else we need to move and adjust the connectivity
		  indicies for all the lines following the one
		  that is being removed.
               */
	       if (buf->ncells == 1)
		  /* Delay the actual delete of the cell set until
		     we have updated the connectivity arrays of other
		     cell sets affected by the delete, else we can't
		     do it properly.
                  */
		  deleted = DELETE_CELL_SET;
               else {
		  for (i=buf->pick_data.primi; i<buf->ncells-1; i++) {
		     buf->conn_array[conni] = buf->conn_array[conni+1] - 1;
		     conni++;
		  }
		  deleted = DELETE_CELL;
	       }
	       break;
            case 2:	/* Line */
	       /* We need to move the coordinates to remove the
		  one that is being deleted. We move all the coords
		  after the one being deleted up one lines worth.
               */
	       conni = buf->pick_data.primi * 2;
	       verti1 = buf->conn_array[conni];
	       size = (buf->nnodes - (verti1 + 2)) * buf->nspace;
	       /* Avoid copy if we are deleting the last line */
	       if (size)
	          memcpy(&buf->coord_array[verti1*buf->nspace],
			 &buf->coord_array[(verti1+2)*buf->nspace],
			 size*sizeof(float));

	       /* Setup the number of verts we deleted. For a line
		  cell set, this is always 2. This is needed at the
		  end of the routine when nnode and the connectivity
		  arrays for other cell sets are updated.
               */
	       size = 2;
	       /* If we have only 1 cell - delete the cell set.
		  Else we need to move and adjust the connectivity
		  indicies for all the lines following the one
		  that is being removed.
               */
	       if (buf->ncells == 1)
		  /* Delay the actual delete of the cell set until
		     we have updated the connectivity arrays of other
		     cell sets affected by the delete, else we can't
		     do it properly.
                  */
		  deleted = DELETE_CELL_SET;
               else {
		  for (i=buf->pick_data.primi; i<buf->ncells-1; i++) {
		     buf->conn_array[conni] = buf->conn_array[conni+2] - 2;
		     buf->conn_array[conni+1] = buf->conn_array[conni+3] - 2;
		     conni += 2;
		  }
		  deleted = DELETE_CELL;
	       }
	       break;
            case 3:	/* Polyline */
	       /* Delete line in polyline. */
	       if (buf->mode == 1) {
	          /* We need up all the coords in the polyline
		     after the line that we are deleting.
                  */
	          conni = buf->pick_data.conni;
		  /* If the size of the polyline is 2 segments
		     or less, delete the whole thing since what
		     will be left will only be a point at most.
                  */
		  if (buf->conn_array[conni+1] - buf->conn_array[conni] <= 2) {
	             verti1 = buf->conn_array[conni];
	             verti2 = buf->conn_array[conni+1] + 1;
		     if (buf->ncells == 1)
		        deleted = DELETE_CELL_SET;
                     else deleted = DELETE_CELL;
		  }
		  else {
	             verti1 = buf->conn_array[conni] + buf->pick_data.primi;
	             verti2 = verti1 + 2;
		     deleted = MODIFY_CELL;
                  }
	          size = (buf->nnodes - verti2) * buf->nspace;
	          /* Avoid copy if we are deleting the last line segment. */
	          if (size)
	             memcpy(&buf->coord_array[verti1*buf->nspace],
			    &buf->coord_array[verti2*buf->nspace],
			    size*sizeof(float));

	          /* Setup the number of verts we deleted. For a line
		     in a polyline this is always 2.  This is needed at
		     the end of the routine when nnode and the connectivity
		     arrays for other cell sets are updated.
                  */
	          size = verti2 - verti1;

		  /* We need to move and adjust the connectivity
		     indicies for all the lines following the one
		     that is being removed.
                  */
		  if (deleted == MODIFY_CELL) {
		     for (i=conni+1; i<buf->conn_size; i++)
		        buf->conn_array[i] -= size;
                  }
		  else if (deleted == DELETE_CELL) {
		     for (i=conni/2; i<buf->ncells-1; i++) {
		        buf->conn_array[conni] = buf->conn_array[conni+2] - size;
		        buf->conn_array[conni+1] = buf->conn_array[conni+3] - size;
		        conni += 2;
		     }
		  }
	       }
	       /* Delete polyline. */
	       else {
	          /* We need to move the coordinates to remove the
		     one that is being deleted. We move all the coords
		     after the one being deleted up one polylines worth.
                  */
	          conni = buf->pick_data.conni;
	          verti1 = buf->conn_array[conni];
	          verti2 = buf->conn_array[conni+1] + 1;
	          size = (buf->nnodes - verti2) * buf->nspace;
	          /* Avoid copy if we are deleting the last polyline */
	          if (size)
	             memcpy(&buf->coord_array[verti1*buf->nspace],
			    &buf->coord_array[verti2*buf->nspace],
			    size*sizeof(float));

	          /* Setup the number of verts we deleted. For a polyline
		     cell set, this is always end-start. This is needed at the
		     end of the routine when nnode and the connectivity
		     arrays for other cell sets are updated.
                  */
	          size = verti2 - verti1;

	          /* If we have only 1 cell - delete the cell set.
		     Else we need to move and adjust the connectivity
		     indicies for all the lines following the one
		     that is being removed.
                  */
	          if (buf->ncells == 1)
		     /* Delay the actual delete of the cell set until
		        we have updated the connectivity arrays of other
		        cell sets affected by the delete, else we can't
		        do it properly.
                     */
		     deleted = DELETE_CELL_SET;
                  else {
		     for (i=conni/2; i<buf->ncells-1; i++) {
		        buf->conn_array[conni] = buf->conn_array[conni+2] - size;
		        buf->conn_array[conni+1] = buf->conn_array[conni+3] - size;
		        conni += 2;
		     }
		     deleted = DELETE_CELL;
		  }
	       }
	       break;
            case 4:	/* Tri */
               ERRerror("GDedit_mesh_delfield", 0, ERR_ORIG, "Delete tri cell not supported");
	       break;
            case 5:	/* Quad */
               ERRerror("GDedit_mesh_delfield", 0, ERR_ORIG, "Delete quad cell not supported");
	       break;
            case 10:	/* Polytri */
               ERRerror("GDedit_mesh_delfield", 0, ERR_ORIG, "Delete polytri cell not supported");
	       break;
            default:
	       break;
         }
      }

      /* Release any reference we have to field data. This will
	 cause a set array to take place and anyone who is
	 attached to the field will find out about the changes.
      */
      GDedit_mesh_free_fieldinfo(elem_id, buf);

      /* If we have deleted a point/primitive/cell we need to
	 cause the change to actuall take effect. We had the
	 coords/conn_array out for RW and have modified them.
	 We released these refs with the ARRfree above. Now we
	 need to make the array smaller. The one thing we need
	 to do is then ask for the arrays RW again so they
	 will be realloced - else there may be a problem if the
	 first request is read only. The OM currently won't
	 realloc in that situation. This will cause an error.

	 It is expected that size will be the number of verts
	 that we deleted. This will be used to update the
	 number of nodes in the coordinate array and the
	 connectivity indicies in the cell set arrays.
      */
      if (deleted) {
	 /* "Zero" the field if we have removed all the verts */
	 if (buf->nnodes - size == 0) {
	    UTILclear_mesh(buf->pick_data.field_id);
	    return;
	 }

	 /* Update the coord array */
         FLDset_nnodes(buf->pick_data.field_id, buf->nnodes-size);
         FLDget_coord(buf->pick_data.field_id, &buf->coord_array,
		&buf->coord_size, OM_GET_ARRAY_RW);
         ARRfree(buf->coord_array);

	 /* See if the connectivity array has been modified */
	 if (deleted == DELETE_CELL) {
	    /* Update the connectivity array. If we have a polyline
	       or polytri cell type, we need to use a different
	       interface.
            */
	    if (buf->cell_type == 3 || buf->cell_type == 10) {
	       FLDset_npolys(buf->pick_data.data_id, buf->ncells-1);
               FLDget_poly_connect(buf->pick_data.data_id, &buf->conn_array,
			&buf->conn_size, OM_GET_ARRAY_RW);
	    }
	    else {
	       FLDset_ncells(buf->pick_data.data_id, buf->ncells-1);
               FLDget_node_connect(buf->pick_data.data_id, &buf->conn_array,
			&buf->conn_size, OM_GET_ARRAY_RW);
            }
            ARRfree(buf->conn_array);
	    /* Update the cell data array if we have any */
            if (FLDget_cell_data_ncomp(buf->pick_data.data_id, &ncomp) == 1) {
               FLDget_cell_data(buf->pick_data.data_id, 0, &type, (char **)&buf->conn_array,
			&buf->conn_size, OM_GET_ARRAY_RW);
               ARRfree(buf->conn_array);
            }
         }

	 /* Now we need to go thru all the cell sets that are
	    after the one we edited and update the connectivity
	    indicies for those cell sets since we have changed
	    the coordinate array.
         */
         if (FLDget_ncell_sets(buf->pick_data.field_id, &ncell_sets) != 1) {
	    return;
	 }
	 /* Loop thru the cell sets. Start updating the connectivity
	    arrays of the cell sets once we are past the one we just
	    edited.
         */
	 found = 0;
	 for (i=0; i<ncell_sets; i++) {
            if (FLDget_cell_set(buf->pick_data.field_id, (int)i, &cset_id) != 1)
	       continue;
	    if (found) {
               if (FLDget_cell_type(cset_id, &buf->cell_type) == 1) {
	          /* Edit the connectivity indicies of the cell set */
	          if (buf->cell_type == 3 || buf->cell_type == 10) {
                     FLDget_poly_connect(cset_id, &buf->conn_array,
			   &buf->conn_size, OM_GET_ARRAY_RW);
                  }
	          else {
                     FLDget_node_connect(cset_id, &buf->conn_array,
			   &buf->conn_size, OM_GET_ARRAY_RW);
	          }
                  for (j=0; j<buf->conn_size; j++)
		     buf->conn_array[j] -= size;
                  ARRfree(buf->conn_array);
               }
            }
	    if (OMequal_objs(buf->pick_data.data_id, cset_id))
	       found = 1;
	 }
         /* If we need to delete the cell set - do it. */
	 if (deleted == DELETE_CELL_SET)
            FLDdel_cell_set(buf->pick_data.field_id, buf->pick_data.data_id);
      }
   }
}

/* 64-bit porting. Only Modified Internally */
void UTILclear_mesh(OMobj_id mesh_id)
{
   int tmp_int;
   xp_long tmp_long;

   /* Make sure we have coords - this will avoid
      unecessary activations.
   */
   if ((FLDget_nnodes(mesh_id, &tmp_long) != 1) || tmp_long) {
      FLDset_nnodes(mesh_id, 0);	/* no coords */
      FLDset_nspace(mesh_id, 0);	/* make sure values is dimensioned [][] */
      /* Cause the OM to release any copy he might have
         allocated. No need to free since we will get
         a NULL pointer back.
      */
      FLDset_coord(mesh_id, NULL, 0, OM_SET_ARRAY_COPY);
   }

   if ((FLDget_node_data_ncomp(mesh_id, &tmp_int) != 1) || tmp_int)
      FLDset_node_data_ncomp(mesh_id, 0);	/* no node data */

   if ((FLDget_ncell_sets(mesh_id, &tmp_int) != 1)|| tmp_int)
      FLDset_ncell_sets(mesh_id, 0);		/* no cell sets */
}

