/*
			Copyright (c) 1994 by
			Advanced Visual Systems Inc.
			All Rights Reserved
	
	This software comprises unpublished confidential information of
	Advanced Visual Systems Inc. and may not be used, copied or made
	available to anyone, except in accordance with the license
	under which it is furnished.
	
	This file is under Perforce control
	$Id: //depot/express/fcs70/gd/samples.c#1 $
*/

#include <stdio.h>
#include <stdlib.h>

#define XP_WIDE_API	/* Use Wide APIs */
#include <avs/util.h>
#include <avs/om.h>
#include <avs/gd.h>

/* Buffers for Doodle drawing */
static int line_flag;
static int nlines = 0, linebuf_size = 0;
static short *linebuf = NULL;
static int box_flag;
static int nbox = 0, boxbuf_size = 0;
static short *boxbuf = NULL;
static int free_flag;
static int nfree = 0, freebuf_size = 0, nverts = 0;
static short *freebuf = NULL;
static short *bufptr = NULL;

/* Room for 201 shorts: a number upto 200 and 200 verts */
#define FREEBUF_INCR	201 * 2

void DrawCursor(OMobj_id elem_id);
void Doodle(OMobj_id elem_id);

static void Doodle_setup (GDview *);
static void Doodle_line  (GDview *, int, int, int, int);
static void Doodle_box   (GDview *, int, int, int, int);
static void Doodle_freehand (GDview *, int, int, int, int);


void samples_init()
{
   OMadd_named_func((OMpfi)DrawCursor, OMstr_to_name("DrawCursor"));
   OMadd_named_func((OMpfi)Doodle, OMstr_to_name("Doodle"));
}

/* This routine uses GD state & primitive routines to draw directly 
   on top of the GD view. One advantage of this method is that this
   provides a window system independent drawing package (ie. this
   will port between X and Windows).
*/
void DrawCursor(OMobj_id elem_id)
{
   OMobj_id view_id;
   GDview *view;
   int state, x, y;
   static int prev_x, prev_y;
   short lines[16];
   float white[3];
   int xmax, ymax;

   white[0]=white[1]=white[2]= 1.0;

   view_id = OMfind_subobj(elem_id, OMstr_to_name("view_in"), OM_OBJ_RD);
   if (OMis_null_obj(view_id)) {
      printf("[DrawCursor]: Can't get view\n");
      return;
   }

   if (!GDget_int_val(elem_id, "state", &state)) {
      printf("[DrawCursor]: Can't get state\n");
      return;
   }
   if (!GDget_int_val(elem_id, "x", &x)) {
      printf("[DrawCursor]: Can't get x\n");
      return;
   }
   if (!GDget_int_val(elem_id, "y", &y)) {
      printf("[DrawCursor]: Can't get y\n");
      return;
   }

   /* get pointer to local view structure so we can find 
      the state and winfo.
   */
   view = (GDview *)GDget_local(view_id, (OMpfi)GDview_init);
   if (!view) {
      printf("[DrawCursor]: Can't get view information\n");
      return;
   }

   /* Setup stuff to draw the cursor */
   if (!view->Winfo || !view->State) {
      printf("[DrawCursor]: Winfo or State not initialized in view\n");
      return;
   }
     
   /* Get view width and height. We should provide a routine
      to access this info (or a macro).
   */
   xmax = view->Winfo->w;
   ymax = view->Winfo->h;

   /* Push start of rendering marker onto the stack so we can
      restore original attributes when we are done.
   */
   GDstack_push(view);

   /* Set the color to be used for drawing */
   GDstate_set_color(view, white); 

   /* Currently allowed modes are 0 = copy and non-zero = XOR.
      This needs to be improved so we have defines for each draw mode.
   */
   GDstate_set_draw_mode(view, 1);

   /* Tell GD to use the front buffer (ie. window) as the
      drawble. This allows our drawing to be visible even if
      we are in double buffer mode.
   */
   GDstate_set_drawable(view, 1);

   switch (state) {
      case 1:		/* start event */
	 /* draw full screen cursor */
	 lines[0] = x;
	 lines[1] = 0;
	 lines[2] = x;
	 lines[3] = ymax;
	 lines[4] = 0;
	 lines[5] = y;
	 lines[6] = xmax;
	 lines[7] = y;
	 /* This is the screen space entry point. There is also an
	    entry for passing world space coords that will be
	    transformed to screen space.  The additional
	    entries on the call are for an array of facet colors
	    and an array of vertex colors respectively. The vertex
	    colors are averaged.
         */
	 GD2d_draw_lines(view->State, 4, lines, NULL, NULL);
	 break;
      case 2:		/* run event */
	 /* erase line at previous position */
	 lines[0] = prev_x;
	 lines[1] = 0;
	 lines[2] = prev_x;
	 lines[3] = ymax;
	 lines[4] = 0;
	 lines[5] = prev_y;
	 lines[6] = xmax;
	 lines[7] = prev_y;
	 /* draw line at new position */
	 lines[8] = x;
	 lines[9] = 0;
	 lines[10] = x;
	 lines[11] = ymax;
	 lines[12] = 0;
	 lines[13] = y;
	 lines[14] = xmax;
	 lines[15] = y;
	 /* This is the screen space entry point. */
	 GD2d_draw_lines(view->State, 8, lines, NULL, NULL);
	 break;
      case 3:		/* stop event */
	 /* erase line at previous position */
	 lines[0] = prev_x;
	 lines[1] = 0;
	 lines[2] = prev_x;
	 lines[3] = ymax;
	 lines[4] = 0;
	 lines[5] = prev_y;
	 lines[6] = xmax;
	 lines[7] = prev_y;
	 /* This is the screen space entry point. */
	 GD2d_draw_lines(view->State, 4, lines, NULL, NULL);
	 break;
      default:		/* unknow event */
	 printf("[DrawCursor]: Unknown event\n");
	 break;
   }
   /* save xy as prev_xy for erasure on other events */
   prev_x = x;
   prev_y = y;

   GDstack_pop(view);
}

void Doodle(OMobj_id elem_id)
{
   OMobj_id seq_id, state_id, view_id;
   int seq, state_seq, i;
   int state, x, y, mode, draw_mode, option;
   GDview *view;
   short *bufptr;

   /* Every time the update function runs the sequence number goes up. This
      means the sequence number of the element itself will be the same
      as the sequence number of the update function after the update function
      executes.  We can use this fact to determine if any of the sub-elements
      has changed since the last time we ran.  If any of the elements have a
      sequence number higher than this, we know that the element has been
      modified since the last execution.
   */
   seq_id = OMfind_subobj(elem_id, OMstr_to_name("update"), OM_OBJ_RD);
   if (OMis_null_obj(seq_id))
      return;
   seq = OMget_obj_seq(seq_id, OMnull_obj, 0);

   state_id = OMfind_subobj(elem_id, OMstr_to_name("state"), OM_OBJ_RW);
   if (OMis_null_obj(state_id)) {
      printf("[Doodle]: Can't get state\n");
      return;
   }
   state_seq = OMget_obj_seq(state_id, OMnull_obj, OM_SEQ_VAL);
   if (seq > state_seq) {
      printf("[Doodle]: State has not changed\n");
      return;
   }
   OMget_int_val(state_id, &state);

   view_id = OMfind_subobj(elem_id, OMstr_to_name("view_in"), OM_OBJ_RD);
   if (OMis_null_obj(view_id)) {
      printf("[Doodle]: Can't get view\n");
      return;
   }

   /* If mode = 0, clear all lines/boxes/freehand buffers.
      Else if mode = 1, append to the buffers.
   */
   if (!GDget_int_val(elem_id, "mode", &mode)) {
      printf("[Doodle]: Can't get mode\n");
      return;
   }
   if (!mode) {
      /* Clear all buffers used for drawing */
      if (nlines) {
	 free(linebuf);
	 linebuf = NULL;
	 nlines = 0;
      }
      if (nbox) {
	 free(boxbuf);
	 boxbuf = NULL;
	 nbox = 0;
      }
      if (nfree) {
	 free(freebuf);
	 freebuf = NULL;
	 nfree = 0;
      }
   }

   /* State must have changed.  Get the 
      rest of the inputs 
   */
   if (!GDget_int_val(elem_id, "x", &x)) {
      printf("[Doodle]: Can't get x\n");
      return;
   }
   if (!GDget_int_val(elem_id, "y", &y)) {
      printf("[Doodle]: Can't get y\n");
      return;
   }
   if (!GDget_int_val(elem_id, "option", &option)) {
      printf("[Doodle]: Can't get option\n");
      return;
   }
   if (!GDget_int_val(elem_id, "draw_mode", &draw_mode)) {
      printf("[Doodle]: Can't get draw_mode\n");
      return;
   }

   /* get pointer to local view structure so we can find 
      the state and winfo.
   */
   view = (GDview *)GDget_local(view_id, (OMpfi)GDview_init);
   if (!view) {
      printf("[Doodle]: Can't get view information\n");
      return;
   }

   /* Setup stuff to draw the cursor */
   if (!view->Winfo || !view->State) {
      printf("[Doodle]: Winfo or State not initialized in view\n");
      return;
   }

   /* If we are in copy mode, we need to restore the contents
      of the window before we begin drawing.
      0 = copy
      1 = xor
   */
   if (!draw_mode) {
      /* COPY mode - Copy the back buffer to the 
	 front buffer (ie. the window) to refresh 
	 the contents of the view.
      */
      view->status = GDview_status(view);
      GDview_call_func(view, "view_swap2", 0);
      /* Do setup after we swap since it will set
	 the drawable to the window. we want the
	 swap to pay attention to what the double
	 buffer flag actually says.
      */
      Doodle_setup(view);

      /* If mode is append - draw all existing lines, boxes and freehands */
      if (mode) {
         GDstate_set_draw_mode(view, 0);
	 if (nlines)
	    GD2d_draw_lines(view->State, nlines*2, linebuf, NULL, NULL);
	 for (i=0; i<nbox; i++)
	    GD2d_draw_polyline(view->State, 5, &boxbuf[i*10], NULL, NULL, 0);
	 for (i=0; i<nfree; i++) {
	    bufptr = &freebuf[i*FREEBUF_INCR];
	    GD2d_draw_polyline(view->State, bufptr[0], &bufptr[1], NULL, NULL, 0);
	 }
      }
   }
   else {
      /* XOR mode */
      /* Else on start event refresh the buffer */
      if (state == 1) {
         view->status = GDview_status(view);
         GDview_call_func(view, "view_swap2", 0);
	 /* setup after swap for reason stated above. */
         Doodle_setup(view);
         /* If mode is append - draw all existing lines, boxes and freehands */
         if (mode) {
            GDstate_set_draw_mode(view, 0);
	    if (nlines)
	       GD2d_draw_lines(view->State, nlines*2, linebuf, NULL, NULL);
	    for (i=0; i<nbox; i++)
	       GD2d_draw_polyline(view->State, 5, &boxbuf[i*10], NULL, NULL, 0);
	    for (i=0; i<nfree; i++) {
	       bufptr = &freebuf[i*FREEBUF_INCR];
	       GD2d_draw_polyline(view->State, bufptr[0], &bufptr[1], NULL, NULL, 0);
	    }
         }
      }
      else Doodle_setup(view);
   }

   switch (option) {
      case 0:		/* draw line */
	 Doodle_line(view, state, x, y, draw_mode);
	 break;
      case 1:		/* draw box */
	 Doodle_box(view, state, x, y, draw_mode);
	 break;
      case 2:		/* draw freehand */
	 Doodle_freehand(view, state, x, y, draw_mode);
	 break;
      default:
	 printf("[Doodle]: Unknown option\n");
	 break;
   }
}

static void Doodle_setup(GDview *view)
{
   float white[3];
   white[0]=white[1]=white[2]= 1.0;
   
   /* Push start of rendering marker onto the stack so we can
      restore original attributes when we are done.
   */
   GDstack_push(view);

   /* Set the color to be used for drawing */
   GDstate_set_color(view, white); 

   /* Tell GD to use the front buffer (ie. window) as the
      drawble. This allows our drawing to be visible even if
      we are in double buffer mode.
   */
   GDstate_set_drawable(view, 1);
}

static void Doodle_line(GDview *view, int state, int x, int y, int draw_mode)
{
   switch (state) {
      case 1:		/* begin event */
	 /* Deal with line buffer allocation */
         if (!nlines) {
	    linebuf_size = 10;
	    ALLOCN(linebuf, short, linebuf_size * 4, "can't alloc linebuf");
	 }
	 else if (nlines == linebuf_size) {
	    linebuf_size += 10;
	    REALLOC(linebuf, short, linebuf_size * 4, "can't alloc linebuf");
         }

	 /* indicate start of a new line */
	 line_flag = 0;
	 /* Setup start point of line */
	 linebuf[nlines*4] = x;
	 linebuf[nlines*4+1] = y;
	 break;
      case 2:		/* motion event */
	 /* This is the screen space entry point. There is also an
	    entry for passing world space coords that will be
	    transformed to screen space.  The additional
	    entries on the call are for an array of facet colors
	    and an array of vertex colors respectively. The vertex
	    colors are averaged.
         */

         GDstate_set_draw_mode(view, draw_mode);

	 /* If we have made a line before, we need to erase the
	    old one before drawing the new one.
         */
	 if (draw_mode) {
	    if (line_flag)
	       GD2d_draw_lines(view->State, 2, &linebuf[nlines*4], NULL, NULL);
	    else line_flag = 1;
	 }

	 /* Setup end point of line & draw it */
	 linebuf[nlines*4+2] = x;
	 linebuf[nlines*4+3] = y;
	 GD2d_draw_lines(view->State, 2, &linebuf[nlines*4], NULL, NULL);
	 break;
      case 3:		/* stop event */
         GDstate_set_draw_mode(view, draw_mode);

	 /* Erase the old line one final time. */
	 if (draw_mode) {
	    GD2d_draw_lines(view->State, 2, &linebuf[nlines*4], NULL, NULL);

	    /* Draw the final line in copy mode */
            GDstate_set_draw_mode(view, 0);
	 }

	 /* Setup end point of line & draw it */
	 linebuf[nlines*4+2] = x;
	 linebuf[nlines*4+3] = y;
	 GD2d_draw_lines(view->State, 2, &linebuf[nlines*4], NULL, NULL);

	 /* Increment the number of lines we have. */
	 nlines++;
	 break;
      default:
	 printf("[Doodle]: Unknown state\n");
	 break;
   }
   GDstack_pop(view);
}

static void Doodle_box(GDview *view, int state, int x, int y, int draw_mode)
{
   switch (state) {
      case 1:		/* begin event */
	 /* Deal with box buffer allocation */
         if (!nbox) {
	    boxbuf_size = 10;
	    ALLOCN(boxbuf, short, boxbuf_size * 10, "can't alloc boxbuf");
	 }
	 else if (nbox == boxbuf_size) {
	    boxbuf_size += 10;
	    REALLOC(boxbuf, short, boxbuf_size * 10, "can't alloc boxbuf");
         }

	 /* indicate start of a new box */
	 box_flag = 0;
	 /* Setup start point of box */
	 boxbuf[nbox*10] = x;
	 boxbuf[nbox*10+1] = y;
	 break;
      case 2:		/* motion event */
	 /* Use xor mode while moving the box around */
         GDstate_set_draw_mode(view, draw_mode);
	 /* Use dashed lines while moving the box around */
         GDstate_set_line_style(view, 1);

	 /* If we have made a line before, we need to erase the
	    old one before drawing the new one.
         */
	 if (draw_mode) {
	    if (box_flag)
	       GD2d_draw_polyline(view->State, 5, &boxbuf[nbox*10], NULL, NULL, 0);
	    else box_flag = 1;
	 }

	 /* setup new box & draw it */
	 boxbuf[nbox*10+2] = x;
	 boxbuf[nbox*10+3] = boxbuf[nbox*10+1];
	 boxbuf[nbox*10+4] = x;
	 boxbuf[nbox*10+5] = y;
	 boxbuf[nbox*10+6] = boxbuf[nbox*10];
	 boxbuf[nbox*10+7] = y;
	 boxbuf[nbox*10+8] = boxbuf[nbox*10];
	 boxbuf[nbox*10+9] = boxbuf[nbox*10+1];
	 GD2d_draw_polyline(view->State, 5, &boxbuf[nbox*10], NULL, NULL, 0);
	 break;
      case 3:		/* stop event */
	 /* Use xor mode while moving the line around */
         GDstate_set_draw_mode(view, draw_mode);
	 /* Use dashed lines while moving the box around */
         GDstate_set_line_style(view, 1);

	 if (draw_mode) {
	    GD2d_draw_polyline(view->State, 5, &boxbuf[nbox*10], NULL, NULL, 0);
	    /* Use copy mode to draw final box */
            GDstate_set_draw_mode(view, 0);
	 }

	 /* setup new box & draw it */
	 boxbuf[nbox*10+2] = x;
	 boxbuf[nbox*10+3] = boxbuf[nbox*10+1];
	 boxbuf[nbox*10+4] = x;
	 boxbuf[nbox*10+5] = y;
	 boxbuf[nbox*10+6] = boxbuf[nbox*10];
	 boxbuf[nbox*10+7] = y;
	 boxbuf[nbox*10+8] = boxbuf[nbox*10];
	 boxbuf[nbox*10+9] = boxbuf[nbox*10+1];
	 
	 /* Use solid lines to draw final box */
         GDstate_set_line_style(view, 0);
	 GD2d_draw_polyline(view->State, 5, &boxbuf[nbox*10], NULL, NULL, 0);
	 nbox++;

	 break;
      default:
	 printf("[Doodle]: Unknown state\n");
	 break;
   }
   GDstack_pop(view);
}

static void Doodle_freehand(GDview *view, int state, int x, int y, int draw_mode)
{
   switch (state) {
      case 1:		/* begin event */
	 /* Deal with buffer allocation. Alloc space for 200
	    lines for each freehand.
	 */
         if (!nfree) {
	    freebuf_size = FREEBUF_INCR;
	    ALLOCN(freebuf, short, freebuf_size, "can't alloc freebuf");
	 }
	 else {
	    freebuf_size += FREEBUF_INCR;
	    REALLOC(freebuf, short, (nfree + 1) * freebuf_size, "can't alloc freebuf");
         }

	 /* Setup start point of free buffer */
	 free_flag = 0;

	 bufptr = &freebuf[nfree * FREEBUF_INCR];

	 /* reserve first entry for number of segments in freehand */
	 bufptr[1] = x;
	 bufptr[2] = y;
	 nverts = 1;
	 break;
      case 2:		/* motion and stop events */
      case 3:
         GDstate_set_draw_mode(view, draw_mode);

	 /* If we have made a line before, we need to erase the
	    old one before drawing the new one.
         */
	 if (draw_mode) {
	    if (free_flag)
	       GD2d_draw_polyline(view->State, nverts, &bufptr[1], NULL, NULL, 0);
	    else free_flag = 1;
	    if (state == 3)
               GDstate_set_draw_mode(view, 0);
	 }

	 /* Add to the buffer if there is room. */
	 if (nverts < 200) {
	    /* Setup end point of line & draw it */
	    bufptr[nverts*2+1] = x;
	    bufptr[nverts*2+2] = y;
	    nverts++;
         }
	 GD2d_draw_polyline(view->State, nverts, &bufptr[1], NULL, NULL, 0);

	 if (state == 3) {
	    bufptr[0] = nverts;
	    nfree++;
	 }
	 break;
      default:
	 printf("[Doodle]: Unknown state\n");
	 break;
   }
   GDstack_pop(view);
}

