/*
			Copyright (c) 1994 by
			Advanced Visual Systems Inc.
			All Rights Reserved

	This software comprises unpublished confidential information of
	Advanced Visual Systems Inc. and may not be used, copied or made
	available to anyone, except in accordance with the license
	under which it is furnished.

	This file is under Perforce control
	$Id: //depot/express/fcs70/gdif/light.c#1 $
*/

#define XP_WIDE_API	/* Use Wide APIs */

#include <avs/dll_in.h>
#include <avs/util.h>
#include <avs/om.h>
#include <avs/gd.h>

/* This function is called in a "delayed" fashion via the
   GDget_local routine. When we try to get the local structure 
   pointer - if it doesn't exist the create routine is called. 
   This will happen in two cases: either thru the update function 
   that is called at instance time or via the attach function.
*/
int GDlight_create(OMobj_id light_id)
{
   GDlight *light;

   /* allocate a light structure and clear it */
   ALLOCN(light, GDlight, 1, "can't allocate light");

   /* save OM element id in local data structures */
   light->light_id = light_id;
   /* Save the id of the update function. */
   light->upd_func = OMfind_subobj(light_id, OMstr_to_name("upd_func"), OM_OBJ_RW);

   GDset_local(light_id, (char *)light);
   return(1);
}

int GDlight_delete(OMobj_id light_id)
{
   GDlight *light;

   if (OMis_null_obj(light_id))
      return(1);
   light = (GDlight *)GDget_local(light_id, (OMpfi)GDlight_create);

   if (light) {
      GDlight_detach_xform(light_id, 0);
      FREE(light);
   }
   GDclear_local(light_id);
   return(1);
}

/* Something in the light database structure has changed.
   Update the local storage to stay in sync.
*/
/* 64-bit porting. Only Modified Internally */
int GDlight_update(OMobj_id light_id)
{
   GDlight *light = NULL;
   int tmp_int;
   xp_long tmp_long;
   float tmp_float, *tmp_arr;

   /* get pointer to local storage, return if NULL */
   light = (GDlight *)GDget_local(light_id, (OMpfi)GDlight_create);
   if (!light)
      return(0);

   if (GDget_int_val(light_id, "type", &tmp_int))
      light->type = tmp_int;
   else light->type = GD_LIGHT_DIR;
   if (GDget_int_val(light_id, "state", &tmp_int))
      light->state= tmp_int;
   else light->state = 0;
   if (GDget_float_array(light_id, "col", &tmp_long, &tmp_arr)) {
      memcpy(light->col, tmp_arr, tmp_long*sizeof(float));
      ARRfree(tmp_arr);
   }
   else {
      light->col[0] = GD_DEFAULT_RED;
      light->col[1] = GD_DEFAULT_GREEN;
      light->col[2] = GD_DEFAULT_BLUE;
   }
   if (GDget_float_array(light_id, "att", &tmp_long, &tmp_arr)) {
      memcpy(light->att, tmp_arr, tmp_long*sizeof(float));
      ARRfree(tmp_arr);
   }
   else {
      light->att[0] = GD_DEFAULT_ATTEN;
      light->att[1] = GD_DEFAULT_ATTEN;
   }
   if (GDget_float_val(light_id, "angle", &tmp_float))
      light->angle = tmp_float;
   else light->angle = GD_DEFAULT_ANGLE;
   if (GDget_float_val(light_id, "concen", &tmp_float))
      light->concen = tmp_float;
   else light->concen = GD_DEFAULT_CONCEN;

   GDlight_update_xform(light);
   return(1);
}

void GDlight_update_xform(GDlight *light)
{
   OMobj_id xform_id;

   /* See if anything is attached already.  If so
      this activation must be to detach the xform.
   */
   if (!GDget_refer_db(light->light_id, "xform", &xform_id)) {
      if (light->Xform)
         GDlight_detach_xform(light->light_id, 0);
   }
   /* Else this activation must be to attach an xform
      to a light 
   */
   else {
      /* valid xform_id attached to light 
         see if we have a valid local xform
      */
      if (!light->Xform)
         GDlight_attach_xform(light->light_id, xform_id, 0);
      /* Else see if anything in the xform has changed. */
      else GDxform_update(xform_id, light->Xform);
   }
}

void GDlight_attach_xform(OMobj_id light_id, OMobj_id xform_id, int flag)
{
   GDlight *light = NULL;

   light = (GDlight *)GDget_local(light_id, (OMpfi)GDlight_create);
   if (light == NULL)
      return;

   /* allocate an xform structure and initialize it */
   light->Xform = GDxform_alloc(xform_id);

   /* Get the current values out of the OM. */
   GDxform_update(xform_id, light->Xform);

   /* Make the connection in the data base. This is needed
      for programmatic usage but not for VP usage.
   */
   if (flag)
      GDset_refer_db(light_id, "xform", xform_id);
}

void GDlight_detach_xform(OMobj_id light_id, int flag)
{
   GDlight *light;

   light = (GDlight *)GDget_local(light_id, (OMpfi)GDlight_create);

   if (light->Xform) {
      /* free alloced data and remove pointer to xform */
      FREE(light->Xform);
      light->Xform = NULL;
   }

   if (flag)
      GDset_refer_db(light_id, "xform", OMnull_obj);
}

/* Resetting the lights should do the following:
   Disable display of the light source representation.
   Set the number of lights to 1. This is a directional light.
   Set the ambient light on with a color of white.
*/
void GDlight_reset(OMobj_id lights_id)
{
   OMobj_id xform_id;
   GDlight *light;

   if (OMis_null_obj(lights_id))
      return;

   /* Update the copy in local storage. */
   light = (GDlight *)GDget_local(lights_id, (OMpfi)GDlight_create);
   light->state = 0;
   light->type = GD_LIGHT_DIR;
   light->col[0] = GD_DEFAULT_RED;
   light->col[1] = GD_DEFAULT_GREEN;
   light->col[2] = GD_DEFAULT_BLUE;
   light->att[0] = GD_DEFAULT_ATTEN;
   light->att[1] = GD_DEFAULT_ATTEN;
   light->angle = GD_DEFAULT_ANGLE;
   light->concen = GD_DEFAULT_CONCEN;

   /* Update the copy in the OM. */
   GDlight_set_state(lights_id, 0);
   GDlight_set_type(lights_id, GD_LIGHT_DIR);
   GDlight_set_color(lights_id, light->col);
   GDlight_set_att(lights_id, light->att);
   GDlight_set_angle(lights_id, GD_DEFAULT_ANGLE);
   GDlight_set_concen(lights_id, GD_DEFAULT_CONCEN);

   /* Reset the xform if it is attached. */
   if (!GDget_refer_db(lights_id, "xform", &xform_id) ||
       OMis_null_obj(xform_id))
      return;
   GDxform_reset(xform_id);
}

void GDlight_set_type(OMobj_id light_id, int type)
{
   GDset_int_val(light_id, "type", type);
}

void GDlight_get_type(OMobj_id light_id, int *type)
{
   if (GDget_int_val(light_id, "type", type) != 1)
      *type = GD_LIGHT_DIR;
}

void GDlight_set_state(OMobj_id light_id, int state)
{
   GDset_int_val(light_id, "state", state);
}

void GDlight_get_state(OMobj_id light_id, int *state)
{
   if (GDget_int_val(light_id, "state", state) != 1)
      *state = 0;	/* off by default */
}

void GDlight_set_color(OMobj_id light_id, float *color)
{
   GDset_float_array(light_id, "col", 3, color);
}

/* 64-bit porting. Only Modified Internally */
void GDlight_get_color(OMobj_id light_id, float *color)
{
   xp_long size;
   float *curcol;

   if (GDget_float_array(light_id, "col", &size, &curcol) != 1) {
      color[0] = GD_DEFAULT_RED;
      color[1] = GD_DEFAULT_GREEN;
      color[2] = GD_DEFAULT_BLUE;
      return;
   }
   memcpy(color, curcol, size*sizeof(float));
   ARRfree(curcol);
}

void GDlight_set_att(OMobj_id light_id, float *att)
{
   GDset_float_array(light_id, "att", 2, att);
}

/* 64-bit porting. Only Modified Internally */
void GDlight_get_att(OMobj_id light_id, float *att)
{
   xp_long size;
   float *curatt;

   if (GDget_float_array(light_id, "att", &size, &curatt) != 1) {
      att[0] = GD_DEFAULT_ATTEN;
      att[1] = GD_DEFAULT_ATTEN;
      return;
   }
   memcpy(att, curatt, size*sizeof(float));
   ARRfree(curatt);
}

void GDlight_set_concen(OMobj_id light_id, float concen)
{
   GDset_float_val(light_id, "concen", concen);
}

void GDlight_get_concen(OMobj_id light_id, float *concen)
{
   if (GDget_float_val(light_id, "concen", concen) != 1)
      *concen = GD_DEFAULT_CONCEN;
}

void GDlight_set_angle(OMobj_id light_id, float angle)
{
   GDset_float_val(light_id, "angle", angle);
}

void GDlight_get_angle(OMobj_id light_id, float *angle)
{
   if (GDget_float_val(light_id, "angle", angle) != 1)
      *angle = GD_DEFAULT_ANGLE;
}
