/*
			Copyright (c) 1994 by
			Advanced Visual Systems Inc.
			All Rights Reserved

	This software comprises unpublished confidential information of
	Advanced Visual Systems Inc. and may not be used, copied or made
	available to anyone, except in accordance with the license
	under which it is furnished.

	This file is under Perforce control
	$Id: //depot/express/fcs70/gdif/om_util.c#1 $
*/

#define XP_WIDE_API	/* Use Wide APIs */

#include <avs/dll_in.h>
#include <avs/err.h>
#include <avs/om.h>
#include <avs/om_att.h>
#include <avs/gd.h>

OMobj_id GDcreate_instance(OMobj_id elem_id,
                           const char *templ_name, const char *inst_name)
{
   OMobj_id parent_id, templ_id, inst_id;

   /* find template the user asked for */
   templ_id = OMfind_str_subobj(OMtempl_obj, templ_name, OM_OBJ_RD);
   if (OMis_null_obj(templ_id))
      return(OMnull_obj);

   /* Create the instance using the elem_id the user specified
      as the parent if it isn't null else use Instances as the parent.
   */
   if (OMis_null_obj(elem_id))
      parent_id = OMinst_obj;
   else parent_id = elem_id;

   /* Make an instance of the template element */
   inst_id = OMcopy_obj(templ_id, OM_NULL_NAME,  parent_id, OMlocal_proc_id,
			OM_COPY_CLOSE);
   if (OMis_null_obj(inst_id))
      return(OMnull_obj);

   /* Give the instance the name the user specified. */
   OMset_obj_name(inst_id, OMstr_to_name(inst_name));
   return(inst_id);
}

void GDset_local(OMobj_id id, char *local_ptr)
{
   OMobj_id ptr_id;

   /* find pointer to local data structure */
   ptr_id = OMfind_subobj(id, OMstr_to_name("local_ptr"), OM_OBJ_RW);
   if (!OMis_null_obj(ptr_id)) {
      /* set pointer to structure in data layer */
      OMset_ptr_val(ptr_id, local_ptr, 0);
   }
}

/* This routine will break the connection to the local pointer.
   This is used when we get the deinstance function to invalidate
   the local pointer so the next time we get instanced we will
   behave properly.
*/
void GDclear_local(OMobj_id id)
{
   OMobj_id ptr_id;

   /* find pointer to local data structure */
   ptr_id = OMfind_subobj(id, OMstr_to_name("local_ptr"), OM_OBJ_RW);
   if (!OMis_null_obj(ptr_id)) {
      /* set pointer to null obj which will break the connection */
      OMset_obj_val(ptr_id, OMnull_obj, 0);
   }
}

char *GDget_local(OMobj_id id, OMpfi create_func)
{
   OMobj_id ptr_id;
   char *local_ptr;
   int stat;

   ptr_id = OMfind_subobj(id, OMstr_to_name("local_ptr"), OM_OBJ_RW);
   if (OMis_null_obj(ptr_id)) {
      ERRerror("GDget_local", 0, ERR_ORIG, "Can't find local pointer");
      return(NULL);
   }

   if ((stat = OMget_ptr_val(ptr_id, (void *)&local_ptr, 0)) == 1)
      /* return pointer to local structure */
      return(local_ptr);
   else {
      /*
       * This is likely because we were given a remote object... we can't
       * deal with that case at all...
       */
      if (stat == -1) return(NULL);
      /* local pointer not valid - this means we need to
	 call the element's create function directly to setup
	 the local pointer, add notifies for the element, etc.
      */
      (*create_func)(id);
      /* now we should be able to get the local pointer */
      if (OMget_ptr_val(ptr_id, (void *)&local_ptr, 0) == 1)
	 return(local_ptr);
      else {
         ERRerror("GDget_local", 0, ERR_ORIG, "Can't create local pointer");
	 return(NULL);
      }
   }
}

void GDset_refer_db(OMobj_id id, const char *name, OMobj_id refer_id)
{
   OMobj_id elem_id;

   /* attach refer_id to id in database */
   elem_id = OMfind_subobj(id, OMstr_to_name(name), OM_OBJ_RW);
   if (!OMis_null_obj(elem_id))
      OMset_obj_ref(elem_id, refer_id, 0);
}

int GDget_refer_db(OMobj_id id, const char *name, OMobj_id *refer_id)
{
   OMobj_id elem_id;

   /* attach refer_id to id in database */
   elem_id = OMfind_subobj(id, OMstr_to_name(name), OM_OBJ_RW);
   if (OMis_null_obj(elem_id))
      return(0);
   else return(OMget_obj_val_user(elem_id, refer_id));
}

int GDget_refer_db2(OMobj_id id, const char *name, OMobj_id *refer_id)
{
   OMobj_id elem_id;
   int stat;

   elem_id = OMfind_subobj(id, OMstr_to_name(name), OM_OBJ_RW);
   if (OMis_null_obj(elem_id))
      return(0);
   else {
      /* This is essentially a replacement for OMget_obj_val_user,
       * which does not corectly handle the arrays of fields that
       * are produced by merge_ARR.  Of course, plain ol' OMget_obj_val
       * seems to work as well ... but that would be too easy.
       *
       * Starting at the start of the chain of references,
       * step down the chain, remembering the last one that
       * does not have the value attribute set.
       */
      OMobj_id tmp_id;
      *refer_id = OMnull_obj;

      while(1) {
         stat = OMget_obj_ref(elem_id, &tmp_id, 0);

         if( stat != OM_STAT_SUCCESS || OMequal_objs(tmp_id, elem_id) ) {
            if(OMis_null_obj(*refer_id)) return OM_STAT_UNDEF;
            else if ((stat = OMget_obj_val(*refer_id, &tmp_id)) != OM_STAT_SUCCESS)
               return(stat);
            else return OM_STAT_SUCCESS;
         }
         else if(OMget_obj_atts(tmp_id, OM_atts_value) == 1) {
            /* take another step */
            elem_id = tmp_id;
            /* but do not update the last good one */
         }
         else {
            /* take another step */
            elem_id = tmp_id;
            /* will return the last good one in refer_id */
            *refer_id = elem_id;
         }
      }
   }
}

int GDget_seqref_db(OMobj_id id, const char *name, int mode,
                    int *seq, OMobj_id *refer_id)
{
   OMobj_id elem_id;

   /* attach refer_id to id in database */
   elem_id = OMfind_subobj(id, OMstr_to_name(name), OM_OBJ_RW);
   if (OMis_null_obj(elem_id))
      return(0);
   else {
      *seq = OMget_obj_seq(elem_id, OMnull_obj, mode);
      return(OMget_obj_val(elem_id, refer_id));
   }
}

int GDget_obj_seq(OMobj_id id, const char *name, int *seq, int mode)
{
   OMobj_id elem_id;

   /* lookup element id of requested sub-element */
   elem_id = OMfind_subobj(id, OMstr_to_name(name), OM_OBJ_RW);
   /* return if null element */
   if (OMis_null_obj(elem_id))
      return(0);
   *seq = OMget_obj_seq(elem_id, OMnull_obj, mode);
   return(1);
}

void GDset_int_val(OMobj_id id, const char *name, int value)
{
   OMobj_id elem_id;

   elem_id = OMfind_subobj(id, OMstr_to_name(name), OM_OBJ_RW);
   if (OMis_null_obj(elem_id)) {
      return;
   }
   OMset_int_val(elem_id, value);
}

int GDget_int_val(OMobj_id id, const char *name, int *value)
{
   OMobj_id elem_id;

   *value = 0;
   /* lookup element id of requested sub-element */
   elem_id = OMfind_subobj(id, OMstr_to_name(name), OM_OBJ_RD);
   /* return if null element */
   if (OMis_null_obj(elem_id))
      return(0);

   if ((OMget_int_val(elem_id, value) == OM_STAT_SUCCESS)) {
      return(1);
   }
   else return(0);
}

/* This routine sets the value of the requested sub-element
   given the name of the element, and updates the sequence number.
*/
void GDset_newint_val(OMobj_id id, const char *name, int *seq, int value)
{
   OMobj_id elem_id;

   /* lookup element id of requested sub-element */
   elem_id = OMfind_subobj(id, OMstr_to_name(name), OM_OBJ_RW);
   if (OMis_null_obj(elem_id))
      return;
   OMset_int_val(elem_id, value);

   /* return updated sequence number */
   *seq = OMget_obj_seq(elem_id, OMnull_obj, OM_SEQ_VAL);
}

/* This routine returns the value of the requested sub-element
   given the name of the element, the current sequence number
   and a pointer to the return data. Return status is 1
   if the data has changed. */
int GDget_newint_val(OMobj_id id, const char *name, int *seq, int *value)
{
   OMobj_id elem_id;
   int elem_seq, ret_val = 0;

   *value = 0;
   /* lookup element id of requested sub-element */
   elem_id = OMfind_subobj(id, OMstr_to_name(name), OM_OBJ_RD);
   /* return if null element */
   if (OMis_null_obj(elem_id))
      return(0);

   /* if sequence number is different set return value */
   elem_seq = OMget_obj_seq(elem_id, OMnull_obj, OM_SEQ_VAL);
   if (elem_seq != *seq) {
      *seq = elem_seq;		/* update element sequence number */
      ret_val = 1;
   }
   if ((OMget_int_val(elem_id, value) == OM_STAT_SUCCESS)) {
      return(ret_val);
   }
   else return(0);
}

/* This routine returns the value of the requested sub-element
   given the name of the element.  Return status is 1 if the
   passed sequence number is less than the sequence number
   of the requested element.
*/
int GDget_updint_val(OMobj_id id, const char *name, int seq, int *value)
{
   OMobj_id elem_id;
   int elem_seq, ret_val = 0;

   *value = 0;
   /* lookup element id of requested sub-element */
   elem_id = OMfind_subobj(id, OMstr_to_name(name), OM_OBJ_RD);
   /* return if null element */
   if (OMis_null_obj(elem_id))
      return(0);

   /* if sequence number is different set return value */
   elem_seq = OMget_obj_seq(elem_id, OMnull_obj, OM_SEQ_VAL);
   if (elem_seq > seq)
      ret_val = 1;

   if ((OMget_int_val(elem_id, value) == OM_STAT_SUCCESS)) {
      return(ret_val);
   }
   else return(0);
}

/* 64-bit porting. Newly Introduced */
void GDset_long_val(OMobj_id id, const char *name, xp_long value)
{
   OMobj_id elem_id;

   elem_id = OMfind_subobj(id, OMstr_to_name(name), OM_OBJ_RW);
   if (OMis_null_obj(elem_id)) {
      return;
   }
   OMset_long_val(elem_id, value);
}

/* 64-bit porting. Newly Introduced */
int GDget_long_val(OMobj_id id, const char *name, xp_long *value)
{
   OMobj_id elem_id;

   *value = 0;
   /* lookup element id of requested sub-element */
   elem_id = OMfind_subobj(id, OMstr_to_name(name), OM_OBJ_RD);
   /* return if null element */
   if (OMis_null_obj(elem_id))
      return(0);

   if ((OMget_long_val(elem_id, value) == OM_STAT_SUCCESS)) {
      return(1);
   }
   else return(0);
}

/* This routine sets the value of the requested sub-element
   given the name of the element, and updates the sequence number.
*/
/* 64-bit porting. Newly Introduced */
void GDset_newlong_val(OMobj_id id, const char *name, int *seq, xp_long value)
{
   OMobj_id elem_id;

   /* lookup element id of requested sub-element */
   elem_id = OMfind_subobj(id, OMstr_to_name(name), OM_OBJ_RW);
   if (OMis_null_obj(elem_id))
      return;
   OMset_long_val(elem_id, value);

   /* return updated sequence number */
   *seq = OMget_obj_seq(elem_id, OMnull_obj, OM_SEQ_VAL);
}

/* This routine returns the value of the requested sub-element
   given the name of the element, the current sequence number
   and a pointer to the return data. Return status is 1
   if the data has changed. */
/* 64-bit porting. Newly Introduced */
int GDget_newlong_val(OMobj_id id, const char *name, int *seq, xp_long *value)
{
   OMobj_id elem_id;
   int elem_seq, ret_val = 0;

   *value = 0;
   /* lookup element id of requested sub-element */
   elem_id = OMfind_subobj(id, OMstr_to_name(name), OM_OBJ_RD);
   /* return if null element */
   if (OMis_null_obj(elem_id))
      return(0);

   /* if sequence number is different set return value */
   elem_seq = OMget_obj_seq(elem_id, OMnull_obj, OM_SEQ_VAL);
   if (elem_seq != *seq) {
      *seq = elem_seq;		/* update element sequence number */
      ret_val = 1;
   }
   if ((OMget_long_val(elem_id, value) == OM_STAT_SUCCESS)) {
      return(ret_val);
   }
   else return(0);
}

/* This routine returns the value of the requested sub-element
   given the name of the element.  Return status is 1 if the
   passed sequence number is less than the sequence number
   of the requested element.
*/
/* 64-bit porting. Newly Introduced */
int GDget_updlong_val(OMobj_id id, const char *name, int seq, xp_long *value)
{
   OMobj_id elem_id;
   int elem_seq, ret_val = 0;

   *value = 0;
   /* lookup element id of requested sub-element */
   elem_id = OMfind_subobj(id, OMstr_to_name(name), OM_OBJ_RD);
   /* return if null element */
   if (OMis_null_obj(elem_id))
      return(0);

   /* if sequence number is different set return value */
   elem_seq = OMget_obj_seq(elem_id, OMnull_obj, OM_SEQ_VAL);
   if (elem_seq > seq)
      ret_val = 1;

   if ((OMget_long_val(elem_id, value) == OM_STAT_SUCCESS)) {
      return(ret_val);
   }
   else return(0);
}

/* This routine returns the value of the requested sub-element
   given the name of the element.  Return status is 1 if the
   passed sequence number is less than the sequence number
   of the requested element.
*/
int GDget_updstr_val(OMobj_id id, const char *name, int seq, char **value)
{
   OMobj_id elem_id;
   int elem_seq, ret_val = 0;

   /* lookup element id of requested sub-element */
   elem_id = OMfind_subobj(id, OMstr_to_name(name), OM_OBJ_RD);
   /* return if null element */
   if (OMis_null_obj(elem_id))
      return(0);

   /* if sequence number is different set return value */
   elem_seq = OMget_obj_seq(elem_id, OMnull_obj, OM_SEQ_VAL);
   if (elem_seq > seq)
      ret_val = 1;

   if ((OMget_str_val(elem_id, value, 0) == OM_STAT_SUCCESS)) {
      return(ret_val);
   }
   else return(0);
}

#ifdef USE_PROTOS
void GDset_float_val(OMobj_id id, const char *name, float value)
#else
void GDset_float_val(id, name, value)
OMobj_id id;
char *name;
float value;
#endif
{
   OMobj_id elem_id;

   elem_id = OMfind_subobj(id, OMstr_to_name(name), OM_OBJ_RW);
   if (OMis_null_obj(elem_id))
      return;
   OMset_real_val(elem_id, value);
}

int GDget_float_val(OMobj_id id, const char *name, float *value)
{
   OMobj_id elem_id;
   double tmp_val;

   /* lookup element id of requested sub-element */
   elem_id = OMfind_subobj(id, OMstr_to_name(name), OM_OBJ_RD);

   /* return if null element */
   if (OMis_null_obj(elem_id))
      return(0);

   if ((OMget_real_val(elem_id, &tmp_val) == OM_STAT_SUCCESS)) {
      *value = tmp_val;
      return(1);
   }
   else return(0);
}

/* This routine sets the value of the requested sub-element
   given the name of the element, and updates the sequence number.
*/
#ifdef USE_PROTOS
void GDset_newfloat_val(OMobj_id id, const char *name, int *seq, float value)
#else
void GDset_newfloat_val(id, name, seq, value)
OMobj_id id;
char *name;
int *seq;
float value;
#endif
{
   OMobj_id elem_id;

   /* lookup element id of requested sub-element */
   elem_id = OMfind_subobj(id, OMstr_to_name(name), OM_OBJ_RW);
   if (OMis_null_obj(elem_id))
      return;
   OMset_real_val(elem_id, value);

   /* return updated sequence number */
   *seq = OMget_obj_seq(elem_id, OMnull_obj, OM_SEQ_VAL);
}

/* This routine returns the value of the requested sub-element
   given the name of the element, the current sequence number
   and a pointer to the return data. Return status is 1
   if the data has changed. */
int GDget_newfloat_val(OMobj_id id, const char *name, int *seq, float *value)
{
   OMobj_id elem_id;
   int elem_seq, ret_val = 0;
   double tmp_val;

   *value = 0.0;
   /* lookup element id of requested sub-element */
   elem_id = OMfind_subobj(id, OMstr_to_name(name), OM_OBJ_RD);
   /* return if null element */
   if (OMis_null_obj(elem_id))
      return(0);

   /* if sequence number is different set return value */
   elem_seq = OMget_obj_seq(elem_id, OMnull_obj, OM_SEQ_VAL);
   if (elem_seq != *seq) {
      *seq = elem_seq;		/* update element sequence number */
      ret_val = 1;
   }
   if ((OMget_real_val(elem_id, &tmp_val) == OM_STAT_SUCCESS)) {
      *value = tmp_val;
      return(ret_val);
   }
   else return(0);
}

/* This routine returns the value of the requested sub-element
   given the name of the element.  Return status is 1 if the
   passed sequence number is less than the sequence number
   of the requested element.
*/
int GDget_updfloat_val(OMobj_id id, const char *name, int seq, float *value)
{
   OMobj_id elem_id;
   int elem_seq, ret_val = 0;
   double tmp_val;

   *value = 0.0;
   /* lookup element id of requested sub-element */
   elem_id = OMfind_subobj(id, OMstr_to_name(name), OM_OBJ_RD);
   /* return if null element */
   if (OMis_null_obj(elem_id))
      return(0);

   /* if sequence number is different set return value */
   elem_seq = OMget_obj_seq(elem_id, OMnull_obj, OM_SEQ_VAL);
   if (elem_seq > seq)
      ret_val = 1;

   if ((OMget_real_val(elem_id, &tmp_val) == OM_STAT_SUCCESS)) {
      *value = tmp_val;
      return(ret_val);
   }
   else return(0);
}

/* 64-bit porting. Directly Modified */
void GDset_int_array(OMobj_id id, const char *name, xp_long size, int *values)
	/* size is number of elements - NOT size in bytes */
{
   OMobj_id elem_id;

   elem_id = OMfind_subobj(id, OMstr_to_name(name), OM_OBJ_RW);
   if (OMis_null_obj(elem_id))
      return;
   OMset_array(elem_id, OM_TYPE_INT, (char *)values, size,
      OM_SET_ARRAY_COPY);
}

/* This routine returns the value of the requested sub-element
   given the name of the element, the current sequence number
   and a pointer to the return data. Return status is 1
   if the data has changed. */
/* 64-bit porting. Modified with Parallel  APIs */
int GDget_newint_array(OMobj_id id, const char *name, int *seq,
                       xp_long *size, int **values)
{
   OMobj_id elem_id;
   int type = OM_TYPE_INT, elem_seq, ret_val = 0;

   /* lookup element id of requested sub-element */
   elem_id = OMfind_subobj(id, OMstr_to_name(name), OM_OBJ_RD);
   /* return if null element */
   if (OMis_null_obj(elem_id))
      return(ret_val);

   /* if sequence number is the same return 1 */
   elem_seq = OMget_obj_seq(elem_id, OMnull_obj, OM_SEQ_VAL);
   if (elem_seq != *seq) {
      *seq = elem_seq;		/* update element sequence number */
      ret_val = 1;
   }
   *size = 0;
   *values = (int *)NULL;
   if ((OMget_array_sz(elem_id, &type, (char **)values, size,
	OM_GET_ARRAY_RD_COPY) == OM_STAT_SUCCESS)) {
      /* If we are returning a zero value but we also got the array,
         we need to free it because the caller will not.
       */
      if (ret_val != 1 && *values != NULL) ARRfree(*values);
      return(ret_val);
   }
   else return(0);
}

#ifdef WORDLENGTH_64
int GDget_newint_array_n(OMobj_id id, const char *name, int *seq,
                       int *size, int **values)
{
   xp_long size_w;
   int status;
   
   status = GDget_newint_array(id, name, seq, &size_w, values);
   
   *size = (int)size_w;
   
   return status;
}
#endif

/* This routine returns the value of the requested sub-element
   given the name of the element, the current sequence number
   and a pointer to the return data. Return status is 1
   if the data has changed. */
/* 64-bit porting. Modified with Parallel  APIs */
int GDget_updint_array(OMobj_id id, const char *name, int seq,
                       xp_long *size, int **values)
{
   OMobj_id elem_id;
   int type = OM_TYPE_INT, elem_seq, ret_val = 0;

   /* lookup element id of requested sub-element */
   elem_id = OMfind_subobj(id, OMstr_to_name(name), OM_OBJ_RD);
   /* return if null element */
   if (OMis_null_obj(elem_id))
      return(ret_val);

   /* if sequence number is the same return 1 */
   elem_seq = OMget_obj_seq(elem_id, OMnull_obj, OM_SEQ_VAL);
   if (elem_seq > seq)
      ret_val = 1;

   *size = 0;
   *values = (int *)NULL;
   if ((OMget_array_sz(elem_id, &type, (char **)values, size,
	OM_GET_ARRAY_RD_COPY) == OM_STAT_SUCCESS)) {
      /* If we are returning a zero value but we also got the array,
         we need to free it because the caller will not.
       */
      if (ret_val != 1 && *values != NULL) ARRfree(*values);
      return(ret_val);
   }
   else return(0);
}

#ifdef WORDLENGTH_64
int GDget_updint_array_n(OMobj_id id, const char *name, int seq,
                       int *size, int **values)
{
   xp_long size_w;
   int status;
   
   status = GDget_updint_array(id, name, seq, &size_w, values);
   
   *size = (int)size_w;
   
   return status;
}
#endif

/* 64-bit porting. Modified with Parallel  APIs */
int GDget_int_array(OMobj_id id, const char *name, xp_long *size, int **values)
{
   OMobj_id elem_id;
   int type = OM_TYPE_INT;

   /* lookup element id of requested sub-element */
   elem_id = OMfind_subobj(id, OMstr_to_name(name), OM_OBJ_RD);
   /* return if null element */
   if (OMis_null_obj(elem_id))
      return(0);

   *size = 0;
   *values = (int *)NULL;
   /* Ask for the array as RD_COPY so if the user changes the
      returned array, it won't actually changes the values in
      the OM.
   */
   if ((OMget_array_sz(elem_id, &type, (char **)values, size,
	OM_GET_ARRAY_RD_COPY) == OM_STAT_SUCCESS))
      return(1);
   else return(0);
}

#ifdef WORDLENGTH_64
int GDget_int_array_n(OMobj_id id, const char *name, int *size, int **values)
{
   xp_long size_w;
   int status;
   
   status = GDget_int_array(id, name, &size_w, values);
   
   *size = (int)size_w;
   
   return status;
}
#endif

/* 64-bit porting. Newly Introduced */
void GDset_long_array(OMobj_id id, const char *name, xp_long size, xp_long *values)
	/* size is number of elements - NOT size in bytes */
{
   OMobj_id elem_id;

   elem_id = OMfind_subobj(id, OMstr_to_name(name), OM_OBJ_RW);
   if (OMis_null_obj(elem_id))
      return;
   OMset_array(elem_id, OM_TYPE_LONG, (char *)values, size,
      OM_SET_ARRAY_COPY);
}

/* This routine returns the value of the requested sub-element
   given the name of the element, the current sequence number
   and a pointer to the return data. Return status is 1
   if the data has changed. */
/* 64-bit porting. Newly Introduced */
int GDget_newlong_array(OMobj_id id, const char *name, int *seq,
                       xp_long *size, xp_long **values)
{
   OMobj_id elem_id;
   int type = OM_TYPE_LONG, elem_seq, ret_val = 0;

   /* lookup element id of requested sub-element */
   elem_id = OMfind_subobj(id, OMstr_to_name(name), OM_OBJ_RD);
   /* return if null element */
   if (OMis_null_obj(elem_id))
      return(ret_val);

   /* if sequence number is the same return 1 */
   elem_seq = OMget_obj_seq(elem_id, OMnull_obj, OM_SEQ_VAL);
   if (elem_seq != *seq) {
      *seq = elem_seq;		/* update element sequence number */
      ret_val = 1;
   }
   *size = 0;
   *values = (xp_long *)NULL;
   if ((OMget_array_sz(elem_id, &type, (char **)values, size,
	OM_GET_ARRAY_RD_COPY) == OM_STAT_SUCCESS)) {
      /* If we are returning a zero value but we also got the array,
         we need to free it because the caller will not.
       */
      if (ret_val != 1 && *values != NULL) ARRfree(*values);
      return(ret_val);
   }
   else return(0);
}

/* This routine returns the value of the requested sub-element
   given the name of the element, the current sequence number
   and a pointer to the return data. Return status is 1
   if the data has changed. */
/* 64-bit porting. Newly Introduced */
int GDget_updlong_array(OMobj_id id, const char *name, int seq,
                       xp_long *size, xp_long **values)
{
   OMobj_id elem_id;
   int type = OM_TYPE_LONG, elem_seq, ret_val = 0;

   /* lookup element id of requested sub-element */
   elem_id = OMfind_subobj(id, OMstr_to_name(name), OM_OBJ_RD);
   /* return if null element */
   if (OMis_null_obj(elem_id))
      return(ret_val);

   /* if sequence number is the same return 1 */
   elem_seq = OMget_obj_seq(elem_id, OMnull_obj, OM_SEQ_VAL);
   if (elem_seq > seq)
      ret_val = 1;

   *size = 0;
   *values = (xp_long *)NULL;
   if ((OMget_array_sz(elem_id, &type, (char **)values, size,
	OM_GET_ARRAY_RD_COPY) == OM_STAT_SUCCESS)) {
      /* If we are returning a zero value but we also got the array,
         we need to free it because the caller will not.
       */
      if (ret_val != 1 && *values != NULL) ARRfree(*values);
      return(ret_val);
   }
   else return(0);
}

/* 64-bit porting. Newly Introduced */
int GDget_long_array(OMobj_id id, const char *name, xp_long *size, xp_long **values)
{
   OMobj_id elem_id;
   int type = OM_TYPE_LONG;

   /* lookup element id of requested sub-element */
   elem_id = OMfind_subobj(id, OMstr_to_name(name), OM_OBJ_RD);
   /* return if null element */
   if (OMis_null_obj(elem_id))
      return(0);

   *size = 0;
   *values = (xp_long *)NULL;
   /* Ask for the array as RD_COPY so if the user changes the
      returned array, it won't actually changes the values in
      the OM.
   */
   if ((OMget_array_sz(elem_id, &type, (char **)values, size,
	OM_GET_ARRAY_RD_COPY) == OM_STAT_SUCCESS))
      return(1);
   else return(0);
}

/* 64-bit porting. Directly Modified */
void GDset_float_array(OMobj_id id, const char *name, xp_long size, float *values)
	/* size is number of elements - NOT size in bytes */
{
   OMobj_id elem_id;

   elem_id = OMfind_subobj(id, OMstr_to_name(name), OM_OBJ_RW);
   if (OMis_null_obj(elem_id))
      return;
   OMset_array(elem_id, OM_TYPE_FLOAT, (char *)values, size,
      OM_SET_ARRAY_COPY);
}

/* This routine returns the value of the requested sub-element
   given the name of the element, the current sequence number
   and a pointer to the return data. Return status is 1
   if the data has changed. */
/* 64-bit porting. Modified with Parallel  APIs */
int GDget_newfloat_array(OMobj_id id, const char *name, int *seq,
                         xp_long *size, float **values)
{
   OMobj_id elem_id;
   int type = OM_TYPE_FLOAT, elem_seq, ret_val = 0;

   /* lookup element id of requested sub-element */
   elem_id = OMfind_subobj(id, OMstr_to_name(name), OM_OBJ_RD);
   /* return if null element */
   if (OMis_null_obj(elem_id))
      return(0);

   /* if sequence number is the same just return */
   elem_seq = OMget_obj_seq(elem_id, OMnull_obj, OM_SEQ_VAL);
   if (elem_seq != *seq) {
      *seq = elem_seq;		/* update element sequence number */
      ret_val = 1;
   }
   *size = 0;
   *values = (float *)NULL;
   if ((OMget_array_sz(elem_id, &type, (char **)values, size,
	OM_GET_ARRAY_RD_COPY) == OM_STAT_SUCCESS)) {
      /* If we are returning a zero value but we also got the array,
         we need to free it because the caller will not.
       */
      if (ret_val != 1 && *values != NULL) ARRfree(*values);
      return(ret_val);
   }
   else return(0);
}

#ifdef WORDLENGTH_64
int GDget_newfloat_array_n(OMobj_id id, const char *name, int *seq,
                         int *size, float **values)
{
   xp_long size_w;
   int status;
   
   status = GDget_newfloat_array(id, name, seq, &size_w, values);
   
   *size = (int)size_w;
   
   return status;	
}
#endif

/* This routine returns the values of the requested sub-element
   given the name of the element.  Return status is 1 if the
   passed sequence number is less than the sequence number
   of the requested element.
*/
/* 64-bit porting. Modified with Parallel  APIs */
int GDget_updfloat_array(OMobj_id id, const char *name, int seq,
                         xp_long *size, float **values)
{
   OMobj_id elem_id;
   int type = OM_TYPE_FLOAT, elem_seq, ret_val = 0;

   /* lookup element id of requested sub-element */
   elem_id = OMfind_subobj(id, OMstr_to_name(name), OM_OBJ_RD);
   /* return if null element */
   if (OMis_null_obj(elem_id))
      return(0);

   /* See if this element has actually changed. */
   elem_seq = OMget_obj_seq(elem_id, OMnull_obj, OM_SEQ_VAL);
   if (elem_seq > seq)
      ret_val = 1;

   *size = 0;
   *values = (float *)NULL;
   if ((OMget_array_sz(elem_id, &type, (char **)values, size,
	OM_GET_ARRAY_RD_COPY) == OM_STAT_SUCCESS)) {
      /* If we are returning a zero value but we also got the array,
         we need to free it because the caller will not.
       */
      if (ret_val != 1 && *values != NULL) ARRfree(*values);
      return(ret_val);
   }
   else return(0);
}

#ifdef WORDLENGTH_64
int GDget_updfloat_array_n(OMobj_id id, const char *name, int seq,
                         int *size, float **values)
{
   xp_long size_w;
   int status;
   
   status = GDget_updfloat_array(id, name, seq, &size_w, values);
   
   *size = (int)size_w;
   
   return status;	
}
#endif

/* 64-bit porting. Modified with Parallel  APIs */
int GDget_float_array(OMobj_id id, const char *name, xp_long *size, float **values)
{
   OMobj_id elem_id;
   int type = OM_TYPE_FLOAT;

   /* lookup element id of requested sub-element */
   elem_id = OMfind_subobj(id, OMstr_to_name(name), OM_OBJ_RD);

   /* return if null element */
   if (OMis_null_obj(elem_id))
      return(0);

   *size = 0;
   *values = (float *)NULL;
   if ((OMget_array_sz(elem_id, &type, (char **)values, size,
	OM_GET_ARRAY_RD_COPY) == OM_STAT_SUCCESS))
      return(1);
   else return(0);
}

#ifdef WORDLENGTH_64
int GDget_float_array_n(OMobj_id id, const char *name, int *size, float **values)
{
   xp_long size_w;
   int status;
   
   status = GDget_float_array(id, name, &size_w, values);
   
   *size = (int)size_w;
   
   return status;	
}
#endif

int GDget_str_val(OMobj_id id, const char *name, char **value)
{
   OMobj_id elem_id;

   /* lookup element id of requested sub-element */
   elem_id = OMfind_subobj(id, OMstr_to_name(name), OM_OBJ_RD);
   /* return if null element */
   if (OMis_null_obj(elem_id))
      return(0);

   if ((OMget_str_val(elem_id, value, 0) == OM_STAT_SUCCESS)) {
      return(1);
   }
   else return(0);
}

void GDset_newstr_val(OMobj_id id, const char *name, int *seq, char *value)
{
   OMobj_id elem_id;

   /* lookup element id of requested sub-element */
   elem_id = OMfind_subobj(id, OMstr_to_name(name), OM_OBJ_RW);
   /* return if null element */
   if (OMis_null_obj(elem_id))
      return;

   OMset_str_val(elem_id, value);

   /* return updated sequence number */
   *seq = OMget_obj_seq(elem_id, OMnull_obj, OM_SEQ_VAL);
}

/* This routine returns the value of the requested sub-element
   given the name of the element, the current sequence number
   and a pointer to the return data. Return status is 1
   if the data has changed. */
int GDget_newstr_val(OMobj_id id, const char *name, int *seq, char **value)
{
   OMobj_id elem_id;
   int elem_seq, ret_val = 0;

   /* lookup element id of requested sub-element */
   elem_id = OMfind_subobj(id, OMstr_to_name(name), OM_OBJ_RD);
   /* return if null element */
   if (OMis_null_obj(elem_id))
      return(0);

   /* if sequence number is different set return value */
   elem_seq = OMget_obj_seq(elem_id, OMnull_obj, OM_SEQ_VAL);
   if (elem_seq != *seq) {
      *seq = elem_seq;		/* update element sequence number */
      ret_val = 1;
   }
   if ((OMget_str_val(elem_id, value, 0) == OM_STAT_SUCCESS)) {
      return(ret_val);
   }
   else return(0);
}
